// Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef TAPIOCA_SESSION_CONNECTOR_H
#define TAPIOCA_SESSION_CONNECTOR_H

#include <gtkmm.h>
#include <string>
#include <libsoupmm/session.h>
#include <libsoupmm/message.h>

namespace Tapioca 
{
  class Document;
  class Item;
  class Gallery;
  class Session;
  class ChangedDocument;
  class ChangedGallery;
class SessionChanges
  {
public:
    SessionChanges(){};
    ~SessionChanges(){};
    bool empty() {return (modified_galleries.empty() == true &&
                          modified_docs.empty() == true && 
                          removed_galleries.empty() == true &&
                          added_galleries.empty() == true && 
                          removed_documents.empty() == true &&
                          added_documents.empty() == true);}

    void find_moved_docs();
    std::list<ChangedGallery> modified_galleries;
    std::list<ChangedDocument> modified_docs;
    std::list<ChangedGallery> removed_galleries;
    std::list<ChangedGallery> added_galleries;
    std::list<ChangedDocument> removed_documents;
    std::list<ChangedDocument> added_documents;
    std::list<std::pair<ChangedDocument, ChangedDocument> > moved_documents;
  };
class SessionConnector
{
    public:
  
        static double timeout; //in minutes.

	SessionConnector(Session *s);

	//! Destructor.
        virtual ~SessionConnector();

        void revert(SessionChanges reverts);
        void apply(SessionChanges changes);

        typedef sigc::slot2<void, Document *, std::string> SlotDownloadDocument;
        typedef sigc::slot2<void, Gallery*, std::list<std::string>& > SlotPullGallery;
        typedef sigc::slot1<void, Gallery*> SlotAddGallery;
        typedef sigc::slot1<void, Gallery*> SlotRemoveGallery;
        typedef sigc::slot2<void, Document*, Document*> SlotMoveDocument;
        typedef sigc::slot1<void, Document*> SlotRemoveDocument;
        typedef sigc::slot1<void, Document*> SlotAddDocument;
        typedef sigc::slot1<void, Gallery*> SlotModifyGallery;
        typedef sigc::slot1<void, Document*> SlotModifyDocument;
        typedef sigc::slot1<void, Item*> SlotUpdateTags;
        typedef sigc::slot1<void, Document*> SlotPullThumbnail;

        void download_document(Document *document, const SessionConnector::SlotDownloadDocument slot);

        void add_gallery(Gallery *gallery, const SessionConnector::SlotAddGallery slot, bool retry = false);
        void remove_gallery(Gallery *gallery, const SessionConnector::SlotRemoveGallery slot);
        void update_item_tags(Item *item, const SessionConnector::SlotUpdateTags slot, bool gallery);
        void move_document(Document *src, Document *dest, const SessionConnector::SlotMoveDocument slot);
        void pull_thumbnail(std::string url, const SessionConnector::SlotPullThumbnail slot);
        void add_document(Tapioca::Document *document, const SessionConnector::SlotAddDocument slot);
        void remove_document(Tapioca::Document *document, const SessionConnector::SlotRemoveDocument slot);
        void modify_gallery(Gallery *gallery, const SessionConnector::SlotModifyGallery slot);
        void modify_document(Document *document, const SessionConnector::SlotModifyDocument slot);


        sigc::signal<void> pulled_gallery(Gallery *g);
        sigc::signal<void> pulled_documents_for_gallery(Gallery *g);
        sigc::signal<void> pulled_thumbnail_for_document(Document *d);
        sigc::signal<void> last_document_page_processed;
        sigc::signal<void, Document*> downloaded_document;
        sigc::signal<void, Document*> downloaded_chunk_of_document;
        sigc::signal<void, Document*> download_document_failed;
        sigc::signal<void, Session*> pull_galleries_failed;
        sigc::signal<void, std::list<Gallery*> > pulled_galleries;
        sigc::signal<void, Gallery*> add_gallery_failed;
        sigc::signal<void, Document*> add_document_failed;
        sigc::signal<void, Item*> update_tags_failed;
        sigc::signal<void, Gallery*> remove_gallery_failed;
        sigc::signal<void, Document*> remove_document_failed;
        sigc::signal<void, Gallery*> modify_gallery_failed;
        sigc::signal<void, Document*> modify_document_failed;

        void pull_galleries();
        void login(sigc::slot<void, bool> slot);
        
        void pull_documents_for_gallery(Gallery *gallery, const SessionConnector::SlotPullGallery slot, std::list<std::string> &thumbnail_urls);
        void get_galleries_from_gallery_listing_page(std::string data);
        void get_documents_from_document_listing_page(Gallery *gallery, std::string data, std::list<std::string> &thumbnail_urls);
        sigc::signal<void, Gallery*> processed_gallery;
        sigc::signal<void, Gallery*, Document*> processed_document_for_gallery;
        sigc::signal<void, Document*> processed_thumbnail_for_document;
        void process_thumbnail(Document *document, const char *data, guint len);
        static Glib::ustring decode_xml(Glib::ustring data);
    private:
        Session *session;

        Glib::RefPtr<Soup::Session> web;
        time_t last_login;

        //callbacks
        void on_login_attempted(Glib::RefPtr<Soup::Message> &msg,
                                sigc::slot<void, bool> slot);
        void on_document_downloaded(Glib::RefPtr<Soup::Message> msg, Document *document, const SessionConnector::SlotDownloadDocument slot, sigc::connection timer);

        void on_new_document_downloaded(Glib::RefPtr<Soup::Message> msg);

        std::string get_document_link_from_link_listing(Glib::RefPtr<Soup::Message> msg, Document *document) const;

        void on_pull_galleries_attempted(Glib::RefPtr<Soup::Message> &msg);

        void on_pull_documents_attempted(Glib::RefPtr<Soup::Message> &msg, Gallery *gallery, const SessionConnector::SlotPullGallery slot, std::list<std::string> &thumbnail_urls);

        void on_thumbnail_downloaded(Glib::RefPtr<Soup::Message> &msg, Document *document, const SessionConnector::SlotPullThumbnail slot);

        bool on_download_document_timeout(Glib::RefPtr<Soup::Message> &msg);

        void on_update_tags_attempted(Glib::RefPtr<Soup::Message> & msg, Item *item, const SessionConnector::SlotUpdateTags slot);

        void on_add_gallery_attempted(Glib::RefPtr<Soup::Message> &msg, Gallery *gallery, const SessionConnector::SlotAddGallery slot, bool retried);

        void on_add_document_attempted(Glib::RefPtr<Soup::Message> &msg, Document *document, const SessionConnector::SlotAddDocument slot);

        void on_remove_gallery_attempted(Glib::RefPtr<Soup::Message> &msg, Gallery *gallery, const SessionConnector::SlotRemoveGallery slot);

        void on_remove_document_attempted(Glib::RefPtr<Soup::Message> &msg, Document *document, const SessionConnector::SlotRemoveDocument slot);

        void on_modify_gallery_attempted(Glib::RefPtr<Soup::Message> &msg, Gallery *gallery, const SessionConnector::SlotModifyGallery slot);

        void on_modify_document_attempted(Glib::RefPtr<Soup::Message> &msg, Document *document, const SessionConnector::SlotModifyDocument slot);

        void on_new_gallery_updated_tag(Item *item, const SessionConnector::SlotAddGallery slot, Gallery *gallery);

        void on_new_document_updated_tag(Item *item, const SessionConnector::SlotAddDocument slot, Document *document);

        void on_modify_gallery_updated_tag(Item *item, const SessionConnector::SlotModifyGallery, Gallery *gallery);

        void on_modify_document_updated_tag(Item *item, const SessionConnector::SlotModifyDocument slot, Document *document);

        void on_document_downloaded_for_move(Tapioca::Document *src, std::string file, Tapioca::Document *dest, const SessionConnector::SlotMoveDocument slot);

        void on_document_added_for_move(Tapioca::Document *dest, Tapioca::Document *src, const SessionConnector::SlotMoveDocument slot);

        void on_document_removed_for_move(Tapioca::Document *src, Tapioca::Document *dest, const SessionConnector::SlotMoveDocument slot);

        //helpers
        bool is_login_necessary();
        std::string extract_document_id_from_thumbnail_url(std::string u) const;
        Glib::RefPtr<Soup::Message> login();
        guint revert_modified_galleries(std::list<ChangedGallery> l);
        guint revert_modified_documents(std::list<ChangedDocument> l);
        guint revert_added_galleries(std::list<ChangedGallery> l);
        guint revert_removed_galleries(std::list<ChangedGallery> l);
        guint revert_added_documents(std::list<ChangedDocument> l);
        guint revert_removed_documents(std::list<ChangedDocument> l);
        bool revert_modified_gallery(ChangedGallery g);
        bool revert_modified_document(ChangedDocument d);
        bool revert_added_gallery(ChangedGallery g);
        bool revert_removed_gallery(ChangedGallery g);
        bool revert_added_document(ChangedDocument d);
        bool revert_removed_document(ChangedDocument d);
        guint update_modified_galleries(std::list<ChangedGallery> l);
        guint update_modified_documents(std::list<ChangedDocument> l);
        guint update_added_galleries(std::list<ChangedGallery> l);
        guint update_added_documents(std::list<ChangedDocument> l);
        bool update_modified_gallery(ChangedGallery g);
        bool update_modified_document(ChangedDocument d);
        bool update_added_gallery(ChangedGallery g);
        bool update_added_document(ChangedDocument d);
        guint revert_moved_documents(std::list<std::pair<ChangedDocument, ChangedDocument> > &moved_docs);
        bool revert_moved_document(ChangedDocument &added_document, ChangedDocument &removed_document);
        std::string get_session_id_from_page(std::string data);
        std::string get_gallery_id_from_gallery_insert_page(std::string data);
        std::string get_document_id_from_document_insert_page(std::string data);
        void fill_in_document_values_from_document_page(Document *document, std::string data);
        void fill_in_gallery_values_from_gallery_page(Gallery *gallery, std::string data);
        std::string get_title_from_document_page(std::string data);
        std::string get_description_from_document_page(std::string data);
        std::string get_priority_from_document_page(std::string data);
        double get_latitude_from_document_page(std::string data);
        double get_longitude_from_document_page(std::string data);
        bool fill_in_document_values(Document *document);
        bool fill_in_gallery_values(Gallery *gallery);
        Gallery *process_gallery_chunk(std::string data) const;
        std::string get_uid_from_gallery_listing_page(std::string data) const;
        std::list<std::string> get_gallery_tags_from_gallery_listing_page(std::string data) const;
        std::string get_gallery_title_from_gallery_listing_page(std::string data) const;
        std::string get_gallery_description_from_gallery_listing_page(std::string data) const;
        std::string get_gallery_id_from_gallery_listing_page(std::string data) const;
        std::string get_gallery_priority_from_gallery_listing_page(std::string data) const;
        void get_gallery_lat_lon_from_gallery_listing_page(std::string data, double &lat, double &lon) const;
        std::string get_gallery_thumbnail_id_from_gallery_listing_page(std::string data) const;

        Tapioca::Document* get_document_from_gallery_listing_page(std::string data) const;
        std::string get_document_id_from_gallery_listing_page(std::string data) const;
        std::string get_document_order_from_gallery_listing_page(std::string data) const;
        std::string get_original_document_filename_from_gallery_listing_page(std::string data) const;
        std::string get_document_title_from_gallery_listing_page(std::string data) const;
        std::string get_document_description_from_gallery_listing_page(std::string data) const;
        std::list<std::string> get_document_tags_from_gallery_listing_page(std::string data) const;
        void get_document_lat_long_from_gallery_listing_page(std::string data, double &lat, double &lon) const;

        bool get_gallery_audience_from_gallery_listing_page(std::string data) const;

        std::string get_thumbnail_url_from_gallery_listing_page(std::string data) const;
    
        static void replace(Glib::ustring &data, const Glib::ustring src, const Glib::ustring dest);
};
}

#endif // TAPIOCA_SESSION_CONNECTOR_H
