// Copyright (C) 2010, 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <fstream>
#include "gallerylist.h"
#include "changed-gallery.h"
#include "changed-document.h"
#include "gallery.h"

using namespace Tapioca;

GalleryList::GalleryList(const GalleryList& g)
{
  for (const_iterator it = g.get_galleries().begin(); 
       it != g.get_galleries().end(); it++)
    _galleries.push_back(new Gallery(*(*it)));
}

Gallery* GalleryList::find_by_id(const std::string id) const
{
  for (const_iterator it = _galleries.begin(); it != _galleries.end(); it++)
    {
      if ((*it)->get_id() == id)
        return (*it);
    }
  return NULL;
}

Document * GalleryList::find_doc_by_id(const std::string id) const
{
  for (const_iterator it = get_galleries().begin(); 
       it != get_galleries().end(); it++)
    {
      for (Gallery::const_iterator git = (*it)->get_docs().begin(); 
           git != (*it)->get_docs().end(); git++)
        {
          if ((*git)->get_id() == id)
            return (*git);
        }
    }
  return NULL;
}

void GalleryList::add(const GalleryList *g)
{
  if (g)
    {
      for (const_iterator it = g->get_galleries().begin();
           it != g->get_galleries().end(); it++)
        _galleries.push_back(new Gallery(*(*it)));
    }
}

std::list<ChangedGallery> GalleryList::get_changed_galleries(const GalleryList *g) const
{
  std::list<ChangedGallery> difflist;
  //which one of these is different from those?
  for (const_iterator it = get_galleries().begin(); 
       it != get_galleries().end(); it++)
    {
      //does this id appear in the given list?
      Gallery *gallery = g->find_by_id((*it)->get_id());
      if (gallery != NULL)
        {
          if (*gallery == *(*it))
            ;
          else
            {
              if (gallery->same_except_for_thumbnail_id(*(*it)) == false)
                {
                  ChangedGallery changed_gallery(*(*it), *gallery);
                  difflist.push_back(changed_gallery);
                }
            }
        }
    }
  return difflist;
}

std::list<ChangedGallery> GalleryList::get_removed_galleries(const GalleryList *g) const
{
  std::list<ChangedGallery> difflist;
  for (const_iterator it = get_galleries().begin(); 
       it != get_galleries().end(); it++)
    {
      Gallery *gallery = g->find_by_id((*it)->get_id());
      if (gallery == NULL)
        {
          ChangedGallery changed_gallery(*(*it), Gallery());
          difflist.push_back(changed_gallery);
        }
    }
  return difflist;
}

std::list<ChangedDocument> GalleryList::get_changed_documents(const GalleryList *g) const
{
  std::list<ChangedDocument> difflist;
  for (const_iterator it = get_galleries().begin(); 
       it != get_galleries().end(); it++)
    {
      for (Gallery::const_iterator git = (*it)->get_docs().begin(); 
           git != (*it)->get_docs().end(); git++)
        {
          Document *document = g->find_doc_by_id((*git)->get_id());
          if (document != NULL)
            {
              if (*document == *(*git))
                ;
              else
                {
                  if (document->get_gallery_id() == (*git)->get_gallery_id())
                    difflist.push_back(ChangedDocument(*(*git), *document));
                }
            }
        }
    }
  return difflist;
}

std::list<ChangedDocument> GalleryList::get_removed_documents(const GalleryList *g) const
{
  std::list<ChangedDocument> difflist;
  for (const_iterator it = get_galleries().begin(); it != get_galleries().end(); it++)
    {
      for (Gallery::const_iterator git = (*it)->get_docs().begin(); 
           git != (*it)->get_docs().end(); git++)
        {
          Document *document = g->find_doc_by_id((*git)->get_id());
          if (document == NULL)
            difflist.push_back(ChangedDocument(*(*git), Document()));
          else
            {
              if (document->get_gallery_id() != (*git)->get_gallery_id())
                difflist.push_back(ChangedDocument(*(*git), Document()));
            }
        }
    }
  return difflist;
}

GalleryList::GalleryList()
{
}

GalleryList::~GalleryList()
{
  remove_all_galleries();
}

void GalleryList::update_thumbnails()
{
  for (iterator it = _galleries.begin(); it != _galleries.end(); it++)
    (*it)->update_thumbnails();
}

void GalleryList::find_deleted_documents(std::list<Document*> &removed_docs)
{
  for (iterator it = _galleries.begin(); it != _galleries.end(); it++)
    (*it)->find_deleted_documents(removed_docs);
}

void GalleryList::remove_all_galleries()
{
  for (iterator it = _galleries.begin(); it != _galleries.end(); it++)
    delete (*it);
  _galleries.clear();
}

size_t GalleryList::count_documents() const
{
  size_t count = 0;
  for (const_iterator it = get_galleries().begin(); 
       it != get_galleries().end(); it++)
    count += (*it)->size();
  return count;
}

void GalleryList::export_html_pages(std::string u, std::string export_dir) const
{
  std::string top_index = export_dir + "/index.html";
  std::ofstream ofs(top_index.c_str());
  if (ofs.is_open() == false)
    return;
  ofs << 
    "<html>\n\t<title>AtPic Galleries</title>\n\t<body>" << std::endl;
  ofs << "\t\t<div class=\"maintitleblock\">" << std::endl;
  ofs <<"\t\t" << Glib::Markup::escape_text(u) << "'s Galleries" << std::endl;
  ofs << "\t\t</div>" << std::endl;
  for (const_iterator i = get_galleries().begin(); 
       i != get_galleries().end(); i++)
    {
      Gallery *g = *i;
      ofs << "\t\t<div class=\"galleryblock\">" << std::endl;
      ofs << "\t\t\t";
      if (g->get_thumbnail_id() != "")
        {
          Document *d= find_doc_by_id(g->get_thumbnail_id());
          if (d)
            {
              ofs << "\t\t\t";
              ofs << "<img height=\"20%\" src=\"" + g->get_id() +"/"+ d->get_id() +"-" +d->get_original_filename() + "\"/>" << std::endl;
            }
        }
      ofs << "<a href=\"" + g->get_id() + "/index.html\"> " + 
        Glib::Markup::escape_text(g->get_title()) + "</a>"<< std::endl;
      ofs << "\t\t\t\t<div class=\"gallerystats\">" << std::endl;
      ofs << "\t\t\t\t<div class=\"gallerydescription\">" << std::endl;
      if (g->get_description() != "")
        ofs << "\t\t\t\t\t" << 
          Glib::Markup::escape_text(g->get_description()) << std::endl;
      ofs << "\t\t\t\t</div>" << std::endl;
      if (g->get_latitude() != 0.0 && g->get_longitude() != 0.0)
        {
          ofs << "\t\t\t\t<div class=\"galleryposition\">" << std::endl;
          ofs << "\t\t\t\t\tLatitude: " << g->get_latitude() << std::endl;
          ofs << "\t\t\t\t\tLongitude: " << g->get_longitude() << std::endl;
          ofs << "\t\t\t\t</div>" << std::endl;
        }
      std::list<std::string> tags = g->get_tags();
      if (tags.size())
        {
          ofs << "\t\t\t\t<div class=\"gallerytags\">" << std::endl;
          ofs << "\t\t\t\t\tTags:";
          for (std::list<std::string>::iterator t = tags.begin(); 
               t != tags.end(); t++)
            ofs << " " << Glib::Markup::escape_text(*t);
          ofs << std::endl;
          ofs << "\t\t\t\t</div>" << std::endl;
        }
      ofs << "\t\t\t\t</div>" << std::endl;
      ofs << "\t\t</div>" << std::endl;
    }
  ofs << "\t</body>\n</html>" << std::endl;
  ofs.close();
  for (const_iterator j = get_galleries().begin(); 
       j != get_galleries().end(); j++)
    (*j)->export_html_page(export_dir);
}

bool GalleryList::is_same(const GalleryList &g) const
{
  if (size() != g.size())
    return false;
  for (const_iterator i = get_galleries().begin();
       i != get_galleries().end(); i++)
    {
      Gallery *gallery = g.find_by_id((*i)->get_id());
      if (!gallery)
        return false;
      if (gallery->size() != (*i)->size())
        return false;
      if (*gallery == *(*i))
        ;
      else
        return false;
      for (Gallery::const_iterator j = (*i)->get_docs().begin(); 
           j != (*i)->get_docs().end(); j++)
        {
          Document *document = gallery->find_by_id((*j)->get_id());
          if (document == NULL)
            return false;
          if (*document == *(*j))
            ;
          else
            return false;
        }
    }
  return true;
}
        
std::list<std::string> GalleryList::get_tags() const
{
  std::list<std::string> tags;
  for (const_iterator i = get_galleries().begin();
       i != get_galleries().end(); i++)
    {
      std::list<std::string> t = (*i)->get_tags();
      tags.merge(t);
      for (Gallery::const_iterator j = (*i)->get_docs().begin(); 
           j != (*i)->get_docs().end(); j++)
        {
          std::list<std::string> tt = (*j)->get_tags();
          tags.merge(tt);
        }
    }
  tags.sort();
  tags.unique();
  return tags;
}
