//  Copyright (C) 2010, 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>
#include <gtkmm.h>
#include <glibmm.h>
#include "main.h"

#include "changes-editor-dialog.h"
#include "ucompose.hpp"
#include "document-dialog.h"
#include "gallery-dialog.h"

ChangesEditorDialog * ChangesEditorDialog::create(Tapioca::Session *s)
{
  Glib::RefPtr<Gtk::Builder> xml
    = Gtk::Builder::create_from_file(Main::get_glade_path()
				    + "/changes-editor-dialog.gtk");

  ChangesEditorDialog *dialog;
  xml->get_widget_derived("dialog", dialog);
  dialog->on_dialog_loaded(s);
  return dialog;
}

void ChangesEditorDialog::on_dialog_loaded(Tapioca::Session *s)
{
  session = s;
  changes = session->get_changes();
  fill_modified_galleries(changes.modified_galleries);
  fill_modified_documents(changes.modified_docs);
  fill_removed_galleries(changes.removed_galleries);
  fill_added_galleries(changes.added_galleries);
  fill_removed_documents(changes.removed_documents);
  fill_added_documents(changes.added_documents);
  update_buttons();
}

ChangesEditorDialog::ChangesEditorDialog(BaseObjectType* baseObject, 
                                         const Glib::RefPtr<Gtk::Builder>& xml)
 : Gtk::Dialog(baseObject)
{
  set_icon_from_file(Main::get_data_path() + "/tapioca.png");
  xml->get_widget("modified_galleries_expander", modified_galleries_expander);
  xml->get_widget("modified_galleries_treeview", modified_galleries_treeview);
  xml->get_widget("modified_galleries_label", modified_galleries_label);
  xml->get_widget("modified_documents_expander", modified_documents_expander);
  xml->get_widget("modified_documents_treeview", modified_documents_treeview);
  xml->get_widget("modified_documents_label", modified_documents_label);
  xml->get_widget("removed_galleries_expander", removed_galleries_expander);
  xml->get_widget("removed_galleries_treeview", removed_galleries_treeview);
  xml->get_widget("removed_galleries_label", removed_galleries_label);
  xml->get_widget("added_galleries_expander", added_galleries_expander);
  xml->get_widget("added_galleries_treeview", added_galleries_treeview);
  xml->get_widget("added_galleries_label", added_galleries_label);
  xml->get_widget("removed_documents_expander", removed_documents_expander);
  xml->get_widget("removed_documents_treeview", removed_documents_treeview);
  xml->get_widget("removed_documents_label", removed_documents_label);
  xml->get_widget("added_documents_expander", added_documents_expander);
  xml->get_widget("added_documents_treeview", added_documents_treeview);
  xml->get_widget("added_documents_label", added_documents_label);
  xml->get_widget("accept_button", accept_button);

  modified_galleries_treeview->signal_row_activated().connect
    (sigc::mem_fun(*this, &ChangesEditorDialog::on_modified_gallery_activated));
  modified_documents_treeview->signal_row_activated().connect
    (sigc::mem_fun(*this, &ChangesEditorDialog::on_modified_document_activated));
  removed_galleries_treeview->signal_row_activated().connect
    (sigc::mem_fun(*this, &ChangesEditorDialog::on_removed_gallery_activated));
  added_galleries_treeview->signal_row_activated().connect
    (sigc::mem_fun(*this, &ChangesEditorDialog::on_added_gallery_activated));
  removed_documents_treeview->signal_row_activated().connect
    (sigc::mem_fun(*this, &ChangesEditorDialog::on_removed_document_activated));
  added_documents_treeview->signal_row_activated().connect
    (sigc::mem_fun(*this, &ChangesEditorDialog::on_added_document_activated));

  modified_galleries_treeview->signal_button_press_event().connect_notify
    (sigc::bind<Gtk::Menu&>(sigc::mem_fun(*this, &ChangesEditorDialog::on_treeview_clicked), modified_galleries_context_menu));
  modified_documents_treeview->signal_button_press_event().connect_notify
    (sigc::bind<Gtk::Menu&>(sigc::mem_fun(*this, &ChangesEditorDialog::on_treeview_clicked), modified_documents_context_menu));
  removed_galleries_treeview->signal_button_press_event().connect_notify
    (sigc::bind<Gtk::Menu&>(sigc::mem_fun(*this, &ChangesEditorDialog::on_treeview_clicked), removed_galleries_context_menu));
  added_galleries_treeview->signal_button_press_event().connect_notify
    (sigc::bind<Gtk::Menu&>(sigc::mem_fun(*this, &ChangesEditorDialog::on_treeview_clicked), added_galleries_context_menu));
  removed_documents_treeview->signal_button_press_event().connect_notify
    (sigc::bind<Gtk::Menu&>(sigc::mem_fun(*this, &ChangesEditorDialog::on_treeview_clicked), removed_documents_context_menu));
  added_documents_treeview->signal_button_press_event().connect_notify
    (sigc::bind<Gtk::Menu&>(sigc::mem_fun(*this, &ChangesEditorDialog::on_treeview_clicked), added_documents_context_menu));
    
  modified_galleries_list = Gtk::ListStore::create(modified_galleries_columns);
  modified_galleries_treeview->set_model(modified_galleries_list);
  modified_galleries_treeview->get_selection()->set_mode(Gtk::SELECTION_SINGLE);
  modified_galleries_treeview->append_column("", modified_galleries_columns.image);
  modified_galleries_treeview->append_column("Nature of Change", modified_galleries_columns.desc);
  Gtk::Menu::MenuList& menulist = 
    modified_galleries_context_menu.items();
  menulist.push_back
      (Gtk::Menu_Helpers::MenuElem
       ("_Revert", 
        sigc::mem_fun(*this, &ChangesEditorDialog::on_revert_modified_gallery)));
  modified_galleries_context_menu.accelerate(*this);

  modified_documents_list = Gtk::ListStore::create(modified_documents_columns);
  modified_documents_treeview->set_model(modified_documents_list);
  modified_documents_treeview->get_selection()->set_mode(Gtk::SELECTION_SINGLE);
  modified_documents_treeview->append_column("", modified_documents_columns.image);
  modified_documents_treeview->append_column("Nature of Change", modified_documents_columns.desc);
  menulist = modified_documents_context_menu.items();
  menulist.push_back
      (Gtk::Menu_Helpers::MenuElem
       ("_Revert", 
        sigc::mem_fun(*this, &ChangesEditorDialog::on_revert_modified_document)));
  modified_documents_context_menu.accelerate(*this);

  removed_galleries_list = Gtk::ListStore::create(removed_galleries_columns);
  removed_galleries_treeview->set_model(removed_galleries_list);
  removed_galleries_treeview->get_selection()->set_mode(Gtk::SELECTION_SINGLE);
  removed_galleries_treeview->append_column("", 
                                            removed_galleries_columns.image);
  removed_galleries_treeview->append_column("Name", 
                                            removed_galleries_columns.desc);
  menulist = 
    removed_galleries_context_menu.items();
  menulist.push_back
      (Gtk::Menu_Helpers::MenuElem
       ("_Revert", 
        sigc::mem_fun(*this, &ChangesEditorDialog::on_revert_removed_gallery)));
  removed_galleries_context_menu.accelerate(*this);

  added_galleries_list = Gtk::ListStore::create(added_galleries_columns);
  added_galleries_treeview->set_model(added_galleries_list);
  added_galleries_treeview->get_selection()->set_mode(Gtk::SELECTION_SINGLE);
  added_galleries_treeview->append_column("", added_galleries_columns.image);
  added_galleries_treeview->append_column("Name", added_galleries_columns.desc);
  menulist = added_galleries_context_menu.items();
  menulist.push_back
      (Gtk::Menu_Helpers::MenuElem
       ("_Revert", 
        sigc::mem_fun(*this, &ChangesEditorDialog::on_revert_added_gallery)));
  added_galleries_context_menu.accelerate(*this);

  removed_documents_list = Gtk::ListStore::create(removed_documents_columns);
  removed_documents_treeview->set_model(removed_documents_list);
  removed_documents_treeview->get_selection()->set_mode(Gtk::SELECTION_SINGLE);
  removed_documents_treeview->append_column("", removed_documents_columns.image);
  removed_documents_treeview->append_column("Name", removed_documents_columns.desc);
  menulist = removed_documents_context_menu.items();
  menulist.push_back
      (Gtk::Menu_Helpers::MenuElem
       ("_Revert", 
        sigc::mem_fun(*this, &ChangesEditorDialog::on_revert_removed_document)));
  removed_documents_context_menu.accelerate(*this);

  added_documents_list = Gtk::ListStore::create(added_documents_columns);
  added_documents_treeview->set_model(added_documents_list);
  added_documents_treeview->get_selection()->set_mode(Gtk::SELECTION_SINGLE);
  added_documents_treeview->append_column("", added_documents_columns.image);
  added_documents_treeview->append_column("Name", added_documents_columns.desc);
  menulist = added_documents_context_menu.items();
  menulist.push_back
      (Gtk::Menu_Helpers::MenuElem
       ("_Revert", 
        sigc::mem_fun(*this, &ChangesEditorDialog::on_revert_added_document)));
  added_documents_context_menu.accelerate(*this);
}

void ChangesEditorDialog::update_buttons()
{
  if (changes.modified_galleries.size() == 0 &&
      changes.modified_docs.size() == 0 &&
      changes.removed_galleries.size() == 0 &&
      changes.added_galleries.size() == 0 &&
      changes.removed_documents.size() == 0 &&
      changes.added_documents.size() == 0 &&
      reverts.empty() == true)
    accept_button->set_sensitive(false);
}

void ChangesEditorDialog::fill_modified_galleries(std::list<Tapioca::ChangedGallery> &l)
{
  modified_galleries_list->clear();
  for (std::list<Tapioca::ChangedGallery>::iterator i = l.begin(); 
       i != l.end(); i++)
    add_row_modified_gallery(&(*i));
  size_t num = modified_galleries_list->children().size();
  modified_galleries_label->set_text 
    (String::ucompose("Modified Galleries (%1)", num));
  modified_galleries_expander->set_expanded(num != 0);
  modified_galleries_expander->set_sensitive(num != 0);
}

void ChangesEditorDialog::fill_modified_documents(std::list<Tapioca::ChangedDocument> &l)
{
  modified_documents_list->clear();
  for (std::list<Tapioca::ChangedDocument>::iterator i = l.begin(); 
       i != l.end(); i++)
    add_row_modified_document(&(*i));
  size_t num = modified_documents_list->children().size();
  modified_documents_label->set_text 
    (String::ucompose("Modified Documents (%1)", num));
  modified_documents_expander->set_expanded(num != 0);
  modified_documents_expander->set_sensitive(num != 0);
}

void ChangesEditorDialog::fill_removed_galleries(std::list<Tapioca::ChangedGallery> &l)
{
  removed_galleries_list->clear();
  for (std::list<Tapioca::ChangedGallery>::iterator i = l.begin(); 
       i != l.end(); i++)
    add_row_removed_gallery(&(*i));
  size_t num = removed_galleries_list->children().size();
  removed_galleries_label->set_text 
    (String::ucompose("Removed Galleries (%1)", num));
  removed_galleries_expander->set_expanded(num != 0);
  removed_galleries_expander->set_sensitive(num != 0);
}

void ChangesEditorDialog::fill_added_galleries(std::list<Tapioca::ChangedGallery> &l)
{
  added_galleries_list->clear();
  for (std::list<Tapioca::ChangedGallery>::iterator i = l.begin(); 
       i != l.end(); i++)
    add_row_added_gallery(&(*i));
  size_t num = added_galleries_list->children().size();
  added_galleries_label->set_text 
    (String::ucompose("Added Galleries (%1)", num));
  added_galleries_expander->set_expanded(num != 0);
  added_galleries_expander->set_sensitive(num != 0);
}

void ChangesEditorDialog::fill_removed_documents(std::list<Tapioca::ChangedDocument> &l)
{
  removed_documents_list->clear();
  for (std::list<Tapioca::ChangedDocument>::iterator i = l.begin(); 
       i != l.end(); i++)
    {
      //only add it if we're not removing the gallery.
      //Gallery *parent = session->find_by_id((*i).first.get_gallery_id());
      //if (parent)
        add_row_removed_document(&(*i));
    }
  size_t num = removed_documents_list->children().size();
  removed_documents_expander->set_expanded(num != 0);
  removed_documents_expander->set_sensitive(num != 0);
  removed_documents_label->set_text 
    (String::ucompose("Removed Documents (%1)", num));
}

void ChangesEditorDialog::fill_added_documents(std::list<Tapioca::ChangedDocument> &l)
{
  added_documents_list->clear();
  for (std::list<Tapioca::ChangedDocument>::iterator i = l.begin(); 
       i != l.end(); i++)
    add_row_added_document(&(*i));
  size_t num = added_documents_list->children().size();
  added_documents_label->set_text 
    (String::ucompose("Added Documents (%1)", num));
  added_documents_expander->set_expanded(num != 0);
  added_documents_expander->set_sensitive(num != 0);
}

void ChangesEditorDialog::add_row_modified_gallery(Tapioca::ChangedGallery *gallery)
{
  Gtk::TreeIter i = modified_galleries_list->append();
  (*i)[modified_galleries_columns.image] = 
    gallery->first.get_thumbnail_image();
  (*i)[modified_galleries_columns.desc] = gallery->get_change_description();
  (*i)[modified_galleries_columns.data] = gallery;
}

void ChangesEditorDialog::add_row_modified_document(Tapioca::ChangedDocument *document)
{
  Gtk::TreeIter i = modified_documents_list->append();
  (*i)[modified_documents_columns.image] = document->first.get_thumbnail();
  (*i)[modified_documents_columns.desc] = document->get_change_description();
  (*i)[modified_documents_columns.data] = document;
}

void ChangesEditorDialog::add_row_removed_gallery(Tapioca::ChangedGallery *gallery)
{
  Gtk::TreeIter i = removed_galleries_list->append();
  Tapioca::Gallery *g = &gallery->first;
  Tapioca::Gallery *orig_g = session->get_profile()->find_by_id(g->get_id());
  if (orig_g)
    (*i)[removed_galleries_columns.image] = orig_g->get_thumbnail_image();
  else
    (*i)[removed_galleries_columns.image] = g->get_thumbnail_image();
  (*i)[removed_galleries_columns.desc] = g->get_title();
  (*i)[removed_galleries_columns.data] = gallery;
}

void ChangesEditorDialog::add_row_added_gallery(Tapioca::ChangedGallery *gallery)
{
  Gtk::TreeIter i = added_galleries_list->append();
  Tapioca::Gallery *g = session->find_by_id(gallery->first.get_id());
  (*i)[added_galleries_columns.image] = g->get_thumbnail_image();
  (*i)[added_galleries_columns.desc] = g->get_title();
  (*i)[added_galleries_columns.data] = gallery;
}

void ChangesEditorDialog::add_row_removed_document(Tapioca::ChangedDocument *document)
{
  Gtk::TreeIter i = removed_documents_list->append();
  Tapioca::Document *d = &document->first;
  (*i)[removed_documents_columns.image] = d->get_thumbnail();
  (*i)[removed_documents_columns.desc] = d->get_title();
  (*i)[removed_documents_columns.data] = document;
}

void ChangesEditorDialog::add_row_added_document(Tapioca::ChangedDocument *document)
{
  Gtk::TreeIter i = added_documents_list->append();
  Tapioca::Document *d = &document->first;
  (*i)[added_documents_columns.image] = d->get_thumbnail();
  (*i)[added_documents_columns.desc] = d->get_title();
  (*i)[added_documents_columns.data] = document;
}

ChangesEditorDialog::~ChangesEditorDialog()
{
}

void ChangesEditorDialog::on_modified_gallery_activated(const Gtk::TreeModel::Path& path,
                                                  Gtk::TreeViewColumn *col)
{
  Gallery *gallery = new Gallery(get_selected_modified_gallery()->first);
  GalleryDialog *d = GalleryDialog::create
    (gallery, &get_selected_modified_gallery()->second, 
     session->get_tags());
  d->set_transient_for(*this);
  int response = d->run();
  d->hide();
  if (response == Gtk::RESPONSE_ACCEPT)
    {
      Gallery *orig_gallery = &get_selected_modified_gallery()->first;
      if (*orig_gallery == *gallery)
        ;
      else
        {
          *orig_gallery = *gallery;
          update_selected_modified_gallery_changes();
        }
    }
  delete d;
}

void ChangesEditorDialog::on_modified_document_activated(const Gtk::TreeModel::Path& path,
                                                  Gtk::TreeViewColumn *col)
{
  Document *document = new Document(get_selected_modified_document()->first);
  DocumentDialog *d = DocumentDialog::create
    (document, &get_selected_modified_document()->second,
     session->get_tags());
  d->view_document.connect(sigc::mem_fun(view_document, &sigc::signal<void, Tapioca::Document*>::emit));
  d->set_transient_for(*this);
  int response = d->run();
  d->hide();
  if (response == Gtk::RESPONSE_ACCEPT)
    {
      Document *orig_document = &get_selected_modified_document()->first;
      if (*orig_document == *document)
        ;
      else
        {
          *orig_document = *document;
          update_selected_modified_document_changes();
        }
    }
  delete d;
}

void ChangesEditorDialog::on_removed_gallery_activated(const Gtk::TreeModel::Path& path,
                                                  Gtk::TreeViewColumn *col)
{
  ChangedGallery * gallery= get_selected_removed_gallery();
  GalleryDialog *d = GalleryDialog::create (&gallery->first, &gallery->second, 
                                            session->get_tags(), true);
  d->set_transient_for(*this);
  d->run();
  delete d;
}

void ChangesEditorDialog::on_added_gallery_activated(const Gtk::TreeModel::Path& path,
                                                  Gtk::TreeViewColumn *col)
{
  Gallery *gallery = new Gallery(get_selected_added_gallery()->first);
  GalleryDialog* d = GalleryDialog::create
    (gallery, &get_selected_added_gallery()->second, session->get_tags());
  d->set_transient_for(*this);
  int response = d->run();
  d->hide();
  if (response == Gtk::RESPONSE_ACCEPT)
    {
      Gallery *orig_gallery = &get_selected_added_gallery()->first;
      if (*orig_gallery == *gallery)
        ;
      else
        {
          *orig_gallery = *gallery;
          update_selected_added_gallery_name();
        }
    }
  delete d;
}

void ChangesEditorDialog::on_removed_document_activated(const Gtk::TreeModel::Path& path,
                                                  Gtk::TreeViewColumn *col)
{
  ChangedDocument* document = get_selected_removed_document();
  DocumentDialog *d = 
    DocumentDialog::create(&document->first, &document->second, 
                           session->get_tags(), true);
  d->view_document.connect(sigc::mem_fun(view_document, &sigc::signal<void, Tapioca::Document*>::emit));
  d->set_transient_for(*this);
  d->run();
  delete d;
}

void ChangesEditorDialog::on_added_document_activated(const Gtk::TreeModel::Path& path,
                                                  Gtk::TreeViewColumn *col)
{
  Document *document = new Document(get_selected_added_document()->first);
  DocumentDialog *d = DocumentDialog::create
    (document, &get_selected_added_document()->second,
     session->get_tags());
  d->view_document.connect(sigc::mem_fun(view_document, &sigc::signal<void, Tapioca::Document*>::emit));
  d->set_transient_for(*this);
  int response = d->run();
  d->hide();
  if (response == Gtk::RESPONSE_ACCEPT)
    {
      Document *orig_document = &get_selected_added_document()->first;
      if (*orig_document == *document)
        ;
      else
        {
          *orig_document = *document;
          update_selected_added_document_name();
        }
    }
  delete d;
}

void ChangesEditorDialog::update_selected_added_document_name()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = 
    added_documents_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      ChangedDocument *changed_document = row[added_documents_columns.data];
      Document *document = &changed_document->first;
      row[added_documents_columns.desc] = document->get_title();
    }
}
    
void ChangesEditorDialog::update_selected_modified_document_changes()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = 
    modified_documents_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      ChangedDocument *changed_document = row[modified_documents_columns.data];
      row[added_documents_columns.desc] = 
        changed_document->get_change_description();
    }
}

void ChangesEditorDialog::update_selected_modified_gallery_changes()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = 
    modified_galleries_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      ChangedGallery *changed_gallery = row[modified_galleries_columns.data];
      row[added_galleries_columns.desc] = 
        changed_gallery->get_change_description();
    }
}

void ChangesEditorDialog::update_selected_added_gallery_name()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = 
    added_galleries_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      ChangedGallery *changed_gallery = row[added_galleries_columns.data];
      Gallery *gallery = &changed_gallery->first;
      row[added_galleries_columns.desc] = gallery->get_title();
    }
}
    
Tapioca::ChangedGallery* ChangesEditorDialog::get_selected_modified_gallery() const
{
  Glib::RefPtr<Gtk::TreeSelection> selection = 
    modified_galleries_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Tapioca::ChangedGallery *gallery = row[modified_galleries_columns.data];
      return gallery;
    }
  return NULL;
}

Tapioca::ChangedDocument* ChangesEditorDialog::get_selected_modified_document() const
{
  Glib::RefPtr<Gtk::TreeSelection> selection = 
    modified_documents_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Tapioca::ChangedDocument *document = row[modified_documents_columns.data];
      return document;
    }
  return NULL;
}

Tapioca::ChangedDocument* ChangesEditorDialog::get_selected_removed_document() const
{
  Glib::RefPtr<Gtk::TreeSelection> selection = 
    removed_documents_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Tapioca::ChangedDocument *document = row[removed_documents_columns.data];
      return document;
    }
  return NULL;
}

Tapioca::ChangedGallery * ChangesEditorDialog::get_selected_added_gallery() const
{
  Glib::RefPtr<Gtk::TreeSelection> selection = 
    added_galleries_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Tapioca::ChangedGallery *gallery = row[added_galleries_columns.data];
      return gallery;
    }
  return NULL;
}

Tapioca::ChangedGallery * ChangesEditorDialog::get_selected_removed_gallery() const
{
  Glib::RefPtr<Gtk::TreeSelection> selection = 
    removed_galleries_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Tapioca::ChangedGallery *gallery = row[removed_galleries_columns.data];
      return gallery;
    }
  return NULL;
}

Tapioca::ChangedDocument* ChangesEditorDialog::get_selected_added_document() const
{
  Glib::RefPtr<Gtk::TreeSelection> selection = 
    added_documents_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Tapioca::ChangedDocument *document = row[added_documents_columns.data];
      return document;
    }
  return NULL;
}

void ChangesEditorDialog::on_treeview_clicked(GdkEventButton *event, Gtk::Menu &context_menu)
{
  if (event->type == GDK_BUTTON_PRESS &&event->button == 3)
    context_menu.popup(event->button, event->time);
}

void ChangesEditorDialog::on_revert_modified_gallery()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = modified_galleries_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      ChangedGallery *gallery = row[modified_galleries_columns.data];
      reverts.modified_galleries.push_back(*gallery);
      changes.modified_galleries.remove(*gallery);
      modified_galleries_list->erase(iterrow);
      fill_modified_galleries(changes.modified_galleries);
      update_buttons();
    }
}

void ChangesEditorDialog::on_revert_modified_document()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = 
    modified_documents_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      ChangedDocument *document = row[modified_documents_columns.data];
      reverts.modified_docs.push_back(*document);
      changes.modified_docs.remove(*document);
      modified_documents_list->erase(iterrow);
      fill_modified_documents(changes.modified_docs);
      update_buttons();
    }
}

void ChangesEditorDialog::remove_removed_parent_gallery(Tapioca::ChangedDocument *document)
{
  if (session->find_by_id(document->first.get_gallery_id()) != NULL)
    return;

  for (std::list<ChangedGallery>::iterator i = 
       changes.removed_galleries.begin(); i != changes.removed_galleries.end();
       i++)
    {
      if ((*i).first.get_id() == document->first.get_gallery_id())
        {
          reverts.removed_galleries.push_back(*i);
          changes.removed_galleries.remove(*i);
          fill_removed_galleries(changes.removed_galleries);
          return;
        }
    }
}

void ChangesEditorDialog::on_revert_removed_document()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = removed_documents_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      ChangedDocument *document = row[removed_documents_columns.data];
      remove_same_added_document(document);
      remove_removed_parent_gallery(document);
      reverts.removed_documents.push_back(*document);
      changes.removed_documents.remove(*document);
      removed_documents_list->erase(iterrow);
      fill_removed_documents(changes.removed_documents);
      update_buttons();
    }
}

void ChangesEditorDialog::on_revert_added_gallery()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = added_galleries_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      ChangedGallery *gallery = row[added_galleries_columns.data];
      reverts.added_galleries.push_back(*gallery);
      remove_documents_belonging_to_reverted_added_gallery(gallery);
      changes.added_galleries.remove(*gallery);
      added_galleries_list->erase(iterrow);
      fill_added_galleries(changes.added_galleries);
      update_buttons();
    }
}

void ChangesEditorDialog::on_revert_removed_gallery()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = removed_galleries_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      ChangedGallery *gallery = row[removed_galleries_columns.data];
      reverts.removed_galleries.push_back(*gallery);
      remove_documents_belonging_to_reverted_removed_gallery(gallery);
      changes.removed_galleries.remove(*gallery);
      removed_galleries_list->erase(iterrow);
      fill_removed_galleries(changes.removed_galleries);
      update_buttons();
    }
}

void ChangesEditorDialog::on_revert_added_document()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = added_documents_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      ChangedDocument *document = row[added_documents_columns.data];
      remove_same_removed_document(document);
      reverts.added_documents.push_back(*document);
      changes.added_documents.remove(*document);
      added_documents_list->erase(iterrow);
      fill_added_documents(changes.added_documents);
      update_buttons();
    }
}

void ChangesEditorDialog::remove_same_removed_document(Tapioca::ChangedDocument *document)
{
  //the idea here is that some docs were moved between galleries.
  //this event shows up as two separate add, and remove changes.
  //so we need to remove the corresponding event, when we're dealing with a 
  //moved document.
  std::list<ChangedDocument>::iterator it = changes.removed_documents.begin();
  for (; it != changes.removed_documents.end(); it++)
    {
      if ((*it).first.get_id() == document->first.get_id())
        {
          reverts.removed_documents.push_back(*it);
          changes.removed_documents.erase(it);
          fill_removed_documents(changes.removed_documents);
          update_buttons();
          break;
        }
    }
}

void ChangesEditorDialog::remove_same_added_document(Tapioca::ChangedDocument *document)
{
  std::list<ChangedDocument>::iterator it = changes.added_documents.begin();
  for (; it != changes.added_documents.end(); it++)
    {
      if ((*it).first.get_id() == document->first.get_id())
        {
          reverts.added_documents.push_back(*it);
          changes.added_documents.erase(it);
          fill_added_documents(changes.added_documents);
          update_buttons();
          break;
        }
    }
}

void ChangesEditorDialog::remove_documents_belonging_to_reverted_removed_gallery(Tapioca::ChangedGallery *changed_gallery)
{
  //okay, this gallery is coming back.  so should the docs belonging to it.
  //this means that the gallery and associated docs are removed from the 
  //lists of removed galleries and documents
  Gallery *gallery = session->get_profile()->find_by_id(changed_gallery->first.get_id());
  std::list<ChangedDocument> docs_to_remove;
  for (std::list<ChangedDocument>::iterator i = 
       changes.removed_documents.begin(); 
       i != changes.removed_documents.end(); i++)
    {
      if ((*i).first.get_gallery_id() == gallery->get_id() &&
          session->find_doc_by_id((*i).first.get_id()) == NULL) //was it moved?
        docs_to_remove.push_back(*i);
    }

  for (std::list<ChangedDocument>::iterator i = docs_to_remove.begin();
       i != docs_to_remove.end(); i++)
    {
      reverts.removed_documents.push_back(*i);
      changes.removed_documents.remove(*i);
    }
  if (gallery->size())
    {
      fill_removed_documents(changes.removed_documents);
      update_buttons();
    }
}

void ChangesEditorDialog::remove_documents_belonging_to_reverted_added_gallery(Tapioca::ChangedGallery *changed_gallery)
{
  //okay, this gallery is going away.  so should the docs belonging to it.
  Gallery *gallery = session->find_by_id(changed_gallery->first.get_id());
  std::list<ChangedDocument> docs_to_remove;
  for (std::list<ChangedDocument>::iterator i = changes.added_documents.begin();
       i != changes.added_documents.end(); i++)
    {
      if ((*i).first.get_gallery_id() == gallery->get_id())
        docs_to_remove.push_back(*i);
    }

  for (std::list<ChangedDocument>::iterator i = docs_to_remove.begin();
       i != docs_to_remove.end(); i++)
    {
      remove_same_removed_document(&*i);
      reverts.added_documents.push_back(*i);
      changes.added_documents.remove(*i);
    }
  if (gallery->size())
    {
      fill_added_documents(changes.added_documents);
      update_buttons();
    }
}
/*
test cases:
reverting a modified gallery
reverting a modified document
reverting an added gallery
reverting an added document
reverting an added gallery with a new document in it.
reverting a removed document
reverting a removed gallery with an existing document in it.
reverting a moved document (when reverting removed document).
reverting a moved document (when reverting added document).
reverting a new gallery when it has a moved document in it.
reverting a removed gallery when it has a document moved out of it.
reverting a removed document that had a parent gallery also removed
*/
