//  Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <gtkmm.h>

#include "item-tagging-widget.h"
#include "main.h"
#include "ucompose.hpp"
#include <cstring>

std::string ItemTaggingWidget::default_tag_name = "type your tag";
ItemTaggingWidget::ItemTaggingWidget()
    : name_column("Name", name_renderer)
{
  inhibit = false;
  Glib::RefPtr<Gtk::Builder> xml
    = Gtk::Builder::create_from_file(Main::get_glade_path()
                                     + "/item-tagging-dialog.gtk");


  Gtk::Dialog *d;
  xml->get_widget("dialog", d);
  xml->get_widget("contents", contents);
  xml->get_widget("tags_count_label", tags_count_label);
  xml->get_widget("treeview", treeview);
  tags_list = Gtk::ListStore::create(tags_columns);
  treeview->set_model(tags_list);
  treeview->get_selection()->set_mode(Gtk::SELECTION_SINGLE);
  treeview->get_selection()->signal_changed().connect
    (sigc::mem_fun(*this, &ItemTaggingWidget::on_tag_selected));
  name_renderer.property_editable() = true;
  name_renderer.signal_edited().connect
    (sigc::mem_fun(*this, &ItemTaggingWidget::on_tag_edited));
  name_renderer.signal_editing_canceled().connect
    (sigc::mem_fun(*this, &ItemTaggingWidget::on_cancel_tag_edit));
  name_column.set_cell_data_func
    (name_renderer, sigc::mem_fun(*this, &ItemTaggingWidget::cell_data_tag));
  treeview->append_column(name_column);
  xml->get_widget("add_button", add_button);
  add_button->signal_clicked().connect
    (sigc::mem_fun(this,&ItemTaggingWidget::on_add_clicked));
  xml->get_widget("remove_button", remove_button);
  remove_button->signal_clicked().connect
    (sigc::mem_fun(this,&ItemTaggingWidget::on_remove_clicked));
}

ItemTaggingWidget::~ItemTaggingWidget()
{
}

void ItemTaggingWidget::fill_tags(std::list<std::string> t)
{
  tags = t;
  for (std::list<std::string>::iterator it = tags.begin(); 
       it != tags.end(); it++)
    add_tag_row(*it);
  tags_count_label->set_text(String::ucompose("%1", tags.size()));
}

void ItemTaggingWidget::add_tag_row(std::string tag)
{
  Gtk::TreeIter i = tags_list->append();
  (*i)[tags_columns.name] = tag;
}

std::string ItemTaggingWidget::get_selected_tag()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Glib::ustring tag = row[tags_columns.name];
      return tag;
    }
  return "";
}

bool ItemTaggingWidget::has_tag(std::string tag) const
{
  return (std::find(tags.begin(), tags.end(), tag) != tags.end());
}

void ItemTaggingWidget::on_tag_selected()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();

  Gtk::TreeModel::iterator last = iterrow;
  for (Gtk::TreeModel::iterator it = tags_list->children().begin();
       it != tags_list->children().end(); it++)
    {
      last = it;
    }

  if ((*iterrow)[tags_columns.name] == (*last)[tags_columns.name])
    return;

  Glib::ustring tag = (*last)[tags_columns.name];
  if (validate_tag(tag, true) == false)
    {
      treeview->get_selection()->select(last);
      on_remove_clicked();
    }
}

void ItemTaggingWidget::on_add_clicked()
{
  inhibit = true;
  std::string tag = default_tag_name;
  if (get_selected_tag() != default_tag_name)
    {
      Gtk::TreeIter i = tags_list->append();
      (*i)[tags_columns.name] = tag;
      Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
      selection->select(i);
    }
  Gtk::TreeViewColumn *column = NULL;
  Gtk::TreeModel::Path path;
  treeview->get_cursor(path, column);
  path = Gtk::TreeModel::Path(String::ucompose("%1", tags_list->children().size()-1));
  treeview->set_cursor(path, name_column, true);
  tags_count_label->set_text(String::ucompose("%1", tags.size()));
  treeview->set_cursor(path, name_column, true);
  inhibit = false;
}

void ItemTaggingWidget::on_remove_clicked()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();

  if (iterrow) 
    {
      Gtk::TreeModel::Row row = *iterrow;
      Glib::ustring tag = row[tags_columns.name];
      if (std::find(tags.begin(), tags.end(), tag) != tags.end())
        tags.remove(tag);
      tags_list->erase(iterrow);
      tags_count_label->set_text(String::ucompose("%1", tags.size()));
    }
}

void
ItemTaggingWidget::cell_data_tag(Gtk::CellRenderer *renderer, const Gtk::TreeIter &i)
{
  Gtk::CellRendererText*cell = dynamic_cast<Gtk::CellRendererText*>(renderer);
  cell->property_text() = (*i)[tags_columns.name];
}

void
ItemTaggingWidget::on_tag_edited(const Glib::ustring &path,
                             const Glib::ustring &new_text)
{
  Gtk::TreeModel::iterator iterrow = tags_list->get_iter(Gtk::TreePath(path));
  Gtk::TreeModel::Row row = *iterrow;
  Glib::ustring old_tag = row[tags_columns.name];
  row[tags_columns.name] = new_text;
  if (validate_tag(new_text) == true)
    update_tag(old_tag, new_text);
  else
    {
      Gtk::TreeViewColumn *column = NULL;
      Gtk::TreeModel::Path path;
      treeview->get_cursor(path, column);
      path = Gtk::TreeModel::Path(String::ucompose("%1", tags_list->children().size()-1));
      treeview->get_selection()->select(path);
      on_remove_clicked();
    }
  tags_count_label->set_text(String::ucompose("%1", tags.size()));
}

bool ItemTaggingWidget::validate_tag(std::string tag, bool show_error)
{
  std::string msg;
  if (tag == "")
    msg = "Tags cannot be empty.";
  else if (tag.find(' ') != std::string::npos)
    msg = "Tags cannot have spaces.";
  else if (strspn (tag.c_str(), "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0124356789~") != tag.length())
    msg = "Tags must be alphanumeric.";
  else
    msg = "";

  return msg == "";
}

void ItemTaggingWidget::on_cancel_tag_edit()
{
  if (inhibit)
    return;
  inhibit = false;
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();

  if (iterrow) 
    {
      Gtk::TreeModel::Row row = *iterrow;
      Glib::ustring tag = row[tags_columns.name];
      if (tag == default_tag_name || validate_tag(tag, false) == false)
        on_remove_clicked();
    }
}

void ItemTaggingWidget::add_tag(std::string tag)
{
  if (std::find(tags.begin(), tags.end(), tag) == tags.end())
    tags.push_back(tag);
}

void ItemTaggingWidget::update_tag(std::string oldvalue, std::string newvalue)
{
  std::list<std::string>::iterator i = 
    std::find(tags.begin(), tags.end(), oldvalue);
  if (i != tags.end())
    (*i) = newvalue;
  else
    tags.push_back(newvalue);
}

void ItemTaggingWidget::remove_tag(std::string tag)
{
  if (std::find(tags.begin(), tags.end(), tag) != tags.end())
    tags.remove(tag);
}

std::list<std::string> ItemTaggingWidget::get_tags()
{
  remove_tag(default_tag_name);
  return tags;
}

