// Copyright (C) 2010, 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>
#include "document.h"

#include <fstream>
#include <cstdio>
#include "main.h"
#include "ucompose.hpp"
using namespace Tapioca;

Document::Document(const Document& d)
  :Item(d), thumbnail(d.thumbnail), thumbnail_filename(d.thumbnail_filename), thumbnail_file_data(d.thumbnail_file_data), gallery_id(d.gallery_id), image_filename(d.image_filename), original_filename(d.original_filename)
{
}

Document::Document()
: Item(), thumbnail(0), thumbnail_filename(""), gallery_id(""), image_filename(""), original_filename("")
{
}

Document::~Document()
{
}
        
void Document::set_thumbnail(std::string filename)
{
  thumbnail_filename = filename;
  thumbnail = Gdk::Pixbuf::create_from_file(filename);
  FILE *fileptr = fopen (filename.c_str(), "rb");
  thumbnail_file_data.clear();
  if (fileptr != NULL)
    {
      gchar block[1024];
      while (!feof(fileptr))
        {
          size_t n = fread(block, sizeof (gchar), 1024, fileptr);
          for (unsigned int i = 0; i < n; i++)
            thumbnail_file_data.push_back(block[i]);
        }
      fclose(fileptr);
    }
  thumbnail_filename = "";
}

void Document::update_thumbnail_from_loaded_data()
{
  size_t len = thumbnail_file_data.size();
  if (thumbnail_file_data.size() == 0)
    {
      Gtk::Image image(Gtk::Stock::MISSING_IMAGE, Gtk::ICON_SIZE_DIALOG);
      thumbnail = image.render_icon(Gtk::Stock::MISSING_IMAGE, 
                                    Gtk::ICON_SIZE_DIALOG);
      return;
    }
  gchar *data = (gchar *) malloc(len * sizeof (gchar));
  size_t count = 0;
  for (std::list<gchar>::iterator i = thumbnail_file_data.begin(); 
       i != thumbnail_file_data.end(); i++)
    {
      data[count] = (*i);
      count++;
    }
  std::string temp_filename = "/tmp/" PACKAGE_NAME ".XXXXXXX";
  int fd = Glib::mkstemp(temp_filename);
  close(fd);
  Glib::file_set_contents(temp_filename, data, len);
  thumbnail = Gdk::Pixbuf::create_from_file(temp_filename);
  thumbnail_filename = "";
  remove(temp_filename.c_str());
}

void Document::set_image_filename(std::string f)
{
  video_filename = "";
  image_filename = f;
  //now make a new thumb.
  //what size of thumb does atpic make?
  //160 pixels in the largest dimension
  Glib::RefPtr<Gdk::Pixbuf> image = Gdk::Pixbuf::create_from_file(f);
  size_t width = image->get_width();
  size_t height = image->get_height();
  size_t new_width, new_height = 0;
  size_t max_thumb_size = 160;
  if (width > height)
    {
      new_width = max_thumb_size;
      new_height = height * (float)((float)max_thumb_size / (float)width);
    }
  else
    {
      new_height = max_thumb_size;
      new_width = width * (float)((float)max_thumb_size / (float)height);
    }

  if (width <= 160 && height <= 160)
    set_thumbnail(f);
  else
    {
      Glib::RefPtr<Gdk::Pixbuf> thumb = 
        image->scale_simple(new_width, new_height, Gdk::INTERP_NEAREST);
      std::string temp_filename = PACKAGE_NAME ".png.XXXX";
      int fd = Glib::mkstemp(temp_filename);
      close(fd);
      thumb->save(temp_filename, "png");
      set_thumbnail(temp_filename);
      remove(temp_filename.c_str());
    }
}

void Document::set_video_filename(std::string f)
{
  image_filename = "";
  video_filename = f;
  //bah, we'll sadly just use our own thumbnail.
  //instead of using /usr/bin/totem-video-thumbnailer
  std::string prog = Glib::find_program_in_path("totem-video-thumbnailer");
  if (prog == "")
    {
      set_thumbnail(Main::get_data_path() + "/video.png");
      return;
    }

  std::string temp_filename = PACKAGE_NAME ".png.XXXX";
  int fd = Glib::mkstemp(temp_filename);
  close(fd);
  std::string cmd = prog + " --size 160 " + f + " " + temp_filename;
  std::string out;
  std::string err;
  int exit_status = 0;
  Glib::spawn_command_line_sync(cmd, &out, &err, &exit_status);
  if (exit_status != 0)
    set_thumbnail(Main::get_data_path() + "/video.png");
  else
    set_thumbnail(temp_filename);
  remove(temp_filename.c_str());
}

bool Document::name_ends_with(std::string filename, std::string extension) 
{
  std::string::size_type idx = filename.rfind(extension);
  if (idx == std::string::npos)
    return false;
  if (idx == filename.length() - extension.length())
    return true;
  return false;
}

bool Document::is_video_filename(std::string f)
{
  return 
    (name_ends_with(f, ".avi") == true ||
    name_ends_with(f, ".ogv") == true ||
    name_ends_with(f, ".mpg") == true ||
    name_ends_with(f, ".AVI") == true ||
    name_ends_with(f, ".OGV") == true ||
    name_ends_with(f, ".MPG") == true);
}

int Document::remove_file(std::string filename)
{
  return remove(filename.c_str());
}

Glib::ustring Document::find_free_file(std::string file)
{
  size_t count = 0;
  while (1)
    {
      Glib::ustring tryfile = file;
      if (count > 0)
        tryfile = file + String::ucompose(".%1", count);
      std::string a = tryfile;
      std::ifstream ifs(a.c_str());
      if (ifs.is_open() == false)
        {
          file = tryfile;
          break;
        }
      else
        ifs.close();
      count++;
    }
  return file;
}

bool Document::exists(std::string f)
{
  std::ifstream ifs(f.c_str());
  if (ifs.is_open())
    {
      ifs.close();
      return true;
    }
  return false;
}

void Document::clean_dir(std::string dirname)
{
  if (Document::exists(dirname) == false)
    return;
  if (Document::name_ends_with(dirname, "/") == false)
    dirname += "/";
  Glib::Dir dir(dirname);
  for (Glib::DirIterator it = dir.begin(); it != dir.end(); it++)
    Document::remove_file(dirname + *it);
  dir.close();
  rmdir(dirname.c_str());

}

bool Document::create_dir(std::string dir)
{
    struct stat testdir;
#ifndef __WIN32__
    if (stat(dir.c_str(), &testdir) || !S_ISDIR(testdir.st_mode))
    {
        guint32 mask = 0755; //make directory only readable for user and group
        if (mkdir(dir.c_str(), mask))
	  return false;
    }
#else
    return false;
#endif
    return true;
}

std::string Document::get_export_filename() const
{
  if (get_original_filename() == "")
    return "";
  if (get_id() == "")
    return "";
  if (id.length() > 0)
    {
      if (id.c_str()[0] == '/')
        return "";
    }
  return get_id() + "-" + get_original_filename();
}

bool Document::same_except_for_tags(const Document& document) const
{
  Document another_document(document);
  another_document.set_tags(get_tags());
  if (another_document == *this)
    return true;
  return false;
}

