// Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>
#include <fstream>
#include <argz.h>
#include <libsoupmm/session.h>
#include <libsoupmm/message.h>
#include <libsoupmm/uri.h>
#include <libsoupmm/message-body.h>
#include <libsoupmm/cookie-jar.h>
#include "session.h"
#include "gallery.h"
#include "ucompose.hpp"
#include <libsoup/soup.h>

using namespace Tapioca;

  
double Session::timeout = 10.0;
Session::Session(const Session& p)
: GalleryList(p)
{
  profile = new Profile(*p.profile);
  add(profile); //copying galleries from profile into session
  web= Soup::Session::create(true);
  web->property_timeout() = 5;
  web->property_idle_timeout() = 5;
  web->property_max_conns_per_host () = 1;
  web->add_feature_by_type(SOUP_TYPE_COOKIE_JAR);
  last_login = 0;
}

void Session::copy_galleries_from_profile()
{
  remove_all_galleries();
  add(profile);
}

Session::Session()
: GalleryList(), profile(0)
{
  web = Soup::Session::create(true);
  web->property_timeout() = 5;
  web->property_idle_timeout() = 5;
  web->property_max_conns_per_host () = 1;
  web->add_feature_by_type(SOUP_TYPE_COOKIE_JAR);
  last_login = 0;
}

Session::~Session()
{
  delete profile;
}

Session* Session::create(Profile *p)
{
  Session *session = new Session();
  session->set_profile(p);
  session->add(p);
  return session;
}

void Session::on_pull_documents_attempted(Glib::RefPtr<Soup::Message> &msg,
                                          Gallery *gallery,
                                          const Session::SlotPullGallery slot,
                                          std::list<std::string> &thumbnail_urls)
{
  if (msg->property_status_code() == 200)
    {
      web->cancel_message(msg, msg->property_status_code());
      Soup::MessageBody body(msg->gobj()->response_body);
      body.set_accumulate(true);
      Soup::Buffer buffer = body.flatten();
      const char *data = buffer.gobj()->data;
      guint len = buffer.gobj()->length;
      if (len == 0)
        {
          pull_galleries_failed.emit(this);
          return;
        }
        
      profile->process_gallery_documents(gallery, std::string(data), thumbnail_urls);
      slot(gallery, thumbnail_urls);
    }
  else
    {
      pull_galleries_failed.emit(this);
      return;
    }
}

void Session::pull_all_documents_from_all_galleries()
{
  for (Profile::iterator it = profile->begin(); it != profile->end(); it++)
    {
      Glib::RefPtr<Soup::Message> msg = Soup::Message::create 
        ("POST", String::ucompose("%1/gallery_view.php?galleryid=%2",
                                  profile->get_url(), (*it)->get_id()));
      //web->queue_message(msg, sigc::bind(sigc::mem_fun
                         //(*this, &Session::on_pull_documents_attempted), *it,
                         //it == --profile->end()));
    }
}

std::string Session::extract_document_id_from_thumbnail_url(std::string u) const
{
   // -->http://88.198.21.167/atpic2/8119/38089/0/1977121/160.jpg
  std::vector<Glib::ustring> a = Glib::Regex::split_simple("/", u);
  if (a.size() < 8)
    return "";
  return a[7];
}

void Session::on_thumbnail_downloaded(Glib::RefPtr<Soup::Message> &msg, Document *document, const Session::SlotPullThumbnail slot)
{
  if (msg->property_status_code() == 200)
    {
      web->cancel_message(msg, msg->property_status_code());
      Soup::MessageBody body(msg->gobj()->response_body);
      body.set_accumulate(true);
      Soup::Buffer buffer = body.flatten();

      const char *data = buffer.gobj()->data;
      guint len = buffer.gobj()->length;
      if (len != 0)
        {
          profile->process_thumbnail(document, data, len);
          msg.reset();
          slot(document);
        }
      else
        {
          msg.reset();
          pull_galleries_failed.emit(this);
          return;
        }
    }
  else
    {
      msg.reset();
      pull_galleries_failed.emit(this);
      return;
    }
}

void Session::on_pull_galleries_attempted(Glib::RefPtr<Soup::Message> &msg)
{
  if (msg->property_status_code() == 200)
    {
      web->cancel_message(msg, msg->property_status_code());
      Soup::MessageBody body(msg->gobj()->response_body);
      body.set_accumulate(true);
      Soup::Buffer buffer = body.flatten();
      const char *data = buffer.gobj()->data;
      guint len = buffer.gobj()->length;
      if (len == 0)
        {
          pull_galleries_failed.emit(this);
          return;
        }
      profile->process_galleries(std::string(data));
      //go get the docs for each gallery.
      pulled_galleries.emit(*profile);
    }
  else
    pull_galleries_failed.emit(this);
}

void Session::pull_galleries()
{
  //thumbnail_urls.clear();
  Glib::ustring homepage = profile->get_url();
  Glib::RefPtr<Soup::Message> msg = 
    Soup::Message::create("POST", homepage + "/home.php");
  web->queue_message(msg, sigc::mem_fun(*this, &Session::on_pull_galleries_attempted));
  return;
}

              
bool Session::on_download_document_timeout(Glib::RefPtr<Soup::Message> &msg)
{
  web->cancel_message(msg, 998);
  return false;
}

void Session::download_document(Document *document, const Session::SlotDownloadDocument slot)
{
  if (document)
    {
      std::map<std::string, std::string> formdata;
      formdata["galleryid"] = document->get_gallery_id();
      formdata["format"] = "txt";
      Glib::RefPtr<Soup::Message> msg;
      msg = Soup::Message::create_form_request("GET", profile->get_url() + 
                                                 "/pic_list_urls.php", formdata);
      guint status = web->send_message(msg);
      if (status != 200)
        {
          download_document_failed.emit(document);
          return;
        }

      if (status == 200)
        {
          std::string uri = get_document_link_from_link_listing(msg, document);
          msg.reset();
          if (uri != "")
            {
              Glib::RefPtr<Soup::Message> m = Soup::Message::create("GET", uri);
             m->signal_wrote_chunk().connect
                (sigc::bind(sigc::mem_fun(downloaded_chunk_of_document, &sigc::signal<void,Document*>::emit), document));
              sigc::connection timer = Glib::signal_timeout().connect(sigc::bind(sigc::mem_fun(*this, &Session::on_download_document_timeout), m), 10*1000);
              web->queue_message(m, sigc::bind<Document*, const Session::SlotDownloadDocument,sigc::connection >(sigc::mem_fun(this, &Session::on_document_downloaded), document, slot, timer));
            }
          else
            download_document_failed.emit(document);
        }
      else
        msg.reset();
    }
  return;
}

bool Session::is_login_necessary()
{
  double now = (double) time(NULL);
  if ((double)last_login + (timeout * 60.0) < now)
    return true;
  else
    return false;
}

Glib::RefPtr<Soup::Message> Session::login()
{
  std::map<std::string, std::string> formdata;
  formdata["login"] = profile->get_username();
  formdata["password"] = profile->get_password();
  Glib::ustring homepage = profile->get_url();
  Glib::RefPtr<Soup::Message> msg = 
    Soup::Message::create_form_request("POST", homepage + 
                                       "/artist_login_check.php", formdata);
  //web->queue_message(msg, sigc::mem_fun(*this, &Session::on_login_attempted));
  web->send_message(msg);
  return msg;
}

void Session::login(sigc::slot<void, bool> slot)
{
  if (get_profile()->get_username() == "" ||
      get_profile()->get_password() == "")
    {
      slot(false);
      return;
    }
  if (is_login_necessary() == false)
    {
      slot(true);
      return;
    }
  Glib::RefPtr<Soup::Message> msg = login();
  if (msg->property_status_code() == 200)
    on_login_attempted(msg, slot);
  else
    slot(false);
}

void Session::on_login_attempted(Glib::RefPtr<Soup::Message> &msg,
                                 sigc::slot<void, bool> slot)
{
  bool success = false;
  if (msg->property_status_code() == 200)
    {
      Soup::MessageBody body(msg->gobj()->response_body);
      body.set_accumulate(true);
      Soup::Buffer buffer = body.flatten();;
      const char *data = buffer.gobj()->data;
      guint len = buffer.gobj()->length;
      if (len == 0)
        {
          slot(success);
          return;
        }
      if (strstr(data, "class=\"error\"") == NULL)
        {
          last_login = time(NULL);
          success = true;
        }
    }
  slot(success);
}

std::string Session::get_document_link_from_link_listing(Glib::RefPtr<Soup::Message> msg, Document *document) const
{
  Soup::MessageBody body(msg->gobj()->response_body);
  body.set_accumulate(true);
  Soup::Buffer buffer = body.flatten();
  const char *d = buffer.gobj()->data;
  size_t len = buffer.gobj()->length;
  if (len == 0)
    return "";
  std::string needle = "/" + document->get_id() + "/";
  char *argz = NULL;
  size_t argz_len = 0;
  argz_create_sep(d, '\n', &argz, &argz_len);
  char *entry = NULL;
  char *link = NULL;
  while ((entry = argz_next(argz, argz_len, entry)))
    {
      if (strstr(entry, needle.c_str()) != NULL)
        link = entry;
    }
  if (!link)
    return "";
  return std::string(link);
}

void Session::on_document_downloaded(Glib::RefPtr<Soup::Message> msg, Document *document, const Session::SlotDownloadDocument slot, sigc::connection timer)
{
  int status = msg->property_status_code();
  if (status == 998)
    {
      msg.reset();
      download_document_failed.emit(document);
    return;
    }
  timer.disconnect();
  if (msg->property_status_code() == 200)
    {
      web->cancel_message(msg, msg->property_status_code());
      downloaded_chunk_of_document.emit(document);
      Soup::MessageBody body(msg->gobj()->response_body);
      body.set_accumulate(true);
      Soup::Buffer buffer = body.flatten();

      const char *data = buffer.gobj()->data;
      guint len = buffer.gobj()->length;
      if (len == 0)
        {
          msg.reset();
          download_document_failed.emit(document);
          return;
        }
  
      downloaded_chunk_of_document.emit(document);
      std::string temp_filename = "/tmp/" PACKAGE_NAME ".XXXXXXX";
      int fd = Glib::mkstemp(temp_filename);
      close(fd);

      downloaded_chunk_of_document.emit(document);
      std::ofstream outfile (temp_filename.c_str(), std::ofstream::binary);
      if (outfile.is_open())
        {
          outfile.write(data, len);
          outfile.close();
          msg.reset();
          downloaded_document.emit(document);
          slot(document, temp_filename);
        }
      else
        {
          msg.reset();
          downloaded_document.emit(document);
        }
    }
  else
    {
      msg.reset();
      download_document_failed.emit(document);
    }
}

void Session::merge_changes(Session *session)
{
  //FIXME
  //maybe the old session has some changes in it
  //that we'd like to bring into this session.
  //what's happened is that we've just done an update from the server
  //and our changes are going to be wiped away unless we do something
  //about it.
  SessionChanges c = session->get_changes();
  for (std::list<ChangedGallery>::iterator i = c.modified_galleries.begin(); 
       i != c.modified_galleries.end(); i++)
    {
      Gallery *gallery = find_by_id((*i).first.get_id());
      if (gallery)
        {
          remove(gallery);
          Gallery *modified_gallery = new Gallery((*i).second);
          push_back(modified_gallery);
        }
    }
  for (std::list<ChangedDocument>::iterator i = c.modified_docs.begin(); 
       i != c.modified_docs.end(); i++)
    {
      Document *document = find_doc_by_id((*i).first.get_id());
      if (document)
        {
          Gallery *gallery = find_by_id(document->get_gallery_id());
          if (gallery)
            {
              gallery->remove_document(document);
              Document *modified_document = new Document((*i).second);
              gallery->add_document(modified_document);
            }
        }
    }
}

void Session::pull_documents_for_gallery(Gallery *gallery, const Session::SlotPullGallery slot, std::list<std::string> &thumbnail_urls)
{
  Glib::RefPtr<Soup::Message> msg = Soup::Message::create 
    ("POST", String::ucompose("%1/gallery_view.php?galleryid=%2",
                              profile->get_url(), gallery->get_id()));
  web->queue_message(msg, sigc::bind(sigc::mem_fun
                                     (*this, &Session::on_pull_documents_attempted), gallery, slot, thumbnail_urls));
}

void Session::pull_thumbnail(std::string url, const Session::SlotPullThumbnail slot)
{
  std::string doc_id = extract_document_id_from_thumbnail_url(url);
  Document *document = profile->find_doc_by_id(doc_id);

  if (!document)
    {
      pull_galleries_failed.emit(this);
      return;
    }

  Glib::RefPtr<Soup::Message> msg = Soup::Message::create("GET", url);
  web->queue_message(msg, sigc::bind(sigc::mem_fun
                                     (*this, &Session::on_thumbnail_downloaded), document, slot));
}

SessionChanges Session::get_changes() const
{
  SessionChanges changes;
  changes.modified_galleries = get_changed_galleries(profile);
  changes.modified_docs = get_changed_documents(profile);
  changes.removed_galleries = profile->get_removed_galleries(this);
  changes.added_galleries = get_removed_galleries(profile);
  changes.removed_documents = profile->get_removed_documents(this);
  changes.added_documents = get_removed_documents(profile);
  return changes;
}

void Session::revert(SessionChanges reverts)
{
  revert_removed_galleries(reverts.removed_galleries);
  revert_removed_documents(reverts.removed_documents);
  revert_modified_galleries(reverts.modified_galleries);
  revert_modified_documents(reverts.modified_docs);
  revert_added_galleries(reverts.added_galleries);
  revert_added_documents(reverts.added_documents);
}

void Session::revert_modified_galleries(std::list<ChangedGallery> l)
{
  for (std::list<ChangedGallery>::iterator i = l.begin(); i != l.end(); i++)
    revert_modified_gallery(*i);
}

void Session::revert_modified_documents(std::list<ChangedDocument> l)
{
  for (std::list<ChangedDocument>::iterator i = l.begin(); i != l.end(); i++)
    revert_modified_document(*i);
}

void Session::revert_added_galleries(std::list<ChangedGallery> l)
{
  for (std::list<ChangedGallery>::iterator i = l.begin(); i != l.end(); i++)
    revert_added_gallery(*i);
}

void Session::revert_removed_galleries(std::list<ChangedGallery> l)
{
  for (std::list<ChangedGallery>::iterator i = l.begin(); i != l.end(); i++)
    revert_removed_gallery(*i);
}

void Session::revert_added_documents(std::list<ChangedDocument> l)
{
  for (std::list<ChangedDocument>::iterator i = l.begin(); i != l.end(); i++)
    revert_added_document(*i);
}

void Session::revert_removed_documents(std::list<ChangedDocument> l)
{
  for (std::list<ChangedDocument>::iterator i = l.begin(); i != l.end(); i++)
    revert_removed_document(*i);
}

void Session::revert_modified_gallery(ChangedGallery g)
{
  Gallery *gallery = find_by_id(g.second.get_id());
  if (gallery)
    *gallery = g.second;
}

void Session::revert_modified_document(ChangedDocument d)
{
  Document *document = find_doc_by_id(d.second.get_id());
  if (document)
    *document = d.second;
}

void Session::revert_added_gallery(ChangedGallery g)
{
  Gallery *gallery = find_by_id(g.first.get_id());
  if (gallery)
    {
      remove(gallery);
      delete gallery;
    }
}

void Session::revert_removed_gallery(ChangedGallery g)
{
  Gallery *gallery = profile->find_by_id(g.first.get_id());
  if (gallery)
    push_back(new Gallery(*gallery));
}

void Session::revert_added_document(ChangedDocument d)
{
  Document *document = find_doc_by_id(d.first.get_id());
  if (document)
    {
      Gallery *gallery = find_by_id(document->get_gallery_id());
      if (gallery)
        {
          gallery->remove_document(document);
          delete document;
        }
    }
}

void Session::revert_removed_document(ChangedDocument d)
{
  Document *document = profile->find_doc_by_id(d.first.get_id());
  if (document)
    {
      Gallery *gallery = profile->find_by_id(document->get_gallery_id());
      if (gallery)
        gallery->add_document(new Document(*document));
    }
}

void Session::apply(SessionChanges changes)
{
  update_modified_galleries(changes.modified_galleries);
  update_modified_documents(changes.modified_docs);
  update_added_galleries(changes.added_galleries);
  update_added_documents(changes.added_documents);
  //removed docs and galleries can't have been edited.
}

void Session::update_modified_galleries(std::list<ChangedGallery> l)
{
  for (std::list<ChangedGallery>::iterator i = l.begin(); i != l.end(); i++)
    update_modified_gallery(*i);
}

void Session::update_modified_documents(std::list<ChangedDocument> l)
{
  for (std::list<ChangedDocument>::iterator i = l.begin(); i != l.end(); i++)
    update_modified_document(*i);
}

void Session::update_added_galleries(std::list<ChangedGallery> l)
{
  for (std::list<ChangedGallery>::iterator i = l.begin(); i != l.end(); i++)
    update_added_gallery(*i);
}

void Session::update_added_documents(std::list<ChangedDocument> l)
{
  for (std::list<ChangedDocument>::iterator i = l.begin(); i != l.end(); i++)
    update_added_document(*i);
}

void Session::update_modified_document(ChangedDocument d)
{
  Document *document = find_doc_by_id(d.second.get_id());
  if (document)
    *document = d.first;
}

void Session::update_modified_gallery(ChangedGallery g)
{
  Gallery *gallery = find_by_id(g.second.get_id());
  if (gallery)
    *gallery = g.first;
}

void Session::update_added_document(ChangedDocument d)
{
  Document *document = find_doc_by_id(d.first.get_id());
  if (document)
    *document = d.first;
}

void Session::update_added_gallery(ChangedGallery g)
{
  Gallery *gallery = find_by_id(g.first.get_id());
  if (gallery)
    {
      Gallery *new_gallery = new Gallery(g.first);
      new_gallery->add(gallery);
      remove(gallery);
      push_back(new_gallery);
      delete gallery;
    }
}
