// Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>
#include <fstream>
#include "profile.h"
#include "gallery.h"

using namespace Tapioca;

Profile::Profile(const Profile& p)
: GalleryList(p), url(p.url), id(p.id), username(p.username), 
    password(p.password)
{
}

Profile::Profile()
: url(""), id(""), username(""), password("")
{
}

Profile::~Profile()
{
}

std::string Profile::extract_id(std::string data) const
{
  std::vector<Glib::ustring> c= Glib::Regex::split_simple("UID", data);
  std::vector<Glib::ustring> d = Glib::Regex::split_simple("\\)", c[1]);
  return d[0];
}

Gallery *Profile::process_gallery_chunk(std::string data) const
{
  Gallery *gallery = new Gallery();

  std::vector<Glib::ustring> a = 
    Glib::Regex::split_simple("span class=\"field\"", data);
  std::string gallery_id = extract_gallery_id(a[0]);
  gallery->set_id(gallery_id);

  std::string gallery_thumbnail_id = extract_gallery_thumbnail_id(a[0]);
  gallery->set_thumbnail(gallery_thumbnail_id);

  std::string title = extract_gallery_title(a[2]);
  if (title != " ")
    gallery->set_title(title);

  std::list<std::string> tags = extract_gallery_tags(a[4]);
  gallery->set_tags(tags);

  std::string description = extract_gallery_description(a[5]);
  gallery->set_description(description);

  std::string order = extract_gallery_priority(a[6]);
  gallery->set_order(order);

  double lat, lon;
  extract_gallery_lat_lon(a[6], lat, lon);
  gallery->set_lat_long(lat, lon);
  return gallery;
}

void Profile::extract_gallery_lat_lon(std::string data, double &lat, double &lon) const
{
  std::vector<Glib::ustring> a = 
    Glib::Regex::split_simple(" \\<\\/span\\>", data);
  std::vector<Glib::ustring> b = 
    Glib::Regex::split_simple("\\<td class=\"geo\"\\>", a[1]);
  std::vector<Glib::ustring> c = 
    Glib::Regex::split_simple("\\<", b[2]);
  lat = atof(std::string(c[0]).c_str());
  std::vector<Glib::ustring> d = 
    Glib::Regex::split_simple("\\<", b[4]);
  lon = atof(std::string(d[0]).c_str());
}

std::string Profile::extract_gallery_priority(std::string data) const
{
  std::vector<Glib::ustring> a = 
    Glib::Regex::split_simple(" \\<\\/span\\>", data);
  std::vector<Glib::ustring> b = 
    Glib::Regex::split_simple("\\<", a[1]);
  std::string s = b[0];
  if (b[0] == "Not set")
    return "";
  return b[0];
}

std::string Profile::extract_gallery_id(std::string data) const
{
  std::list<std::string> tags;
  std::vector<Glib::ustring> a = 
    Glib::Regex::split_simple("/", data);
  return a[5];
}

std::string Profile::extract_gallery_thumbnail_id(std::string data) const
{
  std::list<std::string> tags;
  std::vector<Glib::ustring> a = 
    Glib::Regex::split_simple("/", data);
  return a[7];
}

std::list<std::string> Profile::extract_gallery_tags(std::string data) const
{
  std::list<std::string> tags;
  std::vector<Glib::ustring> a = 
    Glib::Regex::split_simple(" \\<\\/span\\>", data);
  std::vector<Glib::ustring> b = 
    Glib::Regex::split_simple("\\<", a[1]);
  std::vector<Glib::ustring> c = 
    Glib::Regex::split_simple(" ", b[0]);
  for (unsigned int i = 0; i < c.size(); i++)
    {
      if (c[i] != " " && c[i] != "")
        tags.push_back(c[i]);
    }
  return tags;
}

std::string Profile::extract_gallery_title(std::string data) const
{
  std::vector<Glib::ustring> a = 
    Glib::Regex::split_simple("\\<\\/span\\> ", data);
  std::vector<Glib::ustring> b = 
    Glib::Regex::split_simple("\\<", a[1]);
  return decode_xml(b[0]);
}

std::string Profile::extract_gallery_description(std::string data) const
{
  std::vector<Glib::ustring> a = 
    Glib::Regex::split_simple("\\<\\/span\\> ", data);
  std::vector<Glib::ustring> b = 
    Glib::Regex::split_simple("\\</td\\>", a[1]);
  return decode_xml(b[0]);
}

void Profile::process_galleries(std::string data)
{
  remove_all_galleries();
  std::vector<Glib::ustring> chunks = 
    Glib::Regex::split_simple("class=\"homegalimage\"", data);
  std::vector<Glib::ustring>::iterator it  = chunks.begin();
  it++;
  if (get_id() == "")
    set_id(extract_id(*it));
  it++;
  for (; it != chunks.end(); it++)
    {
      Gallery *gallery = process_gallery_chunk(*it);
      if (gallery)
        {
          push_back(gallery);
          processed_gallery(gallery);
        }
    }
  return;
}

std::string Profile::extract_document_id(std::string data) const
{
  std::vector<Glib::ustring> a = Glib::Regex::split_simple("picid=", data);
  if (a.size() <= 1)
    return "";
  std::vector<Glib::ustring> b = Glib::Regex::split_simple("&", a[1]);
  return b[0];
}

std::string Profile::extract_document_order(std::string data) const
{
  std::vector<Glib::ustring> a = Glib::Regex::split_simple(": ", data);
  if (a.size() <= 1)
    return "";
  return a[1];
}

std::string Profile::extract_document_title(std::string data) const
{
  std::vector<Glib::ustring> a = Glib::Regex::split_simple(": ", data);
  if (a.size() <= 1)
    return "";
  return decode_xml(a[1]);
}

std::string Profile::extract_original_document_filename(std::string data) const
{
  std::vector<Glib::ustring> a = Glib::Regex::split_simple(": ", data);
  if (a.size() <= 1)
    return "";
  std::vector<Glib::ustring> b = Glib::Regex::split_simple("\n", a[1]);
  return decode_xml(b[0]);
}

std::string Profile::extract_document_description(std::string data) const
{
  std::vector<Glib::ustring> a = 
    Glib::Regex::split_simple("\\<div class=\"center\"\\>", data);
  std::vector<Glib::ustring> b = Glib::Regex::split_simple("\\<br /\\>", a[0]);
  if (b.size() <= 4)
    return "";
  Glib::ustring desc = "";
  for (unsigned int i = 4; i < b.size(); i++)
    desc += b[i];
  Glib::ustring::size_type colon = desc.find(":");
  if (colon == Glib::ustring::npos)
    return "";
  Glib::ustring::size_type space = desc.find(" ", colon);
  if (space == Glib::ustring::npos)
    return "";
  desc = desc.substr(space + 1, desc.length());
  return decode_xml(desc);
}

std::list<std::string> Profile::extract_document_tags(std::string data) const
{
  std::list<std::string> tags;
  std::vector<Glib::ustring> a = 
    Glib::Regex::split_simple("\\<div class=\"tags\"\\>", data);
  if (a.size() <= 1)
    return tags;
  std::vector<Glib::ustring> b = Glib::Regex::split_simple("\\<i\\>", a[1]);
  if (b.size() <= 1)
    return tags;
  std::vector<Glib::ustring> c = Glib::Regex::split_simple("\\</i\\>", b[1]);
  std::vector<Glib::ustring> d = Glib::Regex::split_simple("\\</i\\>", c[0]);
  for (unsigned int i = 0; i < d.size(); i++)
    {
      if (d[i] == " " || d[i] == "\n" || d[i] == "")
        continue;
      tags.push_back(d[i]);
    }
  return tags;
}

void Profile::extract_document_lat_long(std::string data, double &lat, double &lon) const
{
  std::vector<Glib::ustring> a = 
    Glib::Regex::split_simple("\\<td class=\"technical\"\\>", data);
  if (a.size() <= 1)
    return;
  std::vector<Glib::ustring> b = Glib::Regex::split_simple("\\</td\\>", a[1]);
  std::vector<Glib::ustring> c = Glib::Regex::split_simple("\\<br /\\>", b[0]);
  for (unsigned int i = 0; i < c.size(); i++)
    {
      if (c[i].find("latitude") != Glib::ustring::npos)
        {
          std::vector<Glib::ustring> d = Glib::Regex::split_simple(" ", c[i]);
          if (d.size() > 1)
            {
              std::string latstr = d[1];
              lat = atof(latstr.c_str());
            }
        }
      if (c[i].find("longitude") != Glib::ustring::npos)
        {
          std::vector<Glib::ustring> d = Glib::Regex::split_simple(" ", c[i]);
          if (d.size() > 1)
            {
              std::string lonstr = d[1];
              lon = atof(lonstr.c_str());
            }
        }
    }
}

Tapioca::Document* Profile::extract_document(std::string data) const
{
  Document *document = new Document();
  std::string id = extract_document_id(data);
  document->set_id(id);
  std::vector<Glib::ustring> a = 
    Glib::Regex::split_simple("\\<td valign=\"top\"\\>", data);
  if (a.size() <= 1)
    {
      delete document;
      return NULL;
    }
  std::vector<Glib::ustring> b = Glib::Regex::split_simple("\\<br /\\>", a[1]);
  std::string order = extract_document_order(b[1]);
  document->set_order(order);

  std::string orig = extract_original_document_filename(b[2]);
  document->set_original_filename(orig);

  std::string title = extract_document_title(b[3]);
  document->set_title(title);

  std::string desc = extract_document_description(a[1]);
  document->set_description(desc);

  std::list<std::string> tags = extract_document_tags(a[1]);
  document->set_tags(tags);

  double lat, lon = 0.0;
  extract_document_lat_long(a[1], lat, lon);
  document->set_lat_long(lat, lon);

  return document;
}

          
std::string Profile::extract_thumbnail_url(std::string data) const
{
  std::vector<Glib::ustring> a = 
    Glib::Regex::split_simple("\\<td class=\"rotate\"\\>", data);
  if (a.size() <= 2)
    return "";
  std::vector<Glib::ustring> b = 
    Glib::Regex::split_simple("\\<img src=\"", a[2]);
  if (b.size() <= 1)
    return "";
  std::vector<Glib::ustring> c = 
    Glib::Regex::split_simple("\"", b[1]);
  return c[0];
}

void Profile::process_gallery_documents(Gallery *gallery, std::string data, std::list<std::string> &thumbnail_urls)
{
  gallery->remove_all_documents();
  std::vector<Glib::ustring> chunks = 
    Glib::Regex::split_simple("\\<table class=\"rotate\"\\>", data);
  if (chunks.size() == 1)
    return;
  for (unsigned int i = 1; i < chunks.size(); i++)
    {
      Document *document = extract_document(chunks[i]);
      if (document)
        {
          document->set_gallery_id(gallery->get_id());
          gallery->add_document(document);
          std::string url = extract_thumbnail_url(chunks[i]);
          thumbnail_urls.push_back(url);
          processed_document_for_gallery(gallery, document);
        }
    }
  return;
}

void Profile::process_thumbnail(Document *document, const char *data, guint len)
{
  std::string temp_filename = "/tmp/" PACKAGE_NAME ".XXXXXXX";
  int fd = Glib::mkstemp(temp_filename);
  close(fd);

  std::ofstream outfile (temp_filename.c_str(), std::ofstream::binary);
  if (outfile.is_open())
    {
      outfile.write(data, len);
      outfile.close();
    }
  document->set_thumbnail(temp_filename);
  Document::remove_file(temp_filename);
  processed_thumbnail_for_document.emit(document);
}

void Profile::replace(Glib::ustring &data, const Glib::ustring src, const Glib::ustring dest)
{
  Glib::RefPtr<Glib::Regex> p = Glib::Regex::create(Glib::Regex::escape_string(src));
  data = p->replace(data, 0, dest, static_cast<Glib::RegexMatchFlags>(0));
}
Glib::ustring Profile::decode_xml(Glib::ustring data)
{
  if (data.find("&") == Glib::ustring::npos)
    return data;
  replace(data, "&gt;", ">");
  replace(data, "&lt;", "<");
  replace(data, "&apos;", "'");
  replace(data, "&amp;", "&");
  replace(data, "&quot;", "\"");
  return data;
}
