//  Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>

#include <fstream>
#include <gtkmm.h>
#include <time.h>
#include "main.h"
#include "tapioca-window.h"
#include "preferences-dialog.h"
#include "gallery-dialog.h"
#include "document-dialog.h"
#include "connect-dialog.h"
#include "changed-gallery.h"
#include "changed-document.h"
#include "ucompose.hpp"

TapiocaWindow::TapiocaWindow()
{
  Glib::RefPtr<Gtk::Builder> xml = 
    Gtk::Builder::create_from_file(Main::get_glade_path() + "/tapioca.gtk");

    Gtk::Window *w = 0;
    xml->get_widget("window", w);
    window.reset(w);
    xml->get_widget("contents", contents);

    w->set_icon_from_file(Main::get_data_path() + "/icon.png");
    std::list<Gtk::TargetEntry> targets;
    targets.push_back(Gtk::TargetEntry("TapiocaDocument", Gtk::TARGET_SAME_APP));
    targets.push_back(Gtk::TargetEntry("GTK_TREE_MODEL_ROW", Gtk::TARGET_SAME_WIDGET));

    xml->get_widget("treeview", treeview);

    galleries_list = Gtk::ListStore::create(galleries_columns);
    treeview->set_model(galleries_list);
    treeview->get_selection()->set_mode(Gtk::SELECTION_SINGLE);
    treeview->append_column("Galleries", galleries_columns.name);
    session = NULL;
    treeview->get_selection()->signal_changed().connect
      (sigc::mem_fun(*this, &TapiocaWindow::on_gallery_selected));
    treeview->signal_row_activated().connect
      (sigc::mem_fun(*this, &TapiocaWindow::on_gallery_activated));
    treeview->signal_button_press_event().connect_notify
      (sigc::mem_fun(*this, &TapiocaWindow::on_treeview_clicked));
    treeview->enable_model_drag_dest(targets, Gdk::ACTION_MOVE);
    treeview->signal_drag_data_received().connect
      (sigc::mem_fun
       (*this, &TapiocaWindow::on_galleries_drop_drag_data_received));
    fill_in_galleries();
        
    Gtk::Menu::MenuList& menulist = treeview_context_menu.items();
    menulist.push_back
      (Gtk::Menu_Helpers::MenuElem
       ("_Properties", 
        sigc::mem_fun(*this, &TapiocaWindow::on_gallery_properties_clicked)));
    menulist.push_back
      (Gtk::Menu_Helpers::MenuElem
       ("_Add", sigc::mem_fun(*this, &TapiocaWindow::on_add_gallery_clicked)));
    menulist.push_back
      (Gtk::Menu_Helpers::MenuElem
       ("_Remove", sigc::mem_fun(*this, 
                                 &TapiocaWindow::on_remove_gallery_clicked)));
    treeview_context_menu.accelerate(*window);
      

    xml->get_widget("iconview", iconview);
    documents_list = Gtk::ListStore::create (documents_columns);
    iconview->set_model(documents_list);
    iconview->set_pixbuf_column(documents_columns.image);
    iconview->set_text_column(documents_columns.name);
    iconview->signal_selection_changed().connect
      (sigc::mem_fun(*this, &TapiocaWindow::on_document_selected));

    iconview->signal_item_activated().connect(sigc::mem_fun(*this, &TapiocaWindow::on_document_activated));
    iconview->signal_selection_changed().connect(sigc::mem_fun(*this, &TapiocaWindow::on_document_selected));
    iconview->signal_button_press_event().connect_notify
      (sigc::mem_fun(*this, &TapiocaWindow::on_iconview_clicked));
    iconview->signal_drag_end().connect_notify
      (sigc::mem_fun(*this, &TapiocaWindow::on_iconview_reordered));
    iconview->enable_model_drag_source(targets, Gdk::MODIFIER_MASK, 
                                       Gdk::ACTION_MOVE);
    iconview->enable_model_drag_source(targets);
    iconview->enable_model_drag_dest(targets, Gdk::ACTION_MOVE);
    iconview->signal_drag_data_get().connect
      (sigc::mem_fun(*this, &TapiocaWindow::on_document_drag_data_get));
    iconview->signal_drag_data_received().connect
      (sigc::mem_fun
       (*this, &TapiocaWindow::on_documents_drop_drag_data_received));


    xml->get_widget("gallery_label", gallery_label);
    menulist = iconview_context_menu.items();
    menulist.push_back
      (Gtk::Menu_Helpers::MenuElem
       ("_Properties", 
        sigc::mem_fun(*this, &TapiocaWindow::on_image_properties_clicked)));
    menulist.push_back
      (Gtk::Menu_Helpers::MenuElem
       ("_Add", sigc::mem_fun(*this, &TapiocaWindow::on_add_document_clicked)));
    menulist.push_back
      (Gtk::Menu_Helpers::MenuElem
       ("_Remove", sigc::mem_fun(*this, 
                                 &TapiocaWindow::on_remove_document_clicked)));
    treeview_context_menu.accelerate(*window);


    //menubar callbacks
    xml->get_widget("connect_menuitem", connect_menuitem);
    connect_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_connect_clicked));
    xml->get_widget("open_offline_menuitem", open_offline_menuitem);
    open_offline_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_open_offline_changes_clicked));
    xml->get_widget("save_offline_menuitem", save_offline_menuitem);
    save_offline_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_save_offline_changes_clicked));
    xml->get_widget("save_offline_as_menuitem", save_offline_as_menuitem);
    save_offline_as_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_save_offline_changes_as_clicked));
    xml->get_widget("save_changes_menuitem", save_changes_menuitem);
    save_changes_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_save_changes_to_atpic_clicked));
    xml->get_widget("export_menuitem", export_menuitem);
    export_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_export_clicked));
    export_menuitem->set_sensitive(false);
    xml->get_widget("exit_menuitem", exit_menuitem);
    exit_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_exit_clicked));
    xml->get_widget("preferences_menuitem", preferences_menuitem);
    preferences_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_preferences_clicked));
    xml->get_widget("image_properties_menuitem", image_properties_menuitem);
    image_properties_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_image_properties_clicked));
    image_properties_menuitem->set_sensitive(false);
    xml->get_widget("gallery_properties_menuitem", gallery_properties_menuitem);
    gallery_properties_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_gallery_properties_clicked));
    gallery_properties_menuitem->set_sensitive(false);
    xml->get_widget("about_menuitem", about_menuitem);
    about_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_about_clicked));
    fill_in_documents();
    update_menuitems();
}

void TapiocaWindow::hide()
{
  window->hide();
  if (session)
    delete session;
}

bool TapiocaWindow::run()
{
  window->show_all();
  return true;
}

void TapiocaWindow::on_connect_clicked()
{
  Profile *profile = new Profile();
  ConnectDialog d(profile);
  d.set_parent_window(*window);
  int response = d.run();
  if (response == Gtk::RESPONSE_ACCEPT)
    {
      //FIXME
      //connect to homepage with username and password
      //fill up profile with galleries and documents.
      //have a progressbar for this operation.
      Profile *fake_profile = Main::populate_fake_profile();
      fake_profile->set_username(profile->get_username());
      fake_profile->set_password(profile->get_password());
      delete profile;
      Session *new_session = Session::create(fake_profile);
      if (session)
        delete session;
      session = new_session;
      fill_in_galleries();
      fill_in_documents();
      if (session->size() > 0)
        treeview->set_cursor(Gtk::TreePath("0"));
    }
  d.hide();
}

void TapiocaWindow::on_open_offline_changes_clicked()
{
    Gtk::FileChooserDialog chooser(*window, "Choose File to Load");
    Gtk::FileFilter sav_filter;
    sav_filter.add_pattern("*" + std::string(TAPIOCA_EXT));
    chooser.set_filter(sav_filter);
    chooser.set_current_folder(Glib::get_home_dir());

    chooser.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
    chooser.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_ACCEPT);
    chooser.set_default_response(Gtk::RESPONSE_ACCEPT);
	
    chooser.show_all();
    int res = chooser.run();
    
    if (res == Gtk::RESPONSE_ACCEPT)
      {
        std::string old_save_filename = current_save_filename;
	current_save_filename = chooser.get_filename();
	chooser.hide();
        load_session();
      }
}

void TapiocaWindow::load_session()
{
  if (session)
    delete session;
  std::ifstream ifs(current_save_filename.c_str());
  boost::archive::text_iarchive ia(ifs);
  session = new Session();
  ia >> session;
  session->update_thumbnails();
  session->get_profile()->update_thumbnails();
  remove_deleted_documents();
  fill_in_galleries();
  fill_in_documents();
  if (session->size() > 0)
    treeview->set_cursor(Gtk::TreePath("0"));
}

void TapiocaWindow::remove_deleted_documents()
{
  std::list<Document*> removed_docs;
  session->find_deleted_documents(removed_docs);
  std::string removed = "The following documents were not found:\n";
  size_t count = 0;
  size_t max_listed = 10;
  for (std::list<Document*>::iterator it = removed_docs.begin();
       it != removed_docs.end(); it++)
    {
      Gallery *gallery = session->find_by_id((*it)->get_gallery_id());
      if (count < max_listed)
        removed += (*it)->get_image_filename() + "\n";
      count++;
      gallery->remove_document(*it);
    }
  if (removed_docs.size() > max_listed)
    removed += 
      String::ucompose("and %1 more", removed_docs.size() - max_listed);

  if (removed_docs.size() > 0)
    {
      Gtk::MessageDialog d(removed, Gtk::MESSAGE_ERROR);
      d.set_modal();
      d.set_title("Some documents were not loaded");
      d.set_icon_from_file(Main::get_data_path() + "/icon.png");
      d.run();
    }
}
void TapiocaWindow::save_session()
{
  std::ofstream ofs(current_save_filename.c_str());
  boost::archive::text_oarchive oa(ofs);
  oa << session;
}

void TapiocaWindow::on_save_offline_changes_clicked()
{
  if (current_save_filename.empty() == true)
    on_save_offline_changes_as_clicked();
  else
    save_session();
}

void TapiocaWindow::on_save_offline_changes_as_clicked()
{
    Gtk::FileChooserDialog chooser(*window, "Choose a File to Save To",
				   Gtk::FILE_CHOOSER_ACTION_SAVE);
    Gtk::FileFilter sav_filter;
    sav_filter.add_pattern("*" + std::string(TAPIOCA_EXT));
    chooser.set_filter(sav_filter);
    chooser.set_current_folder(Glib::get_home_dir());

    chooser.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
    chooser.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_ACCEPT);
    chooser.set_default_response(Gtk::RESPONSE_ACCEPT);
    
    chooser.show_all();
    int res = chooser.run();
    
    if (res == Gtk::RESPONSE_ACCEPT)
    {
        std::string old_save_filename = current_save_filename;
	current_save_filename = chooser.get_filename();
	chooser.hide();
        save_session();
    }
}

void TapiocaWindow::on_save_changes_to_atpic_clicked()
{
  Profile *profile = session->get_profile();
  printf("number of original galleries on atpic: %d\n", profile->size());
  printf("number of galleries here: %d\n", session->size());
  std::list<ChangedGallery> changed_galleries;
  changed_galleries = session->get_changed_galleries(profile);
  printf("number of galleries modified: %d\n", changed_galleries.size());
  std::list<ChangedDocument> changed_docs;
  changed_docs = session->get_changed_documents(profile);
  printf("number of documents modified: %d\n", changed_docs.size());

  std::list<ChangedGallery> removed_galleries;
  removed_galleries = profile->get_removed_galleries(session);
  printf("number of galleries removed: %d\n", removed_galleries.size());

  std::list<ChangedGallery> added_galleries;
  added_galleries = session->get_removed_galleries(profile);
  printf("number of galleries added: %d\n", added_galleries.size());

  std::list<ChangedDocument> removed_documents;
  removed_documents = profile->get_removed_documents(session);
  printf("number of documents removed: %d\n", removed_documents.size());

  std::list<ChangedDocument> added_documents;
  added_documents = session->get_removed_documents(profile);
  printf("number of documents added: %d\n", added_documents.size());



}

void TapiocaWindow::on_export_clicked()
{
}

void TapiocaWindow::on_exit_clicked()
{
  window->hide();
}

void TapiocaWindow::on_preferences_clicked()
{
  PreferencesDialog d;
  d.set_parent_window(*window);
  d.run();
  d.hide();
}

void TapiocaWindow::on_image_properties_clicked()
{
  DocumentDialog d(get_selected_document());
  d.run();
  d.set_parent_window(*window);
  d.hide();
  update_selected_document_name();
}

void TapiocaWindow::on_gallery_properties_clicked()
{
  Gallery *g = get_selected_gallery();
  GalleryDialog d(g);
  d.set_parent_window(*window);
  d.run();
  d.hide();
  update_selected_gallery_name();
}

void TapiocaWindow::update_selected_document_name()
{
  typedef std::list<Gtk::TreeModel::Path> type_list_paths;
  type_list_paths selected = iconview->get_selected_items();
  if (!selected.empty())
    {
      const Gtk::TreeModel::Path &path = *selected.begin();
      Gtk::TreeModel::iterator iter = documents_list->get_iter(path);
      Gtk::TreeModel::Row row = *iter;
      Document *document = row[documents_columns.data];
      row[documents_columns.name] = document->get_title();
    }
}

void TapiocaWindow::update_selected_gallery_name()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Gallery *gallery = row[galleries_columns.data];
      row[galleries_columns.name] = gallery->get_title();
      gallery_label->set_text(gallery->get_title());
    }
}

void TapiocaWindow::on_about_clicked()
{
  Gtk::AboutDialog* dialog;

  Glib::RefPtr<Gtk::Builder> xml
    = Gtk::Builder::create_from_file(Main::get_glade_path() + "/about-dialog.gtk");

  xml->get_widget("dialog", dialog);
  dialog->set_transient_for(*window);
  dialog->set_icon_from_file(Main::get_data_path() + "/icon.png");

  dialog->set_version(PACKAGE_VERSION);
  dialog->set_logo(Gdk::Pixbuf::create_from_file(Main::get_data_path() + "/icon.png"));
  dialog->show_all();
  dialog->run();
  delete dialog;
  return;
}

Gallery* TapiocaWindow::get_selected_gallery()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Gallery *gallery = row[galleries_columns.data];
      return gallery;
    }
  return NULL;
}

Document* TapiocaWindow::get_selected_document()
{
  typedef std::list<Gtk::TreeModel::Path> type_list_paths;
  type_list_paths selected = iconview->get_selected_items();
  if (!selected.empty())
    {
      const Gtk::TreeModel::Path &path = *selected.begin();
      Gtk::TreeModel::iterator iter = documents_list->get_iter(path);
      Gtk::TreeModel::Row row = *iter;
      return row[documents_columns.data];
    }
  return NULL;
}

void TapiocaWindow::update_menuitems()
{
  Gallery *gallery = get_selected_gallery();
  if (session != NULL)
    {
      save_offline_menuitem->set_sensitive(true);
      save_offline_as_menuitem->set_sensitive(true);
      save_changes_menuitem->set_sensitive(true);
      contents->set_sensitive(true);
      if (gallery)
        gallery_properties_menuitem->set_sensitive(true);
      else
        gallery_properties_menuitem->set_sensitive(false);
    }
  else
    {
      save_offline_menuitem->set_sensitive(false);
      save_offline_as_menuitem->set_sensitive(false);
      save_changes_menuitem->set_sensitive(false);
      gallery_properties_menuitem->set_sensitive(false);
      contents->set_sensitive(false);
    }

  if (get_selected_document() != NULL)
    image_properties_menuitem->set_sensitive(true);
  else
    image_properties_menuitem->set_sensitive(false);
}

void TapiocaWindow::on_gallery_selected()
{
  Gallery *gallery = get_selected_gallery();
  fill_in_documents();
  if (gallery != NULL)
    gallery_label->set_text(gallery->get_title());
  update_menuitems();
}

void TapiocaWindow::on_document_selected()
{
  update_menuitems();
}

void TapiocaWindow::on_gallery_activated(const Gtk::TreeModel::Path& path,
                                      Gtk::TreeViewColumn *col)
{
  update_menuitems();
  on_gallery_properties_clicked();
}

void TapiocaWindow::on_document_activated(const Gtk::TreeModel::Path& path)
{
  update_menuitems();
  on_image_properties_clicked();
}

void TapiocaWindow::add_gallery(Tapioca::Gallery *gallery)
{
  Gtk::TreeIter i = galleries_list->append();
  (*i)[galleries_columns.name] = gallery->get_title();
  (*i)[galleries_columns.data] = gallery;
}

void TapiocaWindow::fill_in_galleries()
{
  galleries_list->clear();
  if (session)
    for (Tapioca::Session::iterator it = session->begin(); 
         it != session->end(); it++)
      add_gallery((*it));
}

void TapiocaWindow::fill_in_documents()
{
  Gallery *g = get_selected_gallery();
  if (g)
    {
      documents_list->clear();
      for (Gallery::iterator  it = g->begin(); it != g->end(); it++)
        add_document(*it);
    }
  else
    {
      documents_list->clear();
      gallery_label->set_text("");
    }
}

void TapiocaWindow::add_document(Tapioca::Document *document)
{
  if (document)
    {
      Gtk::TreeModel::Row row = *(documents_list->append());
      row[documents_columns.image] = document->get_thumbnail();
      row[documents_columns.name] = document->get_title();
      row[documents_columns.data] = document;
    }
}
      
void TapiocaWindow::on_treeview_clicked(GdkEventButton* event)
{
  if (event->type == GDK_BUTTON_PRESS &&event->button == 3)
    {
      treeview_context_menu.popup(event->button, event->time);
    }
}

void TapiocaWindow::on_iconview_clicked(GdkEventButton* event)
{
  if (event->type == GDK_BUTTON_PRESS &&event->button == 3)
    {
      Gtk::Menu::MenuList& menulist = iconview_context_menu.items();
      Gtk::TreeModel::Path path = iconview->get_path_at_pos(event->x, event->y);
      if (path.empty() == false)
        iconview->select_path(path);
      Document *document = get_selected_document();
      menulist.front().set_sensitive(document != NULL);
      iconview_context_menu.popup(event->button, event->time);
    }
}

void TapiocaWindow::on_add_gallery_clicked()
{
  Gtk::TreeIter i = galleries_list->append();
  Gallery *gallery = new Gallery();
  gallery->set_title("new gallery");
  gallery->set_id(String::ucompose("%1", time(NULL)));
  session->push_back(gallery);
  (*i)[galleries_columns.name] = gallery->get_title();
  (*i)[galleries_columns.data] = gallery;
  //now select the new gallery
  Glib::ustring p = String::ucompose("%1", session->size()-1);
  treeview->set_cursor(Gtk::TreePath(p));
  on_gallery_properties_clicked();
}

void TapiocaWindow::on_remove_gallery_clicked()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Gallery *gallery = row[galleries_columns.data];
      session->remove(gallery);
      delete gallery;
      galleries_list->erase(iterrow);
      update_menuitems();
    }
}

void TapiocaWindow::on_add_document_clicked()
{
  //ask for a document
  Gtk::FileChooserDialog chooser(*window, "Choose Document to Add");
  Gtk::FileFilter doc_filter;
  doc_filter.add_pixbuf_formats();
  chooser.set_filter(doc_filter);
  chooser.set_current_folder(Glib::get_home_dir());

  chooser.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  chooser.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_ACCEPT);
  chooser.set_default_response(Gtk::RESPONSE_ACCEPT);
	
  chooser.show_all();
  int res = chooser.run();
    
  if (res == Gtk::RESPONSE_ACCEPT)
    {
      std::string filename = chooser.get_filename();
      Document *document = new Document();
      document->set_image_filename(filename);
      document->set_id(filename);
      document->set_title("new document");
      Gallery *gallery = get_selected_gallery();
      gallery->add_document(document);
      add_document(document);
    }
}

void TapiocaWindow::on_remove_document_clicked()
{
  typedef std::list<Gtk::TreeModel::Path> type_list_paths;
  type_list_paths selected = iconview->get_selected_items();
  if (!selected.empty())
    {
      const Gtk::TreeModel::Path &path = *selected.begin();
      Gtk::TreeModel::iterator iter = documents_list->get_iter(path);
      Gtk::TreeModel::Row row = *iter;
      Document *document = row[documents_columns.data];
      Gallery *gallery = get_selected_gallery();
      gallery->remove(document);
      delete document;
      documents_list->erase(iter);
    }
}

void TapiocaWindow::on_iconview_reordered(const Glib::RefPtr<Gdk::DragContext>&context)
{
  //sort 'em steve dave.
  //okay only one of the icons is out of order.
      
  Document *prev_document = NULL;
  Document *document = NULL;
  Gallery *gallery = get_selected_gallery();
  for (Gtk::TreeModel::iterator it = documents_list->children().begin();
       it != documents_list->children().end(); it++)
    {
      prev_document = document;
      document = (*it)[documents_columns.data];
      if (prev_document && document)
        {
          if (prev_document->get_order() < document->get_order())
            gallery->set_document_order(document, prev_document);
        }
      else 
        gallery->set_document_order(document, NULL);
    }
}
      
void TapiocaWindow::on_document_drag_data_get(const Glib::RefPtr<Gdk::DragContext> &drag_context,
                                                     
                                           Gtk::SelectionData &data, 
                                           guint info, guint time)

{
  drag_context->get_source_window()->show();
  Document *document = get_selected_document();
  data.set(data.get_target(), 8, (const guchar*)document->get_id().c_str(), strlen(document->get_id().c_str()));

}
       
void TapiocaWindow::on_galleries_drop_drag_data_received(const Glib::RefPtr<Gdk::DragContext> &context, int x, int y, const Gtk::SelectionData& selection_data, guint c, guint time)
{
  const int length = selection_data.get_length();
  if (length >= 0 && selection_data.get_format() == 8)
    {
      std::string id = selection_data.get_data_as_string();
      int nx = 0, ny = 0;
      treeview->convert_widget_to_bin_window_coords(x, y, nx, ny);
      Gtk::TreeModel::Path path;
      treeview->get_path_at_pos(nx, ny, path);
      Gtk::TreeModel::iterator iter = galleries_list->get_iter(path);
      Gtk::TreeModel::Row row = *iter;
      Gallery *dest_gallery = row[galleries_columns.data];
      Document *document = session->find_doc_by_id(id);
      Gallery *orig_gallery = session->find_by_id(document->get_gallery_id());
      orig_gallery->remove_document(document);
      fill_in_documents();
      Document *after = NULL;
      if (dest_gallery->size() > 0)
        after = dest_gallery->back();
      document->set_gallery_id(dest_gallery->get_id());
      dest_gallery->set_document_order(document, after);
      dest_gallery->add_document(document);
    }

  context->drag_finish (false, false, time);

}

void TapiocaWindow::on_documents_drop_drag_data_received(const Glib::RefPtr<Gdk::DragContext> &context, int x, int y, const Gtk::SelectionData& selection_data, guint c, guint time)
{
  bool append = false;
  Gallery *gallery = get_selected_gallery();
  const int length = selection_data.get_length();
  Gtk::TreeModel::Path dest;
  if (length >= 0 && selection_data.get_format() == 8)
    {
      Gtk::IconViewDropPosition pos;
      int nx = 0, ny = 0;
      iconview->convert_widget_to_bin_window_coords(x, y, nx, ny);
      iconview->get_dest_item_at_pos(nx, ny, pos);
      dest = iconview->get_path_at_pos(nx, ny);
      if (dest.empty())
        {
          dest = documents_list->get_path(--documents_list->children().end());
          pos = Gtk::ICON_VIEW_DROP_RIGHT;
        }

      if (pos == Gtk::ICON_VIEW_DROP_RIGHT || pos == Gtk::ICON_VIEW_DROP_BELOW)
        {
          if (documents_list->get_iter(dest) == documents_list->children().end() ||
              documents_list->get_iter(dest) == --documents_list->children().end())
            append = true;
          else
            dest = documents_list->get_path(++documents_list->get_iter(dest));
        }
      iconview->set_drag_dest_item(dest, pos);
    }
  Gtk::TreeModel::Path src;
  iconview->get_cursor(src);
  if (append)
  documents_list->move(documents_list->get_iter(src), documents_list->children().end());
  else
  documents_list->move(documents_list->get_iter(src), documents_list->get_iter(dest));
}
