// Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <fstream>
#include "gallery.h"
#include "document.h"

using namespace Tapioca;

Gallery::Gallery(const Gallery& g)
: Item(g), document_thumbnail_id(g.document_thumbnail_id)
{
  for (Gallery::const_iterator it = g.begin(); it != g.end(); it++)
    push_back(new Document(*(*it)));
}

Gallery::Gallery()
: Item(), document_thumbnail_id("")
{
}

Gallery::~Gallery()
{
  for (iterator it = begin(); it != end(); it++)
    delete (*it);
  clear();
}

        
Glib::RefPtr<Gdk::Pixbuf> Gallery::get_thumbnail_image() const
{
  for (Gallery::const_iterator it = begin(); it != end(); it++)
    {
      if ((*it)->get_id() == document_thumbnail_id)
        return (*it)->get_thumbnail();
    }
  return Gtk::Image(Gtk::Stock::MISSING_IMAGE, 
                    Gtk::ICON_SIZE_DIALOG).get_pixbuf();
}

std::string Gallery::get_next_in_order_sequence(std::string seq)
{
  std::string new_seq;
  if (seq.length() == 0)
    return " ";
  char last_letter = seq.at(seq.length() - 1);
  //starts with !, ends with ~
  if (last_letter >= '~')
    new_seq = seq + ' ';
  else
    {
      char new_letter = last_letter + 1;
      new_seq = seq;
      new_seq[new_seq.length() - 1] = new_letter;
    }
  return new_seq;
}

void Gallery::set_document_order(Document *document, Document *after)
{
  std::string new_order = "";
  if (after == NULL && front() != document && document)
    new_order = "";
  else if (after == NULL && front() == document && document)
    return;
  else
    {
      while (true)
        {
          std::string new_order = 
            get_next_in_order_sequence(after->get_order());
          if (has_order(new_order) == false)
            break;
        }
    }
  if (document)
    document->set_order(new_order);
  return;
}

bool Gallery::has_order(std::string o) const
{
  for (Gallery::const_iterator it = begin(); it != end(); it++)
    {
      if ((*it)->get_order() == o)
        return true;
    }
  return false;
}

void Gallery::remove_document(Document *document)
{
  remove(document);
  if (document_thumbnail_id == document->get_id())
    {
      if (size() > 0)
        document_thumbnail_id = front()->get_id();
      else
        document_thumbnail_id = "";
    }
}

void Gallery::update_thumbnails()
{
  for (iterator it = begin(); it != end(); it++)
    (*it)->update_thumbnail_from_loaded_data();
}

void Gallery::add_document(Document *document)
{
  push_back(document);
  document->set_gallery_id(get_id());
  if (size() == 1)
    document_thumbnail_id = front()->get_id();
}

void Gallery::find_deleted_documents(std::list<Document*> &removed)
{
  for (iterator it = begin(); it != end(); it++)
    {
      Document *document = (*it);
      if (document->get_image_filename() != "")
        {
          std::ifstream ifs(document->get_image_filename().c_str());
          if (ifs.is_open() == false)
            removed.push_back(document);
          else
            ifs.close();
        }
    }
}
