# frozen_string_literal: true

require 'uri'

module GitlabQuality
  module TestTooling
    module FeatureReadiness
      class EpicReadinessNotifier
        include Concerns::WorkItemConcern

        FEATURE_READINESS_NOTIFICATION_ID = '<!-- FEATURE READINESS NOTIFICATION -->'
        FEATURE_READINESS_CANDIDATE_LABEL = 'feature readiness candidate'

        def initialize(token:, epic_urls: nil, epic_urls_file: nil, message: nil, dry_run: false)
          @token = token
          @epic_urls = epic_urls
          @epic_urls_file = epic_urls_file
          @custom_message = message
          @dry_run = dry_run
          @results = []
        end

        def invoke!
          urls = collect_epic_urls
          print_header(urls.size)
          process_all_urls(urls)
          print_summary
        end

        private

        attr_reader :token, :epic_urls, :epic_urls_file, :custom_message, :dry_run, :results

        def print_header(url_count)
          puts "Processing #{url_count} epic URL(s)..."
          puts "Dry run mode: #{dry_run ? 'enabled' : 'disabled'}"
          puts
        end

        def process_all_urls(urls)
          urls.each_with_index do |url, index|
            puts "[#{index + 1}/#{urls.size}] Processing: #{url}"
            process_single_url(url)
            puts
          end
        end

        def process_single_url(url)
          result = process_epic_url(url)
          @results << result
          print_result(result)
        rescue StandardError => e
          error_result = { url: url, success: false, error: e.message }
          @results << error_result
          puts "  ❌ Error: #{e.message}"
        end

        def print_result(result)
          if result[:success]
            author_info = result[:author] ? (result[:author]).to_s : "epic (no author assigned)"
            puts "  ✅ Successfully notified #{author_info} and added label '#{FEATURE_READINESS_CANDIDATE_LABEL}' to epic: #{result[:title]}"
          else
            puts "  ❌ Failed: #{result[:error]}"
          end
        end

        def collect_epic_urls
          urls = []
          urls.concat(epic_urls) if epic_urls
          urls.concat(collect_urls_from_file) if epic_urls_file
          validate_urls_present(urls)
          urls.uniq
        end

        def collect_urls_from_file
          validate_file_exists
          File.readlines(epic_urls_file, chomp: true)
              .map(&:strip)
              .reject(&:empty?)
              .reject { |line| line.start_with?('#') }
        end

        def validate_file_exists
          return if File.exist?(epic_urls_file)

          raise ArgumentError, "Epic URLs file not found: #{epic_urls_file}"
        end

        def validate_urls_present(urls)
          return unless urls.empty?

          raise ArgumentError, "No epic URLs provided"
        end

        def process_epic_url(url)
          epic_info = parse_epic_url(url)
          client = work_items_client_for_group(epic_info[:group])
          epic = fetch_and_validate_epic(epic_info, client, url)

          return epic if epic.key?(:error)

          existing_note = existing_note_containing_text(FEATURE_READINESS_NOTIFICATION_ID, epic_info[:epic_iid], client)
          return build_existing_notification_result(url, epic) if existing_note

          post_notification_and_label(epic, epic_info, client) unless dry_run
          build_success_result(url, epic, epic_info)
        end

        def fetch_and_validate_epic(epic_info, client, url)
          epic = fetch_work_item(epic_info[:epic_iid], client, [:notes])
          return { url: url, success: false, error: "Epic not found or not accessible" } unless epic
          return { url: url, success: false, error: "Work item is not an epic (type: #{epic[:workItemType][:name]})" } unless epic[:workItemType][:name] == "Epic"

          epic
        end

        def build_existing_notification_result(url, epic)
          {
            url: url,
            success: false,
            error: "Notification already exists",
            title: epic[:title],
            author: epic[:author] ? epic[:author][:username] : nil
          }
        end

        def post_notification_and_label(epic, epic_info, client)
          message = build_notification_message(epic)

          # Create the discussion/comment
          client.post(
            <<~GQL
              mutation CreateDiscussion {
                  createDiscussion(input: {noteableId: "#{epic[:id]}", body: #{message.inspect}}) {
                      clientMutationId
                      errors
                      note {
                          id
                          body
                      }
                  }
              }
            GQL
          )

          # Apply group label if it exists
          apply_group_label_if_exists(epic, epic_info, client)
        end

        def build_success_result(url, epic, epic_info)
          {
            url: url,
            success: true,
            title: epic[:title],
            author: epic[:author] ? epic[:author][:username] : nil,
            group: epic_info[:group],
            epic_iid: epic_info[:epic_iid]
          }
        end

        def parse_epic_url(url)
          uri = URI.parse(url)
          path_parts = uri.path.split('/')
          groups_index, epics_index = find_url_indices(path_parts, url)
          group_path, epic_iid = extract_group_and_epic_id(path_parts, groups_index, epics_index, url)

          {
            host: "#{uri.scheme}://#{uri.host}",
            group: group_path,
            epic_iid: epic_iid,
            url: url
          }
        end

        def find_url_indices(path_parts, url)
          groups_index = path_parts.index('groups')
          epics_index = path_parts.index('epics')

          raise ArgumentError, "Invalid epic URL format: #{url}" unless groups_index && epics_index && epics_index > groups_index

          [groups_index, epics_index]
        end

        def extract_group_and_epic_id(path_parts, groups_index, epics_index, url)
          group_parts = path_parts[(groups_index + 1)...(epics_index - 1)]
          group_path = group_parts.join('/')
          epic_iid = path_parts[epics_index + 1]

          validate_extracted_parts(group_path, epic_iid, url)
          [group_path, epic_iid]
        end

        def validate_extracted_parts(group_path, epic_iid, url)
          return if group_path && epic_iid&.match?(/^\d+$/)

          raise ArgumentError, "Could not extract group and epic IID from URL: #{url}"
        end

        def work_items_client_for_group(group)
          client_class = dry_run ? GitlabClient::WorkItemsDryClient : GitlabClient::WorkItemsClient
          client_class.new(token: token, project: nil, group: group)
        end

        def group_labels_client_for_group(group)
          GitlabClient::GroupLabelsClient.new(token: token, group: group)
        end

        def apply_group_label_if_exists(epic, epic_info, client)
          group_labels_client = group_labels_client_for_group(epic_info[:group])
          label_ids = ids_for_group_labels([FEATURE_READINESS_CANDIDATE_LABEL], group_labels_client)

          if label_ids.empty?
            puts "Warning: Group label '#{FEATURE_READINESS_CANDIDATE_LABEL}' not found in group #{epic_info[:group]}. Skipping label application."
            return
          end

          puts "Applying group label '#{FEATURE_READINESS_CANDIDATE_LABEL}' to epic..." unless dry_run
          add_labels(label_ids, epic[:id], client) unless dry_run
        rescue StandardError => e
          puts "Warning: Could not apply group label '#{FEATURE_READINESS_CANDIDATE_LABEL}': #{e.message}"
        end

        def build_notification_message(epic)
          author_mention = build_author_mention(epic)

          message = custom_message || default_message_template

          # Replace placeholders in the message
          message = message.gsub('{author}', author_mention)
                          .gsub('{epic_title}', epic[:title])
                          .gsub('{epic_url}', epic[:webUrl])

          # Add the notification ID for tracking
          "#{FEATURE_READINESS_NOTIFICATION_ID}\n#{message}"
        end

        def build_author_mention(epic)
          return '' unless epic[:author] && epic[:author][:username]

          "@#{epic[:author][:username]}"
        end

        def default_message_template
          <<~MESSAGE
            ### Platform Readiness Enablement Process (PREP) Potentially Required

            {author} The feature **"{epic_title}"** has been selected as a potential candidate for Platform Readiness Enablement Process.

            ### What is PREP?
            The **Platform Readiness Enablement Process** (PREP) is a comprehensive process
            that guides product teams through evaluating GitLab features for readiness
            across GitLab.com, GitLab Dedicated, and GitLab Self-Managed deployment platforms.

            Please note that a completed and approved PREP is **mandatory** for qualified features to reach GA status.

            ## Next Steps:
             1. Review the [handbook](https://internal.gitlab.com/handbook/product/platforms/feature-readiness-assessment/#when-prep-is-required) to confirm if you feature qualifies for PREP.
             2. If it does, start the assessment by following the guidelines in the [documentation](https://gitlab.com/gitlab-org/architecture/readiness).
             3. Reach out to the Feature Readiness team on Slack at `#g_feature_readiness` if you have questions.

            Please react with ✅ if this classification is relevant or ❌ if not relevant to help improve our system.

          MESSAGE
        end

        def print_summary
          successful, failed = calculate_summary_counts
          print_summary_header
          print_summary_stats(successful, failed)
          print_failed_urls(failed) if failed.positive?
          print_completion_message
        end

        def calculate_summary_counts
          successful = results.count { |r| r[:success] }
          failed = results.count { |r| !r[:success] }
          [successful, failed]
        end

        def print_summary_header
          puts "=" * 50
          puts "SUMMARY"
          puts "=" * 50
        end

        def print_summary_stats(successful, failed)
          puts "Total processed: #{results.size}"
          puts "Successful: #{successful}"
          puts "Failed: #{failed}"
        end

        def print_failed_urls(_failed)
          puts
          puts "Failed URLs:"
          results.select { |r| !r[:success] }.each do |result|
            puts "  - #{result[:url]}: #{result[:error]}"
          end
        end

        def print_completion_message
          puts
          message = dry_run ? "Dry run completed - no notifications were sent or labels added." : "Notification process completed."
          puts message
        end
      end
    end
  end
end
