# frozen_string_literal: true

module GitlabQuality
  module TestTooling
    module FeatureReadiness
      module AnalyzedItems
        class AnalyzedEpic
          include Concerns::WorkItemConcern

          def initialize(epic:, token:, project:, group:, dry_run:)
            @token = token
            @project = project
            @group = group
            @dry_run = dry_run
            @epic = epic
            @analyzed_issues = []
          end

          def analyze
            issue_iids = get_issue_iids(epic, project)
            return if issue_iids.none?

            open_issues = fetch_open_issues(issue_iids)

            puts "\nProcessing epic: #{epic[:webUrl]} with #{open_issues.count} open feature addition issues."

            process_issues(open_issues)
          end

          def process_issues(issues)
            @analyzed_issues.concat(
              issues.map do |issue|
                AnalyzedIssue.new(issue: issue, token: token, project: project, group: group, dry_run: dry_run)
                             .tap(&:analyze)
                             .result
              end
            )
          end

          def result
            {
              epic_iid: epic[:iid],
              epic_id: epic[:id],
              epic_web_url: epic[:webUrl],
              issues: analyzed_issues,
              doc_mrs: doc_mrs,
              feature_spec_mrs: feature_spec_mrs,
              e2e_spec_mrs: e2e_spec_mrs,
              feature_flag_mrs: feature_flag_mrs
            }
          end

          private

          attr_reader :epic, :token, :project, :dry_run, :group, :analyzed_issues

          def work_items_client
            @work_items_client ||= (dry_run ? GitlabClient::WorkItemsDryClient : GitlabClient::WorkItemsClient).new(token: token, project: project, group: group)
          end

          def issue_client
            @issue_client ||= (dry_run ? GitlabClient::IssuesDryClient : GitlabClient::IssuesClient).new(token: token, project: project)
          end

          def fetch_open_issues(issue_iids)
            issue_client.find_issues(
              options: {
                iids: issue_iids,
                labels: Evaluation::BASE_LABELS_FOR_SEARCH,
                state: 'opened'
              }
            )
          end

          def doc_mrs
            analyzed_issues.flat_map { |issue| issue[:doc_mrs].any? ? issue[:doc_mrs] : [] }
          end

          def feature_spec_mrs
            analyzed_issues.flat_map { |issue| issue[:feature_spec_mrs].any? ? issue[:feature_spec_mrs] : [] }
          end

          def e2e_spec_mrs
            analyzed_issues.flat_map { |issue| issue[:e2e_spec_mrs].any? ? issue[:e2e_spec_mrs] : [] }
          end

          def feature_flag_mrs
            analyzed_issues.flat_map { |issue| issue[:feature_flag_mrs].any? ? issue[:feature_flag_mrs] : [] }
          end
        end
      end
    end
  end
end
