# frozen_string_literal: true

module GitlabQuality
  module TestTooling
    module CodeCoverage
      class CoverageData
        RESPONSIBLE = 'responsible'
        DEPENDENT = 'dependent'

        # @param [Hash<String, Hash>] code_coverage_by_source_file Source file
        #   mapped to test coverage data
        # @param [Hash<String, Array<String>>] source_file_to_tests Source files
        #   mapped to all test files testing them
        # @param [Hash<String, Array<String>>] tests_to_feature_categories Test files
        #   mapped to all feature categories they belong to
        # @param [Hash<String, Hash>] feature_categories_to_teams Mapping of feature categories
        #   to teams (i.e., groups, stages, sections)
        # @param [Hash<String, String>] source_file_types Mapping of source files
        #   to their types (frontend, backend, etc.)
        # @param [Hash<String, String>] test_classifications Mapping of test files
        #   to their responsibility classification (responsible or dependent)
        def initialize(
          code_coverage_by_source_file, source_file_to_tests, tests_to_feature_categories,
          feature_categories_to_teams, source_file_types = {}, test_classifications = {}
        )
          @code_coverage_by_source_file = code_coverage_by_source_file
          @source_file_to_tests = source_file_to_tests
          @tests_to_feature_categories = tests_to_feature_categories
          @feature_categories_to_teams = feature_categories_to_teams
          @source_file_types = source_file_types
          @test_classifications = test_classifications
        end

        # @return [Array<Hash<Symbol, String>>] Mapping of column name to row
        #   value
        # @example Return value
        #   [
        #     {
        #       file: "app/channels/application_cable/channel.rb"
        #       line_coverage: 100.0
        #       branch_coverage: 95.0
        #       function_coverage: 100.0
        #       source_file_type: "backend"
        #       is_responsible: true
        #       is_dependent: false
        #       feature_category: "team_planning"
        #       group: "project_management"
        #       stage: "plan"
        #       section: "dev"
        #     },
        #     ...
        #   ]
        def as_db_table
          all_files.flat_map { |file| records_for_file(file) }
        end

        private

        def records_for_file(file)
          base_data = base_data_for(file)
          feature_categories_with_flags = feature_categories_with_responsibility_flags_for(file)

          if feature_categories_with_flags.empty?
            base_data.merge(no_owner_info).merge(is_responsible: nil, is_dependent: nil)
          else
            feature_categories_with_flags.map do |feature_category, flags|
              base_data.merge(owner_info(feature_category)).merge(
                is_responsible: flags[:is_responsible],
                is_dependent: flags[:is_dependent]
              )
            end
          end
        end

        def base_data_for(file)
          coverage_data = @code_coverage_by_source_file[file]

          {
            file: file,
            line_coverage: coverage_data&.dig(:percentage),
            branch_coverage: coverage_data&.dig(:branch_percentage),
            function_coverage: coverage_data&.dig(:function_percentage),
            source_file_type: @source_file_types[file] || 'other'
          }
        end

        def no_owner_info
          {
            feature_category: nil,
            group: nil,
            stage: nil,
            section: nil
          }
        end

        def owner_info(feature_category)
          owner_info = @feature_categories_to_teams[feature_category]

          {
            feature_category: feature_category,
            group: owner_info&.dig(:group),
            stage: owner_info&.dig(:stage),
            section: owner_info&.dig(:section)
          }
        end

        # Returns a hash of feature_category => { is_responsible: bool, is_dependent: bool }
        # for a given source file. A feature category can have both flags true if it has
        # both unit tests (responsible) and integration/E2E tests (dependent).
        def feature_categories_with_responsibility_flags_for(file)
          test_files = @source_file_to_tests[file] || []
          return {} if test_files.empty?

          test_files.each_with_object({}) do |test_file, feature_category_to_flags|
            feature_categories = @tests_to_feature_categories[test_file] || []
            classification = @test_classifications[test_file]

            feature_categories.each do |feature_category|
              feature_category_to_flags[feature_category] ||= { is_responsible: false, is_dependent: false }

              case classification
              when RESPONSIBLE
                feature_category_to_flags[feature_category][:is_responsible] = true
              when DEPENDENT
                feature_category_to_flags[feature_category][:is_dependent] = true
              end
            end
          end
        end

        def all_files
          @all_files ||= @code_coverage_by_source_file.keys
        end
      end
    end
  end
end
