﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/docdb/DocDB_EXPORTS.h>
#include <aws/docdb/model/CertificateDetails.h>
#include <aws/docdb/model/DBInstanceStatusInfo.h>
#include <aws/docdb/model/DBSubnetGroup.h>
#include <aws/docdb/model/Endpoint.h>
#include <aws/docdb/model/PendingModifiedValues.h>
#include <aws/docdb/model/VpcSecurityGroupMembership.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace DocDB {
namespace Model {

/**
 * <p>Detailed information about an instance. </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/docdb-2014-10-31/DBInstance">AWS
 * API Reference</a></p>
 */
class DBInstance {
 public:
  AWS_DOCDB_API DBInstance() = default;
  AWS_DOCDB_API DBInstance(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_DOCDB_API DBInstance& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_DOCDB_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_DOCDB_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>Contains a user-provided database identifier. This identifier is the unique
   * key that identifies an instance.</p>
   */
  inline const Aws::String& GetDBInstanceIdentifier() const { return m_dBInstanceIdentifier; }
  inline bool DBInstanceIdentifierHasBeenSet() const { return m_dBInstanceIdentifierHasBeenSet; }
  template <typename DBInstanceIdentifierT = Aws::String>
  void SetDBInstanceIdentifier(DBInstanceIdentifierT&& value) {
    m_dBInstanceIdentifierHasBeenSet = true;
    m_dBInstanceIdentifier = std::forward<DBInstanceIdentifierT>(value);
  }
  template <typename DBInstanceIdentifierT = Aws::String>
  DBInstance& WithDBInstanceIdentifier(DBInstanceIdentifierT&& value) {
    SetDBInstanceIdentifier(std::forward<DBInstanceIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains the name of the compute and memory capacity class of the
   * instance.</p>
   */
  inline const Aws::String& GetDBInstanceClass() const { return m_dBInstanceClass; }
  inline bool DBInstanceClassHasBeenSet() const { return m_dBInstanceClassHasBeenSet; }
  template <typename DBInstanceClassT = Aws::String>
  void SetDBInstanceClass(DBInstanceClassT&& value) {
    m_dBInstanceClassHasBeenSet = true;
    m_dBInstanceClass = std::forward<DBInstanceClassT>(value);
  }
  template <typename DBInstanceClassT = Aws::String>
  DBInstance& WithDBInstanceClass(DBInstanceClassT&& value) {
    SetDBInstanceClass(std::forward<DBInstanceClassT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides the name of the database engine to be used for this instance.</p>
   */
  inline const Aws::String& GetEngine() const { return m_engine; }
  inline bool EngineHasBeenSet() const { return m_engineHasBeenSet; }
  template <typename EngineT = Aws::String>
  void SetEngine(EngineT&& value) {
    m_engineHasBeenSet = true;
    m_engine = std::forward<EngineT>(value);
  }
  template <typename EngineT = Aws::String>
  DBInstance& WithEngine(EngineT&& value) {
    SetEngine(std::forward<EngineT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the current state of this database.</p>
   */
  inline const Aws::String& GetDBInstanceStatus() const { return m_dBInstanceStatus; }
  inline bool DBInstanceStatusHasBeenSet() const { return m_dBInstanceStatusHasBeenSet; }
  template <typename DBInstanceStatusT = Aws::String>
  void SetDBInstanceStatus(DBInstanceStatusT&& value) {
    m_dBInstanceStatusHasBeenSet = true;
    m_dBInstanceStatus = std::forward<DBInstanceStatusT>(value);
  }
  template <typename DBInstanceStatusT = Aws::String>
  DBInstance& WithDBInstanceStatus(DBInstanceStatusT&& value) {
    SetDBInstanceStatus(std::forward<DBInstanceStatusT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the connection endpoint.</p>
   */
  inline const Endpoint& GetEndpoint() const { return m_endpoint; }
  inline bool EndpointHasBeenSet() const { return m_endpointHasBeenSet; }
  template <typename EndpointT = Endpoint>
  void SetEndpoint(EndpointT&& value) {
    m_endpointHasBeenSet = true;
    m_endpoint = std::forward<EndpointT>(value);
  }
  template <typename EndpointT = Endpoint>
  DBInstance& WithEndpoint(EndpointT&& value) {
    SetEndpoint(std::forward<EndpointT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides the date and time that the instance was created.</p>
   */
  inline const Aws::Utils::DateTime& GetInstanceCreateTime() const { return m_instanceCreateTime; }
  inline bool InstanceCreateTimeHasBeenSet() const { return m_instanceCreateTimeHasBeenSet; }
  template <typename InstanceCreateTimeT = Aws::Utils::DateTime>
  void SetInstanceCreateTime(InstanceCreateTimeT&& value) {
    m_instanceCreateTimeHasBeenSet = true;
    m_instanceCreateTime = std::forward<InstanceCreateTimeT>(value);
  }
  template <typename InstanceCreateTimeT = Aws::Utils::DateTime>
  DBInstance& WithInstanceCreateTime(InstanceCreateTimeT&& value) {
    SetInstanceCreateTime(std::forward<InstanceCreateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Specifies the daily time range during which automated backups are created if
   * automated backups are enabled, as determined by the
   * <code>BackupRetentionPeriod</code>. </p>
   */
  inline const Aws::String& GetPreferredBackupWindow() const { return m_preferredBackupWindow; }
  inline bool PreferredBackupWindowHasBeenSet() const { return m_preferredBackupWindowHasBeenSet; }
  template <typename PreferredBackupWindowT = Aws::String>
  void SetPreferredBackupWindow(PreferredBackupWindowT&& value) {
    m_preferredBackupWindowHasBeenSet = true;
    m_preferredBackupWindow = std::forward<PreferredBackupWindowT>(value);
  }
  template <typename PreferredBackupWindowT = Aws::String>
  DBInstance& WithPreferredBackupWindow(PreferredBackupWindowT&& value) {
    SetPreferredBackupWindow(std::forward<PreferredBackupWindowT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the number of days for which automatic snapshots are retained.</p>
   */
  inline int GetBackupRetentionPeriod() const { return m_backupRetentionPeriod; }
  inline bool BackupRetentionPeriodHasBeenSet() const { return m_backupRetentionPeriodHasBeenSet; }
  inline void SetBackupRetentionPeriod(int value) {
    m_backupRetentionPeriodHasBeenSet = true;
    m_backupRetentionPeriod = value;
  }
  inline DBInstance& WithBackupRetentionPeriod(int value) {
    SetBackupRetentionPeriod(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides a list of VPC security group elements that the instance belongs
   * to.</p>
   */
  inline const Aws::Vector<VpcSecurityGroupMembership>& GetVpcSecurityGroups() const { return m_vpcSecurityGroups; }
  inline bool VpcSecurityGroupsHasBeenSet() const { return m_vpcSecurityGroupsHasBeenSet; }
  template <typename VpcSecurityGroupsT = Aws::Vector<VpcSecurityGroupMembership>>
  void SetVpcSecurityGroups(VpcSecurityGroupsT&& value) {
    m_vpcSecurityGroupsHasBeenSet = true;
    m_vpcSecurityGroups = std::forward<VpcSecurityGroupsT>(value);
  }
  template <typename VpcSecurityGroupsT = Aws::Vector<VpcSecurityGroupMembership>>
  DBInstance& WithVpcSecurityGroups(VpcSecurityGroupsT&& value) {
    SetVpcSecurityGroups(std::forward<VpcSecurityGroupsT>(value));
    return *this;
  }
  template <typename VpcSecurityGroupsT = VpcSecurityGroupMembership>
  DBInstance& AddVpcSecurityGroups(VpcSecurityGroupsT&& value) {
    m_vpcSecurityGroupsHasBeenSet = true;
    m_vpcSecurityGroups.emplace_back(std::forward<VpcSecurityGroupsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the name of the Availability Zone that the instance is located
   * in.</p>
   */
  inline const Aws::String& GetAvailabilityZone() const { return m_availabilityZone; }
  inline bool AvailabilityZoneHasBeenSet() const { return m_availabilityZoneHasBeenSet; }
  template <typename AvailabilityZoneT = Aws::String>
  void SetAvailabilityZone(AvailabilityZoneT&& value) {
    m_availabilityZoneHasBeenSet = true;
    m_availabilityZone = std::forward<AvailabilityZoneT>(value);
  }
  template <typename AvailabilityZoneT = Aws::String>
  DBInstance& WithAvailabilityZone(AvailabilityZoneT&& value) {
    SetAvailabilityZone(std::forward<AvailabilityZoneT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies information on the subnet group that is associated with the
   * instance, including the name, description, and subnets in the subnet group.</p>
   */
  inline const DBSubnetGroup& GetDBSubnetGroup() const { return m_dBSubnetGroup; }
  inline bool DBSubnetGroupHasBeenSet() const { return m_dBSubnetGroupHasBeenSet; }
  template <typename DBSubnetGroupT = DBSubnetGroup>
  void SetDBSubnetGroup(DBSubnetGroupT&& value) {
    m_dBSubnetGroupHasBeenSet = true;
    m_dBSubnetGroup = std::forward<DBSubnetGroupT>(value);
  }
  template <typename DBSubnetGroupT = DBSubnetGroup>
  DBInstance& WithDBSubnetGroup(DBSubnetGroupT&& value) {
    SetDBSubnetGroup(std::forward<DBSubnetGroupT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the weekly time range during which system maintenance can occur, in
   * Universal Coordinated Time (UTC).</p>
   */
  inline const Aws::String& GetPreferredMaintenanceWindow() const { return m_preferredMaintenanceWindow; }
  inline bool PreferredMaintenanceWindowHasBeenSet() const { return m_preferredMaintenanceWindowHasBeenSet; }
  template <typename PreferredMaintenanceWindowT = Aws::String>
  void SetPreferredMaintenanceWindow(PreferredMaintenanceWindowT&& value) {
    m_preferredMaintenanceWindowHasBeenSet = true;
    m_preferredMaintenanceWindow = std::forward<PreferredMaintenanceWindowT>(value);
  }
  template <typename PreferredMaintenanceWindowT = Aws::String>
  DBInstance& WithPreferredMaintenanceWindow(PreferredMaintenanceWindowT&& value) {
    SetPreferredMaintenanceWindow(std::forward<PreferredMaintenanceWindowT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies that changes to the instance are pending. This element is included
   * only when changes are pending. Specific changes are identified by
   * subelements.</p>
   */
  inline const PendingModifiedValues& GetPendingModifiedValues() const { return m_pendingModifiedValues; }
  inline bool PendingModifiedValuesHasBeenSet() const { return m_pendingModifiedValuesHasBeenSet; }
  template <typename PendingModifiedValuesT = PendingModifiedValues>
  void SetPendingModifiedValues(PendingModifiedValuesT&& value) {
    m_pendingModifiedValuesHasBeenSet = true;
    m_pendingModifiedValues = std::forward<PendingModifiedValuesT>(value);
  }
  template <typename PendingModifiedValuesT = PendingModifiedValues>
  DBInstance& WithPendingModifiedValues(PendingModifiedValuesT&& value) {
    SetPendingModifiedValues(std::forward<PendingModifiedValuesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the latest time to which a database can be restored with
   * point-in-time restore.</p>
   */
  inline const Aws::Utils::DateTime& GetLatestRestorableTime() const { return m_latestRestorableTime; }
  inline bool LatestRestorableTimeHasBeenSet() const { return m_latestRestorableTimeHasBeenSet; }
  template <typename LatestRestorableTimeT = Aws::Utils::DateTime>
  void SetLatestRestorableTime(LatestRestorableTimeT&& value) {
    m_latestRestorableTimeHasBeenSet = true;
    m_latestRestorableTime = std::forward<LatestRestorableTimeT>(value);
  }
  template <typename LatestRestorableTimeT = Aws::Utils::DateTime>
  DBInstance& WithLatestRestorableTime(LatestRestorableTimeT&& value) {
    SetLatestRestorableTime(std::forward<LatestRestorableTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates the database engine version.</p>
   */
  inline const Aws::String& GetEngineVersion() const { return m_engineVersion; }
  inline bool EngineVersionHasBeenSet() const { return m_engineVersionHasBeenSet; }
  template <typename EngineVersionT = Aws::String>
  void SetEngineVersion(EngineVersionT&& value) {
    m_engineVersionHasBeenSet = true;
    m_engineVersion = std::forward<EngineVersionT>(value);
  }
  template <typename EngineVersionT = Aws::String>
  DBInstance& WithEngineVersion(EngineVersionT&& value) {
    SetEngineVersion(std::forward<EngineVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Does not apply. This parameter does not apply to Amazon DocumentDB. Amazon
   * DocumentDB does not perform minor version upgrades regardless of the value
   * set.</p>
   */
  inline bool GetAutoMinorVersionUpgrade() const { return m_autoMinorVersionUpgrade; }
  inline bool AutoMinorVersionUpgradeHasBeenSet() const { return m_autoMinorVersionUpgradeHasBeenSet; }
  inline void SetAutoMinorVersionUpgrade(bool value) {
    m_autoMinorVersionUpgradeHasBeenSet = true;
    m_autoMinorVersionUpgrade = value;
  }
  inline DBInstance& WithAutoMinorVersionUpgrade(bool value) {
    SetAutoMinorVersionUpgrade(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Not supported. Amazon DocumentDB does not currently support public endpoints.
   * The value of <code>PubliclyAccessible</code> is always <code>false</code>.</p>
   */
  inline bool GetPubliclyAccessible() const { return m_publiclyAccessible; }
  inline bool PubliclyAccessibleHasBeenSet() const { return m_publiclyAccessibleHasBeenSet; }
  inline void SetPubliclyAccessible(bool value) {
    m_publiclyAccessibleHasBeenSet = true;
    m_publiclyAccessible = value;
  }
  inline DBInstance& WithPubliclyAccessible(bool value) {
    SetPubliclyAccessible(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of a read replica. If the instance is not a read replica, this is
   * blank.</p>
   */
  inline const Aws::Vector<DBInstanceStatusInfo>& GetStatusInfos() const { return m_statusInfos; }
  inline bool StatusInfosHasBeenSet() const { return m_statusInfosHasBeenSet; }
  template <typename StatusInfosT = Aws::Vector<DBInstanceStatusInfo>>
  void SetStatusInfos(StatusInfosT&& value) {
    m_statusInfosHasBeenSet = true;
    m_statusInfos = std::forward<StatusInfosT>(value);
  }
  template <typename StatusInfosT = Aws::Vector<DBInstanceStatusInfo>>
  DBInstance& WithStatusInfos(StatusInfosT&& value) {
    SetStatusInfos(std::forward<StatusInfosT>(value));
    return *this;
  }
  template <typename StatusInfosT = DBInstanceStatusInfo>
  DBInstance& AddStatusInfos(StatusInfosT&& value) {
    m_statusInfosHasBeenSet = true;
    m_statusInfos.emplace_back(std::forward<StatusInfosT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains the name of the cluster that the instance is a member of if the
   * instance is a member of a cluster.</p>
   */
  inline const Aws::String& GetDBClusterIdentifier() const { return m_dBClusterIdentifier; }
  inline bool DBClusterIdentifierHasBeenSet() const { return m_dBClusterIdentifierHasBeenSet; }
  template <typename DBClusterIdentifierT = Aws::String>
  void SetDBClusterIdentifier(DBClusterIdentifierT&& value) {
    m_dBClusterIdentifierHasBeenSet = true;
    m_dBClusterIdentifier = std::forward<DBClusterIdentifierT>(value);
  }
  template <typename DBClusterIdentifierT = Aws::String>
  DBInstance& WithDBClusterIdentifier(DBClusterIdentifierT&& value) {
    SetDBClusterIdentifier(std::forward<DBClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether or not the instance is encrypted.</p>
   */
  inline bool GetStorageEncrypted() const { return m_storageEncrypted; }
  inline bool StorageEncryptedHasBeenSet() const { return m_storageEncryptedHasBeenSet; }
  inline void SetStorageEncrypted(bool value) {
    m_storageEncryptedHasBeenSet = true;
    m_storageEncrypted = value;
  }
  inline DBInstance& WithStorageEncrypted(bool value) {
    SetStorageEncrypted(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> If <code>StorageEncrypted</code> is <code>true</code>, the KMS key
   * identifier for the encrypted instance. </p>
   */
  inline const Aws::String& GetKmsKeyId() const { return m_kmsKeyId; }
  inline bool KmsKeyIdHasBeenSet() const { return m_kmsKeyIdHasBeenSet; }
  template <typename KmsKeyIdT = Aws::String>
  void SetKmsKeyId(KmsKeyIdT&& value) {
    m_kmsKeyIdHasBeenSet = true;
    m_kmsKeyId = std::forward<KmsKeyIdT>(value);
  }
  template <typename KmsKeyIdT = Aws::String>
  DBInstance& WithKmsKeyId(KmsKeyIdT&& value) {
    SetKmsKeyId(std::forward<KmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services Region-unique, immutable identifier for the instance.
   * This identifier is found in CloudTrail log entries whenever the KMS key for the
   * instance is accessed.</p>
   */
  inline const Aws::String& GetDbiResourceId() const { return m_dbiResourceId; }
  inline bool DbiResourceIdHasBeenSet() const { return m_dbiResourceIdHasBeenSet; }
  template <typename DbiResourceIdT = Aws::String>
  void SetDbiResourceId(DbiResourceIdT&& value) {
    m_dbiResourceIdHasBeenSet = true;
    m_dbiResourceId = std::forward<DbiResourceIdT>(value);
  }
  template <typename DbiResourceIdT = Aws::String>
  DBInstance& WithDbiResourceId(DbiResourceIdT&& value) {
    SetDbiResourceId(std::forward<DbiResourceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the CA certificate for this DB instance.</p>
   */
  inline const Aws::String& GetCACertificateIdentifier() const { return m_cACertificateIdentifier; }
  inline bool CACertificateIdentifierHasBeenSet() const { return m_cACertificateIdentifierHasBeenSet; }
  template <typename CACertificateIdentifierT = Aws::String>
  void SetCACertificateIdentifier(CACertificateIdentifierT&& value) {
    m_cACertificateIdentifierHasBeenSet = true;
    m_cACertificateIdentifier = std::forward<CACertificateIdentifierT>(value);
  }
  template <typename CACertificateIdentifierT = Aws::String>
  DBInstance& WithCACertificateIdentifier(CACertificateIdentifierT&& value) {
    SetCACertificateIdentifier(std::forward<CACertificateIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that indicates whether to copy tags from the DB instance to snapshots
   * of the DB instance. By default, tags are not copied.</p>
   */
  inline bool GetCopyTagsToSnapshot() const { return m_copyTagsToSnapshot; }
  inline bool CopyTagsToSnapshotHasBeenSet() const { return m_copyTagsToSnapshotHasBeenSet; }
  inline void SetCopyTagsToSnapshot(bool value) {
    m_copyTagsToSnapshotHasBeenSet = true;
    m_copyTagsToSnapshot = value;
  }
  inline DBInstance& WithCopyTagsToSnapshot(bool value) {
    SetCopyTagsToSnapshot(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that specifies the order in which an Amazon DocumentDB replica is
   * promoted to the primary instance after a failure of the existing primary
   * instance.</p>
   */
  inline int GetPromotionTier() const { return m_promotionTier; }
  inline bool PromotionTierHasBeenSet() const { return m_promotionTierHasBeenSet; }
  inline void SetPromotionTier(int value) {
    m_promotionTierHasBeenSet = true;
    m_promotionTier = value;
  }
  inline DBInstance& WithPromotionTier(int value) {
    SetPromotionTier(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) for the instance.</p>
   */
  inline const Aws::String& GetDBInstanceArn() const { return m_dBInstanceArn; }
  inline bool DBInstanceArnHasBeenSet() const { return m_dBInstanceArnHasBeenSet; }
  template <typename DBInstanceArnT = Aws::String>
  void SetDBInstanceArn(DBInstanceArnT&& value) {
    m_dBInstanceArnHasBeenSet = true;
    m_dBInstanceArn = std::forward<DBInstanceArnT>(value);
  }
  template <typename DBInstanceArnT = Aws::String>
  DBInstance& WithDBInstanceArn(DBInstanceArnT&& value) {
    SetDBInstanceArn(std::forward<DBInstanceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of log types that this instance is configured to export to CloudWatch
   * Logs.</p>
   */
  inline const Aws::Vector<Aws::String>& GetEnabledCloudwatchLogsExports() const { return m_enabledCloudwatchLogsExports; }
  inline bool EnabledCloudwatchLogsExportsHasBeenSet() const { return m_enabledCloudwatchLogsExportsHasBeenSet; }
  template <typename EnabledCloudwatchLogsExportsT = Aws::Vector<Aws::String>>
  void SetEnabledCloudwatchLogsExports(EnabledCloudwatchLogsExportsT&& value) {
    m_enabledCloudwatchLogsExportsHasBeenSet = true;
    m_enabledCloudwatchLogsExports = std::forward<EnabledCloudwatchLogsExportsT>(value);
  }
  template <typename EnabledCloudwatchLogsExportsT = Aws::Vector<Aws::String>>
  DBInstance& WithEnabledCloudwatchLogsExports(EnabledCloudwatchLogsExportsT&& value) {
    SetEnabledCloudwatchLogsExports(std::forward<EnabledCloudwatchLogsExportsT>(value));
    return *this;
  }
  template <typename EnabledCloudwatchLogsExportsT = Aws::String>
  DBInstance& AddEnabledCloudwatchLogsExports(EnabledCloudwatchLogsExportsT&& value) {
    m_enabledCloudwatchLogsExportsHasBeenSet = true;
    m_enabledCloudwatchLogsExports.emplace_back(std::forward<EnabledCloudwatchLogsExportsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The details of the DB instance's server certificate.</p>
   */
  inline const CertificateDetails& GetCertificateDetails() const { return m_certificateDetails; }
  inline bool CertificateDetailsHasBeenSet() const { return m_certificateDetailsHasBeenSet; }
  template <typename CertificateDetailsT = CertificateDetails>
  void SetCertificateDetails(CertificateDetailsT&& value) {
    m_certificateDetailsHasBeenSet = true;
    m_certificateDetails = std::forward<CertificateDetailsT>(value);
  }
  template <typename CertificateDetailsT = CertificateDetails>
  DBInstance& WithCertificateDetails(CertificateDetailsT&& value) {
    SetCertificateDetails(std::forward<CertificateDetailsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Set to <code>true</code> if Amazon RDS Performance Insights is enabled for
   * the DB instance, and otherwise <code>false</code>.</p>
   */
  inline bool GetPerformanceInsightsEnabled() const { return m_performanceInsightsEnabled; }
  inline bool PerformanceInsightsEnabledHasBeenSet() const { return m_performanceInsightsEnabledHasBeenSet; }
  inline void SetPerformanceInsightsEnabled(bool value) {
    m_performanceInsightsEnabledHasBeenSet = true;
    m_performanceInsightsEnabled = value;
  }
  inline DBInstance& WithPerformanceInsightsEnabled(bool value) {
    SetPerformanceInsightsEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The KMS key identifier for encryption of Performance Insights data. The KMS
   * key ID is the Amazon Resource Name (ARN), KMS key identifier, or the KMS key
   * alias for the KMS encryption key.</p>
   */
  inline const Aws::String& GetPerformanceInsightsKMSKeyId() const { return m_performanceInsightsKMSKeyId; }
  inline bool PerformanceInsightsKMSKeyIdHasBeenSet() const { return m_performanceInsightsKMSKeyIdHasBeenSet; }
  template <typename PerformanceInsightsKMSKeyIdT = Aws::String>
  void SetPerformanceInsightsKMSKeyId(PerformanceInsightsKMSKeyIdT&& value) {
    m_performanceInsightsKMSKeyIdHasBeenSet = true;
    m_performanceInsightsKMSKeyId = std::forward<PerformanceInsightsKMSKeyIdT>(value);
  }
  template <typename PerformanceInsightsKMSKeyIdT = Aws::String>
  DBInstance& WithPerformanceInsightsKMSKeyId(PerformanceInsightsKMSKeyIdT&& value) {
    SetPerformanceInsightsKMSKeyId(std::forward<PerformanceInsightsKMSKeyIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBInstanceIdentifier;

  Aws::String m_dBInstanceClass;

  Aws::String m_engine;

  Aws::String m_dBInstanceStatus;

  Endpoint m_endpoint;

  Aws::Utils::DateTime m_instanceCreateTime{};

  Aws::String m_preferredBackupWindow;

  int m_backupRetentionPeriod{0};

  Aws::Vector<VpcSecurityGroupMembership> m_vpcSecurityGroups;

  Aws::String m_availabilityZone;

  DBSubnetGroup m_dBSubnetGroup;

  Aws::String m_preferredMaintenanceWindow;

  PendingModifiedValues m_pendingModifiedValues;

  Aws::Utils::DateTime m_latestRestorableTime{};

  Aws::String m_engineVersion;

  bool m_autoMinorVersionUpgrade{false};

  bool m_publiclyAccessible{false};

  Aws::Vector<DBInstanceStatusInfo> m_statusInfos;

  Aws::String m_dBClusterIdentifier;

  bool m_storageEncrypted{false};

  Aws::String m_kmsKeyId;

  Aws::String m_dbiResourceId;

  Aws::String m_cACertificateIdentifier;

  bool m_copyTagsToSnapshot{false};

  int m_promotionTier{0};

  Aws::String m_dBInstanceArn;

  Aws::Vector<Aws::String> m_enabledCloudwatchLogsExports;

  CertificateDetails m_certificateDetails;

  bool m_performanceInsightsEnabled{false};

  Aws::String m_performanceInsightsKMSKeyId;
  bool m_dBInstanceIdentifierHasBeenSet = false;
  bool m_dBInstanceClassHasBeenSet = false;
  bool m_engineHasBeenSet = false;
  bool m_dBInstanceStatusHasBeenSet = false;
  bool m_endpointHasBeenSet = false;
  bool m_instanceCreateTimeHasBeenSet = false;
  bool m_preferredBackupWindowHasBeenSet = false;
  bool m_backupRetentionPeriodHasBeenSet = false;
  bool m_vpcSecurityGroupsHasBeenSet = false;
  bool m_availabilityZoneHasBeenSet = false;
  bool m_dBSubnetGroupHasBeenSet = false;
  bool m_preferredMaintenanceWindowHasBeenSet = false;
  bool m_pendingModifiedValuesHasBeenSet = false;
  bool m_latestRestorableTimeHasBeenSet = false;
  bool m_engineVersionHasBeenSet = false;
  bool m_autoMinorVersionUpgradeHasBeenSet = false;
  bool m_publiclyAccessibleHasBeenSet = false;
  bool m_statusInfosHasBeenSet = false;
  bool m_dBClusterIdentifierHasBeenSet = false;
  bool m_storageEncryptedHasBeenSet = false;
  bool m_kmsKeyIdHasBeenSet = false;
  bool m_dbiResourceIdHasBeenSet = false;
  bool m_cACertificateIdentifierHasBeenSet = false;
  bool m_copyTagsToSnapshotHasBeenSet = false;
  bool m_promotionTierHasBeenSet = false;
  bool m_dBInstanceArnHasBeenSet = false;
  bool m_enabledCloudwatchLogsExportsHasBeenSet = false;
  bool m_certificateDetailsHasBeenSet = false;
  bool m_performanceInsightsEnabledHasBeenSet = false;
  bool m_performanceInsightsKMSKeyIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace DocDB
}  // namespace Aws
