#![cfg(feature = "std")]

extern crate roaring;

use roaring::RoaringBitmap;

// Test data from https://github.com/RoaringBitmap/RoaringFormatSpec/tree/master/testdata
static BITMAP_WITHOUT_RUNS: &[u8] = include_bytes!("bitmapwithoutruns.bin");
static BITMAP_WITH_RUNS: &[u8] = include_bytes!("bitmapwithruns.bin");

fn test_data_bitmap() -> RoaringBitmap {
    (0..100)
        .map(|i| i * 1000)
        .chain((100_000..200_000).map(|i| i * 3))
        .chain(700_000..800_000)
        .collect::<RoaringBitmap>()
}

fn serialize_and_deserialize(bitmap: &RoaringBitmap) -> RoaringBitmap {
    let mut buffer = vec![];
    bitmap.serialize_into(&mut buffer).unwrap();
    assert_eq!(buffer.len(), bitmap.serialized_size());
    RoaringBitmap::deserialize_from(&buffer[..]).unwrap()
}

#[test]
fn test_deserialize_without_runs_from_provided_data() {
    assert_eq!(RoaringBitmap::deserialize_from(BITMAP_WITHOUT_RUNS).unwrap(), test_data_bitmap());
}

#[test]
fn test_deserialize_with_runs_from_provided_data() {
    assert_eq!(
        RoaringBitmap::deserialize_from(&mut &BITMAP_WITH_RUNS[..]).unwrap(),
        test_data_bitmap()
    );
}

#[test]
fn test_serialize_into_provided_data() {
    let bitmap = test_data_bitmap();
    let mut buffer = vec![];
    bitmap.serialize_into(&mut buffer).unwrap();
    assert!(BITMAP_WITHOUT_RUNS == &buffer[..]);
}

#[test]
fn test_empty() {
    let original = RoaringBitmap::new();
    let new = serialize_and_deserialize(&original);
    assert_eq!(original, new);
}

#[test]
fn test_one() {
    let original = (1..2).collect::<RoaringBitmap>();
    let new = serialize_and_deserialize(&original);
    assert_eq!(original, new);
}

#[test]
fn test_array() {
    let original = (1000..3000).collect::<RoaringBitmap>();
    let new = serialize_and_deserialize(&original);
    assert_eq!(original, new);
}

#[test]
fn test_array_boundary() {
    let original = (1000..5096).collect::<RoaringBitmap>();
    let new = serialize_and_deserialize(&original);
    assert_eq!(original, new);
}

#[test]
fn test_bitmap_boundary() {
    let original = (1000..5097).collect::<RoaringBitmap>();
    let new = serialize_and_deserialize(&original);
    assert_eq!(original, new);
}

#[test]
fn test_bitmap_high16bits() {
    let mut bitmap = RoaringBitmap::new();
    for i in 0..1 << 16 {
        let value = i << 16;
        bitmap.insert(value);
    }

    let mut buffer = vec![];
    bitmap.serialize_into(&mut buffer).unwrap();

    let new = RoaringBitmap::deserialize_from(&buffer[..]);
    assert!(new.is_ok());
    assert_eq!(bitmap, new.unwrap());
}

#[test]
fn test_bitmap() {
    let original = (1000..6000).collect::<RoaringBitmap>();
    let new = serialize_and_deserialize(&original);
    assert_eq!(original, new);
}

#[test]
fn test_arrays() {
    let original = (1000..3000).chain(70000..74000).collect::<RoaringBitmap>();
    let new = serialize_and_deserialize(&original);
    assert_eq!(original, new);
}

#[test]
fn test_bitmaps() {
    let original = (1000..6000).chain(70000..77000).collect::<RoaringBitmap>();
    let new = serialize_and_deserialize(&original);
    assert_eq!(original, new);
}

#[test]
fn test_mixed() {
    let original = (1000..3000).chain(70000..77000).collect::<RoaringBitmap>();
    let new = serialize_and_deserialize(&original);
    assert_eq!(original, new);
}

#[test]
fn test_strange() {
    const ARRAY: &[u32] = &[
        6619162, 6619180, 6619181, 6619217, 6619218, 6619257, 6619258, 6619259, 6619260, 6619261,
        6619262, 6619263, 6619264, 6619265, 6619266, 6619292, 6619294, 6619322, 6619461, 6619485,
        6619490, 6619500, 6619594, 6619619, 6619620, 6619623, 6619630, 6619632, 6619700, 6619701,
        6619702, 6619703, 6619813, 6619896, 6619967, 6620022, 6620034, 6620038, 6620110, 6620182,
        6620189, 6620194, 6620200, 6620262, 6620289, 6620321, 6620353, 6620354, 6620355, 6620356,
        6620384, 6620403, 6620431, 6620501, 6620529, 6620596, 6620674, 6620691, 6620770, 6620813,
        6620889, 6620890, 6620891, 6620892, 6620893, 6620894, 6620895, 6620928, 6620929, 6620948,
        6621023, 6621035, 6621046, 6621183, 6621224, 6621227, 6621261, 6621274, 6621345, 6621346,
        6621347, 6621348, 6621349, 6621350, 6621351, 6621352, 6621353, 6621354, 6621355, 6621356,
        6621357, 6621358, 6621359, 6621360, 6621361, 6621362, 6621363, 6621364, 6621365, 6621395,
        6621442, 6621457, 6621518, 6621519, 6621520, 6621521, 6621522, 6621523, 6621524, 6621525,
        6621526, 6621527, 6621538, 6621539, 6621540, 6621541, 6621542, 6621543, 6621544, 6621545,
        6621546, 6621547, 6621548, 6621553, 6621666, 6621715, 6621724, 6621738, 6621844, 6621845,
        6621859, 6621875, 6621891, 6621912, 6621964, 6621969, 6621998, 6621999, 6622037, 6622068,
        6622125, 6622126, 6622137, 6622196, 6622327, 6622328, 6622331, 6622333, 6622355, 6622374,
        6622390, 6622410, 6622411, 6622412, 6622417, 6622418, 6622419, 6622420, 6622421, 6622422,
        6622423, 6622424, 6622425, 6622426, 6622427, 6622428, 6622429, 6622430, 6622431, 6622432,
        6622433, 6622434, 6622518, 6622519, 6622551, 6622571, 6622588, 6622597, 6622599, 6622619,
        6622638, 6622703, 6622704, 6622705, 6622714, 6622734, 6622778, 6622833, 6622834, 6622849,
        6622872, 6622891, 6622964, 6622986, 6623000, 6623004, 6623037, 6623053, 6623054, 6623128,
        6623153, 6623233, 6623243, 6623247, 6623249, 6623267, 6623271, 6623302, 6623339, 6623341,
        6623342, 6623350, 6623353, 6623419, 6623437, 6623513, 6623603, 6623632, 6623656, 6623683,
        6623684, 6623685, 6623686, 6623687, 6623688, 6623689, 6623690, 6623691, 6623692, 6623693,
        6623694, 6623765, 6623767, 6623769, 6623776, 6623785, 6623816, 6623824, 6623840, 6623888,
        6623949, 6623974, 6623979, 6624005, 6624019, 6624053, 6624054, 6624069, 6624079, 6624085,
        6624122, 6624140, 6624144, 6624265, 6624275, 6624276, 6624277, 6624278, 6624350, 6624365,
        6624371, 6624379, 6624384, 6624389, 6624390, 6624415, 6624433, 6624483, 6624492, 6624495,
        6624509, 6624537, 6624563, 6624622, 6624628, 6624748, 6624766, 6624801, 6624813, 6624870,
        6624874, 6624887, 6624936, 6624961, 6625015, 6625042, 6625064, 6625065, 6625066, 6625072,
        6625113, 6625114, 6625115, 6625116, 6625117, 6625118, 6625132, 6625209, 6625235, 6625250,
        6625282, 6625308, 6625404, 6625410, 6625411, 6625415, 6625445, 6625472, 6625501, 6625502,
        6625544, 6625578, 6625581, 6625627, 6625654, 6625771, 6625772, 6625773, 6625774, 6625775,
        6625776, 6625829, 6625831, 6625841, 6625863, 6625872, 6625885, 6625925, 6625933, 6625950,
        6626001, 6626026, 6626038, 6626089, 6626148, 6626273, 6626326, 6626327, 6626328, 6626329,
        6626330, 6626331, 6626332, 6626333, 6626334, 6626335, 6626336, 6626337, 6626423, 6626436,
        6626477, 6626570, 6626571, 6626572, 6626573, 6626583, 6626585, 6626607, 6626608, 6626609,
        6626610, 6626611, 6626612, 6626613, 6626614, 6626615, 6626687, 6626688, 6626689, 6626690,
        6626705, 6626751, 6626835, 6626843, 6626891, 6626903, 6626925, 6626999, 6627003, 6627071,
        6627077, 6627085, 6627118, 6627182, 6627183, 6627184, 6627185, 6627186, 6627187, 6627188,
        6627189, 6627190, 6627191, 6627192, 6627234, 6627241, 6627270, 6627304, 6627305, 6627306,
        6627307, 6627308, 6627309, 6627310, 6627311, 6627312, 6627313, 6627314, 6627315, 6627316,
        6627317, 6627332, 6627333, 6627334, 6627335, 6627358, 6627453, 6627507, 6627542, 6627543,
        6627544, 6627545, 6627560, 6627571, 6627580, 6627636, 6627637, 6627638, 6627639, 6627640,
        6627641, 6627642, 6627643, 6627644, 6627645, 6627707, 6627708, 6627709, 6627710, 6627711,
        6627712, 6627713, 6627714, 6627715, 6627716, 6627717, 6627718, 6627746, 6627747, 6627748,
        6627749, 6627760, 6627769, 6627809, 6627861, 6627989, 6627995, 6628007, 6628009, 6628034,
        6628035, 6628036, 6628037, 6628038, 6628039, 6628040, 6628041, 6628105, 6628124, 6628132,
        6628149, 6628228, 6628229, 6628386, 6628387, 6628406, 6628440, 6628503, 6628546, 6628711,
        6628714, 6628715, 6628716, 6628721, 6628760, 6628761, 6628804, 6628920, 6629022, 6629086,
        6629088, 6629100, 6629102, 6629164, 6629167, 6629168, 6629177, 6629180, 6629181, 6629182,
        6629183, 6629204, 6629222, 6629228, 6629245, 6629270, 6629271, 6629272, 6629273, 6629274,
        6629275, 6629276, 6629277, 6629278, 6629279, 6629280, 6629281, 6629282, 6629283, 6629291,
        6629292, 6629345, 6629503, 6629555, 6629556, 6629594, 6629595, 6629596, 6629597, 6629614,
        6629647, 6629648, 6629649, 6629650, 6629673, 6629692, 6629701, 6629713, 6629719, 6629720,
        6629721, 6629722, 6629723, 6629724, 6629725, 6629728, 6629753, 6629786, 6629789, 6629809,
        6629812, 6629819, 6629879, 6629890, 6629980, 6629981, 6629982, 6629983, 6629984, 6629985,
        6629986, 6629987, 6629988, 6629989, 6629990, 6629991, 6629992, 6630012, 6630026, 6630076,
        6630110, 6630190, 6630191, 6630195, 6630221, 6630227, 6630237, 6630245, 6630340, 6630341,
        6630342, 6630343, 6630482, 6630524, 6630531, 6630532, 6630535, 6630543, 6630571, 6630591,
        6630607, 6630611, 6630615, 6630624, 6630634, 6630643, 6630665, 6630666, 6630667, 6630668,
        6630669, 6630670, 6630671, 6630672, 6630673, 6630674, 6630675, 6630676, 6630677, 6630731,
        6630743, 6630745, 6630753, 6630797, 6630849, 6630850, 6630851, 6630852, 6630853, 6630854,
        6630855, 6630856, 6630857, 6630858, 6630859, 6630860, 6630884, 6630893, 6630896, 6630902,
        6630910, 6630918, 6630922, 6630923, 6630979, 6630986, 6630998, 6631010, 6631060, 6631061,
        6631062, 6631063, 6631064, 6631065, 6631066, 6631067, 6631068, 6631069, 6631070, 6631071,
        6631099, 6631130, 6631141, 6631148, 6631158, 6631176, 6631181, 6631183, 6631238, 6631261,
        6631289, 6631294, 6631410, 6631453, 6631458, 6631470, 6631581, 6631603, 6631627, 6631650,
        6631709, 6631711, 6631734, 6631789, 6631793, 6631815, 6631834, 6631897, 6631950, 6631995,
        6631997, 6632005, 6632007, 6632199, 6632223, 6632224, 6632295, 6632365, 6632440, 6632441,
        6632466, 6632473, 6632491, 6632492, 6632495, 6632496, 6632497, 6632498, 6632499, 6632500,
        6632501, 6632502, 6632503, 6632504, 6632505, 6632506, 6632510, 6632514, 6632524, 6632541,
        6632565, 6632646, 6632647, 6632648, 6632649, 6632650, 6632651, 6632652, 6632653, 6632654,
        6632655, 6632662, 6632746, 6632822, 6632831, 6632840, 6632859, 6632901, 6632902, 6632908,
        6632911, 6632912, 6632913, 6632914, 6632915, 6632916, 6632917, 6632918, 6632919, 6632920,
        6632921, 6632924, 6632927, 6632983, 6632987, 6632991, 6632994, 6633069, 6633071, 6633103,
        6633110, 6633126, 6633155, 6633172, 6633249, 6633275, 6633407, 6633446, 6633537, 6633560,
        6633576, 6633680, 6633735, 6633752, 6633791, 6633845, 6633846, 6633847, 6633848, 6633855,
        6633860, 6633867, 6633885, 6633894, 6633898, 6633903, 6633929, 6633980, 6633995, 6634093,
        6634114, 6634167, 6634170, 6634215, 6634270, 6634399, 6634410, 6634546, 6634588, 6634599,
        6634821, 6634822, 6634823, 6634824, 6634825, 6634826, 6634901, 6634911, 6634913, 6634914,
        6634915, 6634920, 6634921, 6634924, 6634996, 6635018, 6635037, 6635082, 6635117, 6635140,
        6635163, 6635165, 6635167, 6635170, 6635172, 6635223, 6635241, 6635295, 6635296, 6635361,
        6635365, 6635416, 6635417, 6635418, 6635419, 6635420, 6635421, 6635422, 6635447, 6635603,
        6635682, 6635732, 6635735, 6635738, 6635765, 6635793, 6635815, 6635832, 6635840, 6635856,
        6636059, 6636060, 6636071, 6636124, 6636125, 6636163, 6636164, 6636165, 6636166, 6636167,
        6636168, 6636169, 6636170, 6636171, 6636172, 6636173, 6636174, 6636212, 6636248, 6636272,
        6636284, 6636367, 6636403, 6636412, 6636414, 6636453, 6636454, 6636455, 6636456, 6636457,
        6636458, 6636459, 6636460, 6636461, 6636462, 6636463, 6636464, 6636465, 6636466, 6636467,
        6636468, 6636469, 6636470, 6636471, 6636472, 6636473, 6636474, 6636475, 6636476, 6636477,
        6636478, 6636479, 6636480, 6636481, 6636482, 6636483, 6636534, 6636666, 6636735, 6636746,
        6636757, 6636797, 6636846, 6636863, 6636864, 6636913, 6636980, 6636981, 6636982, 6636983,
        6636984, 6636985, 6636986, 6636987, 6636988, 6636989, 6636998, 6637054, 6637055, 6637056,
        6637057, 6637058, 6637059, 6637060, 6637061, 6637062, 6637063, 6637065, 6637068, 6637071,
        6637076, 6637113, 6637218, 6637240, 6637429, 6637453, 6637468, 6637526, 6637563, 6637581,
        6637587, 6637615, 6637619, 6637656, 6637680, 6637681, 6637682, 6637713, 6637757, 6637813,
        6637895, 6637899, 6637912, 6637917, 6637948, 6637949, 6638051, 6638052, 6638075, 6638093,
        6638229, 6638250, 6638315, 6638334, 6638335, 6638336, 6638337, 6638372, 6638373, 6638383,
        6638384, 6638386, 6638387, 6638401, 6638415, 6638545, 6638546, 6638560, 6638561, 6638562,
        6638563, 6638564, 6638565, 6638566, 6638567, 6638568, 6638569, 6638570, 6638571, 6638575,
        6638608, 6638609, 6638631, 6638673, 6638675, 6638683, 6638687, 6638694, 6638697, 6638701,
        6638706, 6638735, 6638736, 6638737, 6638738, 6638761, 6638769, 6638809, 6638810, 6638864,
        6638873, 6638874, 6638875, 6638876, 6638877, 6638878, 6638879, 6638880, 6638881, 6638882,
        6638883, 6638936, 6638966, 6639029, 6639058, 6639066, 6639196, 6639236, 6639266, 6639267,
        6639268, 6639269, 6639273, 6639282, 6639293, 6639295, 6639299, 6639303, 6639444, 6639467,
        6639491, 6639560, 6639573, 6639574, 6639575, 6639576, 6639577, 6639578, 6639602, 6639619,
        6639636, 6639677, 6639693, 6639694, 6639709, 6639763, 6639770, 6639810, 6639898, 6639920,
        6639983, 6639991, 6640019, 6640020, 6640021, 6640022, 6640023, 6640024, 6640025, 6640026,
        6640027, 6640028, 6640029, 6640030, 6640031, 6640032, 6640033, 6640034, 6640035, 6640036,
        6640037, 6640081, 6640134, 6640139, 6640140, 6640141, 6640142, 6640143, 6640144, 6640145,
        6640146, 6640147, 6640148, 6640149, 6640150, 6640151, 6640152, 6640153, 6640239, 6640272,
        6640273, 6640274, 6640275, 6640276, 6640282, 6640287, 6640404, 6640419, 6640420, 6640421,
        6640422, 6640442, 6640510, 6640512, 6640527, 6640528, 6640532, 6640566, 6640585, 6640588,
        6640678, 6640708, 6640736, 6640738, 6640767, 6640768, 6640844, 6640856, 6640881, 6640928,
        6641043, 6641082, 6641090, 6641116, 6641117, 6641125, 6641126, 6641127, 6641198, 6641213,
        6641217, 6641222, 6641226, 6641247, 6641326, 6641365, 6641373, 6641394, 6641456, 6641459,
        6641464, 6641515, 6641527, 6641536, 6641572, 6641573, 6641574, 6641575, 6641576, 6641577,
        6641578, 6641579, 6641580, 6641581, 6641582, 6641583, 6641584, 6641585, 6641586, 6641590,
        6641593, 6641604, 6641612, 6641733, 6641734, 6641735, 6641736, 6641737, 6641738, 6641762,
        6641771, 6641793, 6641811, 6641830, 6641862, 6642040, 6642073, 6642083, 6642084, 6642085,
        6642086, 6642087, 6642088, 6642089, 6642090, 6642091, 6642092, 6642093, 6642094, 6642159,
        6642167, 6642172, 6642173, 6642177, 6642183, 6642269, 6642270, 6642271, 6642304, 6642306,
        6642308, 6642338, 6642384, 6642459, 6642545, 6642558, 6642559, 6642560, 6642561, 6642562,
        6642563, 6642564, 6642565, 6642566, 6642567, 6642568, 6642569, 6642570, 6642571, 6642572,
        6642732, 6642752, 6642754, 6642761, 6642849, 6642850, 6643035, 6643047, 6643075, 6643093,
        6643107, 6643113, 6643130, 6643154, 6643161, 6643169, 6643170, 6643171, 6643172, 6643173,
        6643174, 6643175, 6643176, 6643177, 6643178, 6643179, 6643180, 6643181, 6643230, 6643241,
        6643261, 6643351, 6643353, 6643354, 6643355, 6643370, 6643379, 6643537, 6643538, 6643563,
        6643572, 6643655, 6643656, 6643661, 6643662, 6643663, 6643664, 6643665, 6643666, 6643667,
        6643668, 6643669, 6643670, 6643671, 6643672, 6643675, 6643766, 6643806, 6643837, 6643838,
        6643839, 6643879, 6643899, 6643900, 6643902, 6643984, 6644030, 6644032, 6644043, 6644050,
        6644051, 6644052, 6644053, 6644054, 6644055, 6644056, 6644057, 6644058, 6644059, 6644060,
        6644061, 6644066, 6644067, 6644075, 6644085, 6644089, 6644109, 6644154, 6644155, 6644164,
        6644165, 6644172, 6644173, 6644174, 6644175, 6644176, 6644177, 6644178, 6644179, 6644180,
        6644181, 6644182, 6644183, 6644184, 6644185, 6644201, 6644250, 6644260, 6644263, 6644345,
        6644427, 6644428, 6644432, 6644522, 6644551, 6644584, 6644619, 6644620, 6644621, 6644661,
        6644680, 6644681, 6644777, 6644779, 6644785, 6644789, 6644802, 6644803, 6644900, 6644919,
        6644930, 6645031, 6645133, 6645171, 6645197, 6645235, 6645236, 6645237, 6645238, 6645239,
        6645240, 6645241, 6645242, 6645243, 6645244, 6645245, 6645246, 6645247, 6645248, 6645250,
        6645251, 6645267, 6645268, 6645269, 6645270, 6645271, 6645272, 6645273, 6645274, 6645289,
        6645303, 6645307, 6645308, 6645309, 6645310, 6645311, 6645312, 6645313, 6645314, 6645315,
        6645316, 6645317, 6645318, 6645319, 6645344, 6645362, 6645365, 6645412, 6645413, 6645414,
        6645415, 6645416, 6645417, 6645418, 6645419, 6645420, 6645421, 6645422, 6645427, 6645435,
        6645438, 6645458, 6645479, 6645498, 6645499, 6645500, 6645501, 6645523, 6645536, 6645645,
        6645646, 6645647, 6645648, 6645649, 6645650, 6645651, 6645652, 6645653, 6645654, 6645655,
        6645656, 6645657, 6645658, 6645659, 6645660, 6645661, 6645662, 6645663, 6645664, 6645665,
        6645692, 6645709, 6645710, 6645711, 6645712, 6645751, 6645852, 6645908, 6645909, 6645910,
        6645930, 6645931, 6645932, 6645933, 6645935, 6645936, 6645950, 6646040, 6646050, 6646060,
        6646064, 6646065, 6646069, 6646082, 6646103, 6646202, 6646240, 6646241, 6646242, 6646243,
        6646244, 6646245, 6646246, 6646247, 6646248, 6646249, 6646266, 6646267, 6646269, 6646275,
        6646276, 6646277, 6646278, 6646279, 6646280, 6646281, 6646282, 6646283, 6646284, 6646302,
        6646303, 6646304, 6646305, 6646390, 6646391, 6646392, 6646393, 6646394, 6646395, 6646396,
        6646397, 6646398, 6646399, 6646400, 6646414, 6646420, 6646436, 6646473, 6646494, 6646529,
        6646536, 6646537, 6646538, 6646539, 6646540, 6646541, 6646569, 6646570, 6646571, 6646572,
        6646573, 6646574, 6646575, 6646576, 6646577, 6646578, 6646585, 6646661, 6646682, 6646771,
        6646845, 6646855, 6646865, 6646866, 6646867, 6646868, 6646869, 6646870, 6646871, 6646872,
        6646873, 6646874, 6646875, 6646876, 6646877, 6646882, 6646932, 6646941, 6646974, 6647028,
        6647119, 6647124, 6647165, 6647193, 6647207, 6647226, 6647230, 6647268, 6647269, 6647270,
        6647271, 6647322, 6647364, 6647375, 6647489, 6647508, 6647518, 6647560, 6647573, 6647590,
        6647629, 6647770, 6647781, 6647783, 6647903, 6647977, 6647978, 6647979, 6647980, 6647981,
        6647982, 6647983, 6647984, 6647985, 6647986, 6647987, 6647988, 6647989, 6647990, 6647991,
        6647992, 6647993, 6648069, 6648083, 6648084, 6648085, 6648086, 6648111, 6648179, 6648184,
        6648203, 6648218, 6648243, 6648335, 6648342, 6648389, 6648391, 6648481, 6648548, 6648587,
        6648588, 6648589, 6648590, 6648591, 6648592, 6648593, 6648594, 6648595, 6648596, 6648597,
        6648598, 6648599, 6648600, 6648601, 6648602, 6648603, 6648604, 6648605, 6648606, 6648607,
        6648608, 6648609, 6648687, 6648688, 6648689, 6648690, 6648691, 6648692, 6648693, 6648694,
        6648695, 6648696, 6648697, 6648698, 6648699, 6648700, 6648763, 6648781, 6648788, 6648845,
        6648846, 6648847, 6648848, 6648849, 6648850, 6648851, 6648852, 6648853, 6648854, 6648857,
        6648866, 6648966, 6648967, 6648992, 6649011, 6649012, 6649015, 6649032, 6649047, 6649064,
        6649065, 6649066, 6649067, 6649068, 6649069, 6649070, 6649071, 6649072, 6649073, 6649074,
        6649075, 6649076, 6649077, 6649084, 6649087, 6649156, 6649159, 6649374, 6649410, 6649495,
        6649505, 6649673, 6649683, 6649684, 6649718, 6649733, 6649758, 6649773, 6649784, 6649809,
        6649872, 6649931, 6649935, 6650007, 6650014, 6650016, 6650071, 6650072, 6650073, 6650074,
        6650075, 6650076, 6650077, 6650106, 6650107, 6650108, 6650109, 6650110, 6650111, 6650112,
        6650113, 6650118, 6650119, 6650120, 6650121, 6650122, 6650123, 6650124, 6650180, 6650181,
        6650182, 6650183, 6650184, 6650185, 6650186, 6650187, 6650188, 6650189, 6650190, 6650220,
        6650221, 6650269, 6650270, 6650271, 6650272, 6650299, 6650303, 6650364, 6650380, 6650390,
        6650392, 6650435, 6650469, 6650471, 6650523, 6650524, 6650525, 6650526, 6650527, 6650528,
        6650529, 6650530, 6650531, 6650532, 6650533, 6650534, 6650535, 6650536, 6650537, 6650538,
        6650582, 6650673, 6650693, 6650711, 6650712, 6650767, 6650886, 6650921, 6650955, 6650970,
        6650977, 6651003, 6651039, 6651060, 6651061, 6651062, 6651063, 6651064, 6651065, 6651066,
        6651067, 6651068, 6651073, 6651087, 6651088, 6651089, 6651090, 6651091, 6651138, 6651175,
        6651262, 6651270, 6651271, 6651272, 6651273, 6651297, 6651315, 6651316, 6651317, 6651319,
        6651321, 6651342, 6651351, 6651432, 6651468, 6651469, 6651470, 6651506, 6651549, 6651561,
        6651592, 6651604, 6651616, 6651627, 6651646, 6651682, 6651687, 6651719, 6651720, 6651729,
        6651734, 6651754, 6651795, 6651837, 6651856, 6651881, 6651934, 6651935, 6651936, 6651937,
        6651939, 6651948, 6651949, 6651950, 6651951, 6651954, 6651967, 6651970, 6651989, 6651995,
        6652053, 6652115, 6652140, 6652147, 6652151, 6652154, 6652161, 6652185, 6652220, 6652221,
        6652222, 6652223, 6652224, 6652225, 6652226, 6652227, 6652229, 6652230, 6652242, 6652282,
        6652370, 6652376, 6652385, 6652471, 6652472, 6652473, 6652474, 6652481, 6652482, 6652502,
        6652549, 6652552, 6652651, 6652652, 6652653, 6652654, 6652671, 6652683, 6652687, 6652688,
        6652689, 6652690, 6652691, 6652692, 6652700, 6652741, 6652754, 6652790, 6652895, 6653068,
        6653122, 6653171, 6653189, 6653191, 6653202, 6653264, 6653268, 6653274, 6653279, 6653288,
        6653396, 6653413, 6653414, 6653415, 6653416, 6653417, 6653441, 6653442, 6653443, 6653444,
        6653445, 6653484, 6653534, 6653536, 6653583, 6653628, 6653677, 6653683, 6653736, 6653811,
        6653865, 6653866, 6653867, 6653868, 6653869, 6653870, 6653871, 6653872, 6653873, 6653896,
        6653922, 6653923, 6653924, 6653925, 6653926, 6653927, 6653928, 6653929, 6653930, 6653931,
        6653932, 6653933, 6653936, 6653941, 6654031, 6654050, 6654055, 6654155, 6654166, 6654254,
        6654257, 6654258, 6654259, 6654260, 6654290, 6654316, 6654317, 6654325, 6654342, 6654408,
        6654409, 6654410, 6654411, 6654412, 6654413, 6654414, 6654415, 6654416, 6654417, 6654418,
        6654419, 6654469, 6654470, 6654471, 6654472, 6654511, 6654564, 6654565, 6654566, 6654567,
        6654568, 6654569, 6654570, 6654571, 6654572, 6654719, 6654746, 6654799, 6654815, 6654829,
        6654831, 6654837, 6654853, 6654854, 6654855, 6654856, 6654857, 6654858, 6654859, 6654860,
        6654861, 6654862, 6654863, 6654864, 6654865, 6654866, 6654908, 6654912, 6654949, 6654951,
        6654952, 6654954, 6654956, 6654959, 6654961, 6655039, 6655040, 6655041, 6655042, 6655043,
        6655044, 6655045, 6655046, 6655047, 6655048, 6655049, 6655050, 6655051, 6655052, 6655053,
        6655054, 6655055, 6655059, 6655060, 6655096, 6655127, 6655128, 6655129, 6655130, 6655131,
        6655132, 6655133, 6655134, 6655135, 6655136, 6655137, 6655138, 6655139, 6655195, 6655217,
        6655230, 6655385, 6655386, 6655424, 6655425, 6655426, 6655427, 6655428, 6655429, 6655430,
        6655431, 6655432, 6655433, 6655434, 6655450, 6655536, 6655562, 6655566, 6655600, 6655627,
        6655793, 6655830, 6655856, 6655943, 6655978, 6656053, 6656184, 6656201, 6656205, 6656301,
        6656304, 6656315, 6656316, 6656343, 6656419, 6656424, 6656442, 6656447, 6656535, 6656536,
        6656537, 6656538, 6656539, 6656540, 6656542, 6656558, 6656583, 6656680, 6656696, 6656774,
        6656828, 6656829, 6656830, 6656831, 6656834, 6656865, 6656906, 6656935, 6656964, 6657189,
        6657198, 6657215, 6657229, 6657238, 6657242, 6657284, 6657290, 6657298, 6657303, 6657420,
        6657426, 6657429, 6657430, 6657509, 6657560, 6657574, 6657589, 6657591, 6657617, 6657631,
        6657644, 6657649, 6657653, 6657694, 6657741, 6657747, 6657797, 6657874, 6657875, 6657876,
        6657877, 6657878, 6657885, 6657897, 6657898, 6657899, 6657900, 6657901, 6657902, 6657903,
        6657904, 6657905, 6657906, 6657951, 6658009, 6658039, 6658040, 6658041, 6658043, 6658061,
        6658126, 6658130, 6658158, 6658267, 6658268, 6658271, 6658272, 6658273, 6658274, 6658275,
        6658276, 6658277, 6658278, 6658279, 6658280, 6658281, 6658282, 6658283, 6658284, 6658285,
        6658286, 6658287, 6658288, 6658289, 6658290, 6658291, 6658333, 6658400, 6658422, 6658433,
        6658477, 6658515, 6658516, 6658517, 6658518, 6658519, 6658604, 6658688, 6658689, 6658701,
        6658702, 6658703, 6658704, 6658706, 6658743, 6658764, 6658767, 6658796, 6658885, 6658912,
        6658920, 6659115, 6659130, 6659208, 6659209, 6659282, 6659289, 6659326, 6659345, 6659346,
        6659347, 6659348, 6659349, 6659350, 6659351, 6659352, 6659353, 6659354, 6659355, 6659356,
        6659371, 6659430, 6659497, 6659528, 6659539, 6659564, 6659591, 6659638, 6659662, 6659663,
        6659664, 6659665, 6659666, 6659667, 6659668, 6659669, 6659670, 6659671, 6659672, 6659673,
        6659674, 6659675, 6659676, 6659677, 6659678, 6659679, 6659680, 6659681, 6659682, 6659683,
        6659684, 6659685, 6659686, 6659687, 6659688, 6659689, 6659690, 6659691, 6659692, 6659693,
        6659694, 6659695, 6659696, 6659697, 6659698, 6659713, 6659715, 6659753, 6659844, 6659845,
        6659846, 6659847, 6659848, 6659849, 6659850, 6659851, 6659852, 6659867, 6659893, 6659977,
        6659978, 6659979, 6659980, 6659981, 6659982, 6659983, 6659984, 6659985, 6659986, 6659987,
        6659988, 6660062, 6660079, 6660083, 6660173, 6660184, 6660185, 6660186, 6660212, 6660242,
        6660248, 6660265, 6660277, 6660278, 6660302, 6660304, 6660311, 6660355, 6660357, 6660358,
        6660362, 6660366, 6660368, 6660369, 6660386, 6660408, 6660409, 6660484, 6660492, 6660529,
        6660530, 6660531, 6660734, 6660755, 6660756, 6660762, 6660770, 6660862, 6660863, 6660868,
        6660869, 6660870, 6660871, 6660872, 6660873, 6660989, 6661021, 6661026, 6661030, 6661074,
        6661075, 6661076, 6661079, 6661176, 6661188, 6661272, 6661276, 6661278, 6661282, 6661291,
        6661331, 6661398, 6661412, 6661452, 6661466, 6661475, 6661477, 6661490, 6661492, 6661512,
        6661579, 6661646, 6661711, 6661712, 6661713, 6661714, 6661715, 6661716, 6661717, 6661718,
        6661719, 6661727, 6661736, 6661778, 6661779, 6661780, 6661781, 6661853, 6661860, 6661900,
        6661980, 6662016, 6662017, 6662018, 6662019, 6662020, 6662021, 6662022, 6662023, 6662024,
        6662027, 6662028, 6662062, 6662083, 6662104, 6662119, 6662143, 6662144, 6662371, 6662474,
        6662540, 6662558, 6662572, 6662573, 6662574, 6662575, 6662576, 6662577, 6662653, 6662666,
        6662721, 6662740, 6662798, 6662856, 6663006, 6663021, 6663046, 6663047, 6663048, 6663049,
        6663050, 6663051, 6663052, 6663053, 6663054, 6663055, 6663056, 6663057, 6663058, 6663059,
        6663060, 6663061, 6663062, 6663063, 6663064, 6663065, 6663066, 6663067, 6663068, 6663069,
        6663070, 6663071, 6663072, 6663073, 6663074, 6663084, 6663085, 6663086, 6663087, 6663088,
        6663089, 6663090, 6663091, 6663092, 6663093, 6663094, 6663095, 6663096, 6663097, 6663098,
        6663152, 6663251, 6663252, 6663255, 6663256, 6663257, 6663258, 6663259, 6663260, 6663261,
        6663262, 6663297, 6663298, 6663300, 6663302, 6663319, 6663340, 6663345, 6663368, 6663369,
        6663370, 6663371, 6663372, 6663373, 6663374, 6663391, 6663412, 6663462, 6663463, 6663527,
        6663568, 6663569, 6663570, 6663571, 6663572, 6663573, 6663574, 6663575, 6663576, 6663577,
        6663578, 6663579, 6663601, 6663602, 6663603, 6663604, 6663605, 6663606, 6663607, 6663608,
        6663609, 6663610, 6663658, 6663659, 6663666, 6663669, 6663696, 6663697, 6663744, 6663777,
        6663778, 6663779, 6663780, 6663781, 6663782, 6663783, 6663784, 6663785, 6663786, 6663787,
        6663788, 6663789, 6663790, 6663791, 6663792, 6663793, 6663794, 6663795, 6663796, 6663797,
        6663798, 6663799, 6663800, 6663801, 6663809, 6663810, 6663838, 6663843, 6663975, 6663989,
        6663990, 6663991, 6663994, 6663995, 6663996, 6663998, 6664001, 6664003, 6664004, 6664035,
        6664055, 6664142, 6664143, 6664159, 6664188, 6664197, 6664203, 6664254, 6664354, 6664368,
        6664487, 6664530, 6664561, 6664562, 6664563, 6664564, 6664565, 6664566, 6664567, 6664568,
        6664569, 6664570, 6664571, 6664572, 6664573, 6664574, 6664575, 6664576, 6664577, 6664578,
        6664579, 6664580, 6664581, 6664582, 6664583, 6664629, 6664631, 6664641, 6664646, 6664670,
        6664686, 6664687, 6664688, 6664705, 6664770, 6664798, 6664799, 6664800, 6664801, 6664802,
        6664803, 6664804, 6664805, 6664806, 6664863, 6664864, 6664909, 6664910, 6664983, 6665030,
        6665147, 6665157, 6665174, 6665175, 6665176, 6665180, 6665183, 6665190, 6665244, 6665245,
        6665249, 6665250, 6665251, 6665252, 6665253, 6665254, 6665255, 6665256, 6665268, 6665428,
        6665447, 6665485, 6665486, 6665487, 6665488, 6665489, 6665490, 6665491, 6665499, 6665578,
        6665579, 6665640, 6665641, 6665642, 6665643, 6665644, 6665645, 6665646, 6665647, 6665648,
        6665649, 6665650, 6665651, 6665652, 6665653, 6665654, 6665704, 6665711, 6665712, 6665719,
        6665786, 6665818, 6665896, 6665963, 6665986, 6666041, 6666042, 6666051, 6666096, 6666097,
        6666098, 6666107, 6666116, 6666129, 6666138, 6666164, 6666168, 6666325, 6666339, 6666359,
        6666365, 6666366, 6666367, 6666368, 6666369, 6666370, 6666371, 6666372, 6666381, 6666455,
        6666515, 6666526, 6666527, 6666545, 6666546, 6666547, 6666548, 6666549, 6666550, 6666551,
        6666552, 6666553, 6666554, 6666555, 6666556, 6666557, 6666558, 6666559, 6666587, 6666588,
        6666589, 6666590, 6666591, 6666592, 6666619, 6666624, 6666625, 6666626, 6666627, 6666628,
        6666629, 6666630, 6666636, 6666678, 6666708, 6666716, 6666752, 6666753, 6666776, 6666777,
        6666778, 6666779, 6666780, 6666781, 6666782, 6666783, 6666784, 6666785, 6666786, 6666787,
        6666788, 6666789, 6666790, 6666791, 6666873, 6666895, 6666921, 6666922, 6666923, 6666924,
        6666925, 6666926, 6666927, 6666928, 6666929, 6666930, 6666931, 6666932, 6666933, 6666934,
        6666935, 6666936, 6666937, 6666938, 6666939, 6666940, 6666941, 6666942, 6666998, 6667028,
        6667044, 6667045, 6667046, 6667047, 6667048, 6667084, 6667106, 6667116, 6667181, 6667235,
        6667236, 6667237, 6667238, 6667239, 6667240, 6667241, 6667242, 6667243, 6667244, 6667245,
        6667246, 6667247, 6667248, 6667249, 6667250, 6667251, 6667252, 6667253, 6667254, 6667255,
        6667256, 6667257, 6667258, 6667259, 6667260, 6667261, 6667262, 6667263, 6667264, 6667341,
        6667342, 6667343, 6667344, 6667345, 6667346, 6667347, 6667349, 6667350, 6667351, 6667352,
        6667353, 6667354, 6667355, 6667356, 6667357, 6667358, 6667374, 6667381, 6667391, 6667444,
        6667458, 6667459, 6667460, 6667461, 6667462, 6667463, 6667464, 6667465, 6667466, 6667467,
        6667468, 6667469, 6667496, 6667561, 6667592, 6667621, 6667622, 6667682, 6667683, 6667684,
        6667685, 6667686, 6667687, 6667700, 6667727, 6667728, 6667729, 6667730, 6667731, 6667732,
        6667733, 6667734, 6667735, 6667736, 6667737, 6667738, 6667739, 6667740, 6667741, 6667748,
        6667752, 6667767, 6667768, 6667769, 6667770, 6667771, 6667772, 6667773, 6667774, 6667775,
        6667776, 6667777, 6667778, 6667779, 6667780, 6667781, 6667782, 6667783, 6667784, 6667785,
        6667794, 6667811, 6667812, 6667853, 6667883, 6667884, 6667887, 6667896, 6667906, 6668079,
        6668240, 6668266, 6668281, 6668347, 6668348, 6668351, 6668369, 6668370, 6668371, 6668372,
        6668456, 6668471, 6668492, 6668503, 6668507, 6668508, 6668521, 6668542, 6668635, 6668651,
        6668658, 6668700, 6668701, 6668708, 6668833, 6668930, 6668931, 6668932, 6668933, 6668934,
        6668935, 6668936, 6668937, 6668938, 6668939, 6668946, 6668972, 6668998, 6669023, 6669026,
        6669032, 6669033, 6669034, 6669035, 6669076, 6669120, 6669143, 6669150, 6669166, 6669224,
        6669279, 6669295, 6669367, 6669373, 6669374, 6669375, 6669376, 6669377, 6669378, 6669379,
        6669380, 6669381, 6669382, 6669441, 6669590, 6669603, 6669654, 6669655, 6669715, 6669773,
        6669778, 6669787, 6669811, 6669812, 6669868, 6669869, 6669886, 6669980, 6669981, 6669982,
        6669983, 6669984, 6669985, 6669986, 6670014, 6670015, 6670016, 6670017, 6670060, 6670066,
        6670074, 6670077, 6670078, 6670079, 6670111, 6670145, 6670147, 6670172, 6670182, 6670252,
        6670266, 6670268, 6670352, 6670420, 6670422, 6670522, 6670623, 6670641, 6670649, 6670650,
        6670655, 6670696, 6670705, 6670709, 6670713, 6670727, 6670788, 6670808, 6670891, 6670892,
        6670893, 6670894, 6670895, 6670896, 6670897, 6670898, 6670899, 6670900, 6670901, 6670902,
        6670903, 6670920, 6670921, 6670922, 6670923, 6670924, 6670947, 6670970, 6671006, 6671007,
        6671022, 6671089, 6671134, 6671135, 6671136, 6671137, 6671138, 6671139, 6671140, 6671162,
        6671201, 6671205, 6671223, 6671277, 6671365, 6671419, 6671423, 6671427, 6671431, 6671435,
        6671439, 6671443, 6671447, 6671451, 6671455, 6671459, 6671463, 6671467, 6671471, 6671473,
        6671477, 6671481, 6671485, 6671489, 6671496, 6671497, 6671498, 6671499, 6671500, 6671501,
        6671502, 6671503, 6671504, 6671505, 6671506, 6671507, 6671508, 6671526, 6671540, 6671575,
        6671581, 6671639, 6671695, 6671697, 6671737, 6671753, 6671793, 6671830, 6671842, 6671943,
        6671990, 6671991, 6671992, 6672084, 6672086, 6672109, 6672110, 6672172, 6672196, 6672197,
        6672200, 6672201, 6672252, 6672257, 6672258, 6672259, 6672260, 6672264, 6672326, 6672327,
        6672328, 6672329, 6672330, 6672331, 6672332, 6672333, 6672334, 6672335, 6672336, 6672337,
        6672344, 6672347, 6672376, 6672386, 6672391, 6672392, 6672393, 6672394, 6672395, 6672396,
        6672397, 6672398, 6672399, 6672400, 6672413, 6672431, 6672432, 6672483, 6672566, 6672567,
        6672572, 6672596, 6672606, 6672686, 6672687, 6672719, 6672735, 6672750, 6672751, 6672770,
        6672825, 6672857, 6672934, 6672946, 6672950, 6673007, 6673020, 6673047, 6673061, 6673064,
        6673143, 6673172, 6673173, 6673174, 6673175, 6673176, 6673177, 6673178, 6673179, 6673192,
        6673219, 6673220, 6673225, 6673280, 6673281, 6673284, 6673285, 6673301, 6673302, 6673402,
        6673404, 6673424, 6673468, 6673514, 6673522, 6673525, 6673526, 6673551, 6673662, 6673681,
        6673760, 6673863, 6673871, 6673872, 6673873, 6673874, 6673875, 6673876, 6673877, 6673878,
        6673879, 6673880, 6673892, 6673893, 6673894, 6673895, 6673908, 6673909, 6673910, 6673911,
        6673912, 6673913, 6673914, 6673915, 6673916, 6673917, 6673918, 6673919, 6673920, 6673921,
        6673922, 6673923, 6673924, 6673929, 6673988, 6674035, 6674053, 6674054, 6674093, 6674103,
        6674107, 6674111, 6674147, 6674240, 6674254, 6674255, 6674256, 6674257, 6674258, 6674259,
        6674260, 6674261, 6674262, 6674263, 6674264, 6674280, 6674283, 6674302, 6674328, 6674333,
        6674342, 6674346, 6674350, 6674385, 6674386, 6674387, 6674388, 6674389, 6674390, 6674391,
        6674392, 6674393, 6674394, 6674395, 6674396, 6674397, 6674398, 6674399, 6674418, 6674453,
        6674454, 6674594, 6674607, 6674739, 6674745, 6674756, 6674761, 6674763, 6674836, 6674848,
        6674854, 6674863, 6674904, 6674906, 6674913, 6674914, 6674923, 6674924, 6674925, 6674926,
        6674927, 6674928, 6674929, 6674930, 6674939, 6675054, 6675067, 6675069, 6675090, 6675093,
        6675102, 6675114, 6675121, 6675123, 6675154, 6675170, 6675171, 6675238, 6675256, 6675290,
        6675324, 6675371, 6675372, 6675373, 6675374, 6675375, 6675376, 6675377, 6675378, 6675379,
        6675380, 6675383, 6675386, 6675410, 6675411, 6675423, 6675434, 6675484, 6675485, 6675507,
        6675511, 6675530, 6675540, 6675547, 6675553, 6675554, 6675642, 6675643, 6675644, 6675645,
        6675646, 6675647, 6675648, 6675649, 6675650, 6675651, 6675652, 6675653, 6675654, 6675664,
        6675713, 6675725, 6675733, 6675744, 6675775, 6675824, 6675879, 6675900, 6675902, 6675945,
        6675949, 6676070, 6676099, 6676108, 6676121, 6676129, 6676130, 6676131, 6676132, 6676133,
        6676134, 6676135, 6676136, 6676137, 6676138, 6676139, 6676140, 6676181, 6676183, 6676215,
        6676346, 6676367, 6676395, 6676412, 6676440, 6676468, 6676507, 6676538, 6676580, 6676631,
        6676634, 6676664, 6676675, 6676676, 6676677, 6676678, 6676679, 6676680, 6676681, 6676714,
        6676715, 6676716, 6676717, 6676718, 6676719, 6676720, 6676721, 6676722, 6676723, 6676724,
        6676725, 6676726, 6676727, 6676728, 6676729, 6676751, 6676756, 6676800, 6676853, 6676854,
        6676952, 6676953, 6676954, 6676955, 6676959, 6677111, 6677112, 6677123, 6677128, 6677152,
        6677170, 6677235, 6677238, 6677244, 6677254, 6677262, 6677281, 6677293, 6677303, 6677316,
        6677398, 6677449, 6677450, 6677451, 6677452, 6677453, 6677454, 6677455, 6677456, 6677457,
        6677458, 6677459, 6677460, 6677461, 6677462, 6677463, 6677464, 6677465, 6677466, 6677467,
        6677468, 6677469, 6677470, 6677471, 6677472, 6677473, 6677474, 6677475, 6677476, 6677477,
        6677478, 6677479, 6677480, 6677481, 6677530, 6677531, 6677532, 6677533, 6677534, 6677535,
        6677536, 6677537, 6677561, 6677570, 6677605, 6677606, 6677657, 6677660, 6677683, 6677710,
        6677772, 6677787, 6677810, 6677819, 6677821, 6677848, 6677898, 6677922, 6677931, 6677939,
        6677944, 6677945, 6677946, 6677947, 6677948, 6677949, 6677950, 6677951, 6677952, 6677953,
        6677954, 6677955, 6677956, 6677957, 6677958, 6678075, 6678089, 6678126, 6678217, 6678218,
        6678219, 6678221, 6678237, 6678331, 6678349, 6678350, 6678411, 6678412, 6678413, 6678414,
        6678415, 6678416, 6678417, 6678418, 6678419, 6678420, 6678421, 6678422, 6678423, 6678424,
        6678425, 6678426, 6678427, 6678428, 6678429, 6678430, 6678431, 6678432, 6678433, 6678434,
        6678435, 6678436, 6678437, 6678438, 6678439, 6678448, 6678454, 6678608, 6678609, 6678655,
        6678821, 6678866, 6678915, 6678917, 6678994, 6679027, 6679029, 6679036, 6679044, 6679100,
        6679101, 6679102, 6679103, 6679104, 6679105, 6679106, 6679107, 6679108, 6679109, 6679110,
        6679111, 6679112, 6679148, 6679163, 6679195, 6679196, 6679197, 6679198, 6679199, 6679200,
        6679201, 6679202, 6679203, 6679204, 6679235, 6679236, 6679237, 6679238, 6679239, 6679240,
        6679241, 6679242, 6679243, 6679244, 6679245, 6679246, 6679261, 6679264, 6679296, 6679316,
        6679342, 6679404, 6679511, 6679534, 6679548, 6679549, 6679585, 6679596, 6679598, 6679623,
        6679628, 6679641, 6679646, 6679671, 6679693, 6679701, 6679710, 6679716, 6679731, 6679735,
        6679737, 6679764, 6679812, 6679813, 6679814, 6679815, 6679816, 6679817, 6679818, 6679819,
        6679820, 6679821, 6679822, 6679823, 6679824, 6679825, 6679826, 6679827, 6679901, 6679902,
        6679903, 6679904, 6679946, 6679974, 6680017, 6680160, 6680161, 6680162, 6680163, 6680164,
        6680165, 6680167, 6680169, 6680171, 6680172, 6680173, 6680174, 6680175, 6680176, 6680178,
        6680179, 6680180, 6680182, 6680183, 6680203, 6680265, 6680304, 6680319, 6680342, 6680375,
        6680434, 6680441, 6680442, 6680448, 6680449, 6680450, 6680451, 6680452, 6680453, 6680454,
        6680455, 6680456, 6680457, 6680458, 6680459, 6680460, 6680461, 6680462, 6680508, 6680539,
        6680567, 6680594, 6680616, 6680628, 6680646, 6680676, 6680745, 6680746, 6680747, 6680748,
        6680749, 6680750, 6680751, 6680752, 6680753, 6680754, 6680755, 6680756, 6680757, 6680758,
        6680759, 6680760, 6680761, 6680762, 6680763, 6680764, 6680765, 6680766, 6680767, 6680768,
        6680769, 6680770, 6680771, 6680772, 6680773, 6680774, 6680775, 6680776, 6680777, 6680778,
        6680779, 6680780, 6680781, 6680782, 6680783, 6680784, 6680785, 6680786, 6680787, 6680788,
        6680789, 6680790, 6680791, 6680792, 6680793, 6680794, 6680795, 6680796, 6680797, 6680798,
        6680799, 6680800, 6680801, 6680802, 6680820, 6680955, 6680956, 6680957, 6680958, 6680959,
        6680960, 6680961, 6680962, 6680963, 6680964, 6680965, 6680966, 6680967, 6680968, 6680969,
        6680970, 6680971, 6680972, 6680973, 6680974, 6680975, 6680976, 6680977, 6680978, 6680979,
        6680980, 6680981, 6680982, 6680983, 6680984, 6680985, 6680986, 6680987, 6680988, 6680989,
        6680990, 6680991, 6680992, 6680993, 6680994, 6680995, 6680996, 6680997, 6680998, 6680999,
        6681060, 6681111, 6681145, 6681190, 6681201, 6681227, 6681229, 6681235, 6681237, 6681257,
        6681383, 6681436, 6681476, 6681477, 6681478, 6681479, 6681480, 6681481, 6681482, 6681483,
        6681484, 6681485, 6681550, 6681554, 6681576, 6681600, 6681614, 6681621, 6681789, 6681798,
        6681818, 6681819, 6681820, 6681821, 6681822, 6681823, 6681824, 6681825, 6681826, 6681827,
        6681828, 6681829, 6681830, 6681831, 6681832, 6681833, 6681864, 6681951, 6681952, 6681953,
        6681954, 6681955, 6681956, 6681957, 6681958, 6681959, 6681960, 6681961, 6681962, 6681966,
        6681993, 6682073, 6682074, 6682093, 6682115, 6682131, 6682176, 6682204, 6682291, 6682359,
        6682393, 6682399, 6682433, 6682445, 6682508, 6682522, 6682523, 6682524, 6682528, 6682629,
        6682697, 6682724, 6682754, 6682829, 6682830, 6682831, 6682832, 6682833, 6682834, 6682835,
        6682836, 6682837, 6682838, 6682839, 6682840, 6682841, 6682842, 6682843, 6682844, 6682845,
        6682846, 6682847, 6682880, 6682897, 6682932, 6682966, 6682969, 6682981, 6682992, 6682999,
        6683023, 6683030, 6683038, 6683042, 6683062, 6683080, 6683081, 6683082, 6683083, 6683084,
        6683085, 6683086, 6683087, 6683088, 6683089, 6683090, 6683091, 6683092, 6683093, 6683094,
        6683095, 6683096, 6683097, 6683098, 6683099, 6683100, 6683101, 6683102, 6683103, 6683104,
        6683105, 6683106, 6683107, 6683108, 6683136, 6683140, 6683211, 6683212, 6683213, 6683233,
        6683248, 6683251, 6683263, 6683339, 6683363, 6683365, 6683369, 6683370, 6683371, 6683372,
        6683418, 6683431, 6683441, 6683447, 6683541, 6683604, 6683626, 6683627, 6683628, 6683629,
        6683630, 6683631, 6683632, 6683633, 6683634, 6683635, 6683636, 6683747, 6683758, 6683878,
        6683984, 6684040, 6684070, 6684115, 6684116, 6684117, 6684118, 6684119, 6684120, 6684145,
        6684154, 6684156, 6684258, 6684261, 6684262, 6684282, 6684283, 6684285, 6684314, 6684316,
        6684318, 6684320, 6684323, 6684324, 6684325, 6684326, 6684327, 6684328, 6684329, 6684330,
        6684331, 6684332, 6684333, 6684334, 6684335, 6684336, 6684337, 6684378, 6684407, 6684414,
        6684416, 6684424, 6684472, 6684563, 6684574, 6684575, 6684576, 6684577, 6684601, 6684635,
        6684636, 6684639, 6684640, 6684641, 6684642, 6684666, 108658947,
    ];
    let original = ARRAY.iter().cloned().collect::<RoaringBitmap>();
    let new = serialize_and_deserialize(&original);
    assert_eq!(original, new);
}
