/*
 * NASPRO - The NASPRO Architecture for Sound PROcessing
 * Portable runtime library
 *
 * Copyright (C) 2007-2014 Stefano D'Angelo
 *
 * See the COPYING file for license conditions.
 */

#define _WIN32_WINNT	0x501
#define UNICODE

#include "internal.h"

typedef struct
  {
	void (*start_routine)(void *arg);
	void *arg;
  } thread_data_t;

static unsigned
thread_start(void *arg)
{
	thread_data_t *data;
	void (*start_routine)(void *arg) sr;

	data = (thread_data_t *)arg;
	sr = data->start_routine;
	arg = data->arg;

	free(data);

	start_routine(arg);

	return 0;
}

_NACORE_DEF int
nacore_thread_create(nacore_thread *thread, size_t stack_size,
		     void (*start_routine)(void *arg), void *arg)
{
	thread_data_t *data;

	data = malloc(sizeof(thread_data_t));
	if (data == NULL)
		return ENOMEM;

	data->start_routine = start_routine;
	data->arg = arg;

	thread->handle = (HANDLE)_beginthreadex(NULL, stack_size, thread_start,
						data, 0, &thread->id);

	if (thread->v.handle == NULL)
	  {
		free(data);
		return EAGAIN;
	  }

	thread->flags = 0;

	return 0;
}

_NACORE_DEF void
nacore_thread_exit()
{
	_endthreadex(0);
}

_NACORE_DEF nacore_thread
nacore_thread_self()
{
	nacore_thread ret;

	ret.handle = NULL;
	ret.id = GetCurrentThreadId();

	return ret;
}

_NACORE_DEF char
nacore_thread_equal(nacore_thread t1, nacore_thread t2)
{
	return t1.id == t2.id;
}

_NACORE_DEF int
nacore_thread_join(nacore_thread thread)
{
	int ret;
	HANDLE h;

	if (thread.id == GetCurrentThreadId())
		return EDEADLK;

	if (thread.handle == NULL)
	  {
		h = OpenThread(SYNCHRONIZE, FALSE, thread.id);
		if (h == NULL)
			return ESRCH;
	  }
	else
		h = thread.handle;

	ret = 0;
	if (WaitForSingleObject(h, INFINITE) != WAIT_OBJECT_0)
		ret = NACORE_EUNKNOWN;

	if (thread.handle == NULL)
		CloseHandle(h);

	return ret;
}
