/* See LICENSE for licensing and NOTICE for copyright. */
package org.ldaptive.transport.netty;

import java.util.ArrayList;
import java.util.List;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import org.testng.Assert;
import org.testng.annotations.DataProvider;
import org.testng.annotations.Test;

/**
 * Unit test for {@link MessageFrameDecoder}.
 *
 * @author  Middleware Services
 */
public class MessageFrameDecoderTest
{

  /** To test with. */
  private final MessageFrameDecoder decoder = new MessageFrameDecoder();


  /**
   * Decoder test data.
   *
   * @return  response test data
   */
  @DataProvider(name = "response")
  public Object[][] createData()
  {
    return
      new Object[][] {
        new Object[] {
          new byte[] {0x30},
          false,
        },
        new Object[] {
          new byte[] {0x30, (byte) 0x82},
          false,
        },
        new Object[] {
          new byte[] {0x30, (byte) 0x82, 0x01},
          false,
        },
        new Object[] {
          new byte[] {
            0x30, (byte) 0x82, 0x01, 0x4b},
          false,
        },
        new Object[] {
          new byte[] {
            0x30, (byte) 0x82, 0x01, 0x4b, 0x02, 0x01},
          false,
        },
        new Object[] {
          new byte[] {
            0x30, (byte) 0x82, 0x01, 0x4b, 0x02, 0x01, 0x01, 0x64, (byte) 0x82, 0x01, 0x44, 0x04, 0x21, 0x75, 0x69},
          false,
        },
        new Object[] {
          new byte[] {
            0x30, (byte) 0x82, 0x01, 0x4b, 0x02, 0x01, 0x01, 0x64, (byte) 0x82, 0x01, 0x44, 0x04, 0x21, 0x75, 0x69,
            0x64, 0x3d, 0x38, 0x31, 0x38, 0x30, 0x33, 0x37, 0x2c, 0x6f, 0x75, 0x3d, 0x50, 0x65, 0x6f, 0x70, 0x6c, 0x65,
            0x2c, 0x64, 0x63, 0x3d, 0x76, 0x74, 0x2c, 0x64, 0x63, 0x3d, 0x65, 0x64, 0x75, 0x30, (byte) 0x82, 0x01, 0x1d,
            0x30, 0x23, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x14, 0x04,
            0x12, 0x76, 0x69, 0x72, 0x67, 0x69, 0x6e, 0x69, 0x61, 0x54, 0x65, 0x63, 0x68, 0x50, 0x65, 0x72, 0x73, 0x6f,
            0x6e, 0x30, 0x12, 0x04, 0x05, 0x75, 0x75, 0x70, 0x69, 0x64, 0x31, 0x09, 0x04, 0x07, 0x64, 0x66, 0x69, 0x73,
            0x68, 0x65, 0x72, 0x30, 0x0f, 0x04, 0x03, 0x75, 0x69, 0x64, 0x31, 0x08, 0x04, 0x06, 0x38, 0x31, 0x38, 0x30,
            0x33, 0x37, 0x30, 0x17, 0x04, 0x02, 0x63, 0x6e, 0x31, 0x11, 0x04, 0x0f, 0x44, 0x61, 0x6e, 0x69, 0x65, 0x6c,
            0x20, 0x57, 0x20, 0x46, 0x69, 0x73, 0x68, 0x65, 0x72, 0x30, (byte) 0x81, (byte) 0xb7, 0x04, 0x13, 0x73},
          false,
        },
        new Object[] {
          new byte[] {
            0x30, (byte) 0x82, 0x01, 0x4b, 0x02, 0x01, 0x01, 0x64, (byte) 0x82, 0x01, 0x44, 0x04, 0x21, 0x75, 0x69,
            0x64, 0x3d, 0x38, 0x31, 0x38, 0x30, 0x33, 0x37, 0x2c, 0x6f, 0x75, 0x3d, 0x50, 0x65, 0x6f, 0x70, 0x6c, 0x65,
            0x2c, 0x64, 0x63, 0x3d, 0x76, 0x74, 0x2c, 0x64, 0x63, 0x3d, 0x65, 0x64, 0x75, 0x30, (byte) 0x82, 0x01, 0x1d,
            0x30, 0x23, 0x04, 0x0b, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x31, 0x14, 0x04,
            0x12, 0x76, 0x69, 0x72, 0x67, 0x69, 0x6e, 0x69, 0x61, 0x54, 0x65, 0x63, 0x68, 0x50, 0x65, 0x72, 0x73, 0x6f,
            0x6e, 0x30, 0x12, 0x04, 0x05, 0x75, 0x75, 0x70, 0x69, 0x64, 0x31, 0x09, 0x04, 0x07, 0x64, 0x66, 0x69, 0x73,
            0x68, 0x65, 0x72, 0x30, 0x0f, 0x04, 0x03, 0x75, 0x69, 0x64, 0x31, 0x08, 0x04, 0x06, 0x38, 0x31, 0x38, 0x30,
            0x33, 0x37, 0x30, 0x17, 0x04, 0x02, 0x63, 0x6e, 0x31, 0x11, 0x04, 0x0f, 0x44, 0x61, 0x6e, 0x69, 0x65, 0x6c,
            0x20, 0x57, 0x20, 0x46, 0x69, 0x73, 0x68, 0x65, 0x72, 0x30, (byte) 0x81, (byte) 0xb7, 0x04, 0x13, 0x73,
            0x75, 0x70, 0x70, 0x72, 0x65, 0x73, 0x73, 0x65, 0x64, 0x41, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65,
            0x31, (byte) 0x81, (byte) 0x9f, 0x04, 0x14, 0x6d, 0x61, 0x69, 0x6c, 0x50, 0x72, 0x65, 0x66, 0x65, 0x72,
            0x72, 0x65, 0x64, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x04, 0x14, 0x75, 0x73, 0x65, 0x72, 0x53, 0x4d,
            0x49, 0x4d, 0x45, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x04, 0x04, 0x6d, 0x61,
            0x69, 0x6c, 0x04, 0x0b, 0x6d, 0x61, 0x69, 0x6c, 0x41, 0x63, 0x63, 0x6f, 0x75, 0x6e, 0x74, 0x04, 0x12, 0x6c,
            0x6f, 0x63, 0x61, 0x6c, 0x50, 0x6f, 0x73, 0x74, 0x61, 0x6c, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x04,
            0x15, 0x6d, 0x61, 0x69, 0x6c, 0x46, 0x6f, 0x72, 0x77, 0x61, 0x72, 0x64, 0x69, 0x6e, 0x67, 0x41, 0x64, 0x64,
            0x72, 0x65, 0x73, 0x73, 0x04, 0x09, 0x68, 0x6f, 0x6d, 0x65, 0x50, 0x68, 0x6f, 0x6e, 0x65, 0x04, 0x09, 0x6d,
            0x61, 0x69, 0x6c, 0x41, 0x6c, 0x69, 0x61, 0x73, 0x04, 0x11, 0x68, 0x6f, 0x6d, 0x65, 0x50, 0x6f, 0x73, 0x74,
            0x61, 0x6c, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x04, 0x0a, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x50, 0x68,
            0x6f, 0x6e, 0x65},
          true,
        },
      };
  }


  /**
   * @param  berValue  to decode.
   * @param  frame  whether the frame is complete
   *
   * @throws  Exception  On test failure.
   */
  @Test(groups = "transport", dataProvider = "response")
  public void decode(final byte[] berValue, final boolean frame)
    throws Exception
  {
    final List<Object> out = new ArrayList<>();
    decoder.decode(null, Unpooled.wrappedBuffer(berValue), out);
    Assert.assertEquals(!out.isEmpty(), frame);
    if (frame) {
      Assert.assertEquals(((ByteBuf) out.get(0)).array(), berValue);
    }
  }
}
