# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Gitlab::BackgroundMigration::BackfillArchivedAndTraversalIdsToVulnerabilityReads,
  feature_category: :vulnerability_management do
  before(:all) do
    # This migration will not work if a sec database is configured. It should be finalized and removed prior to
    # sec db rollout.
    # Consult https://gitlab.com/gitlab-org/gitlab/-/merge_requests/171707 for more info.
    skip_if_multiple_databases_are_setup(:sec)
  end

  let(:vulnerability_identifiers) { table(:vulnerability_identifiers) }
  let(:vulnerability_findings) { table(:vulnerability_occurrences) }
  let(:vulnerabilities) { table(:vulnerabilities) }
  let(:vulnerability_reads) { table(:vulnerability_reads) }
  let(:projects) { table(:projects) }
  let(:users) { table(:users) }
  let(:namespaces) { table(:namespaces) }
  let(:scanners) { table(:vulnerability_scanners) }

  let(:user) { users.create!(username: 'john_doe', email: 'johndoe@gitlab.com', projects_limit: 1) }

  let(:args) do
    min, max = vulnerability_reads.pick('MIN(id)', 'MAX(id)')

    {
      start_id: min,
      end_id: max,
      batch_table: 'vulnerability_reads',
      batch_column: 'id',
      sub_batch_size: 100,
      pause_ms: 0,
      connection: ApplicationRecord.connection
    }
  end

  let!(:group_namespace) do
    namespaces.create!(
      name: 'gitlab-org',
      path: 'gitlab-org',
      type: 'Group'
    ).tap { |namespace| namespace.update!(traversal_ids: [namespace.id]) }
  end

  let!(:other_group_namespace) do
    namespaces.create!(
      name: 'gitlab-com',
      path: 'gitlab-com',
      type: 'Group'
    ).tap { |namespace| namespace.update!(traversal_ids: [namespace.id]) }
  end

  let!(:project) { create_project('gitlab', group_namespace) }
  let!(:other_project) { create_project('www-gitlab-com', other_group_namespace) }

  subject(:perform_migration) { described_class.new(**args).perform }

  before do
    [project, other_project].each do |p|
      create_vulnerability_read(project_id: p.id)
    end
  end

  it 'backfills traversal_ids and archived', :aggregate_failures do
    perform_migration

    vulnerability_reads.find_each do |vulnerability_read|
      project = projects.find(vulnerability_read.project_id)
      namespace = namespaces.find(project.namespace_id)

      expect(vulnerability_read.traversal_ids).to eq(namespace.traversal_ids)
      expect(vulnerability_read.archived).to eq(project.archived)
    end
  end

  def create_vulnerability_read(project_id:)
    scanner = scanners.create!(project_id: project_id, external_id: 'foo', name: 'bar')

    identifier = vulnerability_identifiers.create!(
      project_id: project_id,
      external_id: "CVE-2018-1234",
      external_type: "CVE",
      name: "CVE-2018-1234",
      fingerprint: SecureRandom.hex(20)
    )

    finding = vulnerability_findings.create!(
      project_id: project_id,
      scanner_id: scanner.id,
      severity: 5, # medium
      report_type: 99, # generic
      primary_identifier_id: identifier.id,
      project_fingerprint: SecureRandom.hex(20),
      location_fingerprint: SecureRandom.hex(20),
      uuid: SecureRandom.uuid,
      name: "CVE-2018-1234",
      raw_metadata: "{}",
      metadata_version: "test:1.0"
    )

    vulnerability = vulnerabilities.create!(
      project_id: project_id,
      author_id: user.id,
      title: 'Vulnerability 1',
      severity: 1,
      confidence: 1,
      report_type: 1,
      state: 1,
      finding_id: finding.id
    )

    vulnerability_reads.create!(
      vulnerability_id: vulnerability.id,
      namespace_id: project.namespace_id,
      project_id: project_id,
      scanner_id: scanner.id,
      report_type: 1,
      severity: 1,
      state: 1,
      uuid: SecureRandom.uuid,
      archived: false,
      traversal_ids: []
    )
  end

  def create_project(name, group)
    project_namespace = namespaces.create!(
      name: name,
      path: name,
      type: 'Project'
    )

    projects.create!(
      namespace_id: group.id,
      project_namespace_id: project_namespace.id,
      name: name,
      path: name,
      archived: true
    )
  end
end
