/*
 * ZClock - A simple but powerful clock applet for GNOME
 * Copyright (C) 2000  Zack Hobson <zack@malfunktion.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <config.h>
#include <applet-widget.h>

#ifdef HAVE_LIBGTOP
#include <glibtop/uptime.h>
#endif

#include "gtk_zclock.h"
#include "gnome_zclock_preview.h"

#define TOOLTIP_BUF_SIZE 64
#define TOOLTIP_UPDATE_INTERVAL 60000

typedef struct _AppletData AppletData;
struct _AppletData {
	GtkWidget* applet;	/* well, DUH! */

	GtkWidget* ttbox;	/* hbox for tooltip format entry */

	gchar* format;		/* applet format */
	gchar* format_prop;

	gchar* tooltip;		/* tooltip format */
	gchar* tooltip_prop;

	gint show_tooltip;	/* display the tooltip? */
	gint show_tooltip_prop;

	GList *formats; /* all available formats */

	GtkShadowType shadow;	/* shadow style */
	GtkShadowType shadow_prop;

	GtkJustification jtype;	/* label justification */
	GtkJustification jtype_prop;

	gint xpad; /* horizontal padding */
	gint xpad_prop;

	gint ypad; /* vertical padding */
	gint ypad_prop;

	GtkWidget *zclock; /* zclock widget */
	GtkWidget *props;  /* properties window */
};

static gint save_session(GtkWidget*, const char*, const char*);
static void load_session(GtkWidget*, AppletData*);
static void show_properties(AppletWidget*, gpointer);
static void properties_destroy_cb (GtkWidget*, gpointer);
static void properties_help_cb (GtkWidget*, gint, gpointer);
static void properties_apply_cb (GtkWidget*, gint, gpointer);
static void entry_changed_cb (GtkWidget*, gpointer);
static void browse_clicked_cb (GtkWidget*, gpointer);
static void padding_changed_cb (GtkAdjustment*, gpointer);
static void radio_tooltip_toggle_cb (GtkWidget*, gpointer);
static void radio_shadow_toggle_cb (GtkWidget*, gpointer);
static void radio_jtype_toggle_cb (GtkWidget*, gpointer);
static void show_about(AppletWidget*, gpointer);
static void show_help(AppletWidget*, gpointer);
static gint update_tooltip (gpointer);

int
main(int argc, char **argv)
{
	GtkWidget *applet;
	AppletData *config = g_new(AppletData, 1);

	bindtextdomain(PACKAGE, GNOMELOCALEDIR);
	textdomain(PACKAGE);

	applet_widget_init("zclock", NULL, argc, argv, NULL, 0, NULL);

	applet = applet_widget_new("zclock");
	if (!applet)
		g_error("Can't create applet!\n");

	load_session(applet, config);
	gtk_object_set_user_data(GTK_OBJECT(applet), config);
	gtk_signal_connect(GTK_OBJECT(applet), "save_session",
			GTK_SIGNAL_FUNC(save_session), NULL);

	config->zclock = gtk_zclock_new(config->format);
	gtk_frame_set_shadow_type(GTK_FRAME(config->zclock), config->shadow);
	gtk_zclock_set_justify(GTK_ZCLOCK(config->zclock), config->jtype);
	gtk_zclock_set_label_padding(GTK_ZCLOCK(config->zclock),
					config->xpad, config->ypad);

	applet_widget_add(APPLET_WIDGET(applet), config->zclock);

	applet_widget_register_stock_callback(APPLET_WIDGET(applet),
			"properties", GNOME_STOCK_MENU_PROP, _("Properties..."),
			&show_properties, NULL);
	applet_widget_register_stock_callback(APPLET_WIDGET(applet),
			"help", GNOME_STOCK_PIXMAP_HELP, _("Help"),
			&show_help, NULL);
	applet_widget_register_stock_callback(APPLET_WIDGET(applet),
			"about", GNOME_STOCK_MENU_ABOUT, _("About..."),
			&show_about, NULL);

	gtk_widget_show_all(applet);

	update_tooltip(config);
	gtk_timeout_add(TOOLTIP_UPDATE_INTERVAL, update_tooltip, config);

	applet_widget_gtk_main();

	g_list_free(config->formats);
	g_free(config->format);
	g_free(config->format_prop);
	g_free(config->tooltip);
	g_free(config->tooltip_prop);
	g_free(config);

	return 0;
}

static gint
update_tooltip (gpointer data)
{
	char		buf[TOOLTIP_BUF_SIZE] = "";
	time_t		tmp = 0;
	struct tm	*now;
	AppletData	*config = data;

	if (config->show_tooltip == 0)
	{ /* No tooltip */
		applet_widget_set_tooltip(APPLET_WIDGET (config->applet), "");
		return TRUE;
	}

#ifdef HAVE_LIBGTOP
	if (config->show_tooltip == 1)
	{ /* Uptime in tooltip */
		glibtop_uptime	gtu;
		gint		up;	/* uptime in seconds */
		gint		m;	/* minutes */
		gint		h;	/* hours */
		gint		d;	/* days */
		gchar*		uptime;	/* uptime string */

		glibtop_get_uptime(&gtu);
		up = (gint) gtu.uptime;
		m = (up /= 60) % 60;
		h = (up /= 60) % 24;
		d = up / 24;
		uptime = g_strdup_printf ("Up %d days, %d:%02d", d, h, m);
		applet_widget_set_tooltip (APPLET_WIDGET (config->applet), uptime);
		g_free (uptime);
		return TRUE;
	}
#endif

	tmp = time(NULL);
	now = localtime(&tmp);
	if (strftime(buf, TOOLTIP_BUF_SIZE, config->tooltip, now) == TOOLTIP_BUF_SIZE)
		buf[TOOLTIP_BUF_SIZE - 1] = '\0';
	applet_widget_set_tooltip(APPLET_WIDGET (config->applet), buf);

	return TRUE;
}

static void
load_session(GtkWidget *applet, AppletData *config)
{
	gboolean def;
	gint fmt_count;
	gchar **fmt_list;
	gint i;

	static gchar *default_formats[] = {
			"%l:%M:%S %p",
			"%A %B %d",
			"%A%n%l:%M:%S %p",
			"%A%n%B %e%n%l:%M %p",
			NULL
		};

	config->applet = applet;

	gnome_config_push_prefix(APPLET_WIDGET(applet)->privcfgpath);

	config->format = gnome_config_get_string_with_default(
			"format/default=%I:%M:%S %p", &def);
	config->tooltip = gnome_config_get_string_with_default(
			"format/tooltip=%A %B %d", &def);
	config->show_tooltip = gnome_config_get_int_with_default(
			"format/show_tooltip=2", &def);
	config->jtype = CLAMP(gnome_config_get_int_with_default(
			"display/jtype=2", &def), GTK_JUSTIFY_LEFT,
			GTK_JUSTIFY_CENTER);
	config->shadow = CLAMP(gnome_config_get_int_with_default(
			"display/shadow=0", &def), GTK_SHADOW_NONE,
			GTK_SHADOW_ETCHED_OUT);
	config->xpad = gnome_config_get_int_with_default(
			"display/xpad=0", &def);
	config->ypad = gnome_config_get_int_with_default(
			"display/ypad=0", &def);

	gnome_config_get_vector_with_default(
			"format/formats=",
			&fmt_count, &fmt_list, &def);

	config->formats = NULL;
	if (def)
	{
		i = 0;
		while (default_formats[i] != NULL)
		{
			config->formats = g_list_prepend(config->formats,
					g_strdup(default_formats[i]));
			i++;
		}
	}
	else
	{
		for (i = 0; i < fmt_count; i++)
		{
			config->formats = g_list_prepend(config->formats,
					g_strdup(fmt_list[i]));
			g_free(fmt_list[i]);
		}
	}
	config->formats = g_list_reverse(config->formats);

	gnome_config_pop_prefix();

	config->props = NULL;
	config->zclock = NULL;

	config->format_prop = g_strdup(config->format);
	config->tooltip_prop = g_strdup(config->tooltip);
	config->show_tooltip_prop = config->show_tooltip;
	config->shadow_prop = config->shadow;
	config->jtype_prop = config->jtype;
	config->xpad_prop = config->xpad;
	config->ypad_prop = config->ypad;

	return;
}

static gint
save_session(GtkWidget *w, const char *privcfgpath, const char *globcfgpath)
{
	AppletData *config = gtk_object_get_user_data(GTK_OBJECT(w));

	gint fmt_count = g_list_length(config->formats);
	gchar *fmt_list[fmt_count];
	gint i;

	gnome_config_push_prefix(privcfgpath);
	gnome_config_set_string("format/default", config->format);
	gnome_config_set_string("format/tooltip", config->tooltip);
	gnome_config_set_int("format/show_tooltip", config->show_tooltip);
	gnome_config_set_int("display/shadow", config->shadow);
	gnome_config_set_int("display/jtype", config->jtype);
	gnome_config_set_int("display/xpad", config->xpad);
	gnome_config_set_int("display/ypad", config->ypad);

	for (i = 0; i < fmt_count; i++)
	{
		fmt_list[i] = (gchar *) g_list_nth_data(config->formats, i);
	}
	gnome_config_set_vector("format/formats", fmt_count,
			(const char * const *)fmt_list);

	gnome_config_pop_prefix();
	gnome_config_sync();
	gnome_config_drop_all();

	return FALSE;
}

static void
show_properties(AppletWidget *applet, gpointer data)
{
	AppletData *config = gtk_object_get_user_data(GTK_OBJECT(applet));

	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *entry;
	GtkWidget *button;
	GtkWidget *frame;
	GtkWidget *spin;
	GtkWidget *radio = NULL;
	GtkObject *adj;
	gint i;

	static gchar *tooltip_option[] = {
		N_("No Tooltip"),
		N_("Display Uptime in Tooltip"),
		N_("Display Date/Time in Tooltip"),
		NULL
	};

	static gchar *shadow_name[] = {
		N_("No Border"),
		N_("Sunken"),
		N_("Raised"),
		N_("Etched In"),
		N_("Etched Out"),
		NULL
	};

	static gchar *justify_name[] = {
		N_("Left"),
		N_("Right"),
		N_("Center"),
		NULL
	};

	if (config->props != NULL)
	{
		gdk_window_raise(config->props->window);
		return;
	}

	/* create a happy peppy property box */
	config->props = gnome_property_box_new();
	gtk_window_set_title(GTK_WINDOW(config->props),
			_("ZClock properties"));

	/* this vbox will contain the date/time format options */
	vbox = gtk_vbox_new(FALSE, GNOME_PAD);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), GNOME_PAD);
	gnome_property_box_append_page(GNOME_PROPERTY_BOX(config->props),
			vbox, gtk_label_new(_("Date/Time Format")));

	/* this hbox holds the applet format label, entry and button */
	hbox = gtk_hbox_new(FALSE, GNOME_PAD);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, GNOME_PAD);
	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Clock format:")),
			FALSE, FALSE, 0);

	/* applet format entry box */
	entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(entry), config->format_prop);
	gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

	gtk_object_set_user_data(GTK_OBJECT(entry), config);
	gtk_signal_connect(GTK_OBJECT(entry), "changed",
			GTK_SIGNAL_FUNC(entry_changed_cb), &(config->format_prop));

	/* browse button for the applet format */
	button = gtk_button_new_with_label(_("Browse..."));
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_object_set_user_data(GTK_OBJECT(button), config);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(browse_clicked_cb), entry);

	/* frame for tooltip options */
	frame = gtk_frame_new(_("Tooltip"));
	gtk_box_pack_start_defaults(GTK_BOX(vbox), frame);

	/* vbox for tooltip options */
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), GNOME_PAD);
	gtk_container_add(GTK_CONTAINER (frame), vbox);

	/* populate the vbox with tooltip option radio buttons */
	i = 0;
	while (tooltip_option[i] != NULL)
	{
#ifndef HAVE_LIBGTOP
		if (i == 1) { i++; continue; } /* skip this option */
#endif
		radio = gtk_radio_button_new_with_label_from_widget(
				( i ? GTK_RADIO_BUTTON(radio) : NULL ),
				_(tooltip_option[i]));
		gtk_box_pack_start(GTK_BOX(vbox), radio, FALSE, FALSE, 0);

		if (config->show_tooltip_prop == i)
			gtk_toggle_button_set_active(
					GTK_TOGGLE_BUTTON(radio),
					TRUE);

		gtk_object_set_user_data(GTK_OBJECT(radio), config);
		gtk_signal_connect(GTK_OBJECT(radio), "toggled",
				GTK_SIGNAL_FUNC(radio_tooltip_toggle_cb),
				(gpointer) i);
		i++; /* increment index */
	};

	/* this hbox holds the tooltip format label, entry and button */
	hbox = gtk_hbox_new(FALSE, GNOME_PAD);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, GNOME_PAD);
	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Tooltip format:")),
			FALSE, FALSE, 0);

	config->ttbox = hbox;
	gtk_widget_set_sensitive(config->ttbox,
			config->show_tooltip_prop == 2 ? TRUE : FALSE);

	/* tooltip format entry box */
	entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(entry), config->tooltip_prop);
	gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

	gtk_object_set_user_data(GTK_OBJECT(entry), config);
	gtk_signal_connect(GTK_OBJECT(entry), "changed",
			GTK_SIGNAL_FUNC(entry_changed_cb), &(config->tooltip_prop));

	/* browse button for the tooltip format */
	button = gtk_button_new_with_label(_("Browse..."));
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_object_set_user_data(GTK_OBJECT(button), config);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(browse_clicked_cb), entry);

	/* vbox to hold display options */
	vbox = gtk_vbox_new(FALSE, GNOME_PAD);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), GNOME_PAD);
	gnome_property_box_append_page(GNOME_PROPERTY_BOX(config->props),
			vbox, gtk_label_new(_("Display")));


	/* hbox to hold the padding spinbuttons */
	hbox = gtk_hbox_new(FALSE, GNOME_PAD);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), GNOME_PAD);
	gtk_box_pack_start_defaults(GTK_BOX(vbox), hbox);
	gtk_box_pack_start(GTK_BOX(hbox),
			gtk_label_new(_("Horizontal padding:")),
			TRUE, FALSE, 0);

	/* spinbutton for x padding */
	adj = gtk_adjustment_new((gfloat) config->xpad, 0.0, 64.0, 1.0, 1.0, 1.0);
	spin = gtk_spin_button_new(GTK_ADJUSTMENT (adj), 1, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON (spin), TRUE);
	gtk_box_pack_start(GTK_BOX(hbox), spin, TRUE, TRUE, 0);
	gtk_object_set_user_data(GTK_OBJECT(adj), config);
	gtk_signal_connect(adj, "value-changed",
			GTK_SIGNAL_FUNC(padding_changed_cb),
			&(config->xpad_prop));

	gtk_box_pack_start(GTK_BOX(hbox),
			gtk_label_new(_("Vertical padding:")),
			TRUE, FALSE, 0);

	/* spinbutton for y padding */
	adj = gtk_adjustment_new((gfloat) config->ypad, 0.0, 64.0, 1.0, 1.0, 1.0);
	spin = gtk_spin_button_new(GTK_ADJUSTMENT (adj), 1, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON (spin), TRUE);
	gtk_box_pack_start(GTK_BOX(hbox), spin, TRUE, TRUE, 0);
	gtk_object_set_user_data(GTK_OBJECT(adj), config);
	gtk_signal_connect(adj, "value-changed",
			GTK_SIGNAL_FUNC(padding_changed_cb), &(config->ypad_prop));

	/* hbox to hold the display radio groups */
	hbox = gtk_hbox_new(FALSE, GNOME_PAD);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), GNOME_PAD);
	gtk_box_pack_start_defaults(GTK_BOX (vbox), hbox);

	/* frame for border styles */
	frame = gtk_frame_new(_("Border Style"));
	gtk_box_pack_start_defaults(GTK_BOX(hbox), frame);

	/* vbox to hold radio buttons for border style options */
	vbox = gtk_vbox_new(FALSE, 1);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), GNOME_PAD);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	/* populate the vbox with border style radio buttons */
	i = 0;
	while (shadow_name[i] != NULL)
	{
		radio = gtk_radio_button_new_with_label_from_widget(
				( i ? GTK_RADIO_BUTTON(radio) : NULL ),
				_(shadow_name[i]));
		gtk_box_pack_start(GTK_BOX(vbox), radio, FALSE, FALSE, 0);

		if (config->shadow_prop == i)
			gtk_toggle_button_set_active(
					GTK_TOGGLE_BUTTON(radio),
					TRUE);

		gtk_object_set_user_data(GTK_OBJECT(radio), config);
		gtk_signal_connect(GTK_OBJECT(radio), "toggled",
				GTK_SIGNAL_FUNC(radio_shadow_toggle_cb),
				(gpointer) i);
		i++; /* increment index */
	};

	/* frame for jusification types */
	frame = gtk_frame_new(_("Text Justification"));
	gtk_box_pack_start_defaults(GTK_BOX(hbox), frame);

	/* vbox to hold radio buttons for justification options */
	vbox = gtk_vbox_new(FALSE, 1);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), GNOME_PAD);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	/* populate the vbox with justification radio buttons */
	i = 0;
	while (justify_name[i] != NULL)
	{
		radio = gtk_radio_button_new_with_label_from_widget(
				( i ? GTK_RADIO_BUTTON(radio) : NULL ),
				_(justify_name[i]));
		gtk_box_pack_start(GTK_BOX(vbox), radio, FALSE, FALSE, 0);

		if (config->jtype_prop == i)
			gtk_toggle_button_set_active(
					GTK_TOGGLE_BUTTON(radio),
					TRUE);

		gtk_object_set_user_data(GTK_OBJECT(radio), config);
		gtk_signal_connect(GTK_OBJECT(radio), "toggled",
				GTK_SIGNAL_FUNC(radio_jtype_toggle_cb),
				(gpointer) i);
		i++; /* increment index */
	};

	/* set the handlers for the properties window */
	gtk_object_set_user_data(GTK_OBJECT(config->props), config);
	gtk_signal_connect(GTK_OBJECT(config->props), "apply",
			GTK_SIGNAL_FUNC(properties_apply_cb), NULL);
	gtk_signal_connect(GTK_OBJECT(config->props), "help",
			GTK_SIGNAL_FUNC(properties_help_cb), NULL);
	gtk_signal_connect(GTK_OBJECT(config->props), "destroy",
			GTK_SIGNAL_FUNC(properties_destroy_cb), NULL);

	gtk_widget_show_all(config->props);

	return;
}

static void 
properties_destroy_cb (GtkWidget *w, gpointer data)
{
	AppletData *config = gtk_object_get_user_data(GTK_OBJECT(w));
	g_assert(config != NULL);
	config->props = NULL;

	g_free(config->format_prop);
	config->format_prop = g_strdup(config->format);

	g_free(config->tooltip_prop);
	config->tooltip_prop = g_strdup(config->tooltip);

	config->show_tooltip_prop = config->show_tooltip;
	config->shadow_prop = config->shadow;
	config->jtype_prop = config->jtype;
}

static void
properties_help_cb (GtkWidget *w, gint page_num, gpointer data)
{
		GnomeHelpMenuEntry help_entry = {
				"zclock_applet",
				"index.html#ZCLOCK-PREFS"
		};
		gnome_help_display(NULL, &help_entry);
}

static void
properties_apply_cb (GtkWidget *w, gint page_num, gpointer data)
{
	AppletData *config = gtk_object_get_user_data(GTK_OBJECT(w));
	if (page_num != -1) return; /* Only apply once */

	g_assert(config != NULL);
	g_assert(config->format != NULL);
	g_assert(config->format_prop != NULL);
	g_assert(config->tooltip != NULL);
	g_assert(config->tooltip_prop != NULL);

	g_free(config->format);
	g_free(config->tooltip);

	config->format = g_strdup(config->format_prop);
	config->tooltip = g_strdup(config->tooltip_prop);

	config->show_tooltip = config->show_tooltip_prop;
	config->shadow = config->shadow_prop;
	config->jtype = config->jtype_prop;
	config->xpad = config->xpad_prop;
	config->ypad = config->ypad_prop;

	update_tooltip(config);

	gtk_zclock_set_format(GTK_ZCLOCK(config->zclock), config->format);
	gtk_frame_set_shadow_type(GTK_FRAME(config->zclock), config->shadow);
	gtk_zclock_set_justify(GTK_ZCLOCK(config->zclock), config->jtype);
	gtk_zclock_set_label_padding(GTK_ZCLOCK(config->zclock),
					config->xpad, config->ypad);

	/* save new settings immediately */
	applet_widget_sync_config(APPLET_WIDGET (config->applet));
}

static void
entry_changed_cb (GtkWidget *w, gpointer data)
{
	AppletData *config = gtk_object_get_user_data(GTK_OBJECT(w));
	gchar** value = (gchar **) data;
	g_assert(config != NULL);
	g_assert(config->props != NULL);

	g_free(*value);
	*value = gtk_editable_get_chars(GTK_EDITABLE (w), 0, -1);

	gnome_property_box_changed(GNOME_PROPERTY_BOX (config->props));
}

static void
browse_clicked_cb (GtkWidget *w, gpointer data)
{
	AppletData*	config = gtk_object_get_user_data(GTK_OBJECT (w));
	GtkEntry*	entry = GTK_ENTRY (data);
	gchar*		format = gtk_editable_get_chars(GTK_EDITABLE (entry), 0, -1);
	GList*		list = g_list_copy(config->formats);

	GtkWidget* zp = NULL;

	zp = gnome_zclock_preview_new(&list, &format);
	gtk_window_set_title(GTK_WINDOW (zp), _("ZClock formats"));
	gnome_dialog_set_parent(GNOME_DIALOG (zp),
			GTK_WINDOW (config->props));

	/* Maybe I should export functions for this...? */
	gtk_frame_set_shadow_type(GTK_FRAME (GNOME_ZCLOCK_PREVIEW (zp)->zclock),
					config->shadow_prop);
	gtk_zclock_set_justify(GTK_ZCLOCK (GNOME_ZCLOCK_PREVIEW (zp)->zclock),
					config->jtype_prop);
	
	if (gnome_dialog_run(GNOME_DIALOG(zp)) == 0)
	{
		gtk_entry_set_text(entry, format);
		g_free(format);

		gtk_editable_changed(GTK_EDITABLE (entry));

		g_list_free(config->formats);
		config->formats = list;
	}
}

static void
padding_changed_cb (GtkAdjustment* adj, gpointer data)
{
	AppletData* config = gtk_object_get_user_data(GTK_OBJECT(adj));
	gint *padding = data;
	*padding = (gint) adj->value;
	gnome_property_box_changed(GNOME_PROPERTY_BOX (config->props));
}

static void
radio_tooltip_toggle_cb (GtkWidget* w, gpointer data)
{
	g_assert(w != NULL);

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	{
		AppletData* config = gtk_object_get_user_data(GTK_OBJECT(w));
		g_assert(config != NULL);
		g_assert(config->props != NULL);

		config->show_tooltip_prop = GPOINTER_TO_INT(data);

		gtk_widget_set_sensitive(config->ttbox,
				config->show_tooltip_prop == 2 ? TRUE : FALSE);

		gnome_property_box_changed(GNOME_PROPERTY_BOX(config->props));
	}
}

static void
radio_shadow_toggle_cb (GtkWidget* w, gpointer data)
{
	g_assert(w != NULL);

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	{
		AppletData* config = gtk_object_get_user_data(GTK_OBJECT(w));
		g_assert(config != NULL);
		g_assert(config->props != NULL);

		config->shadow_prop = GPOINTER_TO_INT(data);
		gnome_property_box_changed(GNOME_PROPERTY_BOX(config->props));
	}
}

static void
radio_jtype_toggle_cb (GtkWidget* w, gpointer data)
{
	g_assert(w != NULL);

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	{
		AppletData* config = gtk_object_get_user_data(GTK_OBJECT(w));
		g_assert(config != NULL);
		g_assert(config->props != NULL);

		config->jtype_prop = GPOINTER_TO_INT(data);
		gnome_property_box_changed(GNOME_PROPERTY_BOX(config->props));
	}
}

static void
show_help(AppletWidget* applet, gpointer data)
{
		GnomeHelpMenuEntry help_entry = { "zclock_applet", "index.html" };
		gnome_help_display(NULL, &help_entry);
}

static void
show_about(AppletWidget *applet, gpointer data)
{
	static GtkWidget *about = NULL;
	static const gchar *authors[] = {
		"Zack Hobson <zack@malfunktion.net>",
		NULL
	};

	if (about != NULL)
	{
		gdk_window_show(about->window);
		gdk_window_raise(about->window);
		return;
	}

	about = gnome_about_new(_("ZClock Applet"), VERSION,
		"Copyright (C) 2000 Zack Hobson", authors,
		_("A simple but highly configurable clock applet for GNOME"),
		NULL);
	gtk_signal_connect(GTK_OBJECT(about), "destroy",
		    GTK_SIGNAL_FUNC(gtk_widget_destroyed), &about);
	gtk_widget_show(about);
	return;
}

