/* -*- C++ -*-

  This file is part of ViPEC
  Copyright (C) 1991-2001 Johan Rossouw (jrossouw@alcatel.altech.co.za)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <TLinPhysical.h>

#include <Strings.h>
#include <TwoPorts.h>
#include <Exception.h>
#include <MainWindow.h>

#include "../images/tlin2.xpm"

QPixmap* TLinPhysical::pixmap_ = 0;

//-----------------------------------------------------------------
TLinPhysical::TLinPhysical( const QPoint& center )
  : Component( center )
{
}

//-----------------------------------------------------------------
TLinPhysical::~TLinPhysical()
{
}

//-----------------------------------------------------------------
const QString& TLinPhysical::getCatagory() const
{
  return Strings::TransmissionLineCatagory;
}

//-----------------------------------------------------------------
const QString& TLinPhysical::getName() const
{
  return Strings::TLinPhysicalName;
}

//-----------------------------------------------------------------
const QPixmap& TLinPhysical::getPixmap() const
{
  if (!pixmap_)
    {
      pixmap_ = new QPixmap( tlin2_xpm );
    }
  return *pixmap_;
}

//-----------------------------------------------------------------
void TLinPhysical::initComponent()
{
  boundingRect_ = QRect(-24, -10, 48, 20);
  addNode(-24,0, FALSE, FALSE, TRUE );
  addNode(24,0, FALSE, FALSE, TRUE );
  addAttribute( Strings::AttrLineWidth, "W" );
  addAttribute( Strings::AttrLineLength, "L" );
  addAttribute( Strings::AttrSubstrateName, "S" );
}

//-----------------------------------------------------------------
void TLinPhysical::drawSymbol(QPainter* p)
{
  p->drawRect(-24,-6,48,12);
}

//-----------------------------------------------------------------
bool TLinPhysical::initSweep()
{
  bool changed = FALSE;
  TReal w = w_;
  TReal l = l_;
  QString substrate = substrate_;
  w_ = getAttributeValue( "W" ) * getDimensionValue( Strings::LengthDim );
  l_ = getAttributeValue( "L" ) * getDimensionValue( Strings::LengthDim );
  substrate_ = getAttributeStr( "S" );

  const SubstrateDefinition* sub = MainWindow::instance()->getSubstrate( substrate_ );
  if ( sub == 0 )
    {
      throw Exception::NoSuchSubstrate();
    }

  bool hasChanged = sub->changed();
  if ( (w!=w_) || (l!=l_) || ( substrate!=substrate_) || hasChanged )
    {
      changed = TRUE;
    }
  return changed;
}

//-----------------------------------------------------------------
void TLinPhysical::addToAdmittanceMatrix( TReal freq, Matrix* yn )
{
  const SubstrateDefinition* sub = MainWindow::instance()->getSubstrate( substrate_ );
  ASSERT( sub != 0 );
  TReal er = sub->er();
  TReal h = sub->height() * getDimensionValue( Strings::LengthDim ) ;
  TReal t = sub->thickness() * getDimensionValue( Strings::LengthDim );
  TReal rho = sub->rho();
  
  Matrix y;
  switch ( sub->type() )
    {
    case SubstrateDefinition::microstrip:
      TwoPorts::getMLIN( freq, w_, l_, t, h, er, rho, y);
      break;
    case SubstrateDefinition::stripline:
      TwoPorts::getSLIN( freq, w_, l_, t, h, er, rho, y );
      break;
    }
  uint nodes [4];
  nodes[0] = node(0)->getNodeNumber();
  nodes[1] = 0;
  nodes[2] = node(1)->getNodeNumber();
  nodes[3] = 0;
  yn->insertTwoPort(y, nodes);
}
