/*
 * ===========================
 * VDK Visual Development Kit
 * Version 1.0.
 * Revision 5
 * January 2000
 * ===========================
 *
 * Copyright (C) 1998, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */
/*
OVERVIEW
--------
This file has the aim to be a footstep that shows how to make a
gtk+ widget wrapper in vdk.
We choose here to wrap gtk_calendar() widget.
*/
#include <vdkb/vdkextext.h>
#include <vdk/forms.h>
#include <stdio.h>
#include <sys/stat.h>
/*
    is a composite widget, actually a scrolled window
    with both horizontal and vertical scrollbars
    that contains a gtkextext widget.
    So <widget> will point to scrolled window
    and <sigwid> to gtkextext.
    Signals will interest <sigwid>
 */
VDKExText::VDKExText(VDKForm* owner):
  VDKObject(owner),
  // sets properties
  Editable("Editable",this,true,
    	   &VDKExText::SetEditable,&VDKExText::GetEditable),
  Pointer("Pointer",this,0,
	  &VDKExText::SetPointer,&VDKExText::GetPointer),
  Length("Length",this,0,&VDKExText::GetLength),
  Column("Column",this,0,
	 &VDKExText::SetColumn,&VDKExText::GetColumn),
  Line("Line",this,0,
       &VDKExText::SetLine,&VDKExText::GetLine),
  MaxUndo("MaxUndo",this,5,&VDKExText::SetMaxUndo),
  ShowLineNumbers("ShowLineNumbers",this,false,&VDKExText::SetShowLineNumbers),
  LineAutoSelect("LineAutoSelect",this,false,&VDKExText::SetLineAutoSelect),
  Changed(false)
{
    GdsEditor *text = GDS_EDITOR(gds_editor_new());
    sigwid = GTK_WIDGET(text);
    widget = gtk_scrolled_window_new(GTK_EXTEXT(text)->hadj,
              GTK_EXTEXT(text)->vadj);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(widget),
                      GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    gtk_container_add(GTK_CONTAINER(widget),sigwid); 
    gtk_widget_show(sigwid);
    // connects to "realize" (into VDKObject class) 
    ConnectDefaultSignals();
    // 
    LocalConnect();
  }
/*
*/
void
VDKExText::LocalConnect()
{
    gtk_signal_connect (GTK_OBJECT (sigwid), "changed", 
			GTK_SIGNAL_FUNC (VDKExText::ChangedHandler), this);
    gtk_signal_connect (GTK_OBJECT (sigwid), "move_to_row", 
			GTK_SIGNAL_FUNC (VDKExText::MoveToRowHandler), this);
    gtk_signal_connect (GTK_OBJECT (sigwid), "move_to_column", 
			GTK_SIGNAL_FUNC (VDKExText::MoveToColumnHandler), this);

}

/*
*/
void
VDKExText::ChangedHandler(GtkWidget* wid, gpointer gp)
{
    g_return_if_fail (wid != NULL); 
    g_return_if_fail (gp != NULL);
    VDKExText* text = reinterpret_cast<VDKExText*>(gp);
    text->Changed = true;
    text->SignalEmit(text_changed_signal);
}
/*
*/
void
VDKExText::MoveToRowHandler(GtkWidget* wid,
                            gint row,
                            gpointer gp)
{
    g_return_if_fail (wid != NULL); 
    g_return_if_fail (gp != NULL);
    VDKExText* text = reinterpret_cast<VDKExText*>(gp);
    text->SignalEmit(move_to_row_signal);
    /*
    since signature is not the default one
    we can connect with this using dynamic signal
    and <gtk> arg set to false
    */
    text->SignalEmit("move_to_row_signal");
}
/*
*/
void
VDKExText::MoveToColumnHandler(GtkWidget* wid,
                            gint col,
                            gpointer gp)
{
    g_return_if_fail (wid != NULL); 
    g_return_if_fail (gp != NULL);
    VDKExText* text = reinterpret_cast<VDKExText*>(gp);
    text->SignalEmit(move_to_column_signal);
    /*
    since signature is not the default one
    we can connect with this using dynamic signal
    and <gtk> arg set to false
    */
    text->SignalEmit("move_to_column_signal");
}

/*
 */
VDKExText::~VDKExText()
{
  // nothing to do
}
/*
 */
void 
VDKExText::TextInsert(char* t, int nchar)
{
  int pos;
  if(!GTK_WIDGET_REALIZED(sigwid))
    return;
  else
    {
      g_return_if_fail(t != NULL);
      nchar = nchar > 0 ? nchar : strlen(t);
      Freeze();
      pos = Pointer;
      gds_editor_insert_text(GDS_EDITOR(sigwid),t,nchar,&pos);
      Thaw();
      Changed = true;
    }
}

/*
*/

void 
VDKExText::Clear()
{
  if(!GTK_WIDGET_REALIZED(sigwid))
    return;
  else
    {
      Pointer = 0;
      Freeze();
      ForwardDelete(Length);
      Thaw();
      Changed = true;
    }
}
/*
 */
int 
VDKExText::LoadFromFile(char* filename)
{
    if(!GTK_WIDGET_REALIZED(GTK_EXTEXT(sigwid)) )
        return 0;
    FILE* fp;
    char* edbuff,*p;
    fp = fopen(filename,"r");
    if(fp)
        {
	  int c;
	  int maxundo = MaxUndo;
	  struct stat info;
	  stat(filename,&info);
	  unsigned int size = info.st_size;
	  edbuff = p = new char[size];
	  while( (c = fgetc(fp)) != EOF)
	    *p++ = (char) c;
	  fclose(fp);
	  // reset undo
	  MaxUndo = 0;
	  Clear();
	  Freeze();
	  TextInsert(edbuff, size);
	  Thaw();
	  Line = 0;
	  MaxUndo = maxundo;
	  delete edbuff;
	  Changed = false;
	  return TRUE;
        }
    else
        return FALSE;
}
/*
 */
int 
VDKExText::SaveToFile(char* filename)
{
  char* p;
  FILE* fp;
  int result = 0;
  if(!GTK_WIDGET_REALIZED(sigwid))
    return FALSE;
  p = GetChars(0);
  if(p)
    {
      fp = fopen(filename,"w+b");
      if(fp)
	  result = fwrite(p, GetLength(), 1,fp);
      fclose(fp);
      g_free(p);
      return result == 1;
    }
  else 
    return FALSE;
}
/*
*/
void 
VDKExText::SetFont(VDKFont* font)
{
    GtkStyle* style = gtk_style_copy(gtk_widget_get_style(sigwid));
    g_return_if_fail(style != NULL);
    gtk_style_ref(style);
    style->font = (GdkFont*) (*font);
    gtk_widget_set_style(sigwid,style);
}
/*
 */
void 
VDKExText::SetBackground(VDKRgb rgb, GtkStateType state)
{
    if(!GTK_WIDGET_REALIZED(sigwid))
        return;
    VDKColor *color = new VDKColor(Owner(),rgb.red,rgb.green,rgb.blue);
    GtkStyle* style = gtk_style_copy(gtk_widget_get_style(sigwid));
    g_return_if_fail(style != NULL); 
    gtk_style_ref(style); 
    style->bg[state] = *(color->Color()); 
     gtk_widget_set_style(sigwid,style);
}


/*
 */
void 
VDKExText::SetForeground(VDKRgb rgb, GtkStateType state)
{
    if(!GTK_WIDGET_REALIZED(sigwid))
        return;
    VDKColor *color = new VDKColor(Owner(),rgb.red,rgb.green,rgb.blue);
    GtkStyle* style = gtk_style_copy(gtk_widget_get_style(sigwid));
    g_return_if_fail(style != NULL); 
    gtk_style_ref(style); 
    style->text[state] = *(color->Color()); 
    gtk_widget_set_style(sigwid,style);
}
/*
*/
char* 
VDKExText::GetChars(int start, int end = -1)
{ 
  return sigwid ? 
  gds_editor_get_chars(GDS_EDITOR(sigwid),
  			     start,
  			     end == -1 ? GetLength(): end) : NULL;
}
/*
*/
void 
VDKExText::Thaw() 
{ 
    if(sigwid)
        gtk_extext_thaw(GTK_EXTEXT(sigwid)); 
}
/*
*/
void 
VDKExText::Eol()
{ 
    if(sigwid)
        TextInsert("\n");
}
/*
*/
void
VDKExText::BackwardDelete(int nchars)
{ 
  if(sigwid)
    {
      int pos = gtk_editable_get_position(GTK_EDITABLE(sigwid));
      Changed = true;
      gds_editor_delete_text(GDS_EDITOR(sigwid),pos-nchars,pos);
    }
}
/*
*/
void
VDKExText::ForwardDelete(int nchars)
{ 
    if(sigwid)
    {
      int pos = gtk_editable_get_position(GTK_EDITABLE(sigwid));
      Changed = true;
      gds_editor_delete_text(GDS_EDITOR(sigwid),pos,nchars);
    }
}
/*
*/
void 
VDKExText::Freeze() 
{ 
    if(sigwid)
        gtk_extext_freeze(GTK_EXTEXT(sigwid)); 
}
/*
 PROPERTIES SETTING/GETTING
*/
 void 
VDKExText::SetPointer(int pos)
{ 
    if(sigwid)
        gtk_editable_set_position(GTK_EDITABLE(sigwid),pos);
}

int 
VDKExText::GetPointer() 
{ 
return sigwid ? gtk_editable_get_position(GTK_EDITABLE(sigwid)) :(int) NULL; 
}

void 
VDKExText::SetColumn(int col)
{ 
    if(sigwid)
        gtk_extext_set_column(GTK_EXTEXT(sigwid), col); 
}

int 
VDKExText::GetColumn()
{ 
    return sigwid ? gtk_extext_get_column(GTK_EXTEXT(sigwid)) : -1; 
}

void 
VDKExText::SetLine(int line)
{ 
        if(sigwid)
            gtk_extext_set_line(GTK_EXTEXT(sigwid),line); 
}

int
VDKExText::GetLine(void)
{ 
    return  sigwid ? gtk_extext_get_line(GTK_EXTEXT(sigwid)) : -1; 
}
    
void 
VDKExText::SetEditable(bool flag)
{ 
    if(sigwid)
    gtk_editable_set_editable(GTK_EDITABLE(sigwid),flag); 
}

bool 
VDKExText::GetEditable(void)
{ 
    return sigwid ? gtk_extext_get_editable(GTK_EXTEXT(sigwid)) : false; 
}

unsigned int 
VDKExText::GetLength() 
{ 
    return sigwid ? gtk_extext_get_length(GTK_EXTEXT(sigwid)) : 0; 
}
/*
*/
void 
VDKExText::SetMaxUndo(int m)
{
  if(sigwid)
    gtk_extext_undo_set_max(GTK_EXTEXT(sigwid),m);
}
/*
 */
bool
VDKExText::Undo(void)
{
   if(sigwid && !gtk_extext_undo_is_empty(GTK_EXTEXT(sigwid)))
     {
       gtk_extext_undo(GTK_EXTEXT(sigwid));
       return true;
     }
   else
     return false;
}
/*
 */
void 
VDKExText::SetShowLineNumbers(bool flag)
{
  gtk_extext_set_draw_numbers(GTK_EXTEXT(sigwid), (gboolean) flag);
  gtk_widget_queue_draw(GTK_WIDGET(sigwid));
}

/*
 */
void 
VDKExText::SetLineAutoSelect(bool flag)
{
  gtk_extext_set_select_line(GTK_EXTEXT(sigwid), (gboolean) flag);
}
