/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_nodes.h"

#include <stdlib.h>

#include "visu_tools.h"

/**
 * SECTION:visu_nodes
 * @short_description: Defines the elementary structure to store
 * informations about an element in a box.
 *
 * <para>In V_Sim, elements are drawn in a box. The #VisuNode
 * structure is used to represent an instance of an element position
 * somewhere in the box. This element can have several characteristics
 * such as its translation or its visibility.</para>
 *
 * <para>All nodes are stored in a #VisuData object in a two
 * dimensional array. The first dimension is indexed by the
 * #VisuElement of the node and the second corresponds to the number
 * of this node for this element. When a node is own by a #VisuData,
 * the two integers, that control the indexes in this array, are not
 * negative. See the #VisuNode structure for further
 * explanations.</para>
 *
 * <para>The only basic informations own by the #VisuNode structure is
 * basicaly its position. To add further informations (such as
 * orientation for the spin), define a node property using
 * visu_node_property_newPointer().</para>
 */

/**
 * VisuNode:
 * @xyz: (in) (array fixed-size=3): an array of three floating point values that positions the node in (x, y, z) ;
 * @translation: (in) (array fixed-size=3): an array of three floating point values that translates the
 *               node to its drawn position from (x, y, z) ;
 * @number: an integer that corresponds to its position in the entry file, it references
 *          also the node itself in the array 'fromNumberToVisuNode' of the #VisuData
 *          that contains the node ;
 * @posElement: an integer that is the position of the #VisuElement of the node
 *              in the array 'fromIntToVisuElement' of the #VisuData object that
 *              contains the node ;
 * @posNode: an integer that is the position of the node itself in the array
 *           'nodes' of the #VisuData object that contains the node ;
 * @rendered: a boolean to store if the node is drwn or not.
 *
 * Structure to store primary data of a node.
 */

/* Local routines. */
static void freePropertiesStruct(gpointer data);
static void removeNodeProperty(gpointer key, gpointer value, gpointer data);
static void reallocNodeProperty(gpointer key, gpointer value, gpointer data);
static void createNodeproperty(gpointer key, gpointer value, gpointer data);
static VisuNode* newOrCopyNode(VisuNodeArray *nodeArray, int iEle, int oldNodeId);

/* The number of nodes to be reallocated each time the visu_node_array_getNewNode()
   is called. */
#define REALLOCATION_STEP 100

/* The key of the original node property. */
#define ORIGINAL_ID       "originalId"

struct twoNodes
{
  VisuNode *oldNode;
  VisuNode *newNode;
};

struct _VisuNodeProperty
{
  /* A label to define the property. */
  gchar *name;

  /* A pointer to the array of nodes these properties are related to. */
  VisuNodeArray *array;

  /* The type of the property. */
  GType gtype;

  /* This table has the same size and structure
     than the node array object it is related to.
     Only one of the following data array is allocated. */
  gpointer **data_pointer;
  int **data_int;

  /* In the case of pointer data, one can give the new, copy and free routine. */
  /* This method is called for each stored token,
     if not NULL when the table is freed. */
  GFunc freeTokenFunc;
  /* This method is used to create/copy a token of the data array. */
  GCopyFunc newOrCopyTokenFunc;
  /* This value stores a pointer on a user data given when
     the object is created. This pointer is given to the copy
     or the free function. */
  gpointer user_data;
};

/**
 * visu_node_setVisibility:
 * @node: a #VisuNode object ;
 * @visibility: a boolean.
 *
 * This method is used to turn on or off the drawing of the specified node.
 *
 * Returns: true if the calling method should recreate the node list with
 * a call to the visu_data_createAllNodes() method.
 */
int visu_node_setVisibility(VisuNode* node, gboolean visibility)
{
  g_return_val_if_fail(node, 0);

  if (node->rendered == visibility)
    return 0;
  node->rendered = visibility;
  return 1;
}

/**
 * visu_node_getVisibility:
 * @node: a #VisuNode object.
 *
 * This method is used get the status of the drawing state of a node.
 *
 * Returns: true if the node is rendered, false otherwise.
 */
gboolean visu_node_getVisibility(VisuNode* node)
{
  g_return_val_if_fail(node, FALSE);

  return node->rendered;
}

/**
 * visu_node_newValues:
 * @node: an allocated #VisuNode object ;
 * @xyz: (in) (array fixed-size=3): the coordinates to set.
 * 
 * Set the coordinates and set all other values to default.
 */
void visu_node_newValues(VisuNode *node, float xyz[3])
{
  g_return_if_fail(node);

  DBG_fprintf(stderr, "Visu Node: set new position for node %d (%g;%g;%g).\n",
              node->number, xyz[0], xyz[1], xyz[2]);
  node->xyz[0]         = xyz[0];
  node->xyz[1]         = xyz[1];
  node->xyz[2]         = xyz[2];
  node->translation[0] = 0.;
  node->translation[1] = 0.;
  node->translation[2] = 0.;
  node->rendered       = TRUE;
}

/**
 * visu_node_copy:
 * @nodeTo: an allocated #VisuNode object ;
 * @nodeFrom: an allocated #VisuNode object.
 * 
 * Copy all attributes of the object @nodeFrom to @nodeTo.
 */
void visu_node_copy(VisuNode *nodeTo, VisuNode *nodeFrom)
{
  g_return_if_fail(nodeTo && nodeFrom);

  nodeTo->xyz[0]         = nodeFrom->xyz[0];
  nodeTo->xyz[1]         = nodeFrom->xyz[1];
  nodeTo->xyz[2]         = nodeFrom->xyz[2];
  nodeTo->translation[0] = nodeFrom->translation[0];
  nodeTo->translation[1] = nodeFrom->translation[1];
  nodeTo->translation[2] = nodeFrom->translation[2];
  nodeTo->rendered       = nodeFrom->rendered;
}



/***************/
/* Node Arrays */
/***************/

/**
 * visu_node_array_newNodes:
 * @nTypes: the size of nNodes.
 * @nNodes: (in) (array length=nTypes): an array giving the number of nodes per element.
 *
 * Create a new #VisuNodeArray structure, allocate all necessary values.
 *
 * Returns: (transfer none): a newly created #VisuNodeArray object.
 */
VisuNodeArray* visu_node_array_newNodes(unsigned int nTypes, unsigned int *nNodes)
{
  VisuNodeArray *array;
  unsigned int i, j;

  g_return_val_if_fail(nTypes > 0 && nNodes, (VisuNodeArray*)0);

  array = g_malloc(sizeof(VisuNodeArray));
  DBG_fprintf(stderr, "Visu Node: creating a new VisuNodeArray %p (%d types).\n",
	      (gpointer)array, nTypes);

  array->ntype                = nTypes;
  array->idCounter            = 0;
  array->nodes                = g_malloc(sizeof(VisuNode*) * array->ntype);
  array->numberOfNodes        = g_malloc(sizeof(unsigned int) * array->ntype);
  array->numberOfStoredNodes  = g_malloc(sizeof(unsigned int) * array->ntype);

  array->nNodes = 0;
  for(i = 0; i < array->ntype; i++)
    {
      g_return_val_if_fail(nNodes[i] > 0, (VisuNodeArray*)0);
      DBG_fprintf(stderr, " | for type %d -> %d nodes allocated.\n",
		  i, nNodes[i]);

      array->nodes[i]               = g_malloc(sizeof(VisuNode) * nNodes[i]);
      array->numberOfNodes[i]       = nNodes[i];
      array->numberOfStoredNodes[i] = 0;
      for (j = 0; j < nNodes[i]; j++)
	{
/* 	  array->nodes[i][j].number     = array->nNodes; */
	  array->nodes[i][j].posElement = i;
	  array->nodes[i][j].posNode    = j;
	  array->nNodes += 1;
	}
    }
  DBG_fprintf(stderr, " | total number of allocated nodes %d.\n",
	      array->nNodes);
  array->nbOfAllStoredNodes = 0;
  array->nodeTableSize      = array->nNodes;
  array->nodeTable          = g_malloc(sizeof(VisuNode*) * array->nNodes);

  array->properties = g_hash_table_new_full(g_str_hash, g_str_equal,
					    NULL, freePropertiesStruct);
  DBG_fprintf(stderr, " | size = %do\n",
	      (int)(array->nNodes * sizeof(VisuNode) +
		    sizeof(unsigned int) * array->ntype * 2 +
		    sizeof(VisuNode*) * array->ntype +
		    sizeof(VisuNode*) * array->nNodes + sizeof(VisuNodeArray)));

  /* We add a node property that is > 0 if the node is a duplicate
     node and negative if not. The value is the node id that the
     duplicate refers to. */
  DBG_fprintf(stderr, " | create the original node property.\n");
  array->origProp   = visu_node_property_newInteger(array, ORIGINAL_ID);
  array->nOrigNodes = 0;

  return array;
}

/**
 * visu_node_array_removeNodes:
 * @nodeArray: a #VisuNodeArray object.
 * @nodeNumbers: (in) (array): an array of integers (negative terminated).
 *
 * Remove the given #VisuNode from the @nodeArray. The properties
 * are also updated.
 */
void visu_node_array_removeNodes(VisuNodeArray *nodeArray, int *nodeNumbers)
{
  VisuNode *node;
  unsigned int i, iEle, iNode;

  g_return_if_fail(nodeArray && nodeNumbers);

  /* For each element in the given node array, we take the last node
     of the same element and it takes it position. */
  DBG_fprintf(stderr, "Visu Node: removing nodes from array %p.\n",
	      (gpointer)nodeArray);
  for (i = 0; nodeNumbers[i] >= 0; i++)
    {
      node = nodeArray->nodeTable[nodeNumbers[i]];
DBG_fprintf(stderr, "node : %p\n",(gpointer)node);
      g_return_if_fail(node);
DBG_fprintf(stderr, "OK\n");
      iEle  = node->posElement;
      iNode = node->posNode;
      g_return_if_fail(node->number == (unsigned int)nodeNumbers[i]);

      nodeArray->numberOfStoredNodes[iEle] -= 1;

      if (nodeArray->numberOfStoredNodes[iEle] > 0)
	{
	  /* First, we copy the properies following the same scheme,
	     the last node of the given element is copy instead of
	     the given one. */
	  g_hash_table_foreach(nodeArray->properties, removeNodeProperty,
			       nodeArray->nodes[iEle] + iNode);

	  /* Then, we copy the node values themselves. */
	  visu_node_copy(nodeArray->nodes[iEle] + iNode,
		       nodeArray->nodes[iEle] + nodeArray->numberOfStoredNodes[iEle]);
	  /* We update the index values. */
	  nodeArray->nodes[iEle][iNode].posNode = iNode;
	  nodeArray->nodes[iEle][iNode].number =
	    nodeArray->nodes[iEle][nodeArray->numberOfStoredNodes[iEle]].number;
	  nodeArray->nodeTable[nodeArray->nodes[iEle][iNode].number] =
	    nodeArray->nodes[iEle] + iNode;
	}
      /* Nullify the removed node. */
      nodeArray->nodeTable[nodeNumbers[i]] = (VisuNode*)0;
      /* Update counters. */
      nodeArray->nbOfAllStoredNodes -= 1;

      DBG_fprintf(stderr, "Visu Node: %d removed, population for element %d is now:", nodeNumbers[i], iEle);
      DBG_fprintf(stderr, " %d/%d # %d/%d\n", nodeArray->nbOfAllStoredNodes,
		  nodeArray->nNodes, nodeArray->numberOfStoredNodes[iEle],
		  nodeArray->numberOfNodes[iEle]);
    }
  /* We get the last non NULL node pointer in the nodeTable array to make the
     idCounter having this value to avoid to much reallocation of the nodeTable
     array. */
  for (; nodeArray->idCounter > 0 && !nodeArray->nodeTable[nodeArray->idCounter - 1];
       nodeArray->idCounter -= 1);
  DBG_fprintf(stderr, "Visu Node: idCounter is set to %d.\n", nodeArray->idCounter);
  DBG_fprintf(stderr, " | size = %do\n",
	      (int)(nodeArray->nNodes * sizeof(VisuNode) +
		    sizeof(int) * nodeArray->ntype * 2 +
		    sizeof(VisuNode*) * nodeArray->ntype +
		    sizeof(VisuNode*) * nodeArray->nNodes + sizeof(VisuNodeArray)));
}

/**
 * visu_node_array_removeAllDuplicateNodes:
 * @nodeArray: a #VisuNodeArray object.
 * @nodeNumbers: a location to create an arry of int.
 *
 * Remove all nodes that are not original in the box. The list of
 * removed nodes ids are stored in @nodeNumbers. This array is
 * allocated and should be freed with g_free(). If no nodes are
 * removed, this array is not allocated.
 *
 * Returns: TRUE if some nodes have been removed (and @nodeNumbers allocated).
 */
gboolean visu_node_array_removeAllDuplicateNodes(VisuNodeArray *nodeArray,
					  int **nodeNumbers)
{
  int nb;
  unsigned int i, j;

  g_return_val_if_fail(nodeArray && nodeNumbers && !*nodeNumbers, FALSE);
  g_return_val_if_fail(nodeArray->origProp, FALSE);

  nb = 0;
  for (i = 0; i < nodeArray->ntype; i++)
    for (j = 0; j < nodeArray->numberOfStoredNodes[i]; j++)
      if (nodeArray->origProp->data_int[i][j] >= 0)
	nb += 1;

  if (nb > 0)
    {
      *nodeNumbers = g_malloc(sizeof(int) * (nb + 1));
      nb = 0;
      for (i = 0; i < nodeArray->ntype; i++)
	for (j = 0; j < nodeArray->numberOfStoredNodes[i]; j++)
	  if (nodeArray->origProp->data_int[i][j] >= 0)
	    (*nodeNumbers)[nb++] = nodeArray->nodes[i][j].number;
      (*nodeNumbers)[nb] = -1;
      visu_node_array_removeNodes(nodeArray, *nodeNumbers);
      return TRUE;
    }
  else
    return FALSE;
}

/**
 * visu_node_array_getOriginal:
 * @nodeArray: a #VisuNodeArray object.
 * @nodeId: a node id.
 *
 * Test if the given @nodeId is an original or a replica for the
 * periodisation.
 *
 * Returns: TRUE for an original node.
 */
gint visu_node_array_getOriginal(VisuNodeArray *nodeArray, guint nodeId)
{
  VisuNode *node;
  gint orig;

  g_return_val_if_fail(nodeArray && nodeArray->origProp, -1);
  g_return_val_if_fail(nodeId < nodeArray->idCounter, -1);

  orig = (gint)nodeId;
  do
    {
      node = nodeArray->nodeTable[orig];
      orig = nodeArray->origProp->data_int[node->posElement][node->posNode];
    }
  while (orig >= 0);
/*   DBG_fprintf(stderr, "Visu Node: get original from %d: %d (%d).\n", */
/* 	      nodeId, node->number, */
/* 	      nodeArray->origProp->data_int[node->posElement][node->posNode]); */
  return (node->number == nodeId)?-1:(gint)node->number;
}

/**
 * visu_node_array_setOriginal:
 * @nodeArray: a #VisuNodeArray object.
 * @nodeId: a node id.
 *
 * Test if the given @nodeId is an original or a replica for the
 * periodisation.
 *
 * Returns: TRUE for an original node.
 */
gboolean visu_node_array_setOriginal(VisuNodeArray *nodeArray, guint nodeId)
{
  VisuNode *node;
  gint orig;

  g_return_val_if_fail(nodeArray && nodeArray->origProp, -1);
  g_return_val_if_fail(nodeId < nodeArray->idCounter, -1);

  node = nodeArray->nodeTable[nodeId];
  orig = nodeArray->origProp->data_int[node->posElement][node->posNode];
  nodeArray->origProp->data_int[node->posElement][node->posNode] = -1;

  DBG_fprintf(stderr, "Visu Node: set original for %d (%d).\n",
	      nodeId, orig);
  return (orig != -1);
}

/**
 * visu_node_array_freeNodes:
 * @nodeArray: a #VisuNodeArray object.
 *
 * Free the given object and all associated memory.
 */
void visu_node_array_freeNodes(VisuNodeArray *nodeArray)
{
  unsigned int i;

  g_return_if_fail(nodeArray);

  DBG_fprintf(stderr, "Visu Node: freeing the VisuNodeArray %p ...\n",
	      (gpointer)nodeArray);

  /* We first remove the properties, before the node description because
     these descriptions may be relevant. */
  if (nodeArray->properties)
    g_hash_table_destroy(nodeArray->properties);

  if (nodeArray->nodeTable)
    g_free(nodeArray->nodeTable);
  
  if (nodeArray->nodes)
    {
      for(i = 0; i < nodeArray->ntype; i++)
	g_free(nodeArray->nodes[i]);
      g_free(nodeArray->nodes);
    }

  if (nodeArray->numberOfNodes)
    g_free(nodeArray->numberOfNodes);
  if (nodeArray->numberOfStoredNodes)
    g_free(nodeArray->numberOfStoredNodes);

  g_free(nodeArray);
  DBG_fprintf(stderr, "Visu Node: freeing node array ... OK.\n");
}
/**
 * visu_node_array_switchNumber:
 * @nodeArray: a #VisuNodeArray object.
 * @from: a node id.
 * @to: another node id.
 *
 * Two nodes of @nodeArray switches their number.
 *
 * Since: 3.6
 *
 * Returns: TRUE if number is switched.
 */
gboolean visu_node_array_switchNumber(VisuNodeArray *nodeArray, guint from, guint to)
{
  VisuNode *nodeFrom, *nodeTo;

  if (from == to)
    return FALSE;

  nodeFrom = nodeArray->nodeTable[from];
  nodeTo   = nodeArray->nodeTable[to];
  nodeArray->nodeTable[from] = nodeTo;
  nodeArray->nodeTable[to]   = nodeFrom;
  nodeFrom->number = to;
  nodeTo->number   = from;
  return TRUE;
}


/**
 * visu_node_array_allocateNewNodes:
 * @nodeArray: a #VisuNodeArray object ;
 * @iEle: the id of a #VisuElement to expand the number of allocated nodes ;
 * @step: the number of newly allocated nodes.
 *
 * When a new node is required, using visu_node_array_getNewNode() or visu_node_array_getCopyNode()
 * the storing arrays are expand automatically with a fixed increment. If the user
 * wants to control this increment, he should call this routine before the get() ones
 * with the appropriated @step value.
 */
void visu_node_array_allocateNewNodes(VisuNodeArray *nodeArray, unsigned int iEle,
			       unsigned int step)
{
  unsigned int j;
  VisuNode *oldNodeList;

  g_return_if_fail(nodeArray && iEle < nodeArray->ntype);

  oldNodeList = nodeArray->nodes[iEle];
  DBG_fprintf(stderr, "Visu Node: reallocation needed for element "
	      "%d with %d nodes.\n", iEle, step);
  nodeArray->numberOfNodes[iEle] += step;
  nodeArray->nodes[iEle] = g_realloc(nodeArray->nodes[iEle],
				     sizeof(VisuNode) *
				     nodeArray->numberOfNodes[iEle]);
  nodeArray->nNodes += step;
  nodeArray->nodeTableSize += step;
  nodeArray->nodeTable = g_realloc(nodeArray->nodeTable,
				   sizeof(VisuNode*) * nodeArray->nodeTableSize);
  DBG_fprintf(stderr, " | (all)%d/%d # (%d)%d/%d # %d\n",
	      nodeArray->nbOfAllStoredNodes, nodeArray->nNodes, iEle,
	      nodeArray->numberOfStoredNodes[iEle], nodeArray->numberOfNodes[iEle],
	      nodeArray->nodeTableSize);
  /* We set the default values for the new nodes. */
  for (j = nodeArray->numberOfStoredNodes[iEle];
       j < nodeArray->numberOfNodes[iEle]; j++)
    {
      nodeArray->nodes[iEle][j].posElement = iEle;
      nodeArray->nodes[iEle][j].posNode    = j;
    }
  /* If the node list has been moved, we need to reassign pointers
     of array nodeTable. */
  if (oldNodeList != nodeArray->nodes[iEle])
    for (j = 0; j < nodeArray->numberOfStoredNodes[iEle]; j++)
      nodeArray->nodeTable[nodeArray->nodes[iEle][j].number] =
	nodeArray->nodes[iEle] + j;
  /* We reallocate the table properties. */
  g_hash_table_foreach(nodeArray->properties, reallocNodeProperty,
		       GINT_TO_POINTER(iEle));

  /* Test part for the nodeTable array. */
#if DEBUG == 1
  for (j = 0; j < nodeArray->idCounter; j++)
    if (nodeArray->nodeTable[j] &&
	nodeArray->nodeTable[j]->number != j)
      fprintf(stderr, "Visu Node: ERROR inconsistency on node number %d.\n", j);
#endif
  DBG_fprintf(stderr, "Visu Node: reallocation OK.\n");
  DBG_fprintf(stderr, " | size = %do\n",
	      (int)(nodeArray->nNodes * sizeof(VisuNode) +
		    sizeof(int) * nodeArray->ntype * 2 +
		    sizeof(VisuNode*) * nodeArray->ntype +
		    sizeof(VisuNode*) * nodeArray->nNodes + sizeof(VisuNodeArray)));
}

/**
 * visu_node_array_getNewNode:
 * @nodeArray: a #VisuNodeArray object ;
 * @iEle: an integer between 0 and @nodeArray->ntypes - 1.
 *
 * Return the location of an unstored node for the given #VisuElement.
 * The returned node is then added in the list of used nodes.
 *
 * Returns: (transfer none): the location of a newly used node.
 */
VisuNode* visu_node_array_getNewNode(VisuNodeArray *nodeArray, unsigned int iEle)
{
  DBG_fprintf(stderr, "Visu Node: create a new node of element %d.\n", iEle);
  return newOrCopyNode(nodeArray, iEle, -1);
}

/**
 * visu_node_array_getCopyNode:
 * @nodeArray: a #VisuNodeArray object ;
 * @node: a node of the given #VisuNodeArray.
 *
 * Return the location of an unstored node that is the deep copy of the given node.
 * The returned node is then added in the list of used nodes.
 *
 * Returns: (transfer none): the location of a newly used node.
 */
VisuNode* visu_node_array_getCopyNode(VisuNodeArray *nodeArray, VisuNode *node)
{
  VisuNode* out;

  DBG_fprintf(stderr, "Visu Node: copy a new node from node %d (%d-%d).\n",
	      node->number, node->posElement, node->posNode);
  out = newOrCopyNode(nodeArray, node->posElement, node->number);
  DBG_fprintf(stderr, "Visu Node: copy a new node from node -> %d.\n", out->number);
  return out;
}

static VisuNode* newOrCopyNode(VisuNodeArray *nodeArray, int iEle,
			       int oldNodeId)
{
  VisuNode *node, *oldNode;
  int j;
  struct twoNodes nodes;
  GValue idValue = {0, {{0}, {0}}};

  g_return_val_if_fail(nodeArray, (VisuNode*)0);
  g_return_val_if_fail((oldNodeId >= 0 && oldNodeId < (int)nodeArray->idCounter) ||
		       (iEle >= 0 && iEle < (int)nodeArray->ntype),
		       (VisuNode*)0);

  if (nodeArray->numberOfStoredNodes[iEle] == nodeArray->numberOfNodes[iEle])
    /* We need to realloc... */
    visu_node_array_allocateNewNodes(nodeArray, iEle, REALLOCATION_STEP);

  /* Update the node internal values. */
  node         = nodeArray->nodes[iEle] + nodeArray->numberOfStoredNodes[iEle];
  node->number = nodeArray->idCounter;
  /* Update the array control arrays. */
  if (nodeArray->idCounter == nodeArray->nodeTableSize)
    {
      nodeArray->nodeTableSize += REALLOCATION_STEP;
      nodeArray->nodeTable = g_realloc(nodeArray->nodeTable,
				       sizeof(VisuNode*) * nodeArray->nodeTableSize);
    }
  nodeArray->numberOfStoredNodes[iEle] += 1;
  nodeArray->nodeTable[node->number]    = node;
  nodeArray->idCounter                 += 1;
  nodeArray->nbOfAllStoredNodes        += 1;

  /* We copy the values from oldNode. */
  oldNode = (VisuNode*)0;
  if (oldNodeId >= 0)
    {
      oldNode = nodeArray->nodeTable[oldNodeId];
      for ( j = 0; j < 3; j++)
	{
	  node->xyz[j]         = oldNode->xyz[j];
	  node->translation[j] = oldNode->translation[j];
	}
      node->rendered       = oldNode->rendered;
    }

  /* Create new properties for the node. */
  nodes.newNode = node;
  nodes.oldNode = oldNode;
  g_hash_table_foreach(nodeArray->properties, createNodeproperty, (gpointer)&nodes);

  /* If we have an old node, we use it as original node id, or we put
     -1 if not. */
  g_value_init(&idValue, G_TYPE_INT);
  g_value_set_int(&idValue, oldNodeId);
  visu_node_property_setValue(nodeArray->origProp, node, &idValue);
  nodeArray->nOrigNodes += (oldNodeId < 0)?1:0;

  return node;
}

/**************************/
/* The property routines. */
/**************************/
static void freePropertiesStruct(gpointer data)
{
  VisuNodeProperty *prop;
  unsigned int i, j;

  prop = (VisuNodeProperty*)data;
  DBG_fprintf(stderr, "Visu Node: freeing node property '%s'.\n", prop->name);

  g_free(prop->name);
  /* The pointer case. */
  if (prop->data_pointer)
    {
      for (i = 0; i < prop->array->ntype; i++)
	{
	  for (j = 0; j < prop->array->numberOfNodes[i]; j++)
	    {
	      if (prop->data_pointer[i][j])
		{
		  if (prop->freeTokenFunc)
		    prop->freeTokenFunc(prop->data_pointer[i][j], prop->user_data);
		  else
		    g_free(prop->data_pointer[i][j]);
		}
	    }
	  g_free(prop->data_pointer[i]);
	}
      g_free(prop->data_pointer);
    }
  /* The integer case */
  if (prop->data_int)
    {
      for (i = 0; i < prop->array->ntype; i++)
	g_free(prop->data_int[i]);
      g_free(prop->data_int);
    }
  g_free(prop);
  DBG_fprintf(stderr, "Visu Node: freeing property ... OK.\n");
}

/* Remove the property of the node given in data and move the
   last property of this element at the place of the removed node. */
static void removeNodeProperty(gpointer key, gpointer value, gpointer data)
{
  VisuNode *node;
  VisuNodeProperty *prop;
  
  node = (VisuNode*)data;
  prop = (VisuNodeProperty*)value;
  g_return_if_fail(prop->array->numberOfStoredNodes[node->posElement] > 0);

  DBG_fprintf(stderr, "Visu Node: remove node property '%s' from %d %d.\n",
	      (gchar*)key, node->posElement, node->posNode);
  /* We first remove the property token. */
  if (prop->data_pointer && prop->data_pointer[node->posElement][node->posNode])
    {
      if (prop->freeTokenFunc)
	prop->freeTokenFunc(prop->data_pointer[node->posElement][node->posNode],
			    prop->user_data);
      else
	g_free(prop->data_pointer[node->posElement][node->posNode]);
    }
  if (prop->data_int)
    prop->data_int[node->posElement][node->posNode] = 0;

  /* Then we copy the pointer from the last position to the given one.
     The last position is given by numberOfStoredNodes since this counter
     has already been lowered. */
  if (prop->data_pointer)
    {
      prop->data_pointer[node->posElement][node->posNode] =
	prop->data_pointer[node->posElement]
	[prop->array->numberOfStoredNodes[node->posElement]];
      prop->data_pointer[node->posElement]
	[prop->array->numberOfStoredNodes[node->posElement]] = (gpointer)0;
    }
  if (prop->data_int)
    {
      prop->data_int[node->posElement][node->posNode] =
	prop->data_int[node->posElement]
	[prop->array->numberOfStoredNodes[node->posElement]];
      prop->data_int[node->posElement]
	[prop->array->numberOfStoredNodes[node->posElement]] = 0;
    }
}

static void reallocNodeProperty(gpointer key, gpointer value, gpointer data)
{
  VisuNodeProperty *prop;
  unsigned int iEle, j;

  iEle = (unsigned int)GPOINTER_TO_INT(data);
  prop = (VisuNodeProperty*)value;
  DBG_fprintf(stderr, "Visu Node: realloc node property '%s' for element %d.\n",
	      (gchar*)key, iEle);

  g_return_if_fail(iEle < prop->array->ntype);

  if (prop->data_pointer)
    {
      prop->data_pointer[iEle] = g_realloc(prop->data_pointer[iEle],
					   sizeof(gpointer) *
					   prop->array->numberOfNodes[iEle]);
      /* We nullify the newly created properties. */
      for (j = prop->array->numberOfStoredNodes[iEle];
	   j < prop->array->numberOfNodes[iEle]; j++)
	prop->data_pointer[iEle][j] = (gpointer)0;
    }
  if (prop->data_int)
    {
      prop->data_int[iEle] = g_realloc(prop->data_int[iEle],
				       sizeof(int) *
				       prop->array->numberOfNodes[iEle]);
      /* We nullify the newly created properties. */
      for (j = prop->array->numberOfStoredNodes[iEle];
	   j < prop->array->numberOfNodes[iEle]; j++)
	prop->data_int[iEle][j] = 0;
    }
}

static void createNodeproperty(gpointer key, gpointer value, gpointer data)
{
  VisuNodeProperty *prop;
  struct twoNodes *nodes;

  prop = (VisuNodeProperty*)value;
  nodes = (struct twoNodes*)data;
  DBG_fprintf(stderr, "Visu Node: create/copy node property '%s' for node %d-%d.\n",
	      (gchar*)key, nodes->newNode->posElement, nodes->newNode->posNode);

  if (prop->data_pointer)
    {
      if (nodes->oldNode)
	prop->data_pointer[nodes->newNode->posElement][nodes->newNode->posNode] =
	  prop->newOrCopyTokenFunc((gconstpointer)prop->data_pointer[nodes->oldNode->posElement][nodes->oldNode->posNode], prop->user_data);
      else
	prop->data_pointer[nodes->newNode->posElement][nodes->newNode->posNode] =
	  prop->newOrCopyTokenFunc((gconstpointer)0, prop->user_data);
    }
  if (prop->data_int)
    {
      if (nodes->oldNode)
	prop->data_int[nodes->newNode->posElement][nodes->newNode->posNode] =
	  prop->data_int[nodes->oldNode->posElement][nodes->oldNode->posNode];
      else
	prop->data_int[nodes->newNode->posElement][nodes->newNode->posNode] = 0;
    }
}
/*****************************/
/* Public property routines. */
/*****************************/

/**
 * visu_node_property_setValue:
 * @nodeProp: a #VisuNodeProperty object ;
 * @node: a #VisuNode object ;
 * @value: A GValue pointer this the value to be stored.
 *
 * This method is used to store some values associated with
 * the given @node of the given @nodeArray. These values can be pointers to
 * anything allocated (will be free automatically when the property is deleted) or
 * they can be static values. This depends on the construction of the node property.
 * These values can be retrieved with the visu_node_property_getValue() method.
 *
 * See visu_node_array_getProperty() to get a property by its name.
 */
void visu_node_property_setValue(VisuNodeProperty* nodeProp, VisuNode* node,
			       GValue *value)
{
  g_return_if_fail(nodeProp && value && nodeProp->gtype == G_VALUE_TYPE(value));
  g_return_if_fail(node && node->posElement < nodeProp->array->ntype &&
		   node->posNode < nodeProp->array->numberOfStoredNodes[node->posElement]);

  switch (nodeProp->gtype)
    {
    case G_TYPE_POINTER:
      /* We free previous pointer. */
      if (nodeProp->freeTokenFunc)
	nodeProp->freeTokenFunc(nodeProp->data_pointer[node->posElement][node->posNode],
				nodeProp->user_data);
      else
	g_free(nodeProp->data_pointer[node->posElement][node->posNode]);
      /* We set the value. */
      nodeProp->data_pointer[node->posElement][node->posNode] =
	g_value_get_pointer(value);
      break;
    case G_TYPE_INT:
      nodeProp->data_int[node->posElement][node->posNode] =
	g_value_get_int(value);
      break;
    default:
      g_warning("Unsupported GValue type for property '%s'.", nodeProp->name);
    }
}

/**
 * visu_node_property_getValue:
 * @nodeProp: a #VisuNodeArray object ;
 * @node: a #VisuNode object ;
 * @value: an initialise GValue location.
 *
 * This method is used to retrieve some data associated to
 * the specified @node, stored in the given @data. These return data
 * should not be freed after used. The read value is stored in the given
 * GValue pointer. This GValue must be of the right type, depending on the
 * creation of the #VisuNodeProperty.
 *
 * Returns: some data associated to the key, stored the given GValue location.
 */
GValue* visu_node_property_getValue(VisuNodeProperty* nodeProp, VisuNode* node,
				  GValue *value)
{
  g_return_val_if_fail(nodeProp && value && nodeProp->gtype == G_VALUE_TYPE(value),
		       value);
  g_return_val_if_fail(node && node->posElement < nodeProp->array->ntype &&
		       node->posNode < nodeProp->array->numberOfStoredNodes[node->posElement], value);

  switch (nodeProp->gtype)
    {
    case G_TYPE_POINTER:
      DBG_fprintf(stderr, "Visu Node: get '%s' for node %d(%d,%d) as pointer %p.\n",
		  nodeProp->name, node->number, node->posElement, node->posNode,
		  (gpointer)nodeProp->data_pointer[node->posElement][node->posNode]);
      g_value_set_pointer(value, nodeProp->data_pointer[node->posElement][node->posNode]);
      return value;
    case G_TYPE_INT:
      DBG_fprintf(stderr, "Visu Node: get property '%s' for node %d as integer %d.\n",
		  nodeProp->name, node->number,
		  nodeProp->data_int[node->posElement][node->posNode]);
      g_value_set_int(value, nodeProp->data_int[node->posElement][node->posNode]);
      return value;
      break;
    default:
      g_warning("Unsupported GValue type for property '%s'.", nodeProp->name);
    }
  return value;
}

/**
 * visu_node_array_getProperty:
 * @nodeArray: a #VisuNodeArray object ;
 * @key: a string.
 *
 * This method is used to retrieve the node property associated to the given @key.
 *
 * Returns: (transfer none): a #VisuNodeProperty.
 */
VisuNodeProperty* visu_node_array_getProperty(VisuNodeArray* nodeArray, const char* key)
{
  VisuNodeProperty *prop;

  g_return_val_if_fail(nodeArray && key, (VisuNodeProperty*)0);

  prop = (VisuNodeProperty*)g_hash_table_lookup(nodeArray->properties, (gpointer)key);
  return prop;
}

/**
 * visu_node_array_freeProperty:
 * @nodeArray: a #VisuNodeArray object.
 * @key: the name of the property to be removed.
 * 
 * This method free the given property and all associated data.
 */
void visu_node_array_freeProperty(VisuNodeArray* nodeArray, const char* key)
{
  g_return_if_fail(nodeArray && key);

  g_hash_table_remove(nodeArray->properties, key);
  DBG_fprintf(stderr, "Visu Node: removing the property called '%s'.\n", key);
}

/**
 * visu_node_property_newPointer:
 * @nodeArray: a #VisuNodeArray object ;
 * @key: a string ;
 * @freeFunc: (allow-none) (scope call): a method to free each token (can be NULL).
 * @newAndCopyFunc: (scope call): a method to create or copy each token.
 * @user_data: (closure): a user defined pointer that will be given to
 * the free and copy routine.
 * 
 * This method creates and allocates a new area to store nodes associated data that
 * can be retrieve with the @key. These data are pointers on allocated memory
 * locations. When the property is removed with the #visu_node_freePropertry (or the
 * associated #VisuNodeArray is free) the area is free and @freeFunc is called for
 * each token (or g_free() if @freeFunc is NULL).
 *
 * The method @newAndCopyFunc is used when the number of nodes is increased,
 * if the const gpointer of the GCopyFunc is not NULL, then we require a copy,
 * if it is NULL, then the routine must create a new token with default values.
 *
 * Returns: (transfer none): the newly created #VisuNodeProperty object.
 */
VisuNodeProperty* visu_node_property_newPointer(VisuNodeArray* nodeArray,
					      const char* key, 
					      GFunc freeFunc,
					      GCopyFunc newAndCopyFunc,
					      gpointer user_data)
{
  VisuNodeProperty *prop;

  unsigned int i, j;
  
  g_return_val_if_fail(nodeArray && key && newAndCopyFunc, (VisuNodeProperty*)0);

  prop = (VisuNodeProperty*)g_hash_table_lookup(nodeArray->properties, key);
  g_return_val_if_fail(!prop, (VisuNodeProperty*)0);

  DBG_fprintf(stderr, "Visu Node: adding a new pointer"
	      " property, called '%s'.\n", key);
  prop                = g_malloc(sizeof(VisuNodeProperty));
  prop->gtype         = G_TYPE_POINTER;
  prop->name          = g_strdup(key);
  prop->array         = nodeArray;
  prop->data_int      = (int**)0;
  prop->data_pointer  = g_malloc(sizeof(gpointer*) * nodeArray->ntype);
  for (i = 0; i < nodeArray->ntype; i++)
    {
      DBG_fprintf(stderr, " | allocate (%d,%d)\n", i, nodeArray->numberOfNodes[i]);
      prop->data_pointer[i]      = g_malloc(sizeof(gpointer) *
					    nodeArray->numberOfNodes[i]);
      for (j = 0; j < nodeArray->numberOfNodes[i]; j++)
	prop->data_pointer[i][j] = (gpointer)0;
    }
  prop->freeTokenFunc      = freeFunc;
  prop->newOrCopyTokenFunc = newAndCopyFunc;
  prop->user_data          = user_data;
  g_hash_table_insert(nodeArray->properties, (gpointer)key, (gpointer)prop);
  
  return prop;
}

/**
 * visu_node_property_newInteger:
 * @nodeArray: a #VisuNodeArray object ;
 * @key: a string.
 * 
 * This method creates and allocates a new area to store nodes associated integer
 * values. This is the same than visu_node_property_newPointer() but for static
 * integers instead of pointers as data.
 *
 * Returns: (transfer none): the newly created #VisuNodeProperty object.
 */
VisuNodeProperty* visu_node_property_newInteger(VisuNodeArray* nodeArray,
					  const char* key)
{
  VisuNodeProperty *prop;

  unsigned int i, j;
  
  g_return_val_if_fail(nodeArray && key, (VisuNodeProperty*)0);

  prop = (VisuNodeProperty*)g_hash_table_lookup(nodeArray->properties, key);
  g_return_val_if_fail(!prop, (VisuNodeProperty*)0);

  DBG_fprintf(stderr, "Visu Node: adding a new int property, called '%s'.\n", key);
  prop                = g_malloc(sizeof(VisuNodeProperty));
  prop->gtype         = G_TYPE_INT;
  prop->name          = g_strdup(key);
  prop->array         = nodeArray;
  prop->data_pointer  = (gpointer**)0;
  prop->data_int      = g_malloc(sizeof(gpointer*) * nodeArray->ntype);
  for (i = 0; i < nodeArray->ntype; i++)
    {
      DBG_fprintf(stderr, " | allocate (%d,%d)\n", i, nodeArray->numberOfNodes[i]);
      prop->data_int[i]      = g_malloc(sizeof(int) * nodeArray->numberOfNodes[i]);
      for (j = 0; j < nodeArray->numberOfNodes[i]; j++)
	prop->data_int[i][j] = 0;
    }
  prop->freeTokenFunc      = (GFunc)0;
  prop->newOrCopyTokenFunc = (GCopyFunc)0;
  prop->user_data          = (gpointer)0;
  g_hash_table_insert(nodeArray->properties, (gpointer)key, (gpointer)prop);
  
  return prop;
}

/**
 * visu_node_array_traceProperty:
 * @array: a #VisuNodeArray object ;
 * @id: a property name.
 *
 * This is a debug method. It outputs on stderr the values for all
 * nodes of the property @id.
 */
void visu_node_array_traceProperty(VisuNodeArray *array, const gchar *id)
{
  VisuNodeProperty* prop;
  unsigned int i, j;
  
  prop = visu_node_array_getProperty(array, id);
  
  fprintf(stderr, "Visu Node: output node property '%s'.\n", id);
  fprintf(stderr, " | type= %d\n", (int)prop->gtype);
  if (prop->data_int)
    {
      for (i = 0; i < prop->array->ntype; i++)
	for (j = 0; j < prop->array->numberOfStoredNodes[i]; j++)
	  fprintf(stderr, " | %7d %3d %7d -> %d\n", array->nodes[i][j].number,
		  i, j, prop->data_int[i][j]);
    }
  if (prop->data_pointer)
    {
      for (i = 0; i < prop->array->ntype; i++)
	for (j = 0; j < prop->array->numberOfStoredNodes[i]; j++)
	  fprintf(stderr, " | %7d %3d %7d -> %p\n", array->nodes[i][j].number,
		  i, j, prop->data_pointer[i][j]);
    }
}
