/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef VISU_EXTENSION_H
#define VISU_EXTENSION_H

#include <glib.h>
#include <glib-object.h>

#include "openGLFunctions/objectList.h"
#include "openGLFunctions/renderingMode.h"
#include "visu_data.h"

/***************/
/* Public part */
/***************/

/**
 * VISU_EXTENSION_PRIORITY_BACKGROUND
 *
 * An extension with this priority is drawn first.
 */
#define VISU_EXTENSION_PRIORITY_BACKGROUND 0
/**
 * VISU_EXTENSION_PRIORITY_NODES
 *
 * An extension with this priority is drawn alsmost first with the nodes.
 */
#define VISU_EXTENSION_PRIORITY_NODES 2
/**
 * VISU_EXTENSION_PRIORITY_NODE_DECORATIONS
 *
 * An extension with this priority is drawn just after the nodes.
 */
#define VISU_EXTENSION_PRIORITY_NODE_DECORATIONS 5
/**
 * VISU_EXTENSION_PRIORITY_HIGH
 *
 * An extension with this priority is drawn after the higher priorities.
 */
#define VISU_EXTENSION_PRIORITY_HIGH 20
/**
 * VISU_EXTENSION_PRIORITY_NORMAL
 *
 * An extension with this priority is drawn after the higher priorities.
 */
#define VISU_EXTENSION_PRIORITY_NORMAL 50
/**
 * VISU_EXTENSION_PRIORITY_LOW
 *
 * An extension with this priority is drawn among last extensions.
 */
#define VISU_EXTENSION_PRIORITY_LOW 80
/**
 * VISU_EXTENSION_PRIORITY_LAST
 *
 * An extension with this priority is drawn last.
 */
#define VISU_EXTENSION_PRIORITY_LAST 100

/**
 * VisuExtension:
 * 
 * Common name to refer to a #VisuExtension_struct.
 */
typedef struct _VisuExtension VisuExtension;

/**
 * VisuExtensionRebuildFunc:
 * @dataObj: the #VisuData object to be rebuilt.
 *
 * Prototypes used to recreate OpenGL list of objects for each extension.
 */
typedef void (*VisuExtensionRebuildFunc)(VisuData *dataObj);

VisuExtension* visu_extension_new(const gchar* name, const gchar *nameI18n,
				  const gchar* description,
				  int objectListId, VisuExtensionRebuildFunc rebuild);
/**
 * visu_extension_free:
 * @extension: the extension to delete.
 *
 * Free all the allocated attributes of the specified method.
 */
void visu_extension_free(VisuExtension* extension);
/**
 * visuExtensions_getList:
 *
 * This method is used to get the list of all registered VisuExtension.
 * This list is own by V_Sim and should not be freed.
 *
 * Returns: the list of all VisuExtension.
 */
GList* visuExtensions_getList(void);

/**
 * visu_extension_setPriority:
 * @extension: a #VisuExtension object ;
 * @priority: an integer value.
 *
 * Extentions are drawn in an order that depends on their priority.
 * The lower is the number, the sooner the extension is drawn. Flags,
 * such as #VISU_EXTENSION_PRIORITY_NORMAL or #VISU_EXTENSION_PRIORITY_LOW,
 * can be used or user defined values are also possible.
 */
void visu_extension_setPriority(VisuExtension* extension, guint priority);

/**
 * visu_extension_setSaveOpenGLState:
 * @extension: a #VisuExtension object ;
 * @saveState: an boolean value.
 *
 * If the extension needs to change some OpenGL state (to disable the fog for example,
 * or the cullface), a flag should be set to enable V_Sim to restore the right values after
 * the extensio have been called. Because the OpenGL list of an extension is just GL_COMPILE
 * the extension can't just save and restore state itself because when the list is called,
 * the state can have been changed.
 */
void visu_extension_setSaveOpenGLState(VisuExtension *extension, gboolean saveState);

/**
 * visu_extension_setSensitiveToRenderingMode:
 * @extension: a #VisuExtension object ;
 * @status: an boolean value.
 *
 * If @status is TRUE, when the extension is rendered, OpenGL context is
 * switched to the rendering mode preferd for the extension. Use
 * visu_extension_setPreferedRenderingMode() to choose one.
 */
void visu_extension_setSensitiveToRenderingMode(VisuExtension* extension, gboolean status);
/**
 * visu_extension_setPreferedRenderingMode:
 * @extension: a #VisuExtension object ;
 * @value: see #VisuRenderingModeId to choose one.
 *
 * This method is used to specify the rendering mode that the extension should use
 * to be drawn (if the sensitive flag has been set, see
 * visu_extension_setSensitiveToRenderingMode()). If the @value is set
 * to followGeneralSetting, the extension follows the global setting
 * for rendering mode.
 *
 * Returns: TRUE if the "OpenGLAskForReDraw" should be emitted.
 */
gboolean visu_extension_setPreferedRenderingMode(VisuExtension* extension,
						  VisuRenderingModeId value);


/**
 * visu_extension_getActive:
 * @extension: the extension.
 *
 * Get if the extension is used or not. If not its ObjectList
 * is not rendered.
 *
 * Returns: 1 if used, 0 otherwise.
 */
int visu_extension_getActive(VisuExtension* extension);
/**
 * visu_extension_setActive:
 * @extension: the extension,
 * @value: the new value.
 *
 * Set if an extension is actually used or not.
 */
void visu_extension_setActive(VisuExtension* extension, int value);

/**
 * visuExtensions_add:
 * @extension: an extension.
 *
 * A method used by user to registered a new extension.
 */
void visuExtensions_add(VisuExtension *extension);

/**
 * VisuExtensionGet_fromName:
 * @name: a string.
 *
 * Look for the extension with this name.
 *
 * Returns: the extension or NULL if none.
 */
VisuExtension* VisuExtensionGet_fromName(const gchar* name);

/**
 * visuExtensions_remove:
 * @extension: an extension.
 *
 * This method is used to removed fom the list of registered #VisuExtension
 * the given one. Removing @extension does not free it.
 */
void visuExtensions_remove(VisuExtension *extension);

/**
 * visuExtensions_callAllLists:
 *
 * For each registered extension that has a valid list (id > 1000),
 * it calls it with glCallList.
 */
void visuExtensions_callAllLists(void);
/**
 * visuExtensions_callList:
 * @name: the name of the list to be called.
 * @lastOnly: a boolean.
 *
 * Select the #VisuExtension matching the given @name and call
 * it. The call is indeed done only if the extension is used. If
 * @lastOnly is TRUE, the list is called only if it has a
 * #VISU_EXTENSION_PRIORITY_LAST priority. On the contrary the list
 * is called only if its priority is lower than
 * #VISU_EXTENSION_PRIORITY_LAST. To draw all of them, use
 * visuExtensions_callAllLists() instead.
 */
void visuExtensions_callList(const char *name, gboolean lastOnly);
void visuExtensions_callAllLastLists(void);
/**
 * visuExtensions_rebuildAllLists:
 * @dataObj: the #VisuData object to be rebuilt.
 *
 * For each registered extension that has a valid rebuild method,
 * it calls it.
 */
void visuExtensions_rebuildAllLists(VisuData *dataObj);
/**
 * visuExtensions_rebuildList:
 * @dataObj: the #VisuData object to be rebuilt ;
 * @name: the name of the list to be rebuilt.
 *
 * Select the #VisuExtension matching the given @name and rebuild
 * it. This routine does not sort the extension on their priority and
 * should be used only to draw some selected extensions. To draw all
 * of them, use rebuildAllExtensionsLists() instead.
 */
void visuExtensions_rebuildList(VisuData *dataObj, const char *name);



/****************/
/* Private area */
/****************/

/**
 * VisuExtension:
 * @name: name of the extension, in ASCII, used as id in the config
 * files ;
 * @nameI18n: name in UTF8 that can be translated and shown to user ;
 * @description: a short text in UTF-8 ;
 * @objectListId: the id of the OpenGL list that represent this
 *                extension. This id is called whenever the extension need
 *                to be drawn ;
 * @rebuild: method called when the OpenGL context has changed
 *           and the extension need to be recomputed ;
 * @priority: control the order of drawing ;
 * @saveState: if TRUE, all OpenGL parameters are saved and
 *             will be restored after the extension has been called ;
 * @isSensitiveToRenderingMode: if TRUE, the extension can be drawn using
 *                              a rendering mode different from the other
 *                              extension (smooth, wireframe...) ;
 * @preferedRenderingMode: the specific rendering mode ;
 * @used: if FALSE, the OpenGL id is skipped when redraw is necessary.
 * 
 * This structure allows the user to store data
 * about an OpenGL extension.
 */
struct _VisuExtension
{
  /* Some variable to describe this OpenGL extension.
     The attribute name is mandatory since it is
     used to identify the method. */
  gchar *name, *nameI18n;
  gchar *description;

  /* The id of the possible objects list brings by
     the extension is refered by this int. */
  int objectListId;

  /* Function called to rebuild the object list of the extension. */
  VisuExtensionRebuildFunc rebuild;

  /* A priority for the extension. */
  guint priority;

  /* If set, V_Sim save the OpenGL state before the list
     id is called and restore all states after. */
  gboolean saveState;

  /* Fine tune of rendering mode (Wireframe, smooth...).
     The flag isSensitiveToRenderingMode define is the
     extension cares about rendering mode. This flag is FALSE
     by default. When FALSE, the global value for rendering
     mode is used. Otherwise the value is stored in
     preferedRenderingMode. */
  gboolean isSensitiveToRenderingMode;
  VisuRenderingModeId preferedRenderingMode;

  /* A boolean to know if this extension is actually used
     or not. */
  int used;
};

/**
 * initVisuExtensions:
 *
 * Initialise all the variable of this part. It calls all
 * the elements in listInitExtensionFunc (that stores the
 * init function of the extensions). If these elements return
 * valid VisuExtension, they are registered through a call
 * to registerVisuExtension().
 *
 * Returns: 1 if everything goes allright during the initialisation.
 */
int initVisuExtensions(void);
/**
 * loadExtensions:
 *
 * This method is used when V_Sim is initialized and should
 * not be called elsewhere. For each valid init method found
 * in listInitExtensionFunc, it is called.
 */
void loadExtensions(void);


#endif
