/*   EXTRAITS DE LA LICENCE
     Copyright CEA, contributeurs : Luc BILLARD et Damien
     CALISTE, laboratoire L_Sim, (2001-2006)
  
     Adresse ml :
     BILLARD, non joignable par ml ;
     CALISTE, damien P caliste AT cea P fr.

     Ce logiciel est un programme informatique servant  visualiser des
     structures atomiques dans un rendu pseudo-3D. 

     Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
     respectant les principes de diffusion des logiciels libres. Vous pouvez
     utiliser, modifier et/ou redistribuer ce programme sous les conditions
     de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
     sur le site "http://www.cecill.info".

     Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
     pris connaissance de la licence CeCILL, et que vous en avez accept les
     termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
     Copyright CEA, contributors : Luc BILLARD et Damien
     CALISTE, laboratoire L_Sim, (2001-2006)

     E-mail address:
     BILLARD, not reachable any more ;
     CALISTE, damien P caliste AT cea P fr.

     This software is a computer program whose purpose is to visualize atomic
     configurations in 3D.

     This software is governed by the CeCILL  license under French law and
     abiding by the rules of distribution of free software.  You can  use, 
     modify and/ or redistribute the software under the terms of the CeCILL
     license as circulated by CEA, CNRS and INRIA at the following URL
     "http://www.cecill.info". 

     The fact that you are presently reading this means that you have had
     knowledge of the CeCILL license and that you accept its terms. You can
     find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "gtk_renderingWindowWidget.h"

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <math.h> /* for sqrt function... */

#include <pixmaps/icone-observe.xpm>

#include "support.h"
#include "visu_gtk.h"
#include "visu_tools.h"
#include "visu_object.h"
#include "visu_basic.h"
#include "visu_data.h"
#include "visu_configFile.h"
#include "visu_extension.h"
#include "OSOpenGL/visu_openGL.h"
#include "renderingBackend/visu_actionInterface.h"
#include "gtk_openGLWidget.h"
#include "extraFunctions/extraNode.h"
#include "coreTools/toolFileFormat.h"
#include "coreTools/toolConfigFile.h"
#include "extraGtkFunctions/gtk_dumpDialogWidget.h"
#include "openGLFunctions/objectList.h"
#include "openGLFunctions/interactive.h"
#include "opengl.h"

/**
 * SECTION:gtk_renderingWindowWidget
 * @short_description: Defines a complex widget used to render files
 * and print information.
 *
 * <para>This is a complex widget, inheriting from #GtkWindow, with a
 * rendering area and a status bar area. A #VisuData is always
 * attached to this widget, see visu_rendering_window_setData(). If not
 * the V_Sim logo is displayed.</para>
 *
 * <para>The rendering area can receive keyboard or mouse events, see
 * renderingWindowSet_pickEventListener() or
 * renderingWindowSet_observeEventListener(). Then, callbacks are
 * defined using renderingWindowSet_interactiveHandlers().</para>
 *
 * <para>The status bar area has different buttons to load or export a
 * file. It also display some usefull information like the number of
 * rendered nodes. It has also a real status bar location displaying
 * tips about current available actions. One can add news using
 * renderingWindowPush_message().</para>
 */

typedef enum
  {
    event_button_press,
    event_button_release,
    event_motion_notify,
    event_key_press,
    event_key_release,
    event_scroll
  } InteractiveEventsId;

struct InteractiveEvents_struct
{
  gulong callbackId;
  InteractiveEventsId id;
};
typedef struct InteractiveEvents_struct InteractiveEvents;


struct GtkInfoArea_struct
{
  GtkWidget *area;

  GtkWidget *hboxFileInfo;
  GtkWidget *labelSize;
  GtkWidget *labelNb;
  GtkWidget *labelFileInfo;
  gboolean fileInfoFreeze;

  GtkWidget *hboxTools;
  GtkWidget *dumpButton;
  GtkWidget *loadButton;
  GtkWidget *raiseButton;
  GtkWidget *reloadButton;

  GtkWidget *statusInfo, *progress;
  guint progressId;
  GtkWidget *cancelButton;

  GtkWidget *hboxInteractive;
  GtkWidget *clearMarksButton;
  GtkWidget *infoButton;
  gulong infoClicked;

  guint statusInfoId;
};
typedef struct GtkInfoArea_struct GtkInfoArea;

#define GTK_STATUSINFO_NOFILEINFO _("<span style=\"italic\">No description is available</span>")
#define GTK_STATUSINFO_NONB       _("<span style=\"italic\">Nothing is loaded</span>")
#define FLAG_PARAMETER_RED_COORD   "config_showReducedCoordinates"
#define DESC_PARAMETER_RED_COORD   "Display coordinates in reduced values when picking a node ; boolean 0 or 1"
#define FLAG_PARAMETER_AUTO_ADJUST "config_autoAdjustCamera"
#define DESC_PARAMETER_AUTO_ADJUST "Auto adjust zoom capability for the box to be full size at zoom level 1 ; boolean 0 or 1"

#define MENU_CAMERA_SAVE    "<VisuRenderingWindow>/Camera/Save"
#define MENU_CAMERA_RESTORE "<VisuRenderingWindow>/Camera/Restore"
#define MENU_CAMERA_1       "<VisuRenderingWindow>/Camera/select1"
#define MENU_CAMERA_2       "<VisuRenderingWindow>/Camera/select2"
#define MENU_CAMERA_3       "<VisuRenderingWindow>/Camera/select3"
#define MENU_CAMERA_4       "<VisuRenderingWindow>/Camera/select4"
#define MENU_CAMERA_5       "<VisuRenderingWindow>/Camera/select5"
#define MENU_CAMERA_6       "<VisuRenderingWindow>/Camera/select6"
#define MENU_CAMERA_7       "<VisuRenderingWindow>/Camera/select7"
#define MENU_CAMERA_8       "<VisuRenderingWindow>/Camera/select8"
#define MENU_CAMERA_9       "<VisuRenderingWindow>/Camera/select9"
static const gchar* cameraAccels[] = {MENU_CAMERA_1, MENU_CAMERA_2, MENU_CAMERA_3, MENU_CAMERA_4,
                                      MENU_CAMERA_5, MENU_CAMERA_6, MENU_CAMERA_7, MENU_CAMERA_8,
                                      MENU_CAMERA_9};
static guint cameraKeys[] = {GDK_KEY_1, GDK_KEY_2, GDK_KEY_3, GDK_KEY_4, GDK_KEY_5,
                             GDK_KEY_6, GDK_KEY_7, GDK_KEY_8, GDK_KEY_9};

enum {
  EXPORT_SIGNAL,
  OPEN_SIGNAL,
  RELOAD_SIGNAL,
  SHOW_ACTION_DIALOG_SIGNAL,
  SHOW_MAIN_PANEL_SIGNAL,
  LAST_SIGNAL
};

/* Local variables. */
static VisuInteractive *inter;
static guint _signals[LAST_SIGNAL] = { 0 };

/* Local methods. */
static gboolean readConfigReducedCoordinates(gchar **lines, int nbLines, int position,
                                             VisuData *dataObj, GError **error);
static gboolean readConfigAutoAdjust(gchar **lines, int nbLines, int position,
                                     VisuData *dataObj _U_, GError **error);
static void exportParameters(GString *data, VisuData *dataObj);
static GtkInfoArea *gtkStatusInfo_createBar(VisuRenderingWindow *window, gint width,
					    gint height, gboolean withToolBar);
static gulong addInteractiveEventListeners(VisuRenderingWindow *window,
					   InteractiveEventsId id);
static GtkWidget* buildCameraMenu(VisuRenderingWindow *window);
static void _setLabelSize(GtkInfoArea *info, gint width, gint height);
static void _setFileDescription(GtkInfoArea *info, gchar *message);
static void _setNNodes(GtkInfoArea *info, gint nb);
static void getOpenGLAreaSize(VisuRenderingWindow *window,
			      unsigned int *width, unsigned int *height);

/* Local callbacks */
static void visu_rendering_window_dispose (GObject* obj);
static void visu_rendering_window_finalize(GObject* obj);
static void onMarkClearClicked(GtkButton *button, gpointer data);
static void onNodeInfoClicked(GtkToggleButton *button, gpointer data);
static void onRaiseButtonClicked(VisuRenderingWindow *window, gpointer user_data);
static void displayFileInfoOnDataLoaded(VisuRenderingWindow *window);
static void onRenderingMethodChanged(VisuRenderingWindow *window, VisuRendering *method,
				     gpointer data);
static void setFileButtonsSensitive(VisuRenderingWindow *window);
static void onNodePopulationChanged(VisuData *data, int *nodes, gpointer user_data);
static gboolean onDragMotion(GtkWidget *widget, GdkDragContext *context,
			     gint x, gint y, guint t, gpointer user_data);
static void onDropData(VisuRenderingWindow *window, GdkDragContext *context,
		       gint x, gint y, GtkSelectionData *selection_data,
		       guint target_type, guint time, GtkWidget *glArea);
static gboolean onCameraMenu(VisuRenderingWindow *window, GdkEventButton *event,
			     GtkEventBox *ev _U_);
static void onCameraMenuSelected(GtkMenuShell *menushell, gpointer user_data _U_);
static void onCameraMenuClicked(GtkMenuItem *menuitem, gpointer user_data);
static void onCameraMenuCurrentClicked(GtkMenuItem *menuitem, gpointer user_data);
static void minimalPickInfo(VisuInteractive *inter, VisuInteractivePick pick,
			    VisuNode *nodes[3], gpointer data);
static void minimalPickError(VisuInteractive *inter,
			     VisuInteractivePickError error, gpointer data);
static void onCancelButtonClicked(GtkButton *button, gpointer data);
static gboolean onCameraAccel(GtkAccelGroup *accel, GObject *obj, guint key, GdkModifierType mod);

struct _VisuRenderingWindow
{
  GtkVBox generalVBox;
  gboolean dispose_has_run;

  /*********************************/
  /* Dealing with the OpenGL area. */
  /*********************************/
  /* The OpenGL area and it's notification zone. */
  GtkWidget *openGLArea;
  /* Stored size of the OpenGL area. */
  gint socketWidth, socketHeight;
  /* This pointer give the handle to rule all interactive actions. */
  GList *inters;
  /* Printed marks. */
  VisuMarks *marks;
  /* This is a list of currently connected
     signal for the interactive mode. */
  GList *interactiveEvents;
  /* A pointer on the current used cursor. */
  GdkCursor *currentCursor;
  GdkCursor *refCursor;

  /*************************************/
  /* Dealing with the information bar. */
  /*************************************/
  /* TO BE INTEGRATED. */
  GtkInfoArea *info;
  /* TO BE INTEGRATED. */
  int nbStatusMessage;

  GtkAccelGroup *accel;

  /* a pointer to the currently loaded VisuData. */
  VisuData *currentData;
  /* Id of signals currently connected, related to the current #VisuData object. */
  gulong populationIncrease_id, populationDecrease_id;
};

struct _VisuRenderingWindowClass
{
  GtkVBoxClass parent_class;

  void (*renderingWindow) (VisuRenderingWindow *window);

  /* Action signals for keybindings, do not connect to these */
  void (*export) (VisuRenderingWindow *window);
  void (*open)   (VisuRenderingWindow *window);
  void (*reload) (VisuRenderingWindow *window);

  GdkCursor *cursorRotate;
  GdkCursor *cursorWatch;
  GdkCursor *cursorPointer;
  GdkCursor *cursorPirate;
  GdkCursor *cursorGrab;
  GdkCursor *currentCursor;

  gboolean useReducedCoordinates;
  gboolean autoAdjust;
};

static VisuRenderingWindowClass *my_class = NULL;

G_DEFINE_TYPE(VisuRenderingWindow, visu_rendering_window, GTK_TYPE_VBOX)

/* Local callbacks */
static gboolean timeOutPopMessage(gpointer data);
static void onSizeChangeEvent(GtkWidget *widget, GtkAllocation *allocation, gpointer user_data);
static void onRedraw(VisuRenderingWindow *window, gpointer data);
static void onForceRedraw(VisuRenderingWindow *window, gpointer data);
static void onRealiseEvent(GtkWidget *wd, gpointer data);
static void onExport(VisuRenderingWindow *window);
static void onOpen(VisuRenderingWindow *window);
static void onReload(VisuRenderingWindow *window);

/* Interactive mode listeners. */
static gboolean onKeyPressed(GtkWidget *widget, GdkEventKey *event, gpointer data);
static gboolean onKeyRelease(GtkWidget *widget, GdkEventKey *event, gpointer data);
static gboolean onMouseMotion(GtkWidget *widget, GdkEventMotion *event, gpointer user_data);
static gboolean onScrollEvent(GtkWidget *widget, GdkEventScroll *event, gpointer user_data);
static gboolean onButtonAction(VisuRenderingWindow *window, GdkEventButton *event, gpointer user_data);

static void visu_rendering_window_class_init(VisuRenderingWindowClass *klass)
{
  GtkBindingSet *binding_set;
  VisuConfigFileEntry *resourceEntry;

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: creating the class of the widget.\n");

  /* Initialisation des curseurs utiles. */
  klass->cursorPirate  = gdk_cursor_new(GDK_PIRATE);
  klass->cursorRotate  = gdk_cursor_new(GDK_EXCHANGE);
  klass->cursorWatch   = gdk_cursor_new(GDK_WATCH);
  klass->cursorPointer = gdk_cursor_new(GDK_DOTBOX);
  klass->cursorGrab    = gdk_cursor_new(GDK_FLEUR);
  
  klass->export = onExport;
  klass->reload = onReload;
  klass->open   = onOpen;

  klass->useReducedCoordinates = FALSE;
  klass->autoAdjust            = TRUE;

  inter = visuInteractiveNew(interactive_measureAndObserve);

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: connect the signals.\n");
  G_OBJECT_CLASS(klass)->dispose  = visu_rendering_window_dispose;
  G_OBJECT_CLASS(klass)->finalize = visu_rendering_window_finalize;

  /**
   * VisuRenderingWindow::export:
   * @window: the object emitting the signal.
   *
   * Signal emitted when the user ask for data export.
   *
   * Since: 3.6
   */
  _signals[EXPORT_SIGNAL] =
    g_signal_new("export", G_TYPE_FROM_CLASS(klass),
                 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                 G_STRUCT_OFFSET(VisuRenderingWindowClass, export),
                 NULL, NULL, g_cclosure_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);
  /**
   * VisuRenderingWindow::open:
   * @window: the object emitting the signal.
   *
   * Signal emitted when the user ask to open new data.
   *
   * Since: 3.6
   */
  _signals[OPEN_SIGNAL] =
    g_signal_new("open", G_TYPE_FROM_CLASS(klass),
                 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                 G_STRUCT_OFFSET(VisuRenderingWindowClass, open),
                 NULL, NULL, g_cclosure_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);
  /**
   * VisuRenderingWindow::reload:
   * @window: the object emitting the signal.
   *
   * Signal emitted when the user ask to reload current data.
   *
   * Since: 3.6
   */
  _signals[RELOAD_SIGNAL] =
    g_signal_new("reload", G_TYPE_FROM_CLASS(klass),
                 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                 G_STRUCT_OFFSET(VisuRenderingWindowClass, reload),
                 NULL, NULL, g_cclosure_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);
  /**
   * VisuRenderingWindow::show-action-dialog:
   * @window: the object emitting the signal.
   *
   * Signal emitted when the user ask to show the action dialog.
   *
   * Since: 3.6
   */
  _signals[SHOW_ACTION_DIALOG_SIGNAL] =
    g_signal_new("show-action-dialog", G_TYPE_FROM_CLASS(klass),
                 G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS | G_SIGNAL_ACTION,
                 0, NULL, NULL, g_cclosure_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);
  /**
   * VisuRenderingWindow::show-main-panel:
   * @window: the object emitting the signal.
   *
   * Signal emitted when the user ask to raise the main panel.
   *
   * Since: 3.6
   */
  _signals[SHOW_MAIN_PANEL_SIGNAL] =
    g_signal_new("show-main-panel", G_TYPE_FROM_CLASS(klass),
                 G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS | G_SIGNAL_ACTION,
                 0, NULL, NULL, g_cclosure_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: connect the bindings.\n");
  binding_set = gtk_binding_set_by_class(klass);
  gtk_binding_entry_add_signal(binding_set, GDK_KEY_s, GDK_CONTROL_MASK,
                               "export", 0);
  gtk_binding_entry_add_signal(binding_set, GDK_KEY_o, GDK_CONTROL_MASK,
                               "open", 0);
  gtk_binding_entry_add_signal(binding_set, GDK_KEY_r, GDK_CONTROL_MASK,
                               "reload", 0);
  gtk_binding_entry_add_signal(binding_set, GDK_KEY_i, GDK_CONTROL_MASK,
                               "show-action-dialog", 0);
  gtk_binding_entry_add_signal(binding_set, GDK_KEY_Home, 0,
                               "show-main-panel", 0);

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: add the resources.\n");
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_RED_COORD,
					  DESC_PARAMETER_RED_COORD,
					  1, readConfigReducedCoordinates);
  visu_configFile_entry_setVersion(resourceEntry, 3.6f);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_AUTO_ADJUST,
					  DESC_PARAMETER_AUTO_ADJUST,
					  1, readConfigAutoAdjust);
  visu_configFile_entry_setVersion(resourceEntry, 3.6f);
  visu_configFile_addExportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParameters);

  my_class = klass;
}

static void visu_rendering_window_dispose (GObject* obj)
{
  GList *ptList;
  InteractiveEvents *event;
  VisuRenderingWindow *window;

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: dispose object %p.\n", (gpointer)obj);

  window = RENDERING_WINDOW(obj);
  if (window->dispose_has_run)
    return;
  window->dispose_has_run = TRUE;

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: releasing current handles.\n");
  visu_basic_setLoadCancel(TRUE);
  if (window->info->progressId)
    g_source_remove(window->info->progressId);
  visu_marks_setVisuData(window->marks, (VisuData*)0);
  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: release current data handle.\n");
  if (window->currentData)
    g_object_unref(window->currentData);

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: removing interactive listeners.\n");
  for (ptList = window->interactiveEvents; ptList;
       ptList = g_list_next(ptList))
    {
      event = (InteractiveEvents*)ptList->data;
      DBG_fprintf(stderr, "  | disconnecting %d signal.\n", event->id);
      g_signal_handler_disconnect(G_OBJECT(window->openGLArea),
                                  event->callbackId);
      g_free(ptList->data);
    }
  if (window->interactiveEvents)
    g_list_free(window->interactiveEvents);
  window->interactiveEvents = (GList*)0;

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: chain to parent.\n");
  G_OBJECT_CLASS(visu_rendering_window_parent_class)->dispose(obj);
  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: dispose done.\n");
}
static void visu_rendering_window_finalize(GObject* obj)
{
  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: finalize object %p.\n", (gpointer)obj);

  /* Chain up to the parent class */
  G_OBJECT_CLASS(visu_rendering_window_parent_class)->finalize(obj);
}

static void visu_rendering_window_init(VisuRenderingWindow *renderingWindow)
{
  guint n;

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: initializing new object (%p).\n",
	      (gpointer)renderingWindow);

  renderingWindow->currentData           = (VisuData*)0;
  renderingWindow->populationDecrease_id = 0;
  renderingWindow->populationIncrease_id = 0;

  /* Binding for the camera menu. */
  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: connect the camera bindings.\n");
  renderingWindow->accel = gtk_accel_group_new();
  gtk_accel_map_add_entry(g_intern_static_string(MENU_CAMERA_RESTORE), GDK_KEY_r, 0);
  gtk_accel_map_add_entry(g_intern_static_string(MENU_CAMERA_SAVE), GDK_KEY_s, 0);
  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: connect the camera numbered bindings.\n");
  for (n = 0; n < 9; n++)
    {
      gtk_accel_map_add_entry(g_intern_static_string(cameraAccels[n]),
                              cameraKeys[n], GDK_CONTROL_MASK);
      gtk_accel_group_connect_by_path(renderingWindow->accel,
                                      g_intern_static_string(cameraAccels[n]),
                                      g_cclosure_new(G_CALLBACK(onCameraAccel),
                                                     (gpointer)0, (GClosureNotify)0));
    }

  g_signal_connect(G_OBJECT(inter), "node-selection",
		   G_CALLBACK(minimalPickInfo), (gpointer)renderingWindow);
  g_signal_connect(G_OBJECT(inter), "selection-error",
		   G_CALLBACK(minimalPickError), (gpointer)renderingWindow);
}

/**
 * visu_rendering_window_newEmbedded:
 * @width: its desired width ;
 * @height: its desired height.
 *
 * Like visu_rendering_window_new() but create also a #GtkWindow to go
 * around. This widget is reachable using g_object_get_data() with the
 * key #RENDERING_WINDOW_ID. In that case, no frame is drawn around
 * the rendering area.
 *
 * Returns: (transfer full): a newly created #VisuRenderingWindow widget.
 */
GtkWidget* visu_rendering_window_newEmbedded(int width, int height)
{
  GtkWidget *renderingWindow;
  GtkWidget *window;
  gchar *windowRef = "V_Sim";
  gchar *window_name = "v_sim_render";
  gchar *class_name = "V_Sim";
  GdkPixbuf *iconPixBuf;

  renderingWindow = visu_rendering_window_new(width, height, FALSE, TRUE);
  
  /* We create the rendering window. */
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), windowRef);
#if GTK_MAJOR_VERSION > 2
  gtk_window_set_has_resize_grip(GTK_WINDOW(window), FALSE);
#endif
  gtk_window_set_wmclass(GTK_WINDOW(window), window_name, class_name);
  iconPixBuf = gdk_pixbuf_new_from_xpm_data((const char**)icone_observe_xpm);
  gtk_window_set_icon(GTK_WINDOW(window), iconPixBuf);
  gtk_window_add_accel_group(GTK_WINDOW(window),
                             RENDERING_WINDOW(renderingWindow)->accel);
  g_object_set_data(G_OBJECT(window),
                    "RenderingWindow", (gpointer)renderingWindow);

  gtk_container_add(GTK_CONTAINER(window), renderingWindow);
  g_object_set_data(G_OBJECT(renderingWindow), RENDERING_WINDOW_ID, window);

  return renderingWindow;
}

/**
 * visu_rendering_window_new:
 * @width: its desired width ;
 * @height: its desired height ;
 * @withFrame: a boolean ;
 * @withToolBar: a boolean.
 *
 * A #VisuRenderingWindow widget is a GtkWindow that have an area for
 * OpenGL drawing and a statusBar with many stuff like action buttons,
 * real status bar for notifications, ... The rendering area can be
 * drawn with a frame or not. With this routine, only the
 * #VisuRenderingWindow widget is created. To embed it into a window
 * automatically, use visu_rendering_window_newEmbedded() instead.
 *
 * Returns: a newly created #VisuRenderingWindow widget.
 */
GtkWidget* visu_rendering_window_new(int width, int height, gboolean withFrame,
                                     gboolean withToolBar)
{
  VisuRenderingWindow *renderingWindow;
  GtkWidget *wd;
  GtkTargetEntry target_list[] = {
    { "text/plain", 0, 0 },
  };


  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: create a new VisuRenderingWindow object.\n");

  renderingWindow = RENDERING_WINDOW(g_object_new(visu_rendering_window_get_type(), NULL));

  /* We create the statusinfo area. */
  renderingWindow->info = gtkStatusInfo_createBar(renderingWindow, width, height,
						  withToolBar);
  wd = renderingWindow->info->area;
  gtk_box_pack_end(GTK_BOX(renderingWindow), wd, FALSE, FALSE, 0);

  renderingWindow->openGLArea = openGLWidgetNew(TRUE);
  g_signal_connect_swapped(G_OBJECT(renderingWindow->openGLArea), "realize",
			   G_CALLBACK(onRealiseEvent), (gpointer)renderingWindow);
  gtk_drag_dest_set(renderingWindow->openGLArea,
		    GTK_DEST_DEFAULT_DROP,
		    target_list, 1, GDK_ACTION_COPY);
  g_signal_connect(renderingWindow->openGLArea, "drag-motion",
		   G_CALLBACK(onDragMotion), NULL);
  g_signal_connect_swapped(renderingWindow->openGLArea, "drag-data-received",
			   G_CALLBACK(onDropData), (gpointer)renderingWindow);

  gtk_widget_set_size_request(renderingWindow->openGLArea, width, height);
  /* Attach no redraw redraw method. */
  openGLWidgetSet_redraw(OPENGL_WIDGET(renderingWindow->openGLArea),
			 (RedrawMethod)0, (VisuData*)0);
  if (withFrame)
    {
      wd = gtk_frame_new(NULL);
      gtk_frame_set_shadow_type(GTK_FRAME(wd), GTK_SHADOW_ETCHED_IN);
      gtk_box_pack_start(GTK_BOX(renderingWindow), wd, TRUE, TRUE, 0);
      gtk_container_add(GTK_CONTAINER(wd), renderingWindow->openGLArea);
    }
  else
    gtk_box_pack_start(GTK_BOX(renderingWindow),
		       renderingWindow->openGLArea, TRUE, TRUE, 0);
  
  /* We physically show the vbox. */
  gtk_widget_show_all(GTK_WIDGET(renderingWindow));

  /* Unset the size request. */
  renderingWindow->socketWidth = width;
  renderingWindow->socketHeight = height;
  /* Set a listener on the size to show the warning icon accordingly. */
  g_signal_connect(G_OBJECT(renderingWindow->openGLArea), "size-allocate",
		   G_CALLBACK(onSizeChangeEvent), (gpointer)renderingWindow);

  /* Set local variables. */
  DBG_fprintf(stderr, "                - setup the local variables.\n");
  renderingWindow->nbStatusMessage          = 0;
  renderingWindow->interactiveEvents        = (GList*)0;
  renderingWindow->inters                   = (GList*)0;
  renderingWindow->marks                    = visu_marks_new((VisuInteractive*)0);
  renderingWindow->currentCursor            = 
    RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(renderingWindow))->cursorPirate;
  renderingWindow->refCursor                = 
    RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(renderingWindow))->cursorPirate;

  g_signal_connect_swapped(VISU_INSTANCE, "OpenGLAskForReDraw",
			   G_CALLBACK(onRedraw), (gpointer)renderingWindow);
  g_signal_connect_swapped(VISU_INSTANCE, "OpenGLForceReDraw",
			   G_CALLBACK(onForceRedraw), (gpointer)renderingWindow);
  g_signal_connect_swapped(VISU_INSTANCE, "renderingChanged",
			   G_CALLBACK(onRenderingMethodChanged),
			   (gpointer)renderingWindow);
  onRenderingMethodChanged(renderingWindow,
                           visu_object_getRendering(VISU_INSTANCE),
			   (gpointer)0);

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: Building OK.\n");

  return GTK_WIDGET(renderingWindow);
}

static GtkInfoArea *gtkStatusInfo_createBar(VisuRenderingWindow *window, gint width,
					    gint height, gboolean withToolBar)
{
  GtkWidget *hbox;
  GtkWidget *wd, *image, *ev;
  GtkInfoArea *info;
#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION < 12
  GtkTooltips *tooltips;
  tooltips = gtk_tooltips_new ();
#endif

  info = g_malloc(sizeof(GtkInfoArea));

  info->area = gtk_vbox_new(FALSE, 0);

  info->fileInfoFreeze = FALSE;
  info->hboxFileInfo = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(info->area), info->hboxFileInfo, FALSE, FALSE, 1);

  /* Size info */
  wd = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(info->hboxFileInfo), wd, FALSE, FALSE, 5);
  info->labelSize = gtk_label_new("");
  gtk_label_set_use_markup(GTK_LABEL(info->labelSize), TRUE);
  _setLabelSize(info, width, height);
  gtk_box_pack_start(GTK_BOX(wd), info->labelSize, FALSE, FALSE, 0);

  wd = gtk_vseparator_new();
  gtk_box_pack_start(GTK_BOX(info->hboxFileInfo), wd, FALSE, FALSE, 0);

  /* Nb nodes */
  info->labelNb = gtk_label_new("");
  gtk_label_set_use_markup(GTK_LABEL(info->labelNb), TRUE);
  _setNNodes(info, -1);
  gtk_box_pack_start(GTK_BOX(info->hboxFileInfo), info->labelNb, FALSE, FALSE, 5);

  wd = gtk_vseparator_new();
  gtk_box_pack_start(GTK_BOX(info->hboxFileInfo), wd, FALSE, FALSE, 0);

  /* File info */
  wd = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(info->hboxFileInfo), wd, TRUE, TRUE, 5);
#if GTK_MAJOR_VERSION > 2 || GTK_MINOR_VERSION > 7
  image = gtk_image_new_from_stock(GTK_STOCK_INFO,
				   GTK_ICON_SIZE_MENU);
#else
  image = gtk_image_new_from_stock(GTK_STOCK_SAVE,
				   GTK_ICON_SIZE_MENU);
#endif
  gtk_box_pack_start(GTK_BOX(wd), image, FALSE, FALSE, 1);
  info->labelFileInfo = gtk_label_new("");
  gtk_label_set_use_markup(GTK_LABEL(info->labelFileInfo), TRUE);
  gtk_misc_set_alignment(GTK_MISC(info->labelFileInfo), 0., 0.5);
#if GTK_MAJOR_VERSION > 2 || GTK_MINOR_VERSION > 5
  gtk_label_set_ellipsize(GTK_LABEL(info->labelFileInfo), PANGO_ELLIPSIZE_END);
#endif
  _setFileDescription(info, GTK_STATUSINFO_NOFILEINFO);
  gtk_box_pack_start(GTK_BOX(wd), info->labelFileInfo, TRUE, TRUE, 0);
  ev = gtk_event_box_new();
  gtk_widget_set_tooltip_text(ev, _("Click here to get the list of"
				    " saved camera positions.\n"
				    "Use 's' and 'r' keys to save and"
				    " restore camera settings. <Shift> + 's'"
                                    " remove the current camera from the list."));
  g_signal_connect_swapped(G_OBJECT(ev), "button-release-event",
                           G_CALLBACK(onCameraMenu), (gpointer)window);
  gtk_box_pack_end(GTK_BOX(wd), ev, FALSE, FALSE, 0);
  image = gtk_image_new_from_stock(GTK_STOCK_ZOOM_FIT,
				   GTK_ICON_SIZE_MENU);
  gtk_container_add(GTK_CONTAINER(ev), image);

  /* Status */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(info->area), hbox, FALSE, FALSE, 0);

  /* Handle box for action buttons. */
  if (withToolBar)
    {
      /* The container */
      info->hboxTools = gtk_hbox_new(TRUE, 0);
      gtk_box_pack_start(GTK_BOX(hbox), info->hboxTools, FALSE, FALSE, 0);

      /* Load button */
      info->loadButton = gtk_button_new();
      g_object_set(G_OBJECT(info->loadButton), "can-default", FALSE, "can-focus", FALSE,
                   "has-default", FALSE, "has-focus", FALSE, NULL);
      gtk_widget_set_sensitive(info->loadButton, FALSE);
      gtk_button_set_focus_on_click(GTK_BUTTON(info->loadButton), FALSE);
      gtk_widget_set_tooltip_text(info->loadButton,
                                  _("Open Ctrl+s"));
      g_signal_connect_swapped(G_OBJECT(info->loadButton), "clicked",
			       G_CALLBACK(onOpen), (gpointer)window);
      image = gtk_image_new_from_stock(GTK_STOCK_OPEN,
				       GTK_ICON_SIZE_MENU);
      gtk_container_add(GTK_CONTAINER(info->loadButton), image);
      gtk_box_pack_start(GTK_BOX(info->hboxTools), info->loadButton, FALSE, FALSE, 0);
      /* Refresh button */
      wd = gtk_button_new();
      g_object_set(G_OBJECT(wd), "can-default", FALSE, "can-focus", FALSE,
                   "has-default", FALSE, "has-focus", FALSE, NULL);
      gtk_button_set_focus_on_click(GTK_BUTTON(wd), FALSE);
      gtk_widget_set_tooltip_text(wd,
                                  _("Reload the current file Ctrl+r"));
      g_signal_connect_swapped(G_OBJECT(wd), "clicked",
                               G_CALLBACK(onReload), (gpointer)window);
      image = gtk_image_new_from_stock(GTK_STOCK_REFRESH,
				       GTK_ICON_SIZE_MENU);
      gtk_container_add(GTK_CONTAINER(wd), image);
      gtk_box_pack_start(GTK_BOX(info->hboxTools), wd, FALSE, FALSE, 0);
      info->reloadButton = wd;
      /* Save button */
      info->dumpButton = gtk_button_new();
      g_object_set(G_OBJECT(info->dumpButton), "can-default", FALSE, "can-focus", FALSE,
                   "has-default", FALSE, "has-focus", FALSE, NULL);
      gtk_button_set_focus_on_click(GTK_BUTTON(info->dumpButton), FALSE);
      gtk_widget_set_tooltip_text(info->dumpButton,
                                  _("Export Ctrl+s"));
      g_signal_connect_swapped(G_OBJECT(info->dumpButton), "clicked",
                               G_CALLBACK(onExport), (gpointer)window);
      gtk_widget_set_sensitive(info->dumpButton, FALSE);
      image = gtk_image_new_from_stock(GTK_STOCK_SAVE_AS,
				       GTK_ICON_SIZE_MENU);
      gtk_container_add(GTK_CONTAINER(info->dumpButton), image);
      gtk_box_pack_start(GTK_BOX(info->hboxTools), info->dumpButton, FALSE, FALSE, 0);
      /* Auto-raise command panel button */
      info->raiseButton = gtk_button_new();
      g_object_set(G_OBJECT(info->raiseButton), "can-default", FALSE, "can-focus", FALSE,
                   "has-default", FALSE, "has-focus", FALSE, NULL);
      gtk_button_set_focus_on_click(GTK_BUTTON(info->raiseButton), FALSE);
      gtk_widget_set_tooltip_text(info->raiseButton,
                                  _("Raise the command panel window.\n"
                                    "  Use <home> as key binding."));
      g_signal_connect_swapped(G_OBJECT(info->raiseButton), "clicked",
			       G_CALLBACK(onRaiseButtonClicked), (gpointer)window);
      image = gtk_image_new_from_stock(GTK_STOCK_GO_UP,
				       GTK_ICON_SIZE_MENU);
      gtk_widget_show(image);
      gtk_container_add(GTK_CONTAINER(info->raiseButton), image);
      /* gtk_widget_set_no_show_all(info->raiseButton, TRUE); */
      gtk_box_pack_start(GTK_BOX(info->hboxTools), info->raiseButton, FALSE, FALSE, 0);
    }
  else
    {
      info->loadButton = (GtkWidget*)0;
      info->dumpButton = (GtkWidget*)0;
      info->raiseButton = (GtkWidget*)0;
    }

  /* The status bar or progress */
  info->statusInfo = gtk_statusbar_new();
  gtk_box_pack_start(GTK_BOX(hbox), info->statusInfo, TRUE, TRUE, 0);
#if GTK_MAJOR_VERSION < 3
  gtk_statusbar_set_has_resize_grip(GTK_STATUSBAR(info->statusInfo), FALSE);
#endif
  info->statusInfoId = gtk_statusbar_get_context_id(GTK_STATUSBAR(info->statusInfo),
                                                    "OpenGL statusbar.");
  gtk_widget_set_no_show_all(info->statusInfo, TRUE);
  gtk_widget_show(info->statusInfo);
  info->progressId = 0;
  info->progress = gtk_progress_bar_new();
  gtk_box_pack_start(GTK_BOX(hbox), info->progress, TRUE, TRUE, 0);
  gtk_widget_set_no_show_all(info->progress, TRUE);
  info->cancelButton = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
  gtk_box_pack_start(GTK_BOX(hbox), info->cancelButton, FALSE, FALSE, 0);
  gtk_widget_set_no_show_all(info->cancelButton, TRUE);
  g_signal_connect(G_OBJECT(info->cancelButton), "clicked",
                   G_CALLBACK(onCancelButtonClicked), (gpointer)window);
  
  /* The interactive button zone. */
  info->hboxInteractive = gtk_hbox_new(TRUE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), info->hboxInteractive, FALSE, FALSE, 0);

  /* Action button */
  wd = gtk_toggle_button_new();
  gtk_widget_set_sensitive(wd, FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(wd), FALSE);
  gtk_button_set_focus_on_click(GTK_BUTTON(wd), FALSE);
  g_object_set(G_OBJECT(wd), "can-default", FALSE, "can-focus", FALSE,
	       "has-default", FALSE, "has-focus", FALSE, NULL);
  info->infoClicked = g_signal_connect(G_OBJECT(wd), "clicked",
				       G_CALLBACK(onNodeInfoClicked),
				       (gpointer)window);
  gtk_widget_set_tooltip_text(wd, _("Measure / remove information"
				    " for the selected node."));
  image = gtk_image_new_from_stock(GTK_STOCK_PROPERTIES,
				   GTK_ICON_SIZE_MENU);
  gtk_container_add(GTK_CONTAINER(wd), image);
  gtk_box_pack_end(GTK_BOX(info->hboxInteractive), wd, FALSE, FALSE, 0);
  info->infoButton = wd;
  g_object_set_data_full(G_OBJECT(wd), "selectedNodeId",
			 g_malloc(sizeof(gint)), g_free);

  /* Clean marks button */
  wd = gtk_button_new();
  gtk_widget_set_sensitive(wd, FALSE);
  gtk_button_set_focus_on_click(GTK_BUTTON(wd), FALSE);
  g_object_set(G_OBJECT(wd), "can-default", FALSE, "can-focus", FALSE,
	       "has-default", FALSE, "has-focus", FALSE, NULL);
  g_signal_connect(G_OBJECT(wd), "clicked",
		   G_CALLBACK(onMarkClearClicked), (gpointer)window);
  gtk_widget_set_tooltip_text(wd, _("Remove all measurement marks."));
  image = gtk_image_new_from_stock(GTK_STOCK_CLEAR,
				   GTK_ICON_SIZE_MENU);
  gtk_container_add(GTK_CONTAINER(wd), image);
  gtk_box_pack_end(GTK_BOX(info->hboxInteractive), wd, FALSE, FALSE, 0);
  info->clearMarksButton = wd;

  return info;
}

static void onSizeChangeEvent(GtkWidget *widget _U_,
			      GtkAllocation *allocation, gpointer user_data)
{
  VisuRenderingWindow *window;

  window = RENDERING_WINDOW(user_data);
  g_return_if_fail(window);

  /* Return if no changes in size (this event is called even the size
     is not really changed but has been negociated. */
  if (window->socketWidth == allocation->width &&
      window->socketHeight == allocation->height)
    return;

  window->socketWidth = allocation->width;
  window->socketHeight = allocation->height;
  _setLabelSize(window->info,
                window->socketWidth,
                window->socketHeight);

  /* If data are currently rendered on this window,
     we ask these data to update to the new size. */
  if (window->currentData)
    visu_data_setSizeOfView(window->currentData,
                            allocation->width, allocation->height);
}

static void onRealiseEvent(GtkWidget *wd, gpointer data _U_)
{
  guint w, h;
  VisuRenderingWindow *window;

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: initializing OpenGL variable for"
	      "the new OpenGL area.\n");
  /* Set V_Sim OpenGL options. */
  openGLInit_context();
  DBG_fprintf(stderr, " | openGL context OK\n");

  window = RENDERING_WINDOW(wd);

  /* If we have a VisuData object attached, we set its size. */
  if (window->currentData)
    {
      getOpenGLAreaSize(window, &w, &h);
      visu_data_setSizeOfView(window->currentData, w, h);
      visuExtensions_rebuildAllLists(window->currentData);
    }
  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: changing the cursor.\n");
  /* We set the cursor. */
  gdk_window_set_cursor(gtk_widget_get_window(window->openGLArea),
			window->currentCursor);  
  DBG_fprintf(stderr, " | cursor OK.\n");

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: changing the minimum size.\n");
  gtk_widget_set_size_request(window->openGLArea, 100, 100);
}

static gboolean onDragMotion(GtkWidget *widget, GdkDragContext *context,
			     gint x _U_, gint y _U_, guint t, gpointer data _U_)
{
  /*   GList *tmpLst; */
  GdkAtom atom;

  /*   DBG_fprintf(stderr, "VisuRenderingWindow: Hey ! You dnd move something !\n"); */
  /*   for (tmpLst = context->targets; tmpLst; tmpLst = g_list_next(tmpLst)) */
  /*     { */
  /*       DBG_fprintf(stderr, " | dnd: '%s'\n", */
  /* 		  gdk_atom_name(GDK_POINTER_TO_ATOM(tmpLst->data))); */
  /*     } */
  atom = gtk_drag_dest_find_target(widget, context,
				   gtk_drag_dest_get_target_list(widget));
  if (atom != GDK_NONE)
    gdk_drag_status(context, GDK_ACTION_COPY, t);
  else
    gdk_drag_status(context, 0, t);
  return (atom != GDK_NONE);
}

static void onDropData(VisuRenderingWindow *window _U_, GdkDragContext *context _U_,
		       gint x _U_, gint y _U_, GtkSelectionData *selection_data _U_,
		       guint target_type _U_, guint time _U_, GtkWidget *glArea _U_)
{
  /*   gchar **filenames; */
  /*   int i, n, delta; */
  /*   VisuData *newData; */
  /*   guint w, h; */

  /*   if (context->action != GDK_ACTION_COPY) */
  /*     return; */

  g_error("Port drag action to GTK3");

  /*   DBG_fprintf(stderr, " | data: '%s' -> '%s'\n", gdk_atom_name(selection_data->type), */
  /* 	      selection_data->data); */

  /*   filenames = g_strsplit((gchar*)selection_data->data, "\n", -1); */
  /*   gtk_drag_finish(context, TRUE, TRUE, time); */

  /*   if (window->currentData) */
  /*     newData = visu_data_newWithVisuOpenGLView(visu_data_getOpenGLView(window->currentData)); */
  /*   else */
  /*     { */
  /*       if (gtk_widget_get_visible(window)) */
  /* 	{ */
  /* 	  getOpenGLAreaSize(window, &w, &h); */
  /* 	  newData = visu_data_newWithSize(w, h); */
  /* 	} */
  /*       else */
  /* 	newData = visu_data_new(); */
  /*     } */
  /*   g_return_if_fail(newData); */

  /*   n = 0; */
  /*   for (i = 0; filenames[i]; i++) */
  /*     { */
  /*       g_strstrip(filenames[i]); */
  /*       if (filenames[i][0] != '\0') */
  /* 	{ */
  /* 	  delta = (strncmp("file://", filenames[i], 7))?0:7; */
  /* 	  visu_data_addFile(newData, filenames[i] + delta, n, (ToolFileFormat*)0); */
  /* 	  n += 1; */
  /* 	} */
  /*     } */
  /*   g_strfreev(filenames); */

  /*   visu_rendering_window_loadFile(window, newData, 0); */

  /*   VISU_ADD_REDRAW; */

}

static void minimalPickInfo(VisuInteractive *inter _U_, VisuInteractivePick pick,
			    VisuNode *nodes[3], gpointer data)
{
  float posSelect[3], posRef[3], dist;
  GString *str;
  VisuRenderingWindow *window;
  int i;
  gint *id;
  const gchar *comment;

  window = RENDERING_WINDOW(data);
  g_return_if_fail(window);

  while (window->nbStatusMessage > 1)
    renderingWindowPop_message(window);
  gtk_widget_set_sensitive(window->info->infoButton, FALSE);

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: update the status bar after pick.\n");
  switch (pick)
    {
    case PICK_SELECTED:
      gtk_widget_set_sensitive(window->info->infoButton, TRUE);

      id = (gint*)g_object_get_data(G_OBJECT(window->info->infoButton),
				    "selectedNodeId");
      *id = (gint)nodes[0]->number;
      g_signal_handler_block(G_OBJECT(window->info->infoButton),
			     window->info->infoClicked);
      gtk_toggle_button_set_active
	(GTK_TOGGLE_BUTTON(window->info->infoButton),
	 visu_marks_getActive(window->marks, *id));
      g_signal_handler_unblock(G_OBJECT(window->info->infoButton),
			       window->info->infoClicked);
      if (RENDERING_WINDOW_GET_CLASS(window)->useReducedCoordinates)
        visu_data_getReducedNodePosition(window->currentData, nodes[0], posSelect);
      else
        visu_data_getNodePosition(window->currentData, nodes[0], posSelect);
      str = g_string_new(_("Selected node number "));
      g_string_append_printf(str, "%d - %s (%7.3g;%7.3g;%7.3g)",
			     nodes[0]->number + 1,
			     window->currentData->fromIntToVisuElement
			     [nodes[0]->posElement]->name,
			     posSelect[0], posSelect[1], posSelect[2]);
      comment = extraNodeGet_label(window->currentData, nodes[0]);
      if (comment)
	g_string_append_printf(str, " %s", comment);
      renderingWindowPush_message(window, str->str);
      g_string_free(str, TRUE);
      return;
    case PICK_DISTANCE:
      /* Have a ref and a selected node, then distance informations is compued. */
      visu_data_getNodePosition(window->currentData, nodes[0], posSelect);
      visu_data_getNodePosition(window->currentData, nodes[1], posRef);
      str = g_string_new(_("Distance between nodes "));
      dist = 0.;
      for (i = 0; i < 3; i++)
	dist += (posRef[i] - posSelect[i]) * (posRef[i] - posSelect[i]);
      dist = sqrt(dist);
      g_string_append_printf(str, _("%d and %d : %7.3f"),
			     nodes[1]->number + 1, nodes[0]->number + 1, dist);
      renderingWindowPush_message(window, str->str);
      g_string_free(str, TRUE);
      return;
    case PICK_REFERENCE_1:
      renderingWindowPush_message(window, _("<shift> right-click on"
					    " background to unset reference."));
      return;
    case PICK_REFERENCE_2:
      renderingWindowPush_message(window,
				  _("<ctrl> right-click on"
				    " background to unset second reference."));
      return;
    case PICK_UNREFERENCE_1:
    case PICK_UNREFERENCE_2:
      return;
    default:
      return;
    }
}
static void minimalPickError(VisuInteractive *inter _U_,
			     VisuInteractivePickError error, gpointer data)
{
  VisuRenderingWindow *window;

  window = RENDERING_WINDOW(data);
  switch (error)
    {
    case PICK_ERROR_NO_SELECTION:
      renderingWindowPush_message(window, _("No node has been selected."));
      return;
    case PICK_ERROR_SAME_REF:
      renderingWindowPush_message(window, _("Picked node is already used"
					    " as a reference."));
      return;
    case PICK_ERROR_REF1:
      renderingWindowPush_message(window, _("Can't pick a second reference"
					    " without any first one"
					    " (use <shift> right-click)."));
      return;
    case PICK_ERROR_REF2:
      renderingWindowPush_message(window, _("Can't remove first reference"
					    " before removing the second one."));
      return;
    default:
      return;
    }
}

static gulong addInteractiveEventListeners(VisuRenderingWindow *window,
					   InteractiveEventsId id)
{
  GList* ptList;
  InteractiveEvents *event;
  gboolean found;

  g_return_val_if_fail(IS_RENDERING_WINDOW(window), (gulong)0);

  found  = FALSE;
  for (ptList = window->interactiveEvents; ptList && !found;
       ptList = g_list_next(ptList))
    {
      event = (InteractiveEvents*)ptList->data;
      if (event->id == id)
	found = TRUE;
    }
  if (found)
    return (gulong)0;

  event = g_malloc(sizeof(InteractiveEvents));
  event->id = id;
  switch (id)
    {
    case event_button_press:
      event->callbackId = g_signal_connect_swapped(G_OBJECT(window->openGLArea),
						   "button-press-event",
						   G_CALLBACK(onButtonAction), (gpointer)window);
      break;
    case event_button_release:
      event->callbackId = g_signal_connect_swapped(G_OBJECT(window->openGLArea),
						   "button-release-event",
						   G_CALLBACK(onButtonAction), (gpointer)window);
      break;
    case event_motion_notify:
      event->callbackId = g_signal_connect(G_OBJECT(window->openGLArea),
					   "motion-notify-event",
					   G_CALLBACK(onMouseMotion), (gpointer)window);
      break;
    case event_key_press:
      event->callbackId = g_signal_connect(G_OBJECT(window->openGLArea),
					   "key-press-event",
					   G_CALLBACK(onKeyPressed), (gpointer)window);
      break;
    case event_key_release:
      event->callbackId = g_signal_connect(G_OBJECT(window->openGLArea),
					   "key-release-event",
					   G_CALLBACK(onKeyRelease), (gpointer)window);
      break;
    case event_scroll:
      event->callbackId = g_signal_connect(G_OBJECT(window->openGLArea),
					   "scroll-event",
					   G_CALLBACK(onScrollEvent), (gpointer)window);
      break;
    default:
      g_warning("Unknown event to add.");
      g_free(event);
      return (gulong)0;
    };
  window->interactiveEvents = g_list_prepend(window->interactiveEvents,
                                             (gpointer)event);
  return event->callbackId;
}
static void setInteractiveType(VisuRenderingWindow *window,
			       VisuInteractiveId type)
{
  VisuRenderingWindowClass *klass;
  InteractiveEvents *event;
  gulong id;
  GList *ptList;

  g_return_if_fail(IS_RENDERING_WINDOW(window));  
  klass = RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window));
  g_return_if_fail(klass);

  /* We set the cursors. */
  switch (type)
    {
    case interactive_observe:
    case interactive_measureAndObserve:
      if (gtk_widget_get_visible(GTK_WIDGET(window)))
	gdk_window_set_cursor(gtk_widget_get_window(window->openGLArea),
			      klass->cursorRotate);
      window->currentCursor = klass->cursorRotate;
      window->refCursor = klass->cursorRotate;
      break;
    case interactive_measure:
    case interactive_pick:
    case interactive_move:
    case interactive_mark:
      if (gtk_widget_get_visible(GTK_WIDGET(window)))
	gdk_window_set_cursor(gtk_widget_get_window(window->openGLArea),
			      klass->cursorPointer);
      window->currentCursor = klass->cursorPointer;
      window->refCursor = klass->cursorPointer;
      break;
    case interactive_none:
      if (gtk_widget_get_visible(GTK_WIDGET(window)))
	gdk_window_set_cursor(gtk_widget_get_window(window->openGLArea),
			      klass->cursorPirate);
      window->currentCursor = klass->cursorPirate;
      window->refCursor = klass->cursorPirate;
      break;
    }
  /* We set the listeners. */
  if (type != interactive_none)
    {
      DBG_fprintf(stderr, "Gtk VisuRenderingWindow: setup signals.\n");
      id = addInteractiveEventListeners(window, event_button_release);
      if (id)
	DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
      id = addInteractiveEventListeners(window, event_button_press);
      if (id)
	DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
      id = addInteractiveEventListeners(window, event_motion_notify);
      if (id)
	DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
      id = addInteractiveEventListeners(window, event_key_press);
      if (id)
	DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
      id = addInteractiveEventListeners(window, event_key_release);
      if (id)
	DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
      id = addInteractiveEventListeners(window, event_scroll);
      if (id)
	DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
    }
  else
    {
      DBG_fprintf(stderr, "Gtk VisuRenderingWindow: removing interactive listeners.\n");
      for (ptList = window->interactiveEvents; ptList;
           ptList = g_list_next(ptList))
	{
          event = (InteractiveEvents*)ptList->data;
	  DBG_fprintf(stderr, "  | disconnecting %d signal.\n", event->id);
	  g_signal_handler_disconnect(G_OBJECT(window->openGLArea),
                                      event->callbackId);
	  g_free(ptList->data);
	}
      if (window->interactiveEvents)
	g_list_free(window->interactiveEvents);
      window->interactiveEvents = (GList*)0;
    }
}
/**
 * renderingWindowPush_interactive:
 * @window: a #VisuRenderingWindow object.
 * @inter: a #VisuInteractive object.
 *
 * It adds @inter to the stack of interactive sessions currently
 * attached to @window and launch it.
 *
 * Since: 3.6
 */
void renderingWindowPush_interactive(VisuRenderingWindow *window,
				     VisuInteractive *inter)
{
  VisuInteractiveId type;

  g_return_if_fail(IS_RENDERING_WINDOW(window) &&
		   IS_VISU_INTERACTIVE(inter));

  type = visuInteractiveGet_type(inter);
  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: push a new interactive"
	      " session (%d / %d).\n", type, g_list_length(window->inters));
  window->inters = g_list_prepend(window->inters, inter);
  g_object_ref(G_OBJECT(inter));

  setInteractiveType(window, type);
}
/**
 * renderingWindowPop_interactive:
 * @window: a #VisuRenderingWindow object.
 * @inter: a #VisuInteractive object.
 *
 * It removes @inter from the stack of interactive sessions currently
 * attached to @window. If @inter was first on the stack, the next
 * session is launched.
 *
 * Since: 3.6
 */
void renderingWindowPop_interactive(VisuRenderingWindow *window,
				    VisuInteractive *inter)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));

  window->inters = g_list_remove(window->inters, inter);
  g_object_unref(G_OBJECT(inter));

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: pop an old interactive"
	      " session (%d).\n", g_list_length(window->inters));
  
  if (window->inters)
    setInteractiveType(window, visuInteractiveGet_type(VISU_INTERACTIVE(window->inters->data)));
  else
    setInteractiveType(window, interactive_none);
}

static gboolean onButtonAction(VisuRenderingWindow *window, GdkEventButton *event,
			       gpointer user_data _U_)
{
  SimplifiedEvents ev;

  if (!window->inters)
    return TRUE;

  ev.button = 0;
  ev.motion = 0;
  ev.letter = '\0';
  ev.specialKey = Key_None;

  ev.x = event->x;
  ev.y = event->y;
  ev.button = event->button;
  ev.shiftMod = event->state & GDK_SHIFT_MASK;
  ev.controlMod = event->state & GDK_CONTROL_MASK;
  ev.buttonType = 0;
  if (event->type == GDK_BUTTON_PRESS)
    ev.buttonType = BUTTON_TYPE_PRESS;
  else if (event->type == GDK_BUTTON_RELEASE)
    ev.buttonType = BUTTON_TYPE_RELEASE;

  gdk_window_set_cursor(gtk_widget_get_window(window->openGLArea),
			RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorWatch);
  visuInteractiveHandle_event(VISU_INTERACTIVE(window->inters->data),
			      window->currentData, &ev);

  gdk_window_set_cursor(gtk_widget_get_window(window->openGLArea),
			window->currentCursor);

  return TRUE;
}
static gboolean onScrollEvent(GtkWidget *widget _U_, GdkEventScroll *event,
			      gpointer user_data)
{
  SimplifiedEvents ev;
  VisuRenderingWindow *window;

  window = RENDERING_WINDOW(user_data);
  g_return_val_if_fail(window, TRUE);
  if (!window->inters)
    return TRUE;

  ev.button = 0;
  ev.motion = 0;
  ev.letter = '\0';
  ev.specialKey = Key_None;

  ev.x = event->x;
  ev.y = event->y;
  if (event->direction == GDK_SCROLL_UP)
    ev.button = 4;
  else if (event->direction == GDK_SCROLL_DOWN)
    ev.button = 5;
  ev.shiftMod = event->state & GDK_SHIFT_MASK;
  ev.controlMod = event->state & GDK_CONTROL_MASK;

  if (ev.button)
    {
      gdk_window_set_cursor(gtk_widget_get_window(window->openGLArea),
			    RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorWatch);
      visuInteractiveHandle_event(VISU_INTERACTIVE(window->inters->data),
				  window->currentData, &ev);
      gdk_window_set_cursor(gtk_widget_get_window(window->openGLArea),
			    window->currentCursor);
    }

  return TRUE;
}
static gboolean onMouseMotion(GtkWidget *widget _U_, GdkEventMotion *event,
			      gpointer user_data)
{
  SimplifiedEvents ev;
  VisuRenderingWindow *window;
  /* #if DEBUG == 1 */
  /*   GTimer *timer; */
  /*   gulong fractionTimer; */
  /* #endif */

  window = RENDERING_WINDOW(user_data);
  g_return_val_if_fail(window, TRUE);
  if (!window->inters)
    return TRUE;

  /* #if DEBUG == 1 */
  /*   timer = g_timer_new(); */
  /*   g_timer_start(timer); */
  /* #endif */

  ev.button = 0;
  ev.motion = 1;
  ev.letter = '\0';
  ev.specialKey = Key_None;

  /*   DBG_fprintf(stderr, "Gtk VisuRenderingWindow: motion at %gx%g.\n", event->x, event->y); */
  ev.x = event->x;
  ev.y = event->y;
  if (event->state & GDK_BUTTON1_MASK)
    ev.button = 1;
  else if (event->state & GDK_BUTTON2_MASK)
    ev.button = 2;
  else if (event->state & GDK_BUTTON3_MASK)
    ev.button = 3;
  ev.buttonType = BUTTON_TYPE_PRESS;
  ev.shiftMod = event->state & GDK_SHIFT_MASK;
  ev.controlMod = event->state & GDK_CONTROL_MASK;

  if (ev.button)
    {
      gdk_window_set_cursor(gtk_widget_get_window(window->openGLArea),
                            RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorWatch);
      visuInteractiveHandle_event(VISU_INTERACTIVE(window->inters->data),
        			  window->currentData, &ev);
      gdk_window_set_cursor(gtk_widget_get_window(window->openGLArea),
                            window->currentCursor);
    }

  if (event->is_hint)
    {
#if GDK_MINOR_VERSION > 11
      gdk_event_request_motions(event);
#else
      gdk_window_get_pointer(event->window, NULL, NULL, NULL);
#endif
    }

  /* #if DEBUG == 1 */
  /*   g_timer_stop(timer); */
  /*   fprintf(stderr, "Gtk VisuRenderingWindow: handling mouse motion in %g micro-s.\n", */
  /*           g_timer_elapsed(timer, &fractionTimer)*1e6); */
  /*   g_timer_destroy(timer); */
  /* #endif */
  return TRUE;
}
static gboolean onKeyPressed(GtkWidget *widget _U_, GdkEventKey *event,
			     gpointer data)
{
  SimplifiedEvents ev;
  VisuRenderingWindow *window;
  GList *cameras, *head;

  window = RENDERING_WINDOW(data);
  g_return_val_if_fail(window, TRUE);
  if (!window->inters)
    return FALSE;

  ev.button = 0;
  ev.motion = 0;
  ev.letter = '\0';
  ev.specialKey = Key_None;

  if(event->keyval == GDK_KEY_r || event->keyval == GDK_KEY_R)
    {
      ev.letter = 'r';
      /* If any camera, print a message. */
      visuInteractiveGet_savedCameras(VISU_INTERACTIVE(window->inters->data),
				      &cameras, &head);
      if (cameras)
	renderingWindowPush_message(window, _("Restore saved camera position."));
      else
	renderingWindowPush_message(window, _("No saved camera. Use 's' to save one."));
#if GLIB_MINOR_VERSION > 13
      g_timeout_add_seconds(3, timeOutPopMessage, (gpointer)window);
#else
      g_timeout_add(3000, timeOutPopMessage, (gpointer)window);
#endif
    }
  else if(event->keyval == GDK_KEY_s || event->keyval == GDK_KEY_S)
    {
      ev.letter = 's';
      renderingWindowPush_message(window, _("Save current camera position."));
#if GLIB_MINOR_VERSION > 13
      g_timeout_add_seconds(3, timeOutPopMessage, (gpointer)window);
#else
      g_timeout_add(3000, timeOutPopMessage, (gpointer)window);
#endif
    }
  else if(event->keyval == GDK_KEY_space)
    ev.letter = ' ';
  else if(event->keyval == GDK_KEY_Page_Up)
    ev.specialKey = Key_Page_Up;
  else if(event->keyval == GDK_KEY_Page_Down)
    ev.specialKey = Key_Page_Down;
  else if(event->keyval == GDK_KEY_Down)
    ev.specialKey = Key_Arrow_Down;
  else if(event->keyval == GDK_KEY_Up)
    ev.specialKey = Key_Arrow_Up;
  else if(event->keyval == GDK_KEY_Left)
    ev.specialKey = Key_Arrow_Left;
  else if(event->keyval == GDK_KEY_Right)
    ev.specialKey = Key_Arrow_Right;
  ev.shiftMod = event->state & GDK_SHIFT_MASK;
  ev.controlMod = event->state & GDK_CONTROL_MASK;

  if (ev.letter != '\0' || ev.specialKey != Key_None)
    {
      gdk_window_set_cursor(gtk_widget_get_window(window->openGLArea),
			    RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorWatch);
      visuInteractiveHandle_event(VISU_INTERACTIVE(window->inters->data),
				  window->currentData, &ev);
      gdk_window_set_cursor(gtk_widget_get_window(window->openGLArea),
			    window->currentCursor);
    }
  else if (event->keyval == GDK_KEY_Shift_L || event->keyval == GDK_KEY_Shift_R)
    {
      gdk_window_set_cursor(gtk_widget_get_window(window->openGLArea),
			    RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorGrab);
      window->currentCursor = RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorGrab;
    }

  return FALSE;
}
static gboolean onKeyRelease(GtkWidget *widget _U_, GdkEventKey *event,
			     gpointer data)
{
  VisuRenderingWindow *window;

  window = RENDERING_WINDOW(data);
  g_return_val_if_fail(window, TRUE);

  if (event->keyval == GDK_KEY_Shift_L || event->keyval == GDK_KEY_Shift_R)
    {
      gdk_window_set_cursor(gtk_widget_get_window(window->openGLArea),
			    window->refCursor);
      window->currentCursor = window->refCursor;
    }

  return TRUE;
}

static gboolean timeOutPopMessage(gpointer data)
{
  renderingWindowPop_message(RENDERING_WINDOW(data));

  return FALSE;
}

void renderingWindowPush_message(VisuRenderingWindow *window, gchar *message)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));

  gtk_statusbar_push(GTK_STATUSBAR(window->info->statusInfo),
		     window->info->statusInfoId, message);
  window->nbStatusMessage += 1;
}
void renderingWindowPop_message(VisuRenderingWindow *window)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));

  gtk_statusbar_pop(GTK_STATUSBAR(window->info->statusInfo),
		    window->info->statusInfoId);
  window->nbStatusMessage -= 1;
}
static void getOpenGLAreaSize(VisuRenderingWindow *window,
			      unsigned int *width, unsigned int *height)
{
  GtkAllocation alloc;

  g_return_if_fail(IS_RENDERING_WINDOW(window) && width && height);

  gtk_widget_get_allocation(window->openGLArea, &alloc);
  *width = alloc.width;
  *height = alloc.height;
}
void visu_rendering_window_setData(VisuRenderingWindow *window, VisuData* data)
{
  gboolean noEmit;
  gchar *file;
  guint w, h;
  VisuData *oldData;
  gpointer *wd;
  VisuOpenGLView *view;
  float length0;

  g_return_if_fail(IS_RENDERING_WINDOW(window));
  DBG_fprintf(stderr, "Gtk renderingWindow: attach %p (%p) VisuData to %p window.\n",
	      (gpointer)data, (gpointer)window->currentData, (gpointer)window);
  noEmit = (!window->currentData && !data);
  oldData = window->currentData;
  if (oldData != data)
    {
      if (window->currentData)
	{
	  g_signal_handler_disconnect(G_OBJECT(window->currentData),
				      window->populationIncrease_id);
	  g_signal_handler_disconnect(G_OBJECT(window->currentData),
				      window->populationDecrease_id);
	}
    }

  /* Change the rendering window with the new data (may be the same
     but with a different content). */
  window->currentData = data;
  displayFileInfoOnDataLoaded(window);
  setFileButtonsSensitive(window);

  /* Find a widget to change the title of. */
  wd = g_object_get_data(G_OBJECT(window), RENDERING_WINDOW_ID);
  if (!wd)
    wd = (gpointer)visuGtkGet_render();
  DBG_fprintf(stderr, "##### VisuData association to a window #####\n");
  if (data)
    {
      /* Ref the object. */
      g_object_ref(G_OBJECT(data));
      DBG_fprintf(stderr, "Gtk VisuRenderingWindow: ref new object %p.\n",
		  (gpointer)data);

      if (gtk_widget_get_visible(GTK_WIDGET(window)))
	{
	  view = visu_data_getOpenGLView(data);

	  /* Adapt the Camera to the window. */
          length0 = openGLCameraGet_refLength(view->camera, (ToolUnits*)0);
	  if (length0 <= 0.f)
            openGLCameraSet_refLength(view->camera, visu_data_getBoxExtens(data)[0],
                                      visu_data_getUnit(data));
          visu_data_createAllElements(data);
	  openGLModelize(view->camera);

	  /* Adapt the projection. */
	  getOpenGLAreaSize(window, &w, &h);
	  if (!visu_data_setSizeOfView(data, w, h))
	    openGLProject(view->window, view->camera, visu_data_getBoxExtens(data)[1]);
	}
      /* Change the name of the window, according to the file loaded. */
      if (wd)
	{
	  file = visu_data_getFilesAsLabel(data);
	  if (!file)
	    {
	      g_warning("Can't find the filename to label the rendering window.\n");
	      file = g_strdup(_("No filename"));
	    }
	  gtk_window_set_title(GTK_WINDOW(wd), file);
	  g_free(file);
	}

      /* Attach the default redraw method. */
      openGLWidgetSet_redraw(OPENGL_WIDGET(window->openGLArea),
			     openGL_reDraw, data);

      /* Attach signals to the new #VisuData object. */
      window->populationIncrease_id =
	g_signal_connect(G_OBJECT(data), "NodePopulationIncrease",
			 G_CALLBACK(onNodePopulationChanged), window);
      window->populationDecrease_id =
	g_signal_connect(G_OBJECT(data), "NodePopulationDecrease",
			 G_CALLBACK(onNodePopulationChanged), window);
    }
  else
    {
      /* Erase the name of the window. */
      if (wd)
	gtk_window_set_title(GTK_WINDOW(wd), _("No file loaded"));

      if (oldData)
	{
	  /* Attach the default redraw method. */
	  openGLWidgetSet_redraw(OPENGL_WIDGET(window->openGLArea),
				 (RedrawMethod)0, (VisuData*)0);
	  /* Ask for redraw. */
	  renderingWindowRedraw(window, TRUE);
	}
    }

  if (!noEmit)
    {
      DBG_fprintf(stderr, "Gtk renderingWindow: emitting the 'dataReadyForRendering' signal.\n");
      g_signal_emit(VISU_INSTANCE, VISU_SIGNALS[DATAREADYFORRENDERING_SIGNAL],
		    0 /* details */, (gpointer)data, NULL);
      DBG_fprintf(stderr, "Gtk renderingWindow: emition done.\n");
    }
  visu_marks_setVisuData(window->marks, data);

  /* Reset the statusbar informations and other GUI parameters. */
  while (window->nbStatusMessage > 0)
    renderingWindowPop_message(window);
  if (data)
    {
      /* Put a commentary in the statusbar. */
      renderingWindowPush_message(window, _("Rotate with left b.,"
					    " pick with right b.,"
					    " setup ref. with"
					    " <shift> or <control> b."));

      renderingWindowPush_interactive(window, inter);
      gtk_widget_set_sensitive(window->info->clearMarksButton, TRUE);
    }
  else
    {
      if (oldData)
	renderingWindowPop_interactive(window, inter);
      gtk_widget_set_sensitive(window->info->clearMarksButton, FALSE);
    }

  if (oldData)
    {
      DBG_fprintf(stderr, "Gtk VisuRenderingWindow: unref old object %p.\n",
		  (gpointer)oldData);
      g_object_unref(oldData);
    }
}
/**
 * visu_rendering_window_getVisuData:
 * @window: a valid #VisuRenderingWindow object.
 *
 * This method is used to get the #VisuData attached to a window.
 *
 * Returns: (transfer none): the #VisuData attached to the @window or NULL if none.
 */
VisuData* visu_rendering_window_getVisuData(VisuRenderingWindow *window)
{
  g_return_val_if_fail(IS_RENDERING_WINDOW(window), (VisuData*)0);
  return window->currentData;
}

/***************************/
/* GtkStatusInfo functions */
/***************************/
static void _setLabelSize(GtkInfoArea *info, gint width, gint height)
{
  gchar *str;

  g_return_if_fail(info);

  if (info->fileInfoFreeze)
    return;

  str = g_strdup_printf("<span size=\"smaller\"><b>%s</b> %dx%d</span>", _("Size:"), width, height);
  gtk_label_set_markup(GTK_LABEL(info->labelSize), str);
  g_free(str);
}
static void _setFileDescription(GtkInfoArea *info, gchar* message)
{
  gchar *str;

  g_return_if_fail(info);

  str = g_strdup_printf("<span size=\"smaller\">%s</span>", message);
  gtk_label_set_markup(GTK_LABEL(info->labelFileInfo), str);
  g_free(str);
}
static void _setNNodes(GtkInfoArea *info, gint nb)
{
  GString *str;

  g_return_if_fail(info);

  str = g_string_new("<span size=\"smaller\">");
  if (nb > 0)
    g_string_append_printf(str, _("<b>Nb nodes:</b> %d"), nb);
  else
    g_string_append(str, GTK_STATUSINFO_NONB);
  g_string_append_printf(str, "</span>");
  gtk_label_set_markup(GTK_LABEL(info->labelNb), str->str);
  g_string_free(str, TRUE);
}
static void onNodeInfoClicked(GtkToggleButton *button, gpointer data)
{
  VisuRenderingWindow *window;
  gint *id;

  window = RENDERING_WINDOW(data);
  g_return_if_fail(window);

  id = (gint*)g_object_get_data(G_OBJECT(button), "selectedNodeId");
  g_return_if_fail(id && *id >= 0);

  if (visu_marks_setInfos(window->marks, *id,
			 gtk_toggle_button_get_active(button)))
    VISU_FORCE_REDRAW;
}
static void onMarkClearClicked(GtkButton *button _U_, gpointer data)
{
  VisuRenderingWindow *window;

  window = RENDERING_WINDOW(data);
  g_return_if_fail(window);

  if (visu_marks_removeMeasures(window->marks, -1))
    VISU_FORCE_REDRAW;
}
static gboolean onCameraMenu(VisuRenderingWindow *window, GdkEventButton *event,
			     GtkEventBox *ev _U_)
{
  GtkWidget *wd;

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: click on the camera menu.\n");
  wd = buildCameraMenu(window);
  if (!wd)
    return TRUE;

  g_signal_connect(G_OBJECT(wd), "selection-done",
		   G_CALLBACK(onCameraMenuSelected), (gpointer)window);

  gtk_widget_show_all(wd);
  gtk_menu_popup(GTK_MENU(wd), NULL, NULL, NULL, NULL, 
		 1, event->time);

  return TRUE;
}
static GtkWidget* buildCameraMenu(VisuRenderingWindow *window)
{
  GtkWidget *menu, *item;
  gchar *lbl;
  GList *cameras, *head, *tmpLst, *rCameras;
  VisuOpenGLCamera *current;
  guint n;

  if (!window->currentData || !window->inters)
    return (GtkWidget*)0;

  /* All camera. */
  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: get the cameras.\n");
  visuInteractiveGet_savedCameras(VISU_INTERACTIVE(window->inters->data),
				  &cameras, &head);
  /*   if (!cameras) */
  /*     return (GtkWidget*)0; */

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: build the menu.\n");
  menu = gtk_menu_new();
  gtk_menu_set_accel_group(GTK_MENU(menu), window->accel);
  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: create the camera menu %p.\n",
	      (gpointer)menu);
  
  /* Put the current camera. */
  current = visu_data_getOpenGLView(window->currentData)->camera;
  lbl = g_strdup_printf(_("current:\n"
			  "(\316\270 %6.1f\302\260 ; \317\206 %6.1f\302\260 ; \317\211 %6.1f\302\260) "
			  "dx %4.1f dy %4.1f"),
			current->theta, current->phi, current->omega,
			current->xs, current->ys);
  item = gtk_menu_item_new_with_label(lbl);
  g_free(lbl);
  g_signal_connect(G_OBJECT(item), "activate",
                   G_CALLBACK(onCameraMenuCurrentClicked), window);
  gtk_menu_item_set_accel_path(GTK_MENU_ITEM(item),
                               g_intern_static_string(MENU_CAMERA_SAVE));
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
  /* Separator. */
  item = gtk_separator_menu_item_new();
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
  if (!cameras)
    {
      item = gtk_menu_item_new_with_label(_("No saved camera. Use 's' to save one."));
      gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    }
  
  rCameras = g_list_reverse(g_list_copy(cameras));
  for (tmpLst = rCameras, n = 0; tmpLst; tmpLst = g_list_next(tmpLst), n+= 1)
    {
      current = (VisuOpenGLCamera*)tmpLst->data;
      lbl = g_strdup_printf(_("(\316\270 %6.1f\302\260 ; \317\206 %6.1f\302\260 ; \317\211 %6.1f\302\260) "
			      "dx %4.1f dy %4.1f"),
			    current->theta, current->phi, current->omega,
			    current->xs, current->ys);
      item = gtk_menu_item_new_with_label(lbl);
      DBG_fprintf(stderr, " | add menu item %p (%p)\n", (gpointer)item, (gpointer)window);
      g_free(lbl);
      if (current == (VisuOpenGLCamera*)head->data)
        gtk_menu_item_set_accel_path(GTK_MENU_ITEM(item),
                                     g_intern_static_string(MENU_CAMERA_RESTORE));
      else if (n < 9)
        gtk_menu_item_set_accel_path(GTK_MENU_ITEM(item),
                                     g_intern_static_string(cameraAccels[n]));
      g_signal_connect(G_OBJECT(item), "activate",
		       G_CALLBACK(onCameraMenuClicked), window);
      g_object_set_data(G_OBJECT(item), "Camera", (gpointer)current);
      gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    }
  g_list_free(rCameras);
  return menu;
}
static void _setCamera(VisuRenderingWindow *window, VisuOpenGLCamera *camera)
{
  gboolean reDrawNeeded;

  g_return_if_fail(IS_RENDERING_WINDOW(window));
  if (!window->currentData || !window->inters || !camera)
    return;
  
  visuInteractivePush_savedCamera(VISU_INTERACTIVE(window->inters->data), camera);

  reDrawNeeded = visu_data_setAngleOfView(window->currentData,
                                          camera->theta,
                                          camera->phi,
                                          camera->omega,
                                          VISU_CAMERA_THETA | VISU_CAMERA_PHI | VISU_CAMERA_OMEGA);
  reDrawNeeded = visu_data_setPositionOfView(window->currentData,
                                             camera->xs,
                                             camera->ys,
                                             MASK_XS | MASK_YS) ||
    reDrawNeeded;
  reDrawNeeded = visu_data_setZoomOfView(window->currentData,
                                         camera->gross) ||
    reDrawNeeded;
  reDrawNeeded = visu_data_setPerspectiveOfView(window->currentData,
                                                camera->d_red) ||
    reDrawNeeded;
  if (reDrawNeeded)
    VISU_FORCE_REDRAW;
}
static gboolean onCameraAccel(GtkAccelGroup *accel _U_, GObject *obj,
                              guint key, GdkModifierType mod _U_)
{
  VisuRenderingWindow *window;
  GList *cameras, *head, *rCameras;
  VisuOpenGLCamera *camera;

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: get accelerator for object %p.\n", (gpointer)obj);
  window = RENDERING_WINDOW(g_object_get_data(obj, "RenderingWindow"));
  /* All camera. */
  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: get the cameras.\n");
  visuInteractiveGet_savedCameras(VISU_INTERACTIVE(window->inters->data),
				  &cameras, &head);
  if (!cameras)
    return TRUE;

  rCameras = g_list_reverse(g_list_copy(cameras));
  camera = g_list_nth_data(rCameras, key - GDK_KEY_1);
  g_list_free(rCameras);
  _setCamera(window, camera);

  return TRUE;
}
static void onCameraMenuSelected(GtkMenuShell *menushell, gpointer user_data _U_)
{
  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: destroy the camera menu %p.\n",
	      (gpointer)menushell);
  gtk_widget_destroy(GTK_WIDGET(menushell));
}
static void onCameraMenuClicked(GtkMenuItem *menuitem, gpointer user_data)
{
  _setCamera(RENDERING_WINDOW(user_data),
             (VisuOpenGLCamera*)g_object_get_data(G_OBJECT(menuitem), "Camera"));
}
static void onCameraMenuCurrentClicked(GtkMenuItem *menuitem _U_, gpointer user_data)
{
  visuInteractivePush_savedCamera(VISU_INTERACTIVE(RENDERING_WINDOW(user_data)->inters->data),
                                  visu_data_getOpenGLView(RENDERING_WINDOW(user_data)->currentData)->camera);
}
static void displayFileInfoOnDataLoaded(VisuRenderingWindow *window)
{
  gchar* message;
  VisuDataIter iter;

  g_return_if_fail(window);

  if (window->currentData)
    {
      message = visu_data_getFileCommentary(window->currentData,
                                            visu_data_getISubset(window->currentData));
      visu_data_iterNew(window->currentData, &iter);
      _setNNodes(window->info, iter.nAllStoredNodes);
    }
  else
    {
      message = (gchar*)0;
      _setNNodes(window->info, -1);
    }
  if (message && message[0])
    _setFileDescription(window->info, message);
  else
    _setFileDescription(window->info, GTK_STATUSINFO_NOFILEINFO);
}

void visu_rendering_window_lockUI(VisuRenderingWindow *window, gboolean status)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));

  gtk_widget_set_sensitive(window->info->hboxTools, !status);
  gtk_widget_set_sensitive(window->info->hboxInteractive, !status);
  gtk_widget_set_sensitive(window->info->hboxFileInfo, !status);
}

static void stopProgress(VisuRenderingWindow *window)
{
  if (window->info->progressId)
    g_source_remove(window->info->progressId);
  window->info->progressId = 0;

  visu_rendering_window_lockUI(window, FALSE);
}
static gboolean popProgress(gpointer data)
{
  VisuRenderingWindow *window;

  window = RENDERING_WINDOW(data);

  gtk_progress_bar_pulse(GTK_PROGRESS_BAR(window->info->progress));

  return TRUE;
}
static gboolean showProgress(gpointer data)
{
  VisuRenderingWindow *window;

  window = RENDERING_WINDOW(data);

  visu_rendering_window_lockUI(window, TRUE);
  gtk_widget_show(window->info->progress);
  gtk_widget_show(window->info->cancelButton);
  gtk_widget_hide(window->info->statusInfo);

  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(window->info->progress),
                            _("Loading file..."));

  if (window->info->progressId)
    g_source_remove(window->info->progressId);
  window->info->progressId = g_timeout_add(100, popProgress, data);
  
  return FALSE;
}
static void messProgress(gpointer mess, gpointer progress)
{
  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(progress), (const gchar*)mess);
}
static void onCancelButtonClicked(GtkButton *button _U_, gpointer data)
{
  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(RENDERING_WINDOW(data)->info->progress),
                            _("Cancellation request, waiting for reply..."));
  visu_basic_setLoadCancel(TRUE);
}

struct _load_struct
{
  VisuRenderingWindow *window;
  VisuData *data;
  guint iSet;
};
static gboolean _visu_rendering_window_loadFile(gpointer data)
{
  VisuData *obj;
  GError *error;
  gboolean changeElement, res;
  VisuRenderingWindow *window;
  struct _load_struct *pt;
  int iSet;
  guint waitId;

  /* obj is the new object and main the panel that handle the
     loading. */
  pt = (struct _load_struct*)data;
  window = pt->window;
  obj = pt->data;
  iSet = (int)pt->iSet;
  g_free(pt);

  DBG_fprintf(stderr, "VisuRenderingWindow: loading process ... %p points to"
	      " previous VisuData.\n", (gpointer)window->currentData);

  waitId = g_timeout_add(500, showProgress, (gpointer)window);
  visu_basic_setLoadMessageFunc(messProgress, (gpointer)window->info->progress);
  error = (GError*)0;
  res = visu_basic_loadData(obj, (ToolFileFormat*)0, iSet, &error);
  DBG_fprintf(stderr, "VisuRenderingWindow: basic load OK, continue with Gtk loading parts.\n");
  g_source_remove(waitId);
  stopProgress(window);

  gtk_widget_hide(window->info->progress);
  gtk_widget_hide(window->info->cancelButton);
  gtk_widget_show(window->info->statusInfo);
  
  if (!res)
    {
      g_object_unref(obj);
      obj = (VisuData*)0;
      if (error)
	{
	  visuGtkRaise_warning(_("Loading a file"), error->message, NULL);
	  g_error_free(error);
	}
      else
	g_warning("No error message.");
    }
  else
    if (window->currentData && obj)
      {
	changeElement = visu_data_compareElements(window->currentData, obj);
	visu_data_setChangeElementFlag(obj, changeElement);
      }

  DBG_fprintf(stderr, "Visu Gtk: loading process ... try to load %p.\n",
	      (gpointer)obj);
  visu_rendering_window_setData(window, obj);

  if (!obj)
    return FALSE;

  /* We release a ref on obj, since
     visu_rendering_window_setData has increased it. */
  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: release current data.\n");
  g_object_unref(G_OBJECT(obj));
	 
  DBG_fprintf(stderr, "##### All node creation #####\n");

  visu_data_createAllNodes(obj);

  /* A redraw here is necessary since the load is finished.
     A redraw in the parent may not work since the redraw queue
     may be emptied before. */
  VISU_ADD_REDRAW;

  return FALSE;
}
/**
 * visu_rendering_window_loadFile:
 * @window: a valid #VisuRenderingWindow object.
 * @data: the #VisuData to be loaded.
 * @iSet: the id of @data to load.
 *
 * This method calls the general function to load data from file
 * and deals with errors with gtkDialogs. The filenames must have
 * already been set into @data using visu_data_addFile().
 */
void visu_rendering_window_loadFile(VisuRenderingWindow *window, VisuData *data, guint iSet)
{
  struct _load_struct *pt;

  pt = g_malloc(sizeof(struct _load_struct));
  pt->window = window;
  pt->data   = data;
  pt->iSet   = iSet;
  g_idle_add(_visu_rendering_window_loadFile, pt);
}
void renderingWindowOpen_file(VisuRenderingWindow *window, GtkWindow *parent)
{
  gboolean res;
  VisuGtkSetFilesFunc loadAction;
  VisuData *newData;
  guint w, h;

  loadAction = visuGtkGet_renderingSpecificOpen(visu_object_getRendering(VISU_INSTANCE));
  g_return_if_fail(loadAction);

  if (window->currentData)
    newData = visu_data_newWithView(visu_data_getOpenGLView(window->currentData),
                                    my_class->autoAdjust);
  else
    {
      if (gtk_widget_get_visible(GTK_WIDGET(window)))
	{
	  getOpenGLAreaSize(window, &w, &h);
	  newData = visu_data_newWithSize(w, h);
	}
      else
	newData = visu_data_new();
    }
  g_return_if_fail(newData);

  res = loadAction(newData, parent);
  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: 'loadAction' OK.\n");
  
  if (res)
    visu_rendering_window_loadFile(window, newData, 0);
  else
    g_object_unref(newData);
}

static void onOpen(VisuRenderingWindow *window)
{
  renderingWindowOpen_file(window, (GtkWindow*)0);
}

static void onReload(VisuRenderingWindow *window)
{
  VisuData *dataObj;
  int id;

  g_return_if_fail(IS_RENDERING_WINDOW(window));

  /*   if (RENDERING_WINDOW(data)->currentData) */
  /*     rebuildAllExtensionsLists(RENDERING_WINDOW(data)->currentData); */
  /*   renderingWindowRedraw(RENDERING_WINDOW(data), TRUE); */
  dataObj = window->currentData;
  g_return_if_fail(dataObj);

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: reload current file.\n");
  id = visu_data_getISubset(dataObj);
  visu_data_freePopulation(dataObj);
  g_object_ref(dataObj);
  visu_rendering_window_loadFile(window, dataObj, id);
  VISU_FORCE_REDRAW;
}

static void onRenderingMethodChanged(VisuRenderingWindow *window, VisuRendering *method,
				     gpointer data _U_)
{
  if (window->currentData)
    /* First free attached visuData. */
    visu_rendering_window_setData(window, (VisuData*)0);

  /* Customize interface according to new method. */
  if (method)
    {
      renderingWindowPop_message(window);
      if (window->info->loadButton)
	gtk_widget_set_sensitive(window->info->loadButton, TRUE);
      renderingWindowPush_message(window,
				  _("Use the 'open' button to render a file."));
    }
  else
    {
      if (window->info->loadButton)
	gtk_widget_set_sensitive(window->info->loadButton, FALSE);
      renderingWindowPop_message(window);
    }
}
static void onNodePopulationChanged(VisuData *data, int *nodes _U_,
				    gpointer user_data)
{
  VisuDataIter iter;

  if (data != RENDERING_WINDOW(user_data)->currentData)
    return;

  /* Change the count of nodes. */
  visu_data_iterNew(data, &iter);
  _setNNodes(RENDERING_WINDOW(user_data)->info, iter.nAllStoredNodes);
}
static void setFileButtonsSensitive(VisuRenderingWindow *window)
{
  g_return_if_fail(window);

  if (!window->info->dumpButton || !window->info->reloadButton)
    return;

  if (window->currentData)
    {
      gtk_widget_set_sensitive(window->info->dumpButton, TRUE);
      gtk_widget_set_sensitive(window->info->reloadButton, TRUE);
    }
  else
    {
      gtk_widget_set_sensitive(window->info->dumpButton, FALSE);
      gtk_widget_set_sensitive(window->info->reloadButton, FALSE);
    }
}

void updateDumpProgressBar(gpointer data)
{
  gdouble val;

  g_return_if_fail(GTK_PROGRESS_BAR(data));

  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(data), _("Saving image..."));
  val = gtk_progress_bar_get_fraction(GTK_PROGRESS_BAR(data));
  if (val + 0.01 <= 1.0 && val >= 0.)
    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(data), val + 0.01);
  visuGtkWait();
}

/**
 * visu_rendering_window_dump:
 * @window:a valid #VisuRenderingWindow object ;
 * @format: a #VisuDump object, corresponding to the write method ;
 * @fileName: a string that defined the file to write to ;
 * @width: an integer ;
 * @height: an integer ;
 * @error: (allow-none): a location to store some error (not NULL) ;
 * @functionWait: (allow-none) (scope call): a method to call
 * periodically during the dump ;
 * @data: (closure): some pointer on object to be passed to the wait function.
 *
 * Call this method to dump the given @window to a file.
 *
 * Returns: TRUE if everything went right.
 */
gboolean visu_rendering_window_dump(VisuRenderingWindow *window, VisuDump *format,
                                    const char* fileName, gint width, gint height,
                                    GError **error,
                                    ToolVoidDataFunc functionWait, gpointer data)
{
  GArray *imageData;
  gboolean res;

  g_return_val_if_fail(IS_RENDERING_WINDOW(window), FALSE);
  g_return_val_if_fail(window->currentData, FALSE);
  g_return_val_if_fail(error && !*error, FALSE);
  g_return_val_if_fail(format && fileName, FALSE);

  if (format->bitmap)
    {
      DBG_fprintf(stderr, "Gtk VisuRenderingWindow: dumping current OpenGL area.\n");
      DBG_fprintf(stderr, " | requested size %dx%d.\n", width, height);
      imageData = openGLWidgetGet_pixmapData(OPENGL_WIDGET(window->openGLArea),
					     &width, &height, TRUE,
					     format->hasAlpha);
      /*       visu_data_setSizeOfView(VISU_DATA(window->currentData), */
      /* 			     window->socketWidth, window->socketHeight); */
      /*       openGLWidgetSet_redraw(OPENGL_WIDGET(window->openGLArea), */
      /* 			     openGL_reDraw, window->currentData); */
      DBG_fprintf(stderr, " | allocated size %dx%d.\n", width, height);
      if (!imageData)
	{
	  *error = g_error_new(VISU_ERROR_DUMP, DUMP_ERROR_OPENGL,
			       _("Can't dump OpenGL area to data.\n"));
	  return FALSE;
	}
      /* We must put back the viewport... */
      openGLWindowSet_viewport(visu_data_getOpenGLView(window->currentData)->window,
                               window->socketWidth, window->socketHeight);
    }
  else
    imageData = (GArray*)0;
  /* g_message("%p %s %d %d %p %p %s %p %p",(gpointer)format->fileType, fileName, */
  /* 			  width, height, (gpointer)window->currentData, */
  /* 			  (gpointer)imageData, error, (gpointer)functionWait, (gpointer)data); */
  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: call dump routine.\n");
  res = format->writeFunc(format->fileType, fileName,
			  width, height, window->currentData,
			  (guchar*)((imageData)?imageData->data:0),
                          error, functionWait, data);

  if (imageData)
    g_array_free(imageData, TRUE);
  return res;
}

static void onExport(VisuRenderingWindow *window)
{
  GtkWidget *dump;
  char *filename;
  VisuDump *format;
  gboolean res;
  GError *error;
  GdkCursor *cursorWatch;
  GtkProgressBar *dumpBar;

  g_return_if_fail(IS_RENDERING_WINDOW(window));

  dump = dumpDialog_new(window->currentData,
			(GtkWindow*)0, (const gchar*)0);
  if (gtk_dialog_run(GTK_DIALOG(dump)) != GTK_RESPONSE_ACCEPT)
    {
      gtk_widget_destroy(dump);
      return;
    }

  filename = dumpDialogGet_fileName(DUMP_DIALOG(dump));
  format = dumpDialogGet_dumpType(DUMP_DIALOG(dump));
  g_return_if_fail(format && filename);

  DBG_fprintf(stderr, "Gtk VisuRenderingWindow: dump image to file '%s' (format : %s)\n",
	      filename, tool_file_format_getName(format->fileType));
  if (format->writeFunc)
    {
      cursorWatch = gdk_cursor_new(GDK_WATCH);
      dumpBar = dumpDialogGet_progressBar(DUMP_DIALOG(dump));
      dumpDialogStart(DUMP_DIALOG(dump));
      gtk_progress_bar_set_fraction(dumpBar, 0.);
      if (format->bitmap)
	gtk_progress_bar_set_text(dumpBar,
				  _("Waiting for generating image in memory..."));
      visuGtkWait();
      gdk_window_set_cursor(gtk_widget_get_window(GTK_WIDGET(dump)), cursorWatch);

      error = (GError*)0;
      DBG_fprintf(stderr, " | starting dump.\n");
      res = visu_rendering_window_dump(window, format, filename,
                                       dumpDialogGet_widthValue(DUMP_DIALOG(dump)),
                                       dumpDialogGet_heightValue(DUMP_DIALOG(dump)),
                                       &error, updateDumpProgressBar, (gpointer)dumpBar);

      if (!res && error)
	visuGtkRaise_warning(_("Saving a file"), error->message, (GtkWindow*)0);
      gdk_window_set_cursor(gtk_widget_get_window(GTK_WIDGET(dump)), NULL);
      if (error)
	g_error_free(error);
      DBG_fprintf(stderr, " | release UI resources.\n");
    }
  gtk_widget_destroy(dump);
}

static void onRaiseButtonClicked(VisuRenderingWindow *window, gpointer user_data _U_)
{
  g_signal_emit(G_OBJECT(window), _signals[SHOW_MAIN_PANEL_SIGNAL],
		0 /* details */, NULL);
}

void renderingWindowRedraw(VisuRenderingWindow *window, gboolean forceRedraw)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));
  if (!gtk_widget_get_visible(GTK_WIDGET(window)))
    return;

  if (!openGLGet_immediate() && !forceRedraw)
    {
      DBG_fprintf(stderr, "Redraw rejected since drawing is deferred and not forced.\n");
      return;
    }
  DBG_fprintf(stderr, "Redraw accepted let's go...\n");

  openGLWidgetRedraw(OPENGL_WIDGET(window->openGLArea));
}

void renderingWindowSet_current(VisuRenderingWindow *window, gboolean status)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));

  openGLWidgetSet_current(OPENGL_WIDGET(window->openGLArea), status);
}

static void onRedraw(VisuRenderingWindow *window, gpointer data _U_)
{
  renderingWindowRedraw(window, FALSE);
}
static void onForceRedraw(VisuRenderingWindow *window, gpointer data _U_)
{
  renderingWindowRedraw(window, TRUE);
}

/**
 * visu_rendering_window_getMarks:
 * @window: a #VisuRenderingWindow object ;
 *
 * Get the #VisuMarks of the given @window.
 *
 * Returns: (transfer none): a #VisuMarks object owned by V_Sim.
 *
 * Since: 3.6
 */
VisuMarks* visu_rendering_window_getMarks(VisuRenderingWindow *window)
{
  g_return_val_if_fail(IS_RENDERING_WINDOW(window), (VisuMarks*)0);

  return window->marks;
}

/**
 * renderingWindowClassGet_interactive:
 *
 * The user actions on the rendering area are handled by a
 * #VisuInteractive object.
 * 
 * Since: 3.6
 *
 * Returns: the #VisuInteractive object used by the rendering window of
 * V_Sim. This object is owned by V_Sim.
 */
VisuInteractive* renderingWindowClassGet_interactive()
{
  return inter;
}

/**
 * visu_rendering_window_setDisplayCoordinatesInReduce:
 * @status: a boolean.
 *
 * If set, the coordinates of picked nodes are displayed in reduced values.
 *
 * Returns: TRUE if the value has been changed.
 *
 * Since: 3.6
 */
gboolean visu_rendering_window_setDisplayCoordinatesInReduce(gboolean status)
{
  if (!my_class)
    g_type_class_ref(RENDERING_WINDOW_TYPE);

  if (status == my_class->useReducedCoordinates)
    return FALSE;
  my_class->useReducedCoordinates = status;
  return TRUE;
}
/**
 * visu_rendering_window_getDisplayCoordinatesInReduce:
 *
 * Picked nodes have their coordinates displayed in the status bar of the rendering
 * window. This methods retrieves if they are printed in reduced values or not.
 *
 * Returns: TRUE if the coordinates are displayed in reduced values.
 *
 * Since: 3.6
 */
gboolean visu_rendering_window_getDisplayCoordinatesInReduce()
{
  if (!my_class)
    g_type_class_ref(RENDERING_WINDOW_TYPE);

  return my_class->useReducedCoordinates;
}
/**
 * visu_rendering_window_setAutoAdjust:
 * @status: a boolean.
 *
 * If set, the camera auto adjust its zoom capability for the data to be
 * full size at zoom level 1.
 *
 * Returns: TRUE if the value has been changed.
 *
 * Since: 3.6
 */
gboolean visu_rendering_window_setAutoAdjust(gboolean status)
{
  if (!my_class)
    g_type_class_ref(RENDERING_WINDOW_TYPE);

  if (status == my_class->autoAdjust)
    return FALSE;
  my_class->autoAdjust = status;
  return TRUE;
}
/**
 * visu_rendering_window_getAutoAdjust:
 *
 * The camera can be set to auto adjust its zoom capability for the data to be
 * full size at zoom level 1.
 *
 * Returns: TRUE if the camera auto adjust on data loading.
 *
 * Since: 3.6
 */
gboolean visu_rendering_window_getAutoAdjust()
{
  if (!my_class)
    g_type_class_ref(RENDERING_WINDOW_TYPE);

  return my_class->autoAdjust;
}

/*************************************/
/* Routines related to config files. */
/*************************************/
static gboolean readConfigReducedCoordinates(gchar **lines, int nbLines, int position,
                                             VisuData *dataObj _U_, GError **error)
{
  gboolean red;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readBoolean(lines[0], position, &red, 1, error))
    return FALSE;
  visu_rendering_window_setDisplayCoordinatesInReduce(red);
  
  return TRUE;
}
static gboolean readConfigAutoAdjust(gchar **lines, int nbLines, int position,
                                     VisuData *dataObj _U_, GError **error)
{
  gboolean red;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readBoolean(lines[0], position, &red, 1, error))
    return FALSE;
  visu_rendering_window_setAutoAdjust(red);
  
  return TRUE;
}
static void exportParameters(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_RED_COORD);
  g_string_append_printf(data, "%s[gtk]: %d\n\n", FLAG_PARAMETER_RED_COORD,
			 visu_rendering_window_getDisplayCoordinatesInReduce());
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_AUTO_ADJUST);
  g_string_append_printf(data, "%s[gtk]: %d\n\n", FLAG_PARAMETER_AUTO_ADJUST,
			 visu_rendering_window_getAutoAdjust());
}
