/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD, Damien
	CALISTE, Olivier D'Astier, laboratoire L_Sim, (2001-2005)
  
	Adresses ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD and Damien
	CALISTE and Olivier D'Astier, laboratoire L_Sim, (2001-2005)

	E-mail addresses :
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef SCALARFIELDS_H
#define SCALARFIELDS_H

#include <glib.h>

#include <visu_data.h>
#include <coreTools/toolOptions.h>
#include <coreTools/toolMatrix.h>

/**
 * ScalarField_meshflag:
 * @uniform: the mesh has constant divisions along x, y and z axis ;
 * @nonuniform: the mesh has non linear divisions along x, y or z axis.
 *
 * flag (comment) standing at the begining of a Scalar field file,
 * that gives informations concerning the mesh.
 */
typedef enum
  {
    uniform,
    nonuniform
  } ScalarField_meshflag;

/**
 * ScalarField_struct:
 *
 * Structure that stores informations and data of a scalar field.
 */
struct ScalarField_struct;
/**
 * ScalarField:
 *
 * Short name to address #ScalarField_struct objects.
 */
typedef struct ScalarField_struct ScalarField;

/**
 * SCALAR_FIELD_DEFINED_IN_STRUCT_FILE:
 *
 * Flag used to registered a gboolean property in a #VisuData object.
 * If this flag is TRUE, the file used to read the structure can be used to
 * read a density or a potential.
 */
#define SCALAR_FIELD_DEFINED_IN_STRUCT_FILE "fileFormat_hasPotentialOrDensity"

/**
 * scalarFieldNew:
 * @filename: the path to the filename the field should be read from.
 *
 * Create a new #ScalarField object that is empty (all internal pointers
 * are set to NULL and no memory is allocated except for the object itself.
 * The @filename argument is copied.
 *
 * Returns: a newly created #ScalarField object. Use scalarFieldFree() to
 *          delete it.
 */
ScalarField* scalarFieldNew(const gchar *filename);
/**
 * scalarFieldFree:
 * @field: a #ScalarField object.
 *
 * Free the memory used by the given object.
 */
void scalarFieldFree(ScalarField *field);
/**
 * scalarFieldLoad_fromFile:
 * @filename: the path to the file to be loaded ;
 * @fieldList: a #GList to store read field(s) ;
 * @error: a location on a error pointer ;
 * @table: a set of different options (can be NULL).
 *
 * Read the given file and try to load it as a scalar field file. If succeed,
 * all read fields are appended to the @fieldList argument. If an error
 * occurs, it is stored into @error. When entering the routine, *@error must be NULL.
 * If @table is given, it means that the caller routine gives some options to the loader
 * routine. These options are a set of names and values.
 *
 * If the file contains several fields, they must be loaded and added to @fieldList.
 *
 * Returns: TRUE if everything goes with no error.
 */
gboolean scalarFieldLoad_fromFile(const gchar *filename, GList **fieldList,
				  GError **error, GHashTable *table);
/**
 * scalarFieldGet_minMax:
 * @field: a #ScalarField object ;
 * @minmax: two double values.
 *
 * Get the minimum and the maximum values of the given @field.
 */
void scalarFieldGet_minMax(ScalarField *field, double minmax[2]);
/**
 * scalarFieldSet_data:
 * @field: a #ScalarField object ;
 * @data: array containing data to be copied ;
 * @xyzOrder: a boolean.
 *
 * Set the data of the given @field. The array @data should be stored in z direction
 * first, followed by y and x if @xyzOrder is FALSE, or in the other
 * order when TRUE. The number of elements in the x, y and z directions
 * are read from field->nElements. Then use scalarFieldSet_gridSize()
 * before using this method.
 */
void scalarFieldSet_data(ScalarField *field, double *data, gboolean xyzOrder);
/**
 * scalarFieldSet_fitToBox:
 * @data: a #VisuData object ;
 * @field: a #ScalarField object.
 *
 * Change the box of the scalar field to be the same that the one of the
 * given @data object. This may influence the surfaces that may be created
 * from the field later, for instance.
 */
void scalarFieldSet_fitToBox(VisuData *data, ScalarField *field);
/**
 * scalarFieldGet_box:
 * @field: a #ScalarField object ;
 * @box: 6 floating point locations.
 *
 * This method is used to get the definition of the bounding box.
 */
void scalarFieldGet_box(ScalarField *field, double box[6]);
/**
 * scalarFieldSet_box:
 * @field: a #ScalarField object ;
 * @box: 6 floating point values.
 *
 * This method is used to set the definition of the bounding box.
 */
void scalarFieldSet_box(ScalarField *field, double box[6]);
/**
 * scalarFieldSet_boxFull:
 * @field: a #ScalarField object ;
 * @box: 6 floating point values.
 *
 * This method is used to set the definition of the bounding box.
 */
void scalarFieldSet_boxFull(ScalarField *field, double box[3][3]);
/**
 * scalarFieldGet_gridSize:
 * @field: a #ScalarField object ;
 * @grid: 3 integer locations.
 *
 * This method is used to get the division in x, y, and z directions.
 */
void scalarFieldGet_gridSize(ScalarField *field, int grid[3]);
/**
 * scalarFieldSet_gridSize:
 * @field: a #ScalarField object ;
 * @grid: 3 integers.
 *
 * This method is used to set the division in x, y, and z directions.
 * If the size of internal array for data is changed, it is reallocated and
 * previous data are erased. Use scalarFieldGet_data() to get a pointer on this
 * data array.
 */
void scalarFieldSet_gridSize(ScalarField *field, int grid[3]);
/**
 * scalarFieldSet_meshtype:
 * @field: a #ScalarField object ;
 * @meshtype: a #ScalarField_meshflag object.
 *
 * Change the distribution of the vertex of the scalarfield between
 * regular or custom.
 */
void scalarFieldSet_meshtype(ScalarField *field, ScalarField_meshflag meshtype);
/**
 * scalarFieldGet_meshtype:
 * @field: a #ScalarField object ;
 * to be added
 *
 * The vertex may be distributed linearly along the different
 * directions or customily distributed.
 * 
 * Returns: a #ScalarField_meshflag (uniform or nonuniform).
 */
ScalarField_meshflag scalarFieldGet_meshtype(ScalarField *field);
/**
 * scalarFieldGet_meshx:
 * @field: a #ScalarField object.
 *
 * The mesh along x is stored as an array in x increasing.
 *
 * Returns: a pointer on the allocated meshx array (it should not be freed).
 */
double* scalarFieldGet_meshx(ScalarField *field);
/**
 * scalarFieldGet_meshy:
 * @field: a #ScalarField object.
 *
 * The mesh along y is stored as an array in y increasing.
 *
 * Returns: a pointer on the allocated meshy array (it should not be freed).
 */
double* scalarFieldGet_meshy(ScalarField *field);
/**
 * scalarFieldGet_meshz:
 * @field: a #ScalarField object.
 *
 * The mesh along z is stored as an array in z increasing.
 *
 * Returns: a pointer on the allocated meshz array (it should not be freed).
 */
double* scalarFieldGet_meshz(ScalarField *field);
/**
 * scalarFieldGet_data:
 * @field: a #ScalarField object.
 *
 * The data are stored as a 3 indexes array in x, y annd z increasing.
 *
 * Returns: a pointer on the allocated data array (it should not be freed).
 */
double*** scalarFieldGet_data(ScalarField *field);
/**
 * scalarFieldGet_filename:
 * @field: a #ScalarField object.
 *
 * The data are read from a file.
 *
 * Returns: a pointer on the filename (it should not be freed).
 */
gchar* scalarFieldGet_filename(ScalarField *field);
/**
 * scalarFieldGet_commentary:
 * @field: a #ScalarField object.
 *
 * If the file format support a commentary, this is a good method to get it.
 *
 * Returns: a pointer on the commentary (it should not be freed), can be NULL.
 */
gchar* scalarFieldGet_commentary(ScalarField *field);
/**
 * scalarFieldSet_commentary:
 * @field: a #ScalarField object ;
 * @comment: an UTF-8 string to store as a commentary.
 *
 * A commentary can be associated to a #ScalarField, use this method to set it.
 * The value of @comment is NOT copied.
 */
void scalarFieldSet_commentary(ScalarField *field, gchar* comment);
/**
 * scalarFieldGet_periodic:
 * @field: a #ScalarField object.
 *
 * Get the periodicity status of the scalar field.
 *
 * Returns: TRUE if the scalar field is periodic.
 */
gboolean scalarFieldGet_periodic(ScalarField *field);
/**
 * scalarFieldSet_periodic:
 * @field: a #ScalarField object ;
 * @periodic: a boolean.
 *
 * When the scalar field is periodic, the values on the border x = 1,
 * y = 1 or z = 1 can be obtained reading those of border x = 0, y = 0
 * or z = 0. So if there are n values in one direction, the nth is at
 * position 1 - 1/n in box coordinates in that direction. On the
 * contrary, for non-periodic scalar field, the nth value is at
 * coordinate 1 in box system and can be different from value 0.
 */
void scalarFieldSet_periodic(ScalarField *field, gboolean periodic);
/**
 * scalarFieldGet_allOptions:
 * @field: a #ScalarField object.
 *
 * Some #Option can be stored in association to the values of the scalar field.
 * These options are usually values associated to the read data, such as
 * a spin direction when dealing with density of spin...
 *
 * Returns: a newly created GList that should be freed after use with
 *          g_list_free(). But data of the list are owned by V_Sim and
 *          should not be modified or freed.
 */
GList* scalarFieldGet_allOptions(ScalarField *field);
/**
 * scalarFieldAdd_option:
 * @field: a #ScalarField object ;
 * @option: a newly allocated option.
 *
 * This method adds an option to the list of #Option associated to the data. The given
 * @option will not be duplicated and should not be used elsewhere because it will be freed
 * when the @field will be freed.
 */
void scalarFieldAdd_option(ScalarField *field, ToolOption *option);

/**
 * scalarFieldGet_value:
 * @field: a #ScalarField object ;
 * @xyz: a point coordinate (in real space) ;
 * @value: a location to store the value ;
 * @extension: a possible extension in box coordinates.
 *
 * Knowing the point coordinates, it interpolate a value from the
 * scalar field. If the scalar field is periodic (see
 * scalarFieldSet_periodic()), then it allow the coordinates to extend
 * inside the given @extension.
 *
 * Returns: TRUE if the value can be interpolate, FALSE otherwise, for instance,
 *          when the point @xyz is out of bounds.
 */
gboolean scalarFieldGet_value(ScalarField *field, float xyz[3],
			      double *value, float extension[3]);


/* Handling load methos. */
/**
 * ScalarFieldLoadMethod:
 * @filename: the filename (path) the field should be loaded from ;
 * @fieldList: a #GList to store read field(s) ;
 * @error: a location on a error pointer ;
 * @table: a set of different options (can be NULL).
 *
 * Read the given file try to load it as a scalar field file.
 * If succeed (i.e. with none fatal errors) the method should return TRUE, but if not
 * @fieldList must be unchanged and the method should return TRUE. If an error
 * occurs, it is stored into @error. When entering the routine, *@error must be NULL.
 * If @table is given, it means that the caller routine gives some options to the loader
 * routine. These options are a set of names and values.
 *
 * If the file contains several fields, they must be loaded and added to @fieldList.
 * 
 *
 * Returns: TRUE if the read file is in a valid format (even with minor
 *          errors), FALSE otherwise.
 */
typedef gboolean (*ScalarFieldLoadMethod)(const gchar *filename, GList **fieldList, GError **error,
					  GHashTable *table);

/**
 * scalarFieldAdd_loadMethod:
 * @name: the name of the method ;
 * @method: a #ScalarFieldLoadMethod function ;
 * @format: a #ToolFileFormat object ;
 * @priority: a priority value (the lower value, the higher priority).
 *
 * This routine is used to add a new method to load scalar field. The priority uses
 * the scale of the GLib (G_PRIORITY_DEFAULT is 0, G_PRIORITY_LOW 300 for instance).
 */
void scalarFieldAdd_loadMethod(const gchar *name, ScalarFieldLoadMethod method,
			       ToolFileFormat *format, int priority);
/**
 * scalarFieldGet_allLoadMethods:
 *
 * This routine gives access to all the registered load method for scamlar fields.
 *
 * Returns: a list of all methods. This list is read-only and
 *          own by V_Sim.
 */
GList* scalarFieldGet_allLoadMethods(void);


/**
 * ScalarFieldLoadStruct:
 * @name: a descriptive name ;
 * @fmt: file formats associated to this load method ;
 * @load: (scope call): the load method ;
 * @priority: an int.
 *
 * The priority argument is used when a file is tested to be opened.
 * The smaller, the earlier the load method is tested.
 */
typedef struct _ScalarFieldLoadStruct
{
  gchar* name;

  ToolFileFormat *fmt;
  ScalarFieldLoadMethod load;

  int priority;
}ScalarFieldLoadStruct;


/* Private area. */
/**
 * scalarFieldInit:
 *
 * Initialisation method, should not be used.
 */
void scalarFieldInit(void);

#endif
