#!/usr/bin/python
#
# Urwid BigText example program
#    Copyright (C) 2004-2006  Ian Ward
#
#    This library is free software; you can redistribute it and/or
#    modify it under the terms of the GNU Lesser General Public
#    License as published by the Free Software Foundation; either
#    version 2.1 of the License, or (at your option) any later version.
#
#    This library is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    Lesser General Public License for more details.
#
#    You should have received a copy of the GNU Lesser General Public
#    License along with this library; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# Urwid web site: http://excess.org/urwid/

"""
Urwid graphics example program.
"""

import urwid
import urwid.raw_display

try: True # old python?
except: False, True = 0, 1


class EditWithCallback(urwid.Edit):
	def __init__(self, *argl, **argd):
		self.on_change = argd['on_change']
		del argd['on_change']
		self.edit_text = ""
		urwid.Edit.__init__(self, *argl, **argd)
		
	def set_edit_text(self, text):
		if self.on_change and text != self.edit_text:
			self.on_change(self, text)
		urwid.Edit.set_edit_text(self, text)
		

class SwitchingPadding(urwid.Padding):
	def padding_values(self, size, focus):
		maxcol = size[0]
		width, ignore = self.w.pack(focus=focus)
		if maxcol > width:
			self.align_type = "left"
		else:
			self.align_type = "right"
		return urwid.Padding.padding_values(self, size, focus)


class BigTextDisplay:
	palette = [
		('body',         'black',      'light gray', 'standout'),
		('header',       'white',      'dark red',   'bold'),
		('button normal','light gray', 'dark blue', 'standout'),
		('button select','white',      'dark green'),
		('button disabled','dark gray','dark blue'),
		('edit',         'light gray', 'dark blue'),
		('bigtext',      'white',      'black'),
		('chars',        'light gray', 'black'),
		('exit',         'white',      'dark cyan'),
		]
		
	def create_radio_button(self, g, name, font, fn):
		w = urwid.RadioButton(g, name, False, on_state_change=fn)
		w.font = font
		w = urwid.AttrWrap(w, 'button normal', 'button select')
		return w

	def create_disabled_radio_button(self, name):
		w = urwid.Text("    " + name + " (UTF-8 mode required)")
		w = urwid.AttrWrap(w, 'button disabled')
		return w
	
	def create_edit(self, label, text, fn):
		w = EditWithCallback(label, text, on_change = fn)
		w = urwid.AttrWrap(w, 'edit')
		return w

	def set_font_event(self, w, state):
		if state:
			self.bigtext.set_font(w.font)
			self.chars_avail.set_text(w.font.characters())

	def edit_change_event(self, w, text):
		self.bigtext.set_text(text)

	def setup_view(self):
		fonts = urwid.get_all_fonts()
		# setup mode radio buttons
		self.font_buttons = []
		group = []
		utf8 = urwid.get_encoding_mode() == "utf8"
		for name, fontcls in fonts:
			font = fontcls()
			if font.utf8_required and not utf8:
				rb = self.create_disabled_radio_button(name)
			else:
				rb = self.create_radio_button(group, name, font,
					self.set_font_event)
				if fontcls == urwid.Thin6x6Font:
					chosen_font_rb = rb
					exit_font = font
			self.font_buttons.append( rb )
		
		# Create BigText
		self.bigtext = urwid.BigText("", None)
		bt = SwitchingPadding(self.bigtext, 'left', None)
		bt = urwid.AttrWrap(bt, 'bigtext')
		bt = urwid.Filler(bt, 'bottom', None, 7)
		bt = urwid.BoxAdapter(bt, 7)
		
		# Create chars_avail
		cah = urwid.Text("Characters Available:")
		self.chars_avail = urwid.Text("", wrap='any')
		ca = urwid.AttrWrap(self.chars_avail, 'chars')
		
		chosen_font_rb.set_state(True) # causes set_font_event call
	
		# Create Edit widget
		edit = self.create_edit("", "Urwid "+urwid.__version__,
			self.edit_change_event)
		
		# ListBox
		chars = urwid.Pile([cah, ca])
		fonts = urwid.Pile([urwid.Text("Fonts:")] + self.font_buttons,
			focus_item=1)
		col = urwid.Columns([('fixed',16,chars), fonts], 3, 
			focus_column=1)
		bt = urwid.Pile([bt, edit], focus_item=1)
		l = [bt, urwid.Divider(), col]
		w = urwid.ListBox( l )
		
		# Frame
		w = urwid.AttrWrap(w, 'body')
		hdr = urwid.Text("Urwid BigText example program - F8 exits.")
		hdr = urwid.AttrWrap(hdr, 'header')
		w = urwid.Frame(header=hdr, body=w)

		# Exit message
		exit = urwid.BigText(('exit'," Quit? "), exit_font)
		exit = urwid.Overlay(exit, w, 'center', None, 'middle', None)
		return w, exit


	def main(self):
		self.ui = urwid.raw_display.Screen()
		self.ui.register_palette(self.palette)
		self.ui.set_input_timeouts(5)
		self.view, self.exit_view = self.setup_view()
		self.ui.run_wrapper(self.run)
	
	def run(self):
		self.ui.set_mouse_tracking()
		size = self.ui.get_cols_rows()
		show_exit = False
		do_exit = False
		while True:
			if show_exit:
				canvas = self.exit_view.render(size)
			else:
				canvas = self.view.render(size, focus=True)
				
			self.ui.draw_screen(size, canvas)
			if do_exit:
				break
				
			keys = self.ui.get_input()
				
			if show_exit:
				if 'y' in keys or 'Y' in keys:
					do_exit = True
				show_exit = False
				continue

			self.handle_input(size, keys)
			if 'window resize' in keys:
				size = self.ui.get_cols_rows()
			if 'f8' in keys:
				show_exit = True
	
	def handle_input(self, size, keys):
		for k in keys:
			if urwid.is_mouse_event(k):
				event, button, col, row = k
				self.view.mouse_event( size, event, 
					button, col, row, focus=True )
			elif k != 'window resize':
				self.view.keypress( size, k )

def main():
	BigTextDisplay().main()
	
if '__main__'==__name__:
	main()
