/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001  The plex86 developers team
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <signal.h>
#include <errno.h>

#include "bochs.h"
#include "elf.h"
#include "decode.h"

bx_options_t   bx_options;
bx_debug_t     bx_dbg;

plugin_t *bx_plugin;



extern "C" {

static void bochs_opt(unsigned char *s);
static void parse_line_formatted(int num_params, char *params[]);
static void parse_line_unformatted(char *line);
void bx_devices_init(void);


  int
plugin_init(plugin_t *plugin, int argc, char *argv[])
{
  UNUSED(argc);
  UNUSED(argv);

  bx_plugin = plugin;

  memset(&bx_options, 0, sizeof(bx_options));

  bx_options.floppya.type = BX_FLOPPY_NONE;
  bx_options.floppyb.type = BX_FLOPPY_NONE;
  bx_options.ips = 1000000;

  memset(&bx_dbg, 0, sizeof(bx_dbg));

  vm_register_callback("bochs-opt", bochs_opt);

  return 0;
}

  void
bochs_opt(unsigned char *s)
{
  parse_line_unformatted( (char *) s);
}

void plugin_fini(void)
{
  //struct sigaction sg_act;

  // Now we should disable our timer handler, or it can try
  // to access nonexistent mapped memory => SIGSEGV
  //memset(&sg_act, 0, sizeof(sg_act));
  //sg_act.sa_handler = SIG_IGN;
  //sigaction(SIGALRM,&sg_act,NULL);

  // Tell the GUI to shut down 
  bx_gui.exit();       
}

} /* extern "C" */


  void
parse_line_unformatted(char *line)
{
  char *ptr;
  unsigned i, string_i;
  char string[512];
  char *params[40];
  int num_params;
  Boolean inquotes = 0;

  if (line == NULL) return;

  // if passed nothing but whitespace, just return
  for (i=0; i<strlen(line); i++) {
    if (!isspace(line[i])) break;
    }
  if (i>=strlen(line))
    return;

  num_params = 0;

  ptr = strtok(line, ":");
  while (ptr) {
    string_i = 0;
    for (i=0; i<strlen(ptr); i++) {
      if (ptr[i] == '"')
        inquotes = !inquotes;
      else
        if (!isspace(ptr[i]) || inquotes) {
          string[string_i++] = ptr[i];
          }
      }
    string[string_i] = '\0';
    strcpy(ptr, string);
    params[num_params++] = ptr;
    ptr = strtok(NULL, ",");
    }
  parse_line_formatted(num_params, &params[0]);
}

  void
parse_line_formatted(int num_params, char *params[])
{
  int i;

  if (num_params < 1) return;

  if (params[0][0] == '#') return; /* comment */
  else if (!strcmp(params[0], "floppya")) {
    for (i=1; i<num_params; i++) {
      if (!strncmp(params[i], "1_44=", 5)) {
        strcpy(bx_options.floppya.path, &params[i][5]);
        bx_options.floppya.type = BX_FLOPPY_1_44;
        }
      else if (!strncmp(params[i], "1_2=", 4)) {
        strcpy(bx_options.floppya.path, &params[i][4]);
        bx_options.floppya.type = BX_FLOPPY_1_2;
        }
      else if (!strncmp(params[i], "720k=", 5)) {
        strcpy(bx_options.floppya.path, &params[i][5]);
        bx_options.floppya.type = BX_FLOPPY_720K;
        }
      else if (!strncmp(params[i], "status=ejected", 14)) {
        bx_options.floppya.initial_status = BX_EJECTED;
        }
      else if (!strncmp(params[i], "status=inserted", 15)) {
        bx_options.floppya.initial_status = BX_INSERTED;
        }
      else {
        fprintf(stderr, ".bochsrc: floppya attribute '%s' not understood.\n",
          params[i]);
        exit(1);
        }
      }
    }

  else if (!strcmp(params[0], "floppyb")) {
    for (i=1; i<num_params; i++) {
      if (!strncmp(params[i], "1_44=", 5)) {
        strcpy(bx_options.floppyb.path, &params[i][5]);
        bx_options.floppyb.type = BX_FLOPPY_1_44;
        }
      else if (!strncmp(params[i], "1_2=", 4)) {
        strcpy(bx_options.floppyb.path, &params[i][4]);
        bx_options.floppyb.type = BX_FLOPPY_1_2;
        }
      else if (!strncmp(params[i], "720k=", 5)) {
        strcpy(bx_options.floppyb.path, &params[i][5]);
        bx_options.floppyb.type = BX_FLOPPY_720K;
        }
      else if (!strncmp(params[i], "status=ejected", 14)) {
        bx_options.floppyb.initial_status = BX_EJECTED;
        }
      else if (!strncmp(params[i], "status=inserted", 15)) {
        bx_options.floppyb.initial_status = BX_INSERTED;
        }
      else {
        fprintf(stderr, ".bochsrc: floppyb attribute '%s' not understood.\n",
          params[i]);
        exit(1);
        }
      }
    }

  else if (!strcmp(params[0], "diskc")) {
    if (num_params != 5) {
      fprintf(stderr, ".bochsrc: diskc directive malformed.\n");
      exit(1);
      }
    if (strncmp(params[1], "file=", 5) ||
        strncmp(params[2], "cyl=", 4) ||
        strncmp(params[3], "heads=", 6) ||
        strncmp(params[4], "spt=", 4)) {
      fprintf(stderr, ".bochsrc: diskc directive malformed.\n");
      exit(1);
      }
    strcpy(bx_options.diskc.path, &params[1][5]);
    bx_options.diskc.cylinders = atol( &params[2][4] );
    bx_options.diskc.heads     = atol( &params[3][6] );
    bx_options.diskc.spt       = atol( &params[4][4] );
    bx_options.diskc.present = 1;
    }
  else if (!strcmp(params[0], "diskd")) {
    bx_panic("Diskd not implemented. Use cdromd for CD-ROM support.\n");
    if (num_params != 5) {
      fprintf(stderr, ".bochsrc: diskd directive malformed.\n");
      exit(1);
      }
    if (strncmp(params[1], "file=", 5) ||
        strncmp(params[2], "cyl=", 4) ||
        strncmp(params[3], "heads=", 6) ||
        strncmp(params[4], "spt=", 4)) {
      fprintf(stderr, ".bochsrc: diskd directive malformed.\n");
      exit(1);
      }
    strcpy(bx_options.diskd.path, &params[1][5]);
    bx_options.diskd.cylinders = atol( &params[2][4] );
    bx_options.diskd.heads     = atol( &params[3][6] );
    bx_options.diskd.spt       = atol( &params[4][4] );
    bx_options.diskd.present = 1;
    }

  else if (!strcmp(params[0], "cdromd")) {
    if (num_params != 3) {
      fprintf(stderr, ".bochsrc: cdromd directive malformed.\n");
      exit(1);
      }
    if (strncmp(params[1], "dev=", 4) || strncmp(params[2], "status=", 7)) {
      fprintf(stderr, ".bochsrc: cdromd directive malformed.\n");
      exit(1);
      }
    strcpy(bx_options.cdromd.dev, &params[1][4]);
    if (!strcmp(params[2], "status=inserted"))
      bx_options.cdromd.inserted = 1;
    else if (!strcmp(params[2], "status=ejected"))
      bx_options.cdromd.inserted = 0;
    else {
      fprintf(stderr, ".bochsrc: cdromd directive malformed.\n");
      exit(1);
      }
    bx_options.cdromd.present = 1;
    }

  else if (!strcmp(params[0], "boot")) {
    if (!strcmp(params[1], "a") ||
        !strcmp(params[1], "c")) {
      strcpy(bx_options.bootdrive, params[1]);
      }
    else {
      fprintf(stderr, ".bochsrc: boot directive with unknown boot device '%s'.\n",
        params[1]);
      fprintf(stderr, "          use 'a', or 'c'.\n");
      exit(1);
      }
    }
  else if (!strcmp(params[0], "vga_update_interval")) {
    if (num_params != 2) {
      fprintf(stderr, ".bochsrc: vga_update_interval directive: wrong # args.\n");
      exit(1);
      }
    bx_options.vga_update_interval = atol(params[1]);
    if (bx_options.vga_update_interval < 50000) {
      fprintf(stderr, ".bochsrc: vga_update_interval not big enough!\n");
      exit(1);
      }
    }
  else if (!strcmp(params[0], "keyboard_serial_delay")) {
    if (num_params != 2) {
      fprintf(stderr, ".bochsrc: keyboard_serial_delay directive: wrong # args.\n");
      exit(1);
      }
    bx_options.keyboard_serial_delay = atol(params[1]);
    if (bx_options.keyboard_serial_delay < 5) {
      fprintf(stderr, ".bochsrc: keyboard_serial_delay not big enough!\n");
      exit(1);
      }
    }
  else if (!strcmp(params[0], "floppy_command_delay")) {
    if (num_params != 2) {
      fprintf(stderr, ".bochsrc: floppy_command_delay directive: wrong # args.\n");
      exit(1);
      }
    bx_options.floppy_command_delay = atol(params[1]);
    if (bx_options.floppy_command_delay < 100) {
      fprintf(stderr, ".bochsrc: floppy_command_delay not big enough!\n");
      exit(1);
      }
    }
  else if (!strcmp(params[0], "ips")) {
    if (num_params != 2) {
      fprintf(stderr, ".bochsrc: ips directive: wrong # args.\n");
      exit(1);
      }
    bx_options.ips = atol(params[1]);
    if (bx_options.ips < 200000) {
      fprintf(stderr, ".bochsrc: WARNING: ips is AWEFULLY low!\n");
      }
    }

  else if (!strcmp(params[0], "mouse")) {
    if (num_params != 2) {
      fprintf(stderr, ".bochsrc: mouse directive malformed.\n");
      exit(1);
      }
    if (strncmp(params[1], "enabled=", 8)) {
      fprintf(stderr, ".bochsrc: mouse directive malformed.\n");
      exit(1);
      }
    if (params[1][8] == '0')
      bx_options.mouse_enabled = 0;
    else if (params[1][8] == '1')
      bx_options.mouse_enabled = 1;
    else {
      fprintf(stderr, ".bochsrc: mouse directive malformed.\n");
      exit(1);
      }
    }
  else if (!strcmp(params[0], "private_colormap")) {
    if (num_params != 2) {
      fprintf(stderr, ".bochsrc: private_colormap directive malformed.\n");
      exit(1);
      }
    if (strncmp(params[1], "enabled=", 8)) {
      fprintf(stderr, ".bochsrc: private_colormap directive malformed.\n");
      exit(1);
      }
    if (params[1][8] == '0')
      bx_options.private_colormap = 0;
    else if (params[1][8] == '1')
      bx_options.private_colormap = 1;
    else {
      fprintf(stderr, ".bochsrc: private_colormap directive malformed.\n");
      exit(1);
      }
    }

  else if (!strcmp(params[0], "sb16")) {
    for (i=1; i<num_params; i++) {
      if (!strncmp(params[i], "midi=", 5)) {
	bx_options.sb16.midifile = strdup(&params[i][5]);
        }
      else if (!strncmp(params[i], "midimode=", 9)) {
	bx_options.sb16.midimode = atol(&params[i][9]);
        }
      else if (!strncmp(params[i], "wave=", 5)) {
	bx_options.sb16.wavefile = strdup(&params[i][5]);
        }
      else if (!strncmp(params[i], "wavemode=", 9)) {
	bx_options.sb16.wavemode = atol(&params[i][9]);
        }
      else if (!strncmp(params[i], "log=", 4)) {
	bx_options.sb16.logfile = strdup(&params[i][4]);
        }
      else if (!strncmp(params[i], "loglevel=", 9)) {
	bx_options.sb16.loglevel = atol(&params[i][9]);
        }
      else if (!strncmp(params[i], "dmatimer=", 9)) {
	bx_options.sb16.dmatimer = atol(&params[i][9]);
        }
      }
    }

  else if (!strcmp(params[0], "i440fxsupport")) {
    if (num_params != 2) {
      fprintf(stderr, ".bochsrc: i440FXSupport directive malformed.\n");
      exit(1);
      }
    if (strncmp(params[1], "enabled=", 8)) {
      fprintf(stderr, ".bochsrc: i440FXSupport directive malformed.\n");
      exit(1);
      }
    if (params[1][8] == '0')
      bx_options.i440FXSupport = 0;
    else if (params[1][8] == '1')
      bx_options.i440FXSupport = 1;
    else {
      fprintf(stderr, ".bochsrc: i440FXSupport directive malformed.\n");
      exit(1);
      }
    }
  else if (!strcmp(params[0], "newharddrivesupport")) {
    if (num_params != 2) {
      fprintf(stderr, ".bochsrc: newharddrivesupport directive malformed.\n");
      exit(1);
      }
    if (strncmp(params[1], "enabled=", 8)) {
      fprintf(stderr, ".bochsrc: newharddrivesupport directive malformed.\n");
      exit(1);
      }
    if (params[1][8] == '0')
      bx_options.newHardDriveSupport = 0;
    else if (params[1][8] == '1')
      bx_options.newHardDriveSupport = 1;
    else {
      fprintf(stderr, ".bochsrc: newharddrivesupport directive malformed.\n");
      exit(1);
      }
    }
  else if (!strcmp(params[0], "cmosimage")) {
    if (num_params != 2) {
      fprintf(stderr, ".bochsrc: cmosimage directive: wrong # args.\n");
      exit(1);
      }
    bx_options.cmos.path = strdup(params[1]);
    bx_options.cmos.cmosImage = 1;                // CMOS Image is true
    }
  else if (!strcmp(params[0], "time0")) {
    if (num_params != 2) {
      fprintf(stderr, ".bochsrc: time0 directive: wrong # args.\n");
      exit(1);
      }
    bx_options.cmos.time0 = atoi(params[1]);
    }
#ifdef MAGIC_BREAKPOINT
  else if (!strcmp(params[0], "magic_break")) {
    if (num_params != 2) {
      fprintf(stderr, ".bochsrc: magic_break directive: wrong # args.\n");
      exit(1);
      }
    if (strncmp(params[1], "enabled=", 8)) {
      fprintf(stderr, ".bochsrc: magic_break directive malformed.\n");
      exit(1);
      }
    if (params[1][8] == '0') {
      fprintf(stderr, "Ignoring magic break points\n");
      bx_dbg.magic_break_enabled = 0;
      }
    else if (params[1][8] == '1') {
      fprintf(stderr, "Stopping on magic break points\n");
      bx_dbg.magic_break_enabled = 1;
      }
    else {
      fprintf(stderr, ".bochsrc: magic_break directive malformed.\n");
      exit(1);
      }
    }
#endif
  else if (!strcmp(params[0], "ne2k")) {
    int tmp[6];
    bx_options.ne2k.valid = 0;
    if ((num_params < 4) || (num_params > 6)) {
      fprintf(stderr, ".bochsrc: ne2k directive malformed.\n");
      exit(1);
      }
    bx_options.ne2k.ethmod = "null";
    if (strncmp(params[1], "ioaddr=", 7)) {
      fprintf(stderr, ".bochsrc: ne2k directive malformed.\n");
      exit(1);
      }
    if (strncmp(params[2], "irq=", 4)) {
      fprintf(stderr, ".bochsrc: ne2k directive malformed.\n");
      exit(1);
      }
    if (strncmp(params[3], "mac=", 4)) {
      fprintf(stderr, ".bochsrc: ne2k directive malformed.\n");
      exit(1);
      }
    bx_options.ne2k.ioaddr = strtoul(&params[1][7], NULL, 16);
    bx_options.ne2k.irq = atol(&params[2][4]);
    i = sscanf(&params[3][4], "%x:%x:%x:%x:%x:%x",
             &tmp[0],&tmp[1],&tmp[2],&tmp[3],&tmp[4],&tmp[5]);
    if (i != 6) {
      fprintf(stderr, ".bochsrc: ne2k mac address malformed.\n");
      exit(1);
      }
    for (i=0;i<6;i++)
      bx_options.ne2k.macaddr[i] = tmp[i];
    if (num_params > 4) {
      if (strncmp(params[4], "ethmod=", 7)) {
      fprintf(stderr, ".bochsrc: ne2k directive malformed.\n");
      exit(1);
        }
      bx_options.ne2k.ethmod = strdup(&params[4][7]);
      if (num_params == 6) {
      if (strncmp(params[5], "ethdev=", 7)) {
      fprintf(stderr, ".bochsrc: ne2k directive malformed.\n");
      exit(1);
          }
      bx_options.ne2k.ethdev = strdup(&params[5][7]);
        }
      }
    bx_options.ne2k.valid = 1;
    }

  else {
    fprintf(stderr, ".bochsrc: directive '%s' not understood\n", params[0]);
    exit(1);
    }
}
