/****************************************************************************
** ui.h extension file, included from the uic-generated form implementation.
**
** If you wish to add, delete or rename functions or slots use
** Qt Designer which will update this file, preserving your code. Create an
** init() function in place of a constructor, and a destroy() function in
** place of a destructor.
*****************************************************************************/

#include <qapplication.h>
#include <qclipboard.h>
#include <qfileinfo.h>
#include <q3listview.h>
#include <q3popupmenu.h>
#include <q3process.h>

/** @brief Filters the view by the given pattern. 
  *
  * Called whenever the text in the filterinput changed. */
void FilenameView::onFilterChanged( const QString & pattern )
{
	_filterTextEmpty = pattern.isEmpty();
	updateView();
}


/** @brief Clears the content of the widget excluding the filter.
  *
  * The file list and the error message will be cleared.
  */
void FilenameView::clear()
{
	_errorMessage = "";
	_entries.clear();
	_pFilenameView->clear();
}

/** @brief Sets the search pattern for the filter. 
  * 
  * The view will be updated accordingly. */
void FilenameView::setFilterText( const QString & pattern )
{
	_pFilterInput->setText(pattern);
}

/** Adds an entry to the widget. */
void FilenameView::addEntry( const QString & entry )
{
	_entries.push_back(entry);
	insertItem(entry);
}


void FilenameView::init()
{
	_filterTextEmpty = true;
	_pFilenameView->header()->hide();
	_pErrorDisplay->setShown(false);
	connect(_pShowButton, SIGNAL(clicked()), SIGNAL(showRequested()));
}

/** Shows the entries which shall be shown according to the filter currently active. */
void FilenameView::updateView()
{
	_pFilenameView->clear();
	if (_errorMessage.isEmpty()) // if no error was indicated
	{
		_pFilenameView->setShown(true);
		_pErrorDisplay->setShown(false);
		
		for ( QStringList::iterator it = _entries.begin(); it != _entries.end(); ++it)
		{
			insertItem(*it);
		}
	}
	else
	{
		_pErrorDisplay->setText(_errorMessage);
		_pFilenameView->setShown(false);
		_pErrorDisplay->setShown(true);
	}
}

/** @brief Call this function if you want to indicate an error.
  * 
  * The error message will be shown and not affected by the filter.
  * The error state will cleared at next call of the clear() function. */
void FilenameView::setErrorMessage( const QString & errorMessage )
{
	/** The error message holds the error state. If it is empty no error has occured. */
	_errorMessage = errorMessage;
	updateView();
}

/** @brief This adds an item to the filename view.
   *
   * The item is displayed only if it matches the active filter. 
   */
void FilenameView::insertItem( const QString & entry )
{
	if ( _filterTextEmpty || entry.contains(_pFilterInput->text()) )
	{
		Q3ListViewItem* pItem = new Q3ListViewItem(_pFilenameView, entry);
		// allow to copy the item
		pItem->setRenameEnabled(0, true);
	}
	_pFilenameView->setShown(true);
	_pErrorDisplay->setShown(false);
}


void FilenameView::onFileListContextMenuRequested( Q3ListViewItem * pItem, const QPoint & pos, int col )
{
	// if no item was selected return immidiately
	if (pItem == 0)
		return;
	enum {
		COPY_TO_CLIPBOARD = 1,
		COPY_ALL_TO_CLIPBOARD,
		VIEW_FILE,
	};
	Q3PopupMenu menu(this);
	QString filename = pItem->text(0);
	menu.insertItem(tr("Copy to clipboard"), COPY_TO_CLIPBOARD);
	menu.insertItem(tr("Copy all filenames to clipboard"), COPY_ALL_TO_CLIPBOARD);
	int item = menu.insertItem(tr("View file (depends on settings in /etc/mailcap)"), VIEW_FILE);
	if (!isFileViewable(filename))
		menu.setItemEnabled(item, false);
	
	switch (menu.exec(pos))
	{
		case COPY_TO_CLIPBOARD:
		{
			QClipboard *pCb = QApplication::clipboard();
			pCb->setText(filename, QClipboard::Clipboard);
			pCb->setText(filename, QClipboard::Selection);
			break;
		}
		case COPY_ALL_TO_CLIPBOARD:
		{
			QClipboard *pCb = QApplication::clipboard();
			QString itemsString = getAllVisibleItems().join("\n");
			pCb->setText(itemsString, QClipboard::Clipboard);
			pCb->setText(itemsString, QClipboard::Selection);
			break;
		}
		case VIEW_FILE:
			viewFile(filename);
			break;
		default:
			break;
	}
}

// implemented in FilenameViewImpl
void FilenameView::viewFile(QString filename)
{
}


// implemented in FilenameViewImpl
void FilenameView::onProcessExited( Q3Process * pProcess )
{
}

/** Returns the text of all visible items in a string list.
  *
  * Each string will contain one line, the first item in the stringlist corresponds to the first in
  * the listview.
  *
  * If the error state is set, an emtpy list will be returned.
  */
QStringList FilenameView::getAllVisibleItems()
{
	QStringList result;
	if (_errorMessage.isEmpty())
	{
		for (
				Q3ListViewItemIterator it(_pFilenameView, Q3ListViewItemIterator::Visible);
		it.current() != 0;
		++it
				)
		{
			result.push_back((*it)->text(0));
		}
	}
	return result;
}


/** @brief Returns if the file is viewable (i.e. no directory and readable). */
bool FilenameView::isFileViewable(QString filename)
{
	QFileInfo fileinfo(filename);
	return fileinfo.isReadable() && !fileinfo.isDir();
}


void FilenameView::onItemDoubleClicked( Q3ListViewItem * pItem, const QPoint & pos, int column )
{

}
