/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <errno.h>
#include <string.h>
#include <inttypes.h>
#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <ctype.h>
#include <stdlib.h>

#include "libsysactivity.h"
#include "utils.h"

static int iterate_devices(struct sa_data_storage* dst, uint16_t dst_size, uint16_t* count);
static int is_a_real_device(char* dev_name, size_t name_len);
static int read_stat_file(struct sa_data_storage* dst, char* dev_name, size_t name_len);

extern int kernel_version;
__thread DIR* dir_sys_block;

int sa_open_data_storage() {
	dir_sys_block = NULL;

	if (kernel_version < 0)
		return ENOTSUP;

	dir_sys_block = opendir("/sys/block/");
	if (dir_sys_block == NULL)
		return EIO;
	return 0;
}

int sa_close_data_storage() {
	if (dir_sys_block != NULL)
		closedir(dir_sys_block);
	return 0;
}

int sa_count_data_storages(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	if (iterate_devices(NULL, 0, number) != 0)
		return EIO;
	return 0;
}

int sa_get_data_storage(char* name, struct sa_data_storage* dst) {
	if (name == NULL || dst == NULL)
		return EINVAL;

	size_t name_len = strlen(name);
	if (!is_a_real_device(name, name_len))
		return ENODEV;
	return read_stat_file(dst, name, name_len);
}

int sa_get_data_storages(struct sa_data_storage* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	return iterate_devices(dst, dst_size, written);
}

static int iterate_devices(struct sa_data_storage* dst, uint16_t dst_size, uint16_t* count) {
	struct dirent* device_folder;
	size_t name_len;
	*count = 0;
	rewinddir(dir_sys_block);

	while ((device_folder = readdir(dir_sys_block)) != NULL) {
		name_len = strlen(device_folder->d_name);
		if (!is_a_real_device(device_folder->d_name, name_len))
			continue;

		if (dst != NULL) {
			if (*count == dst_size)
				return ENODEV;
			if (read_stat_file(&dst[*count], device_folder->d_name, name_len) != 0)
				return EIO;
		}
		(*count)++;
	}

	return 0;
}

static int is_a_real_device(char* dev_name, size_t name_len) {
	char line_buffer[64];
	DIR* tmp_dir;

	if (name_len > sizeof line_buffer - sizeof("/sys/block//device/") - 1)
		return EIO;
	strcpy(line_buffer, "/sys/block/");
	strcat(line_buffer, dev_name);
	strcat(line_buffer, "/device/");

	tmp_dir = opendir(line_buffer);
	if (tmp_dir == NULL)
		return 0;

	closedir(tmp_dir);
	return 1;
}

static int read_stat_file(struct sa_data_storage* dst, char* dev_name, size_t name_len) {
	char line_buffer[192];
	FILE* file_stat;

	if (name_len > sizeof line_buffer - sizeof("/sys/block//stat") - 1)
		return EIO;
	strcpy(line_buffer, "/sys/block/");
	strcat(line_buffer, dev_name);
	strcat(line_buffer, "/stat");

	file_stat = fopen(line_buffer, "r");
	if (file_stat == NULL)
		return EIO;
	if (fgets(line_buffer, sizeof line_buffer, file_stat) == NULL)
		return EIO;

	char* pos = line_buffer;
	errno = 0;
	while (isblank(*pos))
		pos++;
	dst->reads = strtoull(pos, NULL, 10);
	pos = skip_value(pos);
	dst->reads_merged = strtoull(pos, NULL, 10);
	pos = skip_value(pos);
	dst->sectors_read = strtoull(pos, NULL, 10);
	pos = skip_value(pos);
	dst->time_spent_reading = strtoull(pos, NULL, 10);
	pos = skip_value(pos);
	dst->writes = strtoull(pos, NULL, 10);
	pos = skip_value(pos);
	dst->sectors_written = strtoull(pos, NULL, 10);
	pos = skip_value(pos);
	dst->time_spent_writing = strtoull(pos, NULL, 10);
	if (errno != 0)
		return EIO;

	strlcpy(dst->name, dev_name, sizeof(dst->name) - 1);
	fclose(file_stat);
	return 0;
}
