/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include <libsyncml/syncml.h>
#include <libsyncml/syncml_internals.h>

#ifdef ENABLE_HTTP
#include <libsyncml/sml_transport_internals.h>

#include "http_client.h"
#include "http_client_internals.h"

/**
 * @defgroup GroupIDPrivate Group Description Internals
 * @ingroup ParentGroupID
 * @brief The private part
 * 
 */
/*@{*/


static void _msgReceived(SoupMessage *msg, gpointer userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, msg, userdata);
	SmlTransportHttpClientEnv *env = userdata;
	
	SmlError *error = NULL;
	SmlMimeType mimetype = SML_MIMETYPE_UNKNOWN;
	smlTrace(TRACE_INTERNAL, "Result: %d %s", msg->status_code, msg->reason_phrase);

	smlTransportDataDeref(env->data);
	env->data = NULL;

	if (!SOUP_STATUS_IS_SUCCESSFUL (msg->status_code)) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Call not successfull: %d %s", msg->status_code, msg->reason_phrase);
		//if (SOUP_STATUS_IS_CLIENT_ERROR(msg->status_code))
		goto error;
	}
	
	const char *header = soup_message_get_header(msg->response_headers, "Content-Type");
	smlTrace(TRACE_INTERNAL, "content type: %s", header);
	
	if (header && !g_strncasecmp(header, "application/vnd.syncml+xml", 26))
		mimetype = SML_MIMETYPE_XML;
	else if(header && !g_strncasecmp(header, "application/vnd.syncml+wbxml", 28))
		mimetype = SML_MIMETYPE_WBXML;
	else if (header) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unknown mimetype");
		goto error;
	} else {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Faulty mimetype");
		goto error;
		
	}
	
	char *data = smlTryMalloc0(msg->response.length, &error);
	if (!data)
		goto error;
		
	memcpy(data, msg->response.body, msg->response.length);
	
	SmlTransportData *tspdata = smlTransportDataNew(data, msg->response.length, mimetype, TRUE, &error);
	if (!tspdata)
		goto error_free_data;
	
	smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_DATA, tspdata, NULL);
	
	smlTransportDataDeref(tspdata);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
	
error_free_data:
	g_free(data);
error:
	smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_ERROR, NULL, error);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
}

/*@}*/

/**
 * @defgroup GroupID Group Description
 * @ingroup ParentGroupID
 * @brief What does this group do?
 * 
 */
/*@{*/

static void *smlTransportHttpClientInit(SmlTransport *tsp, const void *data, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, tsp, data, error);
	smlAssert(tsp);
	smlAssert(data);
	
	g_type_init();
	
	SmlTransportHttpClientEnv *env = smlTryMalloc0(sizeof(SmlTransportHttpClientEnv), error);
	if (!env)
		goto error;
	
	const SmlTransportHttpClientConfig *config = data;
	
	if (!(config->port > 0 && config->port < 65535)) {
		smlErrorSet(error, SML_ERROR_MISCONFIGURATION, "specified port was wrong");
		goto error_free_env;
	}
	env->port = config->port;
	
	//FIXME do url checking. use whitelist
	if (!config->url) {
		smlErrorSet(error, SML_ERROR_MISCONFIGURATION, "specified url was wrong");
		goto error_free_env;
	}
	env->url = g_strdup(config->url);
	env->proxy = g_strdup(config->proxy);
	
	smlTrace(TRACE_INTERNAL, "context %p, config: port %i, url %s, proxy %s", tsp->context, env->port, env->url, env->proxy);
	
	env->tsp = tsp;
	
	if (!(env->session = soup_session_async_new_with_options(SOUP_SESSION_ASYNC_CONTEXT, tsp->context, SOUP_SESSION_PROXY_URI, config->proxy, NULL))) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unable to create new session");
		goto error_free_env;
	}
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, env);
	return env;

error_free_env:
	g_free(env);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

static SmlBool smlTransportHttpClientFinalize(void *data, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, data, error);
	smlAssert(data);
	SmlTransportHttpClientEnv *env = data;
	
	smlAssert(env->tsp);
	
	if (env->data)
		smlTransportDataDeref(env->data);
	
	g_object_unref(env->session);
	
	g_free(env->url);
	g_free(env->proxy);
	g_free(env);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
}

static void smlTransportHttpClientSend(void *userdata, void *link, SmlTransportData *data, SmlError *error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, userdata, link, data, error);
	smlAssert(data);
	smlAssert(userdata);
	SmlTransportHttpClientEnv *env = userdata;
	smlAssert(env);
	smlAssert(env->url);
	
	if (error)
		goto error;
		
	smlTrace(TRACE_INTERNAL, "test %s", env->url);
	SoupMessage *msg = soup_message_new(SOUP_METHOD_POST, env->url);
	if (!msg) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to create new message");
		goto error;
	}
	
	switch (data->type) {
		case SML_MIMETYPE_XML:
			soup_message_add_header(msg->request_headers, "Content-Type", "application/vnd.syncml+xml");
			soup_message_add_header(msg->request_headers, "Accept",	"application/vnd.syncml+xml");
			break;
		case SML_MIMETYPE_WBXML:
			soup_message_add_header(msg->request_headers, "Content-Type", "application/vnd.syncml+wbxml");
			soup_message_add_header(msg->request_headers, "Accept",	"application/vnd.syncml+wbxml");
			break;
		default:
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unknown Mimetype");
			goto error_free_message;
	}

	msg->request.body = data->data;
	msg->request.length = data->size;
	msg->request.owner = SOUP_BUFFER_USER_OWNED;
	env->data = data;
	smlTransportDataRef(data);
	
	soup_session_queue_message(env->session, msg, _msgReceived, userdata);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
	
error_free_message:
	g_object_unref(msg);
error:
	smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_ERROR, NULL, error);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
}

/** @brief Function description
 * 
 * @param parameter This parameter does great things
 * @returns What you always wanted to know
 * 
 */
SmlBool smlTransportHttpClientNew(SmlTransport *tsp, SmlError **error)
{
	tsp->functions.initialize = smlTransportHttpClientInit;
	tsp->functions.finalize = smlTransportHttpClientFinalize;
	tsp->functions.send = smlTransportHttpClientSend;
	return TRUE;
}

#endif //ENABLE_HTTP

/*@}*/
