/** Copyright  2003 by Jean-Hugues de Raigniac <jhraigniac@workingfrog.org>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.workingfrog.i18n.swing;

import java.awt.Component;
import java.util.HashMap;

import javax.swing.Icon;
import javax.swing.JScrollBar;
import javax.swing.JTabbedPane;

import org.workingfrog.i18n.util.LocaleEvent;
import org.workingfrog.i18n.util.LocaleListener;
import org.workingfrog.i18n.util.Translator;

/**
 * i18n version of javax.swing.JTabbedPane
 *
 * @author Jean-Hugues de Raigniac
 */
public class I18NJTabbedPane extends JTabbedPane implements LocaleListener {

    /** i18n keys */
    private HashMap i18nKeys = new HashMap();

    /**
     * Override javax.swing.JTabbedPane.JTabbedPane ().
     */
    public I18NJTabbedPane () {
        super();
    }

    /**
     * Override javax.swing.JTabbedPane.JTabbedPane (int).
     *
     * @param tabPlacement the placement for the tabs relative to the content
     */
    public I18NJTabbedPane (int tabPlacement) {
        super(tabPlacement);
    }

    /**
     * Override
     * javax.swing.JTabbedPane.addTab (java.lang.String, java.awt.Component).
     *
     * @param i18nKey the title to be displayed in this tab (i18n bundle key)
     * @param component The component to be displayed when this tab is clicked
     */
    public void addTab(String i18nKey, Component component) {
        super.addTab(initKey(i18nKey), component);
    }

    /**
     * Override
     * javax.swing.JTabbedPane.addTab
     * (java.lang.String, javax.swing.Icon, java.awt.Component).
     *
     * @param i18nKey the title to be displayed in this tab (i18n bundle key)
     * @param icon the icon to be displayed in this tab
     * @param component The component to be displayed when this tab is clicked
     */
    public void addTab(String i18nKey, Icon icon, Component component) {
        super.addTab(initKey(i18nKey), icon, component);
    }

    /**
     * Override
     * javax.swing.JTabbedPane.addTab (java.lang.String, javax.swing.Icon,
     *                                 java.awt.Component, java.lang.String).
     *
     * @param i18nKey the title to be displayed in this tab (i18n bundle key)
     * @param icon the icon to be displayed in this tab
     * @param component The component to be displayed when this tab is clicked
     * @param tip the tooltip to be displayed for this tab
     */
    public void addTab(String i18nKey, Icon icon,
                       Component component, String tip) {
        super.addTab(initKey(i18nKey), icon, component, tip);
    }

    /**
     * Override javax.swing.JTabbedPane.getTitleAt (int).
     *
     * @param index Tab index
     * @return String
     */
    public String getTitleAt (int index) {
        return Translator.checkValue(
            (String) i18nKeys.get(new Integer(index)), this);
    }

    /**
     * Set the i18nKey as an attribute and return it's localization.
     *
     * @param i18nKey i18n bundle key
     * @return A localization of i18nKey
     */
    private String initKey (String i18nKey) {
        i18nKeys.put(new Integer(getTabCount()), i18nKey);
        Translator.checkKey(i18nKey, this);
        return Translator.checkValue(i18nKey, this);
    }

    /**
     * Propagate the LocaleEvent to the components of this JTabbedPane.
     *
     * @param event contains the new Locale
     */
    public void localeChanged (LocaleEvent event) {

        for (int i = 0; i < getTabCount(); i++) {
            setTitleAt(i, Translator.checkValue(
                (String) i18nKeys.get(new Integer(i)), this));
        }

        Component[] components = getComponents();

        for (int i = 0; i < components.length; i++) {

            if ((!(components[i] instanceof JScrollBar))
                && Translator.checkObject(components[i], this)) {
                ((LocaleListener) components[i]).localeChanged(event);
            }
        }
    }
}
