/**
 * @file schema.c Schema functions
 * 
 * $Id: schema.c,v 1.4 2001/07/28 08:39:50 chipx86 Exp $
 *
 * @Copyright (C) 1999-2001 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <gnurdf.h>
#include <libgnurdf/internal.h>

RdfSchema *
rdfNewSchema(void)
{
	RdfSchema    *schema;
	RdfNamespace *rdf;

	rdfInitializeParser();
	
	schema = xmlNewDoc("1.0");
	rdf = rdfGetRdfNamespace(schema);
	schema->xmlRootNode = xmlNewDocNode(schema, rdf, "RDF", NULL);

	return schema;
}

void
rdfDestroySchema(RdfSchema *schema)
{
	if (schema == NULL)
		return;

	xmlFreeDoc((xmlDoc *)schema);
}

RdfSchema *
rdfReadFile(const char *filename)
{
	xmlDoc *res;

	rdfInitializeParser();
	
	res = xmlParseFile(filename);

	if (res == NULL)
		return NULL;
	
	if (res->xmlRootNode == NULL)
	{
		fprintf(stderr, "libgnurdf: rdfReadFile: "
				"%s: Tree is empty\n", filename);
		xmlFreeDoc(res);
		
		return NULL;
	}

	if (strcmp((char *)res->xmlRootNode->name, "RDF"))
	{
		fprintf(stderr, "libgnurdf: rdfReadFile: "
				"%s: Not an RDF file\n", filename);
		xmlFreeDoc(res);

		return NULL;
	}

	return (RdfSchema *)res;
}

RdfSchema *
rdfReadBuffer(char *buffer)
{
	xmlDoc *res;

	rdfInitializeParser();
	
	res = xmlParseMemory(buffer, strlen(buffer));

	if (res == NULL)
		return NULL;
	
	if (res->xmlRootNode == NULL)
	{
		fprintf(stderr, "libgnurdf: rdfReadBuffer: Tree is empty\n");
		xmlFreeDoc(res);
		
		return NULL;
	}

	if (strcmp((char *)res->xmlRootNode->name, "RDF"))
	{
		fprintf(stderr, "libgnurdf: rdfReadBuffer: Not an RDF file\n");
		xmlFreeDoc(res);

		return NULL;
	}

	return (RdfSchema *)res;
}

void
rdfWriteFile(RdfSchema *schema, const char *filename)
{
	int result;

#ifdef HAVE_LIBXML2
	result = xmlSaveFormatFile(filename, (xmlDoc *)schema, 1);
#else
	result = xmlSaveFile(filename, (xmlDoc *)schema);
#endif
	
	if (result < 0)
	{
		fprintf(stderr, "libgnurdf: rdfWriteFile: "
				"Unable to save file to %s: %s\n",
				filename, strerror(errno));
	}
}

void
rdfWriteBuffer(RdfSchema *schema, char **buffer, int *size)
{
	if (schema == NULL)
		return;
	
#ifdef HAVE_LIBXML2
	xmlDocDumpFormatMemory((xmlDoc *)schema, (xmlChar **)buffer, size, 1);
#else
	xmlDocDumpMemory((xmlDoc *)schema, (xmlChar **)buffer, size);
#endif
}

void
rdfSetCompressionLevel(RdfSchema *schema, int level)
{
	if (schema == NULL)
		return;
	
	xmlSetDocCompressMode(schema, level);
}

int
rdfGetCompressionLevel(RdfSchema *schema)
{
	if (schema == NULL)
		return 0;
	
	return xmlGetDocCompressMode(schema);
}

/**************************************************************************
 * Backwards-compatibility
 **************************************************************************/

RdfSchema *
rdfRead(const char *filename)
{
	return rdfReadFile(filename);
}

void
rdfWrite(RdfSchema *schema, const char *filename)
{
	rdfWriteFile(schema, filename);
}

