/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/*****************************************************************
*
*  File: skeleton.c
*
*  Purpose: Variables and functions for skeleton element handling
*/

#include "include.h"

/* extra attribute type sizes */
int attr_type_size[NUMATTRTYPES] = 
  {0,sizeof(REAL),sizeof(int),sizeof(unsigned long), sizeof(unsigned char)};
char *attr_type_name[NUMATTRTYPES] = { NULL,"real","integer","ulong","uchar"};

/* quadratic interpolation coefficients */
/* partials of interpolation polynomials at midpoints of patch edges */
/* control points are numbered 0 to 5 counterclockwise */
/* midpoints are numbered 0 to 2 counterclockwise after control pt 0 */
/* dip[midpoint][control point][partial]  */

REAL dip[FACET_EDGES][FACET_CTRL][2] = {
    { { -0.5, -0.5 }, { 0.0, -1.0 }, { 0.5, 0.0 }, { 0.0, 1.0 },
      { 0.0, -0.5 }, { 0.0, 1.0 } },
    { { 0.5, 0.5 }, { -1.0, -1.0 }, { 0.5, 0.0  }, { 1.0, 1.0 },
      { 0.0, 0.5 }, { -1.0, -1.0 } },
    { { -0.5, -0.5 }, { 1.0, 0.0 }, { -0.5, 0.0 }, { 1.0, 0.0 },
      { 0.0, 0.5 }, { -1.0, 0.0 } }
  };

int Ord(id)    /* useful in debugger */
element_id id;
{ return ordinal(id); }

void vloop(v_id) /* prints vertex edge loop; useful in debugger */
element_id v_id;
{ edge_id e_id,ee_id;
  int n = 0;
  e_id = get_vertex_edge(v_id);
  if ( !valid_id(e_id) ) 
     { puts("No valid edge on vertex."); return; }
  ee_id = e_id;
  do { printf("%d ",inverted(ee_id)?-(Ord(ee_id)+1): (Ord(ee_id)+1)); 
       ee_id = get_next_tail_edge(ee_id);
       if ( ++n > web.skel[EDGE].count )
       { puts("Unclosed loop."); break;}
     } while ( ee_id != e_id );
  printf("\n");
}

void set_attr(id,attrib)
element_id id;
ATTR attrib;
{
  elptr(id)->attr |= attrib;
}

void unset_attr(id,attrib)
element_id id;
ATTR attrib;
{
  elptr(id)->attr &= ~attrib;
}

void set_fe_edge(fe_id,e_id)
facetedge_id fe_id;
edge_id e_id;
{
  if ( inverted(fe_id) ) invert(e_id);
  feptr(fe_id)->fe_edge_id = e_id;
}

#ifndef get_fe_edge
edge_id get_fe_edge(fe_id)
facetedge_id fe_id;
{
  edge_id e_id;
    
  e_id = feptr(fe_id)->fe_edge_id;
  if ( inverted(fe_id) ) invert(e_id);
  return e_id;
}
#endif


void set_fe_facet(fe_id,e_id)
facetedge_id fe_id;
edge_id e_id;
{
  if ( inverted(fe_id) ) invert(e_id);
  feptr(fe_id)->fe_facet_id = e_id;
}

facet_id get_fe_facet(fe_id)
facetedge_id fe_id;
{
  facet_id f_id;
  
  if ( !valid_id(fe_id) ) return NULLFACET;
  f_id = feptr(fe_id)->fe_facet_id;
  if ( inverted(fe_id) ) invert(f_id);
  return f_id;
}

#ifndef get_prev_edge
facetedge_id get_prev_edge(fe_id)
facetedge_id fe_id;
{
  if ( inverted(fe_id) ) return inverse_id(feptr(fe_id)->nextedge[1]);
  else return feptr(fe_id)->nextedge[0];
}
#endif

#ifndef get_next_edge
facetedge_id get_next_edge(fe_id)
facetedge_id fe_id;
{
  if ( inverted(fe_id) ) return inverse_id(feptr(fe_id)->nextedge[0]);
  else return feptr(fe_id)->nextedge[1];
}
#endif

facetedge_id get_prev_facet(fe_id)
facetedge_id fe_id;
{
  if ( inverted(fe_id) ) return inverse_id(feptr(fe_id)->nextfacet[1]);
  else return feptr(fe_id)->nextfacet[0];
}

facetedge_id get_next_facet(fe_id)
facetedge_id fe_id;
{
  if ( inverted(fe_id) ) return inverse_id(feptr(fe_id)->nextfacet[0]);
  else return feptr(fe_id)->nextfacet[1];
}

void set_prev_edge(fe_id,fe)
facetedge_id fe_id;
facet_id fe;
{
  if ( !valid_id(fe_id) ) return;
  if ( inverted(fe_id) )
    { invert(fe);
      feptr(fe_id)->nextedge[1] = fe;
    }
  else
      feptr(fe_id)->nextedge[0] = fe;
}


void set_next_edge(fe_id,fe)
facetedge_id fe_id,fe;
{
  if ( !valid_id(fe_id) ) return;
  if ( inverted(fe_id) )
    { invert(fe);
      feptr(fe_id)->nextedge[0] = fe;
    }
  else
      feptr(fe_id)->nextedge[1] = fe;
}


void set_prev_facet(fe_id,fe)
facetedge_id fe_id,fe;
{
  if ( !valid_id(fe_id) ) return;
  if ( inverted(fe_id) )
    { invert(fe);
      feptr(fe_id)->nextfacet[1] = fe;
    }
  else
      feptr(fe_id)->nextfacet[0] = fe;
}


void set_next_facet(fe_id,fe)
facetedge_id fe_id,fe;
{
  if ( !valid_id(fe_id) ) return;
  if ( inverted(fe_id) )
    { invert(fe);
      feptr(fe_id)->nextfacet[0] = fe;
    }
  else
      feptr(fe_id)->nextfacet[1] = fe;
}


void set_edge_wrap(e_id,wrap)
edge_id e_id;
WRAPTYPE  wrap;
{
 *EINT(e_id,E_WRAP_ATTR) =  inverted(e_id)  ? (*sym_inverse)(wrap) : wrap ;
}

WRAPTYPE get_edge_wrap(e_id)
edge_id e_id;
{
  WRAPTYPE wrap = *EINT(e_id,E_WRAP_ATTR) ;
  return    ( inverted(e_id) ? (*sym_inverse)(wrap) : wrap );
}

void set_edge_fe(e_id,fe)
edge_id e_id;
facetedge_id fe;
{
    if ( inverted(e_id) ) invert(fe);
    eptr(e_id)->fe_id = fe;
}

facetedge_id get_edge_fe(e_id)
edge_id e_id;
{
    facetedge_id fe;
    
    fe = eptr(e_id)->fe_id;
    if ( inverted(e_id) ) invert(fe);
    return fe;
}

#ifndef get_edge_tailv
vertex_id get_edge_tailv(e_id)
edge_id e_id;
{
  if ( inverted(e_id) )
     return get_edge_vertices(e_id)[web.headvnum];
  else
     return get_edge_vertices(e_id)[0];
}
#endif

#ifndef get_edge_headv
vertex_id get_edge_headv(e_id)
edge_id e_id;
{
  if ( inverted(e_id) )
     return get_edge_vertices(e_id)[0];
  else
     return get_edge_vertices(e_id)[web.headvnum];
}
#endif

void set_edge_tailv(e_id,v_id)
edge_id e_id;
vertex_id v_id;
{
  if ( inverted(e_id) )
     get_edge_vertices(e_id)[web.headvnum] = v_id;
  else
     get_edge_vertices(e_id)[0] = v_id;
  insert_vertex_edge(v_id,e_id);
}


void set_edge_headv(e_id,v_id)
edge_id e_id;
vertex_id v_id;
{
  if ( inverted(e_id) )
     get_edge_vertices(e_id)[0] = v_id;
  else
     get_edge_vertices(e_id)[web.headvnum] = v_id;
  insert_vertex_edge(v_id,inverse_id(e_id));
}

void set_edge_midv(e_id,v_id)
edge_id e_id;
vertex_id v_id;
{ get_edge_vertices(e_id)[2] = v_id;
  set_vertex_edge(v_id,e_id);
  set_attr(v_id,Q_MIDPOINT);
}


body_id get_facet_body(f_id)
facet_id f_id;
{
  if ( web.skel[BODY].count == 0 ) return NULLID;
  if ( !valid_id(f_id) ) return NULLID;
  if ( inverted(f_id) ) return FULONG(f_id,F_BODY_LIST_ATTR)[1];
  else  return FULONG(f_id,F_BODY_LIST_ATTR)[0];
}

void set_facet_body(f_id,b_id)
facet_id f_id;
body_id  b_id;
{
  if ( web.skel[BODY].count == 0 ) return;
  if ( !valid_id(f_id) ) return;

  if ( !valid_id(get_body_fe(b_id)) )
    set_body_fe(b_id,get_facet_fe(f_id));

  if ( everything_quantities_flag )
  { 
     if ( inverted(f_id) )  /* back facet */
     { 
        body_id bb_id = FULONG(f_id,F_BODY_LIST_ATTR)[1];
        if ( equal_id(bb_id,b_id) ) return;
        if ( valid_id(bb_id) )
          unapply_method(f_id,-get_body_volmethneg(bb_id)); 
        FULONG(f_id,F_BODY_LIST_ATTR)[1] = b_id;
        if ( valid_id(b_id) )
          apply_method_num(f_id,-get_body_volmethneg(b_id)); 
     }
     else /* front facet */
     { 
        body_id bb_id = FULONG(f_id,F_BODY_LIST_ATTR)[0];
        if ( equal_id(bb_id,b_id) ) return;
        if ( valid_id(bb_id) )
          unapply_method(f_id,get_body_volmethpos(bb_id)); 
        FULONG(f_id,F_BODY_LIST_ATTR)[0] = b_id;
        if ( valid_id(b_id) )
          apply_method_num(f_id,get_body_volmethpos(b_id)); 
     }
  }
  else
  {
     if ( inverted(f_id) )  FULONG(f_id,F_BODY_LIST_ATTR)[1] = b_id;
     else  FULONG(f_id,F_BODY_LIST_ATTR)[0] = b_id;
  }

}

facetedge_id get_facet_fe(f_id)
facet_id f_id;
{
  facetedge_id fe;
  
  fe = fptr(f_id)->fe_id;
  if ( inverted(f_id) ) invert(fe);
  return fe;
}

void set_facet_fe(f_id,fe)
facet_id f_id;
facetedge_id fe;
{
  
  if ( inverted(f_id) ) invert(fe);
  fptr(f_id)->fe_id = fe;
}


REAL get_vertex_length_star(v_id) vertex_id v_id;
{ edge_id e_id = get_vertex_edge(v_id);
  edge_id firste = e_id;
  REAL star = 0.0;
  if ( get_vattr(v_id) & (Q_MIDPOINT|Q_MIDEDGE) ) return get_edge_length(e_id);
  if ( !valid_id(e_id) ) return 0.0;
  do { star += get_edge_length(e_id); e_id = get_next_tail_edge(e_id);}
  while ( !equal_element(e_id,firste) );
  return star;
}

REAL get_vertex_area_star(v_id) vertex_id v_id;
{ 
  REAL star = 0.0;

  if ( web.representation == STRING ) 
      return get_vertex_length_star(v_id);
  else
  {
     facet_id f_id = get_vertex_facet(v_id);
     facet_id firstf = f_id;
     if ( !valid_id(f_id) ) return 0.0;
     if ( vfacet_timestamp < top_timestamp ) make_vfacet_lists();
     if ( get_vattr(v_id) & Q_MIDPOINT )
     { facetedge_id fe,start_fe;
        fe = start_fe = get_vertex_fe(v_id);
        do 
        { star += get_facet_area(get_fe_facet(fe));
          fe = get_next_facet(fe);
        } while ( !equal_id(fe,start_fe) );
     }
     else if ( get_vattr(v_id) & Q_MIDFACET )
        star = get_facet_area(get_vertex_facet(v_id));
     else do 
      { star += get_facet_area(f_id); f_id = get_next_vertex_facet(v_id,f_id);}
     while ( !equal_element(f_id,firstf) );
  } 
  return star;
}

int get_vertex_fvalence(v_id) vertex_id v_id;
{ 
 int valence = 0;
     facet_id f_id = get_vertex_facet(v_id);
     facet_id firstf = f_id;
     if ( !valid_id(f_id) ) return 0;
     if ( vfacet_timestamp < top_timestamp ) make_vfacet_lists();
     if ( get_vattr(v_id) & Q_MIDPOINT )
     { facetedge_id fe,start_fe;
        fe = start_fe = get_vertex_fe(v_id);
        do 
        { valence++;
          fe = get_next_facet(fe);
        } while ( !equal_id(fe,start_fe) );
     }
     else if ( get_vattr(v_id) & Q_MIDFACET )
        valence = 1;
     else do 
      { valence++; f_id = get_next_vertex_facet(v_id,f_id);}
     while ( !equal_element(f_id,firstf) );
  return valence;
}

facet_id get_next_vertex_facet(v_id,f_id) vertex_id v_id; facet_id f_id;
{ int i; 
  vertex_id *v = get_facet_vertices(f_id);
  facet_id *f = FULONG(f_id,F_NEXT_VFACET_ATTR);
  int vmax = web.skel[FACET].ctrlpts; 

  for ( i = 0 ; i < vmax ; i++ )
     if (equal_id(v_id,v[i]))  return f[i] ; 
  sprintf(errmsg,"Internal error: get_next_vertex_facet failure v %d f %d\n",ordinal(v_id)+1,
     ordinal(f_id)+1);
  kb_error(1306,errmsg,RECOVERABLE);

  return NULLID;
}


void set_next_vertex_facet(v_id,f_id,ff_id)
      vertex_id v_id; facet_id f_id,ff_id;
{ 
   vertex_id *v = get_facet_vertices(f_id);
   facet_id *f = FULONG(f_id,F_NEXT_VFACET_ATTR);
   int vmax = web.skel[FACET].ctrlpts; 
   int i;
   for ( i = 0 ; i < vmax ; i++ )
      if (equal_id(v_id,v[i]))  f[i] = ff_id ; 
}

void set_next_body_facet(f_id,ff_id)  facet_id f_id,ff_id;
  { FULONG(f_id,F_NEXT_BFACET_ATTR)[inverted(f_id)?1:0] = (ff_id) ; }

facet_id get_next_body_facet(f_id) facet_id f_id;
{ return    FULONG(f_id,F_NEXT_BFACET_ATTR)[inverted(f_id)?1:0]; }

void set_body_volume(b_id,v)  body_id b_id; REAL v;
{ if ( !valid_id(b_id) ) return;
  bptr(b_id)->volume = v; 
  bptr(b_id)->voltimestamp = global_timestamp;
  if ( web.representation == STRING)
  { facet_id f_id = get_fe_facet(get_body_fe(b_id));
    set_facet_area(f_id,v);
  }
} 

/* parallel functions for some serial macros */
#ifdef PARALLEL_MACHINE
/* for traversing linked lists of adjoining edges */
edge_id get_next_tail_edge(e_id)  edge_id e_id;
      { return eptr(e_id)->next_vedge[inverted(e_id) ?1: 0] ; }
edge_id get_next_head_edge(e_id)  edge_id e_id;
      { return inverse_id(eptr(e_id)->next_vedge[inverted(e_id) ?0: 1]); }
void set_next_tail_edge(e_id,ee_id) edge_id e_id,ee_id;
    { eptr(e_id)->next_vedge[inverted(e_id) ?1: 0] = (ee_id); }
facetedge_id get_body_fe(b_id)  body_id b_id;
  { return ( valid_id(b_id) ? bptr(b_id)->fe_id : NULLID ); }

facetedge_id get_vertex_fe(v_id) vertex_id v_id;
 { edge_id xx_id=vptr(v_id)->e_id;
    return  valid_id(xx_id)?same_sign(eptr(xx_id)->fe_id,xx_id):NULLID;
 }

void set_body_fe(b_id,fe)  body_id b_id; facetedge_id fe;
  {  valid_id(b_id) ?  bptr(b_id)->fe_id = (fe) : NULLID ; }

REAL get_body_density(b_id) body_id b_id;
    { return  ( valid_id(b_id) ?  bptr(b_id)->density : 0.0 ) ; }

REAL get_body_volume(b_id) body_id b_id;
    { return    ( valid_id(b_id) ?  bptr(b_id)->volume : 0.0 ) ; }

REAL get_body_fixvol(b_id) body_id b_id;
     { return ( valid_id(b_id) ?  bptr(b_id)->fixvol : 0.0 ) ; }

REAL  get_body_pressure(b_id) body_id b_id;
    { return  ( valid_id(b_id) ?    bptr(b_id)->pressure : 0.0 ) ; }

REAL get_body_volconst(b_id) body_id b_id;
  { return    ( valid_id(b_id) ?  bptr(b_id)->volconst : 0.0 ) ; }

void set_body_volconst(b_id,v) body_id b_id; REAL v;
      {         ( valid_id(b_id) ?  bptr(b_id)->volconst = (v) : 0.0 ) ; }

void set_body_density(b_id,v) body_id b_id; REAL v;
      {         ( valid_id(b_id) ?  bptr(b_id)->density = (v) : 0.0 ) ; }

void set_body_pressure(b_id,v) body_id b_id; REAL v;
      {         ( valid_id(b_id) ?  bptr(b_id)->pressure = (v) : 0.0 ) ; }
#endif
/* parallel versions of macros */


void set_body_fixvol(b_id,v) 
body_id b_id; 
REAL v;
{ if ( valid_id(b_id) )
  { bptr(b_id)->fixvol = (v); 
    if ( everything_quantities_flag )
    { struct gen_quant *q = GEN_QUANTS + bptr(b_id)->volquant;
      q->target = v;
      if ( !(q->flags & Q_FILLED_SKIP) && !(q->flags & Q_FIXED) )
      { q->flags &= ~(Q_INFO|Q_ENERGY);
        q->flags |= Q_FIXED;
        reset_quant_bits();
      }
    }
  }
  else
  { sprintf(errmsg,"fix body volume: illegal body %d.\n",ordinal(b_id)+1);
    kb_error(1307,errmsg,RECOVERABLE);
  }
}

vertex_id new_vertex(x,parent)
REAL *x;
element_id parent; /* for inherited stuff */
{ 
  int i;
  vertex_id v_id;
  REAL *y;

  v_id = new_element(VERTEX,parent);
  if ( x )
     { y = get_coord(v_id);
        for ( i = 0 ; i < SDIM ; i++ ) y[i] = x[i];
     }
  set_vertex_edge(v_id,NULLID);

  set_v_global(v_id);

  /* interrupt conjugate gradient */
  if ( cg_hvector ) { myfree((char *)cg_hvector); cg_hvector = NULL; }

  return v_id;
}

vertex_id dup_vertex(old_v)
vertex_id old_v;
{ 
  vertex_id v_id;

  v_id = new_element(VERTEX,NULLID);
  memcpy((char *)&(elptr(v_id)->attr),(char *)&(elptr(old_v)->attr),
              web.sizes[VERTEX] - 2*sizeof(element_id));
  elptr(v_id)->self_id = v_id;  /* restore new id */
  vptr(v_id)->e_id = NULLID;
  if ( web.skel[VERTEX].extra_count )
    memcpy(get_v_extra(v_id,0),get_v_extra(old_v,0),web.skel[VERTEX].extra_size);
  return v_id;
}

edge_id    new_edge(tail_id,head_id,parent)
vertex_id tail_id,head_id;
element_id parent; /* for inherited stuff */
{ 
  edge_id e_id;
  vertex_id v_id;
  REAL *x,*h,*t;
  int i,k;

  e_id = new_element(EDGE,parent);
  set_edge_fe(e_id,NULLFACETEDGE);
  set_edge_color(e_id,DEFAULT_EDGE_COLOR);
  if ( valid_id(tail_id) && valid_id(head_id) )
  { set_edge_tailv(e_id,tail_id);
     set_edge_headv(e_id,head_id);
     if ( (web.modeltype == QUADRATIC) && valid_id(head_id) )
     { /* quadratic version; linear interpolation of midpoint */
        v_id = new_element(VERTEX,parent);
        set_edge_midv(e_id,v_id);
        h = get_coord(head_id);
        t = get_coord(tail_id);
        x = get_coord(v_id);
        for ( i = 0 ; i < SDIM ; i++ )
          x[i] = (h[i] + t[i])/2.0;
     }
     else if ( (web.modeltype == LAGRANGE) && valid_id(head_id) )
     { /* Lagrange version; linear interpolation of points */
        vertex_id *v = get_edge_vertices(e_id);
        h = get_coord(head_id);
        t = get_coord(tail_id);
        for ( k = 1 ; k < web.lagrange_order ; k++ )
        { v[k] = new_element(VERTEX,parent);
          set_attr(v[k],Q_MIDEDGE);
          set_vertex_edge(v[k],e_id);
          x = get_coord(v[k]);
          for ( i = 0 ; i < SDIM ; i++ )
             x[i] = (k*h[i] + (web.lagrange_order-k)*t[i])/web.lagrange_order;
        }
     }
  }
  if  ( web.surf_global_map && (web.representation == STRING) )
     { set_attr(e_id,SURF_ENERGY);
        set_e_surfen_map(e_id,web.surf_global_map);
     }
  if  ( web.quant_global_map && (web.representation == STRING) )
     { set_attr(e_id,SURF_QUANTITY);
        set_e_quant_map(e_id, web.quant_global_map);
     }
  return e_id;
}

edge_id dup_edge(old_e)
edge_id old_e;
{ 
  edge_id e_id;
  vertex_id newmid;

  e_id = new_element(EDGE,NULLID);
  memcpy((char *)&(elptr(e_id)->attr),(char *)&(elptr(old_e)->attr),
              web.sizes[EDGE] - 2*sizeof(element_id));
  elptr(e_id)->self_id = e_id; /* restore new id */
  if ( inverted(old_e) ) return inverse_id(e_id);
  eptr(e_id)->next_vedge[0] = NULLID;
  eptr(e_id)->next_vedge[1] = NULLID;
  if ( web.modeltype == QUADRATIC )
    { newmid = dup_vertex(get_edge_midv(old_e));
      set_edge_midv(e_id,newmid);
    } 
  else if ( web.modeltype == LAGRANGE )
    { int i;
      vertex_id *v = get_edge_vertices(e_id);
      vertex_id *oldv = get_edge_vertices(old_e);
      for ( i = 1 ; i < web.lagrange_order ; i++ )
         v[i] = dup_vertex(oldv[i]);
    } 
  if ( web.skel[EDGE].extra_count )
     memcpy(get_e_extra(e_id,0),get_e_extra(old_e,0),web.skel[EDGE].extra_size);
  return e_id;
}

facet_id  new_facet ()
{ 
  facet_id f_id;

  f_id = new_element(FACET,NULLID);
  set_facet_body(f_id,NULLBODY);
  set_facet_body(inverse_id(f_id),NULLBODY);
  set_facet_fe(f_id,NULLFACETEDGE);
  set_facet_color(f_id,DEFAULT_FACET_COLOR);
  set_facet_density(f_id,1.0);
  if  ( web.surf_global_map && (web.representation != STRING) )
     { set_attr(f_id,SURF_ENERGY);
        set_f_surfmap(f_id,web.surf_global_map);
     }
  if  ( web.quant_global_map && (web.representation != STRING) )
     { set_attr(f_id,SURF_QUANTITY);
        set_f_quantmap(f_id,web.surf_global_map);
     }
  return f_id;
}

facet_id dup_facet(old_f)
facet_id old_f;
{ 
  facet_id f_id;

  f_id = new_element(FACET,NULLID);
  memcpy((char *)&(elptr(f_id)->attr),(char *)&(elptr(old_f)->attr),
              web.sizes[FACET] - 2*sizeof(element_id));
  set_attr(f_id,NEWELEMENT);
  elptr(f_id)->self_id = f_id; /* restore new id */
  if ( web.skel[FACET].extra_count )
     memcpy(get_f_extra(f_id,0),get_f_extra(old_f,0),web.skel[FACET].extra_size);
  return f_id;
}

body_id new_body()
{ 
  body_id b_id;

  expand_attribute(FACET,F_BODY_LIST_ATTR,2);
  b_id = new_element(BODY,NULLID);
  set_body_fe(b_id,NULLFACETEDGE);
  if ( everything_quantities_flag ) 
      convert_body_to_quantity(b_id);
  web.bodycount++;
  return b_id;
}

body_id dup_body(old_b)
body_id old_b;
{ 
  body_id b_id;

  b_id = new_element(BODY,NULLID);
  memcpy((char *)&(elptr(b_id)->attr),(char *)&(elptr(old_b)->attr),
              web.sizes[BODY] - 2*sizeof(element_id));
  elptr(b_id)->self_id = b_id; /* restore new id */
  web.bodycount++;
  if ( web.skel[BODY].extra_count )
     memcpy(get_b_extra(b_id,0),get_b_extra(old_b,0),web.skel[BODY].extra_size);
  return b_id;
}

facetedge_id new_facetedge(f_id,e_id)
facet_id f_id;
edge_id e_id;
{ 
  facet_id fe_id;

  fe_id = new_element(FACETEDGE,NULLID);
  set_fe_edge(fe_id,e_id);
  set_fe_facet(fe_id,f_id);
  set_prev_edge(fe_id,NULLFACETEDGE);
  set_next_edge(fe_id,NULLFACETEDGE);
  set_prev_facet(fe_id,NULLFACETEDGE);
  set_prev_facet(fe_id,NULLFACETEDGE);
  set_vertex_edge(get_edge_tailv(e_id),e_id);
  set_vertex_edge(get_edge_headv(e_id),inverse_id(e_id));

  if ( web.representation==STRING && everything_quantities_flag )
  { /* attach volume quantities */
     body_id b_id;
     b_id = get_facet_body(f_id);
     if ( valid_id(b_id) )
     { if ( same_sign(f_id,e_id) )
          apply_method_num(e_id,get_body_volmethpos(b_id));
        else
          apply_method_num(e_id,get_body_volmethneg(b_id));
     }
     b_id = get_facet_body(inverse_id(f_id));
     if ( valid_id(b_id) )
     { if ( same_sign(f_id,e_id) )
          apply_method_num(e_id,get_body_volmethneg(b_id));
        else
          apply_method_num(e_id,get_body_volmethpos(b_id));
     }
  }
  return fe_id;
}

REAL get_edge_length(e_id)
edge_id e_id;
{
  return    (eptr(e_id)->length);
}


REAL get_facet_pressure(f_id)
facet_id f_id;
{ 
  return  (get_body_pressure(get_facet_body(f_id)) - 
        get_body_pressure(get_facet_body(facet_inverse(f_id))));
}


static facet_id cur_facet;
void generate_facet_fe_init()  { cur_facet = NULLFACET; }

int generate_facet_fe(f_id,fe_id)
facet_id f_id;
facetedge_id *fe_id;
{
  static facetedge_id first_fe,gfe;

  if ( !equal_id(f_id,cur_facet) )  /* first call */
     { cur_facet = f_id;
        first_fe = gfe = *fe_id = get_facet_fe(f_id);
        return 1;
     }
  for(;;)
  {
     gfe = *fe_id = get_next_edge(gfe);
     if ( !valid_id(gfe) ||  equal_id(gfe,first_fe)  ) break;
     if ( equal_id(f_id,get_fe_facet(gfe)) ) return 1;
     /* else continue, in case string loop goes around end of edge */
  }
  cur_facet = NULLFACET;
  *fe_id = NULLFACETEDGE;
  return 0;
}


static edge_id cur_edge;
void generate_edge_fe_init() { cur_edge = NULLEDGE; }

int generate_edge_fe(e_id,fe_id)
edge_id e_id;
facetedge_id *fe_id;
{
  static facetedge_id first_fe,fe;

  if ( !equal_id(e_id,cur_edge) )  /* first call */
     { 
        first_fe = get_edge_fe(e_id);
        if ( !valid_id(first_fe) ) { *fe_id = NULLFACETEDGE; return 0; }
        cur_edge = e_id;
        fe = *fe_id = first_fe;
        return 1;
     }
  if ( !equal_id((fe = *fe_id = get_next_facet(fe)),first_fe) ) return 1;
  *fe_id = NULLFACETEDGE;
  cur_edge = NULLEDGE;
  return 0;
}

/********************************************************************
*
* Function: compare_edge_attr(ea,eb)
* 
* Purpose: See if edge eb can be merged with edge ea.
*          Used in eliminate_edge().  Checks constraints
*          and named methods. Those of edge eb must be
*          a subset of those of edge ea.
*
* Return value: 0  Cannot
*               1  Can
*/

int compare_edge_attr(ea,eb)
edge_id ea,eb;
{ int i,j;
  conmap_t * con1=NULL,*con2=NULL;
  int meth_offset = get_meth_offset(EDGE); 
  struct edge *ea_ptr,*eb_ptr;

  /* check constraints */
  con1 = get_e_constraint_map(ea);
  con2 = get_e_constraint_map(eb);

  if ( con2[0] > con1[0] )
  { if ( verbose_flag ) 
    { sprintf(msg,"Edge %d has more constraints than edge %d.\n",
      ordinal(eb)+1,ordinal(ea)+1);
     outstring(msg);
    }
    return 0;
  }
  for ( i = 1 ; i <= (int)con2[0] ; i++ )
  { for ( j = 1 ; j <= (int)con1[0] ; j++ )
      if ( (con1[j]&CONMASK) == (con2[i]&CONMASK) ) break;
    if ( j > (int)con1[0] )
    { if ( verbose_flag )
      { sprintf(msg,"Edge %d is on constraint %d, but edge %d isn't.\n",
        ordinal(eb)+1,(int)con2[i],ordinal(ea)+1);
        outstring(msg);
      }          
      return 0;
    }
  }

  /* check methods */
  ea_ptr = eptr(ea);
  eb_ptr = eptr(eb);
  for ( i = 0 ; i < (int)eb_ptr->method_count ; i++ )
  { int m;
    m = abs(((int*)((char*)eb_ptr+meth_offset))[i]);
    for ( j = 0 ; j < (int)ea_ptr->method_count ; j++ )
    { int n;
      n = abs(((int*)((char*)ea_ptr+meth_offset))[j]);
    if ( n == m ) break;
    }
    if ( j >= (int)ea_ptr->method_count )
    {  if ( verbose_flag )
       { sprintf(msg,"Edge %d is on method %s but edge %d isn't.\n",
           ordinal(eb)+1,METH_INST[m].name,ordinal(ea)+1);
         outstring(msg);
       }       
       return 0;
    }
  }

  return 1;
}

/**************************************************************
*
*  Function:  equal_constr()
*
*  Purpose:    See if two elements have the same set of constraints.
*
*/

int equal_constr(id1,id2)
element_id id1,id2;
{ int i,j;
  conmap_t * con1=NULL,*con2=NULL;

  switch ( id_type(id1) )
     {
        case VERTEX: 
                         con1        = get_v_constraint_map(id1);
                         break;

        case EDGE  : 
                         con1        = get_e_constraint_map(id1);
                         break;

        case FACET : 
                         con1        = get_f_constraint_map(id1);
                         break;
     }


  switch ( id_type(id2) )
     {
        case VERTEX: 
                         con2        = get_v_constraint_map(id2);
                         break;

        case EDGE  :
                         con2        = get_e_constraint_map(id2);
                         break;

        case FACET :
                         con2        = get_f_constraint_map(id2);
                         break;
     }
  if ( con2[0] != con1[0] ) return 0;
  for ( i = 1 ; i <= (int)con1[0] ; i++ )
  { for ( j = 1 ; j <= (int)con2[0] ; j++ )
        if ( (con1[i]&CONMASK) == (con2[j]&CONMASK) ) break;
     if ( j > (int)con2[0] ) return 0;
  }
  return 1;
}

/*************************************************************************
*
*  function: add_attribute()
*
*  purpose: add extra attribute to an element type
*  return:  index of attribute
*/

int add_attribute(e_type,name,attr_type,dim,dumpflag,code)
int e_type; /* VERTEX ,... */
char *name;
int attr_type; /* REAL_ATTR or INTEGER_ATTR or ULONG_ATTR */
int dim; /* number of components */
int dumpflag; /* whether appears in dump file */
struct expnode *code; /* nonnull for function attribute */
{ int newsize;
  struct extra *ex;
  int oldsize;

  if ( web.skel[e_type].extra_count >= MAXEXTRA-1 )
  { sprintf(errmsg,"Too many extra attributes on %ss; limit is %d.\n",
      typenames[e_type], MAXEXTRA-1);
     kb_error(1309,errmsg,RECOVERABLE);
  }
  /* expand space */
  oldsize = web.sizes[e_type];
  if  (oldsize % attr_type_size[attr_type])
    oldsize += attr_type_size[attr_type] - (oldsize % attr_type_size[attr_type]);

  if ( (dim > 0) || (oldsize > web.sizes[e_type]) )
  { newsize =  dim*attr_type_size[attr_type];
     expand(e_type,oldsize + newsize); 
  }
  ex = web.skel[e_type].extras + web.skel[e_type].extra_count;
  strncpy(ex->name,name,ATTR_NAME_SIZE);
  ex->type = attr_type;
  ex->offset = oldsize;
  ex->dim = dim;
  if ( code ) ex->code = *code;
  if ( dumpflag ) ex->flags |= DUMP_ATTR;
  web.skel[e_type].extra_count++;
  return web.skel[e_type].extra_count - 1; /* index */
}


/*************************************************************************
*
*  function: expand_attribute()
*
*  purpose: enlarge space for attribute of an element type
*              or shrink it.
*/

void expand_attribute(e_type,attr_num,newdim)
int e_type; /* VERTEX ,... */
int attr_num; /* number of attribute */
int newdim; /* new number of components */
{ int newsize;
  struct extra *ex;
  int chunksize,offset,available,needed;
  char *spot;
  element_id id;
  int k;

  ex = web.skel[e_type].extras + attr_num;
  if ( ex->dim == newdim ) return;

  if ( newdim <  0 ) 
     kb_error(1310,"Trying to make negative size attribute.\n",RECOVERABLE);


  /* expand space */
  /* see how much extra space is needed */
  if ( attr_num < web.skel[e_type].extra_count-1 )
         available = ex[1].offset - ex->offset;
  else available = web.sizes[e_type] - ex->offset;
  needed = newdim*attr_type_size[ex->type];
  if ( needed > available )
  { /* expand */
     /* check alignment of following fields */
     newsize = needed - available;
     for ( k = attr_num+1 ; k < web.skel[e_type].extra_count ; k++ )
        while ( newsize % attr_type_size[web.skel[e_type].extras[k].type] )
          newsize++;
     expand(e_type,web.sizes[e_type] + newsize); 
     /* move stuff above */
     if ( attr_num < web.skel[e_type].extra_count-1 )
     { offset = web.skel[e_type].extras[attr_num+1].offset;
        chunksize = web.sizes[e_type] - offset - newsize;
        id = NULLID; 
        if ( web.skel[e_type].count > 0 ) 
          while ( generate_all(e_type,&id) )
          { spot = (char*)elptr(id) + offset;
             kb_memmove(spot + newsize,spot,chunksize);
             memset(spot,0,newsize);
          }
        for ( k = attr_num+1 ; k < web.skel[e_type].extra_count ; k++ )
         web.skel[e_type].extras[k].offset += newsize;
     }
  }
  else if ( needed < available )
  { /* maybe shrink */
     /* check alignment of following fields */
     newsize = available - needed;
     for ( k = attr_num+1 ; k < web.skel[e_type].extra_count ; k++ )
        while ( newsize % attr_type_size[web.skel[e_type].extras[k].type] )
          newsize--;
     /* move stuff above */
     if ( attr_num < web.skel[e_type].extra_count-1 )
     { offset = web.skel[e_type].extras[attr_num+1].offset;
        chunksize = web.sizes[e_type] - offset + newsize;
        id = NULLID; 
        if ( web.skel[e_type].count > 0 ) 
          while ( generate_all(e_type,&id) )
          { spot = (char*)elptr(id) + offset;
             kb_memmove(spot - newsize,spot,chunksize);
          }
        for ( k = attr_num+1 ; k < web.skel[e_type].extra_count ; k++ )
         web.skel[e_type].extras[k].offset -= newsize;
     }
     expand(e_type,web.sizes[e_type] - newsize); 
  }
  ex->dim = newdim;
  parallel_update_flag[e_type] = 1;
}

/*************************************************************************
*
*  function: find_attribute()
*
*  purpose: find extra attribute by name, if it exists.
*  return: index number if found, -1 if not.
*/

int find_attribute(etype,name)
int etype;
char *name;
{ struct extra *ex;
  int n;
  ex = web.skel[etype].extras;
  for ( n = 0 ; n < web.skel[etype].extra_count ; n++,ex++ )
    if ( stricmp(ex->name,name) == 0 ) break;
  if ( n == web.skel[etype].extra_count )
     return -1;
  return n;
}

/*************************************************************************
*
*  function: get_extra()
*
*  purpose: return pointer to value of extra attribute of an element.
*/

char *get_extra(id,n)
element_id id;
int n; /* number of extra attribute */
{ int type = id_type(id);

  return ( (char*)elptr(id) + web.skel[type].extras[n].offset);
}

/**************************************************************************
*
* function: find_extra()
*
* purpose: return index of named attribute, searching all element types.
*             return -1 if not found.
*/
int find_extra(name,etype)
char *name;
int *etype; /* for returning element type */
{ int el_type,qnum,n;
  struct extra *ex;

  for ( el_type = VERTEX, qnum = -1 ; el_type <= FACETEDGE ; el_type++ )
     { ex = web.skel[el_type].extras;
        for ( n = 0 ; n < web.skel[el_type].extra_count ; n++,ex++ )
          if ( stricmp(ex->name,name) == 0 )
             {*etype = el_type;qnum = n;break;}
     }
  return qnum;
}

/***********************************************************************
*
* function: get_meth_offset()
*
* purpose: return offset of method instance list in element structure.
*/
int get_meth_offset(type)
int type; /* of element */
{ int meth_offset;
  meth_offset = web.skel[type].extras[web.meth_attr[type]].offset; 
  return meth_offset;
}


/***************************************************************************
  Constraint handling routines
****************************************************************************/

/* Methodology:
    Constraint map is array of conmap_t.
    First entry is number of constraints.
    Then follow constraint numbers, with high bit CON_HIT_BIT set
      if constraint is hit.
    Allocated as an extra attribute if needed.
*/

conmap_t nullcon[2]; /* default empty list */

void set_v_global(v_id)
vertex_id v_id;
{ int k;
  for ( k = 0 ; k < web.con_global_count ; k++ )
     set_v_constraint_map(v_id,web.con_global_map[k]);
}

void set_v_conmap(v_id,map) 
vertex_id v_id;
conmap_t *map;
{ int k, m=(int)map[0];
  for ( k = 1 ; k <= m ; k++ )
    set_v_constraint_map(v_id,map[k]);
  map = get_v_constraint_map(v_id);
  if ( map[0] == 0 ) unset_attr(v_id,CONSTRAINT);
}

void set_e_conmap(e_id,map)  
edge_id e_id;
conmap_t *map;
{ int k, m=(int)map[0];
  for ( k = 1 ; k <= m ; k++ )
    set_e_constraint_map(e_id,map[k]);
  map = get_e_constraint_map(e_id);
  if ( map[0] == 0 ) unset_attr(e_id,CONSTRAINT);
}

void set_f_conmap(f_id,map)  
facet_id f_id;
conmap_t *map;
{ int k, m=(int)map[0];
  for ( k = 1 ; k <= m ; k++ )
    set_f_constraint_map(f_id,map[k]);
  map = get_f_constraint_map(f_id);
  if ( map[0] == 0 ) unset_attr(f_id,CONSTRAINT);
}

void set_v_constraint_map(v_id,n)  
vertex_id v_id;
int n;
{ conmap_t *map;
  int maxcon = web.skel[VERTEX].extras[V_CONSTR_LIST_ATTR].dim;
  struct constraint *constr;
  int k;

  n &= CONMASK;
  if ( maxcon == 0 )
     expand_attribute(VERTEX,V_CONSTR_LIST_ATTR,4);
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
     if ( (map[k] & CONMASK) == ((conmap_t)n & CONMASK) ) return;
  if ( k >= maxcon )
  { expand_attribute(VERTEX,V_CONSTR_LIST_ATTR,maxcon+4);
     map = get_v_constraint_map(v_id);
  } 
  map[k] = (conmap_t)n; 
  map[0]++; /* counter */

  set_attr(v_id,CONSTRAINT);
  constr = get_constraint(n);
  if ( (constr->attr & CON_ENERGY) && (web.representation == STRING) )
  { set_attr(v_id, BDRY_ENERGY);
     if ( everything_quantities_flag )
        apply_method(v_id,METH_INST[constr->energy_method].name);
  }
  if ( (constr->attr & CON_CONTENT) && (web.representation == STRING) )
  { set_attr(v_id, BDRY_CONTENT);
     if ( everything_quantities_flag )
     { apply_method(v_id,METH_INST[constr->content_method_1].name);
        apply_method(v_id,METH_INST[constr->content_method_2].name);
     }
  }
}

void unset_v_constraint_map(v_id,n)    
vertex_id v_id;
int n;
{ conmap_t *map;
  int maxcon = web.skel[VERTEX].extras[V_CONSTR_LIST_ATTR].dim;
  int k,j;

  n &= CONMASK;
  if ( maxcon == 0 ) return;
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
     if ( (map[k] & CONMASK) == (conmap_t)n ) break;
  if ( k > (int)*map ) return;
  map[0]--;
  for ( j = k ; j <= *map ; j++ ) map[j] = map[j+1];
  if ( map[0] == 0 ) unset_attr(v_id,CONSTRAINT);
  if ( everything_quantities_flag )
  { struct constraint *con = get_constraint(n);
     if ( con->attr & CON_ENERGY )
            unapply_method(v_id,con->energy_method);
     if ( con->attr & CON_CONTENT )
         { unapply_method(v_id,con->content_method_1);
            unapply_method(v_id,con->content_method_2);
         }
  }
}

int v_on_constraint(v_id,n)  
vertex_id v_id;
int n;
{ conmap_t *map;
  int maxcon = web.skel[VERTEX].extras[V_CONSTR_LIST_ATTR].dim;
  int k;

  n &= CONMASK;
  if ( maxcon == 0 ) return 0;
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
  { if ( (map[k] & CONMASK) == (conmap_t)n ) 
        return 1;
  }
  return 0;
}

int v_hit_constraint_count(v_id)  
vertex_id v_id;
{ conmap_t *map;
  int maxcon = web.skel[VERTEX].extras[V_CONSTR_LIST_ATTR].dim;
  int count = 0;
  int k;

  if ( maxcon == 0 ) return 0;
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
  { if ( (map[k] & CON_HIT_BIT) ) 
      count++;
  }
  return count;
}

void get_v_common_conmap(v1,v2,conmap)
vertex_id v1,v2;
conmap_t *conmap;
{ conmap_t *map1 = get_v_constraint_map(v1);
  int k;

  conmap[0] = 0;
  for ( k = 1; k <= map1[0] ; k++ )
     if ( v_on_constraint(v2,map1[k]) )
        conmap[++conmap[0]] = map1[k] & (conmap_t)CONMASK;
}

int get_v_constraint_status(v_id,n)  
vertex_id v_id;
int n;
{ conmap_t *map;
  int maxcon = web.skel[VERTEX].extras[V_CONSTR_LIST_ATTR].dim;
  int k;

  n &= CONMASK;
  if ( maxcon == 0 ) return 0;
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
  { if ( (map[k] & CONMASK) == (conmap_t)n ) 
        return (map[k] & CON_HIT_BIT) ? 1 : 0;
  }
  return 0;
}

void clear_v_conmap(v_id)
vertex_id v_id;
{ conmap_t *map = get_v_constraint_map(v_id);
  map[0] = 0;
}

void set_v_constraint_status(v_id,n)  
vertex_id v_id;
int n;
{ conmap_t *map;
  int maxcon = web.skel[VERTEX].extras[V_CONSTR_LIST_ATTR].dim;

  int k;
  n &= CONMASK;
  if ( maxcon == 0 ) return;
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
  { if ( (map[k] & CONMASK) == (conmap_t)n ) 
     { map[k] |= CON_HIT_BIT;
        set_attr(v_id,HIT_WALL);
        return;
     }
  }
}

void unset_v_constraint_status(v_id,n) 
vertex_id v_id;
int n;
{ conmap_t *map;
  int maxcon = web.skel[VERTEX].extras[V_CONSTR_LIST_ATTR].dim;

  int k;
  n &= CONMASK;
  if ( maxcon == 0 ) return;
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
  { if ( (map[k] & CONMASK) == (conmap_t)n ) 
     { map[k] &= ~CON_HIT_BIT;
        return;
     }
  }
}

void clear_v_constraint_status(v_id) 
vertex_id v_id;
{ conmap_t *map;
  int maxcon = web.skel[VERTEX].extras[V_CONSTR_LIST_ATTR].dim;

  int k;
  if ( maxcon == 0 ) return;
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
      map[k] &= ~CON_HIT_BIT;
}

void set_e_constraint_map(e_id,n)  
edge_id e_id;
int n;
{ conmap_t *map;
  int maxcon = web.skel[EDGE].extras[E_CONSTR_LIST_ATTR].dim;
  int k;
  struct constraint *constr;

  n &= CONMASK;
  if ( maxcon == 0 )
     expand_attribute(EDGE,E_CONSTR_LIST_ATTR,4);
  map = get_e_constraint_map(e_id);
  for ( k = 1; k <= (int)*map ; k++ )
     if ( (map[k] & CONMASK) == (conmap_t)n ) return;
  if ( k >= maxcon )
  { expand_attribute(EDGE,E_CONSTR_LIST_ATTR,maxcon+4);
     map = get_e_constraint_map(e_id);
  } 
  map[k] = (conmap_t)n; 
  map[0]++; /* counter */

  set_attr(e_id,CONSTRAINT);
  constr = get_constraint(n);
  if ( constr->attr & CON_ENERGY )
  { set_attr(e_id, BDRY_ENERGY);
     if ( everything_quantities_flag )
        apply_method(e_id,METH_INST[constr->energy_method].name);
  }
  if ( constr->attr & CON_CONTENT )
  { set_attr(e_id, BDRY_CONTENT);  /* BIG PROBLEM HERE GETTING RIGHT BODY!!! */
     if ( everything_quantities_flag )
     { facetedge_id fe = get_edge_fe(e_id);
        if ( valid_id(get_facet_body(get_fe_facet(fe))) )
          apply_method(e_id,METH_INST[constr->content_method_1].name);
        if ( valid_id(get_facet_body(inverse_id(get_fe_facet(fe)))) )
          apply_method(e_id,METH_INST[constr->content_method_2].name);
     }
  }
}

int e_on_constraint(e_id,n)  
edge_id e_id;
int n;
{ conmap_t *map;
  int maxcon = web.skel[EDGE].extras[E_CONSTR_LIST_ATTR].dim;
  int k;

  n &= CONMASK;
  if ( maxcon == 0 ) return 0;
  map = get_e_constraint_map(e_id);
  for ( k = 1; k <= (int)*map ; k++ )
  { if ( (map[k] & CONMASK) == (conmap_t)n ) 
        return 1;
  }
  return 0;
}

void unset_e_constraint_map(e_id,n) 
edge_id e_id;
int n;
{ conmap_t *map;
  int maxcon = web.skel[EDGE].extras[E_CONSTR_LIST_ATTR].dim;
  int j,k;

  n &= CONMASK;
  if ( maxcon == 0 ) return;
  map = get_e_constraint_map(e_id);
  for ( k = 1; k <= (int)*map ; k++ )
    if ( (map[k] & CONMASK) == (conmap_t)n ) break;
  if ( k > *map ) return;

  map[0]--;
  for ( j = k ; j <= (int)*map ; j++ ) map[j] = map[j+1];
  if ( map[0] == 0 ) unset_attr(e_id,CONSTRAINT);

  if ( everything_quantities_flag )
  { struct constraint *con = get_constraint(n);
     if ( con->attr & CON_ENERGY )
            unapply_method(e_id,con->energy_method);
     if ( con->attr & CON_CONTENT )
         { unapply_method(e_id,con->content_method_1);
            unapply_method(e_id,con->content_method_2);
         }
  }

}

void set_f_constraint_map(f_id,n)  
facet_id f_id;
int n;
{ conmap_t *map;
  int maxcon = web.skel[FACET].extras[F_CONSTR_LIST_ATTR].dim;
  int k;

  n &= CONMASK;  /* get rid of hit bit */
  if ( maxcon == 0 )
     expand_attribute(FACET,F_CONSTR_LIST_ATTR,4);
  map = get_f_constraint_map(f_id);
  for ( k = 1; k <= (int)*map ; k++ )
     if ( (map[k] & CONMASK) == (conmap_t)n ) return;
  if ( k >= maxcon )
  { expand_attribute(FACET,F_CONSTR_LIST_ATTR,maxcon+4);
     map = get_f_constraint_map(f_id);
  } 
  map[k] = (conmap_t)n; 
  map[0]++; /* counter */
}

int f_on_constraint(f_id,n)  
facet_id f_id;
int n;
{ conmap_t *map;
  int maxcon = web.skel[FACET].extras[F_CONSTR_LIST_ATTR].dim;
  int k;

  n &= CONMASK;
  if ( maxcon == 0 ) return 0;
  map = get_f_constraint_map(f_id);
  for ( k = 1; k <= (int)*map ; k++ )
  { if ( (map[k] & CONMASK) == (conmap_t)n ) 
        return 1;
  }
  return 0;
}

void unset_f_constraint_map(f_id,n)    
facet_id f_id;
int n;
{ conmap_t *map;
  int maxcon = web.skel[FACET].extras[F_CONSTR_LIST_ATTR].dim;
  int k,j;

  n &= CONMASK;
  if ( maxcon == 0 ) return;
  map = get_f_constraint_map(f_id);
  for ( k = 1; k <= (int)*map ; k++ )
     if ( (map[k] & CONMASK) == (conmap_t)n ) break;
  if ( k > (int)*map ) return;
  map[0]--;
  for ( j = k ; j <= (int)*map ; j++ ) map[j] = map[j+1];
  if ( map[0] == 0 ) unset_attr(f_id,CONSTRAINT);
  if ( everything_quantities_flag )
  { struct constraint *con = get_constraint(n);
     if ( con->attr & CON_ENERGY )
            unapply_method(f_id,con->energy_method);
     if ( con->attr & CON_CONTENT )
         { unapply_method(f_id,con->content_method_1);
            unapply_method(f_id,con->content_method_2);
         }
  }
}

int get_tag(f_id)
facet_id f_id;
{ if ( web.skel[FACET].extras[F_TAG_ATTR].dim > 0 )
     return (*FINT(f_id,F_TAG_ATTR));
  else return 0;
}

void set_tag(f_id,t)
facet_id f_id;
int t;
{ if ( web.skel[FACET].extras[F_TAG_ATTR].dim == 0 )
     expand_attribute(FACET,F_TAG_ATTR,1);
  *FINT(f_id,F_TAG_ATTR) = t;
}

/************************************************************************
*
* function: reset_quant_bits()
*
* purpose: Set flag bits in elements to tell what type methods they have.
*             Also resets global_meth_inst_flags[].
*
*/

void reset_quant_bits()
{ element_id id;
  int etype;
  struct gen_quant *q;
  struct method_instance *mi;
  struct element *e_ptr;
  struct gen_quant_method *gm;
  int k ;

  for ( etype = 0 ; etype < FACETEDGE ; etype++ )
     quant_flags[etype] &= ~(Q_INFO|Q_FIXED|Q_ENERGY);
  for ( k = 0 ; k < meth_inst_count ; k++ )
  { mi = METH_INST + k;
     q = GEN_QUANTS+mi->quant;
     gm = basic_gen_methods + mi->gen_method;
     quant_flags[gm->type] |= q->flags & (Q_ENERGY|Q_FIXED|Q_INFO);
  }
  for ( etype = 0 ; etype < FACETEDGE ; etype++ )
  { int meth_offset = get_meth_offset(etype); 

     global_meth_inst_flags[etype] &= ~(Q_INFO|Q_FIXED|Q_ENERGY);
     for ( k = 0 ; k < global_meth_inst_count[etype] ; k++ )
     { mi = METH_INST + global_meth_inst[etype][k];
        if ( mi->quant < 0 ) continue;
        q = GEN_QUANTS + mi->quant;
        global_meth_inst_flags[etype] |= q->flags&(Q_INFO|Q_FIXED|Q_ENERGY);
     }
     FOR_ALL_ELEMENTS(etype,id)
     { e_ptr = elptr(id);
        e_ptr->qflags &= ~(Q_INFO|Q_FIXED|Q_ENERGY);
        for ( k = 0 ; k < (int)e_ptr->method_count ; k++ )
        { mi = METH_INST + abs(((int*)((char*)e_ptr+meth_offset))[k]);
          if ( mi->quant < 0 ) continue;
          q = GEN_QUANTS + mi->quant;
          e_ptr->qflags |= q->flags&(Q_INFO|Q_FIXED|Q_ENERGY);
        }
     }
  }
}

     

  
