/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/**********************************************************************
*
*  File: gauss.c
*
*  Purpose: Does calculations needed for including gauss curvature
*              integral named method. Linear model only.
*              Implemented as total angle deficit of boundary vertices.
*              Uses general quantity interface.
*/

#include "include.h"

/*********************************************************************
*
* function: gauss_integral_init()
*
* purpose:  Check illegalities
*
*/

void gauss_integral_init(mode,mi)
int mode;
struct method_instance *mi;
{
  if ( web.dimension != 2 )
     kb_error(1590,"gauss_integral method only for 2D facets.\n",RECOVERABLE);

  if ( SDIM != 3 )
     kb_error(1591,"gauss_integral method only for 3D space.\n",RECOVERABLE);

  if ( web.modeltype != LINEAR )
     kb_error(1592,"gauss_integral method only for LINEAR model.\n",RECOVERABLE);

}

/********************************************************************
*
* function: gauss_int_energy()
*
* purpose: single facet contribution to total angle of boundary
*             vertices.
*
*/

REAL gauss_int_energy(f_info)
struct qinfo *f_info;
{ REAL s1[MAXCOORD],s2[MAXCOORD];
  facetedge_id fe;
  vertex_id v_id;
  int k;
  REAL energy = 0.0;
  REAL a;

  fe = get_facet_fe(f_info->id);
  for ( k = 0 ; k < FACET_VERTS ; k++, fe = get_next_edge(fe) )
  {
     if ( get_eattr(get_fe_edge(fe)) & (BOUNDARY|FIXED) )
        energy -= M_PI;  /* normalization */
     v_id = get_fe_tailv(fe);
     if ( !(get_vattr(v_id) & (BOUNDARY|FIXED) ) ) continue;
     get_fe_side(fe,s1);
     get_fe_side(get_prev_edge(fe),s2);
     a = -SDIM_dot(s1,s2)/sqrt(SDIM_dot(s1,s1)*SDIM_dot(s2,s2));
     energy += acos(a);
  }
  return energy;
}


/********************************************************************
*
* function: gauss_int_gradient()
*
* purpose: single facet contribution to gradient of  total angle of 
*             boundary vertices.
*
*/

REAL gauss_int_gradient(f_info)
struct qinfo *f_info;
{ REAL s1[MAXCOORD],s2[MAXCOORD];
  facetedge_id fe;
  vertex_id v_id;
  int j,k;
  REAL energy = 0.0;
  REAL ss1,ss2,ss12;
  REAL a,b;

  fe = get_facet_fe(f_info->id);
  for ( k = 0 ; k < FACET_VERTS ; k++ )
     memset((char*)(f_info->grad[k]),0,SDIM*sizeof(REAL));
  for ( k = 0 ; k < FACET_VERTS ; k++, fe = get_next_edge(fe) )
  {
     if ( get_eattr(get_fe_edge(fe)) & (BOUNDARY|FIXED) )
        energy -= M_PI;  /* normalization */
     v_id = get_fe_tailv(fe);
     if ( !(get_vattr(v_id) & (BOUNDARY|FIXED) ) ) continue; 
     get_fe_side(fe,s1);
     get_fe_side(get_prev_edge(fe),s2);
     ss1 = SDIM_dot(s1,s1);
     ss2 = SDIM_dot(s2,s2);
     ss12 = SDIM_dot(s1,s2);
     a = -ss12/sqrt(ss1*ss2);
     energy += acos(a);
     b = sqrt(1-a*a)*ss1*sqrt(ss1*ss2);
     for ( j = 0 ; j < SDIM ; j++ )
        f_info->grad[(k+1)%3][j] += (ss1*s2[j] - ss12*s1[j])/b;
     b = sqrt(1-a*a)*ss2*sqrt(ss1*ss2);
     for ( j = 0 ; j < SDIM ; j++ )
        f_info->grad[(k+2)%3][j] += -(ss2*s1[j] - ss12*s2[j])/b;

  }
  return energy;
}


/********************************************************************
*
*  Function: sqgauss_energy()
*
*  Purpose:  Does square gauss curvature energy calculation for vertices.
*
*/

void sqgauss_energy()
{ 
  REAL modulus = globals[sqgauss_param].value.real;
  vertex_id v_id,v[3];
  edge_id e_id;
  facet_id f_id;
  facetedge_id fe;
  int fixcount;
  int i;
  REAL area;
  REAL side[3][MAXCOORD];
  REAL ss[3];
  REAL st[3];
  REAL angle;
  REAL c;
  REAL gc;  /* gaussian curvarture */
  struct gvert { REAL angle;
                 REAL area;
                 REAL star_area;
                } *gverts,*gv;

  gverts = (struct gvert*)temp_calloc(web.skel[VERTEX].max_ord+1,
                                            sizeof(struct gvert));

  /* accumulate angles around vertices */
  FOR_ALL_FACETS(f_id)
    { fe = get_facet_fe(f_id);
      fixcount = 0;
      for ( i = 0; i < FACET_VERTS ; i++,fe=get_next_edge(fe) )
        { e_id = get_fe_edge(fe);
          get_edge_side(e_id,side[i]);
          v[i] = get_edge_tailv(e_id);
          if ( get_vattr(v[i]) & (FIXED|BOUNDARY) ) fixcount++;
        }
        for ( i = 0 ; i < FACET_VERTS ; i++ )
         { ss[i] = SDIM_dot(side[i],side[i]);
            st[i] = SDIM_dot(side[i],side[(i+2)%3]);
         }
        area = 0.5*sqrt(ss[0]*ss[1]-st[1]*st[1]);
        for ( i = 0 ; i < FACET_VERTS ; i++ )
        { c = -st[i]/sqrt(ss[i]*ss[(i+2)%3]);
          angle = acos(c);
          gv = gverts + ordinal(v[i]);
          gv->angle += angle;
          gv->area  += area/3;
          gv->star_area  += area/(3-fixcount);
        }
    }
  
  /* calc square gauss curvature at each vertex */
  FOR_ALL_VERTICES(v_id)
    { struct gvert *vg = gverts + ordinal(v_id);
      if ( get_vattr(v_id) & (FIXED|BOUNDARY) ) continue;
      gc = (wedge_angle(v_id) - vg->angle)/vg->area;
      web.total_energy += modulus*gc*gc*vg->star_area;
    }


  temp_free((char*)gverts);
}

/*************************************************************************
*
* function: wedge_angle()
*
* purpose: find topological angle around vertex:
*             2*pi for nonconstrained vertex
*             pi for one constraint
*             calculate angle between 2 constraints
*/

REAL wedge_angle(v_id)
vertex_id v_id;
{ int concount;
  conmap_t *conmap = get_v_constraint_map(v_id);
  REAL normal[2][MAXCOORD];
  REAL c;
  int j;
  struct constraint *con;
  REAL fval;

  concount = conmap[0];

  if ( concount == 0 ) return 2*M_PI;
  if ( concount == 1 ) return M_PI;

  /* two constraints, so find angle between */

  if ( concount > 2 ) 
  { sprintf(errmsg,"gauss_curvature_integral: More than two constraints on vertex %d.\n",
          ordinal(v_id)+1);
     kb_error(1593,errmsg,RECOVERABLE);
  }
  for ( j = 1, concount = 0 ; j <= (int)conmap[0] ; j++ ) 
     { 
        con = get_constraint(conmap[j]);
        eval_all(con->formula,get_coord(v_id),SDIM,&fval,normal[concount],v_id); 
        concount++;
     }
  c = SDIM_dot(normal[0],normal[1])
        /sqrt(SDIM_dot(normal[0],normal[0])) 
        /sqrt(SDIM_dot(normal[1],normal[1]));
  return M_PI - acos(c);
}


/********************************************************************
*
*  Function: sqgauss_force()
*
*  Purpose:  Does square gauss curvature force calculation for vertices.
*
*/

void sqgauss_force()
{ 
  REAL modulus = globals[sqgauss_param].value.real;
  vertex_id v_id,v[3];
  edge_id e_id,e[3];
  facet_id f_id;
  facetedge_id fe;
  int fixcount;
  REAL side[3][MAXCOORD];
  REAL ss[3];
  REAL st[3];
  int j,i;
  REAL area,angle,c,c1,c2;
  REAL *f;
  REAL gc;  /* gaussian curvarture */
  struct gvert { REAL angle;
                 REAL angle_grad[MAXCOORD];
                 REAL area;
                 REAL area_grad[MAXCOORD];
                 REAL star_area;
                 REAL star_area_grad[MAXCOORD];
                 REAL gc;  /* gaussian curvature */
                 } *gverts;
  struct gedge { REAL area_grad[2][MAXCOORD];
                 REAL star_area_grad[2][MAXCOORD];
                 REAL angle_grad[2][MAXCOORD];
                 /* dhead/dtail, dtail/dhead */
               } *gedges;
  gverts = (struct gvert*)temp_calloc(web.skel[VERTEX].max_ord+1,
                                            sizeof(struct gvert));
  gedges = (struct gedge*)temp_calloc(web.skel[EDGE].max_ord+1,
                                            sizeof(struct gedge));

  /* accumulate angles around vertices */
  FOR_ALL_FACETS(f_id)
    { fe = get_facet_fe(f_id);
      fixcount = 0;
      for ( i = 0; i < FACET_VERTS ; i++,fe=get_next_edge(fe) )
        { e[i] = get_fe_edge(fe);
          get_edge_side(e[i],side[i]);
          v[i] = get_edge_tailv(e[i]);
          if ( get_vattr(v[i]) & (FIXED|BOUNDARY) ) fixcount++;
        }
        for ( i = 0 ; i < FACET_VERTS ; i++ )
         { ss[i] = SDIM_dot(side[i],side[i]);
           st[i] = SDIM_dot(side[i],side[(i+2)%3]);
         }
        area = 0.5*sqrt(ss[0]*ss[1]-st[1]*st[1]);
        for ( i = 0 ; i < FACET_VERTS ; i++ )
        { int i1 = (i+1)%3;
          int i2 = (i+2)%3;
          struct gedge *ge1,*ge2;
          struct gvert *gv;
          int jj1,jj2;

          gv = gverts + ordinal(v[i]);
          ge1 = gedges + ordinal(e[i]);
          ge2 = gedges + ordinal(e[i2]);
          jj1 =  inverted(e[i]) ? 1 : 0;
          jj2 =  inverted(e[i2]) ? 0 : 1;

          c = -st[i]/sqrt(ss[i]*ss[i2]);
          angle = acos(c);
          gv->angle += angle;
          c1 = (1+st[i]/ss[i])/area/2;
          c2 = (1+st[i]/ss[i2])/area/2;
          for ( j = 0 ; j < SDIM ; j++ )
             gv->angle_grad[j] += c1*side[i][j] - c2*side[i2][j];
          c1 = st[i]/ss[i]/area/2;
          c2 = 1/area/2;
          for ( j = 0 ; j < SDIM ; j++ )
             ge1->angle_grad[jj1][j] -= c1*side[i][j] - c2*side[i2][j];
          c1 = 1/area/2;
          c2 = st[i]/ss[i2]/area/2;
          for ( j = 0 ; j < SDIM ; j++ )
             ge2->angle_grad[jj2][j] -= c1*side[i][j] - c2*side[i2][j];
          gv->area  += area/3;
          gv->star_area  += area/(3-fixcount);
          c1 = ss[i1]/area/4;
          c2 = st[i1]/area/4;
          for ( j = 0 ; j < SDIM ; j++ )
            { c = c2*side[i1][j] - c1*side[i][j];
              gv->area_grad[j] += c/3;
              gv->star_area_grad[j] += c/(3-fixcount);
              ge1->area_grad[jj1][j] += c/3;
              ge2->area_grad[jj2][j] += c/3;
              ge1->star_area_grad[jj1][j] += c/(3-fixcount);
              ge2->star_area_grad[jj2][j] += c/(3-fixcount);
            }
        }
    }
  
  /* calc square gauss curvature at each vertex */
  FOR_ALL_VERTICES(v_id)
    { struct gvert *gv = gverts + ordinal(v_id);
      if ( get_vattr(v_id) & (FIXED|BOUNDARY) ) continue;
      gc = (wedge_angle(v_id) - gv->angle)/gv->area;
      web.total_energy += modulus*gc*gc*gv->star_area;
      gv->gc = gc;

      /* now self terms in derivative */
      f = get_force(v_id);
      for ( j = 0 ; j < SDIM ; j++ )
         f[j] -= (2*gc*gv->star_area*(-gv->angle_grad[j]/gv->area 
                         - gc*gv->area_grad[j]/gv->area)
                         + gc*gc*gv->star_area_grad[j])*modulus;
    }
  
  /* now cross terms from edges */
  FOR_ALL_EDGES(e_id)
    { struct gvert *gv1 = gverts + ordinal(get_edge_tailv(e_id));
      struct gvert *gv2 = gverts + ordinal(get_edge_headv(e_id));
      struct gedge *ge = gedges + ordinal(e_id);

      /* head energy as function of tail */
      if ( !(get_vattr(get_edge_headv(e_id)) & (FIXED|BOUNDARY) ) )
        { f = get_force(get_edge_tailv(e_id));
          for ( j = 0 ; j < SDIM ; j++ )
             f[j] -= (2*gv2->gc*gv2->star_area*(-ge->angle_grad[0][j]/gv2->area
                        - gv2->gc*ge->area_grad[0][j]/gv2->area)
                        + gv2->gc*gv2->gc*ge->star_area_grad[0][j])*modulus;
        }
      /* tail energy as function of head */
      if ( !(get_vattr(get_edge_tailv(e_id)) & (FIXED|BOUNDARY) ) )
        { f = get_force(get_edge_headv(e_id));
          for ( j = 0 ; j < SDIM ; j++ )
             f[j] -= (2*gv1->gc*gv1->star_area*(-ge->angle_grad[1][j]/gv1->area
                        - gv1->gc*ge->area_grad[1][j]/gv1->area)
                        + gv1->gc*gv1->gc*ge->star_area_grad[1][j])*modulus;
        }
    }

  temp_free((char*)gverts);
  temp_free((char*)gedges);
}


/********************************************************************

                     sqgauss integral as method

*********************************************************************/

static  struct gvert { REAL angle;
                       REAL angle_grad[MAXCOORD];
                       REAL area;
                       REAL area_grad[MAXCOORD];
                       REAL star_area;
                       REAL star_area_grad[MAXCOORD];
                       REAL gc;  /* gaussian curvature */
                     } *gverts,*gv;
static  struct gedge { REAL area_grad[2][MAXCOORD];
                       REAL star_area_grad[2][MAXCOORD];
                       REAL angle_grad[2][MAXCOORD];
                       /* dhead/dtail, dtail/dhead */
                     } *gedges;

/*************************************************************************
*
* function: sqgauss_method_init()
*
* purpose: gather data for sqgauss method.
*             Should really be replaced by local gathering.
*/

void sqgauss_method_init(mode,mi)
int mode;
struct method_instance *mi;
{
  vertex_id v[3];
  edge_id e_id,e[3];
  facet_id f_id;
  facetedge_id fe;
  int fixcount;
  REAL side[3][MAXCOORD];
  REAL ss[3];
  REAL st[3];
  int j,i;
  REAL area,angle,c,c1,c2;

  if ( vedge_timestamp < top_timestamp ) make_vedge_lists();

  if ( gverts ) myfree((char*)gverts);
  if ( gedges ) myfree((char*)gedges);

  gverts = (struct gvert*)mycalloc(web.skel[VERTEX].max_ord+1,
                                            sizeof(struct gvert));

  /* accumulate angles around vertices */
  if ( mode == METHOD_VALUE )
  FOR_ALL_FACETS(f_id)
    { fe = get_facet_fe(f_id);
      fixcount = 0;
      for ( i = 0; i < FACET_VERTS ; i++,fe=get_next_edge(fe) )
        { e_id = get_fe_edge(fe);
          get_edge_side(e_id,side[i]);
          v[i] = get_edge_tailv(e_id);
          if ( get_vattr(v[i]) & (FIXED|BOUNDARY) ) fixcount++;
        }
        for ( i = 0 ; i < FACET_VERTS ; i++ )
         { ss[i] = SDIM_dot(side[i],side[i]);
           st[i] = SDIM_dot(side[i],side[(i+2)%3]);
         }
        area = 0.5*sqrt(ss[0]*ss[1]-st[1]*st[1]);
        for ( i = 0 ; i < FACET_VERTS ; i++ )
        { c = -st[i]/sqrt(ss[i]*ss[(i+2)%3]);
          angle = acos(c);
          gv = gverts + ordinal(v[i]);
          gv->angle += angle;
          gv->area  += area/3;
          gv->star_area  += area/(3-fixcount);
        }
    }
  else if ( mode == METHOD_GRADIENT )
  {
    gedges = (struct gedge*)mycalloc(web.skel[EDGE].max_ord+1,
                                            sizeof(struct gedge));

    /* accumulate angles around vertices */
    FOR_ALL_FACETS(f_id)
    { fe = get_facet_fe(f_id);
      fixcount = 0;
      for ( i = 0; i < FACET_VERTS ; i++,fe=get_next_edge(fe) )
        { e[i] = get_fe_edge(fe);
          get_edge_side(e[i],side[i]);
          v[i] = get_edge_tailv(e[i]);
          if ( get_vattr(v[i]) & (FIXED|BOUNDARY) ) fixcount++;
        }
        for ( i = 0 ; i < FACET_VERTS ; i++ )
         { ss[i] = SDIM_dot(side[i],side[i]);
            st[i] = SDIM_dot(side[i],side[(i+2)%3]);
         }
        area = 0.5*sqrt(ss[0]*ss[1]-st[1]*st[1]);
        for ( i = 0 ; i < FACET_VERTS ; i++ )
        { int i1 = (i+1)%3;
          int i2 = (i+2)%3;
          struct gedge *ge1,*ge2;
          int jj1,j2;

          gv = gverts + ordinal(v[i]);
          ge1 = gedges + ordinal(e[i]);
          ge2 = gedges + ordinal(e[i2]);
          jj1 =  inverted(e[i]) ? 1 : 0;
          j2 =  inverted(e[i2]) ? 0 : 1;

          c = -st[i]/sqrt(ss[i]*ss[i2]);
          angle = acos(c);
          gv->angle += angle;
          c1 = (1+st[i]/ss[i])/area/2;
          c2 = (1+st[i]/ss[i2])/area/2;
          for ( j = 0 ; j < SDIM ; j++ )
             gv->angle_grad[j] += c1*side[i][j] - c2*side[i2][j];
          c1 = st[i]/ss[i]/area/2;
          c2 = 1/area/2;
          for ( j = 0 ; j < SDIM ; j++ )
             ge1->angle_grad[jj1][j] -= c1*side[i][j] - c2*side[i2][j];
          c1 = 1/area/2;
          c2 = st[i]/ss[i2]/area/2;
          for ( j = 0 ; j < SDIM ; j++ )
             ge2->angle_grad[j2][j] -= c1*side[i][j] - c2*side[i2][j];
          gv->area  += area/3;
          gv->star_area  += area/(3-fixcount);
          c1 = ss[i1]/area/4;
          c2 = st[i1]/area/4;
          for ( j = 0 ; j < SDIM ; j++ )
            { c = c2*side[i1][j] - c1*side[i][j];
              gv->area_grad[j] += c/3;
              gv->star_area_grad[j] += c/(3-fixcount);
              ge1->area_grad[jj1][j] += c/3;
              ge2->area_grad[j2][j] += c/3;
              ge1->star_area_grad[jj1][j] += c/(3-fixcount);
              ge2->star_area_grad[j2][j] += c/(3-fixcount);
            }
        }
    }
  } 
}

/********************************************************************
*
*  Function: sqgauss_method_value()
*
*  Purpose:  Does square gauss curvature energy calculation for vertices.
*
*/

REAL sqgauss_method_value(v_info)
struct qinfo *v_info;
{ 
  REAL modulus = sqgauss_flag ? globals[sqgauss_param].value.real : 0.0;
  REAL gc;  /* gaussian curvarture */

  /* calc square gauss curvature at each vertex */
  struct gvert *vg = gverts + ordinal(v_info->id);
  if ( get_vattr(v_info->id) & (FIXED|BOUNDARY) ) return 0.0;
  gc = (wedge_angle(v_info->id) - vg->angle)/vg->area;
  return modulus*gc*gc*vg->star_area;
}


/********************************************************************
*
*  Function: sqgauss_method_grad()
*
*  Purpose:  Does square gauss curvature force calculation for vertices.
*
*/

REAL sqgauss_method_grad(v_info)
struct qinfo *v_info;
{ 
  REAL modulus = sqgauss_flag ? globals[sqgauss_param].value.real : 0.0;
  vertex_id v_id;
  edge_id e_id,start_e;
  int j;
  REAL gc;  /* gaussian curvarture */
  REAL energy = 0.0;

  /* calc square gauss curvature at each vertex */
  v_id = v_info->id;
  gv = gverts + ordinal(v_id);
  if ( get_vattr(v_id) & (FIXED|BOUNDARY) ) return 0.0;
  gc = (wedge_angle(v_id) - gv->angle)/gv->area;
  energy += modulus*gc*gc*gv->star_area;
  gv->gc = gc;

  /* now self terms in derivative */
  for ( j = 0 ; j < SDIM ; j++ )
     v_info->grad[0][j] += (2*gc*gv->star_area*(-gv->angle_grad[j]/gv->area 
                         - gc*gv->area_grad[j]/gv->area)
                         + gc*gc*gv->star_area_grad[j])*modulus;
  
  /* now cross terms from edges */
  start_e = e_id = get_vertex_edge(v_id);
  if ( !(get_vattr(v_id) & (FIXED|BOUNDARY) ) )
    do
    { 
      struct gvert *gv2 = gverts + ordinal(get_edge_headv(e_id));
      struct gedge *ge = gedges + ordinal(e_id);

      /* head energy as function of tail */
        for ( j = 0 ; j < SDIM ; j++ )
             v_info->grad[0][j] += 
                    (2*gv2->gc*gv2->star_area*(-ge->angle_grad[0][j]/gv2->area
                        - gv2->gc*ge->area_grad[0][j]/gv2->area)
                        + gv2->gc*gv2->gc*ge->star_area_grad[0][j])*modulus;

      e_id = get_next_tail_edge(e_id);
    } while ( !equal_element(start_e,e_id) );

 return energy;
}

/*****************************************************************************

          Gaussian curvature at vertices (star model)

******************************************************************************/

/*****************************************************************************

          Square Gaussian curvature integral at vertices (star model)

******************************************************************************/


void star_sqgauss_method_init(mode,mi)
int mode;
struct method_instance *mi;
{
}

REAL star_sqgauss_method_value(v_info)
struct qinfo *v_info;
{ REAL deficit = 2*M_PI;
  REAL area = 0.0;
  int k;

  if ( v_info->vcount <= 1 ) return 0.0;

  for ( k = 1 ; k < v_info->vcount ; k++ )
  { REAL ss1,ss2,s1s2;
    int kk = (k==(v_info->vcount-1)) ? 0 : k;
    ss1 = SDIM_dot(v_info->sides[0][k-1],v_info->sides[0][k-1]);
    ss2 = SDIM_dot(v_info->sides[0][kk],v_info->sides[0][kk]);
    s1s2 = SDIM_dot(v_info->sides[0][k-1],v_info->sides[0][kk]);
    deficit -= acos(s1s2/sqrt(ss1*ss2));
    area += sqrt(ss1*ss2 - s1s2*s1s2)/6;
  }
  return deficit*deficit/area;
}

REAL star_sqgauss_method_grad(v_info)
struct qinfo *v_info;
{ REAL deficit = 2*M_PI;
  REAL area = 0.0;
  int k,i;

  if ( v_info->vcount <= 1 ) return 0.0;

  for ( k = 1 ; k < v_info->vcount ; k++ )
  { REAL ss1,ss2,s1s2;
    int kk = (k==(v_info->vcount-1)) ? 0 : k;
    ss1 = SDIM_dot(v_info->sides[0][k-1],v_info->sides[0][k-1]);
    ss2 = SDIM_dot(v_info->sides[0][kk],v_info->sides[0][kk]);
    s1s2 = SDIM_dot(v_info->sides[0][k-1],v_info->sides[0][kk]);
    deficit -= acos(s1s2/sqrt(ss1*ss2));
    area += sqrt(ss1*ss2 - s1s2*s1s2)/6;
  }

  /* now individual vertex gradients */
  for ( k = 1 ; k < v_info->vcount ; k++ )
  {  REAL ss1,ss2,s1s2;
     int kk = (k==(v_info->vcount-1)) ? 0 : k;
     REAL *s1 = v_info->sides[0][k-1],*s2 = v_info->sides[0][kk];
     REAL denom;

     ss1 = SDIM_dot(v_info->sides[0][k-1],v_info->sides[0][k-1]);
     ss2 = SDIM_dot(v_info->sides[0][kk],v_info->sides[0][kk]);
     s1s2 = SDIM_dot(v_info->sides[0][k-1],v_info->sides[0][kk]);
     
     denom = sqrt(ss1*ss2 - s1s2*s1s2);
     for ( i = 0 ; i < SDIM ; i++ )
     { REAL ddefds1,ddefds2,dads1,dads2,dsqds1,dsqds2;
        ddefds1 = (s2[i] - s1s2/ss1*s1[i])/denom;
        ddefds2 = (s1[i] - s1s2/ss2*s2[i])/denom;
        dads1 = (s1[i]*ss2 - s1s2*s2[i])/denom/6;
        dads2 = (s2[i]*ss1 - s1s2*s1[i])/denom/6;
        dsqds1 = 2*deficit*ddefds1/area - deficit*deficit/area/area*dads1;
        dsqds2 = 2*deficit*ddefds2/area - deficit*deficit/area/area*dads2;
        v_info->grad[k][i] += dsqds1;
        v_info->grad[0][i] -= dsqds1;
        v_info->grad[kk?k+1:1][i] += dsqds2;
        v_info->grad[0][i] -= dsqds2;
     }
  }

  return deficit*deficit/area;
}


