/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/************************************************************************/  
/*    File: bk.c                                                        */
/*                                                                      */
/*     Bunch-Kaufmann factoring                                         */
/*     and general linear system mongering                              */
/************************************************************************/  

#include "include.h"

/**************************************************************************/

#define RSWAP(a,b) {REAL rtmp = a; a = b ; b = rtmp; }
#define RPTRSWAP(a,b) {REAL *rtmp = a; a = b ; b = rtmp; }
#define ISWAP(a,b) {int rtmp = a; a = b ; b = rtmp; }

/*******************************************************************
* 
* function: bk_mul()
*
* purpose: multiply vector by original sparse matrix
*/
void bk_mul(S,w,v)
struct linsys  *S;
REAL *w; /* in */ 
REAL *v; /* out */
{ int row,col,i;
  memset((char*)v,0,S->N*sizeof(REAL));
  for ( row = 0 ; row < S->N ; row++ )
  { for ( i = S->IA[row]-A_OFF ; i < S->IA[row+1]-A_OFF ; i++ )
    { col = S->JA[i] - A_OFF;
      v[row] += S->A[i]*w[col];
      if ( col != row ) v[col] += S->A[i]*w[row];
    }
  }
  if ( (S->CN > 0) && S->CHinvCinv )
  { /* project back */
     REAL *tempv = (REAL*)temp_calloc(2*S->N,sizeof(REAL));
     REAL *tempvv = tempv + S->N;
     matvec_mul(S->C,v,tempv,S->CN,S->N);
     matvec_mul(S->CHinvCinv,tempv,tempvv,S->CN,S->CN);
     vec_mat_mul(tempvv,S->HinvC,tempv,S->CN,S->N);
     for ( i = 0 ; i < S->N ; i++ ) v[i] -= tempv[i];
     temp_free((char*)tempv);
  }
} /* end bk_mul() */


/*******************************************************************
* 
* function: free_system()
*
* purpose: free all memory allocated to linear system structure.
*/

void free_system(S)
struct linsys *S;
{ int i;
  struct BKrow *ri;

  if ( S->IA ) myfree((char*)S->IA);
  if ( S->JA && !(S->flags & S_JA_INCLUDED) ) myfree((char*)S->JA);
  if ( S->A ) myfree((char*)S->A);
  if ( S->pIA ) myfree((char*)S->pIA);
  if ( S->pJA ) myfree((char*)S->pJA);
  if ( S->pA ) myfree((char*)S->pA);
  if ( S->P ) myfree((char*)S->P);
  if ( S->IP ) myfree((char*)S->IP);
  if ( S->ISP ) myfree((char*)S->ISP);
  if ( S->psize ) myfree((char*)S->psize);
  if ( S->rowhead )
  { myfree((char*)S->rowhead);
     for ( i = 0, ri = S->rowhead ; i < S->N ; i++,ri++ )
    myfree((char*)(ri->entry + ri->start));
  }
  if ( S->LIA ) myfree((char*)S->LIA);
  if ( S->LJA ) myfree((char*)S->LJA);
  if ( S->LIJA ) myfree((char*)S->LIJA);
  if ( S->LA ) myfree((char*)S->LA);
  if ( S->coninx ) myfree((char*)S->coninx);
  if ( S->coninxinv ) myfree((char*)S->coninxinv);
  if ( S->apinv ) myfree((char*)S->apinv);
  if ( S->HinvC ) free_matrix(S->HinvC);
  if ( S->CHinvCinv ) free_matrix(S->CHinvCinv);
  for ( i = 0 ; i < S->CN ; i++ ) myfree((char*)(S->C[i]));
  if ( S->C ) myfree((char*)S->C);
  if ( S->stree )
  { int k;
    for (  k = 0 ; k <= S->streemax ; k++ )
    { if ( S->stree[k].u.info.mat) 
      { myfree((char*)S->stree[k].u.info.mat);
        myfree((char*)S->stree[k].u.info.vlist);
      }
    }
  }

  memset((char*)S,0,sizeof(struct linsys));
} /* end free_system() */

/************************************************************************
*
*  function: find_ordering()
*
*  purpose: find order of vertex degree of freedom variables.
*      Currently in x coordinate order. Not very useful.
*  in:  array    Hessian data
*          n     total degrees
*  out:    P     permutation vector, P[0] is first variable to do.
*/

struct xsorting { int inx;  /* original row number */
         REAL x; /* x value of vertex */
         int ord;  /* ordinal of vertex */
         int deg;  /* degree of freedom */
         };

/* comparison function for sorting */
int xcomp(a,b)
struct xsorting *a,*b;
{ if ( a->x < b->x ) return -1;
  if ( a->x > b->x ) return  1;
  if ( a->ord < b->ord ) return -1;
  if ( a->ord > b->ord ) return  1;
  if ( a->deg < b->deg ) return -1;
  if ( a->deg > b->deg ) return  1;
  return 0;
} /* end xcomp() */

void find_ordering(verlist,n,P)
struct hess_verlist *verlist;     /* pointers to rows */
int n;        /* size of system */
int *P;       /* preallocated for return of permutation */
{
  int i,j;
  struct xsorting *xlist,*Bptr;
  vertex_id v_id;

  /* sort vertices in x order */
  xlist = (struct xsorting *)temp_calloc(n,sizeof(struct xsorting));
  for ( i = 0, Bptr = xlist ; i < n ; i++,Bptr++ ) Bptr->inx = i;
  FOR_ALL_VERTICES(v_id)
  { struct hess_verlist *v;      /* current  vertex */
    v = verlist + ordinal(v_id);
    Bptr = xlist + v->rownum;
    for ( j = 0 ; j < v->freedom ; j++, Bptr++ )
    {
      Bptr->ord = ordinal(v->v_id);
      Bptr->deg = j;
      Bptr->x = get_coord(v->v_id)[0];
    }
  }
  qsort((char*)xlist,n,sizeof(struct xsorting),FCAST xcomp);
  for ( i = 0 ; i < n ; i++ ) 
  { P[i] = xlist[i].inx + A_OFF;
  }
  temp_free((char*)xlist);
} /* end find_ordering() */

/**********************************************************************
*
* function: bk_AIJ_setup()
*
* purpose: convert raw Hessian data to standard sparse format
*
*/

void bk_AIJ_setup(harray,N,S)
struct hess_index *harray;     /* pointers to rows */
int N;        /* size of system */
struct linsys *S;  /* pointer to empty structure */
{
  int i,j,n;
  int  total = 0;
  struct hess_entry *e;
  int spot;
  int *ptr,*cptr,*dptr;
  int isize,dsize;

  S->N = N;
  S->flags &= ~ S_ODRV_REORDERED; 
  S->flags |= S_JA_INCLUDED;

  for ( i = 0 ; i < N ; i++ ) total += harray[i].count;

  /* incoming list of hess_entry structs is compacted in place */
  isize = sizeof(int);
  dsize = sizeof(REAL);
  /* delete row numbers */
  e = hashtable;
  ptr = (int*)e;  
  if ( 2*isize == dsize )
   for ( n = 0; n < total; n++,e++,ptr+=3 )
   { ptr[0] = ((int*)e)[0];
     ptr[1] = ((int*)e)[1];
     ptr[2] = e->col;
   }
  else if ( isize == dsize )
   for ( n = 0; n < total; n++,e++,ptr+=2 )
   { ptr[0] = ((int*)e)[0];
     ptr[1] = e->col;
   }
  else if ( 4*isize == dsize )
   for ( n = 0; n < total; n++,e++,ptr+=5 )
   { ptr[0] = ((int*)e)[0];
     ptr[1] = ((int*)e)[1];
     ptr[2] = ((int*)e)[2];
     ptr[3] = ((int*)e)[3];
     ptr[4] = e->col;
   }
  else  /* i.e. 10-byte long double */
  { char *p = (char*)ptr;
    for ( n = 0; n < total; n++,e++,p+=dsize+isize )
     { *(REAL *)p = e->value;
       *(int *)(p+dsize)  = e->col;
     }
    ptr = (int *)p;
  }
  /* now extract cols to room at the end */
  S->JA = cptr = ptr;
  S->A = (REAL*)(hashtable);
  hashtable = NULL;
  ptr = (int*)(S->A);
  dptr = ptr;
  if ( 2*isize == dsize )
   for ( n = 0 ; n < total; n++,dptr+=3,ptr+=2,cptr++)
     { ptr[0] = dptr[0];
       ptr[1] = dptr[1];
       *cptr =  dptr[2]+A_OFF;
     }
  else if ( isize == dsize )
     for ( n = 0 ; n < total; n++,dptr+=2,ptr+=1,cptr++)
     { ptr[0] = dptr[0];
       *cptr =  dptr[1]+A_OFF;
     }
  else if ( 4*isize == dsize )
     for ( n = 0 ; n < total; n++,dptr+=5,ptr+=4,cptr++)
     { ptr[0] = dptr[0];
       ptr[1] = dptr[1];
       ptr[2] = dptr[2];
       ptr[3] = dptr[3];
       *cptr =  dptr[4]+A_OFF;
     }
  else
  { char *p = (char *)ptr;
     REAL *d = (REAL *)ptr;
     for ( n = 0 ; n < total; n++,p+=isize+dsize,d++,cptr++ )
     { *d = *(REAL*)p;
       *cptr = *(int*)(p+dsize)+A_OFF; 
     }
  }

  /* move JA down to end of A */
  kb_memmove((char*)(S->A+total),(char*)(S->JA),total*sizeof(int));

  /* free up leftover room */
  S->A = (REAL*)kb_realloc((char*)(S->A),total*(sizeof(REAL)+sizeof(int)),
    table_size*sizeof(struct hess_entry));
  S->JA = (int*)(S->A+total);

  /* allocate other bits */
  S->IA = (int *)mycalloc(N+1,sizeof(int));
  if ( S->P == NULL ) S->P = (int *)mycalloc(N,sizeof(int));
  if ( S->IP == NULL ) S->IP = (int *)mycalloc(N,sizeof(int));

  for ( i = 0, spot = 0 ; i < N ; i++ )
  { S->IA[i] = spot + A_OFF;
    spot +=  harray[i].count; 
  }
  S->IA[N] = spot + A_OFF;

  /* test for NaN's */
  for ( i = 0 ; i < S->IA[S->N]-A_OFF ; i++ )
  if ( !is_finite(S->A[i]) )
  { kb_error(1822,"NaNs in Hessian. Replacing with big value.\n",WARNING);
    S->A[i] = 1e30; 
  }

  /* some debug printing */
  if ( hess_debug )
  { 
    printf("IA: ");
    for ( i = 0 ; i <= S->N ; i++ ) printf(" %d",S->IA[i]);
    printf("\nJA: ");
    for ( i = 0 ; i < S->IA[S->N]-A_OFF ; i++ ) printf(" %d",S->JA[i]);
    printf("\n");
    for ( i = 0 ; i <= web.skel[VERTEX].max_ord ; i++ )
    { for ( j = 0 ; j < vhead[i].freedom ; j++ )
      { sprintf(msg,"v%d.%d",i+1,j+1); 
        printf("%10s",msg);
      }
    }
    for ( i = 0 ; i < optparamcount ; i++ )
      printf("%10s",globals[optparam[i].pnum].name);
    printf("\n");
    for ( i = 0 ; i < S->N ; i++ ) 
      { int k,m;
        for ( m = 0 ; m < i ; m++ ) printf("          ");
        for ( m = i,  k = S->IA[i]-A_OFF ; m < S->N ; m++ )
          if ( (m == S->JA[k]-A_OFF) && (k < S->IA[i+1]-A_OFF) )
            { printf(" %9.6f",(DOUBLE)S->A[k]); k++; }
          else printf(" %9.6f",0.0);
        printf("\n");
     }
   } /* end hess_debug */
} /* end bk_AIJ_setup() */

/**********************************************************************
*
* function: bk_constraint_setup()
*
* purpose: get matrices needed for handling constraints
*
*/

void bk_constraint_setup(conarray,concount,S)
struct hess_index *conarray;     /* pointers to rows */
int concount;  /* number of constraints */
struct linsys *S; 
{ int i,j;

  if ( concount == 0 ) return;
  S->concount = concount;
  S->coninx = (int*)mycalloc(concount,sizeof(int));
  S->coninxinv = (int*)mycalloc(concount,sizeof(int));
  /* figure out which rows are really constraints */
  for ( i = 0, S->CN = 0 ; i < concount ; i++ )
   if ( conarray[i].u.congrad != NULL )
     { S->coninx[i] = S->CN;
       S->coninxinv[S->CN] = i;
       S->CN++;
     }
     else S->coninx[i] = -1;  /* marker for nonconstraint */
  if ( S->CN == 0 ) return;

  S->C = (REAL**) mycalloc(S->CN,sizeof(REAL*)) ;
  for ( i = 0 ; i < S->CN ; i++ )
     { /* fill in C  (really C transpose) */
       int k = S->coninxinv[i];
       S->C[i] = conarray[k].u.congrad;
     }
  /* some debug printing */
  if ( hess_debug )
  { for ( i = 0 ; i < S->CN ; i++ )
     { printf("C%d:",i+1);
       for ( j = 0 ; j < S->N ; j++ ) printf(" %9.6f",(double)(S->C[i][j]));
       printf("\n");
     }
  }
} /* end bk_constraint_setup */

/*********************************************************************
*
* function: BK_hess_project_setup()
*
* purpose:  set up projection to constraints using hessian metric
*/

void BK_hess_project_setup(S)
struct linsys *S;
{ int i;

  if ( S->CN == 0 ) return;

  /* HinvC */
  if ( S->HinvC ) free_matrix(S->HinvC);
  S->HinvC = dmatrix(0,S->CN-1,0,S->N-1);
  for ( i = 0 ; i < S->CN ; i++ )
     (*sp_solve_func)(S,S->C[i],S->HinvC[i]);

  /* CHinvCinv */
  if ( S->CHinvCinv ) free_matrix(S->CHinvCinv);
  S->CHinvCinv = dmatrix(0,S->CN-1,0,S->CN-1);
  mat_mul_tr(S->HinvC,S->C,S->CHinvCinv,S->CN,S->N,S->CN);
  { /* adjust for constraints */
     int con_index;
     S->neg -= con_index = matrix_index(S->CHinvCinv,S->CN);
     S->pos -= (S->CN - con_index);
     eigen_neg = S->neg; eigen_pos = S->pos;
  }
  if ( mat_inv(S->CHinvCinv,S->CN) < 0 )
     kb_error(1823,"Constraints not independent.\n",RECOVERABLE);

} /* end BK_hess_project_setup */

/*************************************************************************
*
* function: BK_hess_project()
* 
* purpose: project vector onto constraints of system using Hessian metric
*
*/

void BK_hess_project(S,B,x)
struct linsys *S; /* factored and constrained system */
REAL *B;    /* incoming vector */
REAL *x;    /* solution, may be incoming */
{ REAL *T1,*T2,*T3;
  int i;
  if ( S->CN == 0 ) return;
  T1 = vector(0,2*S->CN+S->N);
  T2 = T1 + S->CN;
  T3 = T2 + S->CN;
  matvec_mul(S->C,B,T1,S->CN,S->N);
  matvec_mul(S->CHinvCinv,T1,T2,S->CN,S->CN);
  vec_mat_mul(T2,S->HinvC,T3,S->CN,S->N);
  for ( i = 0 ; i < S->N ; i++ )
     x[i] = B[i] - T3[i];
  myfree((char*)T1);
} /* end BK_hess_project */

/***********************************************************************
*
*  function: lowest_eigenpair()
*
*  purpose: find lowest eigenvalue and corresponding eigenvalue
*
*  return: lowest eigenvalue;
*/

REAL lowest_eigenpair(S,v)
struct linsys *S;
REAL *v; /* eigenvector, preallocated */
{ REAL lo;
  REAL old_ev,new_ev;
  int i;
  int krydim = 20;
  int nlook = 2;
  REAL evalues[5];

  /* find lower bound on lowest eigenvalue */
  lo = -0.01;
  S->lambda = lo; 
  (*sp_factor_func)(S);
  (*sp_hess_project_setup_func)(S);
  while ( S->neg > 0)
    { 
      S->lambda *= 10;
      (*sp_factor_func)(S);
      (*sp_hess_project_setup_func)(S);
    }
  lanczos(S,krydim,evalues,nlook);

  /* inverse iteration to find eigenvector */
  S->lambda = evalues[0] - .0001;
  (*sp_factor_func)(S);
  (*sp_hess_project_setup_func)(S);
  for ( i = 0 ; i < S->N ; i++ ) v[i] = drand48();
  old_ev = new_ev = 11231.0;  /* weird number, nonzero */
  do
    { sp_hessian_solve(S,v,v,NO_SET_PRESSURE);
      old_ev = new_ev;
      new_ev = sqrt(dot(v,v,S->N));
      for ( i = 0 ; i < S->N ; i++ ) v[i] /= new_ev;
    } while ( fabs(1/old_ev-1/new_ev) > 1e-11 );

  last_eigenvalue = S->lambda + 1/new_ev;
  return S->lambda + 1/new_ev;
} /* end lowest_eigenpair */


/***********************************************************************
*
*  function: cg_lowest_eigenpair()
*
*  purpose: find lowest eigenvalue and corresponding eigenvalue
*      Uses conjugate gradient on sphere to minimize XHX
*      Hessian_metric used.
*
*  return: lowest eigenvalue;
*/

REAL old_cg_lowest_eigenpair(S,x)
struct linsys *S;
REAL *x; /* eigenvector, preallocated */
{ REAL norm_inv;
  int i;
  REAL *h; /* search direction */
  REAL *ah; 
  REAL *f,*If; /* gradient of XAX, form and vector */
  REAL *mx; /* MX */
  REAL *mh; /* MH */
  REAL cgamma; /* cg coefficient */
  REAL xax,old_xax;
  int count;
  int maxcount;
  char response[100];
  REAL **CCinv;
  REAL *Cf,*Gf;

  h = (REAL *)temp_calloc(S->N,sizeof(REAL));
  ah = (REAL *)temp_calloc(S->N,sizeof(REAL));
  f = (REAL *)temp_calloc(S->N,sizeof(REAL));
  if ( web.area_norm_flag || hessian_linear_metric_flag ) 
    { mx = (REAL *)temp_calloc(S->N,sizeof(REAL));
      mh = (REAL *)temp_calloc(S->N,sizeof(REAL));
    }
  if ( S->CN )
    { CCinv = dmatrix(0,S->CN,0,S->CN);
      mat_mul_tr(S->C,S->C,CCinv,S->CN,S->N,S->CN);
      mat_inv(CCinv,S->CN);
      Cf = vector(0,S->CN);
      Gf = vector(0,S->CN);
    }

  /* initial random guess */
  for ( i = 0 ; i < S->N ; i++ ) x[i] = drand48() - .5;
  /* project to constraints */
  if ( S->CN )
  { matvec_mul(S->C,x,Cf,S->CN,S->N);
    matvec_mul(CCinv,Cf,Gf,S->CN,S->CN);
    for ( i = 0 ; i < S->CN ; i++ )
       vector_add_smul(x,S->C[i],-Gf[i],S->N);
  }
  if ( web.area_norm_flag || hessian_linear_metric_flag ) 
      norm_inv = 1/sqrt(sparse_metric_dot(x,x,&Met));
  else norm_inv = 1/sqrt(dot(x,x,S->N));
  for ( i = 0 ; i < S->N ; i++ ) x[i] *= norm_inv;

  xax = 1e30; /* silly value for start of convergence test */
  for(;;)
  { prompt("Enter max iterations: ",response,sizeof(response)); 
    maxcount = atoi(response);
    if ( maxcount == 0 ) break;
    if ( maxcount < 0 ) { count = 0 ; maxcount = -maxcount; }
    else count = 0;

    do
    { REAL a,b,c,evalue,denom,q2,q1comp,hnorm,hnorm_inv;
      old_xax = xax;

      count++;
      /* get cg search direction */
      (*sp_mul_func)(S,x,f);  /* AX */
      xax = dot(x,f,S->N);
      if ( web.area_norm_flag || hessian_linear_metric_flag ) 
          (*sp_mul_func)(&Met,x,mx);
      else mx = x;
      for ( i = 0 ; i < S->N ; i++ ) f[i] -= xax*mx[i];

      /* convert form f to vector If with cg metric I */
      If = f;
      /* project to constraints */
      if ( S->CN )
      { matvec_mul(S->C,If,Cf,S->CN,S->N);
        matvec_mul(CCinv,Cf,Gf,S->CN,S->CN);
        for ( i = 0 ; i < S->CN ; i++ )
          vector_add_smul(If,S->C[i],-Gf[i],S->N);
      }
      /* project If tangent to Steifel manifold */
      if ( web.area_norm_flag || hessian_linear_metric_flag ) 
         c = sparse_metric_dot(x,If,&Met);
      else c = dot(x,If,S->N);
      for ( i = 0 ; i < S->N ; i++ ) If[i] -= c*x[i];

      /* compute Ah */
      /* proper Hessian is A - XAX M */
      (*sp_mul_func)(S,h,ah);
      if ( web.area_norm_flag || hessian_linear_metric_flag ) 
          (*sp_mul_func)(&Met,h,mh);
      else mh = h;
      for ( i = 0 ; i < S->N ; i++ ) ah[i] -= xax*mh[i];

      /* compute gamma */
      if ( count <= 1 ) cgamma = 0.0;
      else
      { cgamma = -dot(If,ah,S->N)/dot(h,ah,S->N);
      }

      /* compute search direction */
      for ( i = 0 ; i < S->N ; i++ ) h[i] = If[i] + cgamma*h[i];

      if ( web.area_norm_flag || hessian_linear_metric_flag ) 
         hnorm = sqrt(sparse_metric_dot(h,h,&Met));
      else hnorm = sqrt(dot(h,h,S->N));
      hnorm_inv = 1/hnorm;

      /* recompute Ah */
      (*sp_mul_func)(S,h,ah); 
#ifdef XXX
projecting off XAX M seems to hurt 
      if ( web.area_norm_flag || hessian_linear_metric_flag ) 
    (*sp_mul_func)(&Met,h,mh);
      else mh = h;
      for ( i = 0 ; i < S->N ; i++ ) ah[i] -= xax*mh[i];
#endif

      /* find minimum along geodesic */
      a = xax;
      b = dot(x,ah,S->N)/hnorm;
      c = dot(h,ah,S->N)/hnorm/hnorm;
      evalue = 0.5*(a+c-sqrt((a-c)*(a-c)+4*b*b));  /* smallest ev */
      denom = sqrt(b*b + (evalue - a)*(evalue - a));
      if ( b < 0.0 ) denom = -denom;
      q2 = (evalue - a)/denom;
      q1comp = (evalue-a)*(evalue-a)/denom/(denom+b);
      /* rotate h and x */
      for ( i = 0 ; i < S->N ; i++ )
      { REAL xtmp = x[i]*hnorm;
        REAL htmp = h[i]*hnorm_inv;
        x[i] += -q1comp*x[i] + q2*htmp;
        h[i] += -q2*xtmp  -q1comp*h[i]; 
      }
      if ( (count < maxcount) && (maxcount > 5) && (count % (maxcount/5) == 0) )
      { 
#ifdef LONGDOUBLE
     sprintf(msg,"%3d.    %3.*Lg\n",count,DPREC,xax);
#else
     sprintf(msg,"%3d.    %3.17g\n",count,xax);
#endif 
     outstring(msg);
      }
    } while ( (fabs(xax - old_xax) > 1e-14*fabs(xax)) && (count<maxcount) );
    if ( count < maxcount )
      { 
#ifdef LONGDOUBLE
     sprintf(msg,"%3d.    %3.*Lg  converged\n",count,DPREC,xax);
#else
     sprintf(msg,"%3d.    %3.17g  converged\n",count,xax);
#endif 
     outstring(msg);
      }
    else  { 
#ifdef LONGDOUBLE
     sprintf(msg,"%3d.    %3.*Lg  max iterations\n",count,DPREC,xax);
#else
     sprintf(msg,"%3d.    %3.17g  max iterations\n",count,xax);
#endif 
     outstring(msg);
      }
  }


  temp_free((char*)h);
  temp_free((char*)ah);
  temp_free((char*)f);
  if ( web.area_norm_flag || hessian_linear_metric_flag ) 
  { temp_free((char*)mx);
    temp_free((char*)mh);
  }
  if ( S->CN )
  { free_vector(Gf,0,S->CN);
    free_vector(Cf,0,S->CN);
    free_matrix(CCinv);
  }
  last_eigenvalue = xax;
  return xax;
} /* end old_cg_lowest_eigenpair */

/***********************************************************************
*
*  function: cg_lowest_eigenpair()
*
*  purpose: find lowest eigenvalue and corresponding eigenvalue
*      Uses conjugate gradient on sphere to minimize XHX
*      Hessian_metric used.
*
*  return: lowest eigenvalue;
*/

REAL cg_lowest_eigenpair(S,x)
struct linsys *S;
REAL *x; /* eigenvector, preallocated */
{ REAL norm_inv;
  int i,j;
  REAL *h; /* search direction */
  REAL *ah;  /* (A-XAX M)H */
  REAL *ax; /* AX */
  REAL *mx; /* MX */
  REAL *mh; /* MH */
  REAL xax,old_xax;
  int count;
  int maxcount;
  char response[100];
  REAL *Cf,*Gf;
  REAL **P; /* matrix of coeff for Lagrange mult */
  REAL **Pinv; /* inverse of P */

  h = (REAL *)temp_calloc(S->N,sizeof(REAL));
  ah = (REAL *)temp_calloc(S->N,sizeof(REAL));
  ax = (REAL *)temp_calloc(S->N,sizeof(REAL));
  if ( web.area_norm_flag || hessian_linear_metric_flag ) 
    { mx = (REAL *)temp_calloc(S->N,sizeof(REAL));
      mh = (REAL *)temp_calloc(S->N,sizeof(REAL));
    }
  else { mx = x; mh = h; }
  P = dmatrix(0,S->CN+1,0,S->CN+1);
  Pinv = dmatrix(0,S->CN+1,0,S->CN+1);
  Cf = vector(0,S->CN+2);
  Gf = vector(0,S->CN+2);
  if ( S->CN )
    { /* CGC goes in upper left corner of P */
      mat_mul_tr(S->C,S->C,P,S->CN,S->N,S->CN);
    }

  /* initial random guess */
  for ( i = 0 ; i < S->N ; i++ ) x[i] = drand48() - .5;
  /* project to constraints */
  if ( S->CN )
    { matvec_mul(S->C,x,Cf,S->CN,S->N);
      matcopy(Pinv,P,S->CN,S->CN);
      mat_inv(Pinv,S->CN);
      matvec_mul(Pinv,Cf,Gf,S->CN,S->CN);
      for ( i = 0 ; i < S->CN ; i++ )
      vector_add_smul(x,S->C[i],-Gf[i],S->N);
    }
  if ( web.area_norm_flag || hessian_linear_metric_flag ) 
      norm_inv = 1/sqrt(sparse_metric_dot(x,x,&Met));
  else norm_inv = 1/sqrt(dot(x,x,S->N));
  for ( i = 0 ; i < S->N ; i++ ) x[i] *= norm_inv;

  xax = 1e30; /* silly value for start of convergence test */
  for(;;)
  { prompt("Enter max iterations: ",response,sizeof(response)); 
    maxcount = atoi(response);
    if ( maxcount == 0 ) break;
    if ( maxcount < 0 ) { count = 0 ; maxcount = -maxcount; }
    else count = 0;

    do
    { REAL a,b,c,evalue,denom,q2,q1comp,hnorm,hnorm_inv;
      int pdim;

      old_xax = xax;

      count++;

      /* get cg search direction */

      /* calculate needed vectors */

      /* MX */
      if ( web.area_norm_flag || hessian_linear_metric_flag ) 
         (*sp_mul_func)(&Met,x,mx);

      /* MH */
      if ( web.area_norm_flag || hessian_linear_metric_flag ) 
         (*sp_mul_func)(&Met,h,mh);
      
      /* (A - XAX M)X */
      (*sp_mul_func)(S,x,ax); 
      xax = dot(x,ax,S->N);
      for ( i = 0 ; i < S->N ; i++ ) ax[i] -= xax*mx[i];

      /* (A - XAX M)H */
      (*sp_mul_func)(S,h,ah); 
      for ( i = 0 ; i < S->N ; i++ ) ah[i] -= xax*mh[i];

      /* fill in P, which already has CGC in upper left */
      if ( S->CN > 0 )
      { matvec_mul(S->C,mx,P[S->CN],S->CN,S->N);
        matvec_mul(S->C,ah,P[S->CN+1],S->CN,S->N);
      }
      P[S->CN][S->CN] = dot(mx,mx,S->N);
      P[S->CN+1][S->CN] = dot(ah,mx,S->N);

#ifdef SYMMETRICWAY
      /* this way takes twice as long */
      P[S->CN+1][S->CN+1] = dot(ah,ah,S->N);
      for ( i = 0 ; i <= S->CN ; i++ )
        for ( j = S->CN ; j < S->CN+2 ; j++ )
           P[i][j] = P[j][i];
#else
      P[S->CN+1][S->CN+1] = dot(ah,h,S->N);
      P[S->CN][S->CN+1] = dot(mx,h,S->N);
      for ( j = 0 ; j < S->CN ; j++ )
         P[j][S->CN+1] = dot(h,S->C[j],S->N);
#endif
      
      /* fill in right side */
      if ( S->CN > 0 ) matvec_mul(S->C,ax,Cf,S->CN,S->N);
      Cf[S->CN] = dot(mx,ax,S->N);
      Cf[S->CN+1] = dot(ah,ax,S->N);

      /* solve */ 
      pdim = S->CN + ( count<=1 ? 1 : 2 );
      matcopy(Pinv,P,pdim,pdim);
      mat_inv(Pinv,pdim);
      matvec_mul(Pinv,Cf,Gf,pdim,pdim);

      /* compute new search direction h */
      for ( i = 0 ; i < S->N ; i++ )
      { 
#ifdef SYMMETRICWAY
        if ( count > 1 ) h[i] = -Gf[S->CN+1]*ah[i]; else h[i] = 0.0;
#else
        if ( count > 1 ) h[i] = -Gf[S->CN+1]*h[i]; else h[i] = 0.0;
#endif
        h[i] += ax[i] - Gf[S->CN]*mx[i];
        for ( j = 0 ; j < S->CN ; j++ )
          h[i] -= Gf[j]*S->C[j][i];
      }

      /* recompute Ah */
      (*sp_mul_func)(S,h,ah); 
      if ( web.area_norm_flag || hessian_linear_metric_flag ) 
        hnorm = sqrt(sparse_metric_dot(h,h,&Met));
      else hnorm = sqrt(dot(h,h,S->N));
      hnorm_inv = 1/hnorm;

      /* find minimum along geodesic */
      a = xax;
      b = dot(x,ah,S->N)/hnorm;
      c = dot(h,ah,S->N)/hnorm/hnorm;
      evalue = 0.5*(a+c-sqrt((a-c)*(a-c)+4*b*b));  /* smallest ev */
      denom = sqrt(b*b + (evalue - a)*(evalue - a));
      if ( b < 0.0 ) denom = -denom;
      q2 = (evalue - a)/denom;
      q1comp = (evalue-a)*(evalue-a)/denom/(denom+b); 
      /* rotate h and x */
      for ( i = 0 ; i < S->N ; i++ )
      { REAL xtmp = x[i]*hnorm;
        REAL htmp = h[i]*hnorm_inv;
        x[i] += -q1comp*x[i] + q2*htmp;
        h[i] += -q2*xtmp  -q1comp*h[i]; 
      }
      if ( (count < maxcount) && (maxcount > 5) && (count % (maxcount/5) == 0) )
      { 
#ifdef LONGDOUBLE
     sprintf(msg,"%3d.    %3.*Lg\n",count,DPREC,xax);
#else
     sprintf(msg,"%3d.    %3.17g\n",count,xax);
#endif 
     outstring(msg);
      }
    } while ( (fabs(xax - old_xax) > 1e-14*fabs(xax)) && (count<maxcount) );
    if ( count < maxcount )
      { 
#ifdef LONGDOUBLE
     sprintf(msg,"%3d.    %3.*Lg  converged\n",count,DPREC,xax);
#else
     sprintf(msg,"%3d.    %3.17g  converged\n",count,xax);
#endif 
     outstring(msg);
      }
    else  { 

#ifdef LONGDOUBLE
     sprintf(msg,"%3d.    %3.*Lg  max iterations\n",count,DPREC,xax);
#else
     sprintf(msg,"%3d.    %3.17g  max iterations\n",count,xax);
#endif 
     outstring(msg);
      }
  }


  temp_free((char*)h);
  temp_free((char*)ah);
  temp_free((char*)ax);
  if ( web.area_norm_flag || hessian_linear_metric_flag ) 
  { temp_free((char*)mx);
    temp_free((char*)mh);
  }
  free_vector(Gf,0,S->CN);
  free_vector(Cf,0,S->CN);
  free_matrix(P);
  free_matrix(Pinv);
  last_eigenvalue = xax;
  return xax;
} /* end cg_lowest_eigenpair */

/***********************************************************************
*
*  function: cg_ritz()
*
*  purpose: find lowest eigenvalues and corresponding eigenvalues
*      Uses conjugate gradient on XMX=I to minimize Tr(XHX)
*      Hessian_metric M used.
*
*/

void cg_ritz(S,n,x,ev)
struct linsys *S;
int n;  /* number of eigenpairs desired */
REAL **x; /* eigenvectors, preallocated */
REAL *ev; /* eigenvalues, preallocated */
{ 
  int i,j,k;
  REAL **h; /* search direction */
  REAL **ah; 
  REAL **f,**If; /* gradient of XAX, form and vector */
  REAL **mx; /* MX */
  REAL **mh; /* MH */
  REAL cgamma; /* cg coefficient */
  REAL **xax;
  REAL trxax,old_trxax; /* objective */
  int count;
  int maxcount;
  char response[100];
  REAL **CCinv;
  REAL **Cf,**Gf;
  struct linsys *M;  /* Hessian metric to use */
  REAL **L,**Linv; /* for LQ decomp */
  REAL **Q;
  REAL **hax,**xah,**hah;
  REAL **B;  /* 2n x 2n matrix to find low eigenvalues for */
  REAL **hx,**xh;
  REAL **evectors,*evalues;
  REAL **NN; /* temp */

  if ( web.area_norm_flag || hessian_linear_metric_flag ) M =  &Met;
  else M = NULL;

  if ( n <= 0 ) return;

  h = dmatrix(0,n-1,0,S->N);
  ah = dmatrix(0,n-1,0,S->N);
  f = dmatrix(0,n-1,0,S->N);
  Q = dmatrix(0,n-1,0,S->N);
  hx = dmatrix(0,2*n-1,0,S->N);
  xh = (REAL**)temp_calloc(2*n,sizeof(REAL*));
  for ( i = 0 ; i < n ; i++ ) { xh[i] = x[i]; xh[i+n] = Q[i]; }
  NN = dmatrix(0,n-1,0,n-1);
  L = dmatrix(0,n-1,0,n-1);
  Linv = dmatrix(0,n-1,0,n-1);
  B = dmatrix(0,2*n-1,0,2*n-1);
  xax = B;
  hax = B+n;
  xah = (REAL**)temp_calloc(n,sizeof(REAL*));
  hah = (REAL**)temp_calloc(n,sizeof(REAL*));
  for ( j = 0 ; j < n ; j++ ) { xah[j] = B[j]+n; hah[j] = B[j+n]+n; }
  evalues = (REAL *)temp_calloc(2*n,sizeof(REAL));
  evectors = dmatrix(0,2*n-1,0,2*n-1);
  if ( web.area_norm_flag || hessian_linear_metric_flag ) 
    { mx = dmatrix(0,n-1,0,S->N);
      mh = dmatrix(0,n-1,0,S->N);
    }
  if ( S->CN )
    { CCinv = dmatrix(0,S->CN,0,S->CN);
      mat_mul_tr(S->C,S->C,CCinv,S->CN,S->N,S->CN);
      mat_inv(CCinv,S->CN);
      Cf = dmatrix(0,S->CN-1,0,n-1);
      Gf = dmatrix(0,S->CN-1,0,n-1); 
    }

  /* initial random guess */
  for ( j = 0 ; j < n ; j++ )
    for ( i = 0 ; i < S->N ; i++ ) x[j][i] = drand48() - .5;
  /* project to constraints */
  if ( S->CN )
    { mat_mul_tr(S->C,x,Cf,S->CN,S->N,n);
      mat_mult(CCinv,Cf,Gf,S->CN,S->CN,n);
      for ( j = 0 ; j < n ; j++ )
       for ( i = 0 ; i < S->CN ; i++ )
        vector_add_smul(x[j],S->C[i],-Gf[i][j],S->N);
    }
  /* project to Steifel manifold XMX = I */
  LQ_decomp(x,n,S->N,x,L,M);

  trxax = 1e30; /* silly value for start of convergence test */
  for(;;)
  { prompt("Enter max iterations: ",response,sizeof(response)); 
    maxcount = atoi(response);
    if ( maxcount == 0 ) break;
    if ( maxcount < 0 ) { count = 0 ; maxcount = -maxcount; }
    else count = 0;

    do
    { REAL denom;
      old_trxax = trxax;

      count++;
      /* get cg search direction */
      for ( j = 0 ; j < n ; j++ )
        (*sp_mul_func)(S,x[j],f[j]);  /* AX */
      mat_mul_tr(x,f,xax,n,S->N,n);
      if ( web.area_norm_flag || hessian_linear_metric_flag ) 
        for ( j = 0 ; j < n ; j++ )
          (*sp_mul_func)(&Met,x[j],mx[j]);
      else mx = x;

      mat_mult(xax,mx,ah,n,n,S->N);  /* ah just used as temp */
      for ( j = 0 ; j < n ; j++ )
        for ( i = 0 ; i < S->N ; i++ ) f[j][i] -= ah[j][i];

      /* convert form f to vector If with cg metric I */
      If = f;
      /* project to constraints */
      if ( S->CN )
      { mat_mul_tr(S->C,If,Cf,S->CN,S->N,n);
        mat_mult(CCinv,Cf,Gf,S->CN,S->CN,n);
        for ( j = 0 ; j < n ; j++ )
         for ( i = 0 ; i < S->CN ; i++ )
          vector_add_smul(If[j],S->C[i],-Gf[i][j],S->N);
      }
      /* project If tangent to Steifel manifold */
      mat_mul_tr(mx,If,NN,n,S->N,n); 
      tr_mat_mul(NN,x,ah,n,n,S->N);
      for ( j = 0 ; j < n ; j++ )
        for ( i = 0 ; i < S->N ; i++ ) If[j][i] -= ah[j][i];

      if ( count <= 1 ) cgamma = 0.0;
      else
      { /* proper Hessian is A - XAX M */
        REAL numer;
        for ( j = 0 ; j < n ; j++ )
          (*sp_mul_func)(S,h[j],ah[j]);
        if ( web.area_norm_flag || hessian_linear_metric_flag ) 
         for ( j = 0 ; j < n ; j++ )
          (*sp_mul_func)(&Met,h[j],mh[j]);
        else mh = h;
        for ( j = 0 ; j < n ; j++ )
         for ( i = 0 ; i < S->N ; i++ ) 
          for ( k = 0 ; k < n ; k++ ) ah[j][i] -= xax[j][k]*mh[k][i];
           for ( j = 0, numer = denom = 0.0 ; j < n ; j++ )
           { numer += dot(If[j],ah[j],S->N);
             denom += dot(h[j],ah[j],S->N);
           }
        cgamma = -numer/denom;
      }
      for ( j = 0 ; j < n ; j++ )
        for ( i = 0 ; i < S->N ; i++ ) h[j][i] = If[j][i] + cgamma*h[j][i];

      /* normalize h to be orthonormal wrt M, hMh = I */
      LQ_decomp(h,n,S->N,Q,L,M); 
      matcopy(Linv,L,n,n);
      mat_inv(Linv,n);

      /* compute Ah */
      for ( j = 0 ; j < n ; j++ )
       (*sp_mul_func)(S,h[j],ah[j]);

      /* load up matrix */
      /* B = | xax xah |  */
      /*      | hax hah |  */
      mat_mul_tr(h,ah,hah,n,S->N,n);
      mat_mult(Linv,hah,NN,n,n,n);
      mat_mul_tr(NN,Linv,hah,n,n,n);
      mat_mul_tr(Q,f,hax,n,S->N,n);
      for ( i = 0 ; i < n ; i++ )
       for ( j = 0 ; j < n ; j++ )
         xah[i][j] = hax[j][i];
      jacobi_eigenpairs(B,2*n,evalues,evectors);
      /* returns eigenvectors in columns of evectors, in descending order */

      /* use lowest n eigenvectors */
      for ( j = n, trxax = 0.0; j < 2*n ; j++ ) trxax += evalues[j];

      /* rotate h and x */
      tr_mat_mul(evectors,xh,hx,2*n,2*n,S->N);
      matcopy(x,hx+n,n,S->N);
      mat_mult(L,hx,h,n,n,S->N);

      if ( (count < maxcount) && (maxcount > 5) && (count % (maxcount/5) == 0) )
      { sprintf(msg,"%3d.  ",count);
          for ( j = 0 ; (j < 4)&&(j<n) ; j++ )
#ifdef LONGDOUBLE
              sprintf(msg+strlen(msg)," %*.*Lg",DWIDTH,DPREC,evalues[2*n-1-j]);
#else
              sprintf(msg+strlen(msg)," %18.15g",evalues[2*n-1-j]);
#endif 
        strcat(msg,"\n");
        outstring(msg);
      }
    } while ( (fabs(trxax-old_trxax) > 1e-14*fabs(trxax)) && (count<maxcount) );
    if ( count < maxcount )
      { 
#ifdef LONGDOUBLE
     sprintf(msg,"%3d.    %3.*Lg  converged\n",count,DPREC,trxax);
#else
     sprintf(msg,"%3d.    %3.17g  converged\n",count,trxax);
#endif 
     outstring(msg);
      }
    else  { 
#ifdef LONGDOUBLE
     sprintf(msg,"%3d.    %3.*Lg  max iterations\n",count,DPREC,trxax);
#else
     sprintf(msg,"%3d.    %3.17g  max iterations\n",count,trxax);
#endif 
     outstring(msg);
      }
  }
  last_eigenvalue = evalues[0];

  free_matrix(h);
  free_matrix(ah);
  free_matrix(hx);
  free_matrix(f);
  free_matrix(B);
  free_matrix(L);
  free_matrix(Q);
  free_matrix(Linv);
  temp_free((char*)hah);
  temp_free((char*)xah);
  temp_free((char*)evalues);
  free_matrix(evectors);

  if ( web.area_norm_flag || hessian_linear_metric_flag ) 
  { free_matrix(mx);
    free_matrix(mh);
  }
  if ( S->CN )
  { free_matrix(Gf);
    free_matrix(Cf);
    free_matrix(CCinv);
  }
} /* end cg_ritz() */

/************************************************************************
*
* function: sp_hessian_solve()
*
* purpose: solve hessian with constraints
*
*/

void sp_hessian_solve(S,rhs,X,set_pressure_flag)
struct linsys *S;
REAL *rhs;
REAL *X;  /* solution */
int set_pressure_flag; /* whether to set body pressures */
{ /* solve stuff */
  int i,j,ii,jj;
  REAL *T1,*T2,*T3;
  body_id b_id;
  struct gen_quant*q;
 
  if ( hess_debug )
  { for ( i = 0 ; i <= web.skel[VERTEX].max_ord ; i++ )
    { for ( j = 0 ; j < vhead[i].freedom ; j++ )
      { ii = vhead[i].rownum+j;
        printf("    %f  B[%d] v%d.%d\n",(DOUBLE)rhs[ii],ii,i+1,j+1);
      }
    }
    for ( i = 0 ; i < optparamcount ; i++ )
    { ii = optparam[i].pnum;
      jj = optparam[i].rownum;
      printf("    %f  B[%d] %s\n",(DOUBLE)rhs[jj],jj,globals[ii].name);
    }
  }
  
  (*sp_solve_func)(S,rhs,X);

  if ( S->CN > 0 ) /* have constraints */
  { REAL *con_rhs = (REAL*)temp_calloc(S->CN,sizeof(REAL));
     for ( i = 0 ; i < S->concount ; i++ )
     { j = S->coninx[i];
       if ( j >= 0 ) con_rhs[j] = rhs[S->N+i];
     }
     T1 = vector(0,S->CN-1);
     T2 = vector(0,S->CN-1);
     T3 = vector(0,S->N-1);

     matvec_mul(S->C,X,T1,S->CN,S->N);
     for ( i = 0 ; i < S->CN ; i++ )
     T1[i] += con_rhs[i]; /* corrections */
     matvec_mul(S->CHinvCinv,T1,T2,S->CN,S->CN); /* Lagrange multipliers */
     vec_mat_mul(T2,S->HinvC,T3,S->CN,S->N);
     for ( i = 0 ; i < S->N ; i++ )
       X[i] -= T3[i];
     for ( i = 0 ; i < S->CN ; i++ ) con_rhs[i] = -T2[i];

     if ( set_pressure_flag == SET_PRESSURE )
     { /* remap Lagrange multipliers*/
       for ( i = 0 ; i < web.bodycount + gen_quant_count ; i++ )
         if ( S->coninx[i] >= 0 )
           pressures[i] = con_rhs[S->coninx[i]];

       FOR_ALL_BODIES(b_id)
        { REAL p = get_body_pressure(b_id)+pressures[ordinal(b_id)];
          set_body_pressure(b_id,p);
        }
       for ( i = 0, q = GEN_QUANTS ; i < gen_quant_count ; i++,q++ )
         if ( q->flags & Q_FIXED )
         { q->pressure += pressures[web.bodycount+i];
           if ( valid_id(q->b_id) )
           set_body_pressure(q->b_id,q->pressure);
         }
     }
     myfree((char*)T1);
     myfree((char*)T2);
     myfree((char*)T3);
     temp_free((char*)con_rhs);
  }
  if ( hess_debug )
  { for ( i = 0 ; i <= web.skel[VERTEX].max_ord ; i++ )
    { for ( j = 0 ; j < vhead[i].freedom ; j++ )
      { ii = vhead[i].rownum+j;
        printf("    %f  X[%d] v%d.%d\n",(DOUBLE)X[ii],ii,i+1,j+1);
      } 
    }
    for ( i = 0 ; i < optparamcount ; i++ )
    { ii = optparam[i].pnum;
      jj = optparam[i].rownum;
      printf("    %f  X[%d] %s\n",(DOUBLE)X[jj],jj,globals[ii].name);
    }
  }
} /* end sp_hessian_solve */

/************************************************************************
*
* function: sp_hessian_solve_multi()
*
* purpose: solve hessian with constraints for multiple rhs
*
*/

void sp_hessian_solve_multi(S,rhs,X,rk)
struct linsys *S;
REAL **rhs;
REAL **X;  /* solution */
int rk;    /* number of right sides */
{ /* solve stuff */
  int i,k;
  REAL *T1,*T2,*T3;
 
  (*sp_solve_multi_func)(S,rhs,X,rk);

  if ( S->CN > 0 ) /* have constraints */
  { REAL **con_rhs = dmatrix(0,rk-1,0,S->CN);
     for ( i = 0 ; i < S->concount ; i++ )
     { int j = S->coninx[i];
       if ( j >= 0 ) 
         for ( k = 0 ; k < rk ; k++ ) con_rhs[k][j] = rhs[k][S->N+i];
     }
     T1 = vector(0,S->CN-1);
     T2 = vector(0,S->CN-1);
     T3 = vector(0,S->N-1);

     for ( k = 0 ; k < rk ; k++ )
     { matvec_mul(S->C,X[k],T1,S->CN,S->N);
       for ( i = 0 ; i < S->CN ; i++ )
        T1[i] += con_rhs[k][i]; /* corrections */
       matvec_mul(S->CHinvCinv,T1,T2,S->CN,S->CN); /* Lagrange multipliers */
       vec_mat_mul(T2,S->HinvC,T3,S->CN,S->N);
       for ( i = 0 ; i < S->N ; i++ )
         X[k][i] -= T3[i];
       for ( i = 0 ; i < S->CN ; i++ ) con_rhs[k][i] = -T2[i];
     }
     /* conrhs not used?? */

     myfree((char*)T1);
     myfree((char*)T2);
     myfree((char*)T3);
     free_matrix(con_rhs);
  }
} /* end sp_hessian_solve_multi */


/*******************************************************************************
*
*  function: bk_eigenprobe()
*
*  purpose: find number of eigenvalues less than, equal, or greater than
*      desired value.  Asks user for probe values.
*/

void bk_eigenprobe(S)
struct linsys *S;
{ 
    for(;;)
    { char response[100];
      prompt("Enter probe value: ",response,sizeof(response));
      if ( (response[0] == 0) || (response[0] == 'q') ) break;
      S->lambda = atof(response);
      (*sp_factor_func)(S);
      (*sp_hess_project_setup_func)(S);
      sprintf(msg,"Eigencounts:    %d <,  %d ==,  %d > \n",S->neg,S->zero,S->pos);
      outstring(msg);
    }
    return;
} /* end bk_eigenprobe */

/*******************************************************************************
*
*  function: bk_inverse_it()
*
*  purpose: Find eigenvector near probe value.
*/

void bk_inverse_it(S,V)
struct linsys *S;
REAL *V; /* for eigenvector return */
{ int i,ii,k,its;
  REAL t,oldt;
  char response[100];
  REAL *W = NULL; /* for use when metric is needed */

  prompt("Enter probe value: ",response,sizeof(response));
  if ( (response[0] == 0) || (response[0] == 'q') ) return;
  S->lambda = atof(response);
  (*sp_factor_func)(S);
  (*sp_hess_project_setup_func)(S);
  sprintf(msg,"Eigencounts:    %d <,  %d ==,  %d > \n",S->neg,S->zero,S->pos);
  outstring(msg);

  if ( web.area_norm_flag || hessian_linear_metric_flag ) 
     W = (REAL*)mycalloc(S->N,sizeof(REAL));
  /* inverse iteration */
  /* random starting vector */
  for ( i = 0 ; i < S->N ; i++ ) V[i] = drand48();
  t = 1/sqrt(dot(V,V,S->N));
  for ( i = ii = 0 ; i < S->N ; i++,ii++ ) V[i] *= t;
  prompt("Enter maximum iterations: ",response,sizeof(response));
  its = atoi(response);
  if ( its == 0 ) goto afterits;
  oldt = 1e30; /* for convergence test */
  for ( k = 0, ii=0 ; k < its ; k++,ii++ )
  { REAL oldv0=1.0; /* for sign */
     int oldvi=0;     /* index of nonzero component */
     REAL eps = 1/sqrt((REAL)(S->N))/2;
     for ( i = 0, oldv0 = 0.0 ; i < S->N ; i++ ) 
       if ( fabs(V[i]) > eps ) { oldvi = i; oldv0 = V[i]; break; }
     if ( web.area_norm_flag || hessian_linear_metric_flag ) 
        (*sp_mul_func)(&Met,V,W);
     else W = V;
     sp_hessian_solve(S,W,V,NO_SET_PRESSURE);
     if ( web.area_norm_flag || hessian_linear_metric_flag ) 
       t = 1/sqrt(sparse_metric_dot(V,V,&Met));
     else t = 1/sqrt(dot(V,V,S->N));
     if ( V[oldvi]*oldv0 < 0. ) t = -t;  /* get sign right */
#ifdef LONGDOUBLE
     if ( k % 10 == 0 ) printf("%d  ev = %*.*Lf\n",k,DWIDTH,DPREC,S->lambda+t);
#else
     if ( k % 10 == 0 ) printf("%d  ev = %20.15f\n",k,S->lambda+t);
#endif 
     for ( i = 0 ; i < S->N ; i++ ) V[i] *= t;
     if ( fabs(t-oldt) <= 1e-13*fabs(t) ) 
      { /* does twice as many iterations as needed to get t converged */
        /* so eigenvector gets fully converged also */
        ii -= 2;
        if ( ii <= 0 )break;
      }
     oldt = t;
  }
#ifdef LONGDOUBLE
  printf("%d  ev = %*.*Lf\n",k,DWIDTH,DPREC,S->lambda+t);
#else
  printf("%d  ev = %20.15f\n",k,S->lambda+t);
#endif 
afterits:
  if ( web.area_norm_flag || hessian_linear_metric_flag ) 
      myfree((char*)W);
  return;
} /* end bk_inverse_it() */

/*************************************************************************
*
* function: sp_Hessian_solver()
*
* purpose;  Set up and solve sparse Hessian with given right hand side.
*/
void sp_Hessian_solver(S,rhs,X)
struct linsys *S;  /* system */
REAL *rhs; /* right side */
REAL *X; /* returned vector */
{

  S->P = (int *)mycalloc(A_rows,sizeof(int));    
  (*sp_AIJ_setup_func)(array,A_rows,S);
  if ( sp_ordering_func ) (*sp_ordering_func)(S);
  (*sp_constraint_setup_func)
  (array+bodyrowstart,web.bodycount + gen_quant_count,S);
  (*sp_factor_func)(S);
  (*sp_hess_project_setup_func)(S);

  if ( hess_debug )
  { int i;
    puts("rhs:");
    for ( i = 0 ; i < S->N ; i++ ) printf("%g \n",(DOUBLE)rhs[i]);
  }

  sp_hessian_solve(S,rhs,X,SET_PRESSURE);

  if ( S->neg > 0 )
    { sprintf(msg,"Hessian not positive definite. Index: %d\n",S->neg);
      kb_error(1825,msg,WARNING);
    } 

  if ( hess_debug )
  { int i;
    REAL *out = (REAL*)mycalloc(S->N,sizeof(REAL));
    puts("X:");
    for ( i = 0 ; i < S->N ; i++ ) printf("%g \n",(DOUBLE)X[i]);
    bk_mul(S,X,out);
    puts("check:");
    for ( i = 0 ; i < S->N ; i++ ) printf("%g \n",(DOUBLE)out[i]);
    myfree((char*)out);
  }
} /* end sp_hessian_solver() */

/**************************************************************************
*
* function: sparse_metric_dot()
*
* purpose: Take dot product of vectors when metric is sparse linear system.
*     Metric sparse storage has diagonal element first in each row.
*
*/

REAL sparse_metric_dot(u,v,M)
REAL *u,*v; /* the vectors to be dotted */
struct linsys *M;  /* the metric */
{ int i,j,end;
  REAL sum;

  sum = 0.0;
  for ( i = 0 ; i < M->N ; i++ )
  { j = M->IA[i]-A_OFF;
    end = M->IA[i+1]-A_OFF;
    sum += M->A[j]*u[i]*v[i];
    for ( j++  ; j < end  ; j++ )
     { int ii = M->JA[j]-A_OFF;
       sum += M->A[j]*(u[i]*v[ii] + u[ii]*v[i]);
     }
  }
  return sum;
} /* end sparse_metric_dot() */

