/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: snap_discover.c
 *
 * Functions to help in the discovery of snapshot volumes.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "snapshot.h"

/**
 * read_metadata
 *
 * Read the snapshot metadata from the specified object. The location and
 * size of the metadata is in the object's feature header.
 **/
static int read_metadata(storage_object_t * object,
			 snapshot_metadata_t ** metadata)
{
	void * data;
	int rc;

	LOG_ENTRY();

	/* Allocate an I/O buffer. */
	data = EngFncs->engine_alloc(object->feature_header->feature_data1_size * EVMS_VSECTOR_SIZE);
	if (!data) {
		LOG_ERROR("Memory error creating buffer to read snapshot "
			  "metadata from %s.\n", object->name);
		rc = ENOMEM;
		goto out;
	}

	rc = READ(object, object->feature_header->feature_data1_start_lsn,
		  object->feature_header->feature_data1_size, data);
	if (rc) {
		EngFncs->engine_free(data);
		LOG_ERROR("I/O error reading snapshot metadata from %s.\n",
			  object->name);
		goto out;
	}

	*metadata = data;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * check_metadata_signature
 *
 * Check for a valid signature in the snapshot metadata. The metadata is
 * assumed to be in disk-order.
 **/
static inline int check_metadata_signature(snapshot_metadata_t * metadata)
{
	int rc = 0;
	LOG_ENTRY();
	if (DISK_TO_CPU32(metadata->signature) != SNAPSHOT_SIGNATURE) {
		LOG_WARNING("No snapshot metadata found.\n");
		rc = EINVAL;
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * check_metadata_crc
 *
 * Check the CRC in the metadata. Still assumed to be disk-order.
 **/
static inline int check_metadata_crc(snapshot_metadata_t * metadata)
{
	u_int32_t org_crc, new_crc;
	int rc = 0;

	LOG_ENTRY();

	org_crc = DISK_TO_CPU32(metadata->crc);
	metadata->crc = 0;
	new_crc = EngFncs->calculate_CRC(EVMS_INITIAL_CRC,
					 metadata, sizeof(*metadata));
	if (new_crc != org_crc) {
		LOG_ERROR("Incorrect CRC found in snapshot metadata.\n");
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * calculate_metadata_crc
 *
 * Calculate a new CRC for the specified metadata, which should be in
 * disk-order.
 **/
static void calculate_metadata_crc(snapshot_metadata_t * metadata)
{
	LOG_ENTRY();
	metadata->crc = 0;
	metadata->crc = EngFncs->calculate_CRC(EVMS_INITIAL_CRC, metadata,
					       sizeof(*metadata));
	metadata->crc = DISK_TO_CPU32(metadata->crc);
	LOG_EXIT_VOID();
}

/**
 * convert_metadata
 *
 * Endian-convert the snapshot metadata.
 **/
static void convert_metadata(snapshot_metadata_t * metadata)
{
	LOG_ENTRY();
	metadata->signature		= DISK_TO_CPU32(metadata->signature);
	metadata->crc			= DISK_TO_CPU32(metadata->crc);
	metadata->version.major		= DISK_TO_CPU32(metadata->version.major);
	metadata->version.minor		= DISK_TO_CPU32(metadata->version.minor);
	metadata->version.patchlevel	= DISK_TO_CPU32(metadata->version.patchlevel);
	metadata->flags			= DISK_TO_CPU32(metadata->flags);
	metadata->origin_size		= DISK_TO_CPU64(metadata->origin_size);
	metadata->total_chunks		= DISK_TO_CPU32(metadata->total_chunks);
	metadata->chunk_size		= DISK_TO_CPU32(metadata->chunk_size);
	LOG_EXIT_VOID();
}

/**
 * check_metadata_version
 *
 * Check that the metadata version matches this version of the snapshot
 * plugin. Metadata is now assumed to be in cpu-order.
 **/
static inline int check_metadata_version(snapshot_metadata_t * metadata)
{
	int rc = 0;
	LOG_ENTRY();
	if (metadata->version.major != my_plugin_record->version.major) {
		LOG_ERROR("Version of metadata does not match snapshot plugin version.\n");
		LOG_ERROR("Metadata: %d.%d.%d\n",
			  metadata->version.major,
			  metadata->version.minor,
			  metadata->version.patchlevel);
		LOG_ERROR("Plugin: %d.%d.%d\n",
			  my_plugin_record->version.major,
			  my_plugin_record->version.minor,
			  my_plugin_record->version.patchlevel);
		rc = EINVAL;
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * get_snapshot_metadata
 *
 * Read the snapshot metadata from the specified object. Verify the signature,
 * CRC, and version number. If everything checks out, endian-convert the
 * metadata.
 **/
int get_snapshot_metadata(storage_object_t * snap_child,
			  snapshot_metadata_t ** metadata)
{
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Getting snapshot metadata from %s\n", snap_child->name);

	rc = read_metadata(snap_child, metadata);
	if (rc) {
		goto out;
	}

	rc = check_metadata_signature(*metadata);
	if (rc) {
		goto out;
	}

	rc = check_metadata_crc(*metadata);
	if (rc) {
		goto out;
	}

	convert_metadata(*metadata);

	rc = check_metadata_version(*metadata);
	if (rc) {
		goto out;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * write_snapshot_metadata
 *
 * Write the specified snapshot's metadata to disk.
 **/
int write_snapshot_metadata(snapshot_volume_t * snap_volume, boolean backup)
{
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Writing metadata for %s.\n", snap_volume->parent->name);

	/* Convert from cpu-order to disk-order and get a new CRC. */
	convert_metadata(snap_volume->metadata);
	calculate_metadata_crc(snap_volume->metadata);

	if (backup) {
		rc = EngFncs->save_metadata(snap_volume->parent->name,
					    snap_volume->child->name,
					    snap_volume->child->feature_header->feature_data1_start_lsn,
					    snap_volume->child->feature_header->feature_data1_size,
					    snap_volume->metadata);
	} else {
		rc = WRITE(snap_volume->child,
			   snap_volume->child->feature_header->feature_data1_start_lsn,
			   snap_volume->child->feature_header->feature_data1_size,
			   snap_volume->metadata);
	}

	if (rc) {
		LOG_ERROR("I/O error writing snapshot metadata to %s.\n",
			  snap_volume->child->name);
	}

	/* Convert back to cpu-order. */
	convert_metadata(snap_volume->metadata);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * erase_snapshot_metadata
 *
 * Erase the snapshot metadata from the specified object. The object should
 * still have the feature-header attached. The engine takes care of erasing
 * the feature-headers from disk. This also clears the DM header sector.
 **/
void erase_snapshot_metadata(storage_object_t * snap_child)
{
	LOG_ENTRY();
	KILL_SECTORS(snap_child,
		     snap_child->feature_header->feature_data1_start_lsn,
		     snap_child->feature_header->feature_data1_size);
	KILL_SECTORS(snap_child, 0, 1);
	LOG_EXIT_VOID();
}

/**
 * erase_snapshot_header
 *
 * Erase the DM header from the start of the snapshot child object. This will
 * effectively reset the snapshot, since DM will think it's new the next time
 * it is activated.
 **/
int erase_snapshot_header(snapshot_volume_t * snap_volume, boolean backup)
{
	char * buffer;
	int rc;

	LOG_ENTRY();

	LOG_DETAILS("Erasing header from snapshot %s.\n",
		    snap_volume->parent->name);

	buffer = EngFncs->engine_alloc(EVMS_VSECTOR_SIZE);
	if (!buffer) {
		rc = ENOMEM;
		goto out;
	}

	if (backup) {
		rc = EngFncs->save_metadata(snap_volume->parent->name,
					    snap_volume->child->name,
					    0, 1, buffer);
	} else {
		rc = WRITE(snap_volume->child, 0, 1, buffer);
	}

out:
	EngFncs->engine_free(buffer);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * find_origin
 * @origin_name: Name of the EVMS volume for the desired origin. The
 *		 /dev/evms portion of the name should be removed first.
 *
 * Get the global list of volumes, and search for the specified volume name.
 * If found, return the underlying storage object.
 **/
storage_object_t * find_origin(char * origin_name)
{
	list_anchor_t volumes;
	list_element_t itr;
	logical_volume_t * volume;
	storage_object_t * origin = NULL;
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Searching for origin volume %s.\n", origin_name);

	rc = EngFncs->get_volume_list(NULL, NULL, 0, &volumes);
	if (rc) {
		LOG_ERROR("Error getting volume list from engine.\n");
		goto out;
	}

	LIST_FOR_EACH(volumes, itr, volume) {
		if (!strncmp(skip_dev_path(volume->name), origin_name,
			     EVMS_VOLUME_NAME_SIZE)) {
			origin = volume->object;
			break;
		}
	}
	EngFncs->destroy_list(volumes);

	if (!origin) {
		LOG_ERROR("Cannot find origin volume %s.\n", origin_name);
	}

out:
	LOG_EXIT_PTR(origin);
	return origin;
}

/**
 * check_snapshot_header
 *
 * Read the DM snapshot header from the first sector of the snapshot child
 * object and check the signature and valid bit. This sector is not stored
 * in endian-neutral format.
 **/
static int check_snapshot_header(snapshot_volume_t * snap_volume)
{
	storage_object_t * snap_child = snap_volume->child;
	dm_snapshot_header_t * header;
	int rc;

	LOG_ENTRY();

	/* Allocate an I/O buffer. */
	header = EngFncs->engine_alloc(EVMS_VSECTOR_SIZE);
	if (!header) {
		LOG_ERROR("Memory error creating buffer to read DM snapshot "
			  "header from %s.\n", snap_child->name);
		rc = ENOMEM;
		goto out;
	}

	/* Read the first sector. */
	rc = READ(snap_child, 0, 1, header);
	if (rc) {
		EngFncs->engine_free(header);
		LOG_ERROR("I/O error reading DM snapshot header from %s.\n",
			  snap_child->name);
		goto out;
	}

	if (DISK_TO_CPU32(header->magic) == SNAPSHOT_DM_MAGIC) {
		if (DISK_TO_CPU32(header->valid) == 0) {
			/* This snapshot has been initialized by DM,
			 * and is now marked full or disabled.
			 */
			mark_invalid(snap_volume);
		}
	}

	EngFncs->engine_free(header);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * check_sibling_state
 *
 * This check was added when the naming-scheme for the snapshot sibling was
 * changed. If anyone had an active snapshot at the time they upgraded their
 * version of EVMS to this new scheme, some naming problems would arise, and
 * at the very least create an orphaned kernel device. This check attempts to
 * remedy this situation by trying to rename from the old naming scheme to
 * the new one.
 **/
static void check_sibling_state(snapshot_volume_t * snap_volume)
{
	char old_name[EVMS_NAME_SIZE];
	int rc;

	LOG_ENTRY();

	if (!(snap_volume->sibling->flags & SOFLAG_ACTIVE)) {
		strncpy(old_name, snap_volume->child->name, EVMS_NAME_SIZE);
		strncat(old_name, "#snap#", EVMS_NAME_SIZE-strlen(old_name));

		/* Lie about the active bit to fool dm_rename. The subsequent
		 * call to dm_update_status will correct it if necessary.
		 */
		snap_volume->sibling->flags |= SOFLAG_ACTIVE;

		rc = EngFncs->dm_rename(snap_volume->sibling, old_name,
					snap_volume->sibling->name);
		if (rc) {
			LOG_ERROR("Unable to rename sibling for snapshot %s\n",
				  snap_volume->parent->name);
		}
		EngFncs->dm_update_status(snap_volume->sibling);
	}

	LOG_EXIT_VOID();
}

/**
 * get_snapshot_state
 *
 * Get Device-Mapper's state for this snapshot. If the device is active,
 * retrieve the percent-full value. If the device is not active, check
 * the DM header to see if the snapshot is valid.
 **/
void get_snapshot_state(snapshot_volume_t * snap_volume)
{
	storage_object_t * snap_parent = snap_volume->parent;
	char * info = NULL;
	u_int64_t numerator, denominator;
	int percent, rc;

	LOG_ENTRY();
	LOG_DEBUG("Checking state of snapshot %s.\n", snap_parent->name);

	if (!is_active(snap_volume)) {
		EngFncs->dm_update_status(snap_volume->sibling);
		EngFncs->dm_update_status(snap_parent);
	}

	if (is_active(snap_volume)) {
		check_sibling_state(snap_volume);

		rc = EngFncs->dm_get_info(snap_parent, &info);
		if (!rc) {
			if (sscanf(info, "%"SCNu64"/%"SCNu64,
				   &numerator, &denominator) == 2) {
				snap_volume->percent_full = (numerator * 100) /
							    denominator;
			} else if (sscanf(info, "%d%%", &percent) == 1) {
				snap_volume->percent_full = percent;
			} else if (!strcmp(info, "Invalid")) {
				snap_volume->percent_full = 0;
				mark_invalid(snap_volume);
			} else {
				snap_volume->percent_full = -1;
			}

			EngFncs->engine_free(info);
		}
	} else {
		if (!is_invalid(snap_volume)) {
			check_snapshot_header(snap_volume);
		}
	}

	LOG_EXIT_VOID();
}

/**
 * get_origin_state
 *
 * Get Device-Mapper's state for this origin. If the origin is still inactive
 * after checking the kernel state, we need to grab the child's device
 * numbers.
 **/
void get_origin_state(snapshot_volume_t * org_volume)
{
	LOG_ENTRY();
	LOG_DEBUG("Checking state of origin %s.\n", org_volume->parent->name);

	if (!is_active(org_volume)) {
		EngFncs->dm_update_status(org_volume->parent);
	}

	LOG_EXIT_VOID();
}

