# (C) Copyright IBM Corp. 2004
#
# This program is free software;  you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY;  without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
# the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program;  if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
# Perl module with common subroutines for managing Snapshots.

package Evms::Snapshot;

use strict;
use warnings;

use Evms::Common;
use Evms::Object;

BEGIN {
	use Exporter();
	our (@ISA, @EXPORT);
	@ISA = qw(Exporter);
	@EXPORT = qw(&create_snapshot
		     &rollback_snapshot
		     &reset_snapshot
		     &get_snapshot_percent_full
		     &get_snapshot_state
		    );
}

# create_snapshot
# Create a snapshot object. This does not create a volume from the object. To
# do that, use the create_evms_volume() subroutine from Volume.pm.
#
# Arguments:
#    object: Object to create the snapshot on top of.
#    options: Reference to a hash of options for creating the container.
#             Allowable hash keys and values are:
#               original: Name of volume to be snapshotted.
#               snapshot: Name for the new snapshot object.
#               chunksize: Size of snapshot chunks. Must include size suffix.
#               writeable: Make snapshot writeable. "TRUE" or "FALSE".
sub create_snapshot($ $)
{
	my $object = $_[0];
	my $options_ref = $_[1];
	my %options = %{$options_ref};
	my ($key, $option_args);
	my ($command, $rc);

	$option_args = "";
	foreach $key ("original", "snapshot", "chunksize", "writeable") {
		if (defined($options{$key})) {
			$option_args .= "$key=$options{$key},";
		}
	}
	# Strip trailing commas.
	$option_args =~ s/^(.*)\,$/$1/;

	$command = "create:object,Snapshot={$option_args},$object";
	$rc = run_evms_command($command);

	return $rc;
}

# rollback_snapshot
# Perform a rollback on the specified snapshot object. The snapshot and the
# origin must both be unmounted.
#
# Arguments:
#    snapshot: Name of snapshot object to rollback.
sub rollback_snapshot($)
{
	my $snapshot = $_[0];
	my $command = "task:rollback,$snapshot";
	my $rc;

	$rc = run_evms_command($command);

	return $rc;
}

# reset_snapshot
# Reset the specified snapshot object to the current state of its origin
# volume. The snapshot must be unmounted.
#
# Arguments:
#    snapshot: Name of snapshot object to rollback.
sub reset_snapshot($)
{
	my $snapshot = $_[0];
	my $command = "task:reset,$snapshot";
	my $rc;

	$rc = run_evms_command($command);

	return $rc;
}

# get_snapshot_percent_full
# Return the percentage-full for the specified snapshot object.
#
# Arguments:
#    snapshot: Name of snapshot object to query.
sub get_snapshot_percent_full($)
{
	my $snapshot = $_[0];
	my @output;
	my $percent;

	@output = extended_query_object($snapshot);
	if (!@output) {
		return -1;
	}

	$percent = get_extended_info_value("PercentFull", \@output);
	$percent =~ s/(\d+)\s+percent/$1/;

	return $percent;
}

# get_snapshot_state
# Get the current state of the specified snapshot object.
#
# Arguments:
#    snapshot: Name of snapshot object to query.
# Returns:
#    "Active", "Inactive", "Disabled/Full", "Pending Rollback", or "Unknown".
sub get_snapshot_state($)
{
	my $snapshot = $_[0];
	my @output;
	my $state;

	@output = extended_query_object($snapshot);
	if (!@output) {
		return "Unknown";
	}

	$state = get_extended_info_value("State", \@output);

	return $state;
}

1;

