/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdrivelink.so
 *
 *   File: dl_object.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "dl_common.h"


/*
 *  Function: dl_shrink_drivelink
 *
 *  Each time called, we see if we can remove the child storage object from the parent
 *  drive link storage object.
 *
 *  The job is simply to remove the last child object from the drive link storage
 *  object and clean up the ordering table.
 */
static int dl_shrink_drivelink( storage_object_t  *drivelink,
                                storage_object_t  *child )
{
        int                       rc = 0;
        int                       index;
        drivelink_private_data_t  *pdata;
        struct plugin_functions_s *fncs=NULL;
        evms_feature_header_t     *feature_header;

        LOG_ENTRY();

        pdata = (drivelink_private_data_t *)drivelink->private_data;

        // get index for last (nth) storage object in the drivelink
        index = pdata->drive_link_count - 1;

        // specified object must be last storage object in the drivelink
        REQUIRE(child == pdata->drive_link[index].object);

        // remove child from parent object child list
        EngFncs->remove_thing(drivelink->child_objects, child);

        // remove parent from child parent list
        EngFncs->remove_thing(child->parent_objects, drivelink);

        // cleanup the drivelink metadata

        // shrink drive link size
        drivelink->size -= pdata->drive_link[index].sector_count;

        // decrement link count
        --pdata->drive_link_count;

        // remove our metadata from child storage object
        if ( dl_isa_missing_child(child) == FALSE ) {

                fncs = (struct plugin_functions_s *)child->plugin->functions.plugin;
                feature_header = child->feature_header;

                rc = fncs->add_sectors_to_kill_list( child,
                                                     feature_header->feature_data1_start_lsn,
                                                     feature_header->feature_data1_size );
                if (rc) {
                        LOG_ERROR("error, add kill sectors call failed\n");
                }

                rc = fncs->add_sectors_to_kill_list( child,
                                                     feature_header->feature_data2_start_lsn,
                                                     feature_header->feature_data2_size );
                if (rc) {
                        LOG_ERROR("error, add kill sectors call failed\n");
                }
        }

        (pdata->ordering_table+index)->child_serial_number = 0;
        (pdata->ordering_table+index)->child_vsize         = 0;

        // clear the drive link entry
        memset( &pdata->drive_link[index], 0, sizeof(drive_link_t) );


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  dl_expand_drivelink
 *
 *  Each time called, we add the storage objects to the parent
 *  drive link storage object.
 *
 *  This routine can be called from either dl_create() or dl_expand().
 *  with objects that have NO feature headers and NO feature data.
 *
 *  The job is simply to add child objects to the drive link storage
 *  object who dont have child serial numbers or positions in the
 *  ordering table.
 *
 *  Called with exactly the number of objects we need to add to the
 *  drivelink!
 */
static int dl_expand_drivelink( storage_object_t *drivelink,
                                list_anchor_t     expand_objects )
{
        storage_object_t  *child;
        u_int32_t          child_serial_number;
        u_int64_t          expand_size = 0;
        drivelink_private_data_t *pdata;
        list_element_t iter;
        int  rc=0,i;
        int  original_link_count;

        LOG_ENTRY();

        LOG_DEBUG("expanding drivelink %s\n", drivelink->name );

        pdata = (drivelink_private_data_t *)drivelink->private_data;

	// Check with the engine before starting the expand.
	LIST_FOR_EACH(expand_objects, iter, child) {
		expand_size += (child->size - 2*DRIVELINK_METADATA_SECTOR_COUNT -
				2*FEATURE_HEADER_SECTOR_COUNT);
	}
	rc = EngFncs->can_expand_by(drivelink, &expand_size);
	if (rc) {
		LOG_ERROR("Expand of object %s rejected by the engine.\n", drivelink->name);
		LOG_EXIT_INT(rc);
		return rc;
	}

        // get setup for any possible unwinding of the expand
        original_link_count = pdata->drive_link_count;

        // walk through the supplied list of objects and try to add them all
        // to the drivelink.
        LIST_FOR_EACH(expand_objects, iter, child) {

                LOG_DEBUG("adding child %s\n", child->name );

                // create a feature header for the child object
                child->feature_header = (evms_feature_header_t *) EngFncs->engine_alloc( sizeof(evms_feature_header_t) );

                if (child->feature_header==NULL){
                        rc=ENOMEM;
                        break;
                }

                // create a drive link serial number for the child object we are consuming .
                // The routine will check that it is unique for the ordering table.
                child_serial_number = dl_gen_child_serial_number( drivelink );

                if (child_serial_number == BAD_SERIAL_NUMBER) {
                        rc=EINVAL;
                        break;
                }

                // update the drive link ordering table to maintain a
                // sequence for child links.
                (pdata->ordering_table+pdata->drive_link_count)->child_serial_number = child_serial_number;
                (pdata->ordering_table+pdata->drive_link_count)->child_vsize = 0;
                ++pdata->drive_link_count;

                // add child to the drive link storage object
                rc = dl_add_child_object_to_drivelink( drivelink, child, child_serial_number, NULL);
                if (rc==0 ) {
                        drive_link_t  *dlink = &pdata->drive_link[pdata->drive_link_count-1];
                        rc = dl_build_feature_header( drivelink, dlink, dlink->object->feature_header );
                }
                else {
                        --pdata->drive_link_count;
                }
                if (rc) {
                        break;
                }
        }

        // if there were any errors and we consumed some new children objects
        // then we need to unwind the expand
        if (rc) {
                for (i=original_link_count-1; i<pdata->drive_link_count; i++) {
                        if (pdata->drive_link[i].object != NULL) {
                                if (pdata->drive_link[i].object->feature_header) {
                                        EngFncs->engine_free(pdata->drive_link[i].object->feature_header);
                                }
                                EngFncs->remove_thing(drivelink->child_objects, pdata->drive_link[i].object);
                                EngFncs->remove_thing(pdata->drive_link[i].object->parent_objects, drivelink);
                                memset(&pdata->drive_link[i], 0, sizeof(drive_link_t));
                                pdata->ordering_table[i].child_serial_number = 0;
                                pdata->ordering_table[i].child_vsize         = 0;
                        }
                }

                // restore original drive_link_count
                pdata->drive_link_count = original_link_count;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


int dl_can_set_volume( storage_object_t * object, boolean  flag )
{
        int rc=EINVAL;

        LOG_ENTRY();

        if ( dl_isa_drivelink(object) ) {
                rc = 0;
        }

        LOG_EXIT_INT(rc);
        return rc;
}

int dl_can_delete( storage_object_t * object)
{
        int rc=EINVAL;

        LOG_ENTRY();

        if ( dl_isa_drivelink(object) == TRUE ) {
                rc = 0;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


int dl_can_expand(storage_object_t *drivelink, sector_count_t expand_limit, list_anchor_t expansion_points )
{
        int rc = EINVAL;
        storage_object_t        *x_obj;
        drivelink_private_data_t *pdata;
        expand_object_info_t    *expand_object;
        list_anchor_t acceptable_objects;
        list_element_t iter;
        sector_count_t  dl_expand_size=0;
        storage_object_t *child=NULL;
        sector_count_t  metadata_sectors = (DRIVELINK_METADATA_SECTOR_COUNT*2) + (FEATURE_HEADER_SECTOR_COUNT*2);

        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(drivelink)==TRUE);
        REQUIRE(dl_isa_complete_aggregate(drivelink)==TRUE);
        REQUIRE(expansion_points != NULL);

        acceptable_objects = EngFncs->allocate_list();

        REQUIRE(acceptable_objects != NULL);

        LOG_DEBUG("considering drivelink %s\n", drivelink->name );

        pdata = (drivelink_private_data_t *)drivelink->private_data;

        if (pdata->drive_link_count < EVMS_DRIVELINK_MAX_ENTRIES) {

                rc = EngFncs->get_object_list( 0,
                                                      DATA_TYPE,
                                                      NULL,
                                                      drivelink->disk_group,
                                                      VALID_INPUT_OBJECT,
                                                      &acceptable_objects );

                if (rc==0) {
                        LIST_FOR_EACH(acceptable_objects, iter, x_obj) {
                                if ( x_obj != drivelink &&
                                     x_obj->size > metadata_sectors &&
                                     (x_obj->size - metadata_sectors) <= expand_limit ) {
                                        dl_expand_size += x_obj->size - metadata_sectors;
                                }
                        }
                }

        }

        if (dl_expand_size) {
                expand_object = (expand_object_info_t *) EngFncs->engine_alloc(sizeof(expand_object_info_t));
                if (expand_object) {

                        expand_object->object          = drivelink;
                        expand_object->max_expand_size = min(dl_expand_size, expand_limit);

                        iter = EngFncs->insert_thing(expansion_points,
                                                     expand_object,
                                                     INSERT_AFTER, NULL);

                        if (!iter) {
                                rc = EPERM;
                        }
                }
        }

        child = dl_get_last_child(drivelink);
        if (child) {
                struct plugin_functions_s * fncs = (struct plugin_functions_s *)child->plugin->functions.plugin;
                rc = fncs->can_expand( child, expand_limit, expansion_points );
        }

        if (rc && dl_expand_size==0) {
                LOG_DEBUG("did not find any expansion points for %s\n", drivelink->name);
                rc = ENOMSG;
        }
        else {
                LOG_DEBUG("found expansion points for %s\n", drivelink->name);
                rc = 0;
        }

        LOG_EXIT_INT(rc);
        return rc;
}

int dl_can_expand_by( storage_object_t *object, sector_count_t *size)
{
        int rc=EINVAL;

        LOG_ENTRY();

        if ( dl_isa_drivelink(object) &&
             dl_isa_complete_aggregate(object)==TRUE ) {
                rc = 0;
        }

        LOG_EXIT_INT(rc);
        return rc;
}

int dl_can_shrink( storage_object_t  *drivelink, sector_count_t shrink_limit,  list_anchor_t shrink_points )
{
        int i,rc = EINVAL;
        drivelink_private_data_t  *pdata;
        shrink_object_info_t *shrink_object=NULL;
        storage_object_t *child;
        list_element_t iter;

        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(drivelink));
        REQUIRE(shrink_points != NULL);

        pdata = (drivelink_private_data_t *)drivelink->private_data;

        if ( pdata->drive_link_count > 1 ) {

                if (pdata->drive_link[pdata->drive_link_count - 1].sector_count <= shrink_limit) {

                        shrink_object = (shrink_object_info_t *) EngFncs->engine_alloc( sizeof(shrink_object_info_t) );
                        if (shrink_object) {

                                // Max shrink size is the sum child objects n to 1, as
                                // long as the sum is less than or equal to the shrink
                                // limit.  We must leave at least the 1st object in the
                                // drivelink.

                                shrink_object->object = drivelink;

                                for ( i = pdata->drive_link_count - 1;
                                     (i > 0) &&
                                     (shrink_object->max_shrink_size + pdata->drive_link[i].sector_count <= shrink_limit);
                                      i--) {

                                        shrink_object->max_shrink_size += pdata->drive_link[i].sector_count;
                                }

                                iter = EngFncs->insert_thing(shrink_points,
                                                             shrink_object,
                                                             INSERT_AFTER, NULL);
                                if (!iter) {
                                        rc = EPERM;
                                }
                        }
                        else {
                                LOG_ERROR("error, engine alloc of shrink object failed\n");
                                rc = ENOMEM;
                        }

                }

        }

        child = dl_get_last_child(drivelink);
        if (child) {
                struct plugin_functions_s * fncs = (struct plugin_functions_s *)child->plugin->functions.plugin;
                rc = fncs->can_shrink( child, shrink_limit, shrink_points );
        }

        if (rc && shrink_object==NULL) {
                LOG_DEBUG("did not find any shrink points for %s\n", drivelink->name);
                rc = ENOMSG;
        }
        else {
                LOG_DEBUG("found shrink points for %s\n", drivelink->name);
                rc = 0;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


int dl_can_shrink_by( storage_object_t  *drivelink,  sector_count_t *size)
{
        int rc = EINVAL;
        drivelink_private_data_t *pdata;
        sector_count_t  shrink_size=0;
        sector_count_t  min_object_size=0;


        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(drivelink)==TRUE);

        pdata = (drivelink_private_data_t *) drivelink->private_data;

        if (pdata->drive_link_count == 1) {

                // max size we can shrink an individual object is really
                // 8k of data to keep kernel happy
                // sectors for feature headers
                // sectors for 2 copies of metadata

                min_object_size = (8192/EVMS_VSECTOR_SIZE) + (DRIVELINK_METADATA_SECTOR_COUNT*2) + (FEATURE_HEADER_SECTOR_COUNT*2);

                if ( drivelink->size > min_object_size ) {
                        shrink_size = drivelink->size - min_object_size;
                }
                else {
                        shrink_size = 0;
                }

        }
        else {

                // max shrink size is the sum child objects 1-N
                // this would leave us with a minimum drivelink
                // consisting of just the 1st child object.

                shrink_size = drivelink->size - pdata->drive_link[0].sector_count;
        }

        // test that the child link is sufficiently large to shrink by size sectors
        if ( shrink_size >= *size) {
                rc = 0;
        }
        else {
                *size = shrink_size;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


int dl_can_replace_child( storage_object_t  * object,
                          storage_object_t  * child,
                          storage_object_t  * new_child)
{
        int i, rc;
        drivelink_private_data_t *pdata;
        dot_entry_t  *dot;
        sector_count_t evms_sectors = (DRIVELINK_METADATA_SECTOR_COUNT*2) - (FEATURE_HEADER_SECTOR_COUNT*2);

        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(object)==TRUE);
        REQUIRE(child != NULL);
        REQUIRE(dl_isa_missing_child(child)==FALSE); // till we fix replace of missing child objects that
                                                     // have error mappings that kill the copy job

        pdata = (drivelink_private_data_t *) object->private_data;

        for (i=0,rc=EINVAL; i < pdata->drive_link_count; i++) {

                if (pdata->drive_link[i].object == child){

                        dot = &pdata->ordering_table[i];

                        if (new_child != NULL) {
                                if ( new_child->size >= (dot->child_vsize + evms_sectors) &&
                                     new_child->disk_group == child->disk_group ) {
                                        rc = 0;
                                }
                        }
                        else {
                                rc = 0;
                        }
                        break;
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


int dl_can_add_feature(storage_object_t * input_object, sector_count_t * size)
{
        int rc = 0;

        LOG_ENTRY();

        if (input_object->data_type == DATA_TYPE) {
                /* Subtract off meta data size. */
                *size = input_object->size - (DRIVELINK_METADATA_SECTOR_COUNT*2) - (FEATURE_HEADER_SECTOR_COUNT*2);

                /* Round down to 8K (16 512 byte sectors). */
                *size &= ~15;

        }
        else {
                /* The object is not a DATA_TYPE. */
                rc = EINVAL;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: dl_get_create_options
 *
 *  Called to parse option descriptors and get the name of
 *  the new drivelink object.
 */
static void dl_get_create_options( option_array_t * options,  char  * drive_link_name  )
{
        int i;

        LOG_ENTRY();

        for (i = 0; i < options->count; i++) {

                if (options->option[i].is_number_based) {

                        if (options->option[i].number == DL_CREATE_LINKNAME_INDEX ) {
                                strncpy( drive_link_name, options->option[i].value.s, EVMS_VOLUME_NAME_SIZE );
                        }

                }
                else {

                        if (strcmp(options->option[i].name,  DL_CREATE_LINKNAME_NAME ) == 0) {
                                strncpy( drive_link_name, options->option[i].value.s, EVMS_VOLUME_NAME_SIZE );
                        }

                }

        }

        LOG_EXIT_VOID();
}


/*
 *  Function: dl_create
 *
 *  To create a drive link object I check that:
 *
 *  - the request link count doesnt exceed MAX drive link count
 *  - that there are enough storage objects in the list to satisfy the
 *    requested number of links
 *
 *  I then create a new drive link storage object and consume child
 *  objects found in the objects list till we reach the requested
 *  link count.
 *
 *  Returns: the address of the new drive link object if successful
 *
 */
int dl_create( list_anchor_t input_objects, option_array_t *options,  list_anchor_t output_objects )
{
        int rc=0, drive_link_count=0;
        storage_object_t  *drivelink;
        char   drive_link_name[EVMS_VOLUME_NAME_SIZE+1]="";
        drivelink_private_data_t *pdata=NULL;
        storage_object_t *child = EngFncs->first_thing(input_objects, NULL);
        list_element_t iter;

        LOG_ENTRY();

        // collect create drivelink info
        dl_get_create_options( options, drive_link_name );
        drive_link_count = EngFncs->list_count(input_objects);

        REQUIRE(drive_link_count > 0);
        REQUIRE(drive_link_count <= EVMS_DRIVELINK_MAX_ENTRIES);
        REQUIRE(strlen(drive_link_name)>0);
        REQUIRE(child);

        // allocate a new drive link storage object
        drivelink = dl_malloc_drivelink_object();

        if (drivelink == NULL) {
                rc=ENOMEM;
                LOG_EXIT_INT(rc);
                return rc;
        }

        pdata = (drivelink_private_data_t *) drivelink->private_data;

        drivelink->disk_group = child->disk_group;

        // build cluster ready object name
        if (drivelink->disk_group) {
                strncpy(drivelink->name, drivelink->disk_group->name, EVMS_NAME_SIZE);
                strncat(drivelink->name, "/", EVMS_NAME_SIZE-strlen(drivelink->name));
        }
        strncat(drivelink->name, drive_link_name, EVMS_NAME_SIZE-strlen(drivelink->name));

        strncpy(pdata->parent_object_name, drive_link_name, EVMS_NAME_SIZE);

        // no links yet
        pdata->drive_link_count  = 0;

        // geta drive link parent serial number
        pdata->parent_serial_number = dl_gen_parent_serial_number( (u_int32_t) ((unsigned long)drivelink) );
        if (pdata->parent_serial_number == BAD_SERIAL_NUMBER) {
                rc = ENOMEM;
                dl_free_drivelink_object(drivelink);
                LOG_EXIT_INT(rc);
                return rc;
        }

        // register parent drivelink object name
        if (!rc) {
                if ( EngFncs->register_name(drivelink->name) ) {
                        rc = EINVAL;
                        dl_unregister_serial_number(pdata->parent_serial_number);
                        dl_free_drivelink_object(drivelink);
                        LOG_EXIT_INT(rc);
                        return rc;
                }
        }

        // Now, consume objects from the input objects list
        rc = dl_expand_drivelink(drivelink, input_objects );

        if (rc==0) {
                iter = EngFncs->insert_thing(output_objects, drivelink,
                                             INSERT_BEFORE, NULL);
                if (!iter) {
                        rc = ENOMEM;
                }
        }

        if (rc==0) {
                dl_build_linear_mapping(drivelink);
                dl_setup_geometry(drivelink);   // call routine to figure out a geometry for the aggregate
        }
        else {
                EngFncs->unregister_name(drivelink->name);
                dl_unregister_serial_number(pdata->parent_serial_number);
                dl_free_drivelink_object(drivelink);
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  dl_w_delete
 *
 *  Worker function for dl_delete and dl_discard
 */
int dl_w_delete( storage_object_t * object, list_anchor_t  child_objects, boolean destroy )
{

	int i;
	drivelink_private_data_t *pdata;
	storage_object_t          *child;
	struct plugin_functions_s *fncs;
	evms_feature_header_t     *fh;

	LOG_ENTRY();

	REQUIRE(dl_isa_drivelink(object)==TRUE);

	pdata = (drivelink_private_data_t *) object->private_data;

	for (i=0; i<pdata->drive_link_count; i++ ) {

		child = pdata->drive_link[i].object;

		if (child==NULL)
			continue;

		if ( dl_isa_missing_child(child)==TRUE ) {
			dl_free_missing_child_object(child);
		}
		else {
			if (destroy) {
				// zap our metadata
				fncs = (struct plugin_functions_s *)child->plugin->functions.plugin;
				fh   = child->feature_header;

				fncs->add_sectors_to_kill_list(
					child,
					fh->feature_data1_start_lsn,
					fh->feature_data1_size );

				if (fh->feature_data2_size>0) {
					fncs->add_sectors_to_kill_list(
						child,
						fh->feature_data2_start_lsn,
						fh->feature_data2_size );
				}
			}
					
			// remove drivelink object reference in child parent object list
			EngFncs->remove_thing(child->parent_objects, object);

			// place available child object in engine list
			if (child_objects)
				EngFncs->insert_thing(child_objects, child, INSERT_BEFORE, NULL);
		}
	}

	// free drive link object if we successfully freed up child objects.
	dl_free_drivelink_object( object );

	LOG_EXIT_INT(0);
	return 0;
}

/*
 *  Function:  dl_delete
 *
 *  - Remove the feature from each child object by blasting the feature header sector
 *  - Free any privately allocated data.
 *  - Remove the parent pointer from each child object
 *  - Put child onto the child_objects list provided in the second parameter.
 *
 */
int dl_delete(storage_object_t *object, list_anchor_t child_objects)
{
	LOG_ENTRY();

	dl_w_delete(object, child_objects, TRUE);
	LOG_EXIT_INT(0);
	return 0;
}

/*
 * Function: dl_discard
 *
 * This function is similar to delete.  Just call delete to free all
 * data structures related to the DriveLink objects.
 */
int dl_discard(list_anchor_t objects)
{
	storage_object_t * object;
	list_element_t le;

	LOG_ENTRY();

	LIST_FOR_EACH(objects, le, object) {
		dl_w_delete(object, NULL, FALSE);
	}

	LOG_EXIT_INT(0);
	return 0;
}

/*
 *  Function: dl_expand
 *
 *  A drive link is expanded by adding all the objects in the
 *  objects list to the end of an existing drive link.
 *
 *  I check that:
 *
 *  - I own the object
 *  - there are objects in the list we can add to the drive link
 *  - adding the objects wont exceed the capacity of a drive link
 *
 *  I expand the drive link by consuming the child objects from the
 *  objects list and adding them to the drive link object. They are
 *  removed from the objects list in no particular order but are
 *  added at the end of the drive link object ... expanding by adding
 *  sectors at the end of the drive link.
 */
int dl_expand( storage_object_t * drivelink,
               storage_object_t * expand_object,
               list_anchor_t      objects,
               option_array_t   * options )
{
        int  rc;
        drivelink_private_data_t *pdata;
        uint  expand_link_count=0;
        sector_count_t             child_useable_size;
        sector_count_t             padding_sectors;
        storage_object_t          *child;
        struct plugin_functions_s *funcs;
        int index;

        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(drivelink)==TRUE);
        REQUIRE(expand_object != NULL);
        REQUIRE(objects != NULL);

        LOG_DEBUG("expanding drivelink %s\n", drivelink->name);
        LOG_DEBUG("expand object is %s\n", expand_object->name);

        pdata = (drivelink_private_data_t *) drivelink->private_data;

        if (drivelink == expand_object) {

                expand_link_count = EngFncs->list_count(objects);

                REQUIRE( expand_link_count > 0 );

                REQUIRE( pdata->drive_link_count + expand_link_count <= EVMS_DRIVELINK_MAX_ENTRIES );

                rc = dl_expand_drivelink( drivelink, objects );
        }
        else {
                rc = ENODEV;
                child = dl_get_last_child( drivelink );
                if (child) {
                        funcs = (struct plugin_functions_s *)child->plugin->functions.plugin;
                        if (funcs) {
                                rc = funcs->expand( child,
                                                    expand_object,
                                                    objects,
                                                    options);
                        }
                }
                if (rc == 0) {
                        child_useable_size  = child->size - (DRIVELINK_METADATA_SECTOR_COUNT*2) - (FEATURE_HEADER_SECTOR_COUNT*2);

                        // drive link child objects must be sized to fall
                        // on an 8k boundary ... kernel requirement so
                        // individual i/o requests wont span links.
                        padding_sectors     = child_useable_size % 16;
                        child_useable_size -= padding_sectors;

                        index = pdata->drive_link_count - 1;
                        pdata->drive_link[index].sector_count       = child_useable_size;
                        pdata->drive_link[index].padding            = padding_sectors;
                        pdata->ordering_table[index].child_vsize    = child_useable_size;

                }
        }

        if (rc==0) {
                drivelink->flags |= SOFLAG_DIRTY;
		if (drivelink->flags & SOFLAG_ACTIVE) {
			drivelink->flags |= SOFLAG_NEEDS_ACTIVATE;
		}
                dl_build_linear_mapping(drivelink);
                dl_setup_geometry(drivelink);
        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Function:  dl_shrink
 *
 *  Called by the engine to shrink a drive link object by removing child
 *  objects from the parent storage object and reflecting this in the
 *  drive link private data.
 *
 *  I check that:
 *
 *  - I own the object
 *  - Only 1 shrink point is specified
 *
 *  I shrink by removing the last child in the drivelink and placing it
 *  in the objects list -OR- if there is only 1 child in the link then
 *  I pass the shrink command down to the child object and see if it can
 *  do the shrink.
 */

int dl_shrink( storage_object_t * drivelink,
               storage_object_t * shrink_object,
               list_anchor_t      objects,
               option_array_t   * options )
{
        int index, i, rc = EINVAL;
        storage_object_t          *child;
        drivelink_private_data_t  *pdata=NULL;
        boolean                    object_ok;
        int                        search_starting_index=0;
        uint                       selected_objects_count=0;
        sector_count_t             child_useable_size;
        sector_count_t             padding_sectors;
        sector_count_t             shrink_size = 0;
        list_element_t             iter;


        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(drivelink)==TRUE);

        LOG_DEBUG("object= %s   shrink object= %s\n", drivelink->name, shrink_object->name );

        pdata = (drivelink_private_data_t *) drivelink->private_data;

        // check if we are the shrink point ...
        if ( drivelink == shrink_object ) {

                selected_objects_count = EngFncs->list_count(objects);

                REQUIRE(selected_objects_count > 0);
                REQUIRE(pdata->drive_link_count >= selected_objects_count);

                // make sure all the selected shrink child objects are at the end of the
                // drivelink. we cant shrink by removing objects from the middle or front
                // of the linear drivelink object.

                search_starting_index = pdata->drive_link_count - selected_objects_count;

                LIST_FOR_EACH(objects, iter, child) {
                        object_ok = FALSE;
                        for (i=search_starting_index; i<pdata->drive_link_count; i++) {
                                if (child == pdata->drive_link[i].object) {
                                        object_ok = TRUE;
                                        break;
                                }
                        }
                        if (object_ok==FALSE) {
                                LOG_ERROR("error, declining object %s because it is in the middle/front of the drivelink\n", child->name);
                                rc = EINVAL;
                                LOG_EXIT_INT(rc);
                                return rc;
                        }
                }

		// Check with the engine before starting the expand.
		for (i = 1; i <= selected_objects_count; i++) {
			shrink_size += pdata->drive_link[pdata->drive_link_count - i].sector_count;
		}
		rc = EngFncs->can_shrink_by(drivelink, &shrink_size);
		if (rc) {
			LOG_ERROR("Shrink of object %s rejected by the engine.\n", drivelink->name);
			LOG_EXIT_INT(rc);
			return rc;
		}

                // now do the shrink ... always removing last child in drivelink
                for (i=0,rc=0; i<selected_objects_count && rc==0; i++) {

                        child = dl_get_last_child(drivelink);
                        if (child) {
                                rc = dl_shrink_drivelink( drivelink, child );
                        }
                        else {
                                rc = ENODEV;
                        }

                        if ( rc==0 && dl_isa_missing_child(child)==TRUE ) {
                                EngFncs->remove_thing(objects, child);
                                dl_free_missing_child_object(child);
                        }

                }

        }
        else {

                child = dl_get_last_child(drivelink);
                if (child) {
                        struct plugin_functions_s * fncs = (struct plugin_functions_s *) child->plugin->functions.plugin;
                        rc = fncs->shrink(child, shrink_object, objects, options );
                }
                else {
                        rc = ENODEV;
                }

                if (rc == 0) {

                        child_useable_size  = child->size - (DRIVELINK_METADATA_SECTOR_COUNT*2) - (FEATURE_HEADER_SECTOR_COUNT*2);

                        // drive link child objects must be sized to fall
                        // on an 8k boundary ... kernel requirement so
                        // individual i/o requests wont span links.
                        padding_sectors     = child_useable_size % 16;
                        child_useable_size -= padding_sectors;

                        index = pdata->drive_link_count - 1;
                        pdata->drive_link[index].sector_count       = child_useable_size;
                        pdata->drive_link[index].padding            = padding_sectors;
                        pdata->ordering_table[index].child_vsize    = child_useable_size;

                }

        }

        if ( rc==0 ) {
                dl_build_linear_mapping(drivelink);
                dl_setup_geometry(drivelink);
                drivelink->flags |= SOFLAG_DIRTY;
		if (drivelink->flags & SOFLAG_ACTIVE) {
			drivelink->flags |= SOFLAG_NEEDS_ACTIVATE;
		}
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  dl_replace_child
 *
 *  At this point the replace plugin has already committed its replace
 *  object and the copy has been attempted.  If the copy failed, then
 *  the new_child object will be the original drivelink child.  If the
 *  copy was successful, then the new_child object will indeed be the
 *  new child object chosen for the replace.
 */
int dl_replace_child( storage_object_t * object,
                      storage_object_t * child,
                      storage_object_t * new_child)
{
        int  i, index, rc = EINVAL;
        drivelink_private_data_t  *pdata;
        struct plugin_functions_s *fncs=NULL;
        evms_feature_header_t     *fh;
        list_element_t             iter;

        LOG_ENTRY();

        REQUIRE( dl_isa_drivelink(object) == TRUE );
        REQUIRE( child != NULL);
        REQUIRE( new_child != NULL );
        REQUIRE( new_child->disk_group == object->disk_group);

        LOG_DEBUG("drivelink= %s  child= %s  new_child= %s\n", object->name, child->name, new_child->name);

        pdata = (drivelink_private_data_t *) object->private_data;

        // find the child in our drivelink
        for (i=0,index=-1; i<pdata->drive_link_count; i++) {
                if (pdata->drive_link[i].object == child) {
                        index=i;
                        break;
                }
        }

        REQUIRE(index>=0);

        if (new_child->feature_header == NULL) {
                new_child->feature_header = (evms_feature_header_t *) EngFncs->engine_alloc( sizeof(evms_feature_header_t) );
        }

        REQUIRE(new_child->feature_header != NULL);

        rc = dl_build_feature_header( object, &(pdata->drive_link[index]), new_child->feature_header );

        if (rc) {
                LOG_ERROR("error building feature header for new child object\n");
                LOG_EXIT_INT(rc);
                return rc;
        }

        iter = EngFncs->insert_thing(new_child->parent_objects, object,
                                     INSERT_AFTER, NULL);
        if (!iter) {
                rc = ENOMEM;
        }

        if (rc==0) {
                pdata->drive_link[index].object = new_child;
                dl_build_ordered_child_object_list( object, &object->child_objects );
                object->flags |= SOFLAG_DIRTY;
		if (object->flags & SOFLAG_ACTIVE) {
			object->flags |= SOFLAG_NEEDS_ACTIVATE;
		}

                // remove parent from child parent list
                EngFncs->remove_thing(child->parent_objects, object);

                // wipe out the drivelink metadata
                fncs = (struct plugin_functions_s *)child->plugin->functions.plugin;
                fh   = child->feature_header;

                if (dl_isa_missing_child(child)==FALSE) {

                        fncs->add_sectors_to_kill_list( child,
                                                        fh->feature_data1_start_lsn,
                                                        fh->feature_data1_size );

                        if (fh->feature_data2_size > 0) {
                                fncs->add_sectors_to_kill_list( child,
                                                                fh->feature_data2_start_lsn,
                                                                fh->feature_data2_size );
                        }
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: dl_add_sectors_to_kill_list
 *
 *  Passes the API call down to the appropriate drive link child object.
 *
 *  I check that:
 *
 *  - I own the object
 *  - the logical sectors are found in the drive link
 *
 *  I do it by walking the drive link and determining the child storage
 *  object that contains the specified sectors and then forwarding the
 *  API call to that object.
 */
int dl_add_sectors_to_kill_list( storage_object_t * object,
                                 lsn_t              lsn,
                                 sector_count_t     count)
{
        int i, rc = EINVAL;
        lsn_t                      io_lsn = lsn;
        lsn_t                      link_lsn;
        sector_count_t             io_sector_count;
        sector_count_t             sectors_left_to_write = count;
        sector_count_t             max_sector_count;
        drivelink_private_data_t  *pdata;
        storage_object_t          *child;
        struct plugin_functions_s *fncs;


        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(object)==TRUE);
        REQUIRE(lsn+count <= object->size);

        pdata = (drivelink_private_data_t *) object->private_data;

        for (i=0; i < pdata->drive_link_count; i++) {

                if ( pdata->drive_link[i].end_lsn >= io_lsn ) {

                        child = pdata->drive_link[i].object;

                        max_sector_count = pdata->drive_link[i].end_lsn - io_lsn + 1;

                        if ( max_sector_count >= sectors_left_to_write ) {
                                io_sector_count = sectors_left_to_write;
                        }
                        else {
                                io_sector_count = max_sector_count;
                        }

                        link_lsn = io_lsn - pdata->drive_link[i].start_lsn;

                        if ( dl_isa_missing_child(child)==TRUE ) {
                                rc=0;
                        }
                        else {
                                fncs = (struct plugin_functions_s *)child->plugin->functions.plugin;
                                rc = fncs->add_sectors_to_kill_list( child, link_lsn, io_sector_count);
                        }

                        io_lsn                += io_sector_count;
                        sectors_left_to_write -= io_sector_count;

                        if ((sectors_left_to_write == 0) || (rc)) break;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  dl_read
 *
 *  Passes the API call down to the appropriate drive link child object.
 *
 *  I check that:
 *
 *  - I own the object
 *  - I have a valid buffer
 *  - the logical sectors are found in the drive link
 *
 *  I do the I/O by walking the drive link and determining the child storage
 *  object that contains the specified sectors and then forwarding the
 *  API call to that object. The I/O may span multiple objects and this is
 *  Ok because the I/O will be broken up and multiple Reads performed as
 *  needed.
 */
int dl_read(storage_object_t *object, lsn_t lsn, sector_count_t count, void *buffer)
{
        int i, rc = EINVAL;
        char                      *io_buffer_ptr = (char *)buffer;
        lsn_t                      io_lsn = lsn;
        lsn_t                      link_lsn;
        sector_count_t             io_sector_count;
        sector_count_t             sectors_left_to_write = count;
        sector_count_t             max_sector_count;
        drivelink_private_data_t  *pdata;
        storage_object_t          *child;

        LOG_ENTRY();
        LOG_DEBUG("drivelink= %s  size = %"PRIu64"  lsn= %"PRIu64"  count= %"PRIu64"\n",
                    object->name, object->size, lsn, count);

        REQUIRE(buffer != NULL);
        REQUIRE(lsn+count <= object->size);

        if (dl_isa_missing_child(object)==TRUE) {
                memset(buffer, 0, count*EVMS_VSECTOR_SIZE);
                LOG_EXIT_INT(0);
                return 0;
        }

        REQUIRE(dl_isa_drivelink(object)==TRUE);

        pdata = (drivelink_private_data_t *) object->private_data;

        for (i=0; i < pdata->drive_link_count; i++) {

                if ( pdata->drive_link[i].end_lsn >= io_lsn ) {

                        LOG_DEBUG("\tlsn is in link %d cux link has end_lsn of %d\n", i, (u_int32_t)pdata->drive_link[i].end_lsn);

                        child = pdata->drive_link[i].object;

                        max_sector_count = pdata->drive_link[i].end_lsn - io_lsn + 1;

                        if ( max_sector_count >= sectors_left_to_write ) {
                                io_sector_count = sectors_left_to_write;
                        }
                        else {
                                io_sector_count = max_sector_count;
                        }

                        link_lsn = io_lsn - pdata->drive_link[i].start_lsn;

                        if ( dl_isa_missing_child(child)==TRUE ) {
                                rc = EIO;
                        }
                        else {
                                rc=READ( child, link_lsn, io_sector_count, (void *) io_buffer_ptr );
                        }

                        io_lsn                += io_sector_count;
                        io_buffer_ptr         += io_sector_count*EVMS_VSECTOR_SIZE;
                        sectors_left_to_write -= io_sector_count;

                        if ((sectors_left_to_write == 0) || (rc)) break;
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  dl_write
 *
 *  Passes the API call down to the appropriate drive link child object.
 *
 *  I check that:
 *
 *  - I own the object
 *  - I have a valid buffer
 *  - the logical sectors are found in the drive link
 *
 *  I do the I/O by walking the drive link and determining the child storage
 *  object that contains the specified sectors and then forwarding the
 *  API call to that object. The I/O may span 2 or more child objects and
 *  this is Ok because the I/O will be broken up into multiple Writes as
 *  needed.
 */
int dl_write(storage_object_t *object, lsn_t lsn, sector_count_t count, void *buffer)
{
        int i, rc = EINVAL;
        char                      *io_buffer_ptr = (char *)buffer;
        lsn_t                      io_lsn = lsn;
        lsn_t                      link_lsn;
        sector_count_t             io_sector_count;
        sector_count_t             sectors_left_to_write = count;
        sector_count_t             max_sector_count;
        drivelink_private_data_t  *pdata;
        storage_object_t          *child;


        LOG_ENTRY();
        LOG_DEBUG("drivelink= %s  size = %"PRIu64"  lsn= %"PRIu64"  count= %"PRIu64"\n",
                   object->name, object->size, lsn, count);

        REQUIRE(buffer != NULL);
        REQUIRE(lsn+count <= object->size);

        if (dl_isa_missing_child(object)==TRUE) {
                LOG_EXIT_INT(0);
                return 0;
        }

        REQUIRE(dl_isa_drivelink(object)==TRUE);

        if ( object->flags & SOFLAG_READ_ONLY ) {
                rc = EROFS;
                LOG_EXIT_INT(rc);
                return rc;
        }

        pdata = (drivelink_private_data_t *) object->private_data;

        for (i=0; i < pdata->drive_link_count; i++) {

                if ( pdata->drive_link[i].end_lsn >= io_lsn ) {

                        LOG_DEBUG("\tlsn is in link %d cux link has end_lsn of %d\n", i, (u_int32_t)pdata->drive_link[i].end_lsn);

                        child = pdata->drive_link[i].object;

                        max_sector_count = pdata->drive_link[i].end_lsn - io_lsn + 1;

                        if ( max_sector_count >= sectors_left_to_write ) {
                                io_sector_count = sectors_left_to_write;
                        }
                        else {
                                io_sector_count = max_sector_count;
                        }

                        link_lsn = io_lsn - pdata->drive_link[i].start_lsn;

                        if ( dl_isa_missing_child(child)==TRUE ) {
                                rc = EIO;
                        }
                        else {
                                rc=WRITE( child, link_lsn, io_sector_count, (void *) io_buffer_ptr );
                        }

                        io_lsn                += io_sector_count;
                        io_buffer_ptr         += io_sector_count*EVMS_VSECTOR_SIZE;
                        sectors_left_to_write -= io_sector_count;

                        if ((sectors_left_to_write == 0) || (rc)) break;
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


void dl_set_volume( storage_object_t * object, boolean  flag )
{
    LOG_ENTRY();
    // Nothing to do yet.
    LOG_EXIT_VOID();
}


/*
 * Execute the private action on the object.
 */
int dl_plugin_function( storage_object_t * missing_child,
                        task_action_t      action,
                        list_anchor_t      objects,
                        option_array_t   * options )
{
        int rc=EINVAL;
        storage_object_t *new_child=NULL;
        storage_object_t *drivelink=NULL;

        LOG_ENTRY();

        REQUIRE(dl_isa_missing_child(missing_child)==TRUE);
        REQUIRE(objects != NULL);
        REQUIRE(options != NULL);

        drivelink = dl_get_parent(missing_child);

        REQUIRE(drivelink != NULL);

        switch (action) {
                case EVMS_Replace_Missing_DL_Child:
                        new_child = EngFncs->first_thing(objects, NULL);
                        if (new_child) {
                                rc = dl_replace_missing_child( drivelink,
                                                               missing_child,
                                                               new_child);
                                if (rc==0) {
                                        dl_free_missing_child_object(missing_child);
                                }
                        }
                        break;

                default:
                        rc = ENOSYS;
                        break;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Return an array of private actions that you support for this object.
 */
int dl_get_plugin_functions( storage_object_t        * object,
                             function_info_array_t * * actions)
{
        int rc = EINVAL;
        function_info_array_t *func_info=NULL;

        LOG_ENTRY();

        REQUIRE(dl_isa_missing_child(object)==TRUE);
        REQUIRE(actions != NULL);
        REQUIRE(dl_can_replace_missing_child(object)==TRUE);

        func_info = EngFncs->engine_alloc( sizeof(function_info_array_t) + sizeof(function_info_t) );
        if (func_info) {

                func_info->count = 0;
                func_info->count = 1;

                func_info->info[DL_REPLACE_MISSING_CHILD_OPTION_INDEX].function = EVMS_Replace_Missing_DL_Child;

                func_info->info[DL_REPLACE_MISSING_CHILD_OPTION_INDEX].title = EngFncs->engine_strdup( "Replace Missing Child" );
                func_info->info[DL_REPLACE_MISSING_CHILD_OPTION_INDEX].verb = EngFncs->engine_strdup( _("Replace") );
                func_info->info[DL_REPLACE_MISSING_CHILD_OPTION_INDEX].name = EngFncs->engine_strdup( _("Replace Missing Child") );
                func_info->info[DL_REPLACE_MISSING_CHILD_OPTION_INDEX].help = EngFncs->engine_strdup( _("Use this function to replace a missing drivelink child with an available storage object.") );

                rc = 0;
        }
        else {
                rc = ENOMEM;
        }

        *actions = func_info;

        LOG_EXIT_INT(rc);
        return rc;
}

