/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003, 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: flatten.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>

#include "fullengine.h"
#include "dmonmsg.h"
#include "engine.h"
#include "memman.h"
#include "object.h"
#include "option.h"
#include "common.h"


/*
 * Utility functions
 */

static char * valid_format_characters =
	boolean_f
	count_f
	count32_f
	int_f
	u_int8_f
	u_int16_f
	u_int32_f
	u_int64_f
	ptr_f
	string_f
	strings_f
	bytes_f
	handle_array_f
	handle_object_info_f
	expand_handle_array_f
	shrink_handle_array_f
	change_record_array_f
	function_info_array_f
	declined_handle_array_f
	value_f
	option_array_f
	option_descriptor_f
	extended_info_array_f
	"0123456789"
	"[]{}";


static int validate_format(char * format) {

	int rc = 0;
	char * pch = format;
	int i, j;
	char stack[32];

	LOG_PROC_ENTRY();

	while (*pch != '\0') {
		char * p;

		p = strchr(valid_format_characters, *pch);
		if (p == NULL) {
			LOG_ERROR("Invalid format character %c at offset %d in format string \"%s\".\n",
				  *pch, (int)(pch - format), format);
			rc = EINVAL;
			pch++;

		} else {
			switch (*pch) {
				case count_c:
				case count32_c:
					/* The next character must be a '['. */
					if (*(pch + 1) != '[') {
						LOG_ERROR("A bracket ('[') is required after the %c at offset %d in the format string \"%s\".\n",
							  *pch, (int)(pch - format), format);
						rc = EINVAL;
						pch++;

					} else {
						/*
						 * Skip over the 'c' or 'C' and
						 * the following bracket.
						 */
						pch +=2;
					}
					break;

				case bytes_c:
					{
						/* Skip over the digits that follow. */
						int num_digits;

						pch++;
						num_digits = strspn(pch, "0123456789");
						pch += num_digits;
					}
					break;

				default:
					pch++;
			}
		}
	}

	/*
	 * Make sure the number of open brackets/braces matches the number of
	 * close brackets/braces and that they are nested properly.
	 */
	pch = format;
	i = 0;
	while (*pch != '\0') {
		switch (*pch) {
			case '[':
				stack[i] = ']';
				i++;
				break;

			case ']':
				if (i <= 0) {
					LOG_ERROR("Format \"%s\" has a close bracket (']') at offset %d without a matching open bracket ('[).\n",
						  format, (int)(pch - format));
					rc = EINVAL;

				} else {
					if (stack[i-1] != ']') {
						LOG_ERROR("Expecting a '%c' before the close bracket (']') at offset %d in format \"%s\".\n",
							  stack[i-1], (int)(pch - format), format);
						rc = EINVAL;

					} else {
						i--;
					}
				}
				break;

			case '{':
				stack[i] = '}';
				i++;
				break;

			case '}':
				if (i <= 0) {
					LOG_ERROR("Format \"%s\" has a close brace ('}') at offset %d without a matching open brace ('{').\n",
						  format, (int)(pch - format));
					rc = EINVAL;

				} else {
					if (stack[i-1] != '}') {
						LOG_ERROR("Expecting a '%c' before the close brace ('}') at offset %d in format \"%s\".\n",
							  stack[i-1], (int)(pch - format), format);
						rc = EINVAL;

					} else {
						i--;
					}
				}
				break;
		}
		pch++;
	}

	if (i > 0) {
		for (j = 0; j < i; j++) {
			LOG_ERROR("Format \"%s\" is missing a '%c'.\n", format, stack[j]);
		}
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static char * closing_bracket(char * format) {

	char * pch;
	uint bracket_count;

	for (pch = format, bracket_count = 0;
	    (*pch != '\0') && ((*pch != ']') || (bracket_count != 0));
	    pch++) {
		if (*pch == '[') {
			bracket_count ++;

		} else if (*pch == ']') {
			bracket_count--;
		}
	}

	return pch;
}


static char * closing_brace(char * format) {

	char * pch;
	uint brace_count;

	for (pch = format, brace_count = 0;
	    (*pch != '\0') && ((*pch != '}') || (brace_count != 0));
	    pch++) {
		if (*pch == '{') {
			brace_count ++;

		} else if (*pch == '}') {
			brace_count--;
		}
	}

	return pch;
}


/*
 * Functions for calculating the size needed for a buffer to contain
 * the network stream for given data structures.
 */

static size_t sizeof_netbuf_value(value_t value, value_type_t type, boolean value_is_list) {

	size_t size = sizeof(u_int8_t);	/* boolean that says if the value is a list */
	uint count;
	value_t * p_value;
	int i;

	LOG_PROC_ENTRY();

	if (value_is_list) {
		/* First byte says if the list pointer is not NULL. */
		size += sizeof(u_int8_t);

		if (value.list != NULL) {
			value_list_t * vl = (value_list_t *) value.list;
			count = vl->count;
			p_value = &vl->value[0];
			size += sizeof(u_int32_t);	/* Room for the count */

		} else {
			/* That's all folks. */

			LOG_PROC_EXIT_INT((int)size);
			return size;
		}

	} else {
		count = 1;
		p_value = &value;
	}

	for (i = 0; i < count; i++, p_value++) {
		switch (type) {
			case EVMS_Type_String:
				/*
				 * One for the indicator that says if the string pointer
				 * is NULL.
				 */
				size += sizeof(u_int8_t);

				if ((*p_value).s != NULL) {
					size +=  strlen(value.s) + sizeof(char);
				}
				break;

			case EVMS_Type_Boolean:
			case EVMS_Type_Char:
			case EVMS_Type_Unsigned_Char:
			case EVMS_Type_Int8:
			case EVMS_Type_Unsigned_Int8:
				size += sizeof(u_int8_t);
				break;

			case EVMS_Type_Int16:
			case EVMS_Type_Unsigned_Int16:
				size += sizeof(u_int16_t);
				break;

			case EVMS_Type_Real32:
			case EVMS_Type_Int:
			case EVMS_Type_Int32:
			case EVMS_Type_Unsigned_Int:
			case EVMS_Type_Unsigned_Int32:
				size += sizeof(u_int32_t);
				break;

			case EVMS_Type_Real64:
			case EVMS_Type_Int64:
			case EVMS_Type_Unsigned_Int64:
				size += sizeof(u_int64_t);
				break;
		}
	}

	LOG_PROC_EXIT_INT((int)size);
	return size;
}


static size_t sizeof_netbuf_collection(value_collection_t coll,
				       collection_type_t  coll_type,
				       value_type_t       value_type) {

	size_t size = 0;
	int i;

	LOG_PROC_ENTRY();

	switch (coll_type) {
		case EVMS_Collection_None:
			break;

		case EVMS_Collection_List:
			size += sizeof(u_int32_t);	/* od->constraint.list->count */
			if (value_type == EVMS_Type_String) {
				for (i = 0; i < coll.list->count; i++) {
					size += sizeof_netbuf_value(coll.list->value[i], value_type, FALSE);
				}

			} else {
				size += coll.list->count * sizeof_netbuf_value(coll.list->value[0], value_type, FALSE);
			}
			break;

		case EVMS_Collection_Range:
			/*
			 * Don't have to worry about the value being of type
			 * EVMS_Type_String.  String cannot be used as ranges.
			 */
			size += sizeof_netbuf_value(coll.range->min, value_type, FALSE)
				* 3;  /* coll->range->min       */
				      /* coll->range->max       */
				      /* coll->range->increment */
			break;
	}

	LOG_PROC_EXIT_INT((int)size);
	return size;
}


static void * sizeof_hostbuf_to_netbuf(char * format,
				       void * host_buf,
				       size_t * psize) {

	char * pf = format;
	char curr_f;
	size_t size = 0;
	size_t curr_size;
	int i;
	uint len;
	u_int32_t count;
	char * pch;
	char * string;
	char * * strings;
	char tmp_format[64];
	handle_array_t * ha;
	handle_object_info_t * hoi;
	expand_handle_array_t * eha;
	shrink_handle_array_t * sha;
	change_record_array_t * cra;
	declined_handle_array_t * dha;
	function_info_array_t * fia;
	value_type_t value_type;
	boolean value_is_list;
	option_array_t * oa;
	option_descriptor_t * od;
	extended_info_array_t * eia;

	LOG_PROC_ENTRY();

	LOG_EXTRA("Format is: %s\n", format);

	while (*pf != '\0') {
		curr_size = size;
		curr_f = *pf;

		switch (*pf) {
			case boolean_c:
				size += sizeof(boolean);
				pf++;
				break;

			case count_c:
				/* First is the int of the count. */
				count = (u_int32_t) *((int *) host_buf);

				size += sizeof(u_int32_t);
				host_buf += sizeof(int);

				/* Skip over "c[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					/* call myself recursively to process the substring. */
					host_buf = sizeof_hostbuf_to_netbuf(tmp_format, host_buf, &size);
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case count32_c:
				/* First is the u_int32_t of the count. */
				count = *((u_int32_t *) host_buf);

				size += sizeof(u_int32_t);
				host_buf += sizeof(u_int32_t);

				/* Skip over "C[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					/* call myself recursively to process the substring. */
					host_buf = sizeof_hostbuf_to_netbuf(tmp_format, host_buf, &size);
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case int_c:
				size += sizeof(u_int32_t);
				host_buf += sizeof(int);
				pf++;
				break;

			case u_int8_c:
				size += sizeof(u_int8_t);
				host_buf += sizeof(u_int8_t);
				pf++;
				break;

			case u_int16_c:
				size += sizeof(u_int16_t);
				host_buf += sizeof(u_int16_t);
				pf++;
				break;

			case u_int32_c:
				size += sizeof(u_int32_t);
				host_buf += sizeof(u_int32_t);
				pf++;
				break;

			case u_int64_c:
				size += sizeof(u_int64_t);
				host_buf += sizeof(u_int64_t);
				pf++;
				break;

			case ptr_c:
				if (*(pf + 1) == '{') {
					void * sub_buf = *((int * *) host_buf);

					size += sizeof(u_int8_t);

					if (sub_buf != NULL) {
					}
					/* Skip over "p{". */
					pf += 2;

					/* Parse out the substring between the braces. */
					strcpy(tmp_format, pf);
					pch = closing_brace(tmp_format);
					*pch = '\0';

					/* tmp_format now has the substring between the braces. */

					if (sub_buf != NULL) {
						/* Call myself recursively to process the substring. */
						sizeof_hostbuf_to_netbuf(tmp_format, sub_buf, &size);
					}

					pf += strlen(tmp_format) + sizeof(char);

				} else {
					/*
					 * Size for the largest pointer on
					 * any architecture.
					 */
					size += sizeof(u_int64_t);
					pf++;
				}

				host_buf += sizeof(void *);
				break;

			case string_c:
				/*
				 * One for the indicator that says if the string
				 * pointer is NULL.
				 */
				size += sizeof(u_int8_t);

				string = *((char * *) host_buf);
				if (string != NULL) {
					size += strlen(string) + sizeof(char);
				}

				host_buf += sizeof(char *);
				pf++;
				break;

			case strings_c:
				/*
				 * One for the indicator that says if the
				 * strings pointer is NULL.
				 */
				size += sizeof(u_int8_t);

				strings = *((char * * *) host_buf);
				if (strings != NULL) {
					for (i = 0; strings[i] != NULL; i++) {
						size += strlen(strings[i]) + sizeof(char);
					}

					/* one more for the final terminating nul. */
					size += sizeof(char);
				}

				host_buf += sizeof(char *);
				pf++;
				break;

			case bytes_c:
				pf++;
				len = atoi(pf);
				size += len;

				host_buf += len;
				pf += strspn(pf, "0123456789");
				break;

			case handle_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				ha = *((handle_array_t * *) host_buf);
				if (ha != NULL) {
					sizeof_hostbuf_to_netbuf(handle_array_struct_f, ha, &size);
				}

				host_buf += sizeof(handle_array_t * *);
				pf++;
				break;

			case handle_object_info_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				hoi = *((handle_object_info_t * *) host_buf);
				if (hoi != NULL) {
					/* Space for the object type */
					size += sizeof(u_int32_t);

					switch (hoi->type) {
						case DISK:
						case SEGMENT:
						case REGION:
						case EVMS_OBJECT:
							sizeof_hostbuf_to_netbuf(storage_object_info_f, &hoi->info, &size);
							break;

						case CONTAINER:
							sizeof_hostbuf_to_netbuf(storage_container_info_f, &hoi->info, &size);
							break;

						case VOLUME:
							sizeof_hostbuf_to_netbuf(logical_volume_info_f, &hoi->info, &size);
							break;

						case PLUGIN:
							sizeof_hostbuf_to_netbuf(plugin_info_f, &hoi->info, &size);
							break;

						default:
							break;
					}
				}

				host_buf += sizeof(handle_object_info_t * *);
				pf++;
				break;

			case expand_handle_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				eha = *((expand_handle_array_t * *) host_buf);
				if (eha != NULL) {
					sizeof_hostbuf_to_netbuf(expand_handle_array_struct_f, eha, &size);
				}

				host_buf += sizeof(expand_handle_array_t * *);
				pf++;
				break;

			case shrink_handle_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				sha = *((shrink_handle_array_t * *) host_buf);
				if (sha != NULL) {
					sizeof_hostbuf_to_netbuf(shrink_handle_array_struct_f, sha, &size);
				}

				host_buf += sizeof(shrink_handle_array_t * *);
				pf++;
				break;

			case change_record_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				cra = *((change_record_array_t * *) host_buf);
				if (cra != NULL) {
					sizeof_hostbuf_to_netbuf(change_record_array_struct_f, cra, &size);
				}

				host_buf += sizeof(change_record_array_t * *);
				pf++;
				break;

			case function_info_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				fia = *((function_info_array_t * *) host_buf);
				if (fia != NULL) {
					sizeof_hostbuf_to_netbuf(function_info_array_struct_f, fia, &size);
				}

				host_buf += sizeof(function_info_array_t * *);
				pf++;
				break;

			case declined_handle_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				dha = *((declined_handle_array_t * *) host_buf);
				if (dha != NULL) {
					sizeof_hostbuf_to_netbuf(declined_handle_array_struct_f, dha, &size);
				}

				host_buf += sizeof(declined_handle_array_t * *);
				pf++;
				break;

			case value_c:
				value_type = *((value_type_t *) host_buf);
				host_buf += sizeof(value_type_t);

				value_is_list = *((boolean *) host_buf);
				host_buf += sizeof(boolean);

				size += sizeof(u_int8_t) +
					sizeof(u_int32_t) +
					sizeof_netbuf_value(*((value_t *) host_buf), value_type, value_is_list);
				host_buf += sizeof(value_t);
				pf++;
				break;

			case option_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				oa = *((option_array_t * *) host_buf);
				if (oa != NULL) {
					sizeof_hostbuf_to_netbuf(option_array_struct_f, oa, &size);
				}

				host_buf += sizeof(option_array_t * *);
				pf++;
				break;

			case option_descriptor_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				od = *((option_descriptor_t * *) host_buf);

				if (od != NULL) {
					if (od->name != NULL) {
						size += strlen(od->name) + sizeof(char);
					} else {
						size += sizeof(char);
					}
					if (od->title != NULL) {
						size += strlen(od->title) + sizeof(char);
					} else {
						size += sizeof(char);
					}
					if (od->tip != NULL) {
						size += strlen(od->tip) + sizeof(char);
					} else {
						size += sizeof(char);
					}
					if (od->help != NULL) {
						size += strlen(od->help) + sizeof(char);
					} else {
						size += sizeof(char);
					}

					size += sizeof(u_int32_t)	/* type		   */
						+ sizeof(u_int32_t)	/* unit		   */
						+ sizeof(u_int32_t)	/* format	   */
						+ sizeof(u_int32_t)	/* min_len	   */
						+ sizeof(u_int32_t)	/* max_len	   */
						+ sizeof(u_int32_t)	/* flags	   */
						+ sizeof(u_int32_t);	/* constraint_type */

					size += sizeof_netbuf_collection(od->constraint,
									 od->constraint_type,
									 od->type);

					size += sizeof_netbuf_value(od->value, od->type, od->flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST);

					size += sizeof(u_int32_t)	/* od->group.group_number */
						+ sizeof(u_int32_t);	/* od->group.group_level  */
					if (od->group.group_name != NULL) {
						size += strlen(od->group.group_name) + sizeof(char);

					} else {
						size += sizeof(char);
					}
				}

				host_buf += sizeof(option_descriptor_t * *);
				pf++;
				break;

			case extended_info_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				eia = *((extended_info_array_t * *) host_buf);

				if (eia != NULL) {
					extended_info_t * ei;

					/* First is the u_int32_t of the count. */
					count = *((u_int32_t *) eia);

					size += sizeof(u_int32_t);

					for (i = 0; i < count; i++) {
						ei = &eia->info[i];

						if (ei->name != NULL) {
							size += strlen(ei->name) + sizeof(char);
						} else {
							size += sizeof(char);
						}
						if (ei->title != NULL) {
							size += strlen(ei->title) + sizeof(char);
						} else {
							size += sizeof(char);
						}
						if (ei->desc != NULL) {
							size += strlen(ei->desc) + sizeof(char);
						} else {
							size += sizeof(char);
						}

						size += sizeof(u_int32_t)	/* type		   */
							+ sizeof(u_int32_t)	/* unit		   */
							+ sizeof(u_int32_t);	/* format	   */

						size += sizeof_netbuf_value(ei->value, ei->type, FALSE);

						size += sizeof(u_int32_t);	/* collection_type */

						size += sizeof_netbuf_collection(ei->collection,
										 ei->collection_type,
										 ei->type);

						size += sizeof(u_int32_t)	/* eia->group.group_number */
							+ sizeof(u_int32_t);	/* eia->group.group_level  */
						if (ei->group.group_name != NULL) {
							size += strlen(ei->group.group_name) + sizeof(char);

						} else {
							size += sizeof(char);
						}

						size += sizeof(u_int16_t);	/* flags */
					}
				}

				host_buf += sizeof(extended_info_array_t * *);
				pf++;
				break;

			default:
				LOG_SERIOUS("Format character '%c' slipped past the validater. "
					    "I don't know how to handle it.  I'm skipping it.\n", *pf);
				pf++;
		}

		LOG_EXTRA("Size for format '%c' is %zd.\n", curr_f, size - curr_size);
	}

	LOG_EXTRA("Total size for format \"%s\" is %zd.\n", format, size);

	*psize += size;

	LOG_PROC_EXIT_PTR(host_buf);
	return host_buf;
}


int evms_sizeof_host_to_net(size_t * psize, char * format, ...) {

	int rc;
	va_list args;
	char * pf = format;
	char curr_f;
	size_t size = 0;
	size_t curr_size;
	int i;
	int x;
	u_int64_t u64;
	void * ptr;
	char * pch;
	char * string;
	char * * strings;
	char tmp_format[64];
	handle_array_t * ha;
	handle_object_info_t * hoi;
	expand_handle_array_t * eha;
	shrink_handle_array_t * sha;
	change_record_array_t * cra;
	function_info_array_t * fia;
	declined_handle_array_t * dha;
	value_type_t value_type;
	boolean value_is_list;
	option_array_t * oa;
	option_descriptor_t * od;
	extended_info_array_t * eia;

	LOG_PROC_ENTRY();

	LOG_EXTRA("Format is: %s\n", format);

	rc = validate_format(format);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	va_start(args, format);

	while (*pf != '\0') {
		curr_size = size;
		curr_f = *pf;

		switch (*pf) {
			case boolean_c:
				x = va_arg(args, int);
				size += sizeof(u_int8_t);
				pf++;
				break;

			case count_c:
			case count32_c:
				/*
				 * Arrays are not passed as args.
				 * Pointers to arrays are, but not
				 * arrays directly themsleves.
				 */
				LOG_ERROR("Arrays are not passed as arguments. "
					  "Try using \"p{c[...]}\" for the array arguement.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case int_c:
				x = va_arg(args, int);
				size += sizeof(u_int32_t);
				pf++;
				break;

			case u_int8_c:
				x = va_arg(args, int);
				size += sizeof(u_int8_t);
				pf++;
				break;

			case u_int16_c:
				x = va_arg(args, int);
				size += sizeof(u_int16_t);
				pf++;
				break;

			case u_int32_c:
				x = va_arg(args, int);
				size += sizeof(u_int32_t);
				pf++;
				break;

			case u_int64_c:
				u64 = va_arg(args, u_int64_t);
				size += sizeof(u_int64_t);
				pf++;
				break;

			case ptr_c:
				if (*(pf + 1) == '{') {
					void * host_buf;

					host_buf = va_arg(args, void *);

					size += sizeof(u_int8_t);

					/* Skip over "p{". */
					pf += 2;

					/* Parse out the substring between the braces. */
					strcpy(tmp_format, pf);
					pch = closing_brace(tmp_format);
					*pch = '\0';

					/* tmp_format now has the substring between the braces. */

					if (host_buf != NULL) {
						/* Call myself recursively to process the substring. */
						sizeof_hostbuf_to_netbuf(tmp_format, host_buf, &size);
					}

					pf += strlen(tmp_format) + sizeof(char);

				} else {
					/*
					 * Size for the largest pointer on
					 * any architecture.
					 */
					ptr = va_arg(args, void *);
					size += sizeof(u_int64_t);
					pf++;
				}
				break;

			case string_c:
				/*
				 * One for the indicator that says if the string
				 * pointer is NULL.
				 */
				size += sizeof(u_int8_t);

				string = va_arg(args, char *);
				if (string != NULL) {
					size += strlen(string) + sizeof(char);
				}
				pf++;
				break;

			case strings_c:
				/*
				 * One for the indicator that says if the string
				 * pointer is NULL.
				 */
				size += sizeof(u_int8_t);

				strings = va_arg(args, char * *);
				if (strings != NULL) {
					for (i = 0; strings[i] != NULL; i++) {
						size += strlen(strings[i]) + sizeof(char);
					}

					/* one more for the final terminating nul. */
					size++;
				}

				pf++;
				break;

			case bytes_c:
				/*
				 * Byte strings are not passed as args.
				 * Pointers to them are, but not the
				 * bytes directly themsleves.
				 */
				LOG_ERROR("Byte strings are not passed as arguments. "
					  "Try using \"pxnnn\" for the byte string arguement.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case handle_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				ha = va_arg(args, handle_array_t *);
				if (ha != NULL) {
					sizeof_hostbuf_to_netbuf(handle_array_struct_f, ha, &size);
				}

				pf++;
				break;

			case handle_object_info_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				hoi = va_arg(args, handle_object_info_t *);
				if (hoi != NULL) {
					/* Space for the object type */
					size += sizeof(u_int32_t);

					switch (hoi->type) {
						case DISK:
						case SEGMENT:
						case REGION:
						case EVMS_OBJECT:
							sizeof_hostbuf_to_netbuf(storage_object_info_f, &hoi->info, &size);
							break;

						case CONTAINER:
							sizeof_hostbuf_to_netbuf(storage_container_info_f, &hoi->info, &size);
							break;

						case VOLUME:
							sizeof_hostbuf_to_netbuf(logical_volume_info_f, &hoi->info, &size);
							break;

						case PLUGIN:
							sizeof_hostbuf_to_netbuf(plugin_info_f, &hoi->info, &size);
							break;

						default:
							break;
					}
				}

				pf++;
				break;

			case expand_handle_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				eha = va_arg(args, expand_handle_array_t *);
				if (eha != NULL) {
					sizeof_hostbuf_to_netbuf(expand_handle_array_struct_f, eha, &size);
				}

				pf++;
				break;

			case shrink_handle_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				sha = va_arg(args, shrink_handle_array_t *);
				if (sha != NULL) {
					sizeof_hostbuf_to_netbuf(shrink_handle_array_struct_f, sha, &size);
				}

				pf++;
				break;

			case change_record_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				cra = va_arg(args, change_record_array_t *);
				if (cra != NULL) {
					sizeof_hostbuf_to_netbuf(change_record_array_struct_f, cra, &size);
				}

				pf++;
				break;

			case function_info_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				fia = va_arg(args, function_info_array_t *);
				if (fia != NULL) {
					sizeof_hostbuf_to_netbuf(function_info_array_struct_f, fia, &size);
				}

				pf++;
				break;

			case declined_handle_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				dha = va_arg(args, declined_handle_array_t *);
				if (dha != NULL) {
					sizeof_hostbuf_to_netbuf(declined_handle_array_struct_f, dha, &size);
				}

				pf++;
				break;

			case value_c:
				value_type = va_arg(args, value_type_t);
				
				/*
				 * The following line should be:
				 * value_is_list = va_arg(args, boolean);
				 * but the compiler warns:
				 * "'boolean' is promoted to `int' when passed through `...'"
				 * so we do the casting manually.
				 */
				value_is_list = (boolean) va_arg(args, int);

				size += sizeof(u_int8_t) +
					sizeof(u_int32_t) +
					sizeof_netbuf_value(va_arg(args, value_t), value_type, value_is_list);
				pf++;
				break;

			case option_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				oa = va_arg(args, option_array_t *);
				if (oa != NULL) {
					sizeof_hostbuf_to_netbuf(option_array_struct_f, oa, &size);
				}

				pf++;
				break;

			case option_descriptor_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				od = va_arg(args, option_descriptor_t *);

				if (od != NULL) {
					if (od->name != NULL) {
						size += strlen(od->name) + sizeof(char);
					} else {
						size += sizeof(char);
					}
					if (od->title != NULL) {
						size += strlen(od->title) + sizeof(char);
					} else {
						size += sizeof(char);
					}
					if (od->tip != NULL) {
						size += strlen(od->tip) + sizeof(char);
					} else {
						size += sizeof(char);
					}
					if (od->help != NULL) {
						size += strlen(od->help) + sizeof(char);
					} else {
						size += sizeof(char);
					}

					size += sizeof(u_int32_t)	/* type		   */
						+ sizeof(u_int32_t)	/* unit		   */
						+ sizeof(u_int32_t)	/* format	   */
						+ sizeof(u_int32_t)	/* min_len	   */
						+ sizeof(u_int32_t)	/* max_len	   */
						+ sizeof(u_int32_t)	/* flags	   */
						+ sizeof(u_int32_t);	/* constraint_type */

					size += sizeof_netbuf_collection(od->constraint,
									 od->constraint_type,
									 od->type);

					size += sizeof_netbuf_value(od->value, od->type, od->flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST);

					size += sizeof(u_int32_t)	/* od->group.group_number */
						+ sizeof(u_int32_t);	/* od->group.group_level  */
					if (od->group.group_name != NULL) {
						size += strlen(od->group.group_name) + sizeof(char);

					} else {
						size += sizeof(char);
					}
				}

				pf++;
				break;

			case extended_info_array_c:
				/* One byte that says if it's a non-NULL pointer */
				size += sizeof(u_int8_t);

				eia = va_arg(args, extended_info_array_t *);

				if (eia != NULL) {
					extended_info_t * ei;
					u_int32_t count;

					/* First is the u_int32_t of the count. */
					count = *((u_int32_t *) eia);

					size += sizeof(u_int32_t);

					for (i = 0; i < count; i++) {
						ei = &eia->info[i];

						if (ei->name != NULL) {
							size += strlen(ei->name) + sizeof(char);
						} else {
							size += sizeof(char);
						}
						if (ei->title != NULL) {
							size += strlen(ei->title) + sizeof(char);
						} else {
							size += sizeof(char);
						}
						if (ei->desc != NULL) {
							size += strlen(ei->desc) + sizeof(char);
						} else {
							size += sizeof(char);
						}

						size += sizeof(u_int32_t)	/* type		   */
							+ sizeof(u_int32_t)	/* unit		   */
							+ sizeof(u_int32_t);	/* format	   */

						size += sizeof_netbuf_value(ei->value, ei->type, FALSE);

						size += sizeof(u_int32_t);	/* collection_type */

						size += sizeof_netbuf_collection(ei->collection,
										 ei->collection_type,
										 ei->type);

						size += sizeof(u_int32_t)	/* eia->group.group_number */
							+ sizeof(u_int32_t);	/* eia->group.group_level  */
						if (ei->group.group_name != NULL) {
							size += strlen(ei->group.group_name) + sizeof(char);

						} else {
							size += sizeof(char);
						}

						size += sizeof(u_int16_t);	/* flags */
					}
				}

				pf++;
				break;

			default:
				LOG_SERIOUS("Format character %c slipped past the validater. "
					    "I don't know how to handle it.  I'm skipping it.\n", *pf);
				pf++;
		}

		LOG_EXTRA("Size for format '%c' is %zd.\n", curr_f, size - curr_size);
	}

	va_end(args);

	LOG_EXTRA("Total size for format \"%s\" is %zd.\n", format, size);

	*psize = size;

	LOG_PROC_EXIT_INT(0);
	return 0;
}


/*
 * Functions for converting data into a network stream.
 */

static void * value_to_netbuf(void       * net_buf,
			      value_t      value,
			      value_type_t type,
			      boolean      value_is_list) {

	void * orig_net_buf = net_buf;
	uint count;
	value_t * p_value;
	int i;

	LOG_PROC_ENTRY();

	if (value_is_list) {
		/* First byte says if it is a NULL pointer or not. */
		*((u_int8_t *) net_buf) = (value.list != NULL);
		net_buf += sizeof(u_int8_t);

		if (value.list != NULL) {
			value_list_t * vl = (value_list_t *) value.list;
			count = vl->count;
			p_value = &vl->value[0];
			*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) count);
			net_buf += sizeof(u_int32_t);

		} else {
			/* That's all folks. */
			LOG_PROC_EXIT_PTR(net_buf);
			return net_buf;
		}

	} else {
		count = 1;
		p_value = &value;
	}

	for (i = 0; i < count; i++, p_value++) {
		switch (type) {
			case EVMS_Type_String:
				if ((*p_value).s != NULL) {
					/* A valid string follows. */
					*((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					strcpy(net_buf, (*p_value).s);
					net_buf += strlen((char *) net_buf) + sizeof(char);

				} else {
					/* String is NULL. */
					*((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);
				}
				break;

			case EVMS_Type_Boolean:
			case EVMS_Type_Char:
			case EVMS_Type_Unsigned_Char:
			case EVMS_Type_Int8:
			case EVMS_Type_Unsigned_Int8:
				*((u_int8_t *) net_buf) = (*p_value).uc;
				net_buf += sizeof(u_int8_t);
				break;

			case EVMS_Type_Int16:
			case EVMS_Type_Unsigned_Int16:
				*((u_int16_t *) net_buf) = HOST_TO_NET16((*p_value).ui16);
				net_buf += sizeof(u_int16_t);
				break;

			case EVMS_Type_Real32:
			case EVMS_Type_Int:
			case EVMS_Type_Int32:
			case EVMS_Type_Unsigned_Int:
			case EVMS_Type_Unsigned_Int32:
				*((u_int32_t *) net_buf) = HOST_TO_NET32((*p_value).ui32);
				net_buf += sizeof(u_int32_t);
				break;

			case EVMS_Type_Real64:
			case EVMS_Type_Int64:
			case EVMS_Type_Unsigned_Int64:
				*((u_int64_t *) net_buf) = HOST_TO_NET64((*p_value).ui64);
				net_buf += sizeof(u_int64_t);
				break;
		}
	}

	LOG_EXTRA("Size of output is %d.\n", (int)(net_buf - orig_net_buf));

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void * collection_to_netbuf(void             * net_buf,
				   value_collection_t coll,
				   collection_type_t  coll_type,
				   value_type_t       value_type) {
	int i;
	void * orig_net_buf = net_buf;

	LOG_PROC_ENTRY();

	switch (coll_type) {
		case EVMS_Collection_None:
			break;

		case EVMS_Collection_List:
			*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) coll.list->count);
			net_buf += sizeof(u_int32_t);

			for (i = 0; i < coll.list->count; i++) {
				net_buf = value_to_netbuf(net_buf, coll.list->value[i], value_type, FALSE);
			}
			break;

		case EVMS_Collection_Range:
			net_buf = value_to_netbuf(net_buf, coll.range->min,       value_type, FALSE);
			net_buf = value_to_netbuf(net_buf, coll.range->max,       value_type, FALSE);
			net_buf = value_to_netbuf(net_buf, coll.range->increment, value_type, FALSE);
			break;
	}

	LOG_EXTRA("Size of output is %d.\n", (int)(net_buf - orig_net_buf));

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void hostbuf_to_netbuf(char * format,
			      void * host_buf,       void * net_buf,
			      uint * p_host_buf_inc, uint * p_net_buf_inc) {

	void * orig_host_buf = host_buf;
	void * orig_net_buf  = net_buf;
	void * curr_net_buf;
	uint host_buf_inc;
	uint net_buf_inc;
	char * pf = format;
	char curr_f;
	u_int32_t count;
	u_int8_t  u8;
	u_int16_t u16;
	u_int32_t u32;
	u_int64_t u64;
	int i;
	char * pch;
	char tmp_format[64];
	void * ptr;
	char * string;
	char * * strings;
	uint len;
	handle_array_t * ha;
	handle_object_info_t * hoi;
	expand_handle_array_t * eha;
	shrink_handle_array_t * sha;
	change_record_array_t * cra;
	function_info_array_t * fia;
	declined_handle_array_t * dha;
	value_type_t value_type;
	boolean value_is_list;
	option_array_t * oa;
	option_descriptor_t * od;
	extended_info_array_t * eia;

	LOG_PROC_ENTRY();

	LOG_EXTRA("Format is: %s\n", format);

	while (*pf != '\0') {
		curr_net_buf = net_buf;
		curr_f = *pf;

		switch (*pf) {
			case boolean_c:
				u8 = (u_int8_t) *((boolean *) host_buf);
				*((u_int8_t *) net_buf) = u8;
				host_buf += sizeof(boolean);
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case count_c:
				/* First is the int of the count. */
				count = (u_int32_t) *((int *) host_buf);
				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) count);
				host_buf += sizeof(u_int32_t);
				net_buf += sizeof(u_int32_t);

				/* Skip over "c[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					uint host_buf_inc;
					uint net_buf_inc;

					/* call myself recursively to process the substring. */
					hostbuf_to_netbuf(tmp_format,
							  host_buf, net_buf,
							  &host_buf_inc, &net_buf_inc);

					host_buf += host_buf_inc;
					net_buf  += net_buf_inc;
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case count32_c:
				/* First is the u_int32_t of the count. */
				count = (u_int32_t) *((u_int32_t *) host_buf);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(count);
				host_buf += sizeof(int);
				net_buf += sizeof(u_int32_t);

				/* Skip over "C[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					uint host_buf_inc;
					uint net_buf_inc;

					/* call myself recursively to process the substring. */
					hostbuf_to_netbuf(tmp_format,
							  host_buf, net_buf,
							  &host_buf_inc, &net_buf_inc);

					host_buf += host_buf_inc;
					net_buf  += net_buf_inc;
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case int_c:
				u32 = (u_int32_t) *((int *) host_buf);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(u32);
				host_buf += sizeof(int);
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case u_int8_c:
				*((u_int8_t *) net_buf) = *((u_int8_t *) host_buf);
				host_buf += sizeof(u_int8_t);
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case u_int16_c:
				u16 = *((u_int16_t *) host_buf);
				*((u_int16_t *) net_buf) = HOST_TO_NET16(u16);
				host_buf += sizeof(u_int16_t);
				net_buf += sizeof(u_int16_t);
				pf++;
				break;

			case u_int32_c:
				u32 = *((u_int32_t *) host_buf);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(u32);
				host_buf += sizeof(u_int32_t);
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case u_int64_c:
				u64 = *((u_int64_t *) host_buf);
				*((u_int64_t *) net_buf) = HOST_TO_NET64(u64);
				host_buf += sizeof(u_int64_t);
				net_buf += sizeof(u_int64_t);
				pf++;
				break;

			case ptr_c:
				if (*(pf + 1) == '{') {
					/* It's a pointer to a structure. */
					ptr = *((int * *) host_buf);

					/* Skip over "p{". */
					pf += 2;

					/* Parse out the substring between the braces. */
					strcpy(tmp_format, pf);
					pch = closing_brace(tmp_format);
					*pch = '\0';

					/* tmp_format now has the substring between the braces. */

					if (ptr != NULL) {
						/* Set the marker to say we have a valid pointer. */
						*((u_int8_t *) net_buf) = TRUE;
						net_buf += sizeof(u_int8_t);

						/* Call myself recursively to process the substring. */
						hostbuf_to_netbuf(tmp_format,
								  ptr, net_buf,
								  &host_buf_inc, &net_buf_inc);

						net_buf  += net_buf_inc;

					} else {
						/* Set the marker to say the pointer is NULL. */
						*((u_int8_t *) net_buf) = FALSE;
						net_buf += sizeof(u_int8_t);

					}

					host_buf += sizeof(void *);
					pf += strlen(tmp_format) + sizeof(char);

				} else {
					/* It's a generic pointer. */
					/*
					 * Use the largest archectctural size
					 * for a pointer.
					 */
					u64 = (u_int64_t) (unsigned long) *((char * *) host_buf);
					*((u_int64_t *) net_buf) = HOST_TO_NET64(u64);
					host_buf += sizeof(void *);
					net_buf += sizeof(u_int64_t);
					pf++;
				}
				break;

			case string_c:
				string = *((char * *) host_buf);
				if (string != NULL) {
					/* A valid string follows. */
					*((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					strcpy(net_buf, string);
					net_buf += strlen(string) + sizeof(char);

				} else {
					/* String is NULL. */
					*((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);
				}

				host_buf += sizeof(char *);
				pf++;
				break;

			case strings_c:
				strings = *((char * * *) host_buf);
				if (strings != NULL) {
					/* Valid strings follow. */
					*((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					for (i = 0; strings[i] != NULL; i++) {
						strcpy(net_buf, strings[i]);
						net_buf += strlen(strings[i]) + FALSE;
					}

					/* Slap on the terminating nul. */
					*((char *) net_buf) = '\0';
					net_buf += sizeof(char);

				} else {
					/* Strings is NULL. */
					*((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);
				}

				host_buf += sizeof(char * *);
				pf++;
				break;

			case bytes_c:
				pf++;
				len = atoi(pf);

				memcpy(net_buf, host_buf, len);

				net_buf += len;
				host_buf += len;
				pf += strspn(pf, "0123456789");
				break;

			case handle_array_c:
				ha = *((handle_array_t * *) host_buf);
				if (ha == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					hostbuf_to_netbuf(handle_array_struct_f,
							  ha, net_buf,
							  &host_buf_inc, &net_buf_inc);
					net_buf  += net_buf_inc;
				}

				host_buf += sizeof(handle_array_t * *);
				pf++;
				break;

			case handle_object_info_c:
				hoi = *((handle_object_info_t * *) host_buf);
				if (hoi == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					/* Space for the object type */
					*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) hoi->type);
					net_buf += sizeof(u_int32_t);

					switch (hoi->type) {
						case DISK:
						case SEGMENT:
						case REGION:
						case EVMS_OBJECT:
							hostbuf_to_netbuf(storage_object_info_f,
									  &hoi->info, net_buf,
									  &host_buf_inc, &net_buf_inc);
							break;

						case CONTAINER:
							hostbuf_to_netbuf(storage_container_info_f,
									  &hoi->info, net_buf,
									  &host_buf_inc, &net_buf_inc);
							break;

						case VOLUME:
							hostbuf_to_netbuf(logical_volume_info_f,
									  &hoi->info, net_buf,
									  &host_buf_inc, &net_buf_inc);
							break;

						case PLUGIN:
							hostbuf_to_netbuf(plugin_info_f,
									  &hoi->info, net_buf,
									  &host_buf_inc, &net_buf_inc);
							break;

						default:
							break;
					}

					net_buf += net_buf_inc;
				}

				host_buf += sizeof(handle_object_info_t *);
				pf++;
				break;

			case expand_handle_array_c:
				eha = *((expand_handle_array_t * *) host_buf);
				if (eha == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					hostbuf_to_netbuf(expand_handle_array_struct_f,
							  eha, net_buf,
							  &host_buf_inc, &net_buf_inc);
					net_buf  += net_buf_inc;
				}

				host_buf += sizeof(expand_handle_array_t * *);
				pf++;
				break;

			case shrink_handle_array_c:
				sha = *((shrink_handle_array_t * *) host_buf);
				if (sha == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					hostbuf_to_netbuf(shrink_handle_array_struct_f,
							  sha, net_buf,
							  &host_buf_inc, &net_buf_inc);
					net_buf  += net_buf_inc;
				}

				host_buf += sizeof(shrink_handle_array_t * *);
				pf++;
				break;

			case change_record_array_c:
				cra = *((change_record_array_t * *) host_buf);
				if (cra == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					hostbuf_to_netbuf(change_record_array_struct_f,
							  cra, net_buf,
							  &host_buf_inc, &net_buf_inc);
					net_buf  += net_buf_inc;
				}

				host_buf += sizeof(change_record_array_t * *);
				pf++;
				break;

			case function_info_array_c:
				fia = *((function_info_array_t * *) host_buf);
				if (fia == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					hostbuf_to_netbuf(function_info_array_struct_f,
							  fia, net_buf,
							  &host_buf_inc, &net_buf_inc);
					net_buf  += net_buf_inc;
				}

				host_buf += sizeof(function_info_array_t * *);
				pf++;
				break;

			case declined_handle_array_c:
				dha = *((declined_handle_array_t * *) host_buf);
				if (dha == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					hostbuf_to_netbuf(declined_handle_array_struct_f,
							  dha, net_buf,
							  &host_buf_inc, &net_buf_inc);
					net_buf  += net_buf_inc;
				}

				host_buf += sizeof(declined_handle_array_t * *);
				pf++;
				break;

			case value_c:
				value_type = *((value_type_t *) host_buf);
				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) value_type);
				net_buf += sizeof(u_int32_t);
				host_buf += sizeof(value_type_t);

				value_is_list = *((boolean *) host_buf);
				*((u_int8_t *) net_buf) = (u_int8_t) value_is_list;
				net_buf += sizeof(u_int8_t);
				host_buf += sizeof(boolean);

				net_buf = value_to_netbuf(net_buf, *((value_t *) host_buf), value_type, value_is_list);
				host_buf += sizeof(value_t);
				pf++;
				break;

			case option_array_c:
				oa = *((option_array_t * *) host_buf);
				if (oa == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					hostbuf_to_netbuf(option_array_struct_f,
							  oa, net_buf,
							  &host_buf_inc, &net_buf_inc);
					net_buf  += net_buf_inc;
				}

				host_buf += sizeof(option_array_t * *);
				pf++;
				break;

			case option_descriptor_c:
				od = *((option_descriptor_t * *) host_buf);
				if (od == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					if (od->name != NULL) {
						len = strlen(od->name) + sizeof(char);
						memcpy(net_buf, od->name, len);
						net_buf += len;
					} else {
						*((char *) net_buf) = '\0';
						net_buf += sizeof(char);
					}
					if (od->title != NULL) {
						len = strlen(od->title) + sizeof(char);
						memcpy(net_buf, od->title, len);
						net_buf += len;
					} else {
						*((char *) net_buf) = '\0';
						net_buf += sizeof(char);
					}
					if (od->tip != NULL) {
						len = strlen(od->tip) + sizeof(char);
						memcpy(net_buf, od->tip, len);
						net_buf += len;
					} else {
						*((char *) net_buf) = '\0';
						net_buf += sizeof(char);
					}
					if (od->help != NULL) {
						len = strlen(od->help) + sizeof(char);
						memcpy(net_buf, od->help, len);
						net_buf += len;
					} else {
						*((char *) net_buf) = '\0';
						net_buf += sizeof(char);
					}

					*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) od->type);
					net_buf += sizeof(u_int32_t);
					*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) od->unit);
					net_buf += sizeof(u_int32_t);
					*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) od->format);
					net_buf += sizeof(u_int32_t);
					*((u_int32_t *) net_buf) = HOST_TO_NET32(            od->min_len);
					net_buf += sizeof(u_int32_t);
					*((u_int32_t *) net_buf) = HOST_TO_NET32(            od->max_len);
					net_buf += sizeof(u_int32_t);
					*((u_int32_t *) net_buf) = HOST_TO_NET32(            od->flags);
					net_buf += sizeof(u_int32_t);
					*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) od->constraint_type);
					net_buf += sizeof(u_int32_t);

					net_buf = collection_to_netbuf(net_buf,
								       od->constraint,
								       od->constraint_type,
								       od->type);

					net_buf = value_to_netbuf(net_buf,
								  od->value,
								  od->type,
								  od->flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST);

					*((u_int32_t *) net_buf) = HOST_TO_NET32(od->group.group_number);
					net_buf += sizeof(u_int32_t);
					*((u_int32_t *) net_buf) = HOST_TO_NET32(od->group.group_level);
					net_buf += sizeof(u_int32_t);

					if (od->group.group_name != NULL) {
						len = strlen(od->group.group_name) + sizeof(char);
						memcpy(net_buf, od->group.group_name, len);
						net_buf += len;
					} else {
						*((char *) net_buf) = '\0';
						net_buf += sizeof(char);
					}
				}

				host_buf += sizeof(option_descriptor_t *);
				pf++;
				break;

			case extended_info_array_c:
				eia = *((extended_info_array_t * *) host_buf);
				if (eia == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					/* First is the u_int32_t of the count. */
					count = eia->count;
					*((u_int32_t *) net_buf) = HOST_TO_NET32(count);
					net_buf += sizeof(u_int32_t);

					for (i = 0; i < count; i++) {
						extended_info_t * ei;

						ei = &eia->info[i];

						if (ei->name != NULL) {
							len = strlen(ei->name) + sizeof(char);
							memcpy(net_buf, ei->name, len);
							net_buf += len;
						} else {
							*((char *) net_buf) = '\0';
							net_buf += sizeof(char);
						}
						if (ei->title != NULL) {
							len = strlen(ei->title) + sizeof(char);
							memcpy(net_buf, ei->title, len);
							net_buf += len;
						} else {
							*((char *) net_buf) = '\0';
							net_buf += sizeof(char);
						}
						if (ei->desc != NULL) {
							len = strlen(ei->desc) + sizeof(char);
							memcpy(net_buf, ei->desc, len);
							net_buf += len;
						} else {
							*((char *) net_buf) = '\0';
							net_buf += sizeof(char);
						}

						*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) ei->type);
						net_buf += sizeof(u_int32_t);
						*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) ei->unit);
						net_buf += sizeof(u_int32_t);
						*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) ei->format);
						net_buf += sizeof(u_int32_t);

						net_buf = value_to_netbuf(net_buf,
									  ei->value,
									  ei->type,
									  FALSE);

						*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) ei->collection_type);
						net_buf += sizeof(u_int32_t);

						net_buf = collection_to_netbuf(net_buf,
									       ei->collection,
									       ei->collection_type,
									       ei->type);

						*((u_int32_t *) net_buf) = HOST_TO_NET32(ei->group.group_number);
						net_buf += sizeof(u_int32_t);
						*((u_int32_t *) net_buf) = HOST_TO_NET32(ei->group.group_level);
						net_buf += sizeof(u_int32_t);

						if (ei->group.group_name != NULL) {
							len = strlen(ei->group.group_name) + sizeof(char);
							memcpy(net_buf, ei->group.group_name, len);
							net_buf += len;
						} else {
							*((char *) net_buf) = '\0';
							net_buf += sizeof(char);
						}

						*((u_int16_t *) net_buf) = HOST_TO_NET16(ei->flags);
						net_buf += sizeof(u_int16_t);
					}
				}

				host_buf += sizeof(extended_info_array_t *);
				pf++;
				break;

			default:
				LOG_SERIOUS("Format character %c slipped past the validater. "
					    "I don't know how to handle it.  I'm skipping it.\n", *pf);
				pf++;
		}

		LOG_EXTRA("Size of output for format '%c' is %d.\n", curr_f, (int)(net_buf - curr_net_buf));
	}

	*p_host_buf_inc = host_buf - orig_host_buf;
	*p_net_buf_inc  = net_buf  - orig_net_buf;

	LOG_EXTRA("Total size of output for format \"%s\" is %d.\n", format, (int)(net_buf  - orig_net_buf));

	LOG_PROC_EXIT_VOID();
	return;
}


int evms_host_to_net(void * net_buf, char * format, ...) {

	int rc;
	char * pf = format;
	char curr_f;
	void * orig_net_buf = net_buf;
	void * curr_net_buf;
	uint host_buf_inc;
	uint net_buf_inc;
	va_list args;
	u_int16_t u16;
	u_int32_t u32;
	u_int64_t u64;
	int i;
	char * pch;
	char tmp_format[64];
	void * ptr;
	char * string;
	char * * strings;
	handle_array_t * ha;
	handle_object_info_t * hoi;
	expand_handle_array_t * eha;
	shrink_handle_array_t * sha;
	change_record_array_t * cra;
	function_info_array_t * fia;
	declined_handle_array_t * dha;
	value_type_t value_type;
	boolean value_is_list;
	option_array_t * oa;
	option_descriptor_t * od;
	extended_info_array_t * eia;

	LOG_PROC_ENTRY();

	LOG_EXTRA("Format is: %s\n", format);

	rc = validate_format(format);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	va_start(args, format);

	while (*pf != '\0') {
		curr_net_buf = net_buf;
		curr_f = *pf;

		switch (*pf) {
			case boolean_c:
				*((u_int8_t *) net_buf) = (u_int8_t) va_arg(args, int);
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case count_c:
			case count32_c:
				/*
				 * Arrays are not passed as args.
				 * Pointers to arrays are, but not
				 * arrays directly themsleves.
				 */
				LOG_ERROR("Arrays are not passed as arguments. "
					  "Try using \"p{c[...]}\" for the array arguement.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case int_c:
				u32 = (u_int32_t) va_arg(args, int);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(u32);
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case u_int8_c:
				*((u_int8_t *) net_buf) = (u_int8_t) va_arg(args, int);
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case u_int16_c:
				u16 = (u_int16_t) va_arg(args, int);
				*((u_int16_t *) net_buf) = HOST_TO_NET16(u16);
				net_buf += sizeof(u_int16_t);
				pf++;
				break;

			case u_int32_c:
				u32 = (u_int32_t) va_arg(args, int);
				*((u_int32_t *) net_buf) = HOST_TO_NET32(u32);
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case u_int64_c:
				u64 = (u_int64_t) va_arg(args, u_int64_t);
				*((u_int64_t *) net_buf) = HOST_TO_NET64(u64);
				net_buf += sizeof(u_int64_t);
				pf++;
				break;

			case ptr_c:
				if (*(pf + 1) == '{') {
					uint host_buf_inc;
					uint net_buf_inc;

					/* It's a pointer to a structure. */
					ptr = va_arg(args, void *);

					/* Skip over "p{". */
					pf += 2;

					/* Parse out the substring between the braces. */
					strcpy(tmp_format, pf);
					pch = closing_brace(tmp_format);
					*pch = '\0';

					/* tmp_format now has the substring between the braces. */

					if (ptr != NULL) {
						/* Set the marker to say we have a valid pointer. */
						*((u_int8_t *) net_buf) = TRUE;
						net_buf += sizeof(u_int8_t);

						/* Call myself recursively to process the substring. */
						hostbuf_to_netbuf(tmp_format,
								  ptr, net_buf,
								  &host_buf_inc, &net_buf_inc);

						net_buf += net_buf_inc;

					} else {
						/* Set the marker to say the pointer is NULL. */
						*((u_int8_t *) net_buf) = FALSE;
						net_buf += sizeof(u_int8_t);

					}

					pf += strlen(tmp_format) + sizeof(char);

				} else {
					/* It's a generic pointer. */
					/*
					 * Use the largest archectctural size
					 * for a pointer.
					 */
					u64 = (u_int64_t) (unsigned long) va_arg(args, void *);
					*((u_int64_t *) net_buf) = HOST_TO_NET64(u64);
					net_buf += sizeof(u_int64_t);
					pf++;
				}
				break;

			case string_c:
				string = va_arg(args, char *);
				if (string != NULL) {
					/* A valid string follows. */
					*((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					strcpy(net_buf, string);
					net_buf += strlen(string) + sizeof(char);

				} else {
					/* String is NULL. */
					*((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);
				}

				pf++;
				break;

			case strings_c:
				strings = va_arg(args, char * *);
				if (strings != NULL) {
					/* Valid strings follows. */
					*((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					for (i = 0; strings[i] != NULL; i++) {
						strcpy(net_buf, strings[i]);
						net_buf += strlen(strings[i]) + sizeof(char);
					}

					/* Slap on the terminating nul. */
					*((char *) net_buf) = '\0';
					net_buf += sizeof(char);

				} else {
					/* Strings is NULL. */
					*((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);
				}
				pf++;
				break;

			case bytes_c:
				/*
				 * Byte strings are not passed as args.
				 * Pointers to them are, but not the
				 * bytes directly themsleves.
				 */
				LOG_ERROR("Byte strings are not passed as arguments. "
					  "Try using \"pxnnn\" for the byte string arguement.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case handle_array_c:
				ha = va_arg(args, handle_array_t *);
				if (ha == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					hostbuf_to_netbuf(handle_array_struct_f,
							  ha, net_buf,
							  &host_buf_inc, &net_buf_inc);
					net_buf  += net_buf_inc;
				}

				pf++;
				break;

			case handle_object_info_c:
				hoi = va_arg(args, handle_object_info_t *);
				if (hoi == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					/* Space for the object type */
					*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) hoi->type);
					net_buf += sizeof(u_int32_t);

					switch (hoi->type) {
						case DISK:
						case SEGMENT:
						case REGION:
						case EVMS_OBJECT:
							hostbuf_to_netbuf(storage_object_info_f,
									  &hoi->info, net_buf,
									  &host_buf_inc, &net_buf_inc);
							break;

						case CONTAINER:
							hostbuf_to_netbuf(storage_container_info_f,
									  &hoi->info, net_buf,
									  &host_buf_inc, &net_buf_inc);
							break;

						case VOLUME:
							hostbuf_to_netbuf(logical_volume_info_f,
									  &hoi->info, net_buf,
									  &host_buf_inc, &net_buf_inc);
							break;

						case PLUGIN:
							hostbuf_to_netbuf(plugin_info_f,
									  &hoi->info, net_buf,
									  &host_buf_inc, &net_buf_inc);
							break;

						default:
							break;
					}

					net_buf += net_buf_inc;
				}

				pf++;
				break;

			case expand_handle_array_c:
				eha = va_arg(args, expand_handle_array_t *);
				if (eha == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					hostbuf_to_netbuf(expand_handle_array_struct_f,
							  eha, net_buf,
							  &host_buf_inc, &net_buf_inc);
					net_buf  += net_buf_inc;
				}

				pf++;
				break;

			case shrink_handle_array_c:
				sha = va_arg(args, shrink_handle_array_t *);
				if (sha == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					hostbuf_to_netbuf(shrink_handle_array_struct_f,
							  sha, net_buf,
							  &host_buf_inc, &net_buf_inc);
					net_buf  += net_buf_inc;
				}

				pf++;
				break;

			case change_record_array_c:
				cra = va_arg(args, change_record_array_t *);
				if (cra == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					hostbuf_to_netbuf(change_record_array_struct_f,
							  cra, net_buf,
							  &host_buf_inc, &net_buf_inc);
					net_buf  += net_buf_inc;
				}

				pf++;
				break;

			case function_info_array_c:
				fia = va_arg(args, function_info_array_t *);
				if (fia == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					hostbuf_to_netbuf(function_info_array_struct_f,
							  fia, net_buf,
							  &host_buf_inc, &net_buf_inc);
					net_buf  += net_buf_inc;
				}

				pf++;
				break;

			case declined_handle_array_c:
				dha = va_arg(args, declined_handle_array_t *);
				if (dha == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					hostbuf_to_netbuf(declined_handle_array_struct_f,
							  dha, net_buf,
							  &host_buf_inc, &net_buf_inc);
					net_buf  += net_buf_inc;
				}

				pf++;
				break;

			case value_c:
				value_type = va_arg(args, value_type_t);
				*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) value_type);
				net_buf += sizeof(u_int32_t);

				/*
				 * The following line should be:
				 * value_is_list = va_arg(args, boolean);
				 * but the compiler warns:
				 * "'boolean' is promoted to `int' when passed through `...'"
				 * so we do the casting manually.
				 */
				value_is_list = (boolean) va_arg(args, int);
				*((u_int8_t *) net_buf) = (u_int8_t) value_is_list;
				net_buf += sizeof(u_int8_t);

				net_buf = value_to_netbuf(net_buf, va_arg(args, value_t), value_type, value_is_list);
				pf++;
				break;

			case option_array_c:
				oa = va_arg(args, option_array_t *);
				if (oa == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					hostbuf_to_netbuf(option_array_struct_f,
							  oa, net_buf,
							  &host_buf_inc, &net_buf_inc);
					net_buf  += net_buf_inc;
				}

				pf++;
				break;

			case option_descriptor_c:
				od = va_arg(args, option_descriptor_t *);
				if (od == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
					uint len;

                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					if (od->name != NULL) {
						len = strlen(od->name) + sizeof(char);
						memcpy(net_buf, od->name, len);
						net_buf += len;
					} else {
						*((char *) net_buf) = '\0';
						net_buf += sizeof(char);
					}
					if (od->title != NULL) {
						len = strlen(od->title) + sizeof(char);
						memcpy(net_buf, od->title, len);
						net_buf += len;
					} else {
						*((char *) net_buf) = '\0';
						net_buf += sizeof(char);
					}
					if (od->tip != NULL) {
						len = strlen(od->tip) + sizeof(char);
						memcpy(net_buf, od->tip, len);
						net_buf += len;
					} else {
						*((char *) net_buf) = '\0';
						net_buf += sizeof(char);
					}
					if (od->help != NULL) {
						len = strlen(od->help) + sizeof(char);
						memcpy(net_buf, od->help, len);
						net_buf += len;
					} else {
						*((char *) net_buf) = '\0';
						net_buf += sizeof(char);
					}

					*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) od->type);
					net_buf += sizeof(u_int32_t);
					*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) od->unit);
					net_buf += sizeof(u_int32_t);
					*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) od->format);
					net_buf += sizeof(u_int32_t);
					*((u_int32_t *) net_buf) = HOST_TO_NET32(            od->min_len);
					net_buf += sizeof(u_int32_t);
					*((u_int32_t *) net_buf) = HOST_TO_NET32(            od->max_len);
					net_buf += sizeof(u_int32_t);
					*((u_int32_t *) net_buf) = HOST_TO_NET32(            od->flags);
					net_buf += sizeof(u_int32_t);
					*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) od->constraint_type);
					net_buf += sizeof(u_int32_t);

					net_buf = collection_to_netbuf(net_buf,
								       od->constraint,
								       od->constraint_type,
								       od->type);

					net_buf = value_to_netbuf(net_buf,
								  od->value,
								  od->type,
								  od->flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST);

					*((u_int32_t *) net_buf) = HOST_TO_NET32(od->group.group_number);
					net_buf += sizeof(u_int32_t);
					*((u_int32_t *) net_buf) = HOST_TO_NET32(od->group.group_level);
					net_buf += sizeof(u_int32_t);

					if (od->group.group_name != NULL) {
						len = strlen(od->group.group_name) + sizeof(char);
						memcpy(net_buf, od->group.group_name, len);
						net_buf += len;
					} else {
						*((char *) net_buf) = '\0';
						net_buf += sizeof(char);
					}
				}

				pf++;
				break;

			case extended_info_array_c:
				eia = va_arg(args, extended_info_array_t *);
				if (eia == NULL) {
                                        *((u_int8_t *) net_buf) = FALSE;
					net_buf += sizeof(u_int8_t);

				} else {
					u_int32_t count;

                                        *((u_int8_t *) net_buf) = TRUE;
					net_buf += sizeof(u_int8_t);

					/* First is the u_int32_t of the count. */
					count = eia->count;
					*((u_int32_t *) net_buf) = HOST_TO_NET32(count);
					net_buf += sizeof(u_int32_t);

					for (i = 0; i < count; i++) {
						extended_info_t * ei;
						uint len;

						ei = &eia->info[i];

						if (ei->name != NULL) {
							len = strlen(ei->name) + sizeof(char);
							memcpy(net_buf, ei->name, len);
							net_buf += len;
						} else {
							*((char *) net_buf) = '\0';
							net_buf += sizeof(char);
						}
						if (ei->title != NULL) {
							len = strlen(ei->title) + sizeof(char);
							memcpy(net_buf, ei->title, len);
							net_buf += len;
						} else {
							*((char *) net_buf) = '\0';
							net_buf += sizeof(char);
						}
						if (ei->desc != NULL) {
							len = strlen(ei->desc) + sizeof(char);
							memcpy(net_buf, ei->desc, len);
							net_buf += len;
						} else {
							*((char *) net_buf) = '\0';
							net_buf += sizeof(char);
						}

						*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) ei->type);
						net_buf += sizeof(u_int32_t);
						*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) ei->unit);
						net_buf += sizeof(u_int32_t);
						*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) ei->format);
						net_buf += sizeof(u_int32_t);

						net_buf = value_to_netbuf(net_buf,
									  ei->value,
									  ei->type,
									  FALSE);

						*((u_int32_t *) net_buf) = HOST_TO_NET32((u_int32_t) ei->collection_type);
						net_buf += sizeof(u_int32_t);

						net_buf = collection_to_netbuf(net_buf,
									       ei->collection,
									       ei->collection_type,
									       ei->type);

						*((u_int32_t *) net_buf) = HOST_TO_NET32(ei->group.group_number);
						net_buf += sizeof(u_int32_t);
						*((u_int32_t *) net_buf) = HOST_TO_NET32(ei->group.group_level);
						net_buf += sizeof(u_int32_t);

						if (ei->group.group_name != NULL) {
							len = strlen(ei->group.group_name) + sizeof(char);
							memcpy(net_buf, ei->group.group_name, len);
							net_buf += len;
						} else {
							*((char *) net_buf) = '\0';
							net_buf += sizeof(char);
						}

						*((u_int16_t *) net_buf) = HOST_TO_NET16(ei->flags);
						net_buf += sizeof(u_int16_t);
					}
				}

				pf++;
				break;

			default:
				LOG_SERIOUS("Format character %c slipped past the validater. "
					    "I don't know how to handle it.  I'm skipping it.\n", *pf);
				pf++;
		}

		LOG_EXTRA("Size of output for format '%c' is %d.\n", curr_f, (int)(net_buf - curr_net_buf));
	}

	va_end(args);

	LOG_EXTRA("Total size of output buffer for format \"%s\" is %d.\n", format, (int)(net_buf - orig_net_buf));

	LOG_PROC_EXIT_INT(0);
	return 0;
}


/*
 * Functions for calculating the size needed for a local buffer to receive
 * the data in a network stream.
 */

static void * skip_netbuf_value(void       * net_buf,
				value_type_t type,
				boolean      value_is_list) {
	uint count;
	int i;

	LOG_PROC_ENTRY();

	if (value_is_list) {
		/* First byte says if we have a NUL pointer or not. */
		boolean is_null = *((u_int8_t *) net_buf) == 0;
		net_buf += sizeof(u_int8_t);

		if (is_null) {
			LOG_PROC_EXIT_PTR(net_buf);
			return net_buf;
		}

		count = NET_TO_HOST32(*((u_int32_t *) net_buf));
		net_buf += sizeof(u_int32_t);

	} else {
		count = 1;
	}

	for (i = 0; i < count; i++) {
		switch (type) {
			case EVMS_Type_String:
				if (*((u_int8_t *) net_buf) != 0) {
					/* A valid string pointer follows. */
					net_buf += sizeof(u_int8_t);

					net_buf += strlen((char *) net_buf) + sizeof(char);

				} else {
					/* The string pointer is NULL. */
					net_buf += sizeof(u_int8_t);
				}
				break;

			case EVMS_Type_Boolean:
			case EVMS_Type_Char:
			case EVMS_Type_Unsigned_Char:
			case EVMS_Type_Int8:
			case EVMS_Type_Unsigned_Int8:
				net_buf += sizeof(u_int8_t);
				break;

			case EVMS_Type_Int16:
			case EVMS_Type_Unsigned_Int16:
				net_buf += sizeof(u_int16_t);
				break;

			case EVMS_Type_Real32:
			case EVMS_Type_Int:
			case EVMS_Type_Int32:
			case EVMS_Type_Unsigned_Int:
			case EVMS_Type_Unsigned_Int32:
				net_buf += sizeof(u_int32_t);
				break;

			case EVMS_Type_Real64:
			case EVMS_Type_Int64:
			case EVMS_Type_Unsigned_Int64:
				net_buf += sizeof(u_int64_t);
				break;
		}
	}

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void * skip_netbuf_collection(void            * net_buf,
				     collection_type_t coll_type,
				     value_type_t      value_type) {
	
	uint count;
	int i;

	LOG_PROC_ENTRY();

	switch (coll_type) {
		case EVMS_Collection_None:
			break;

		case EVMS_Collection_List:
			/* First 32-bit value in the net_buf is the count */
			count = (uint) NET_TO_HOST32(*((u_int32_t *) net_buf));
			net_buf += sizeof(u_int32_t);

			for (i = 0; i < count; i++) {
				net_buf = skip_netbuf_value(net_buf, value_type, FALSE);
			}
			break;

		case EVMS_Collection_Range:
			net_buf = skip_netbuf_value(net_buf, value_type, FALSE);
			net_buf = skip_netbuf_value(net_buf, value_type, FALSE);
			net_buf = skip_netbuf_value(net_buf, value_type, FALSE);
			break;
	}

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void * skip_netbuf_option_descriptor(void * net_buf) {

	option_descriptor_t od;

	LOG_PROC_ENTRY();

	/* Skip name */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	/* Skip title */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	/* Skip tip */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	/* Skip help */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	od.type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	od.unit = (value_unit_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	od.format = (value_format_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	od.min_len = NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	od.max_len = NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	od.flags = NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	od.constraint_type = (collection_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	net_buf = skip_netbuf_collection(net_buf,
					 od.constraint_type,
					 od.type);

	net_buf = skip_netbuf_value(net_buf,
				    od.type,
				    od.flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST);

//	od.group.group_number = NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	od.group.group_level = NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	/* Skip group.group_name */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void * skip_netbuf_extended_info(void * net_buf) {

	value_type_t      type;
	collection_type_t collection_type;

	LOG_PROC_ENTRY();

	/* Skip name */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	/* Skip title */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	/* Skip desc */
	net_buf += strlen((char *) net_buf) + sizeof(char);

	type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	ei->unit = (value_unit_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	ei->format = (value_format_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	net_buf = skip_netbuf_value(net_buf,
				    type,
				    FALSE);

	collection_type = (collection_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	net_buf = skip_netbuf_collection(net_buf,
					 collection_type,
					 type);

//	ei->group.group_number = NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

//	ei->group.group_level = NET_TO_HOST32(*((u_int32_t *) net_buf));
	net_buf += sizeof(u_int32_t);

	/* Skip group.group_name */
	net_buf += strlen((char *) net_buf) + sizeof(char);

//	ei->flags = NET_TO_HOST16(*((u_int16_t *) net_buf));
	net_buf += sizeof(u_int16_t);

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void * sizeof_netbuf_to_hostbuf(char * format, void * net_buf, size_t * psize) {

	char * pf = format;
	u_int32_t count;
	int i;
	char * pch;
	char tmp_format[64];
	uint len;
	value_type_t value_type;
	boolean value_is_list;

	LOG_PROC_ENTRY();

	LOG_EXTRA("Format is: %s\n", format);

	while (*pf != '\0') {
		switch (*pf) {
			case boolean_c:
				*psize += sizeof(boolean);
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case count_c:
				/* First is the int of the count. */
				count = NET_TO_HOST32(*((u_int32_t *) net_buf));
				*psize += sizeof(uint);

				/* Counts are always sent on the wire as 32-bint ints. */
				net_buf += sizeof(u_int32_t);

				/* Skip over "c[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					/* call myself recursively to process the substring. */
					net_buf = sizeof_netbuf_to_hostbuf(tmp_format, net_buf, psize);
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case count32_c:
				/* First is the u_int32_t of the count. */
				count = NET_TO_HOST32(*((u_int32_t *) net_buf));
				*psize += sizeof(u_int32_t);

				/* Skip over "C[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					/* Call myself recursively to process the substring. */
					net_buf = sizeof_netbuf_to_hostbuf(tmp_format, net_buf, psize);
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case int_c:
				*psize += sizeof(int);
				/* ints are sent on the wire as 32-bit ints. */
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case u_int8_c:
				*psize += sizeof(u_int8_t);
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case u_int16_c:
				*psize += sizeof(u_int16_t);
				net_buf += sizeof(u_int16_t);
				pf++;
				break;

			case u_int32_c:
				*psize += sizeof(u_int32_t);
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case u_int64_c:
				*psize += sizeof(u_int64_t);
				net_buf += sizeof(u_int64_t);
				pf++;
				break;

			case ptr_c:
				/*
				 * No need to delve into the structure that the
				 * pointer may point to.  As a pointer to a
				 * structure, the structure will be allocated
				 * separately.  It is not part of the base
				 * size being calculated.
				 */
				*psize += sizeof(void *);
				pf++;

				if (*pf == '{') {
					size_t size = 0;
					/* Call myself recursively to skip over
					 * the structure in the net_buf.
					 */
					// FIXME
					/* Skip over the left brace. */
					pf++;

					/* Parse out the substring between the braces. */
					strcpy(tmp_format, pf);
					pch = closing_brace(tmp_format);
					*pch = '\0';

					/* tmp_format now has the substring between the braces. */
                                        /* Call myself recursively to process the substring. */
					net_buf = sizeof_netbuf_to_hostbuf(tmp_format, net_buf, &size);

					pf += strlen(tmp_format) + sizeof(char);
				}
				break;

			case string_c:
				*psize += sizeof(char *);
				net_buf += sizeof(u_int8_t);
				if (*(((u_int8_t *) net_buf) - 1)) {
					net_buf += strlen((char *) net_buf) + 1;
				}
				pf++;
				break;

			case strings_c:
				*psize += sizeof(char * *);
				net_buf += sizeof(u_int8_t);
				if (*(((u_int8_t *) net_buf) - 1)) {
					while (strlen((char *) net_buf) > 0) {
						net_buf += strlen((char *) net_buf) + sizeof(char);
					}
					/* skip over last empty string. */
					net_buf += sizeof(char);
				}
				pf++;
				break;

			case bytes_c:
				pf++;
				len = atoi(pf);
				*psize += len;

				net_buf += len;
				pf += strspn(pf, "0123456789");
				break;

			case handle_array_c:
				*psize += sizeof(handle_array_t *);
				net_buf += sizeof(u_int8_t);

				/*
				 * Skip over byte that says if a handle array
				 * follows.
				 */
				
				if (*(((u_int8_t *) net_buf) - 1)) {
					size_t size = 0;
					
					net_buf = sizeof_netbuf_to_hostbuf(handle_array_struct_f, net_buf, &size);
				}
				pf++;
				break;

			case handle_object_info_c:
				*psize += sizeof(handle_object_info_t *);
				net_buf += sizeof(u_int8_t);
				if (*(((u_int8_t *) net_buf) - 1)) {
					size_t size = 0;
					object_type_t type;

					type = (object_type_t) *((u_int32_t *) net_buf);
					net_buf += sizeof(u_int32_t);

					switch (type) {
						case DISK:
						case SEGMENT:
						case REGION:
						case EVMS_OBJECT:
							net_buf = sizeof_netbuf_to_hostbuf(storage_object_info_f, net_buf, &size);
							break;

						case CONTAINER:
							net_buf = sizeof_netbuf_to_hostbuf(storage_container_info_f, net_buf, &size);
							break;

						case VOLUME:
							net_buf = sizeof_netbuf_to_hostbuf(logical_volume_info_f, net_buf, &size);
							break;

						case PLUGIN:
							net_buf = sizeof_netbuf_to_hostbuf(plugin_info_f, net_buf, &size);
							break;

						default:
							LOG_WARNING("Unknown object type %d (%#x) came in on the wire.\n",
								    type, type);
					}
				}
				pf++;
				break;

			case expand_handle_array_c:
				*psize += sizeof(expand_handle_array_t *);
				net_buf += sizeof(u_int8_t);

				/*
				 * Skip over byte that says if an expand handle
				 * array follows.
				 */
				
				if (*(((u_int8_t *) net_buf) - 1)) {
					size_t size = 0;
					
					net_buf = sizeof_netbuf_to_hostbuf(expand_handle_array_struct_f, net_buf, &size);
				}
				pf++;
				break;

			case shrink_handle_array_c:
				*psize += sizeof(shrink_handle_array_t *);
				net_buf += sizeof(u_int8_t);

				/*
				 * Skip over byte that says if a shrink handle
				 * array follows.
				 */
				
				if (*(((u_int8_t *) net_buf) - 1)) {
					size_t size = 0;

					net_buf = sizeof_netbuf_to_hostbuf(shrink_handle_array_struct_f, net_buf, &size);
				}
				pf++;
				break;

			case change_record_array_c:
				*psize += sizeof(change_record_array_t *);
				net_buf += sizeof(u_int8_t);

				/*
				 * Skip over byte that says if a change record
				 * array follows.
				 */
				
				if (*(((u_int8_t *) net_buf) - 1)) {
					size_t size = 0;
					
					net_buf = sizeof_netbuf_to_hostbuf(change_record_array_struct_f, net_buf, &size);
				}
				pf++;
				break;

			case function_info_array_c:
				*psize += sizeof(function_info_array_t *);
				net_buf += sizeof(u_int8_t);

				/*
				 * Skip over byte that says if a function info
				 * array follows.
				 */

				if (*(((u_int8_t *) net_buf) - 1)) {
					size_t size = 0;
					
					net_buf = sizeof_netbuf_to_hostbuf(function_info_array_struct_f, net_buf, &size);
				}
				pf++;
				break;

			case declined_handle_array_c:
				*psize += sizeof(declined_handle_array_t *);
				net_buf += sizeof(u_int8_t);

				/*
				 * Skip over byte that says if a declined handle
				 * array follows.
				 */
				
				if (*(((u_int8_t *) net_buf) - 1)) {
					size_t size = 0;
					
					net_buf = sizeof_netbuf_to_hostbuf(declined_handle_array_struct_f, net_buf, &size);
				}
				pf++;
				break;

			case value_c:
				*psize += sizeof(value_t);
				
				/* Get the the value type. */
				value_type = (value_type_t) *((u_int32_t *) net_buf);
				net_buf += sizeof(u_int32_t);
				
				/* Get whether it is a list or not. */
				value_is_list = (boolean) *((u_int8_t *) net_buf);
				net_buf += sizeof(u_int8_t);

				net_buf = skip_netbuf_value(net_buf, value_type, value_is_list);
				pf++;
				break;

			case option_array_c:
				*psize += sizeof(handle_object_info_t *);
				net_buf += sizeof(u_int8_t);
				if (*(((u_int8_t *) net_buf) - 1)) {
					size_t size = 0;

					net_buf = sizeof_netbuf_to_hostbuf(option_array_struct_f, net_buf, &size);
				}
				pf++;
				break;

			case option_descriptor_c:
				*psize += sizeof(option_descriptor_t *);

				/*
				 * Skip over byte that says if an option
				 * descriptor follows.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					net_buf = skip_netbuf_option_descriptor(net_buf);
				}
				pf++;
				break;

			case extended_info_array_c:
				*psize += sizeof(option_descriptor_t *);

				/*
				 * Skip over byte that says if an extended
				 * info array follows.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* First is the u_int32_t of the count. */
					count = NET_TO_HOST32(*((u_int32_t *) net_buf));
					net_buf += sizeof(u_int32_t);

					for (i = 0; i < count; i++) {
						net_buf = skip_netbuf_extended_info(net_buf);
					}
				}
				pf++;
				break;

			default:
				LOG_SERIOUS("Format character %c slipped past the validater. "
					    "I don't know how to handle it.  I'm skipping it.\n", *pf);
				pf++;
		}
	}

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


/*
 * Functions for converting a network stream into local buffers.
 */

static void * netbuf_to_value(void       * net_buf,
			      value_t    * pvalue,
			      value_type_t type,
			      boolean      value_is_list) {
	uint count;
	value_t * p_value;
	int i;

	LOG_PROC_ENTRY();

	if (value_is_list) {

		/* First byte says if we have a NULL pointer or not. */
		net_buf += sizeof(u_int8_t);

		if (!*(((u_int8_t *) net_buf) - 1)) {
			if (pvalue != NULL) {
				(*pvalue).list = NULL;
			}

			LOG_PROC_EXIT_PTR(net_buf);
			return net_buf;
		}

		count = NET_TO_HOST32(*((u_int32_t *) net_buf));

		if (pvalue != NULL) {
			if ((*pvalue).list != NULL) {
				(*pvalue).list->count = count;
				p_value = (*pvalue).list->value;

			} else {
				p_value = NULL;
			}

		} else {
			p_value = NULL;
		}
		net_buf += sizeof(u_int32_t);

	} else {
		count = 1;
		p_value = pvalue;
	}

	for (i = 0; i < count; i++, p_value++) {
		switch (type) {
			case EVMS_Type_String:
				if (*((u_int8_t *) net_buf) != 0) {
					/* A valid string pointer follows. */
					net_buf += sizeof(u_int8_t);

					if (p_value != NULL) {
						if ((*p_value).s != NULL) {
							strcpy((*p_value).s, (const char *) net_buf);
						} else {
							(*p_value).s = engine_strdup((const char *) net_buf);
						}
					}

					net_buf += strlen((char *) net_buf) + sizeof(char);

				} else {
					/* The string pointer is NULL. */
					net_buf += sizeof(u_int8_t);

					if (p_value != NULL) {
						(*p_value).s = NULL;
					}
				}
				break;

			case EVMS_Type_Boolean:
			case EVMS_Type_Char:
			case EVMS_Type_Unsigned_Char:
			case EVMS_Type_Int8:
			case EVMS_Type_Unsigned_Int8:
				if (p_value != NULL) {
					(*p_value).uc = *((u_int8_t *) net_buf);
				}
				net_buf += sizeof(u_int8_t);
				break;

			case EVMS_Type_Int16:
			case EVMS_Type_Unsigned_Int16:
				if (p_value != NULL) {
					(*p_value).ui16 = NET_TO_HOST16(*((u_int16_t *) net_buf));
				}
				net_buf += sizeof(u_int16_t);
				break;

			case EVMS_Type_Real32:
			case EVMS_Type_Int:
			case EVMS_Type_Int32:
			case EVMS_Type_Unsigned_Int:
			case EVMS_Type_Unsigned_Int32:
				if (p_value != NULL) {
					(*p_value).ui32 = NET_TO_HOST32(*((u_int32_t *) net_buf));
				}
				net_buf += sizeof(u_int32_t);
				break;

			case EVMS_Type_Real64:
			case EVMS_Type_Int64:
			case EVMS_Type_Unsigned_Int64:
				if (p_value != NULL) {
					(*p_value).ui64 = NET_TO_HOST64(*((u_int64_t *) net_buf));
				}
				net_buf += sizeof(u_int64_t);
				break;
		}
	}

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void * netbuf_to_collection(void               * net_buf,
				   value_collection_t * pcoll,
				   collection_type_t    coll_type,
				   value_type_t         value_type) {

	uint count;
	value_list_t * list;
	value_range_t * range;
	int i;

	LOG_PROC_ENTRY();

	switch (coll_type) {
		case EVMS_Collection_None:
			if (pcoll != NULL) {
				(*(pcoll)).list = NULL;
			}
			break;

		case EVMS_Collection_List:
			/* First 32-bit value in the net_buf is the count */
			count = (uint) NET_TO_HOST32(*((u_int32_t *) net_buf));
			net_buf += sizeof(u_int32_t);

			if (pcoll != NULL) {
				list = engine_alloc(sizeof(value_list_t) + count * sizeof(value_t));
				(*(pcoll)).list = list;

				if (list != NULL) {
					list->count = count;
				}

			} else {
				list = NULL;
			}

			for (i = 0; i < count; i++) {
				net_buf = netbuf_to_value(net_buf,
							  (list != NULL) ? &list->value[i] : NULL,
							  value_type,
							  FALSE);
			}

			break;

		case EVMS_Collection_Range:
			if (pcoll != NULL) {
				range = engine_alloc(sizeof(value_range_t));
				(*(pcoll)).range = range;

			} else {
				range = NULL;
			}

			net_buf = netbuf_to_value(net_buf,
						  (range != NULL) ? &range->min : NULL,
						  value_type,
						  FALSE);

			net_buf = netbuf_to_value(net_buf,
						  (range != NULL) ? &range->max : NULL,
						  value_type,
						  FALSE);

			net_buf = netbuf_to_value(net_buf,
						  (range != NULL) ? &range->increment : NULL,
						  value_type,
						  FALSE);
			break;
	}

	LOG_PROC_EXIT_PTR(net_buf);
	return net_buf;
}


static void netbuf_to_hostbuf(char * format,
			      void * net_buf, void * host_buf,
			      uint * p_net_buf_inc, uint * p_host_buf_inc) {

	void * orig_net_buf  = net_buf;
	void * orig_host_buf = host_buf;
	uint net_buf_inc = 0;
	uint host_buf_inc = 0;
	char * pf = format;
	u_int16_t u16;
	u_int32_t u32;
	u_int64_t u64;
	u_int32_t count;
	int i;
	char * pch;
	char tmp_format[64];
	char * string;
	char * * strings;
	size_t alloc_size;
	uint len;
	object_type_t object_type;
	handle_array_t * ha;
	handle_object_info_t * hoi;
	expand_handle_array_t * eha;
	shrink_handle_array_t * sha;
	change_record_array_t * cra;
	function_info_array_t * fia;
	declined_handle_array_t * dha;
	value_type_t value_type;
	collection_type_t collection_type;
	u_int32_t flags;
	boolean value_is_list;
	option_array_t * oa;
	option_descriptor_t * od;
	extended_info_array_t * eia;

	LOG_PROC_ENTRY();

	LOG_EXTRA("Format is: %s\n", format);

	while (*pf != '\0') {
		switch (*pf) {
			case boolean_c:
				if (host_buf != NULL) {
					*((boolean *) host_buf) = (boolean) *((u_int8_t *) net_buf);
					host_buf += sizeof(boolean);
				}
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case count_c:
				/* First is the int of the count. */
				count = NET_TO_HOST32(*((u_int32_t *) net_buf));

				if (host_buf != NULL) {
					*((uint *) host_buf) = (uint) count;
					host_buf += sizeof(uint);
				}

				net_buf += sizeof(u_int32_t);

				/* Skip over "c[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					/* call myself recursively to process the substring. */
					netbuf_to_hostbuf(tmp_format,
							  net_buf, host_buf,
							  &net_buf_inc, &host_buf_inc);

					if (host_buf != NULL) {
						host_buf += host_buf_inc;
					}
					net_buf  += net_buf_inc;
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case count32_c:
				/* First is the u_int32_ of the count. */
				count = NET_TO_HOST32(*((u_int32_t *) net_buf));

				if (host_buf != NULL) {
					*((u_int32_t *) host_buf) = count;
					host_buf += sizeof(u_int32_t);
				}

				net_buf += sizeof(u_int32_t);

				/* Skip over "C[". */
				pf += 2;

				/* Parse out the substring between the brackets. */
				strcpy(tmp_format, pf);
				pch = closing_bracket(tmp_format);
				*pch = '\0';

				/* tmp_format now has the substring between the brackets. */

				for (i = 0; i < count; i++) {
					/* call myself recursively to process the substring. */
					netbuf_to_hostbuf(tmp_format,
							  net_buf, host_buf,
							  &net_buf_inc, &host_buf_inc);

					if (host_buf != NULL) {
						host_buf += host_buf_inc;
					}
					net_buf  += net_buf_inc;
				}

				pf += strlen(tmp_format) + sizeof(char);
				break;

			case int_c:
				if (host_buf != NULL) {
					u32 = *((u_int32_t *) net_buf);
					*((uint *) host_buf) = (int) NET_TO_HOST32(u32);
					host_buf += sizeof(uint);
				}

				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case u_int8_c:
				if (host_buf != NULL) {
					*((u_int8_t *) host_buf) = *((u_int8_t *) net_buf);
					host_buf += sizeof(u_int8_t);
				}

				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case u_int16_c:
				if (host_buf != NULL) {
					u16 = *((u_int16_t *) net_buf);
					*((u_int16_t *) host_buf) = NET_TO_HOST16(u16);
					host_buf += sizeof(u_int16_t);
				}

				net_buf += sizeof(u_int16_t);
				pf++;
				break;

			case u_int32_c:
				if (host_buf != NULL) {
					u32 = *((u_int32_t *) net_buf);
					*((u_int32_t *) host_buf) = NET_TO_HOST32(u32);
					host_buf += sizeof(u_int32_t);
				}

				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case u_int64_c:
				if (host_buf != NULL) {
					u64 = *((u_int64_t *) net_buf);
					*((u_int64_t *) host_buf) = NET_TO_HOST64(u64);
					host_buf += sizeof(u_int64_t);
				}

				net_buf += sizeof(u_int64_t);
				pf++;
				break;

			case ptr_c:
				if (*(pf + 1) == '{') {
					/* Skip over "p{". */
					pf += 2;

					/* Parse out the substring between the braces. */
					strcpy(tmp_format, pf);
					pch = closing_brace(tmp_format);
					*pch = '\0';

					net_buf += sizeof(u_int8_t);

					if (*(((u_int8_t *) net_buf) - 1)) {
						void * sub_buf;
						uint sub_buf_inc;

						if (host_buf != NULL) {
							alloc_size = 0;
							sizeof_netbuf_to_hostbuf(tmp_format, net_buf, &alloc_size);
							sub_buf = engine_alloc(alloc_size);
							*((int * *) host_buf) = (int *) sub_buf;
							host_buf += sizeof(void *);

						} else {
							sub_buf = NULL;
						}

						/* tmp_format now has the substring between the braces. */

						/* Call myself recursively to process the substring. */
						netbuf_to_hostbuf(tmp_format,
								  net_buf, sub_buf,
								  &sub_buf_inc, &net_buf_inc);


						net_buf  += net_buf_inc;

					} else {
						if (host_buf != NULL) {
							*((int * *) host_buf) = NULL;
							host_buf += sizeof(void *);
						}
					}

					pf += strlen(tmp_format) + sizeof(char);

				} else {
					/* It's a generic pointer. */
					/*
					 * Generic pointers come in 64-bits.
					 */
					if (host_buf != NULL) {
						u64 = NET_TO_HOST64(*((u_int64_t *) net_buf));
						*((unsigned long * *) host_buf) = (unsigned long *) (unsigned long) u64;
						host_buf += sizeof(void *);
					}

					net_buf += sizeof(u_int64_t);
					pf++;
				}
				break;

			case string_c:
				net_buf += sizeof(u_int8_t);
				if (*(((u_int8_t *) net_buf) - 1)) {
					/* A valid string follows. */

					if (host_buf != NULL) {
						*((char * *) host_buf) = engine_strdup(net_buf);
						host_buf += sizeof(char *);
					}

					net_buf += strlen(net_buf) + sizeof(char);

				} else {
					/* The string pointer is NULL. */
					if (host_buf != NULL) {
						*((char * *) host_buf) = NULL;
						host_buf += sizeof(char *);
					}
				}

				pf++;
				break;

			case strings_c:
				net_buf += sizeof(u_int8_t);
				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid strings follow. */

					/*
					 * Count the number of strings so that
					 * we know how big to allocate the array
					 * of pointers to the strings.
					 */
					string = (char *) net_buf;
					count = 0;
					while (*string != '\0') {
						count++;
						string += strlen(string) + sizeof(char);
					}

					strings = engine_alloc((count + 1) * sizeof(char *));
					*((char * * *) host_buf) = strings;

					for (i = 0; *((char *) net_buf) != '\0'; i++) {
						if (strings != NULL) {
							strings[i] = engine_strdup((char *) net_buf);
						}

						net_buf += strlen((char *) net_buf) + sizeof(char);
					}

					if (strings != NULL) {
						strings[count] = NULL;
					}

					net_buf += sizeof(char);

				} else {
					/*
					 * No strings passed.  Set the
					 * string array pointer to NULL.
					 */
					if (host_buf != NULL) {
						*((char * * *) host_buf) = NULL;
					}
				}

				if (host_buf != NULL) {
					host_buf += sizeof(char * *);
				}

				pf++;
				break;

			case bytes_c:
				pf++;
				len = atoi(pf);

				if (host_buf != NULL) {
					memcpy(host_buf, net_buf, len);
					host_buf += len;
				}

				net_buf += len;
				pf += strspn(pf, "0123456789");
				break;

			case handle_array_c:
				ha = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid handle_array_t follows. */
					
					if (host_buf != NULL) {
						/* First is the u_int32_ of the count. */
						count = NET_TO_HOST32(*((u_int32_t *) net_buf));
						ha = engine_alloc(sizeof(handle_array_t) +
								  sizeof(object_handle_t) * count);
					}

					netbuf_to_hostbuf(handle_array_struct_f,
							  net_buf, ha,
							  &net_buf_inc, &host_buf_inc);

					net_buf  += net_buf_inc;
				}

				if (host_buf != NULL) {
					*((handle_array_t * *) host_buf) = ha;
					host_buf += sizeof(handle_array_t *);
				}

				pf++;
				break;

			case handle_object_info_c:
				hoi = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);
				if (host_buf != NULL) {
					if (*(((u_int8_t *) net_buf) - 1)) {
						/* Valid handle_object_info_t follows. */
						hoi = engine_alloc(sizeof(handle_object_info_t));
					}

					*((handle_object_info_t * *) host_buf) = hoi;
					host_buf += sizeof(handle_object_info_t *);
				}

				/* If no pointer was sent we are finished. */
				if (!*(((u_int8_t *) net_buf) - 1)) {
					break;
				}

				object_type = (object_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				if (hoi != NULL) {
					hoi->type = object_type;
				}

				switch (object_type) {
					case DISK:
					case SEGMENT:
					case REGION:
					case EVMS_OBJECT:
						netbuf_to_hostbuf(storage_object_info_f,
								  net_buf, (hoi != NULL) ? &hoi->info : NULL,
								  &net_buf_inc, &host_buf_inc);
						break;

					case CONTAINER:
						netbuf_to_hostbuf(storage_container_info_f,
								  net_buf, (hoi != NULL) ? &hoi->info : NULL,
								  &net_buf_inc, &host_buf_inc);
						break;

					case VOLUME:
						netbuf_to_hostbuf(logical_volume_info_f,
								  net_buf, (hoi != NULL) ? &hoi->info : NULL,
								  &net_buf_inc, &host_buf_inc);
						break;

					case PLUGIN:
						netbuf_to_hostbuf(plugin_info_f,
								  net_buf, (hoi != NULL) ? &hoi->info : NULL,
								  &net_buf_inc, &host_buf_inc);
						break;

					default:
						net_buf_inc = 0;
						break;
				}

				net_buf += net_buf_inc;

				pf++;
				break;

			case expand_handle_array_c:
				eha = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid expand_handle_array_t follows. */

					if (host_buf != NULL) {
						/* First is the u_int32_ of the count. */
						count = NET_TO_HOST32(*((u_int32_t *) net_buf));
						eha = engine_alloc(sizeof(expand_handle_array_t) +
								   sizeof(expand_handle_t) * count);
					}

					netbuf_to_hostbuf(expand_handle_array_struct_f,
							  net_buf, eha,
							  &net_buf_inc, &host_buf_inc);

					net_buf += net_buf_inc;
				}

				if (host_buf != NULL) {
					*((expand_handle_array_t * *) host_buf) = eha;
					host_buf += sizeof(expand_handle_array_t *);
				}

				pf++;
				break;

			case shrink_handle_array_c:
				sha = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid shrink_handle_array_t follows. */

					if (host_buf != NULL) {
						/* First is the u_int32_ of the count. */
						count = NET_TO_HOST32(*((u_int32_t *) net_buf));
						sha = engine_alloc(sizeof(shrink_handle_array_t) +
								   sizeof(shrink_handle_t) * count);
					}

					netbuf_to_hostbuf(shrink_handle_array_struct_f,
							  net_buf, sha,
							  &net_buf_inc, &host_buf_inc);

					net_buf  += net_buf_inc;
				}

				if (host_buf != NULL) {
					*((shrink_handle_array_t * *) host_buf) = sha;
					host_buf += sizeof(shrink_handle_array_t *);
				}

				pf++;
				break;

			case change_record_array_c:
				cra = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid change_record_array_t follows. */

					if (host_buf != NULL) {
						count = NET_TO_HOST32(*((u_int32_t *) net_buf));
						cra = engine_alloc(sizeof(change_record_array_t) +
								   sizeof(change_record_t) * count);
					}

					netbuf_to_hostbuf(change_record_array_struct_f,
							  net_buf, cra,
							  &net_buf_inc, &host_buf_inc);

					net_buf  += net_buf_inc;
                                }

				if (host_buf != NULL) {
					*((change_record_array_t * *) host_buf) = cra;
					host_buf += sizeof(change_record_array_t *);
				}

				pf++;
				break;

			case function_info_array_c:
				fia = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid function_info_array_t follows. */

					if (host_buf != NULL) {
						/* First is the u_int32_ of the count. */
						count = NET_TO_HOST32(*((u_int32_t *) net_buf));
						fia = engine_alloc(sizeof(function_info_array_t) +
								   sizeof(function_info_t) * count);
					}

					netbuf_to_hostbuf(function_info_array_struct_f,
							  net_buf, fia,
							  &net_buf_inc, &host_buf_inc);

					net_buf  += net_buf_inc;
				}

				if (host_buf != NULL) {
					*((function_info_array_t * *) host_buf) = fia;
					host_buf += sizeof(function_info_array_t *);
				}

				pf++;
				break;

			case declined_handle_array_c:
				dha = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid declined_handle_array_t follows. */

					if (host_buf != NULL) {
						/* First is the u_int32_ of the count. */
						count = NET_TO_HOST32(*((u_int32_t *) net_buf));
						dha = engine_alloc(sizeof(declined_handle_array_t) +
								   sizeof(declined_handle_t) * count);
					}

					netbuf_to_hostbuf(declined_handle_array_struct_f,
							  net_buf, dha,
							  &net_buf_inc, &host_buf_inc);
					net_buf  += net_buf_inc;
				}

				if (host_buf != NULL) {
					*((declined_handle_array_t * *) host_buf) = dha;
					host_buf += sizeof(declined_handle_array_t *);
				}

				pf++;
				break;

			case value_c:
				value_type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				value_is_list = (boolean) *((u_int8_t *) net_buf);
				net_buf += sizeof(u_int8_t);

				if ((host_buf != NULL) &&
				    value_is_list &&
				    (((value_t *) host_buf)->list == NULL)) {
					u_int8_t * have_ptr = (u_int8_t *) net_buf;
					u_int32_t * p_count = (u_int32_t *) (have_ptr + 1);
					u_int32_t count = NET_TO_HOST32(*p_count);

					host_buf = engine_alloc(sizeof(value_list_t) + sizeof(value_t) * count);
				}
				net_buf = netbuf_to_value(net_buf,
							  (value_t *) host_buf,
							  value_type,
							  value_is_list);
				host_buf += sizeof(value_t);
				pf++;
				break;

			case option_array_c:
				oa = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid option_array_t follows. */

					if (host_buf != NULL) {
						/* First is the u_int32_ of the count. */
						count = NET_TO_HOST32(*((u_int32_t *) net_buf));
						oa = engine_alloc(sizeof(option_array_t) +
								  sizeof(key_value_pair_t) * count);
					}

					netbuf_to_hostbuf(option_array_f,
							  net_buf, oa,
							  &net_buf_inc, &host_buf_inc);

					net_buf  += net_buf_inc;
				}

				if (host_buf != NULL) {
					*((option_array_t * *) host_buf) = oa;
					host_buf += sizeof(option_array_t *);
				}

				pf++;
				break;

			case option_descriptor_c:
				od = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid option_descriptor_t follows. */
					if (host_buf != NULL) {
						od = engine_alloc(sizeof(option_descriptor_t));
					}
				}

				if (od != NULL) {
					if (*((char *) net_buf) != '\0') {
						od->name = engine_strdup((const char *) net_buf);
					} else {
						od->name = NULL;
					}
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				if (od != NULL) {
					if (*((char *) net_buf) != '\0') {
						od->title = engine_strdup((const char *) net_buf);
					} else {
						od->title = NULL;
					}
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				if (od != NULL) {
					if (*((char *) net_buf) != '\0') {
						od->tip = engine_strdup((const char *) net_buf);
					} else {
						od->tip = NULL;
					}
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				if (od != NULL) {
					if (*((char *) net_buf) != '\0') {
						od->help = engine_strdup((const char *) net_buf);
					} else {
						od->help = NULL;
					}
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				value_type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				if (od != NULL) {
					od->type = value_type;
				}
				net_buf += sizeof(u_int32_t);

				if (od != NULL) {
					od->unit = (value_unit_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				}
				net_buf += sizeof(u_int32_t);

				if (od != NULL) {
					od->format = (value_format_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				}
				net_buf += sizeof(u_int32_t);

				if (od != NULL) {
					od->min_len = NET_TO_HOST32(*((u_int32_t *) net_buf));
				}
				net_buf += sizeof(u_int32_t);

				if (od != NULL) {
					od->max_len = NET_TO_HOST32(*((u_int32_t *) net_buf));
				}
				net_buf += sizeof(u_int32_t);

				flags = NET_TO_HOST32(*((u_int32_t *) net_buf));
				if (od != NULL) {
					od->flags = flags;
				}
				net_buf += sizeof(u_int32_t);

				collection_type = (collection_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				if (od != NULL) {
					od->constraint_type = collection_type;
				}
				net_buf += sizeof(u_int32_t);

				net_buf = netbuf_to_collection(net_buf,
							       (od != NULL) ? &od->constraint : NULL,
							       collection_type,
							       value_type);

				if ((od != NULL) &&
				    (flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST) &&
				    (od->value.list == NULL)) {
					u_int8_t * have_ptr = (u_int8_t *) net_buf;
					u_int32_t * p_count = (u_int32_t *) (have_ptr + 1);
					u_int32_t count = NET_TO_HOST32(*p_count);

					od->value.list = engine_alloc(sizeof(value_list_t) + sizeof(value_t) * count);
				}
				net_buf = netbuf_to_value(net_buf,
							  (od != NULL) ? &od->value : NULL,
							  value_type,
							  flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST);

				if (od != NULL) {
					od->group.group_number = NET_TO_HOST32(*((u_int32_t *) net_buf));
				}
				net_buf += sizeof(u_int32_t);

				if (od != NULL) {
					od->group.group_level = NET_TO_HOST32(*((u_int32_t *) net_buf));
				}
				net_buf += sizeof(u_int32_t);

				if (od != NULL) {
					if (*((char *) net_buf) != '\0') {
						od->group.group_name = engine_strdup((const char *) net_buf);
					} else {
						od->group.group_name = NULL;
					}
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				if (host_buf != NULL) {
					*((option_descriptor_t * *) host_buf) = od;
					host_buf += sizeof(option_descriptor_t *);
				}

				pf++;
				break;

			case extended_info_array_c:
				eia = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid extended_info_array_t follows. */
					/* First is the u_int32_ of the count. */
					count = NET_TO_HOST32(*((u_int32_t *) net_buf));
					eia = engine_alloc(sizeof(extended_info_array_t) +
							   sizeof(extended_info_t) * count);
				}

				/* First is the u_int32_ of the count. */
				count = NET_TO_HOST32(*((u_int32_t *) net_buf));
				if (eia != NULL) {
					eia->count = count;
				}
				net_buf += sizeof(u_int32_t);

				for (i = 0; i < count; i++) {
					extended_info_t * ei = NULL;

					if (eia != NULL) {
						ei = &eia->info[i];
					}

					if (ei != NULL) {
						if (*((char *) net_buf) != '\0') {
							ei->name = engine_strdup((const char *) net_buf);
						} else {
							ei->name = NULL;
						}
					}
					net_buf += strlen((char *) net_buf) + sizeof(char);

					if (ei != NULL) {
						if (*((char *) net_buf) != '\0') {
							ei->title = engine_strdup((const char *) net_buf);
						} else {
							ei->title = NULL;
						}
					}
					net_buf += strlen((char *) net_buf) + sizeof(char);

					if (ei != NULL) {
						if (*((char *) net_buf) != '\0') {
							ei->desc = engine_strdup((const char *) net_buf);
						} else {
							ei->desc = NULL;
						}
					}
					net_buf += strlen((char *) net_buf) + sizeof(char);

					value_type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
					if (ei != NULL) {
						ei->type = value_type;
					}
					net_buf += sizeof(u_int32_t);

					if (ei != NULL) {
						ei->unit = (value_unit_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
					}
					net_buf += sizeof(u_int32_t);

					if (ei != NULL) {
						ei->format = (value_format_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
					}
					net_buf += sizeof(u_int32_t);

					net_buf = netbuf_to_value(net_buf,
								  (ei != NULL) ? &ei->value : NULL,
								  value_type,
								  FALSE);

					collection_type = (collection_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
					if (ei != NULL) {
						ei->collection_type = collection_type;
					}
					net_buf += sizeof(u_int32_t);

					net_buf = netbuf_to_collection(net_buf,
								       (ei != NULL) ? &ei->collection : NULL,
								       collection_type,
								       value_type);

					if (ei != NULL) {
						ei->group.group_number = NET_TO_HOST32(*((u_int32_t *) net_buf));
					}
					net_buf += sizeof(u_int32_t);

					if (ei != NULL) {
						ei->group.group_level = NET_TO_HOST32(*((u_int32_t *) net_buf));
					}
					net_buf += sizeof(u_int32_t);


					if (ei != NULL) {
						if (*((char *) net_buf) != '\0') {
							ei->group.group_name = engine_strdup((const char *) net_buf);
						} else {
							ei->group.group_name = NULL;
						}
					}
					net_buf += strlen((char *) net_buf) + sizeof(char);

					if (ei != NULL) {
						ei->flags = NET_TO_HOST16(*((u_int16_t *) net_buf));
					}
					net_buf += sizeof(u_int16_t);
				}

				if (host_buf != NULL) {
					*((extended_info_array_t * *) host_buf) = eia;
					host_buf += sizeof(extended_info_array_t *);
				}

				pf++;
				break;

			default:
				LOG_SERIOUS("Format character %c slipped past the validater. "
					    "I don't know how to handle it.  I'm skipping it.\n", *pf);
				pf++;
		}
	}

	*p_net_buf_inc  = net_buf  - orig_net_buf;
	*p_host_buf_inc = host_buf - orig_host_buf;

	LOG_PROC_EXIT_VOID();
	return;
}


static void free_string_array_contents(void * arg) {

	char * * strings = (char * *) arg;
	int i;

	for (i = 0; strings[i] != NULL; i++) {
		engine_free(strings[i]);
	}
}


int evms_net_to_host(void * net_buf, char * format, ...) {

	int rc;
	char * pf = format;
	va_list args;
	void * var_ptr;
	uint net_buf_inc;
	uint host_buf_inc;
	u_int16_t u16;
	u_int32_t u32;
	u_int64_t u64;
	int i;
	uint count;
	char * pch;
	char tmp_format[64];
	char * string;
	char * * strings;
	size_t alloc_size;
	object_type_t object_type;
	handle_array_t * ha;
	handle_object_info_t * hoi;
	expand_handle_array_t * eha;
	shrink_handle_array_t * sha;
	change_record_array_t * cra;
	function_info_array_t * fia;
	declined_handle_array_t * dha;
	value_type_t value_type;
	collection_type_t collection_type;
	u_int32_t flags;
	boolean value_is_list;
	option_array_t * oa;
	option_descriptor_t * od;
	extended_info_array_t * eia;

	LOG_PROC_ENTRY();

	LOG_EXTRA("Format is: %s\n", format);

	rc = validate_format(format);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	va_start(args, format);

	while (*pf != '\0') {
		var_ptr = va_arg(args, void *);

		switch (*pf) {
			case boolean_c:
				if (var_ptr != NULL) {
					*((boolean *) var_ptr) = (boolean) *((u_int8_t *) net_buf);
				}
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case count_c:
			case count32_c:
				/*
				 * Arrays are not passed as args.
				 * Pointers to arrays are, but not
				 * arrays directly themsleves.
				 */
				LOG_ERROR("Arrays are not passed as arguments. "
					  "Try using \"p{c[...]}\" for the array arguement.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case int_c:
				if (var_ptr != NULL) {
					u32 = NET_TO_HOST32(*((u_int32_t *) net_buf));
					*((int *) var_ptr) = (int) u32;
				}
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case u_int8_c:
				if (var_ptr != NULL) {
					*((u_int8_t *) var_ptr) = *((u_int8_t *) net_buf);
				}
				net_buf += sizeof(u_int8_t);
				pf++;
				break;

			case u_int16_c:
				if (var_ptr != NULL) {
					u16 = NET_TO_HOST16(*((u_int16_t *) net_buf));
					*((u_int16_t *) var_ptr) = u16;
				}
				net_buf += sizeof(u_int16_t);
				pf++;
				break;

			case u_int32_c:
				if (var_ptr != NULL) {
					u32 = NET_TO_HOST32(*((u_int32_t *) net_buf));
					*((u_int32_t *) var_ptr) = u32;
				}
				net_buf += sizeof(u_int32_t);
				pf++;
				break;

			case u_int64_c:
				if (var_ptr != NULL) {
					u64 = NET_TO_HOST64(*((u_int64_t *) net_buf));
					*((u_int64_t *) var_ptr) = u64;
				}
				net_buf += sizeof(u_int64_t);
				pf++;
				break;

			case ptr_c:
				if (*(pf + 1) == '{') {
					/* Skip over "p{". */
					pf += 2;

					/* Parse out the substring between the braces. */
					strcpy(tmp_format, pf);
					pch = closing_brace(tmp_format);
					*pch = '\0';

					net_buf += sizeof(u_int8_t);

					if (*(((u_int8_t *) net_buf) - 1)) {
						void * sub_buf;
						uint sub_buf_inc;

						if (var_ptr != NULL) {
							alloc_size = 0;
							sizeof_netbuf_to_hostbuf(tmp_format, net_buf, &alloc_size);
							sub_buf = alloc_app_struct(alloc_size, NULL);
							*((int * *) var_ptr) = (int *) sub_buf;
							var_ptr += sizeof(void *);

						} else {
							sub_buf = NULL;
						}

						/* tmp_format now has the substring between the braces. */

						/* Call myself recursively to process the substring. */
						netbuf_to_hostbuf(tmp_format,
								  net_buf, sub_buf,
								  &sub_buf_inc, &net_buf_inc);


						net_buf  += net_buf_inc;

					} else {
						if (var_ptr != NULL) {
							*((int * *) var_ptr) = NULL;
							var_ptr += sizeof(void *);
						}
					}

					pf += strlen(tmp_format) + sizeof(char);

				} else {
					/* It's a generic pointer. */
					/*
					 * Generic pointers come in 64-bits.
					 */
					if (var_ptr != NULL) {
						u64 = NET_TO_HOST64(*((u_int64_t *) net_buf));
						*((unsigned long * *) var_ptr) = (unsigned long *) (unsigned long) u64;
						var_ptr += sizeof(void *);
					}

					net_buf += sizeof(u_int64_t);
					pf++;
				}
				break;

			case string_c:
				net_buf += sizeof(u_int8_t);
				if (*(((u_int8_t *) net_buf) - 1)) {
					/* A valid string follows. */

					if (var_ptr != NULL) {
						string = alloc_app_struct(strlen(net_buf) + sizeof(char), NULL);
						if (string != NULL) {
							strcpy(string, (char *) net_buf);
							*((char * *) var_ptr) = string;
						}
					}

					net_buf += strlen(net_buf) + sizeof(char);

				} else {
					/* The string pointer is NULL. */
					if (var_ptr != NULL) {
						*((char * *) var_ptr) = NULL;
					}
				}

				pf++;
				break;

			case strings_c:
				strings = NULL;

				net_buf += sizeof(u_int8_t);
				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid strings follow. */

					/*
					 * Count the number of strings so that
					 * we know how big to allocate the array
					 * of pointers to the strings.
					 */
					string = (char *) net_buf;
					count = 0;
					while (*string != '\0') {
						count++;
						string += strlen(string) + sizeof(char);
					}

					if (var_ptr != NULL) {
						strings = alloc_app_struct((count + 1) * sizeof(char *),
									   free_string_array_contents);
						*((char * * *) var_ptr) = strings;
					}

					for (i = 0; *((char *) net_buf) != '\0'; i++) {
						if (strings != NULL) {
							strings[i] = engine_strdup((char *) net_buf);
						}

						net_buf += strlen((char *) net_buf) + sizeof(char);
					}

					if (strings != NULL) {
						strings[count] = NULL;
					}

					net_buf += sizeof(char);

				} else {
					/*
					 * No strings passed.  Set the
					 * string array pointer to NULL.
					 */
					if (var_ptr != NULL) {
						*((char * * *) var_ptr) = NULL;
					}
				}

				pf++;
				break;

			case bytes_c:
				/*
				 * Byte strings are not passed as args.
				 * Pointers to them are, but not the
				 * bytes directly themsleves.
				 */
				LOG_ERROR("Byte strings are not passed as arguments. "
					  "Try using \"p{xnnn}\" for the byte string argument.\n");
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
				break;

			case handle_array_c:
				ha = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);
				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid handle_array_t follows. */

					if (var_ptr != NULL) {
						/* First is the u_int32_ of the count. */
						count = NET_TO_HOST32(*((u_int32_t *) net_buf));
						ha = alloc_app_struct(sizeof(handle_array_t) +
								      sizeof(object_handle_t) * count,
								      NULL);
					}

					netbuf_to_hostbuf(handle_array_struct_f,
							  net_buf, ha,
							  &net_buf_inc, &host_buf_inc);

					net_buf += net_buf_inc;
				}

				if (var_ptr != NULL) {
					*((handle_array_t * *) var_ptr) = ha;
				}

				pf++;
				break;

			case handle_object_info_c:
				hoi = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);
				if (var_ptr != NULL) {
					if (*(((u_int8_t *) net_buf) - 1)) {
						/* Valid handle_object_info_t follows. */
						hoi = alloc_app_struct(sizeof(handle_object_info_t),
								       free_info_object_contents);
					}

					*((handle_object_info_t * *) var_ptr) = hoi;
				}

				/* If no pointer was sent we are finished. */
				if (!*(((u_int8_t *) net_buf) - 1)) {
					break;
				}

				object_type = (object_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				if (hoi != NULL) {
					hoi->type = object_type;
				}

				switch (object_type) {
					case DISK:
					case SEGMENT:
					case REGION:
					case EVMS_OBJECT:
						netbuf_to_hostbuf(storage_object_info_f,
								  net_buf, (hoi != NULL) ? &hoi->info : NULL,
								  &net_buf_inc, &host_buf_inc);
						break;

					case CONTAINER:
						netbuf_to_hostbuf(storage_container_info_f,
								  net_buf, (hoi != NULL) ? &hoi->info : NULL,
								  &net_buf_inc, &host_buf_inc);
						break;

					case VOLUME:
						netbuf_to_hostbuf(logical_volume_info_f,
								  net_buf, (hoi != NULL) ? &hoi->info : NULL,
								  &net_buf_inc, &host_buf_inc);
						break;

					case PLUGIN:
						netbuf_to_hostbuf(plugin_info_f,
								  net_buf, (hoi != NULL) ? &hoi->info : NULL,
								  &net_buf_inc, &host_buf_inc);
						break;

					default:
						net_buf_inc = 0;
						break;
				}

				net_buf += net_buf_inc;

				pf++;
				break;

			case expand_handle_array_c:
				eha = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid expand_handle_array_t follows. */

					if (var_ptr != NULL) {
						/* First is the u_int32_ of the count. */
						count = NET_TO_HOST32(*((u_int32_t *) net_buf));
						eha = alloc_app_struct(sizeof(expand_handle_array_t) +
								       sizeof(expand_handle_t) * count,
								       NULL);
					}

					netbuf_to_hostbuf(expand_handle_array_struct_f,
							  net_buf, eha,
							  &net_buf_inc, &host_buf_inc);

					net_buf += net_buf_inc;
				}

				if (var_ptr != NULL) {
					*((expand_handle_array_t * *) var_ptr) = eha;
				}

				pf++;
				break;

			case shrink_handle_array_c:
				sha = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid shrink_handle_array_t follows. */

					if (var_ptr != NULL) {
						/* First is the u_int32_ of the count. */
						count = NET_TO_HOST32(*((u_int32_t *) net_buf));
						sha = alloc_app_struct(sizeof(shrink_handle_array_t) +
								       sizeof(shrink_handle_t) * count,
								       NULL);
					}

					netbuf_to_hostbuf(shrink_handle_array_struct_f,
							  net_buf, sha,
							  &net_buf_inc, &host_buf_inc);

					net_buf  += net_buf_inc;
				}

				if (var_ptr != NULL) {
					*((shrink_handle_array_t * *) var_ptr) = sha;
				}

				pf++;
				break;

			case change_record_array_c:
				cra = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid change_record_array_t follows. */

					if (var_ptr != NULL) {
						count = NET_TO_HOST32(*((u_int32_t *) net_buf));
						cra = alloc_app_struct(sizeof(change_record_array_t) +
								       sizeof(change_record_t) * count,
								       free_changes_pending_record_array_contents);
					}

					netbuf_to_hostbuf(change_record_array_struct_f,
							  net_buf, cra,
							  &net_buf_inc, &host_buf_inc);

					net_buf  += net_buf_inc;
                                }

				if (var_ptr != NULL) {
					*((change_record_array_t * *) var_ptr) = cra;
				}

				pf++;
				break;

			case function_info_array_c:
				fia = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid function_info_array_t follows. */

					if (var_ptr != NULL) {
						/* First is the u_int32_ of the count. */
						count = NET_TO_HOST32(*((u_int32_t *) net_buf));
						fia = alloc_app_struct(sizeof(function_info_array_t) +
								       sizeof(function_info_t) * count,
								       free_function_info_array_contents);
					}

					netbuf_to_hostbuf(function_info_array_struct_f,
							  net_buf, fia,
							  &net_buf_inc, &host_buf_inc);

					net_buf  += net_buf_inc;
				}

				if (var_ptr != NULL) {
					*((function_info_array_t * *) var_ptr) = fia;
				}

				pf++;
				break;

			case declined_handle_array_c:
				dha = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid declined_handle_array_t follows. */

					if (var_ptr != NULL) {
						/* First is the u_int32_ of the count. */
						count = NET_TO_HOST32(*((u_int32_t *) net_buf));
						dha = alloc_app_struct(sizeof(declined_handle_array_t) +
								       sizeof(declined_handle_t) * count,
								       NULL);
					}

					netbuf_to_hostbuf(declined_handle_array_struct_f,
							  net_buf, dha,
							  &net_buf_inc, &host_buf_inc);
					net_buf  += net_buf_inc;
				}

				if (var_ptr != NULL) {
					*((declined_handle_array_t * *) var_ptr) = dha;
				}

				pf++;
				break;

			case value_c:
				value_type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				net_buf += sizeof(u_int32_t);

				value_is_list = (boolean) *((u_int8_t *) net_buf);
				net_buf += sizeof(u_int8_t);

				net_buf = netbuf_to_value(net_buf,
							  (value_t *) var_ptr,
							  value_type,
							  value_is_list);
				pf++;
				break;

			case option_array_c:
				oa = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid option_array_t follows. */

					if (var_ptr != NULL) {
						/* First is the u_int32_ of the count. */
						count = NET_TO_HOST32(*((u_int32_t *) net_buf));
						oa = alloc_app_struct(sizeof(option_array_t) +
								      sizeof(key_value_pair_t) * count,
								      free_option_array_contents);
					}

					netbuf_to_hostbuf(option_array_f,
							  net_buf, oa,
							  &net_buf_inc, &host_buf_inc);

					net_buf  += net_buf_inc;
				}

				if (var_ptr != NULL) {
					*((option_array_t * *) var_ptr) = oa;
				}

				pf++;
				break;

			case option_descriptor_c:
				od = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid option_descriptor_t follows. */
					if (var_ptr != NULL) {
						od = alloc_app_struct(sizeof(option_descriptor_t),
								      free_option_descriptor_contents);
					}
				}

				if (od != NULL) {
					if (*((char *) net_buf) != '\0') {
						od->name = engine_strdup((const char *) net_buf);
					} else {
						od->name = NULL;
					}
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				if (od != NULL) {
					if (*((char *) net_buf) != '\0') {
						od->title = engine_strdup((const char *) net_buf);
					} else {
						od->title = NULL;
					}
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				if (od != NULL) {
					if (*((char *) net_buf) != '\0') {
						od->tip = engine_strdup((const char *) net_buf);
					} else {
						od->tip = NULL;
					}
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				if (od != NULL) {
					if (*((char *) net_buf) != '\0') {
						od->help = engine_strdup((const char *) net_buf);
					} else {
						od->help = NULL;
					}
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				value_type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				if (od != NULL) {
					od->type = value_type;
				}
				net_buf += sizeof(u_int32_t);

				if (od != NULL) {
					od->unit = (value_unit_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				}
				net_buf += sizeof(u_int32_t);

				if (od != NULL) {
					od->format = (value_format_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				}
				net_buf += sizeof(u_int32_t);

				if (od != NULL) {
					od->min_len = NET_TO_HOST32(*((u_int32_t *) net_buf));
				}
				net_buf += sizeof(u_int32_t);

				if (od != NULL) {
					od->max_len = NET_TO_HOST32(*((u_int32_t *) net_buf));
				}
				net_buf += sizeof(u_int32_t);

				flags = NET_TO_HOST32(*((u_int32_t *) net_buf));
				if (od != NULL) {
					od->flags = flags;
				}
				net_buf += sizeof(u_int32_t);

				collection_type = (collection_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
				if (od != NULL) {
					od->constraint_type = collection_type;
				}
				net_buf += sizeof(u_int32_t);

				net_buf = netbuf_to_collection(net_buf,
							       (od != NULL) ? &od->constraint : NULL,
							       collection_type,
							       value_type);

				if ((od != NULL) &&
				    (flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST) &&
				    (od->value.list == NULL)) {
					u_int8_t * have_ptr = (u_int8_t *) net_buf;
					u_int32_t * p_count = (u_int32_t *) (have_ptr + 1);
					u_int32_t count = NET_TO_HOST32(*p_count);

					od->value.list = engine_alloc(sizeof(value_list_t) + sizeof(value_t) * count);
				}
				net_buf = netbuf_to_value(net_buf,
							  (od != NULL) ? &od->value : NULL,
							  value_type,
							  flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST);

				if (od != NULL) {
					od->group.group_number = NET_TO_HOST32(*((u_int32_t *) net_buf));
				}
				net_buf += sizeof(u_int32_t);

				if (od != NULL) {
					od->group.group_level = NET_TO_HOST32(*((u_int32_t *) net_buf));
				}
				net_buf += sizeof(u_int32_t);

				if (od != NULL) {
					if (*((char *) net_buf) != '\0') {
						od->group.group_name = engine_strdup((const char *) net_buf);
					} else {
						od->group.group_name = NULL;
					}
				}
				net_buf += strlen((char *) net_buf) + sizeof(char);

				if (var_ptr != NULL) {
					*((option_descriptor_t * *) var_ptr) = od;
				}

				pf++;
				break;

			case extended_info_array_c:
				eia = NULL;

				/*
				 * Step over byte that says if a valid pointer
				 * was sent.
				 */
				net_buf += sizeof(u_int8_t);

				if (*(((u_int8_t *) net_buf) - 1)) {
					/* Valid extended_info_array_t follows. */
					/* First is the u_int32_ of the count. */
					count = NET_TO_HOST32(*((u_int32_t *) net_buf));
					eia = alloc_app_struct(sizeof(extended_info_array_t) +
							       sizeof(extended_info_t) * count,
							       free_extended_info_array_contents);
				}

				/* First is the u_int32_ of the count. */
				count = NET_TO_HOST32(*((u_int32_t *) net_buf));
				if (eia != NULL) {
					eia->count = count;
				}
				net_buf += sizeof(u_int32_t);

				for (i = 0; i < count; i++) {
					extended_info_t * ei = NULL;

					if (eia != NULL) {
						ei = &eia->info[i];
					}

					if (ei != NULL) {
						if (*((char *) net_buf) != '\0') {
							ei->name = engine_strdup((const char *) net_buf);
						} else {
							ei->name = NULL;
						}
					}
					net_buf += strlen((char *) net_buf) + sizeof(char);

					if (ei != NULL) {
						if (*((char *) net_buf) != '\0') {
							ei->title = engine_strdup((const char *) net_buf);
						} else {
							ei->title = NULL;
						}
					}
					net_buf += strlen((char *) net_buf) + sizeof(char);

					if (ei != NULL) {
						if (*((char *) net_buf) != '\0') {
							ei->desc = engine_strdup((const char *) net_buf);
						} else {
							ei->desc = NULL;
						}
					}
					net_buf += strlen((char *) net_buf) + sizeof(char);

					value_type = (value_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
					if (ei != NULL) {
						ei->type = value_type;
					}
					net_buf += sizeof(u_int32_t);

					if (ei != NULL) {
						ei->unit = (value_unit_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
					}
					net_buf += sizeof(u_int32_t);

					if (ei != NULL) {
						ei->format = (value_format_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
					}
					net_buf += sizeof(u_int32_t);

					net_buf = netbuf_to_value(net_buf,
								  (ei != NULL) ? &ei->value : NULL,
								  value_type,
								  FALSE);

					collection_type = (collection_type_t) NET_TO_HOST32(*((u_int32_t *) net_buf));
					if (ei != NULL) {
						ei->collection_type = collection_type;
					}
					net_buf += sizeof(u_int32_t);

					net_buf = netbuf_to_collection(net_buf,
								       (ei != NULL) ? &ei->collection : NULL,
								       collection_type,
								       value_type);

					if (ei != NULL) {
						ei->group.group_number = NET_TO_HOST32(*((u_int32_t *) net_buf));
					}
					net_buf += sizeof(u_int32_t);

					if (ei != NULL) {
						ei->group.group_level = NET_TO_HOST32(*((u_int32_t *) net_buf));
					}
					net_buf += sizeof(u_int32_t);


					if (ei != NULL) {
						if (*((char *) net_buf) != '\0') {
							ei->group.group_name = engine_strdup((const char *) net_buf);
						} else {
							ei->group.group_name = NULL;
						}
					}
					net_buf += strlen((char *) net_buf) + sizeof(char);

					if (ei != NULL) {
						ei->flags = NET_TO_HOST16(*((u_int16_t *) net_buf));
					}
					net_buf += sizeof(u_int16_t);
				}

				if (var_ptr != NULL) {
					*((extended_info_array_t * *) var_ptr) = eia;
				}

				pf++;
				break;

			default:
				LOG_SERIOUS("Format character %c slipped past the validater. "
					    "I don't know how to handle it.  I'm skipping it.\n", *pf);
				pf++;
		}
	}

	va_end(args);

	LOG_PROC_EXIT_INT(0);
	return 0;
}


