/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: object_cb.c
 */ 
 
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "revert.h"
#include "resize.h"
#include "destroy.h"
#include "plugin_cb.h"
#include "object_cb.h"
#include "logging.h"

/*
 *
 *   void on_revert_object_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates the revert API call using the handle
 *      of the object associated with the last row selected. It then
 *      creates and displays the results popup which destroys both
 *      windows when dismissed.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      The object is reverted and a results screen is displayed.
 *
 */
void on_revert_object_button_clicked (GtkButton *button, gpointer user_data)
{
    gchar *error_msg   = _("An error was encountered attempting to remove the toplevel feature.");
    gchar *success_msg = _("The toplevel feature was successfully removed from the storage object stack.");

    on_revert_thing_button_clicked (button, error_msg, success_msg);
}

/*
 *
 *   void on_revert_object_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of toplevel objects that can be reverted.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Selection list populated with objects
 *
 */
void on_revert_object_clist_realize (GtkWidget *widget, gpointer user_data)
{
    on_revert_thing_clist_realize (widget, EVMS_OBJECT);
}

/*
 *
 *   void on_destroy_object_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates the destroy API call using the handle
 *      of the object associated with the last row  selected. It then 
 *      creates and displays the results popup which destroys both 
 *      windows when dismissed.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      See description.
 *
 */
void on_destroy_object_button_clicked (GtkButton *button, gpointer user_data)
{
    gchar *error_msg   = _("An error was encountered attempting to destroy the storage object.");
    gchar *success_msg = _("The storage object was successfully destroyed.");

    on_destroy_thing_button_clicked (button, error_msg, success_msg);
}

/*
 *
 *   void on_destroy_object_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of objects that can be destroyed.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Selection list populated with suitable objects
 *
 */
void on_destroy_object_clist_realize (GtkWidget *widget, gpointer user_data)
{
    on_destroy_thing_clist_realize (widget, EVMS_OBJECT);
}

/*
 *
 *   void on_feature_plugin_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of different feature plugins to choose from.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Selection list populated with available feature plugins
 *
 */
void on_feature_plugin_clist_realize (GtkWidget *widget, gpointer user_data)
{
    GtkCList *clist = GTK_CLIST (widget);

    set_selection_window_clist_column_titles (clist, _("EVMS Feature Plugin"), NULL, NULL);
    
    on_plugin_selection_clist_realize (clist, EVMS_FEATURE, 0);
    on_plugin_selection_clist_realize (clist, EVMS_ASSOCIATIVE_FEATURE, 0);

    /*
     * Check to see how many things were placed in the list. If there is
     * only one thing then automatically select it so the "Next" button
     * is immediately made available.
     *
     * If the number of rows is something other than one then make sure
     * that all rows are unselected. This is a hack since the
     * on_plugin_selection_clist_realize() is typically only called once
     * and so it sets the selected row if there is only one item found
     * during the invocation. We just so happen to call it multiple times.
     */

    if (clist->rows == 1)
        gtk_clist_select_row (clist, 0, 0);
    else
        gtk_clist_unselect_all (clist);
}

/*
 *
 *   void on_resize_object_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine simply calls the function that handles
 *      displaying expand/shrink points for the selected
 *      storage object.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - contains id of resize action EVMS_Task_Expand or
 *                  EVMS_Task_Shrink
 *
 *   Exit:
 *      See description.
 *
 */
void on_resize_object_button_clicked (GtkButton *button, gpointer user_data)
{
    on_resize_selection_button_clicked (button, GPOINTER_TO_UINT (user_data));
}

/*
 *
 *   void on_resize_object_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of topmost object that can be expanded or shrunk depending
 *      on the action.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - contains id of resize action EVMS_Task_Expand
 *                  or EVMS_Task_Shrink
 *
 *   Exit:
 *      Selection list populated with suitable expandable or shrinkable
 *      topmost storage objects.
 *
 */
void on_resize_object_clist_realize (GtkWidget *widget, gpointer user_data)
{
    gint            rc;
    task_action_t   action = GPOINTER_TO_UINT (user_data);
    handle_array_t *objects;

    set_selection_window_clist_column_titles (GTK_CLIST (widget), _("Size"), _("Storage Object"), NULL);
    
    rc = evms_get_object_list (0, DATA_TYPE, 0, TOPMOST, &objects);
    
    if (rc != SUCCESS)
    {
        log_error ("%s: evms_get_volume_list() returned error code %d.\n", __FUNCTION__, rc);
    }
    else
    {
        if (action == EVMS_Task_Expand)
            add_expandable_objects_to_selection_clist (GTK_CLIST (widget), objects, FALSE);
        else
            add_shrinkable_objects_to_selection_clist (GTK_CLIST (widget), objects, FALSE);
        
        evms_free (objects);
    }
}
