/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module:  os2regmgr.c
 *
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <linux/evms/evms_os2.h>
#include <plugin.h>
#include "os2regmgr.h"
#include "orm_options.h"
#include "orm_discover.h"
#include "orm_commit.h"
#include "orm_io.h"

plugin_record_t     ORM_PluginRecord;     // Used to identify regions as owned by this region manager
engine_functions_t  * ORM_EngFncs;        // EVMS Engine function table.


/*
 *  Called to test if we own the specified storage object and can
 *  work with it.
 */
BOOLEAN orm_ican_modify( storage_object_t *obj )
{
    if ( obj ) {

        if ( obj->plugin == &ORM_PluginRecord ) {

            if ( obj->private_data ) {

                if ( (( orm_private_data_t * )obj->private_data )->signature == ORM_PDATA_SIGNATURE ) {

                    return TRUE;

                }
            }
        }
    }

    return FALSE;
}

/*
 *  Returns the 1st object found in a DLIST.
 */
storage_object_t *  orm_get_first_object_in_list( dlist_t list )
{
    int rc;
    storage_object_t *obj;
    storage_object_t *object_from_list = NULL;
    uint size;
    TAG  tag;

    LOGENTRY();

    rc = GoToStartOfList( list );
    if ( rc == DLIST_SUCCESS ) {

        rc = BlindGetObject( list, &size, &tag, NULL, FALSE, ( void ** )&obj );
        if ( rc == DLIST_SUCCESS ) {

            object_from_list = obj;

        }
    }

    LOGEXIT();
    return  object_from_list;
}


/*
 * Function:  setup_evms_plugin
 *
 *  This function gets called shortly after the plugin is loaded by the
 *   Engine.  It performs all tasks that are necessary before the initial
 *   discovery pass.
 *
 */
static int setup_evms_plugin( engine_mode_t  mode,
                              engine_functions_t  * function_table )
{
    int rc = EINVAL;

    if ( function_table ) {

        function_table->write_log_entry( ENTRY_EXIT, &ORM_PluginRecord, "%s: entry\n", __FUNCTION__ );

        ORM_EngFncs = function_table;

        rc = 0;

        LOGEXITRC();
    }

    return rc;
}


/*
 * Function:  cleanup_evms_plugin
 *
 *  This function gets when the plugin will be unloaded to give it a chance
 *   to clean up after iself.
 *
 */
static void cleanup_evms_plugin( void )
{
    /* Nothing to cleanup. */
}


/*
 *  I can delete any object I own providing:
 *
 *  - if the region is an LVM volume ... that all the segments can be deleted
 *  - if the region is a compatibility volume ... that the segment can be deleted
 *
 */
static int can_delete( storage_object_t * region )
{
    int rc = EINVAL;
    int rc2 = DLIST_SUCCESS;
    storage_object_t *seg;

    LOGENTRY();

    if ( orm_ican_modify( region ) == TRUE ) {

        //
        // for any kind of region ... if all child objects report that they can be
        // deleted then we can safely delete the region
        //
        rc = GoToStartOfList( region->child_objects );
        if ( rc == DLIST_SUCCESS ) {

            rc = GetObject( region->child_objects, sizeof( storage_object_t ), SEGMENT_TAG, NULL, TRUE, ( void ** )&seg );

            while ((rc == DLIST_SUCCESS)&&(rc2 == DLIST_SUCCESS)) {

                rc2 = (( struct plugin_functions_s * )seg->plugin->functions.plugin )->can_delete( seg );

                rc = GetNextObject( region->child_objects, sizeof( storage_object_t ), SEGMENT_TAG, ( void ** )&seg );
            };

            rc = rc2;
        }
    }

    LOGEXITRC();
    return rc;
}


/*
 *  I can expand any object I own providing:
 *
 *  - if the region is an LVM volume ... that it can add a child drivelink object.
 *  - if the region is a compatibility volume ... that the segment can be expanded.
 *
 */
static int can_expand( storage_object_t * region,
                       sector_count_t * expansion_limit,
                       dlist_t  expansion_points )
{
    orm_private_data_t  *pdata;
    storage_object_t    *seg;
    int rc=EINVAL;

    LOGENTRY();

    if ( orm_ican_modify( region ) == TRUE) {

        pdata = ( orm_private_data_t * )region->private_data;

        if ( pdata->flags & COMPATIBILITY_VOLUME ) {

            seg = orm_get_first_object_in_list( region->child_objects );

            if ( seg ) {

                rc = (( struct plugin_functions_s * )seg->plugin->functions.plugin )->can_expand( seg, expansion_limit, expansion_points );

            }
        }
        else {

            if ( pdata->drive_link_count < MAXIMUM_LINKS ) {
                rc = 0;
            }
        }
    }

    LOGEXITRC();
    return rc;
}


static int can_expand_by( storage_object_t  * region,
                          sector_count_t    *  size )
{
    LOGENTRY();
    LOGEXIT();
    return ENOSYS;
}


/*
 *  I can shrink any object I own providing:
 *
 *  - if the region is an LVM volume ... that it can delete a link and still remain a drivelink
 *  - if the region is a compatibility volume ... that the segment can be shrunk.
 *
 */

static int can_shrink( storage_object_t  * region,
                       sector_count_t * shrink_limit,
                       dlist_t shrink_points )
{
    orm_private_data_t  *pdata;
    storage_object_t    *seg;
    int rc=EINVAL;

    LOGENTRY();

    if ( orm_ican_modify( region ) == TRUE ) {

        pdata = ( orm_private_data_t * )region->private_data;

        if ( pdata->flags & COMPATIBILITY_VOLUME ) {

            seg = orm_get_first_object_in_list( region->child_objects );

            if ( seg ) {

                rc = (( struct plugin_functions_s * )seg->plugin->functions.plugin )->can_shrink( seg, shrink_limit, shrink_points );

            }
        }
        else {

            if (( --pdata->drive_link_count ) > 1 ) {
                rc = 0;
            }
        }
    }

    LOGEXITRC();
    return rc;
}


static int can_shrink_by( storage_object_t  * region,
                          sector_count_t    *  size )
{
    LOGENTRY();
    LOGEXIT();
    return ENOSYS;
}


/*
 *  Not supported ...
 */
static int can_move( storage_object_t * region )
{
    LOGENTRY();
    LOGEXIT();
    return ENOSYS;
}


 /*
  *  OS2 regions can be made into a volume as long as they are not
  *  hidden.  Don't know how they are hidden yet ...
  */
static int can_set_volume( storage_object_t * region, BOOLEAN flag )
{
    int rc = EINVAL;
    orm_private_data_t  *pdata;

    LOGENTRY();

    if ( orm_ican_modify( region ) == TRUE ) {

        pdata = ( orm_private_data_t * )region->private_data;

        if ( pdata->flags & HIDDEN_VOLUME ) {
            rc = EINVAL;
        }
        else {
            rc = 0;
        }
    }

    LOGEXITRC();
    return rc;
}



static int create(  dlist_t         freespace_region_list,
                    option_array_t  * options,
                    dlist_t         new_region_list )
{
    int rc = EINVAL;
    LOGENTRY();


    LOGEXITRC();
    return rc;
}



static int orm_delete( storage_object_t * region, dlist_t  children )
{
    int rc = EINVAL;
    int rc2 = DLIST_SUCCESS;
    storage_object_t *seg;

    LOGENTRY();

    if ( orm_ican_modify( region ) == TRUE ) {

        //
        // for any kind of region ... if all child objects are successfully
        // deleted then we can free up the region storage object.
        //
        rc = GoToStartOfList( region->child_objects );
        if ( rc == DLIST_SUCCESS ) {

            rc = GetObject( region->child_objects, sizeof( storage_object_t ), SEGMENT_TAG, NULL, TRUE, ( void ** )&seg );

            while (( rc == DLIST_SUCCESS ) && ( rc2 == DLIST_SUCCESS )) {

                rc2 = (( struct plugin_functions_s * )seg->plugin->functions.plugin )->delete( seg, children );

                rc = GetNextObject( region->child_objects, sizeof( storage_object_t ), SEGMENT_TAG, ( void ** )&seg );
            };

            rc = rc2;
        }
    }

    if ( rc == 0 ) {
        free_os2_region( region );
    }

    LOGEXITRC();
    return rc;
}


static int expand( storage_object_t  * region,
                   storage_object_t  * expand_region,
                   dlist_t           freespace_region,
                   option_array_t    * options )
{
    int rc = ENOSYS;

    LOGENTRY();
    LOGEXITRC();
    return rc;
}


static int shrink( storage_object_t  * region,
                   storage_object_t  * shrink_region,
                   dlist_t           freespace,
                   option_array_t    * options )
{
    int rc = ENOSYS;

    LOGENTRY();
    LOGEXITRC();
    return rc;
}


/*
 *  Not supported ...
 */
static int move( storage_object_t  * source,
                 storage_object_t  * target,
                 option_array_t    * options )
{
    int rc = ENOSYS;

    LOGENTRY();
    LOGEXITRC();
    return rc;
}


static void set_volume( storage_object_t * region,
                        BOOLEAN  flag )
{
    LOGENTRY();
    LOGEXIT();
}


//  Function table for the OS/2 Region Manager...
static plugin_functions_t orm_functions = {
    setup_evms_plugin           : setup_evms_plugin,
    cleanup_evms_plugin         : cleanup_evms_plugin,
    can_delete                  : can_delete,
    can_expand                  : can_expand,
    can_expand_by               : can_expand_by,
    can_shrink                  : can_shrink,
    can_shrink_by               : can_shrink_by,
    can_move                    : can_move,
    can_set_volume              : can_set_volume,
    create                      : create,
    delete                      : orm_delete,
    expand                      : expand,
    shrink                      : shrink,
    move                        : move,
    set_volume                  : set_volume,

    // found in orm_commit.c
    commit_changes              : orm_commit_changes,

    // found in orm_discover.c
    discover                    : orm_discover,

    // found in orm_io.c
    add_sectors_to_kill_list    : orm_add_sectors_to_kill_list,
    read                        : orm_read,
    write                       : orm_write,

    // found in orm_options.c
    get_option_count            : ORM_GetOptionCount,
    init_task                   : ORM_InitTask,
    set_option                  : ORM_SetOption,
    set_objects                 : ORM_SetObjects,
    get_info                    : ORM_GetInfo,
    get_plugin_info             : ORM_GetPluginInfo

};


plugin_record_t ORM_PluginRecord = {
    id:                     SetPluginID( EVMS_OEM_IBM, EVMS_REGION_MANAGER, 0x02 ),

    version:              { major:       MAJOR_VERSION,
                            minor:       MINOR_VERSION,
                            patchlevel:  PATCH_LEVEL },

    required_api_version: { major:       3,
                            minor:       0,
                            patchlevel:  0 },

    short_name:             "OS2RegMgr",
    long_name:              "OS/2 Region Manager",
    oem_name:               "IBM",

    functions:            { plugin:    ( ADDRESS )&orm_functions },

    container_functions:   NULL
};


plugin_record_t * evms_plugin_records[] = { &ORM_PluginRecord, NULL };

