/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: segs.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>

#include "ptables.h"
#include "defsegmgr.h"
#include "checks.h"
#include "segs.h"
#include "display.h"
#include "os2dlat.h"
#include "sn.h"


// dlist for keeping disk private data areas
dlist_t  Disk_PrivateData_List=NULL;
#define DISK_PDATA_TAG  0x08080808




/*
 *  A freespace has a name that has 3 parts to it.
 *  (1) a parent object name like hda
 *  (2) the word freespace
 *  (3) a number used to keep freespace seg names different
 *
 *  Examples:  hda_freespace1
 *             hda_freespace2
 *             dasdh3_freespace4
 *
 *  This routine returns the number found at the end
 *  of the freespace segment name, ignoring any differences
 *  that may be found in other parts of the name.
 *
 *  Returns: success: integer > 0
 *           failure: -1
 *
 */
static int get_freespace_number(DISKSEG *freespace)
{
    int number=-1;
    int i;

    if (freespace) {

        if (freespace->name) {

            // pretty liberal here...
            // minimum string has length of 2 ... freespac[eN]
            if (strlen(freespace->name)>=2) {

                // walk backwards
                for (i=strlen(freespace->name)-1; i>0; i--) {

                    // first e we find ends the search
                    if ( freespace->name[i-1] == 'e' ) {

                        // convert ascii number to integer
                        return (  atoi( &freespace->name[i] ) );

                    }

                }

            }

        }

    }

    return number;
}


/*
 *  Called by the get_name_for_disk_segment() routine to come up
 *  with the next available number to use when naming freespace
 *  segments on this drive.
 */
static int  get_next_avail_freespace_number(LOGICALDISK *ld)
{
    int freespace_number=0;
    int rc;
    DISKSEG *seg;

    LOGENTRY();

    rc = GoToStartOfList( ld->parent_objects );
    if ( rc == DLIST_SUCCESS) {

        rc = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE, (void **) &seg );
        while ( rc == DLIST_SUCCESS ) {

            if (seg->data_type == FREE_SPACE_TYPE) {

                if ( get_freespace_number(seg) > freespace_number ) {
                    freespace_number = get_freespace_number(seg);
                }

            }

            rc = GetNextObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,(void **)&seg );
        }

    }

    ++freespace_number;

    LOGEXIT();
    return freespace_number;
}


/*
 *  Called by insert_DiskSeg_Into_List() to get a name
 *  for the segment storage object being created.
 */
int  get_name_for_disk_segment( DISKSEG *seg )
{
    int                 rc = EINVAL;
    LOGICALDISK        *ld  = get_logical_disk(seg);
    DISK_PRIVATE_DATA  *disk_pdata;
    SEG_PRIVATE_DATA   *pdata;


    if (ld) {

        disk_pdata = get_disk_private_data(ld);
        pdata      = (SEG_PRIVATE_DATA *)seg->private_data;

        if (seg->data_type == DATA_TYPE) {

            if ( pdata->flags & SEG_IS_PRIMARY_PARTITION ) {

                sprintf(seg->name, "%s%d", ld->name, pdata->part_number );

            }
            else if ( ( pdata->flags & SEG_IS_BSD_PARTITION ) ||
                      ( pdata->flags & SEG_IS_UNIXWARE_PARTITION ) ||
                      ( pdata->flags & SEG_IS_SOLARIS_X86_PARTITION )){

                sprintf(seg->name, "%s%d", ld->name, pdata->part_number );

            }
            else {

                DISKSEG *ebr;
                SEG_PRIVATE_DATA *ebr_pdata;

                ebr = pdata->ebr;
                if (ebr) {
                    ebr_pdata = (SEG_PRIVATE_DATA *)ebr->private_data;
                    if (ebr_pdata) {
                        sprintf(seg->name, "%s%d", ld->name, ebr_pdata->ebr_number+5   );
                    }
                }

            }

            rc = 0;

        }
        else if (seg->data_type == META_DATA_TYPE) {

            if ( pdata->flags & SEG_IS_MBR) {
                sprintf(seg->name, "%s_mbr", ld->name );
                rc = 0;
            }
            else if ( pdata->flags & SEG_IS_EBR) {
                sprintf(seg->name, "%s_ebr%d", ld->name, pdata->ebr_number );
                rc = 0;
            }
            else {
                rc = 0; // must be embedded metadata segment
            }

        }
        else if (seg->data_type == FREE_SPACE_TYPE) {

            sprintf(seg->name, "%s_freespace%d", ld->name, get_next_avail_freespace_number(ld) );
            rc = 0;

        }
        else {
            LOG_ERROR("segment has unknown data type (type=%d)\n", seg->data_type );
        }

    }


    return rc;
}


/*
 *  Called to allocate a segment storage object
 */
DISKSEG *  allocate_disk_segment( LOGICALDISK *ld )
{
    int   rc;
    void *handle;
    DISKSEG            *seg=NULL;


    LOGENTRY();


    rc = SegEngFncs->allocate_segment( NULL, &seg );
    if (rc==0) {

        // insert logical disk into new segment's child dlist
        rc = InsertObject ( (dlist_t)          seg->child_objects,
                            (int)              sizeof(storage_object_t),
                            (void *)           ld,
                            (TAG)              DISK_TAG,
                            (void *)           NULL,
                            (Insertion_Modes)  AppendToList,
                            (BOOLEAN)          TRUE,
                            (void **)         &handle );

        if (rc == 0) {

            seg->plugin      = Seg_My_PluginRecord_Ptr;
            seg->object_type = SEGMENT;

            seg->private_data = calloc(1, sizeof(SEG_PRIVATE_DATA));
            if (seg->private_data) {

                ((SEG_PRIVATE_DATA *)seg->private_data)->signature = DEFAULT_SEG_MGR_PDATA_SIGNATURE;
                ((SEG_PRIVATE_DATA *)seg->private_data)->logical_disk = ld;

            }
            else {
                LOG_ERROR("call to malloc segment private storage area failed\n");
                SegEngFncs->free_segment( seg );
                seg = NULL;
            }
        }
        else {
            LOG_ERROR("call to insert DISK storage object in segment child_objects dlist failed, RC= %d\n", rc );
        }

    }
    else {
        LOG_ERROR("call to engine_allocate_segment failed, RC= %d\n", rc);
        seg = NULL;
    }

    LOGEXIT();

    return seg;
}


/*
 * Called to free a segment storage object
 */
void free_disk_segment( DISKSEG *seg )
{
    LOGICALDISK       *ld         = NULL;
    DISK_PRIVATE_DATA *disk_pdata = NULL;
    SEG_PRIVATE_DATA  *seg_pdata  = NULL;

    LOGENTRY();
    LOG_DEBUG("segment name= %s\n", seg->name);

    ld         = get_logical_disk(seg);
    disk_pdata = get_disk_private_data(ld);
    seg_pdata  = (SEG_PRIVATE_DATA *) seg->private_data;

    SegEngFncs->free_segment( seg );

    LOGEXIT();

}


/*
 *  Called to allocate a DISK_PRIVATE_DATA struct and link it into the
 *  list we maintain for disks we manage disk segments on.
 */
int create_disk_private_data( LOGICALDISK *ld )
{
    int rc=ENOMEM;
    DISK_PRIVATE_DATA *disk_pdata;
    void *handle;


    LOGENTRY();

    // make sure dlist is created
    if (Disk_PrivateData_List == NULL) {

        Disk_PrivateData_List = CreateList();

        if (Disk_PrivateData_List == NULL) {
            LOGEXITRC();
            return rc;
        }

    }

    if ( get_disk_private_data(ld ) == NULL ) {

        disk_pdata = (DISK_PRIVATE_DATA *) calloc(1, sizeof(DISK_PRIVATE_DATA) );

        if ( disk_pdata ) {

            disk_pdata->signature      = DEFAULT_SEG_MGR_PDATA_SIGNATURE;
            disk_pdata->key            = (void *)ld;
            disk_pdata->container_segs = CreateList();
            memcpy(&disk_pdata->geometry, &ld->geometry, sizeof(geometry_t));

            if (disk_pdata->container_segs) {

                rc = InsertObject ( (dlist_t)          Disk_PrivateData_List,
                                    (int)              sizeof(DISK_PRIVATE_DATA),
                                    (void *)           disk_pdata,
                                    (TAG)              DISK_PDATA_TAG,
                                    (void *)           NULL,
                                    (Insertion_Modes)  AppendToList,
                                    (BOOLEAN)          TRUE,
                                    (void **)         &handle );
            }
            else {
                rc = ENOMEM;
            }

            if ( rc ) {
                free(disk_pdata);
            }

        }
        else {
            rc = ENOMEM;
        }
    }
    else {
        rc = 0;
    }

    LOGEXITRC();
    return rc;
}

/*
 *  Called to delete the specified disk private data by pruning it from the double
 *  linked list.
 */
int delete_disk_private_data( LOGICALDISK *ld )
{
    int                rc = EINVAL;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld );


    LOGENTRY();

    // make sure dlist exists
    if ( Disk_PrivateData_List == NULL ) {
        LOGEXITRC();
        return rc;
    }

    // delete object from list and free its memory
    if (disk_pdata) {

        DeleteObject( Disk_PrivateData_List, disk_pdata );

        if ( disk_pdata->container_segs ) DestroyList(&disk_pdata->container_segs,FALSE);

        free(disk_pdata);

        rc = 0;
    }
    else {
        rc = EINVAL;
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Returns the private data area for the specified disk.
 */
DISK_PRIVATE_DATA * get_disk_private_data( LOGICALDISK *ld )
{
    int                 rc;
    DISK_PRIVATE_DATA  *disk_pdata = NULL;



    // make sure dlist is created
    if ( Disk_PrivateData_List ) {

        rc = GoToStartOfList( Disk_PrivateData_List );
        if ( rc == DLIST_SUCCESS ) {

            rc = GetObject( Disk_PrivateData_List,sizeof(DISK_PRIVATE_DATA),DISK_PDATA_TAG,NULL,TRUE,(void **)&disk_pdata );

            while (rc == DLIST_SUCCESS) {

                // compare storage object ptr to key field in private data area
                if ( disk_pdata->key == ld ) {

                    return disk_pdata;

                }

                rc = GetNextObject( Disk_PrivateData_List,sizeof(DISK_PRIVATE_DATA), DISK_PDATA_TAG,(void **) &disk_pdata );

            };

        }

    }

    LOG_ERROR("error: get_disk_private_data(%s) returning NULL\n", ld->name);
    return NULL;
}




/*
 *  Tests if the stated LBA is below the 1024 cylinder limit on
 *  the drive.
 */
BOOLEAN below_1024_cylinder_limit(LOGICALDISK *ld, lba_t  lba )
{
    chs_t   chs;

    if ( LBAtoCHS( ld, lba, &chs ) ) {
       return TRUE;
    }
    else {

        if (chs.cylinder < MAX_CYLINDERS ) {
            return TRUE;
        }
        else {
            return FALSE;
        }

    }

}



/*
 *  Returns TRUE if the LBA starts on a cylinder boundary
 */
BOOLEAN  starts_on_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
    chs_t  chs;

    if ( LBAtoCHS( ld, lba, &chs ) ) {
        return TRUE;
    }
    else {

        if ( ( chs.sector == 1 ) && ( chs.head == 0 ) ) {
            return TRUE;
        }
        else {
            return FALSE;
        }
    }

}



/*
 *  Returns TRUE if the LBA ends on a cylinder boundary
 */
BOOLEAN  ends_on_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
    chs_t  chs;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);


    if ( LBAtoCHS( ld, lba, &chs ) ) {
        return TRUE;
    }
    else {

        if ( ( chs.sector == disk_pdata->geometry.sectors_per_track ) &&
             ( chs.head   == disk_pdata->geometry.heads-1 ) ) {
            return TRUE;
        }
        else {
            return FALSE;
        }

    }

}


/*
 *  Called to calculate the cylinder size for the specified
 *  storage object. Storage object might be either DISK or
 *  SEGMENT type of object.
 */
sector_count_t get_cylinder_size( storage_object_t *obj )
{
    sector_count_t     sectors=0;
    DISK_PRIVATE_DATA *disk_pdata;
    LOGICALDISK       *ld;


    if (obj) {

        ld = get_logical_disk(obj);

        if (ld) {

            disk_pdata = get_disk_private_data(ld);

            if (disk_pdata) {
                sectors = (sector_count_t) (disk_pdata->geometry.heads * disk_pdata->geometry.sectors_per_track);
            }

        }

    }


    return sectors;
}


/*
 *  Returns the specified LBA rounded up to a track boundary.
 */
static lba_t  roundup_to_track_boundary( LOGICALDISK *ld, lba_t lba )
{
    lba_t            new_lba = lba;
    sector_count_t   extra_sectors;
    sector_count_t   SectorsPerTrack;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);


    if ( disk_pdata ) {

        SectorsPerTrack = disk_pdata->geometry.sectors_per_track;

        if (SectorsPerTrack) {

            extra_sectors = lba % SectorsPerTrack;

            if ( extra_sectors != 0) {
                new_lba = lba + ( SectorsPerTrack - extra_sectors ) - 1;
            }

        }

    }

    return new_lba;
}



/*
 *  Returns the specified LBA rounded up to a cylinder boundary.
 */
lba_t  roundup_to_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
    lba_t            new_lba = lba;
    sector_count_t   extra_sectors;
    sector_count_t   SectorsPerCylinder;


    SectorsPerCylinder = get_cylinder_size(ld);

    if ( SectorsPerCylinder ) {

        extra_sectors = lba % SectorsPerCylinder;

        if ( extra_sectors != 0) {
            new_lba = lba + ( SectorsPerCylinder - extra_sectors) - 1;
        }

    }

    return new_lba;
}


/*
 *  Returns the specified LBA rounded down to a cylinder boundary.
 */
lba_t  rounddown_to_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
    lba_t            new_lba=lba;
    sector_count_t   extra_sectors;
    sector_count_t   SectorsPerCylinder;



    SectorsPerCylinder = get_cylinder_size(ld);

    if (SectorsPerCylinder) {

        extra_sectors = lba % SectorsPerCylinder;

        if ( extra_sectors != 0) {
            new_lba = lba - extra_sectors;
        }

    }

    return new_lba;
}



/*
 *  Returns TRUE if the disk has an extended partition
 */
BOOLEAN   disk_has_extended_partition(LOGICALDISK *ld)
{
 SEG_PRIVATE_DATA   *pdata;
 BOOLEAN   rc=FALSE;
 DISKSEG   *mbr;


    mbr = get_mbr_from_seglist( ld->parent_objects );
    if (mbr) {

        pdata = (SEG_PRIVATE_DATA *)mbr->private_data;
        if (pdata) {

            // if the mbr chains to an ebr then YES ...
            if (pdata->next_ebr) {
                rc=TRUE;
            }

        }

    }

    return rc;
}


/*
 *  Returns TRUE if the disk segment list has a META_DATA segment
 *  that is an MBR track.
 */
BOOLEAN   seglist_has_mbr_segment( dlist_t seglist )
{
    if ( get_mbr_from_seglist(seglist)==NULL ) {
        return FALSE;
    }
    else {
        return TRUE;
    }
}


/*
 *  Returns an MBR DISKSEG if one is found in the dlist.
 */
DISKSEG * get_mbr_from_seglist( dlist_t seglist )
{
    DISKSEG          *seg;
    int               rc;
    SEG_PRIVATE_DATA *pdata;

    if (seglist==NULL) return NULL;

    rc = GoToStartOfList( seglist );
    if (rc) return NULL;

    rc = GetObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **)&seg );
    if (rc) return NULL;

    do {

        if (seg!=NULL) {

            pdata = (SEG_PRIVATE_DATA *) seg->private_data;
            if (pdata==NULL) return NULL;

            if ( pdata->flags & SEG_IS_MBR ) return seg;

            rc = GetNextObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,(void **)&seg );
            if (rc) return NULL;
        }

    } while (seg != NULL);

    return NULL;
}


/*
 *  Called to find the 1st unused entry in a partition table.  This is normally called
 *  when we are createing new partitions and need to add a partition record to a
 *  table.
 *
 *  If successful:  RC = 0...3  a valid partition table index
 *
 *  If errors:      RC = -1  either there are no free partition table entries or else
 *                  we had difficulties operating on the segment DLIST.
 *
 */
int  get_first_unused_ptable_entry( dlist_t seglist, DISKSEG *ebr )
{
    DISKSEG            *seg;
    int                 rc = -1;
    SEG_PRIVATE_DATA   *pdata;
    BOOLEAN             ptable_entry_in_use[4] = {FALSE,FALSE,FALSE,FALSE};
    int                 i;
    LOGICALDISK        *ld = get_logical_disk(ebr);
    DISK_PRIVATE_DATA  *disk_pdata;

    if (seglist) {

        rc = GoToStartOfList( seglist );
        if (rc==DLIST_SUCCESS) {

            rc = GetObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &seg );
            if (rc == DLIST_SUCCESS) {

                do {

                    if (seg!=NULL) {

                        pdata = (SEG_PRIVATE_DATA *) seg->private_data;
                        if (pdata) {

                            if ( pdata->ebr == ebr ) {
                                if ( pdata->ptable_index >=0  && pdata->ptable_index < 4) {
                                    ptable_entry_in_use[pdata->ptable_index] = TRUE;
                                }
                                else {
                                    rc = -1;
                                }
                            }
                            if (rc==DLIST_SUCCESS) {
                                rc = GetNextObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,(void **) &seg );
                            }
                        }
                        else {
                            rc = -1;
                        }

                    }

                } while ( rc == DLIST_SUCCESS  &&  seg != NULL);
            }
        }
    }


    /*
     * also walk the container seg list, looking for primary partitions
     */
    if (ld  &&  rc != -1) {

        disk_pdata = get_disk_private_data(ld);

        if (disk_pdata) {

            rc = GoToStartOfList( disk_pdata->container_segs );
            if (rc==DLIST_SUCCESS) {

                rc = GetObject( disk_pdata->container_segs,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &seg );
                if (rc == DLIST_SUCCESS) {

                    do {

                        if (seg!=NULL) {

                            pdata = (SEG_PRIVATE_DATA *) seg->private_data;
                            if (pdata) {

                                if ( pdata->ebr == ebr ) {
                                    if ( pdata->ptable_index >=0  && pdata->ptable_index < 4) {
                                        ptable_entry_in_use[pdata->ptable_index] = TRUE;
                                    }
                                    else {
                                        rc = -1;
                                    }
                                }
                                if (rc==DLIST_SUCCESS) {
                                    rc = GetNextObject( disk_pdata->container_segs,sizeof(DISKSEG),SEGMENT_TAG,(void **) &seg );
                                }

                           }
                           else {
                               rc = -1;
                           }

                        }

                    } while ( rc == DLIST_SUCCESS  &&  seg != NULL);
                }
            }

        }

    }


    /*
     * if successful in walking the seglist then look to see which
     * is the first unused slot in the partition table.
     */
    if (rc != -1) {

        rc = -1;

        for(i=0; i < 4; i++) {
            if (ptable_entry_in_use[i] == FALSE) {
                rc = i;
                break;
            }
        }
    }

    /*
     * we are going to return either -1 (error) or else the first
     * free partition table entry.
     */
    return rc;
}







/*
 *  Called to find an entry in the mbr partition table for the extended
 *  partition record.
 *
 *  If successful:  RC = 0...3  a valid partition table index
 *
 *  If errors:      RC = -1  either there are no free partition table entries or else
 *                  we had difficulties operating on the segment DLIST.
 *
 */
static int  get_extd_partition_ptable_entry( dlist_t seglist, DISKSEG *mbr )
{
    DISKSEG            *seg;
    int                 rc = -1;
    SEG_PRIVATE_DATA   *pdata;
    BOOLEAN             ptable_entry_in_use[4] = {FALSE,FALSE,FALSE,FALSE};
    int                 i;
    LOGICALDISK        *ld = get_logical_disk(mbr);
    DISK_PRIVATE_DATA  *disk_pdata;


    if (seglist) {

        rc = GoToStartOfList( seglist );
        if (rc==DLIST_SUCCESS) {

            rc = GetObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &seg );
            while (rc == DLIST_SUCCESS) {

                pdata = (SEG_PRIVATE_DATA *) seg->private_data;

                if ( (pdata) && (pdata->ebr == mbr) ) {

                    if ( pdata->ptable_index >=0  && pdata->ptable_index < 4) {

                        if ( (pdata->flags & SEG_IS_EBR)==FALSE) {
                            ptable_entry_in_use[pdata->ptable_index] = TRUE;
                        }

                    }

                }

                rc = GetNextObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,(void **) &seg );

            }

        }

    }

    /*
     * also walk the container seg list, looking for primary partitions
     */
    if (ld) {

        disk_pdata = get_disk_private_data(ld);

        if (disk_pdata) {

            rc = GoToStartOfList( disk_pdata->container_segs );
            if (rc==DLIST_SUCCESS) {

                rc = GetObject( disk_pdata->container_segs,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &seg );
                while (rc == DLIST_SUCCESS) {

                    pdata = (SEG_PRIVATE_DATA *) seg->private_data;

                    if ( (pdata) && (pdata->ebr == mbr) ) {

                        if ( pdata->ptable_index >=0  && pdata->ptable_index < 4) {

                            ptable_entry_in_use[pdata->ptable_index] = TRUE;

                        }

                    }

                    rc = GetNextObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,(void **) &seg );

                }

            }

        }

    }


    /*
     * if successful in walking the seglist then look to see which
     * of the ptable entries we can use.
     */
    rc = -1;

    for(i=3; i >= 0; i--) {

        if (ptable_entry_in_use[i] == FALSE) {
            rc = i;
            break;
        }

    }

    /*
     * we are going to return either -1 (error) or else the index
     * of the mbr partition table entry to be used for the extd
     * partition record.
     */

    return rc;
}



/*
 *  Called to get a count of used entries in either the mbr
 *  or ebr partition tables. These tables only hold 4 entries
 *  apiece.
 */
int  get_ptable_entry_count( dlist_t seglist, DISKSEG *ebr )
{
    DISKSEG            *seg;
    int                 count = 0;
    int                 rc;
    SEG_PRIVATE_DATA   *pdata;


    if (seglist==NULL) {
        return 0;
    }

    rc = GoToStartOfList( seglist );
    if (rc==DLIST_SUCCESS) {

        rc = GetObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE, (void **) &seg );
        if (rc == DLIST_SUCCESS) {

            do {

                if (seg!=NULL) {

                    pdata = (SEG_PRIVATE_DATA *) seg->private_data;
                    if (pdata==NULL) return FALSE;

                    if (pdata->ebr == ebr ) {
                        ++count;
                    }

                    rc = GetNextObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,(void **) &seg );
                }

            } while ( rc==DLIST_SUCCESS && seg != NULL);
        }
    }

    return count;
}


/*
 *  Called by Assign_SegmentManager_ToDisk() to get a freespace
 *  segment from the segment dlist.
 */
DISKSEG * get_first_freespace_seg_in_list( dlist_t seglist )
{
    int rc;
    DISKSEG *seg=NULL;

    LOGENTRY();

    rc = GoToStartOfList( seglist );
    if ( rc == DLIST_SUCCESS ) {

        rc = GetObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **)&seg );
        if ( rc == DLIST_SUCCESS ) {

            do {
                if (seg->data_type == FREE_SPACE_TYPE) {
                    LOGEXIT();
                    return seg;
                }

                rc = GetNextObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,(void **) &seg );

            } while ( rc == DLIST_SUCCESS );

        }
    }

    LOGEXIT();
    return seg;
}


/*
 *   Returns the DISKSEG struct ptr to the freespace seg following the
 *   specified DISKSEG.  Returns NULL if freespace is not found.
 *
 *  Consider the following seglist ...
 *
 *    Seg_1     Seg2       Seg3       Seg4      Seg5
 *    MBR_SEG   DATA_SEG   Free_Seg   EBR_SEG   DATA_SEG
 *
 *   If we were asked for the freespace seg following Seg2 we would
 *   respond by returning Seg3.  The freespace seg must immediately
 *   follow the specified DISKSEG.
 *
 */
DISKSEG * get_freespace_following_seg(DISKSEG *seg)
{
    dlist_t      seglist=NULL;
    DISKSEG     *prev = NULL;
    DISKSEG     *this = NULL;
    int          rc;
    LOGICALDISK  *ld=NULL;


    // get logical disk and the entire segment list for the disk
    ld = get_logical_disk( seg );
    if ( ld ) {
       seglist = ld->parent_objects;
    }

    if ( (ld) && (seglist) ) {

     rc = GoToStartOfList( seglist );
     if (rc) return NULL;

     rc = GetObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &prev );
     if (rc) return NULL;

     do {

      rc = GetNextObject( seglist,sizeof(DISKSEG),SEGMENT_TAG, (void **) &this );
      if (rc) return NULL;

      if (this != NULL) {

          if (prev == seg) {

              if ( this->data_type==FREE_SPACE_TYPE ) {
                  return this;
              }
              else {
                  return NULL;
              }

          }

          prev=this;
      }

     } while (this != NULL);

     return NULL;
 }
 else {
     return NULL;
 }

}



/*
 *   Called to try and find free space between disk data segments, returning a
 *   new free space DISKSEG struct if a gap between existing segments is found.
 */

DISKSEG * find_freespace_in_seglist( dlist_t seglist )
{
    DISKSEG            *prev = NULL;
    DISKSEG            *this = NULL;
    DISKSEG            *freeseg  = NULL;
    LOGICALDISK        *ld;
    DISK_PRIVATE_DATA  *disk_pdata;
    int                 rc;
    int64_t             gap;


    LOGENTRY();

    if (seglist) {

        rc = GoToStartOfList( seglist );
        if (rc == DLIST_SUCCESS) {

            rc = GetObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &prev );
            if (rc == DLIST_SUCCESS) {

                // need disk private data area
                ld = get_logical_disk( prev );
                disk_pdata = get_disk_private_data( ld );
                if (disk_pdata==NULL) return NULL;

                do {

                    rc = GetNextObject( seglist,sizeof(DISKSEG),SEGMENT_TAG, (void **) &this );
                    if ( rc == DLIST_SUCCESS) {

                        gap = this->start - ( prev->start + prev->size );

                        if ( gap > 0 ) {

                            // allocate a segment storage object from the engine
                            freeseg = allocate_disk_segment(ld);

                            if (freeseg ) {

                                freeseg->data_type    = FREE_SPACE_TYPE;
                                freeseg->size         = gap;
                                freeseg->start        = prev->start + prev->size;

                                // dont expose freespace in bsd,unixware and solaris containers
                                if ( seg_is_within_container_segment( freeseg ) == TRUE ) {
                                    free_disk_segment( freeseg );
                                    prev = this;
                                    continue;
                                }

                                // and dont expose any part of the container seg that wasnt
                                // allocated to embedded partitions. you can easilly show
                                // freespace that overlaps a container seg ... and a container
                                // seg is a primary partition ... yikes!  dont do it.
                                else if ( seg_overlaps_container_segment( freeseg ) == TRUE ) {

                                    // call routine to try and adjust the size of the freespace
                                    if ( remove_container_seg_overlap(freeseg) ) {
                                        free_disk_segment( freeseg );
                                        prev = this;
                                        continue;
                                    }

                                }



                                LOGEXIT();
                                return freeseg;

                            }
                            else {
                                LOGEXIT();
                                return NULL;
                            }

                        }
                        else {
                            prev = this;
                        }
                    }

                } while ( rc==DLIST_SUCCESS );
            }
        }
    }


    LOGEXIT();
    return NULL;
}


/*
 *  Called to look for free space on the logical disk and to create
 *  segments which will expose the free space on the disk.
 */
int  find_freespace_on_disk( LOGICALDISK *ld )
{
    DISKSEG            *freeseg              = NULL;
    DISKSEG            *seg                  = NULL;
    DISKSEG            *segaddr              = NULL;
    uint                segcount;
    sector_count_t      sectors_left_on_disk = 0;
    int                 rc;
    lba_t               freespace_start_lba;
    DISK_PRIVATE_DATA  *disk_pdata;
    dlist_t             seglist = ld->parent_objects;



    LOGENTRY();

    // need disk private data area
    disk_pdata = get_disk_private_data( ld );
    if (disk_pdata==NULL) return EINVAL;

    rc = GetListSize(seglist, &segcount);
    if (rc) segcount = 0;

    if ( segcount > 0 ) {                                      /* IF ... we have disk segments */

        do {                                                   /* THEN ... look for gaps between them */
            seg = find_freespace_in_seglist( seglist );

            if (seg != NULL) {

                segaddr = insert_diskseg_into_list( seglist, seg);

                if (segaddr==NULL) {

                    int i;

                    for (i=0; i<10 && segaddr==NULL; i++) {
                        segaddr = insert_diskseg_into_list( seglist, seg);
                    }

                    if (segaddr==NULL) {

                        free_disk_segment( seg );
                        rc = ENOMEM;
                        LOGEXITRC();
                        return rc;
                    }

                }

            }

        } while (seg != NULL);


        rc = GoToEndOfList( seglist );       /* AND ... calc number of free sectors at the */
        if (rc) {
            LOGEXIT();
            return 0;                        /*         end of disk after the last segment */
        }

        rc = GetObject(seglist,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(ADDRESS *) &seg );
        if (rc) {
            LOGEXIT();
            return 0;
        }

        sectors_left_on_disk = ld->size - ( seg->start + seg->size );

        if ( sectors_left_on_disk > 0 ) {
            freespace_start_lba  =  seg->start + seg->size;
        } else {
            sectors_left_on_disk = 0;
            freespace_start_lba  = 0;
        }
    }
    else {  // there are no segments on the disk at all so just create a
            // single free space segment for the disk!
        freespace_start_lba  = 0;
        sectors_left_on_disk = ld->size;
    }

    if ( sectors_left_on_disk > 0 ) {

        freeseg = allocate_disk_segment(ld);
        if (freeseg) {

            freeseg->data_type          = FREE_SPACE_TYPE;
            freeseg->size               = sectors_left_on_disk;
            freeseg->start              = freespace_start_lba;

            ((SEG_PRIVATE_DATA *)freeseg->private_data)->flags |= SEG_IS_FREE_SPACE_PARTITION;

            // and dont expose any part of the container seg that wasnt
            // allocated to embedded partitions. you can easilly show
            // freespace that overlaps a container seg ... and a container
            // seg is a primary partition ... yikes!  dont do it.
            remove_container_seg_overlap(freeseg);

            segaddr = insert_diskseg_into_list( seglist, freeseg);
            if (segaddr==NULL) {

                int i;

                for (i=1; i<10; i++) {

                    segaddr = insert_diskseg_into_list( seglist, freeseg);
                    if (segaddr==NULL) {
                        LOG_DEBUG("error, insert_DiskSeg_Into_List returned an error\n");
                        free_disk_segment(freeseg);
                        LOGEXIT();
                        return EIO;
                    }
                    else {
                        break;
                    }
                }

            }

        }
        else {
            LOGEXIT();
            return EIO;
        }
    }

    // lastly, merge adjacent free areas
    merge_adjacent_freedisksegs_in_list( ld->parent_objects );

    LOGEXIT();
    return 0;
}





/*
 *   Called to try and join neighboring free segments together.
 *
 *   NOTE:  Assumes that the DISKSEG structs are kept in an ordered list
 *
 */
static int  merge_freespace_segments( dlist_t seglist )
{
    DISKSEG            *prev;
    DISKSEG            *this;
    int                 rc;
    DISK_PRIVATE_DATA  *disk_pdata;
    LOGICALDISK        *ld;


    LOGENTRY();

    rc = GoToStartOfList( seglist );
    if ( rc == DLIST_SUCCESS) {

        rc = GetObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE, (void **) &prev );
        if ( rc == DLIST_SUCCESS ) {

            // need disk private data area
            ld = ((SEG_PRIVATE_DATA *)prev->private_data)->logical_disk;
            disk_pdata = get_disk_private_data( ld );
            if (disk_pdata==NULL) return EINVAL;

            do {

                rc = GetNextObject( seglist,sizeof(DISKSEG),SEGMENT_TAG, (void **) &this );

                if (rc==DLIST_SUCCESS) {

                    if ( ( prev==NULL ) ||
                         ( this->data_type != FREE_SPACE_TYPE )||
                         ( prev->data_type != FREE_SPACE_TYPE )) {
                        prev = this;
                        continue;
                    }

                    if (get_freespace_number(prev) > get_freespace_number(this) ) {
                        rc = DeleteObject( seglist, prev );
                        if (rc == DLIST_SUCCESS) {
                            this->start -= prev->size;
                            this->size  += prev->size;
                            free_disk_segment( prev );
                            LOG_DEBUG("        kept seg: %s  start: %lld  size: %lld\n", this->name, this->start, this->size );
                            return DLIST_SUCCESS;
                        }
                        else {
                            return rc;
                        }
                    }
                    else {
                        rc = DeleteObject( seglist, this );
                        if (rc == DLIST_SUCCESS) {
                            prev->size  += this->size;
                            free_disk_segment( this );
                            LOG_DEBUG("        kept seg: %s  start: %lld  size: %lld\n", prev->name, prev->start, prev->size );
                            return DLIST_SUCCESS;
                        }
                        else {
                            return rc;
                        }
                    }

                }

            } while (rc==DLIST_SUCCESS);  /* continue till we hit the end of the list */
        }
    }

    rc = DLIST_END_OF_LIST;

    LOGEXITRC();
    return rc;
}





/*
 *   Called to try and join neighboring free segments together.
 *
 *   NOTE:  Assumes that the DISKSEG structs are kept in an ordered list
 *
 */
int  merge_adjacent_freedisksegs_in_list( dlist_t seglist )
{
    int         rc;

    LOGENTRY();

    do {

        rc = merge_freespace_segments(seglist);

    } while (rc==DLIST_SUCCESS);  /* continue while there are segs to merge */


    rc = 0;

    LOGEXITRC();
    return rc;
}



/*
 *  Called to add a segment to the logical disk's segment DLIST. First, convert the
 *  partition record to a DISKSEG struct.  Then, insert the DISKSEG struct into
 *  the DLIST hanging off the logical disk.
 */
DISKSEG * build_mbr_disk_segment( LOGICALDISK *ld )
{
    Partition_Record   pr;
    DISKSEG            *mbr=NULL;
    DLA_Table_Sector   *dlat_buffer = NULL;
    DISK_PRIVATE_DATA  *disk_pdata = get_disk_private_data(ld);

    if (disk_pdata) {

        // for OS2 disks we need to get the Drive Link Address Table
        if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

            dlat_buffer = Read_Dlat_Sector( ld, 0 );

            if (dlat_buffer) {

                if ( seg_register_serial_number( dlat_buffer->Disk_Serial_Number ) == 0) {

                    if ( SegEngFncs->register_name( (char *) dlat_buffer->Disk_Name ) ) {

                        seg_unregister_serial_number( dlat_buffer->Disk_Serial_Number);
                        return NULL;

                    }

                }
                else {
                    return NULL;
                }

                // save disk name in convenient place
                strncpy( disk_pdata->disk_name, &dlat_buffer->Disk_Name[0], DISK_NAME_SIZE );
            }
        }



        // now hang the MBR meta data segment off the evms logical drive
        memset( &pr, 0, sizeof(Partition_Record));
        pr.start_sect = 0;
        pr.nr_sects   = CPU_TO_DISK32( disk_pdata->geometry.sectors_per_track );
        pr.sys_ind    = MBR_PARTITION;

        mbr = build_diskseg_from_partition_record(ld, &pr, NULL, 0, FALSE );

        if (mbr==NULL) {
            if (dlat_buffer != NULL) free(dlat_buffer);
        }
        else {
            ((SEG_PRIVATE_DATA *)mbr->private_data)->dlat = dlat_buffer;
        }

    }

    return mbr;
}


/*
 *  Called to add a segment to the logical disk's segment DLIST. First, convert the
 *  partition record to a DISKSEG struct.  Then, insert the DISKSEG struct into
 *  the DLIST hanging off the logical disk.
 */
DISKSEG * build_ebr_disk_segment( LOGICALDISK      *ld,
                                  Partition_Record *part,
                                  DISKSEG          *ebr,
                                  lba_t             ebr_lba,
                                  u_int32_t         ptable_index,
                                  BOOLEAN           primary_partition_flag  )
{
    Partition_Record            pr;
    DISKSEG                    *new_ebr=NULL;
    DLA_Table_Sector           *dlat_buffer = NULL;
    DISK_PRIVATE_DATA          *disk_pdata = get_disk_private_data(ld);
    sector_count_t              ebr_size;
    Extended_Boot_Record        ebr_buffer;
    struct plugin_functions_s  *dft;
    Partition_Record           *tpart=NULL;
    int                         rc, i;


    LOGENTRY();
    LOG_DEBUG("ebr lba= %lld\n", ebr_lba );

    if (disk_pdata) {



        // for OS2 disks we need to get the Drive Link Address Table
        if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

            dlat_buffer = Read_Dlat_Sector( ld, ebr_lba );

            if (dlat_buffer==NULL) {
                return NULL;
            }

        }

        // initialize size to a full track
        ebr_size = disk_pdata->geometry.sectors_per_track;

        // now try to get actual sector allocation for the EBR
        // this will be the distance between the EBR sector and
        // the start of the logical partition.
        dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
        if (dft==NULL) {
            LOG_ERROR("error, logical disk has no plugin function table\n");
            return NULL;
        }
        else {

            rc = dft->read(ld, ebr_lba, 1, (void *) &ebr_buffer );
            if ( rc == 0 ) {

                // find logical partition record
                for (i=0; i<4; i++) {

                    tpart = &ebr_buffer.Partition_Table[i];

                    if ( isa_ebr_partition_record(tpart) == FALSE ) {

                        // we only shrink them ...
                        if ( DISK_TO_CPU32(START_LBA(tpart)) < ebr_size ) {
                            ebr_size = DISK_TO_CPU32(START_LBA(tpart));
                        }
                        break;
                    }

                }

            }

        }


        // build an extended partition EBR meta data segment
        memset( &pr, 0, sizeof(Partition_Record));
        pr.nr_sects   = CPU_TO_DISK32(ebr_size);
        pr.start_sect = START_LBA(part);
        pr.sys_ind    = SYS_IND(part);
        new_ebr = build_diskseg_from_partition_record( ld, &pr, ebr, ptable_index, primary_partition_flag );

        if (new_ebr==NULL) {
            if (dlat_buffer) free(dlat_buffer);
        }
        else {
            ((SEG_PRIVATE_DATA *)new_ebr->private_data)->dlat = dlat_buffer;
        }
    }

    LOGEXIT();
    return new_ebr;
}




/*
 *  Called to add a segment to the logical disk's segment DLIST. First, convert the
 *  partition record to a DISKSEG struct.  Then, insert the DISKSEG struct into
 *  the DLIST hanging off the logical disk.
 *
 *  Note: Because this routine converts from partition records to storage objects it
 *        is only called during discovery when we are reading partition tables.
 *
 */
DISKSEG * build_diskseg_from_partition_record(  LOGICALDISK       *ld,
                                                Partition_Record  *part,
                                                DISKSEG           *ebr,
                                                u_int32_t          ptable_index,
                                                BOOLEAN            primary_partition )
{
    DISKSEG           *seg        = NULL;
    SEG_PRIVATE_DATA  *pdata      = NULL;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);


    LOGENTRY();

    // allocate a new disk segment storage object
    seg = allocate_disk_segment( ld );
    if (seg==NULL) {
        LOGEXIT();
        return NULL;
    }
    else {
        pdata      = (SEG_PRIVATE_DATA *) seg->private_data;
    }

    // copy disk geometry to storage object
    memcpy(&seg->geometry, &disk_pdata->geometry, sizeof(geometry_t) );

    seg->size              = DISK_TO_CPU32(NR_SECTS(part));
    seg->start             = DISK_TO_CPU32(START_LBA(part));

    pdata->sys_id          = (u_int32_t) SYS_IND(part);
    pdata->boot_ind        = (u_int32_t) BOOT_IND(part);


    /* mbr, ebr, primary or logical ? */
    if ( isa_ebr_partition_record(part) ) {
        pdata->flags      |= SEG_IS_EBR;
    }
    else if ( isa_mbr_partition_record(part) ) {
        pdata->flags      |= SEG_IS_MBR;
    }
    else if ( primary_partition == TRUE ) {
        pdata->flags      |= SEG_IS_PRIMARY_PARTITION;
        // seg->flags        |= SOFLAG_PRIMARY;
    }
    else {
        pdata->flags      |= SEG_IS_LOGICAL_PARTITION;
    }

    /*  Data, MetaData segment type */
    if ( ( isa_ebr_partition_record(part) ) ||
         ( isa_mbr_partition_record(part) )) {
        seg->data_type   = META_DATA_TYPE;
    }
    else {
        seg->data_type   = DATA_TYPE;
    }

    /*  Look for special kind of partitions and mark them */
    if ( SYS_IND(part) == LINUX_RAID_PARTITION ) {
        pdata->flags  |= SEG_IS_LINUX_RAID_PARTITION;
    }
    else if ( isa_linux_swap_partition_record(ld, part, disk_pdata->extd_partition_lba)==TRUE ) {
        pdata->flags  |= SEG_IS_LINUX_SWAP_PARTITION;
    }

    /* Bootable or not */
    if ( BOOT_IND(part) == ACTIVE_PARTITION){
        seg->flags |= SOFLAG_BIOS_READABLE ;
    }


    pdata->ptable_index = ptable_index; // remember where we were found in the EBR or MBR
                                        // this is the index in the ptable where this seg
                                        // was found.

    pdata->ebr          = ebr;          // if primary partition then this pts to the MBR
                                        // if logcial partition then this pts to the EBR
                                        // if EBR meta data seg then this is the same as
                                        // pdata->prev_ebr.
                                        // if MBR meta data seg then this is NULL.


    // since logical drives are relative to the start of the extended
    // partition ... we need to add in the extended partitions lba to
    // get an address that is relative to the start of the disk. Also,
    // any EBR segs we lay down must also have their starting LBA
    // adjusted to be disk relative.
    if ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) {
        seg->start += ebr->start;
    }
    else if ( ( pdata->flags & SEG_IS_EBR ) &&
              ( disk_has_extended_partition(ld) == TRUE ) ){
        seg->start += disk_pdata->extd_partition_lba;
    }

    /*  Only logical and primary partitions have entries in the DLAT.
     *  Here we find the DLAT entry and save a ptr to the DLA_Entry
     *  in the DISKSEG struct.
     */
    if ( ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) &&
         ( seg->data_type == DATA_TYPE ) ) {

        pdata->dla_entry = Get_Dlat_Entry_Matching_DiskSegment(ebr,seg );

        if ( pdata->dla_entry == NULL ) {
            LOG_ERROR("disk partition was not found in corresponding DLA Table\n");
            free_disk_segment(seg);
            seg=NULL;
        }

    }

    LOGEXIT();

    return seg;
}


/*
 *  Wrapper routine that is called to remove a segment storage object from a dlist.
 */
int  remove_diskseg_from_list( dlist_t seglist, DISKSEG *seg )
{
    int                rc;
    LOGICALDISK       *ld = get_logical_disk(seg);
    DISK_PRIVATE_DATA *disk_pdata;
    SEG_PRIVATE_DATA  *seg_pdata;

    disk_pdata = get_disk_private_data(ld);
    seg_pdata  = (SEG_PRIVATE_DATA *)seg->private_data;


    LOGENTRY();
    LOG_DEBUG("segment name= %s\n",  seg->name );

    rc = DeleteObject( seglist,seg );

    if (rc == DLIST_SUCCESS) {

        SegEngFncs->unregister_name( seg->name );

        // unregister the os2 partition name
        if ( ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) &&
             ( seg->data_type == DATA_TYPE ) ){

            char pname[PARTITION_NAME_SIZE+12];

            if (seg_pdata->dla_entry->Partition_Name[0]!=0x00) {

                strcpy(pname, "os2_seg_");
                strncat( pname, (char *) &seg_pdata->dla_entry->Partition_Name[0], PARTITION_NAME_SIZE);

                SegEngFncs->unregister_name( pname );
            }

        }

        // test if we need to unregister any serial numbers for this segment
        if ( seg_pdata->flags & SEG_HAS_DLAT_SERIAL_NUMBERS_REGISTERED ) {

            seg_unregister_serial_number( seg_pdata->dla_entry->Partition_Serial_Number );
            seg_unregister_serial_number( seg_pdata->dla_entry->Volume_Serial_Number );

        }

        // if this is the mbr segment then test if we need to unregister os2 disk info
        if ( ( seg_pdata->flags & SEG_IS_MBR ) &&
             ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES )) {

            seg_unregister_serial_number( seg_pdata->dlat->Disk_Serial_Number );
            SegEngFncs->unregister_name( (char *) seg_pdata->dlat->Disk_Name );

        }


    }
    else {
        LOG_ERROR("call to DeleteObject failed\n");
    }

    LOGEXITRC();

    return rc;
}


/*
 *   Called to name a new disk segment and insert it into the
 *   ordered segment list for the logical disk.
 *
 *   Returns the dlist handle if successful.
 */
void *  insert_diskseg_into_list( dlist_t seglist, DISKSEG *seg)
{
    int                rc=EINVAL;
    SEG_PRIVATE_DATA  *seg_pdata = (SEG_PRIVATE_DATA *) seg->private_data;
    LOGICALDISK       *ld = get_logical_disk(seg);
    DISK_PRIVATE_DATA *disk_pdata=NULL;
    void              *result=NULL;


    LOGENTRY();
    LOG_DEBUG("seg start= %lld   size= %lld\n", seg->start, seg->size);


    // need our private disk data
    disk_pdata = get_disk_private_data(ld);
    if (disk_pdata == NULL) {
        LOGEXIT();
        return NULL;
    }


    /*
     *  Storage objects must all have unique names. First get a name
     *  for the disk segment and then register it with the engine.
     *  if successful ... continue.
     */
    rc = get_name_for_disk_segment( seg );
    if (rc) {
        LOG_ERROR("error, get_name_for_disk_segment failed, RC= %d\n", rc);
        LOGEXIT();
        return NULL;
    }
    else {
        rc = SegEngFncs->register_name( seg->name );
        if (rc) {
            LOG_ERROR("error, get_name_for_disk_segment failed, RC= %d\n", rc);
            LOGEXIT();
            return NULL;
        }
    }


/* engine is removing flags
    if ( seg_pdata->flags & SEG_IS_PRIMARY_PARTITION ) {
        seg->flags |= SOFLAG_PRIMARY;
    }
    else if ( seg_pdata->flags & SEG_IS_LOGICAL_PARTITION) {
        seg->flags |= SOFLAG_EXTENDED;
    }
*/

    /*
     * Check if we need to register any DLAT serial numbers
     */
    if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

        if ( (seg_pdata->flags & SEG_IS_PRIMARY_PARTITION) ||
             (seg_pdata->flags & SEG_IS_LOGICAL_PARTITION)) {


            // OS/2 partition names must be unique. But make
            // sure you have a name to register!
            if ( seg_pdata->dla_entry->Partition_Name[0]!=0x00) {

                char pname[PARTITION_NAME_SIZE+12];

                strcpy(pname, "os2_seg_");
                strncat(pname, seg_pdata->dla_entry->Partition_Name, PARTITION_NAME_SIZE);

                rc = SegEngFncs->register_name( pname );
                if (rc) {
                    LOG_ERROR("error, register os2 partition name failed\n");
                    LOGEXIT();
                    return NULL;
                }

            }

            // OS/2 dlat entries will not have serial numbers if the partitions
            // have not been made into volumes yet.
            if ( (seg_pdata->dla_entry->Partition_Serial_Number != 0 ) &&
                 (seg_pdata->dla_entry->Volume_Serial_Number != 0 ) ) {

                rc  = seg_register_serial_number( seg_pdata->dla_entry->Partition_Serial_Number );

                // dont register LVM serial numbers

                if (rc) {
                    LOG_ERROR("error, register dlat entry serial numbers failed\n");
                    LOGEXIT();
                    return NULL;
                }
                else {
                    seg_pdata->flags |= SEG_HAS_DLAT_SERIAL_NUMBERS_REGISTERED;
                }
            }

        }

    }

    result =  insert_diskseg_into_ordered_list( seglist, seg );

    // if we just created an extended partition then chain it to the
    // mbr so that the routine disk_has_extended_partition() will work.
    if ( ( result != NULL) &&
         ( seg_pdata->flags & SEG_IS_EBR ) &&
         ( disk_has_extended_partition(ld) == FALSE ) ) {

        storage_object_t *mbr;

        mbr = get_mbr_from_seglist( ld->parent_objects );

        if (mbr) {
           ((SEG_PRIVATE_DATA *)mbr->private_data)->next_ebr = result;
        }

    }


    LOG_DEBUG("returning %p\n", result);
    LOGEXIT();
    return result;
}


static char
overlap_partitions_msg[] = "\nFound overlapping partitions on drive %s.\nPartition %s overlaps partition %s at lba %lld.\n";


static char
overlap_segments_msg[] = "\nFound overlapping segment storage objects for drive %s.\nEvms segment %s overlaps segment %s at lba %lld.\n";


/*
 *   Called to insert a disk segment into a segment list that we
 *   are trying to keep ordered by the segments starting LBA
 *   number.
 *
 *   Returns the dlist handle if successful.
 */
void *  insert_diskseg_into_ordered_list( dlist_t seglist, DISKSEG *seg )
{
    void              *handle=NULL;
    DISKSEG           *seg2;
    int                rc;
    LOGICALDISK       *ld = get_logical_disk(seg);
    SEG_PRIVATE_DATA  *seg_pdata;
    SEG_PRIVATE_DATA  *seg2_pdata;
    lba_t              seg2_end_lba;
    BOOLEAN            overlapping=FALSE;
    DISK_PRIVATE_DATA *disk_pdata=NULL;


    LOGENTRY();
    LOG_DEBUG("seg name= %s   seg start= %lld  ends= %lld  size= %lld\n", seg->name, seg->start, seg->start+seg->size-1, seg->size );


    rc = GoToStartOfList( seglist );
    if (rc == DLIST_SUCCESS) {

        while (rc == DLIST_SUCCESS) {

            rc = GetObject( seglist, sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE, (ADDRESS *) &seg2 );

            if (rc == DLIST_SUCCESS) {

                seg2_end_lba = seg2->start + seg2->size - 1;

                  // test and set ... overlapping segments flag
                  if ( (  seg->start >= seg2->start )&&
                       (  seg->start <= seg2_end_lba)) {
                      overlapping = TRUE;
                  }
                  else if ( ( seg->start  <  seg2->start ) &&
                            ( seg2->start <= (seg->start + seg->size - 1)) ) {
                      overlapping = TRUE;
                  }
                  else {
                      overlapping = FALSE;
                  }

                  // try and fix overlapping segments ...
                  if ( overlapping == TRUE ) {

                      LOG_DEBUG("Error ... Overlapping Segments ...\n");
                      LOG_DEBUG("seg2:   name: %s\n", seg2->name );
                      LOG_DEBUG("       start: %lld\n", seg2->start );
                      LOG_DEBUG("        size: %lld\n", seg2->size );
                      LOG_DEBUG("         end: %lld\n", seg2_end_lba );
                      LOG_DEBUG(" overlap lba: %lld\n", seg->start );

                      seg_pdata  = (SEG_PRIVATE_DATA *) seg->private_data;
                      seg2_pdata = (SEG_PRIVATE_DATA *) seg2->private_data;

                      // Test if we simply created the mbr metadata seg too big
                      // some tools will start partitions immediately following
                      // the MBR sector, instead of leaving an entire track for the
                      // MBR.
                      if ( ( seg2_pdata->flags & SEG_IS_MBR ) &&
                           ( seg->start > seg2->start  ) ) {

                          seg2->size = seg->start;   // solution ... just shrink
                                                     // the mbr segment
                      }

                      // Test if we have an ebr metadata seg that is too big.
                      // Usually, we give the ebr segment an entire track like
                      // msdos would. Sometimes though, we find an ebr segment
                      // that larger than a track and we need to shrink it down
                      // to fit in another logical drive.  Usually, you will
                      // find the logical drive starting in the sector immediately
                      // following the ebr segment we are overlapping.  This
                      // happens when the ebr is in the 1st sector of the extended
                      // partition but the data partition is at the end of the drive.
                      // Then, other logical partitions are added in between the two.
                      else if ( ( seg2_pdata->flags & SEG_IS_EBR ) &&
                                ( seg->start > seg2->start       ) ) {

                          seg2->size = seg->start - seg2->start; // shrink ebr segment

                          // I'd like to shrink down to a single track if possible
                          disk_pdata = get_disk_private_data(ld);
                          if ( disk_pdata != NULL &&
                               seg2->size > disk_pdata->geometry.sectors_per_track ) {
                               seg2->size = disk_pdata->geometry.sectors_per_track;
                          }

                      }

                      // Test if we have an ebr metadata seg that is being used
                      // to anchor an ebr chain and along comes a logical drive
                      // that sits on top of the anchor seg. In this situation we
                      // simply move and shrink the new ebr segment as you will
                      // see. This situation occurs during a create and we catch
                      // it here to help ease the create seg work.
                      else if ( ( seg_pdata->flags & SEG_IS_EBR  ) &&
                                ( seg2_pdata->flags & SEG_IS_EBR ) &&
                                ( seg->start == seg2->start      ) ) {

                          seg2->size = 1;
                          seg->size -= 1;
                          ++seg->start;

                      }

                      else {
                          rc = EINVAL;    // must be genuine partition overlap
                          break;          // break out of loop ... looking for insertion pt
                      }

                  }


                // test for ... valid insertion point
                if (seg2->start > seg->start ) break;

                rc = NextItem( seglist );
            }

        }

    }


    switch (rc) {

    case EINVAL:

        if ( (seg->data_type == DATA_TYPE) &&
             (seg2->data_type == DATA_TYPE)) {
            POPUP_MSG(NULL,NULL,overlap_partitions_msg, ld->name, seg->name, seg2->name, seg->start);
        }
        else {
            POPUP_MSG(NULL,NULL,overlap_segments_msg, ld->name, seg->name, seg2->name, seg->start );
        }
        break;

    case DLIST_SUCCESS:      /* Ok, found a segment we should insert in front of */

        rc = InsertObject ( (dlist_t)          seglist,
                            (int)              sizeof(DISKSEG),
                            (void *)           seg,
                            (TAG)              SEGMENT_TAG,
                            (void *)           NULL,
                            (Insertion_Modes)  InsertBefore,
                            (BOOLEAN)          TRUE,
                            (void **)         &handle );
        break;

    case DLIST_EMPTY:
    case DLIST_END_OF_LIST:  /* This new segment lays out on the disk at a higher */
                              /* LBA than any other existing segment.  So, just    */
                              /* insert at end of the segment list.                */

        rc = InsertObject ( (dlist_t)          seglist,
                            (int)              sizeof(DISKSEG),
                            (void *)           seg,
                            (TAG)              SEGMENT_TAG,
                            (void *)           NULL,
                            (Insertion_Modes)  AppendToList,
                            (BOOLEAN)          TRUE,
                            (void **)         &handle );
        break;

    default:     /* REAL ERROR ... return NULL ptr */
        LOG_ERROR("error, insertion failed ... RC= %d\n",  rc);
        break;
    }


    LOGEXIT();

    if (rc) {
        return NULL;
    }
    else {
        return handle;
    }
}



/*
 *     A disk can be addressed by LBA or by CHS values. A partition table entry
 *     uses the following field sizes:
 *
 *     LBA - 32 bits
 *
 *     CHS - 24 bits  ( cylinders - 10 bits,  heads - 8 bits, sectors - 6 bits )
 *
 *     So, the CHS value has maximums of: 1024 cylinders, 255 heads, 63 sectors
 *
 *     Since disks have grown beyond 1024 cylinders in size, there is a special
 *     convention used in partition records to signify this.
 *
 *     If a partition record has an LBA that wont fit into either the starting
 *     or ending CHS values ... then ... use the LBA field in the partition record
 *     and calculate the ending address as LBA+SIZE.  The partition record
 *     signifys this by placing max values in the CHS fields.
 *
 *     The below routine was written to intercept calls to the LBAtoCHS() conversion
 *     routine ... when the returned CHS values are to be placed in a partition record.
 *
 *     This was done so we'd have a chance to convert CHS values to MAXIMUMS when
 *     we see that the LBA is above 1024 cylinders ... OR ... that the disk is
 *     using LBA addressing for all partition records.
 */
int LBA_to_Ptable_CHS( LOGICALDISK *ld, lba_t  lba, chs_t  *chs )
{
    int  rc;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
    BOOLEAN            lba_addressing = FALSE;


    // set LBA addressing flag
    if (disk_pdata->flags & DISK_USES_LBA_ADDRESSING) {
        rc = 0;
        lba_addressing = TRUE;
    }
    else {
        rc = LBAtoCHS( ld, lba, chs );
    }

    if (rc==0) {

        /* if above 1024 cylinder limit of a partition table entry ... we need to
         * max out the CHS values to indicate this situation. We also need to
         * do this if we are using LBA addressing on this disk.
         */
        if ( ( chs->cylinder > MAX_CYLINDERS ) ||
             ( lba_addressing == TRUE ) ) {

            if (MAX_CYLINDERS > disk_pdata->geometry.cylinders) {
                chs->cylinder = disk_pdata->geometry.cylinders-1;
            }
            else {
                chs->cylinder = MAX_CYLINDERS;
            }

            chs->head     = disk_pdata->geometry.heads-1;
            chs->sector   = disk_pdata->geometry.sectors_per_track;

        }


    }

    return rc;
}


/*
 *     LBA addresses are converted into CHS addressees by this formula:
 *
 *        Sector    = ( LBA MOD Sectors Per Track ) + 1
 *        Head      = ( LBA DIV Sectors Per Track ) MOD Heads Per Cylinder
 *        Cylinder  = ( LBA DIV Sectors Per Track ) DIV Heads Per Cylinder
 */
int LBAtoCHS( LOGICALDISK *ld, lba_t  lba, chs_t *chs )
{
 u_int32_t          SectorsPerTrack;
 u_int32_t          DriveHeads;
 u_int32_t          SectorsPerCylinder;
 int                rc;
 DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);

    memset(chs, 0, sizeof(chs_t));

    if ( disk_pdata != NULL &&
         chs        != NULL &&
         disk_pdata->geometry.sectors_per_track > 0  ) {

        SectorsPerTrack    = disk_pdata->geometry.sectors_per_track;
        DriveHeads         = disk_pdata->geometry.heads;
        SectorsPerCylinder = SectorsPerTrack * DriveHeads;

        chs->sector   = ( lba % SectorsPerTrack ) + 1;
        chs->head     = ( lba / SectorsPerTrack ) % DriveHeads;
        chs->cylinder =   lba / SectorsPerCylinder;

        rc = 0;
    }
    else {
        rc = EINVAL;
    }

    return rc;
}



/*
 *     CHS addresses are converted into LBA addresses by the following formula:
 *
 *        LBA = (Sector - 1) + (Head * Sectors Per Track) + (Cylinder * Heads Per Cylinder * Sectors Per Track)
 *
 */
int CHStoLBA( LOGICALDISK *ld, chs_t *chs, lba_t *callers_lba )
{
    u_int32_t          SectorsPerTrack;
    u_int32_t          DriveHeads;
    u_int32_t          SectorsPerCylinder;
    lba_t              lba = 0;
    int                rc;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);


    if (  disk_pdata  != NULL &&
          chs         != NULL &&
          callers_lba != NULL &&
          disk_pdata->geometry.sectors_per_track > 0 ) {

        SectorsPerTrack    = disk_pdata->geometry.sectors_per_track;
        DriveHeads         = disk_pdata->geometry.heads;
        SectorsPerCylinder = SectorsPerTrack * DriveHeads;

        lba = (chs->sector - 1 ) + (chs->head * SectorsPerTrack) + (chs->cylinder * SectorsPerCylinder );

        rc = 0;
    }
    else {
        rc = EINVAL;
    }

    *callers_lba = lba;

    return rc;
}



/*
 *  Called to walk an MBR/EBR chain and calculate the size for each
 *  EBR found. This is only called by fixup_EBR_Chain() because this
 *  routine is dependant on the EBR chaining ptrs which are setup by
 *  the routine fixup_disk_extd_partition_dimensions().
 *
 *  The purpose of the routine is to save away the size of each EBR
 *  so that the commit code can use the EBR_SECTOR_COUNT field in
 *  seg private data when committing an EBR table.  This is needed
 *  in advance because the commit code would have to read ahead to
 *  get this info and it screws up the commit.  Look at the routine
 *  Build_EBR_PartitionTable(), found in commit.c.  When it is building
 *  an ebr table, and it encounters an extended partition record, it
 *  needs to know how big the extd partitoin is ... that the record
 *  points to.  Rather than do read aheads ... I just keep the info
 *  in the ebr seg private data.
 *
 */
static void fixup_EBR_Sizes( LOGICALDISK *ld )
{
    int               rc;

    DISKSEG          *seg;
    SEG_PRIVATE_DATA *pdata;

    DISKSEG          *ebr;
    SEG_PRIVATE_DATA *ebr_pdata;

    DISKSEG           *mbr = get_mbr_from_seglist(ld->parent_objects);
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data( ld );


    LOGENTRY();

    // advance MBR -> extended partition
    ebr_pdata = (SEG_PRIVATE_DATA *) mbr->private_data;
    ebr   = ebr_pdata->next_ebr;

    // return if no extended partition exists
    if (ebr==NULL) {
        LOG_DEBUG("no extended partition on this disk\n");
        LOGEXIT();
        return;
    }

    // advance extd partition -> 1st logical drive
    ebr_pdata                   = (SEG_PRIVATE_DATA *) ebr->private_data;
    ebr_pdata->ebr_sector_count = disk_pdata->extd_partition_size;
    ebr                         = ebr_pdata->next_ebr;

    // walk EBR chain
    while (ebr != NULL) {

        ebr_pdata = (SEG_PRIVATE_DATA *) ebr->private_data;

        // initial size is just the EBR track itself
        ebr_pdata->ebr_sector_count = ebr->size;

        // now add in logical partitions to get total size.
        rc = GoToStartOfList( ld->parent_objects );
        if (rc == DLIST_SUCCESS) {

            rc = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &seg );
            if (rc == DLIST_SUCCESS ) {

                do {
                    pdata = (SEG_PRIVATE_DATA *) seg->private_data;

                    if ( ( pdata->ebr == ebr ) &&
                         ( pdata->flags & SEG_IS_LOGICAL_PARTITION )) {
                        ebr_pdata->ebr_sector_count += seg->size;
                    }

                    rc = GetNextObject( ld->parent_objects, sizeof(DISKSEG), SEGMENT_TAG, (void **) &seg );
                } while (rc==DLIST_SUCCESS);

            }
        }

        ebr = ebr_pdata->next_ebr;
    };


    LOGEXIT();
}


/*
 *  Called to fixup names of logial and embedded partitions because their names will
 *  change if the EBR chain is modified by a delete or create.
 */
int   fixup_logical_partition_names( LOGICALDISK *ld )
{
    int                rc;
    DISKSEG           *seg=NULL;
    DISKSEG           *embedded_seg=NULL;
    DISKSEG           *ebr=NULL;
    SEG_PRIVATE_DATA  *pdata=NULL;
    SEG_PRIVATE_DATA  *ebr_pdata=NULL;
    u_int32_t          embedded_minor;
    dlist_t            embedded_seg_list;
    void              *handle;
    int                embedded_partition_count=0;
    int                logical_partition_count=0;
    int                i;
    int                lowest_number;


    LOGENTRY();

    // Get a temp dlist to use
    embedded_seg_list = CreateList();
    if (embedded_seg_list == NULL) {
        rc = DLIST_CORRUPTED;
        LOG_ERROR("error: create embedded seglist failed\n");
        LOGEXITRC();
        return rc;
    }

    // unregister all logical partition names and embedded partition names
    rc = GoToStartOfList( ld->parent_objects );
    if (rc == DLIST_SUCCESS) {

        rc = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &seg );
        while (rc == DLIST_SUCCESS) {

            pdata = (SEG_PRIVATE_DATA *) seg->private_data;

            if ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) {

                ebr = pdata->ebr;
                if (ebr) {

                    ebr_pdata = (SEG_PRIVATE_DATA *) ebr->private_data;
                    if (ebr_pdata) {

                        if (strlen(seg->name)) SegEngFncs->unregister_name( seg->name );

                        // build a name for this logical partition
                        pdata->part_number  = ebr_pdata->ebr_number + 5;
                        sprintf(seg->name, "%s%d", ld->name, pdata->part_number  );

                        ++logical_partition_count;
                    }

                }
                else {
                    LOG_ERROR("error, found a logical partition that has no ebr associated with it.\n");
                    // but continue to get logical partitions named correctly
                }

            }
            else if ( pdata->flags & SEG_IS_EMBEDDED ) {

                if (strlen(seg->name)) SegEngFncs->unregister_name( seg->name );

                rc = InsertObject ( embedded_seg_list,
                                    sizeof(DISKSEG),
                                    seg,
                                    SEGMENT_TAG,
                                    (void *)  NULL,
                                    (Insertion_Modes)  AppendToList,
                                    TRUE,
                                    &handle );

                if (rc == DLIST_SUCCESS) {
                    ++embedded_partition_count;
                }
                else {
                    LOG_ERROR("error, dlist errors constructing embedded seglist\n");
                    // but continue to get logical partitions named correctly
                }
            }


            rc = GetNextObject( ld->parent_objects, sizeof(DISKSEG), SEGMENT_TAG, (void **) &seg );

        }
    }

    embedded_minor = logical_partition_count + 5;

    // build new embedded partition names
    rc = 0;
    for (i=0; (i<embedded_partition_count)&&(rc==0); i++) {

        lowest_number   = MAX_EBR_NUMBER;
        embedded_seg    = NULL;

        rc = GoToStartOfList( embedded_seg_list );
        if (rc == DLIST_SUCCESS) {

            rc = GetObject( embedded_seg_list, sizeof(DISKSEG), SEGMENT_TAG, NULL, TRUE, (void **) &seg );
            if (rc == DLIST_SUCCESS ) {

                do {
                    pdata = (SEG_PRIVATE_DATA *) seg->private_data;

                    if ( pdata->part_number < lowest_number ) {
                        embedded_seg = seg;
                        lowest_number = pdata->part_number;
                    }

                    rc = GetNextObject( embedded_seg_list, sizeof(DISKSEG), SEGMENT_TAG, (void **) &seg );

                } while ( rc == DLIST_SUCCESS );

                if (embedded_seg != NULL) {

                    pdata = (SEG_PRIVATE_DATA *) embedded_seg->private_data;

                    pdata->part_number  = embedded_minor;
                    sprintf(embedded_seg->name, "%s%d", ld->name, pdata->part_number  );
                    ++embedded_minor;

                    rc = DeleteObject( embedded_seg_list, embedded_seg );

                }
                else {
                    rc = ENODEV;
                }

            }

        }

       // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
       if ( ( rc == DLIST_EMPTY ) || ( rc == DLIST_END_OF_LIST )) {
           rc = DLIST_SUCCESS;
       }

    }

    // register all logical and embedded partition names
    rc = GoToStartOfList( ld->parent_objects );
    if (rc == DLIST_SUCCESS) {

        rc = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &seg );
        while (rc == DLIST_SUCCESS) {

            pdata = (SEG_PRIVATE_DATA *) seg->private_data;

            if ( ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) ||
                 ( pdata->flags & SEG_IS_EMBEDDED ) ) {

                SegEngFncs->register_name( seg->name );

            }

            rc = GetNextObject( ld->parent_objects, sizeof(DISKSEG), SEGMENT_TAG, (void **) &seg );

        }
    }

    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY ) || ( rc == DLIST_END_OF_LIST )) {
        rc = DLIST_SUCCESS;
    }


    DestroyList(&embedded_seg_list, FALSE);
    LOGEXITRC();
    return 0;
}



/*
 *  Called when changes are made to the EBR chain to fixup
 *  the names of EBR segments.  These names, and those of
 *  logical partitions, will change as you add and delete
 *  logical drives.
 */
void fixup_EBR_Names( LOGICALDISK *ld )
{
    u_int32_t         ebr_number = 0;
    DISKSEG          *ebr;
    DISKSEG          *mbr;
    SEG_PRIVATE_DATA *pdata;


    LOGENTRY();

    mbr = get_mbr_from_seglist(ld->parent_objects);
    if ( mbr ) {

        // unregister old names and build new names
        // new names depend on position in ebr chain
        ebr = ((SEG_PRIVATE_DATA *)mbr->private_data)->next_ebr;
        while ( ebr ) {

            pdata = (SEG_PRIVATE_DATA *)ebr->private_data;

            if (pdata) {

                // unregister old name if we have one
                if (strlen(ebr->name)) SegEngFncs->unregister_name( ebr->name );

                // set new ebr number
                pdata->ebr_number = ebr_number;

                // build new name
                sprintf(ebr->name, "%s_ebr%d", ld->name, pdata->ebr_number );

                // increment for next ebr
                ++ebr_number;

                // chain to next ebr
                ebr = pdata->next_ebr;

            }
            else {
                return ; // quit ...
            }
        };

        // now register ebr names
        ebr = ((SEG_PRIVATE_DATA *)mbr->private_data)->next_ebr;
        while ( ebr ) {

            pdata = (SEG_PRIVATE_DATA *)ebr->private_data;

            if (pdata) {

                SegEngFncs->register_name( ebr->name );

                ebr = pdata->next_ebr;

            }
            else {
                return; // quit ...
            }
        }

    }


    LOGEXIT();
}


static int do_os2_ebr_chaining( LOGICALDISK       *ld,
                                DISK_PRIVATE_DATA *disk_pdata,
                                DISKSEG           *mbr,
                                dlist_t            ebr_seg_list,
                                int                ebr_count )
{
    int                rc = 0;
    SEG_PRIVATE_DATA  *pdata;
    DISKSEG           *last_ebr=NULL;
    DISKSEG           *ebr;


    LOGENTRY();


    // ebr chain is anchored off mbr disk segment so setup the mbr
    // for hangingin an ebr chain off it.

    pdata = (SEG_PRIVATE_DATA *) mbr->private_data;

    pdata->ebr          = NULL;
    pdata->prev_ebr     = NULL;
    pdata->next_ebr     = NULL;

    disk_pdata->logical_drive_count = ebr_count;
    disk_pdata->flags  |= DISK_HAS_PHYS_ORDERED_LOGICAL_PARTITIONS;

    last_ebr            = mbr;


    //
    // Now, chain EBRs in the order in which they occur in the working
    // dlist.  This is because they are in the correct order already and
    // so all we need to do is to walk the list and touch up ptrs.
    //

    rc = GoToStartOfList( ebr_seg_list );
    if (rc == DLIST_SUCCESS) {

        rc = GetObject( ebr_seg_list, sizeof(DISKSEG), SEGMENT_TAG, NULL, TRUE, (void **) &ebr );
        while (rc == DLIST_SUCCESS ) {

            //  fixup ... ptable index if prev ebr ptr is actually mbr
            if (last_ebr == mbr) {

                int mbr_ptable_index = get_extd_partition_ptable_entry(ld->parent_objects, mbr);

                if ( mbr_ptable_index == -1 ) {
                    LOG_ERROR("error, there are no unused entries in MBR partition table\n");
                    rc = EINVAL;
                    LOGEXITRC();
                    return rc;
                }
                else {
                    pdata->ptable_index = mbr_ptable_index;
                }
            }

            /* fix up THIS EBR chaining ptrs */
            pdata = (SEG_PRIVATE_DATA *) ebr->private_data;
            pdata->ebr          = last_ebr;
            pdata->prev_ebr     = last_ebr;
            pdata->next_ebr     = NULL;

            /* fix up PREV EBR forward ptrs */
            pdata = (SEG_PRIVATE_DATA *) last_ebr->private_data;
            pdata->next_ebr     = ebr;

            /* setup for next pass */
            last_ebr            = ebr;

            rc = GetNextObject( ebr_seg_list, sizeof(DISKSEG), SEGMENT_TAG, (void **) &ebr );
        }
    }

    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY ) || ( rc == DLIST_END_OF_LIST )) {
        rc = DLIST_SUCCESS;
    }

    LOGEXITRC();
    return rc;
}




static int do_linux_ebr_chaining( LOGICALDISK       *ld,
                                  DISK_PRIVATE_DATA *disk_pdata,
                                  DISKSEG           *mbr,
                                  dlist_t            ebr_seg_list,
                                  int                ebr_count )
{
    int                rc = 0;
    int                i;
    SEG_PRIVATE_DATA  *pdata;
    DISKSEG           *last_ebr=NULL;
    DISKSEG           *seg;
    DISKSEG           *ebr;
    u_int32_t          lowest_ebr_number;
    lba_t              last_lba;

    LOGENTRY();


    // ebr chain is anchored off mbr disk segment so setup the mbr
    // for hangingin an ebr chain off it.

    pdata = (SEG_PRIVATE_DATA *) mbr->private_data;

    pdata->ebr          = NULL;
    pdata->prev_ebr     = NULL;
    pdata->next_ebr     = NULL;

    disk_pdata->logical_drive_count = ebr_count;

    last_ebr            = mbr;


    //
    // Now, chain EBRs in the order in which they were discovered on the drive.
    // the EBR chaining ptrs are used by commit code, allowing the code to
    // walk the EBR chain and commit partition tables.
    //
    for (i=0; (i<ebr_count)&&(rc==0); i++) {

        lowest_ebr_number   = MAX_EBR_NUMBER;
        ebr                 = NULL;

        rc = GoToStartOfList( ebr_seg_list );
        if (rc == DLIST_SUCCESS) {

            rc = GetObject( ebr_seg_list, sizeof(DISKSEG), SEGMENT_TAG, NULL, TRUE, (void **) &seg );
            if (rc == DLIST_SUCCESS ) {

                do {
                    pdata = (SEG_PRIVATE_DATA *) seg->private_data;

                    if ( pdata->ebr_number < lowest_ebr_number ) {
                        ebr = seg;
                        lowest_ebr_number = pdata->ebr_number;
                    }

                    rc = GetNextObject( ebr_seg_list, sizeof(DISKSEG), SEGMENT_TAG, (void **) &seg );

                } while ( rc == DLIST_SUCCESS );

                if (ebr != NULL) {

                    rc = DeleteObject( ebr_seg_list, ebr );

                    //  fixup ... ptable index if prev ebr ptr is actually mbr
                    if (last_ebr == mbr) {

                        int mbr_ptable_index = get_extd_partition_ptable_entry(ld->parent_objects, mbr);

                        if ( mbr_ptable_index == -1 ) {
                            LOG_ERROR("error, there are no unused entries in MBR partition table\n");
                            rc = EINVAL;
                            LOGEXITRC();
                            return rc;
                        }
                        else {
                            pdata->ptable_index = mbr_ptable_index;
                        }
                    }

                    /* fix up THIS EBR chaining ptrs */
                    pdata = (SEG_PRIVATE_DATA *) ebr->private_data;
                    pdata->ebr          = last_ebr;
                    pdata->prev_ebr     = last_ebr;
                    pdata->next_ebr     = NULL;

                    /* fix up PREV EBR forward ptrs */
                    pdata = (SEG_PRIVATE_DATA *) last_ebr->private_data;
                    pdata->next_ebr     = ebr;

                    /* setup for next pass */
                    last_ebr            = ebr;

                }
                else {
                    rc = ENODEV;
                }

            }
            else {
                rc = ENODEV;
            }
        }
        else {
            rc = ENODEV;
        }
    }

    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY ) || ( rc == DLIST_END_OF_LIST )) {
        rc = DLIST_SUCCESS;
    }

    // Test if the logical drives are layed down on the disk in
    // physical order.  This is easy to check ... just walk the
    // ebr chain and each LOGICAL PARTITION that we encounter in
    // the chain should start at a higher LBA than the previous
    // LOGICAL PARTITION.
    if (rc == 0 && ebr_count > 0) {

        ebr                 = mbr;
        last_lba            = 0;
        disk_pdata->flags  |= DISK_HAS_PHYS_ORDERED_LOGICAL_PARTITIONS;


        while ( (ebr != NULL) && (disk_pdata->flags & DISK_HAS_PHYS_ORDERED_LOGICAL_PARTITIONS) ){

            rc = GoToStartOfList( ld->parent_objects );
            if (rc == DLIST_SUCCESS) {

                rc = GetObject( ld->parent_objects, sizeof(DISKSEG), SEGMENT_TAG, NULL, TRUE, (void **) &seg );
                while (rc == DLIST_SUCCESS ) {

                    pdata = (SEG_PRIVATE_DATA *) seg->private_data;

                    if ( ( pdata->ebr == ebr ) &&
                         ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) ){

                        if (last_lba > seg->start) {
                            disk_pdata->flags  &= ~DISK_HAS_PHYS_ORDERED_LOGICAL_PARTITIONS;
                            break;
                        }

                        last_lba = seg->start;
                    }

                    rc = GetNextObject( ebr_seg_list, sizeof(DISKSEG), SEGMENT_TAG, (void **) &seg );
                }
            }

            ebr = ((SEG_PRIVATE_DATA *)ebr->private_data)->next_ebr;

            // this new ebr segment must also be in physical order on the disk
            if (ebr != NULL) {

                if ( ebr->start < ((SEG_PRIVATE_DATA *)ebr->private_data)->prev_ebr->start ) {
                    disk_pdata->flags  &= ~DISK_HAS_PHYS_ORDERED_LOGICAL_PARTITIONS;
                }

            }

        }

        rc=0;
    }


    LOGEXITRC();
    return rc;
}


/*
 *  Called to walk an MBR/EBR chain and fixup the forward and backward pointers
 *  that are kept in the segment managers private data area.
 */
int  fixup_EBR_Chain( LOGICALDISK *ld )
{
    int                rc = EINVAL;
    DISKSEG           *seg;
    SEG_PRIVATE_DATA  *pdata;
    DISKSEG           *mbr;
    DISK_PRIVATE_DATA *disk_pdata;
    int                ebr_count=0;
    dlist_t            ebr_seg_list;
    void              *handle;


    LOGENTRY();

    //
    // Test callers parms
    //
    mbr        = get_mbr_from_seglist(ld->parent_objects);
    disk_pdata = get_disk_private_data( ld );

    if ((mbr==NULL)||(disk_pdata==NULL)) {
        LOG_ERROR("error: bad parms, mbr_ptr= %p  disk_pdata_ptr= %p\n", mbr, disk_pdata);
        LOGEXITRC();
        return rc;
    }

    // Get a temp dlist to use
    ebr_seg_list = CreateList();
    if (ebr_seg_list == NULL) {
        LOG_ERROR("error: bad parms, mbr_ptr= %p  disk_pdata_ptr= %p\n", mbr, disk_pdata);
        LOGEXITRC();
        return rc;
    }

    //
    // Copy the EBR segments to our working DLIST
    //
    ebr_count = 0;
    rc = GoToStartOfList( ld->parent_objects );
    if (rc == DLIST_SUCCESS) {
        rc = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &seg );
        while (rc == DLIST_SUCCESS) {

            pdata = (SEG_PRIVATE_DATA *) seg->private_data;

            if (pdata->flags & SEG_IS_EBR) {

                rc = InsertObject ( ebr_seg_list,
                                    sizeof(DISKSEG),
                                    seg,
                                    SEGMENT_TAG,
                                    (void *)  NULL,
                                    (Insertion_Modes)  AppendToList,
                                    TRUE,
                                    &handle );

            }

            if (rc == DLIST_SUCCESS) {
                rc = GetNextObject( ld->parent_objects, sizeof(DISKSEG), SEGMENT_TAG, (void **) &seg );
            }
        };
        rc = 0;
    }


    //
    // see if there are any logical drives to chain together
    //
    rc = GetListSize(ebr_seg_list, &ebr_count);
    if ( rc ) {
        DestroyList(&ebr_seg_list, FALSE);
        LOGEXITRC();
        return rc;
    }


    //
    // OS/2 disks keep logical drives in physical order.
    // A linux disk does not. So, we use two different
    // ordering algorithms to support this.
    //
    if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {
        rc = do_os2_ebr_chaining( ld, disk_pdata, mbr, ebr_seg_list, ebr_count );
    }
    else {
        rc = do_linux_ebr_chaining( ld, disk_pdata, mbr, ebr_seg_list, ebr_count );
    }


    if (rc == 0) {

        // make sure we have a sys_ind value for the extended partition
        // in case we just created the extd partition itself
        if ( (ebr_count > 0) &&
             (disk_pdata->extd_partition_sys_ind == 0)){
            disk_pdata->extd_partition_sys_ind = DOS_EXTENDED_PARTITION;  // use MSDOS default
        }

        // recalc the size of the extended partition
        fixup_disk_extd_partition_dimensions( ld );

        // if logical drives are offset in the extd partition we need to
        // keep an anchor EBR in the 1st sector of the extd partition.
        fixup_disk_extd_partition_anchor(ld);

        // calc sizes of each std ebr partition
        fixup_EBR_Sizes(ld);

        // rename EBR segments due to shifting
        fixup_EBR_Names(ld);

    }


    DestroyList(&ebr_seg_list, FALSE);

    LOGEXITRC();

    return rc;
}



/*
 *  Called to inspect the extended partition on a drive and make sure we have
 *  an anchor EBR in the 1st sector of the extended partition.  This might entail
 *  moving the EBR from the 1st logical drive to this new location.  This really
 *  only happens when we delete the 1st logical drive in a chain and its ebr was
 *  being used to anchor the extd partition.
 *
 *  Simple EBR Chain occurs when logical drives are in physical order ...
 *
 *      MBR -> LD0 -> LD1 -> LD2
 *
 *  More complex EBR chain occcurs when user creates the 1st logical drive, using
 *  an offset into the freespace segment.
 *
 *      MBR -----------------> LD0
 *             LD2 <-- LD1<--- +
 *
 *  This results in requiring an EBR sector in the 1st sector of the extd partition
 *  AT ALL TIMES!  So, if you later delete LD0 you still need to maintain the EBR
 *  anchor.
 *
 *  The extended partition will look kinda like this ...
 *
 *         +----------------------------------------------------------------+
 *         |  +-----------------------------+                               |
 *         |  |     +---------------------+ | +---+                         |
 *         |  |     |  +---+              | | |   |                         |
 *         |  |     V  |   V              | V |   V                         V
 *         +------+-------+----------------+------+---------------------------+------------+
 *  MBR--> | ebr0 |  ebr2 | ld2            | ebr1 | ld1                       | ld0        |
 *         +------+-------+----------------+------+---------------------------+------------+
 *
 *
 *  You can see that the extended partition record, from the MBR partition table, expects
 *  to find an EBR in the 1st sector of the extended partition.  If we deleted LD0 and its
 *  EBR sector there would no longer be an EBR at this location.
 *
 *
 *  SO ... the purpose of this routine is to preserve the anchor EBR if we need one!!!!
 *
 *  Obviously,if the logical drives are in physical order then as we delete drives we'll
 *  simply shrink the extended partition.
 *
 */
void  fixup_disk_extd_partition_anchor( LOGICALDISK *ld )
{
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
    DISKSEG           *mbr;
    DISKSEG           *ebr=NULL;
    DISKSEG           *ebr2=NULL;
    DISKSEG           *anchor_ebr=NULL;
    DISKSEG           *lp;
    DISKSEG           *first_lp=NULL;
    lba_t              ebr_start=0;
    sector_count_t     ebr_size=0;
    int                rc;


    LOGENTRY();

    // OS2 partitions are in physical order ... so we never have to
    // anchor them with an ebr at sector Zero of the ext partition.
    if ( (disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES) == FALSE) {

        LOG_DEBUG("extended start lba: %lld\n", disk_pdata->extd_partition_lba );
        LOG_DEBUG("extended   end lba: %lld\n", disk_pdata->extd_partition_end_lba );
        LOG_DEBUG("extended      size: %lld\n", disk_pdata->extd_partition_size );

        mbr = get_mbr_from_seglist(ld->parent_objects);

        if ( mbr ) {

            ebr = ((SEG_PRIVATE_DATA *)mbr->private_data)->next_ebr;

            if ( ebr ) {

                // if we have an existing extended partition AND ...
                // we need to move the anchor ebr THEN ...
                // move ebr to new location on drive
                if ( ( disk_pdata->extd_partition_size != 0 ) &&
                     ( ebr->start != disk_pdata->extd_partition_lba )) {

                    // save ebr info
                    ebr_start = ebr->start;
                    ebr_size  = ebr->size;

                    // look for an ebr already anchoring the extd partition
                    rc = GoToStartOfList( ld->parent_objects );
                    if (rc == DLIST_SUCCESS) {

                        rc = GetObject( ld->parent_objects, sizeof(DISKSEG), SEGMENT_TAG, NULL, TRUE, (void **) &ebr2 );
                        while (rc == DLIST_SUCCESS && anchor_ebr == NULL) {

                            if (ebr2->start == disk_pdata->extd_partition_lba) {
                                anchor_ebr = ebr2;
                            }
                            else {
                                rc = GetNextObject( ld->parent_objects, sizeof(DISKSEG), SEGMENT_TAG, (void **) &ebr2 );
                            }

                        }
                    }

                    // look for the first logical partition in current ebr chain
                    rc = GoToStartOfList( ld->parent_objects );
                    if (rc == DLIST_SUCCESS) {

                        rc = GetObject( ld->parent_objects, sizeof(DISKSEG), SEGMENT_TAG, NULL, TRUE, (void **) &lp );
                        while (rc == DLIST_SUCCESS && first_lp == NULL) {

                            if ( ( lp->data_type == DATA_TYPE ) &&
                                 ( ((SEG_PRIVATE_DATA *)lp->private_data)->ebr == ebr )) {
                                first_lp = lp;
                            }
                            else {
                                rc = GetNextObject( ld->parent_objects, sizeof(DISKSEG), SEGMENT_TAG, (void **) &lp );
                            }

                        }
                    }


                    // throw away any errors like ... DLIST_END_OF_LIST ... which are not real errors
                    rc = 0;

                    // remove it from the disk seglist
                    DeleteObject(ld->parent_objects, ebr );

                    // set new location and size. whenever we split off
                    // the ebr segment like this well only use a single
                    // sector like other partitioning tools.
                    if (anchor_ebr) {
                        ++anchor_ebr->start;
                        --anchor_ebr->size;
                    }

                    // anchor ebr always goes in 1st sector of ext partition
                    ebr->start = disk_pdata->extd_partition_lba;

                    // now check if we size the new anchor ebr to 1 sector because there
                    // is considerable distance to the logical partition it anchors
                    // ... or ...
                    // if we resize the new anchor ebr to the number of sectors
                    // between the ebr and its logical partition.
                    //
                    // this happens when the ebr is currently in the 2nd sector of its
                    // ebr track because we were anchoring an ebr chain in the extd
                    // partition ... and we just destroyed a logical partition ...
                    // leaving a single logical partition in the extd partition.
                    // anchor.
                    if ( ( first_lp != NULL) &&
                         ( first_lp->start > ebr->start ) &&
                         ( (first_lp->start - ebr->start) <= disk_pdata->geometry.sectors_per_track) ) {
                        ebr->size = first_lp->start - ebr->start;
                    }
                    else {
                        ebr->size = 1;
                    }

                    // insert updated ebr in list ... if any errors ... recover original ebr
                    if ( insert_diskseg_into_ordered_list( ld->parent_objects, ebr )==NULL) {
                        ebr->start = ebr_start;
                        ebr->size  = ebr_size;
                        insert_diskseg_into_ordered_list( ld->parent_objects, ebr );
                    }

                }

            }

        }

    }

    LOGEXIT();
}


/*
 *  Called to inspect the dimensions of an extended partition and adjust the size
 *  to match the EBR chain. It doesn't matter if the disk flags dont show an
 *  extended partition because we will look at the segment dlist and determine if
 *  an extended partition exists, updating the disk flags before exiting.
 */
void  fixup_disk_extd_partition_dimensions( LOGICALDISK *ld )
{
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
    DISKSEG           *seg;
    SEG_PRIVATE_DATA  *pdata;
    DISKSEG           *last_logical = NULL;
    DISKSEG           *first_ebr = NULL;
    lba_t              extd_start;
    sector_count_t     extd_size;
    int                rc;

    LOGENTRY();


    /* walk the dlist looking for an extended partition */
    rc = GoToStartOfList( ld->parent_objects );
    if (rc == DLIST_SUCCESS) {

        rc = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &seg );

        while (rc == DLIST_SUCCESS) {

            pdata = (SEG_PRIVATE_DATA *) seg->private_data;

            if ( (pdata->flags & SEG_IS_EBR) && (first_ebr == NULL)){

                first_ebr = seg;
            }
            else if ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) {

                last_logical = seg;
            }

            rc = GetNextObject( ld->parent_objects, sizeof(DISKSEG), SEGMENT_TAG, (void **) &seg );
        }

    }


    /* test if we found the start and end of the extended partition */
    if (first_ebr  && last_logical) {

        extd_start = first_ebr->start;

        // special check ... if the extd partition is being anchored
        // because the logical drives are out of physical order
        // then we might need to preserve the start of the ext partition.
        // this only occurs when we have freed a single sector at the
        // start of the extd partition, due to destroying a logical
        // parition, and we are moving the start of the extd partition
        // by 1 sector.  However, the anchor sector came from an EBR track
        // and we should not do this. So ... check for it.
        if (extd_start == disk_pdata->extd_partition_lba + 1) {
            extd_start = disk_pdata->extd_partition_lba;
        }

        extd_size  = (last_logical->start + last_logical->size) - extd_start;

        disk_pdata->flags                 |= DISK_HAS_EXTENDED_PARTITION;
        disk_pdata->extd_partition_lba     = extd_start;
        disk_pdata->extd_partition_size    = extd_size;
        disk_pdata->extd_partition_end_lba = extd_start + extd_size - 1;

        rc = 0;

    }
    else {

        disk_pdata->flags                 &= ~DISK_HAS_EXTENDED_PARTITION;
        disk_pdata->extd_partition_lba     = 0;
        disk_pdata->extd_partition_end_lba = 0;
        disk_pdata->extd_partition_size    = 0;

    }

    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY ) || ( rc == DLIST_END_OF_LIST )) {
        rc = DLIST_SUCCESS;
    }


    LOG_DEBUG("extended start lba: %lld\n", disk_pdata->extd_partition_lba );
    LOG_DEBUG("extended   end lba: %lld\n", disk_pdata->extd_partition_end_lba );
    LOG_DEBUG("extended      size: %lld\n", disk_pdata->extd_partition_size );


    LOGEXIT();
}

/*
 *  Called to create a Logical Drive. A logical drive consists of
 *  an EBR track and the logical segment.  Well create the EBR segment
 *  for the logical drive first and then the data segment, inserting
 *  both EBR and logical partition into the DISKSEG DLIST which
 *  hangs off the LOGICALDISK.
 *
 *  (1) actually create an EBR DISKSEG
 *  (2) hang the EBR on the logical disk
 *  (3) adjust the starting lba and size of the DISKSEG
 *      to reflect that we used a track of disk space to
 *      hold the EBR.
 */
int create_primary_partition( LOGICALDISK *ld, DISKSEG *seg, DLA_Entry *dla )
{
    SEG_PRIVATE_DATA  *pdata      = (SEG_PRIVATE_DATA *)seg->private_data;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
    DISKSEG           *mbr        = get_mbr_from_seglist(ld->parent_objects);
    int                rc         = 0;
    void              *handle     = NULL;
    DLA_Entry         *newdla     = NULL;
    int                i;

    LOGENTRY();

    // check create primary partition parms
    if ( ld==NULL || seg==NULL || pdata==NULL || mbr == NULL ) {
        LOG_ERROR("error, invalid parms\n");
        rc = EINVAL;
        LOGEXITRC();
        return rc;
    }

    //  A primary partition points back to the mbr
    pdata->ebr = mbr;

    //  Look for an unused entry in the MBR partition table.
    i = get_first_unused_ptable_entry(ld->parent_objects, mbr);
    if ( i == -1 ) {
        LOG_ERROR("error, there are no unused entries in MBR partition table\n");
        rc = EINVAL;
        LOGEXITRC();
        return rc;
    }

    // found unused entry in partition table so assign
    // this new partition to this ptable entry
    pdata->ptable_index = i;
    pdata->part_number  = i+1;


    // if OS2 partition ... setup dlat entry info
    if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

        /*  A primary partition's DLAT ptr will point to the DLAT
         *  owned by the MBR meta data segment.
         */
        pdata->dlat = ((SEG_PRIVATE_DATA *)(mbr->private_data))->dlat;

        /*  A primary partition will have a DLA entry in the DLAT
         *  found in the MBR track.  This is the dlat owned by our
         *  MBR meta data segment. So, the dla entry for this new
         *  primary partition we are creating will be the first
         *  available entry in the DLA table owned by the MBR.
         */
        pdata->dla_entry = NULL;

        for (i=0; i<4; i++) {

            newdla = &pdata->dlat->DLA_Array[i];

            if ( (newdla->Partition_Size  == 0 ) &&
                 (newdla->Partition_Start == 0 )) {

                /* found an unused dla entry to use, so copy our dla entry
                 * to this location and set the dla entry ptr in this data
                 * segment.
                 */
                pdata->dla_entry = newdla;
                memcpy( pdata->dla_entry, dla, sizeof(DLA_Entry));
                break;
            }
        }

        if (pdata->dla_entry == NULL) {
            LOG_ERROR("error, need a DLA entry but there are none available\n");
            rc = ENOMEM;
            LOGEXITRC();
            return rc;
        }

    }

    /*
     *  Now we can insert this data segment in the segment list that
     *  hangs off the LOGICALDISK struct.
     */
    handle = insert_diskseg_into_list( ld->parent_objects, seg );
    if (handle==NULL) {
        LOG_ERROR("error, some kind of dlist insert error");
        rc = ENOMEM;
    }


    LOGEXITRC();
    return rc;
}


/*
 *  Called to create a Logical Drive. A logical drive consists of
 *  an EBR track and the logical segment.  Well create the EBR segment
 *  for the logical drive first and then the data segment, inserting
 *  both EBR and logical partition into the DISKSEG DLIST which
 *  hangs off the LOGICALDISK.
 *
 *  (1) actually create an EBR DISKSEG
 *  (2) hang the EBR on the logical disk
 *  (3) adjust the starting lba and size of the DISKSEG
 *      to reflect that we used a track of disk space to
 *      hold the EBR.
 *  (4) if we are offsetting within freespace for the 1st
 *      logical drive then we need to split apart the EBR sector
 *      from its data partition.
 */
int create_logical_partition( LOGICALDISK *ld, DISKSEG *seg, DLA_Entry *dla, DISKSEG *freespace, sector_count_t offset )
{
    DISKSEG           *ebr;
    SEG_PRIVATE_DATA  *pdata=NULL;
    SEG_PRIVATE_DATA  *ebr_pdata=NULL;
    int                rc=0;
    void              *handle;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data( ld );
    DLA_Table_Sector  *dlat_buffer = NULL;
    BOOLEAN            split_logical_drive = FALSE;


    LOGENTRY();

    LOG_DEBUG("seg: start= %lld  size= %lld\n", seg->start, seg->size );
    LOG_DEBUG("free: start= %lld size= %lld\n", freespace->start, freespace->size );

    /* first malloc a new EBR storage object */
    ebr = allocate_disk_segment(ld);
    if (ebr==NULL) {
        rc = ENOMEM;
        LOG_ERROR("alloc of new segment storage object failed\n");
        LOGEXITRC();
        return rc;
    }
    else {
        ebr_pdata = (SEG_PRIVATE_DATA *) ebr->private_data;
    }

    // if OS2 partition ... malloc a DLAT buffer and hang it off
    // the new EBR diskseg.
    if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

        dlat_buffer = Allocate_Dlat( ld );
        if (dlat_buffer==NULL) {
            free_disk_segment(ebr);
            rc = ENOMEM;
            LOG_ERROR("alloc of dlat buffer failed\n");
            LOGEXITRC();
            return rc;
        }

    }

    // figure out where to put the ebr and how big the metadata
    // segment should be.  If we are offsetting the logical partition
    // and no extd partition exists yet ... then anchor the extd
    // by placing the ebr at sector 1 in the extd partition
    if ( ( disk_pdata->logical_drive_count == 0 ) &&
         ( (disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES) == FALSE) &&
         ( seg->start > freespace->start ) ) {

        if (freespace->size > disk_pdata->geometry.sectors_per_track) {

            ebr->start  = freespace->start;
            ebr->size   = 1; //disk_pdata->geometry.sectors_per_track;
            freespace->start += ebr->size;
            freespace->size  -= ebr->size;
            split_logical_drive = TRUE;

        }
        else {
            free_disk_segment(ebr);
            rc = ENOMEM;
            POPUP_MSG(NULL, NULL, "\nerror, if you absolutely must use an offset when creating a logical drive, it cannot be less than 1 track in size.\n"
                      "Try an offset of at least %d sectors.\n",
                      disk_pdata->geometry.sectors_per_track );
            LOGEXITRC();
            return rc;
        }

    }
    else {  // ensure the logical partition starts on a track bdy by
            // making sure the ebr metadata segment ends on track bdy

        lba_t ebr_end = roundup_to_track_boundary( ld, seg->start );

        if (ebr_end == seg->start) {  // already on track bdy then just add track
            ebr_end = seg->start + disk_pdata->geometry.sectors_per_track - 1;
        }

        ebr->start  = seg->start;
        ebr->size   = ebr_end - seg->start + 1;

    }

    // finish building the EBR segment
    ebr->data_type          = META_DATA_TYPE;
    ebr->flags             |= SOFLAG_DIRTY;

    ebr_pdata->dlat         = dlat_buffer;
    ebr_pdata->sys_id       = DOS_EXTENDED_PARTITION ;
    ebr_pdata->flags        = SEG_IS_EBR;
    ebr_pdata->ebr_number   = MAX_EBR_NUMBER-1;

    sprintf(ebr->name, "%s_ebr%d", ld->name, ebr_pdata->ebr_number);

    // insert the EBR segment into the ordered list so we can
    // tell where it is in the EBR chain and name it correctly.
    handle = insert_diskseg_into_ordered_list( ld->parent_objects, ebr );
    if (handle==NULL) {
        rc = ENOMEM;
        if (dlat_buffer) free(dlat_buffer);
        if ( split_logical_drive == TRUE ){
            freespace->size  += ebr->size;
            freespace->start -= ebr->size;
        }
        free_disk_segment(ebr);
        LOGEXITRC();
        return rc;
    }

    // Fixup the EBR chain before adding the logical partition
    fixup_EBR_Chain( ld );

    /*
     * the EBR meta data segment exists as an extended partition record in
     * some partition table. Since we just added an EBR segment to the segment
     * DLIST we must find out which partition table the extended partition
     * record will reside within and also which partition table entry will be
     * used.  The partition table is identified by the bkwd pointer in our
     * segments private data area.  And the exact index will be returned by
     * calling first_unused_ptable_entry() with this info.
     */
    if (ebr_pdata->ptable_index == 0) {

        ebr_pdata->ptable_index = get_first_unused_ptable_entry( ld->parent_objects,
                                                             ebr_pdata->prev_ebr );

        if (ebr_pdata->ptable_index == -1) {
            if ( remove_diskseg_from_list(ld->parent_objects, ebr ) == 0) {
                if (dlat_buffer) free(dlat_buffer);
                if (split_logical_drive == TRUE){
                    freespace->size  += ebr->size;
                    freespace->start -= ebr->size;
                }
                free_disk_segment(ebr);
                --disk_pdata->logical_drive_count;
                fixup_EBR_Chain( ld );
            }
            rc = ENOMEM;
        }
    }

    if (rc == 0) {

        /* subtract EBR track from size of logical partition and insert the
         * segment into the DLIST.
         */
        if (split_logical_drive == FALSE){
            seg->start += ebr->size;
            seg->size  -= ebr->size;
        }

        // seg->flags |= SOFLAG_EXTENDED;

        /* setup seg private data */
        pdata = (SEG_PRIVATE_DATA *) seg->private_data;
        pdata->ptable_index = 0;      // this logical partition will be the 1st partition record in table
        pdata->ebr          = ebr;
        pdata->dlat         = dlat_buffer;

        /* update dla entry info and copy dla to dlat */
        if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

            pdata->dla_entry     = &pdata->dlat->DLA_Array[0];

            dla->Partition_Start = seg->start;
            dla->Partition_Size  = seg->size;

            memcpy(pdata->dla_entry, dla, sizeof(DLA_Entry));

        }

        handle = insert_diskseg_into_ordered_list( ld->parent_objects, seg );
        if ( handle ) {

            // because partitions shift when we add logical partitions
            // we need to fixup their names.
            fixup_logical_partition_names(ld);

            // update count
            ++disk_pdata->logical_drive_count;

            // set the size of this logical drive in the EBR private data
            // area. it is simply the size of the EBR (1 track) plus the
            // size of the logical partition itself.
            ebr_pdata->ebr_sector_count = ebr->size + seg->size;

            /*
             *  Lastly, fixup the size of the extended partition if we changed its size
             *  when adding this logical drive.
             */

            fixup_disk_extd_partition_dimensions(ld);
        }
        else {

            // recover the space used by the EBR segment
            if (split_logical_drive == TRUE){
                freespace->size  += ebr->size;
                freespace->start += ebr->size;
            }
            else {
                seg->start -= ebr->size;
                seg->size  += ebr->size;
            }

            remove_diskseg_from_list( ld->parent_objects, ebr );
            if (dlat_buffer) free(dlat_buffer);
            free_disk_segment(ebr);
            fixup_EBR_Chain( ld );
            rc = ENOMEM;
        }

    }


    if (rc == 0) {
        LOG_DEBUG("New EBR ...\n");
        LOG_DEBUG("    Start LBA: %lld\n", ebr->start);
        LOG_DEBUG("         Size: %lld\n", ebr->size);
        LOG_DEBUG("         Name: %s\n", ebr->name );
        LOG_DEBUG("New Logical Partition ...\n");
        LOG_DEBUG("    Start LBA: %lld\n", seg->start);
        LOG_DEBUG("         Size: %lld\n", seg->size);
        LOG_DEBUG("         Name: %s\n", seg->name);
        LOG_DEBUG("FreeSpace ...\n");
        LOG_DEBUG("    Start LBA: %lld\n", freespace->start);
        LOG_DEBUG("         Size: %lld\n", freespace->size);
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Called to find out if the specified DISKSEG will fall adjacant to an extended partition
 *  on the drive.  This question is asked by SEG create code to help it determine if the
 *  new SEG should be a logical drive or not.
 */
BOOLEAN seg_is_within_or_adjacant_to_extended_partition( LOGICALDISK *ld, DISKSEG *seg )
{
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
    DISKSEG           *this_seg;
    SEG_PRIVATE_DATA  *pdata;
    BOOLEAN            rc = FALSE;
    int                error;
    lba_t              extd_start;
    lba_t              extd_end;



    LOGENTRY();

    if (disk_pdata == NULL) {
        LOG_DEBUG("disk has no private data\n");
        LOGEXIT();
        return FALSE;
    }


    if ( disk_has_extended_partition( ld ) == TRUE ) {

        /* need starting and ending LBA of the extended partition */
        extd_start = disk_pdata->extd_partition_lba;
        extd_end   = disk_pdata->extd_partition_end_lba;

        if ( (  seg->start >= extd_start ) &&
             ( (seg->start+seg->size) <= extd_end )) {

            /*  seg falls entirely within extended partition */
            rc = TRUE;
        }
        else if (  (  seg->start >= extd_start ) &&
                   (  seg->start <  extd_end ) &&
                   ( (seg->start+seg->size) > extd_end ) ) {

            /*  seg starts within the extended partition but ends beyond the
             *  end of the extended partition. So the extended partition can
             *  just be expanded.
             */

            rc = TRUE;
        }
        else if ( seg->start < extd_start ) {

            /*
             * seg starts before extended partition so see if there are interveneing
             * segments between this guy and the extended partition.
             */

            error = GoToStartOfList( ld->parent_objects );
            if (error == DLIST_SUCCESS) {

                error = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &this_seg );
                if (error == DLIST_SUCCESS ) {

                    rc = TRUE;

                    while ( error == DLIST_SUCCESS && (this_seg->start < extd_start) ) {

                        if ( this_seg->start > seg->start ) {

                            pdata = (SEG_PRIVATE_DATA *) this_seg->private_data;

                            /* ebr segs and freespace segs are Ok ... all else fail */
                            if ( ( pdata->flags & SEG_IS_PRIMARY_PARTITION ) ||
                                 ( pdata->flags & SEG_IS_LOGICAL_PARTITION )) {
                                rc = FALSE;
                                break;
                            }

                        }

                        error = GetNextObject( ld->parent_objects, sizeof(DISKSEG), SEGMENT_TAG, (void **) &this_seg );
                    }
                }
            }
        }
        else {

            /*
            * seg starts beyond the end of the extended partition so look for
            * interveneing segments between the extended partition and this guy.
            */

            error = GoToStartOfList( ld->parent_objects );
            if (error == DLIST_SUCCESS) {

                error = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &this_seg );
                if (error == DLIST_SUCCESS) {

                    rc = TRUE;

                    while ( error == DLIST_SUCCESS && (this_seg->start < seg->start ) ) {

                        if ( this_seg->start > extd_end  ) {

                            pdata = (SEG_PRIVATE_DATA *) this_seg->private_data;

                            /* ebr segs and freespace segs are Ok ... all else fail */
                            if ( ( pdata->flags & SEG_IS_PRIMARY_PARTITION ) ||
                                 ( pdata->flags & SEG_IS_LOGICAL_PARTITION )) {
                                rc = FALSE;
                                break;
                            }
                        }

                        error = GetNextObject( ld->parent_objects, sizeof(DISKSEG), SEGMENT_TAG, (void **) &this_seg );
                    }
                }
            }
        }
    }


    LOGEXIT();
    return rc;
}


/*
 *  Called because the disk doesn't have an MBR yet.  So, we create one, hang
 *  it off the disk, and fill in a DLAT sector.
 */
int  create_mbr_For_Disk( LOGICALDISK *ld, char *DiskName, BOOLEAN isa_os2_disk )
{
    DISKSEG           *mbr;
    DISKSEG           *freeseg;
    int                rc=EINVAL;
    DLA_Table_Sector  *dlat_buffer = NULL;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);

    LOGENTRY();
    LOG_DEBUG("disk= %s disk_name= %s os2_flag= %d\n", ld->name, DiskName, isa_os2_disk );

    // get the existing freespace segment
    rc = GoToStartOfList( ld->parent_objects );
    if ( rc != DLIST_SUCCESS ) {
        rc = DLIST_OBJECT_NOT_FOUND;
        LOG_ERROR("cant create MBR, no freespace segments found on disk %s\n", ld->name);
        LOGEXITRC();
        return rc;
    }

    rc = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &freeseg );
    if ( rc != DLIST_SUCCESS ) {
        rc = DLIST_OBJECT_NOT_FOUND;
        LOG_ERROR("cant create MBR, no freespace segments found on disk %s\n", ld->name);
        LOGEXITRC();
        return rc;
    }

    /* Ok, we got the first diskseg on the logical drive.  It must be freespace
     * because this is where we need to lay down the MBR track. It must also
     * start at LBA 0 else some non FREESPACE seg exists where the MBR track
     * must reside.
     */
    if ( ( freeseg->data_type != FREE_SPACE_TYPE ) ||
         ( freeseg->start != 0 ) ) {
        rc = DLIST_CORRUPTED;
        LOG_ERROR("cant create MBR, first segment on disk (%s) not FREESPACE seg or not at LBA 0\n", ld->name);
        LOGEXITRC();
        return rc;
    }

    /* hang an MBR meta data segment off the logical drive */
    disk_pdata->flags = 0;
    mbr = build_mbr_disk_segment( ld );
    if (mbr == NULL) {
        rc = DLIST_CORRUPTED;
        free(dlat_buffer);
        LOG_ERROR("cant create MBR, build MBR storage object failed\n");
        LOGEXITRC();
        return rc;
    }


    if ( isa_os2_disk ) { // register Disk Serial Number and Disk Name

        dlat_buffer = Allocate_Dlat(ld);
        if (dlat_buffer == NULL) {
            free_disk_segment(mbr);
            rc = ENOMEM;
            LOG_ERROR("cant create dlat for MBR track\n");
            LOGEXITRC();
            return rc;
        }

        // copy os2 disk name to dlat and disk pdata
        strcpy(dlat_buffer->Disk_Name, DiskName);
        strcpy(disk_pdata->disk_name, DiskName );

        // gen a serial number for the disk
        dlat_buffer->Disk_Serial_Number = seg_gen_serial_number( (u_int32_t) &dlat_buffer->Disk_Serial_Number );
        if (dlat_buffer->Disk_Serial_Number != BAD_SERIAL_NUMBER) {
            rc = seg_register_serial_number( dlat_buffer->Disk_Serial_Number);
        }
        else {
            rc = ENOTUNIQ;
        }

        if (rc != 0) {
            free(dlat_buffer);
            free_disk_segment(mbr);
            LOG_ERROR("cant create MBR, unable to generate disk serial number for DLAT\n");
            LOGEXIT();
            return rc;
        }


        ((SEG_PRIVATE_DATA *)mbr->private_data)->dlat = dlat_buffer;

        if ( SegEngFncs->register_name( dlat_buffer->Disk_Name) != 0 ) {
            rc = ENOTUNIQ;
            seg_unregister_serial_number( dlat_buffer->Disk_Serial_Number );
            free(dlat_buffer);
            free_disk_segment(mbr);
            LOG_ERROR("cant create MBR, unable to register OS2 disk name\n");
            LOGEXITRC();
            return rc;
        }

        disk_pdata->flags |= DISK_HAS_OS2_DLAT_TABLES;

    }


    /*
     *  Things look good so before trying to insert the new data segment
     *  we will need to adjust the freespace segment to leave a 1 track gap at
     *  the start of the drive for the MBR meta data segment we are about to hang.
     */
    freeseg->start  += mbr->size;
    freeseg->size   -= mbr->size;

    if ( insert_diskseg_into_list(ld->parent_objects, mbr) == NULL ) {
        rc = DLIST_CORRUPTED;
        freeseg->start  -= mbr->size;
        freeseg->size   += mbr->size;
        free_disk_segment(mbr);
        free(dlat_buffer);
        LOG_ERROR("cant create MBR, call to insert MBR storage object into disk dlist failed\n");
        LOGEXITRC();
        return rc;
    }


    // Success ...
    rc = 0;
    mbr->flags |= SOFLAG_DIRTY;

    LOGEXITRC();
    return rc;
}


/*
 *  Returns TRUE if the specified disk segment falls within the extended partition
 *  extent on the drive.
 */
BOOLEAN  seg_is_within_the_extended_partition( LOGICALDISK *ld, DISKSEG *seg )
{
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);


    if (disk_pdata) {

        if ( ( seg->start >= disk_pdata->extd_partition_lba )&&
             ( seg->start <  disk_pdata->extd_partition_end_lba )) {

            LOGEXIT();
            return TRUE;

        }

    }

    return FALSE;
}


/*
 * Called to see if a segment can be resized
 */
BOOLEAN seg_is_volitile( DISKSEG *seg )
{
    SEG_PRIVATE_DATA  *pdata = (SEG_PRIVATE_DATA *)seg->private_data;
    LOGICALDISK       *ld = get_logical_disk(seg);
    DISK_PRIVATE_DATA *disk_pdata;


    // cant be any kind of embedded segment
    if ( ( pdata->flags & SEG_IS_EMBEDDED ) ||
         ( pdata->flags & SEG_IS_EMBEDDED_METADATA ) ||
         ( pdata->flags & SEG_IS_CONSUMED ) ) {

        return FALSE;

    }

    // cant reside on disk with screwball geometry
    if (ld) {

        disk_pdata = get_disk_private_data(ld);

        if (disk_pdata) {

            if ( disk_pdata->flags & DISK_HAS_FORCED_LBA_ADDRESSING ) {
                return FALSE;
            }
            else {
                return TRUE;
            }
        }
    }

    return FALSE;
}


/*
 * Called to make a container from a disk segment. A
 * container segment is one that holds an embedded
 * partitioning scheme, e.g. unixware.
 *
 * Make sure it isnt already in the container_seg list.
 */
int  diskseg_to_container_segment( DISKSEG *seg )
{
    DISKSEG           *seg2;
    SEG_PRIVATE_DATA  *pdata = (SEG_PRIVATE_DATA *)seg->private_data;
    LOGICALDISK       *ld;
    DISK_PRIVATE_DATA *disk_pdata;
    void              *handle;
    int                rc=EINVAL;

    LOGENTRY();

    ld = get_logical_disk(seg);

    if (ld) {

        disk_pdata = get_disk_private_data(ld);

        if (disk_pdata) {

            if ( disk_pdata->container_segs ) {

                rc = GoToStartOfList( disk_pdata->container_segs );
                if ( rc == DLIST_SUCCESS ) {

                    rc = GetObject( disk_pdata->container_segs,
                                    sizeof(DISKSEG),
                                    SEGMENT_TAG,
                                    NULL,
                                    TRUE,
                                    (void **)&seg2 );

                    while (rc == DLIST_SUCCESS) {

                        if ( seg2 == seg ) {

                            LOGEXIT();
                            return 0;

                        }

                        rc = GetNextObject( disk_pdata->container_segs,
                                            sizeof(DISKSEG),
                                            SEGMENT_TAG,
                                            (void **) &seg2 );

                    };

                }

            }

            rc = InsertObject ( disk_pdata->container_segs,
                                (int)              sizeof(DISKSEG),
                                (void *)           seg,
                                (TAG)              SEGMENT_TAG,
                                (void *)           NULL,
                                (Insertion_Modes)  InsertBefore,
                                (BOOLEAN)          TRUE,
                                (void **)         &handle );

            if (rc == DLIST_SUCCESS) {
                pdata->flags |= SEG_IS_CONSUMED;
            }

        }

    }

    LOGEXITRC();
    return rc;
}

/*
 *  Called to release a segment from the container list.
 */
void revert_container_segment( DISKSEG *seg )
{
    SEG_PRIVATE_DATA  *pdata = (SEG_PRIVATE_DATA *)seg->private_data;
    LOGICALDISK       *ld;
    DISK_PRIVATE_DATA *disk_pdata;

    LOGENTRY();

    ld = get_logical_disk(seg);

    if (ld) {

        disk_pdata = get_disk_private_data(ld);

        if (disk_pdata) {

            if ( disk_pdata->container_segs ) {

                DeleteObject( disk_pdata->container_segs, seg );

                insert_diskseg_into_list( ld->parent_objects, seg);

                pdata->flags &= ~SEG_IS_CONSUMED;

            }

        }

    }

}


/*
 *  Called to see if the specified disk segment resides
 *  entirely within a container segment.
 *
 *  Returns: TRUE if it is contained by a container seg on the disk
 *           FALSE otherwise
 *
 */
BOOLEAN seg_is_within_container_segment( DISKSEG *seg )
{
    DISKSEG           *container_seg;
    LOGICALDISK       *ld;
    DISK_PRIVATE_DATA *disk_pdata;
    int                rc;

    LOGENTRY();

    ld = get_logical_disk(seg);

    if (ld) {

        disk_pdata = get_disk_private_data(ld);

        if (disk_pdata) {

            if ( disk_pdata->container_segs ) {

                rc = GoToStartOfList( disk_pdata->container_segs );
                if ( rc == DLIST_SUCCESS ) {

                    rc = GetObject( disk_pdata->container_segs,
                                    sizeof(DISKSEG),
                                    SEGMENT_TAG,
                                    NULL,
                                    TRUE,
                                    (void **)&container_seg );

                    while (rc == DLIST_SUCCESS) {

                        if ( ( seg->start >= container_seg->start ) &&
                             ( (seg->start+seg->size-1) <= (container_seg->start+container_seg->size-1))) {
                            LOGEXIT();
                            return TRUE;

                        }

                        rc = GetNextObject( disk_pdata->container_segs,
                                            sizeof(DISKSEG),
                                            SEGMENT_TAG,
                                            (void **) &container_seg );

                    };

                }

            }

        }

    }


    LOGEXIT();
    return FALSE;
}


/*
 *  Called to see if the specified disk segment overlaps
 *  a container segment. It may start before a container and
 *  finish within it ... or else ... it may start within a
 *  container and finish afterwards.
 *
 *  Returns: TRUE if it does overlap some container seg on the disk
 *           FALSE otherwise
 *
 */
BOOLEAN seg_overlaps_container_segment( DISKSEG *seg )
{
    DISKSEG           *container_seg;
    LOGICALDISK       *ld;
    DISK_PRIVATE_DATA *disk_pdata;
    int                rc;
    BOOLEAN            overlapping=FALSE;


    LOGENTRY();

    ld = get_logical_disk(seg);

    if (ld) {

        disk_pdata = get_disk_private_data(ld);

        if (disk_pdata) {

            if ( disk_pdata->container_segs ) {

                rc = GoToStartOfList( disk_pdata->container_segs );
                if ( rc == DLIST_SUCCESS ) {

                    rc = GetObject( disk_pdata->container_segs,
                                    sizeof(DISKSEG),
                                    SEGMENT_TAG,
                                    NULL,
                                    TRUE,
                                    (void **)&container_seg );

                    while (rc == DLIST_SUCCESS) {

                        // test and set ... overlapping segments flag
                        if ( (  seg->start >= container_seg->start )&&
                             (  seg->start <= container_seg->start+container_seg->size-1) ) {
                            overlapping = TRUE;
                        }
                        else if ( ( seg->start  <  container_seg->start ) &&
                                  ( container_seg->start <= (seg->start + seg->size - 1)) ) {
                            overlapping = TRUE;
                        }
                        else {
                            overlapping = FALSE;
                        }

                        if ( overlapping == TRUE) {
                            LOGEXIT();
                            return TRUE;
                        }

                        rc = GetNextObject( disk_pdata->container_segs,
                                            sizeof(DISKSEG),
                                            SEGMENT_TAG,
                                            (void **) &container_seg );

                    };

                }

            }

        }

    }

    LOGEXIT();
    return FALSE;
}




/*
 *  Called to remove any part of a segment object that overlaps
 *  a container segment. This will happen by either shortening
 *  the end of a segment or by removing sectors from its start.
 *
 *  Returns: 0 if segment was successfully adjusted
 *           -1 otherwise
 *
 */
int  remove_container_seg_overlap( DISKSEG *seg)
{
    DISKSEG           *container_seg;
    LOGICALDISK       *ld;
    DISK_PRIVATE_DATA *disk_pdata;
    int                rc;
    sector_count_t     delta;

    LOGENTRY();

    ld = get_logical_disk(seg);

    if (ld) {

        disk_pdata = get_disk_private_data(ld);

        if (disk_pdata) {

            if ( disk_pdata->container_segs ) {

                rc = GoToStartOfList( disk_pdata->container_segs );
                if ( rc == DLIST_SUCCESS ) {

                    rc = GetObject( disk_pdata->container_segs,
                                    sizeof(DISKSEG),
                                    SEGMENT_TAG,
                                    NULL,
                                    TRUE,
                                    (void **)&container_seg );

                    while (rc == DLIST_SUCCESS) {

                        if ( (  seg->start >= container_seg->start )&&
                             (  seg->start <= container_seg->start+container_seg->size-1) ) {

                            delta = (container_seg->start + container_seg->size) - seg->start;

                            if (delta < seg->size) {
                                seg->start  += delta;
                                seg->size   -= delta;
                                return 0;
                            }
                            else {
                                return -1;
                            }

                        }
                        else if ( ( seg->start  <  container_seg->start ) &&
                                  ( container_seg->start <= (seg->start + seg->size - 1)) ) {

                            delta = (seg->start + seg->size) - container_seg->start;

                            if (delta < seg->size) {
                                seg->size   -= delta;
                                return 0;
                            }
                            else {
                                return -1;
                            }

                        }

                        rc = GetNextObject( disk_pdata->container_segs,
                                            sizeof(DISKSEG),
                                            SEGMENT_TAG,
                                            (void **) &container_seg );

                    };

                }

            }

        }

    }

    LOGEXIT();
    return -1;
}



/*
 *  Returns a storage object from the child object list
 *  of the specified storage object ... as long as there
 *  is one and only one object in the child object list.
 */
DISKSEG *  only_child( DISKSEG *seg )
{
    int               rc;
    storage_object_t *only_child = NULL;
    storage_object_t *obj;

    LOGENTRY();

    rc = GoToStartOfList( seg->child_objects );
    if ( rc == DLIST_SUCCESS ) {

        rc = GetObject( seg->child_objects, sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE, (void **) &only_child );
        if ( rc == DLIST_SUCCESS ) {

            // test if there is another object which means we dont have one
            // and only one child object.
            rc = GetNextObject( seg->child_objects,sizeof(DISKSEG),SEGMENT_TAG, (ADDRESS *) &obj );
            if (rc == DLIST_SUCCESS) {
                only_child = NULL;   // return NULL ... cuz not only child
            }                        // condition failed.

        }

    }

    LOGEXIT();
    return  only_child;
}



