/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: expand.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include <fullengine.h>

#include "expand.h"
#include "engine.h"
#include "handlemgr.h"
#include "common.h"
#include "commit.h"
#include "discover.h"
#include "object.h"
#include "memman.h"
#include "internalAPI.h"
#include "message.h"


static inline int get_object_expand_points(storage_object_t * obj, u_int64_t * max_delta_size, dlist_t expand_points) {

    int rc = 0;

    LOG_PROC_ENTRY();

    rc = obj->plugin->functions.plugin->can_expand(obj, max_delta_size, expand_points);

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static inline int get_volume_expand_points(logical_volume_t * volume, dlist_t expand_points) {

    int rc = 0;
    u_int64_t max_delta_size = -1;

    LOG_PROC_ENTRY();

    /*
     * If the volume has an FSIM, gets its maximum delta expand size.  We reget
     * the current size and limits because the file system could be mounted and
     * be active.
     */
    if (volume->file_system_manager != NULL) {
        rc = volume->file_system_manager->functions.fsim->get_fs_size(volume, &volume->fs_size);
        if (rc == 0) {
            rc = volume->file_system_manager->functions.fsim->get_fs_limits(volume,
                                                                            &volume->min_fs_size,
                                                                            &volume->max_fs_size,
                                                                            &volume->max_vol_size);
            if (rc == 0) {
                max_delta_size = volume->max_vol_size - volume->vol_size;
            }
        }
    }

    if (rc == 0) {
        if (!(volume->flags & VOLFLAG_MKFS)) {
            rc = get_object_expand_points(volume->object, &max_delta_size, expand_points);

        } else {
            LOG_ERROR("Volume \"%s\" cannot be expanded because it is scheduled to have a file system installed on it.\n", volume->name);
            rc = EINVAL;
        }
    }

    if (rc == 0) {
        /*
         * If the volume has an FSIM and there are expand points, check if the
         * FSIM can handle the expand.
         */
        if (volume->file_system_manager != NULL) {
            if (!ListEmpty(expand_points)) {
                rc = volume->file_system_manager->functions.fsim->can_expand_by(volume, &max_delta_size);

                /*
                 * The FSIM cannot handle the expand.  Delete all the expand
                 * points from the list.
                 */
                if (rc != 0) {
                    DeleteAllItems(expand_points, TRUE);
                }
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Find out if an object can be expanded.  The "thing" must be either a volume
 * or a top level object.  What we'll do is call the internal
 * get_expand_points() and see if any expand candidates show up.  If we get
 * expand points then return 0 to say that the object can be expanded.  Else,
 * return an error.
 */
int evms_can_expand(object_handle_t thing) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(thing,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            dlist_t expand_point_list = CreateList();

            if (expand_point_list != NULL) {
                switch (type) {
                    case DISK:
                    case SEGMENT:
                    case REGION:
                    case EVMS_OBJECT:
                        {
                            storage_object_t * obj = object;

                            /*
                             * Expands can only be done on top level objects.
                             */
                            if (is_top_object(obj)) {
                                dlist_t expand_points = CreateList();

                                if (expand_points != NULL) {
                                    u_int64_t max_delta_size = -1;

                                    rc = get_object_expand_points(obj, &max_delta_size, expand_points);

                                    if (rc == 0) {
                                        uint count = 0;

                                        GetListSize(expand_points, &count);

                                        if (count == 0) {
                                            rc = ENOENT;
                                        }
                                    }

                                    DestroyList(&expand_points, FALSE);

                                } else {
                                    LOG_CRITICAL("Error allocating memory for an expand point list.\n");
                                    rc = ENOMEM;
                                }

                            } else {
                                LOG_ERROR("Object %s is not a top level object.  Expands can only be done on volumes or top level objects.\n", obj->name);
                                rc = EINVAL;
                            }
                        }
                        break;

                    case VOLUME:
                        {
                            logical_volume_t * volume = (logical_volume_t *) object;
                            dlist_t expand_points = CreateList();

                            if (expand_points != NULL) {
                                rc = get_volume_expand_points(volume, expand_points);

                                if (rc == 0) {
                                    uint count = 0;

                                    GetListSize(expand_points, &count);

                                    if (count == 0) {
                                        rc = ENOENT;
                                    }
                                }

                                DestroyList(&expand_points, FALSE);

                            } else {
                                LOG_CRITICAL("Error allocating memory for an expand point list.\n");
                                rc = ENOMEM;
                            }
                        }
                        break;

                    default:
                        rc = EINVAL;
                        break;
                }

                DestroyList(&expand_point_list, FALSE );

            } else {
                rc = ENOMEM;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Given an expand_info_t, add the app_handle for the object and the size to an
 * expand_handle_array_t.
 * The parameters of this function are structured so that it can be called by
 * ForEachItem().
 */
static int make_expand_handle_entry(ADDRESS object,
                                    TAG     object_tag,
                                    uint    object_size,
                                    ADDRESS object_handle,
                                    ADDRESS parameters) {
    int rc = 0;

    expand_object_info_t * object_info = (expand_object_info_t *) object;
    expand_handle_array_t * eha = (expand_handle_array_t *) parameters;

    LOG_PROC_ENTRY();

    LOG_DEBUG("Current number of entries in handle array:  %d.\n", eha->count);

    if (object_tag == EXPAND_OBJECT_TAG) {
        LOG_DEBUG("Add entry for storage object %s.\n", object_info->object->name);

        rc = ensure_app_handle(object_info->object,
                               object_info->object->object_type,
                               &(object_info->object->app_handle));
        if (rc == 0) {
            eha->expand_point[eha->count].object = object_info->object->app_handle;
            eha->expand_point[eha->count].max_expand_size = object_info->max_expand_size;
            eha->count++;
        }

    } else {
        LOG_WARNING("Cannot make an expand handle entry from an object with tag %ld.\n", object_tag);
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Make an array of expand handles (expand_handle_array_t) for the objects in
 * a dlist_t.
 */
static int make_expand_handle_array(dlist_t list, expand_handle_array_t * * eha) {
    int rc = 0;
    uint count;
    uint size;

    LOG_PROC_ENTRY();

    rc = GetListSize(list, &count);

    if (rc == 0) {
        LOG_DEBUG("Number of objects in the list:  %d\n", count);
        if (count > 1) {
            size = sizeof(expand_handle_array_t) + ((count -1) * sizeof(expand_handle_t));
        } else {
            size = sizeof(expand_handle_array_t);
        }

        *eha = alloc_app_struct(size, NULL);
        if (*eha != NULL) {
            rc = ForEachItem(list,
                             make_expand_handle_entry,
                             *eha,
                             TRUE);
        } else {
            rc = ENOMEM;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Find out which objects in the stack can expand.  evms_get_expand_points()
 * must be targeted at either a volume or a top level object.  Any plug-in
 * that can expand will add an expand_object_info_t to the expand_points
 * dlist.  The dlist is then converted into an expand_handle_array_t for return
 * to the user.
 */
int evms_get_expand_points(object_handle_t thing, expand_handle_array_t * * expand_points) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {

        rc = translate_handle(thing,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {

            if (expand_points != NULL) {
                dlist_t expand_point_list = CreateList();

                if (expand_point_list != NULL) {

                    switch (type) {
                        case VOLUME:
                            {
                                logical_volume_t * volume = (logical_volume_t *) object;

                                rc = get_volume_expand_points(volume, expand_point_list);
                            }
                            break;

                        case EVMS_OBJECT:
                        case REGION:
                        case SEGMENT:
                        case DISK:
                            {
                                storage_object_t * obj = (storage_object_t *) object;
                                /*
                                 * Expands can only be done on top level objects.
                                 */
                                if (is_top_object(obj)) {
                                    u_int64_t max_delta_size = -1;

                                    rc = get_object_expand_points(obj, &max_delta_size, expand_point_list);

                                } else {
                                    LOG_ERROR("Object %s is not a top level object.  Expands can only be done on volumes or top level objects.\n", obj->name);
                                    rc = EINVAL;
                                }
                            }
                            break;

                        default:
                            LOG_ERROR("An object of type %d cannot be expanded.\n", type);
                            rc = EINVAL;
                            break;
                    }

                    if (rc == 0) {
                        rc = make_expand_handle_array(expand_point_list, expand_points);
                    }

                    DestroyList(&expand_point_list, FALSE);
                }

            } else {
                LOG_ERROR("The pointer to the expand points list cannot be NULL.\n");
                rc = EINVAL;
            }

        } else {
            LOG_CRITICAL("Error when allocating memory for an expand point list.\n");
            rc = ENOMEM;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * If the object is part of a volume and the volume has no FSIM, warn the
 * user that there is no FSIM to handle the expand.  The user has the
 * choice of continuing or aborting.
 * Returns 0 if the user wants to continue, EINTR if the user wants to abort.
 */
static int warn_if_no_fsim(storage_object_t * obj) {

    int rc = 0;

    LOG_PROC_ENTRY();

    if (obj->volume != NULL) {
        if (obj->volume->file_system_manager == NULL) {
            char * choices[] = {"Continue", "Abort", NULL};
            int answer = 0;     /* Initialize to "Continue" */

            engine_user_message(&answer,
                                choices,
                                "WARNING: Volume \"%s\" does not have an associated File System Interface Module which could coordinate the expansion of the file system on the volume.  "
                                "The file system will not be expanded.  "
                                "You may need to run a separate utility to expand the file system after this operation completes.  "
                                "Expanding the volume may make the file system unusable.  "
                                "Do you want to continue with the expand or abort?\n",
                                obj->volume->name);

            if (answer != 0) {
                rc = EINTR;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Expand the size of an object.  The expand gets targeted at the top object of
 * the feature stack.  The plug-ins pass it on down until it hits the object
 * that is to do the real expansion.
 */
int evms_expand(object_handle_t thing, handle_array_t * input_objects, option_array_t * options) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {

        rc = translate_handle(thing,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {

            if ((type == EVMS_OBJECT) ||
                (type == REGION) ||
                (type == SEGMENT) ||
                (type == DISK)) {

                storage_object_t * obj = (storage_object_t *) object;

                if (obj->volume != NULL) {
                    if (obj->volume->flags & VOLFLAG_MKFS) {
                        LOG_ERROR("Volume \"%s\" cannot be expanded because it is scheduled to have a file system installed on it.\n", obj->volume->name);
                        rc = EINVAL;
                    }
                }

                if (rc == 0) {
                    dlist_t input_object_list = CreateList();

                    if (input_object_list != NULL) {
                        rc = make_dlist(input_objects, input_object_list);

                        if (rc == 0) {

                            /*
                             * Make sure all of the input objects are storage
                             * objects.  (The caller could have supplied handles
                             * for bogus items such as plug-ins or volumes.)
                             */
                            rc = ForEachItem(input_object_list,
                                             isa_valid_input_object,
                                             NULL,
                                             TRUE);

                            if (rc == DLIST_SUCCESS) {

                                /*
                                 * Find the top object in the feature stack.
                                 */
                                storage_object_t * top_object = obj;
                                uint parent_count = 0;

                                rc = GetListSize(top_object->parent_objects, &parent_count);

                                while ((rc == DLIST_SUCCESS) && (parent_count > 0)) {
                                    rc = GetObject(top_object->parent_objects,
                                                   sizeof(storage_object_t),
                                                   EVMS_OBJECT_TAG,
                                                   NULL,
                                                   FALSE,
                                                   (ADDRESS *) &top_object);

                                    if (rc == DLIST_SUCCESS) {
                                        rc = GetListSize(top_object->parent_objects, &parent_count);
                                    }
                                }

                                /* Clean up error codes that are OK. */
                                if ((rc == DLIST_EMPTY) || (rc == DLIST_END_OF_LIST)) {
                                    rc = DLIST_SUCCESS;
                                }

                                if (rc == DLIST_SUCCESS) {

                                    /*
                                     * Warn the user if there is no FSIM to
                                     * handle the expand.
                                     */
                                    rc = warn_if_no_fsim(obj);

                                    if (rc == 0) {

                                        /*
                                         * Start the expand at the top level
                                         * object.
                                         */
                                        rc = top_object->plugin->functions.plugin->expand(top_object, obj, input_object_list, options);

                                        if (rc == 0) {

                                            if (obj->object_type == EVMS_OBJECT) {
                                                /*
                                                 * Mark all the child feature
                                                 * headers dirty to ensure that
                                                 * they get the correct depth
                                                 * and get written in the right
                                                 * place.
                                                 */
                                                ForEachItem(obj->child_objects,
                                                            mark_feature_headers_dirty,
                                                            NULL,
                                                            TRUE);

                                                /*
                                                 * Make sure the child objects
                                                 * are marked as being in the
                                                 * same volume.
                                                 */
                                                ForEachItem(obj->child_objects,
                                                            set_volume_in_object,
                                                            obj->volume,
                                                            TRUE);
                                            }

                                            /*
                                             * If the object is part of a
                                             * volume...
                                             */
                                            if (obj->volume != NULL) {

                                                /* Mark it for rediscovery. */
                                                engine_rediscover_volume(obj->volume, FALSE);

                                                /*
                                                 * Mark it dirty so that the
                                                 * feature headers get written
                                                 * to their new locations.
                                                 */
                                                obj->volume->flags |= VOLFLAG_DIRTY;

                                                /*
                                                 * Update the volume size.  It
                                                 * may not be the resulting
                                                 * volume size after the FSIM
                                                 * does its expand, but that
                                                 * will be updated after the
                                                 * commit.
                                                 */
                                                obj->volume->vol_size = top_object->size;
                                                if (top_object->feature_header != NULL) {
                                                    /*
                                                     * Object has EVMS volume
                                                     * feature headers.
                                                     * Subtract them from the
                                                     * size.
                                                     */
                                                    obj->volume->vol_size -= FEATURE_HEADER_SECTORS * 2;
                                                }
                                            }
                                        }
                                    }

                                } else {
                                    LOG_WARNING("Error code %d encountered when trying to find the top level object.\n", rc);
                                }

                            } else {
                                LOG_ERROR("One or more items in the input object list is not a storage object.\n");
                                rc = EINVAL;
                            }

                        } else {
                            LOG_ERROR("Error code %d when making a dlist from the input_objects handle array.\n", rc);
                            rc = EINVAL;
                        }

                        DestroyList(&input_object_list, FALSE);

                    } else {
                        LOG_CRITICAL("Error allocating memory to create the input object list.\n");
                        rc = ENOMEM;
                    }
                }

            } else {
                LOG_ERROR("Object of type %d cannot be expanded.\n", type);
                rc = EINVAL;
            }
        }

        if (rc == 0) {
            changes_pending = TRUE;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * engine_can_expand_by() is a service that the Engine provides for the
 * plug-ins.  A plug-in calls engine_can_expand_by() to find out if all of its
 * parent objects, including the volume, would approve of an expansion by the
 * specified amount.  The Engine walks up the object tree calling each
 * plug-in's can_expand_by() function to see if it would approve of the
 * expansion.  Each plug-in can update the "size" if the resulting size of its
 * object would have a different delta size than the delta size of its child
 * object.  If the top of the object stack is reached and the object is part of
 * a volume, the Engine then checks the resulting delta size against any limits
 * that an FSIM may have imposed on the volume.
 */
int engine_can_expand_by(storage_object_t * object, sector_count_t * delta_size) {

    int rc = 0;
    storage_object_t * curr_object = object;
    uint parent_count = 0;
    sector_count_t original_delta_size = *delta_size;

    LOG_PROC_ENTRY();

    rc = GetListSize(object->parent_objects, &parent_count);

    while ((rc == DLIST_SUCCESS) && (parent_count > 0)) {
        rc = GetObject(curr_object->parent_objects,
                       sizeof(storage_object_t),
                       EVMS_OBJECT_TAG,
                       NULL,
                       FALSE,
                       (ADDRESS *) &curr_object);

        if (rc == DLIST_SUCCESS) {
            rc = curr_object->plugin->functions.plugin->can_expand_by(curr_object, delta_size);

            if (rc == 0) {
                rc = GetListSize(curr_object->parent_objects, &parent_count);
            }
        }
    }

    /* Clean up error codes that are OK. */
    if ((rc == DLIST_EMPTY) || (rc == DLIST_END_OF_LIST)) {
        rc = DLIST_SUCCESS;
    }

    if (rc == 0) {
        /*
         * If this object is part of a volume and the volume has an FSIM, check
         * to make sure the final size does not go over the FSIM's maximum size
         * it can handle for its backing storage object.
         */
        if (object->volume != NULL) {
            if (object->volume->file_system_manager != NULL) {
                rc = object->volume->file_system_manager->functions.fsim->can_expand_by(object->volume, delta_size);
            }
        }
    }

    if (rc == 0) {
        if (*delta_size != original_delta_size) {
            rc = EAGAIN;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

