/* ethos-plugin.h
 *
 * Copyright (C) 2009 Christian Hergert <chris@dronelabs.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 
 * 02110-1301 USA
 */

#include "ethos-plugin.h"

/**
 * SECTION:ethos-plugin
 * @title: EthosPlugin
 * @short_description: plugin base class
 *
 * The #EthosPlugin class is a base class that can be inherited to create plugins.  It
 * can also be instantiated directly and have signals attached to build a plugin during
 * runtime.
 *
 * Plugins should inherit this class within their module.
 */

G_DEFINE_TYPE (EthosPlugin, ethos_plugin, G_TYPE_OBJECT)

enum
{
	ACTIVATED,
	DEACTIVATED,
	LAST_SIGNAL
};

static guint signals [LAST_SIGNAL] = {0,};

struct _EthosPluginPrivate
{
	gpointer dummy;
};

static void
ethos_plugin_finalize (GObject *object)
{
	G_OBJECT_CLASS (ethos_plugin_parent_class)->finalize (object);
}

static void
ethos_plugin_class_init (EthosPluginClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	object_class->finalize = ethos_plugin_finalize;

	/**
	 * EthosPlugin::activated:
	 * @plugin: the #EthosPlugin instance
	 *
	 * Signal emitted when the plugin has been activated by the #EthosManager
	 * for the process.
	 */
	signals [ACTIVATED] =
		g_signal_new (g_intern_static_string ("activated"),
		              G_TYPE_FROM_CLASS (klass),
		              G_SIGNAL_RUN_FIRST,
		              G_STRUCT_OFFSET (EthosPluginClass, activated),
		              NULL, NULL,
		              g_cclosure_marshal_VOID__VOID,
		              G_TYPE_NONE,
		              0);

	/**
	 * EthosPlugin::deactivated:
	 * @plugin: the #EthosPlugin instance
	 *
	 * Signal emitted when the plugin has been deactivated by the #EthosManager
	 * for the process.
	 */
	signals [DEACTIVATED] =
		g_signal_new (g_intern_static_string ("deactivated"),
		              G_TYPE_FROM_CLASS (klass),
		              G_SIGNAL_RUN_FIRST,
		              G_STRUCT_OFFSET (EthosPluginClass, deactivated),
		              NULL, NULL,
		              g_cclosure_marshal_VOID__VOID,
		              G_TYPE_NONE,
		              0);
}

static void
ethos_plugin_init (EthosPlugin *plugin)
{
}

/**
 * ethos_plugin_activate:
 * @plugin: An #EthosPlugin
 *
 * Activates the plugin so that it may prepare itself for execution.
 */
void
ethos_plugin_activate (EthosPlugin *plugin)
{
	g_return_if_fail (ETHOS_IS_PLUGIN (plugin));
	g_signal_emit (plugin, signals [ACTIVATED], 0);
}

/**
 * ethos_plugin_deactivate:
 * @plugin: An #EthosPlugin
 *
 * Requests the plugin to deactivate itself so that it may be destroyed.
 */
void
ethos_plugin_deactivate (EthosPlugin *plugin)
{
	g_return_if_fail (ETHOS_IS_PLUGIN (plugin));
	g_signal_emit (plugin, signals [DEACTIVATED], 0);
}
