#include <glib.h>
#include <pcre.h>
#include <string.h>
#include "entity.h"

static ENode *
enode_lookup_by_path (gchar * path)
{
    ENode *curnode;
    gchar **dirname;
    GSList *tmp;
    gint i;
    gint path_length;
    gint found;

    /* do a few simple sanity checks */
    ECHECK_RETVAL (path != NULL, NULL);

    path_length = strlen (path);

    /* break up the dir names */
    dirname = g_strsplit (path, "/", 65535);

    curnode = enode_root_node ();

    EDEBUG (("enode-search", "element = %s\n", curnode->element->str));

    /* Starting at 1 assuming '/'. */
    for (i = 1; dirname[i]; i++) {
	found = FALSE;

	EDEBUG (("enode-search", "dirname[%i] = %s", i, dirname[i]));

	if (dirname[i][0] == '\0')
	    continue;

	tmp = curnode->children;

	while (tmp) {
	    ENode *test_node = tmp->data;
	    EBuf *basename;

	    basename = enode_basename (test_node);

	    if (ebuf_equal_str (basename, dirname[i])) {
		curnode = test_node;
		found = TRUE;
		ebuf_free (basename);
		break;
	    }

	    ebuf_free (basename);

	    tmp = tmp->next;
	}

	if (!found) {
	    g_strfreev (dirname);
	    return (NULL);
	}
    }

    g_strfreev (dirname);
    return (curnode);
}


static gint
enode_basename_match (ENode * node, gchar * basename)
{
    EBuf *name;
    int i;
    char *type;
    
    ECHECK_RETVAL (node != NULL, FALSE);
    ECHECK_RETVAL (basename != NULL, FALSE);

    type = node->element->str;
    
    for (i = 0; ; i++) {
	if (basename[i] == '.') {
	    if (type[i] == '\0') {
	     	/* The basename and type match, so we should
		 * check the name */
		name = enode_attrib (node, "name", NULL);
		if (ebuf_equal_str (name, &basename[i + 1]))
		    return (TRUE);
	    } else {
		/* The basename has a '.' here, but the type still
		 * has some chars.. no match. */
		return (FALSE);
	    }
	}

	/* Straight compare.. note that this will also go off
	 * if the type is end of string '\0', and basename
	 * is not */
	if (basename[i] != type[i])
	    return FALSE;

	/* At this point they both have to be the same, so if one
	 * is \0, the other must be too */
	if (basename[i] == '\0')
	    return (TRUE);
    }
  
    /* NOT REACHED */
    return (FALSE);
}

static ENode *
enode_reference_object (ENode * node)
{
    ENode *objnode;

    if (ebuf_equal_str (node->element, "object"))
	objnode = node;
    else
	objnode = enode_parent (node, "object");

    return (objnode);
}



/* Returns immediate parent node if search is NULL, or will search upwards
 * if given search string, for type[.name] */
ENode *
enode_parent (ENode * node, gchar * search)
{
    ENode *tmpnode;

    ECHECK_RETVAL (node != NULL, NULL);

    if (!search)
	return (node->parent);

    tmpnode = node->parent;

    while (tmpnode) {
	if (enode_basename_match (tmpnode, search))
	    return (tmpnode);

	tmpnode = tmpnode->parent;
    }

    return (NULL);
}



static gint
enode_child_search (ENodeTreeWalk * walker)
{
    gchar *search = walker->user_data1;
    ENode *curnode = walker->curnode;

    ECHECK_RETVAL (curnode != NULL, FALSE);
    ECHECK_RETVAL (search != NULL, FALSE);

    if (enode_basename_match (curnode, search)) {
	walker->user_data2 = curnode;
	return (FALSE);
    }

    return (TRUE);
}

/* Search for a child by name[.type] */
/* NOT synonymous with enode (); Enode() looks from <object> down. */
ENode *
enode_child (ENode * node, gchar * search)
{
    ENode *found = NULL;
    ENodeTreeWalk *walker;

    ECHECK_RETVAL (node != NULL, NULL);
    ECHECK_RETVAL (search != NULL, NULL);
    ECHECK_RETVAL (strlen (search) != 0, NULL);

    walker = enode_treewalk_new (node);
    walker->user_data1 = search;
    walker->user_data2 = NULL;
    enode_treewalk (walker, enode_child_search, NULL);
    found = walker->user_data2;
    enode_treewalk_free (walker);

    return (found);
}

ENode *
enode (gchar * path)
{
    gint path_length;
    ENode *objnode;
    ENode *node;
    ENode *refnode;

    refnode = enode_call_reference ();

    ECHECK_RETVAL (refnode != NULL, NULL);
    ECHECK_RETVAL (path != NULL, NULL);

    /* first step is to find what mode we're looking up in */
    path_length = strlen (path);
    ECHECK_RETVAL (path_length != 0, NULL);

    /* check for full lookup from root */
    if ((path_length >= 1) && (path[0] == '/'))
	return (enode_lookup_by_path (path));

    /* The remaining lookups find relative to 'object' if applicable */
    objnode = enode_reference_object (refnode);

    /* Check the current object node too */
    if (enode_basename_match (objnode, path))
	return (objnode);

    node = enode_child (objnode, path);
    return (node);
}

static gint
enode_child_rx_search (ENodeTreeWalk * walker)
{
    pcre *re = walker->user_data1;
    ENode *curnode = walker->curnode;
    EBuf *basename = enode_basename (curnode);
    int result;
    int retval = TRUE;

    ECHECK_RETVAL (curnode != NULL, FALSE);
    ECHECK_RETVAL (re != NULL, FALSE);
    ECHECK_RETVAL (basename != NULL, FALSE);

    result = pcre_exec (re, NULL, basename->str, basename->len, 0, 0, NULL, 0);

    if (result >= 0) {
	walker->user_data2 = curnode;
	retval = FALSE;
    }

    ebuf_free (basename);

    return (retval);
}

/* Search for a child given regex on type.name.  Returns first correct
 * match */
ENode *
enode_child_rx (ENode * node, gchar * regex)
{
    pcre *re;
    ENodeTreeWalk *walker;
    ENode *found = NULL;
    const char *error_str = NULL;
    int error_offset;

    ECHECK_RETVAL (node != NULL, NULL);
    ECHECK_RETVAL (regex != NULL, NULL);

    re = pcre_compile (regex, 0, &error_str, &error_offset, NULL);

    if (!re) {
	g_warning ("Compilation of regex '%s' failed at index %d: %s", regex,
		   error_offset, error_str);
	return (NULL);
    }

    walker = enode_treewalk_new (node);
    walker->user_data1 = re;
    walker->user_data2 = NULL;
    enode_treewalk (walker, enode_child_rx_search, NULL);
    found = walker->user_data2;
    enode_treewalk_free (walker);

    pcre_free (re);

    return (found);
}

ENode *
enode_rx (gchar * regex)
{
    ENode *objnode;
    ENode *node;
    ENode *refnode;

    refnode = enode_call_reference ();

    ECHECK_RETVAL (refnode != NULL, NULL);
    ECHECK_RETVAL (regex != NULL, NULL);

    objnode = enode_reference_object (refnode);

    node = enode_child_rx (objnode, regex);
    return (node);
}


static gint
enode_children_search (ENodeTreeWalk * treewalk)
{
    GSList *list = treewalk->user_data1;
    GSList *tail = treewalk->user_data2;
    gchar *search = treewalk->user_data3;

    if (enode_basename_match (treewalk->curnode, search)) {
	list = g_slist_append_tail (list, treewalk->curnode, &tail);
	treewalk->user_data1 = list;
	treewalk->user_data2 = tail;
    }

    return (TRUE);
}

GSList *
enode_children (ENode * node, gchar * search)
{
    ENodeTreeWalk *walker;
    GSList *children = NULL;

    ECHECK_RETVAL (node != NULL, NULL);

    /* Passing a NULL search argument, returns list of direct children */
    if (!search)
	return (g_slist_copy (node->children));

    walker = enode_treewalk_new (node);
    walker->user_data3 = search;
    enode_treewalk (walker, enode_children_search, NULL);
    children = walker->user_data1;
    enode_treewalk_free (walker);

    return (children);
}

GSList *
elist (gchar * search)
{
    ENode *objnode;
    GSList *list;
    ENode *refnode;

    refnode = enode_call_reference ();

    ECHECK_RETVAL (refnode != NULL, NULL);
    ECHECK_RETVAL (search != NULL, NULL);

    objnode = enode_reference_object (refnode);

    list = enode_children (objnode, search);
    return (list);
}


static gint
enode_children_rx_search (ENodeTreeWalk * treewalk)
{
    GSList *list = treewalk->user_data1;
    GSList *tail = treewalk->user_data2;
    pcre *re = treewalk->user_data3;
    EBuf *basename = enode_basename (treewalk->curnode);
    gint result;

    EDEBUG (("enode-search", "Checking regex against %s", basename->str));
    result = pcre_exec (re, NULL, basename->str, basename->len, 0, 0, NULL, 0);

    if (result >= 0) {
	EDEBUG (("enode-search", "Matched regex against %s", basename->str));
	list = g_slist_append_tail (list, treewalk->curnode, &tail);
	treewalk->user_data1 = list;
	treewalk->user_data2 = tail;
    }
    ebuf_free (basename);

    return (TRUE);
}

GSList *
enode_children_rx (ENode * node, gchar * regex)
{
    GSList *children = NULL;
    ENodeTreeWalk *walker;
    pcre *re;
    const char *error_str = NULL;
    int error_offset;

    ECHECK_RETVAL (node != NULL, NULL);
    ECHECK_RETVAL (regex != NULL, NULL);

    /* Optimize the match-all case */
    if (g_str_equal (regex, ".*")) {
	return (enode_child_list (node, TRUE));
    }

    EDEBUG (("enode-search", "Doing search with regex %s", regex));
    re = pcre_compile (regex, 0, &error_str, &error_offset, NULL);
    if (!re) {
	g_warning ("Compilation of regex '%s' failed at index %d: %s", regex,
		   error_offset, error_str);
	return (NULL);
    }

    walker = enode_treewalk_new (node);
    walker->user_data3 = re;
    enode_treewalk (walker, enode_children_rx_search, NULL);
    children = walker->user_data1;
    enode_treewalk_free (walker);

    pcre_free (re);

    return (children);
}

GSList *
elist_rx (gchar * regex)
{
    ENode *objnode;
    GSList *list;
    ENode *refnode;

    refnode = enode_call_reference ();

    ECHECK_RETVAL (refnode != NULL, NULL);
    ECHECK_RETVAL (regex != NULL, NULL);

    objnode = enode_reference_object (refnode);

    list = enode_children_rx (objnode, regex);
    return (list);
}


/* Search the tree for children who have an attribute that matches a
 * specific value. */
GSList *children_attrib (ENode * node, gchar * attrib, gchar * value);

static gint
enode_children_attrib_search (ENodeTreeWalk * treewalk)
{
    GSList *list = treewalk->user_data1;
    GSList *tail = treewalk->user_data2;
    gchar *attrib = treewalk->user_data3;
    EBuf *value = treewalk->user_data4;
    EBuf *realval;

    realval = enode_attrib (treewalk->curnode, attrib, NULL);

    if (ebuf_not_empty (realval) && (ebuf_equal_ebuf (realval, value))) {
	list = g_slist_append_tail (list, treewalk->curnode, &tail);
	treewalk->user_data1 = list;
	treewalk->user_data2 = tail;
    }

    return (TRUE);
}


/* Search the tree for children who have an attribute that * matches a *
 * specific value. */
GSList *
enode_children_attrib (ENode * node, gchar * attrib, EBuf * value)
{
    ENodeTreeWalk *walker;
    GSList *children = NULL;

    ECHECK_RETVAL (node != NULL, NULL);
    ECHECK_RETVAL (attrib != NULL, NULL);
    ECHECK_RETVAL (value != NULL, NULL);

    walker = enode_treewalk_new (node);
    walker->user_data3 = attrib;
    walker->user_data4 = value;
    enode_treewalk (walker, enode_children_attrib_search, NULL);
    children = walker->user_data1;
    enode_treewalk_free (walker);

    return (children);
}


static gint
enode_children_attrib_rx_search (ENodeTreeWalk * treewalk)
{
    GSList *list = treewalk->user_data1;
    GSList *tail = treewalk->user_data2;
    gchar *attrib = treewalk->user_data3;
    pcre *re = treewalk->user_data4;
    EBuf *realval;

    realval = enode_attrib (treewalk->curnode, attrib, NULL);

    if (ebuf_not_empty (realval) &&
	(pcre_exec (re, NULL, realval->str, realval->len, 0, 0, NULL, 0) >= 0)) {
	list = g_slist_append_tail (list, treewalk->curnode, &tail);
	treewalk->user_data1 = list;
	treewalk->user_data2 = tail;
    }

    return (TRUE);
}

/* Search the tree for children who have an attribute that * matches a regex. */
GSList *
enode_children_attrib_rx (ENode * node, gchar * attrib, gchar * regex)
{
    ENodeTreeWalk *walker;
    GSList *children = NULL;
    const char *error_str = NULL;
    int error_offset;
    pcre *re;

    ECHECK_RETVAL (node != NULL, NULL);
    ECHECK_RETVAL (attrib != NULL, NULL);
    ECHECK_RETVAL (regex != NULL, NULL);

    re = pcre_compile (regex, 0, &error_str, &error_offset, NULL);

    if (!re) {
	g_warning ("Compilation of regex '%s' failed at index %d: %s", regex,
		   error_offset, error_str);
	return (NULL);
    }

    walker = enode_treewalk_new (node);
    walker->user_data3 = attrib;
    walker->user_data4 = re;
    enode_treewalk (walker, enode_children_attrib_rx_search, NULL);
    children = walker->user_data1;
    enode_treewalk_free (walker);

    return (children);
}

