#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/i2c.h>
#include <linux/types.h>
#include <linux/videodev.h>
#include <linux/init.h>

#include "tuner.h"

/* Addresses to scan */
static unsigned short normal_i2c[] = {I2C_CLIENT_END};
static unsigned short normal_i2c_range[] = {0x61,0x67,I2C_CLIENT_END};
static unsigned short probe[2]        = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short probe_range[2]  = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short ignore[2]       = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short ignore_range[2] = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short force[2]        = { I2C_CLIENT_END, I2C_CLIENT_END };
static struct i2c_client_address_data addr_data = {
	normal_i2c, normal_i2c_range, 
	probe, probe_range, 
	ignore, ignore_range, 
	force
};

static int debug =  0; /* insmod parameter */
static int type  = -1; /* insmod parameter */

static int addr  =  0;
static int this_adap;

#define dprintk     if (debug) printk

MODULE_PARM(debug,"i");
MODULE_PARM(type,"i");
MODULE_PARM(addr,"i");

struct tuner
{
	int type;            /* chip type */
	int freq;            /* keep track of the current settings */
	int std;

	int radio;
	int mode;            /* PAL(0)/SECAM(1) mode (PHILIPS_SECAM only) */
};

static struct i2c_driver driver;
static struct i2c_client client_template;

/* ---------------------------------------------------------------------- */

struct tunertype 
{
	char *name;
	unsigned char Vendor;
	unsigned char Type;
  
	/* all frequencies in units of Hz */
	u32 min;     /* minimum frequency */  
	u32 max;     /* maximum frequency */
	u32 res;     /* frequency resolution in Hz */
u32 step;    /* stepsize in Hz (units of setfrequency ioctl) */
	u32 IFPCoff; /* frequency offset */
	u32 thresh1; /* frequency Range for UHF,VHF-L, VHF_H */   
	u32 thresh2;  
	u8 VHF_L;   
	u8 VHF_H;
	u8 UHF;
	u8 config; 
	u8 mode; /* mode change value (tested PHILIPS_SECAM only) */
	         /* 0x01 -> ??? no change ??? */
	         /* 0x02 -> PAL BDGHI / SECAM L */
	         /* 0x04 -> ??? PAL others / SECAM others ??? */
	int capability;
};

static struct tunertype tuners[] = {
        {"Temic PAL", TEMIC, PAL, 
	          55000000, 802000000, 62500, 62500, 38937500, 
	         140250000, 463250000, 0x02, 0x04, 0x01, 0x8e},
	{"Philips PAL_I", Philips, PAL_I,
	          55000000, 802000000, 62500, 62500, 38937500,
	         140250000, 463250000, 0xa0, 0x90, 0x30, 0x8e},
	{"Philips NTSC", Philips, NTSC,
	          55000000, 802000000, 62500, 62500, 45750000,
                 157250000, 451250000, 0xA0, 0x90, 0x30, 0x8e},
	{"Philips SECAM", Philips, SECAM,
	          55000000, 802000000, 62500, 62500, 38937500,
                 168250000, 447250000, 0xA7, 0x97, 0x37, 0x8e, 0x02},

	{"NoTuner", NoTuner, NOTUNER,
	         0, 0, 0, 0, 0,
                 0, 0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{"Philips PAL", Philips, PAL,
                 55000000, 802000000, 62500, 62500, 38937500,
	        168250000, 447250000, 0xA0, 0x90, 0x30, 0x8e},
	{"Temic NTSC", TEMIC, NTSC,
                 55000000, 802000000, 62500, 62500, 45750000,
	        157250000, 463250000, 0x02, 0x04, 0x01, 0x8e},
	{"TEMIC PAL_I", TEMIC, PAL_I,
                 55000000, 802000000, 62500, 62500, 38937500,
	        170000000, 450000000, 0x02, 0x04, 0x01, 0x8e},

	{"Temic 4036 FY5 NTSC", TEMIC, NTSC,
                 55000000, 802000000, 62500, 62500, 45750000,
	        157250000, 463250000, 0xa0, 0x90, 0x30, 0x8e},
	{"Alps HSBH1",TEMIC,NTSC,
                 55000000, 802000000, 62500, 62500, 45750000,
	        137250000, 385250000, 0x01, 0x02, 0x08, 0x8e},
	{"Alps TSBE1",TEMIC,PAL,
                 55000000, 802000000, 62500, 62500, 45750000,
	        137250000, 385250000, 0x01, 0x02, 0x08, 0x8e},
	{"Alps TSBB5",Alps,PAL_I,
                 55000000, 802000000, 62500, 62500, 39500000,
		133250000, 351250000, 0x01, 0x02, 0x08, 0x8e},

	{"Alps TSBE5",Alps,PAL,
                 55000000, 802000000, 62500, 62500, 38875000,
		133250000, 351250000, 0x01, 0x02, 0x08, 0x8e},
	{"Alps TSBC5",Alps,PAL,
                 55000000, 802000000, 62500, 62500, 38000000,
		133250000, 351250000, 0x01, 0x02, 0x08, 0x8e},
	{ "Temic 4006FH5", TEMIC, PAL_I,
                 55000000, 802000000, 62500, 62500, 38937500,
		170000000, 450000000, 0xa0, 0x90, 0x30, 0x8e},
  	{ "Alps TSCH6",Alps,NTSC,
                 55000000, 802000000, 62500, 62500, 45750000,
		137250000, 385250000, 0x14, 0x12, 0x11, 0x8e},

  	{ "Temic PAL_DK (4016 FY5)",TEMIC,PAL,
	         55000000, 802000000, 62500, 62500, 38937500,
  	        168250000, 456250000, 0xa0, 0x90, 0x30, 0x8e},
  	{ "Philips NTSC_M (MK2)",Philips,NTSC,
	         55000000, 802000000, 62500, 62500, 45750000,
  	        160000000, 454000000, 0xa0,0x90,0x30,0x8e},
        { "Temic PAL_I (4066 FY5)", TEMIC, PAL_I,
	         55000000, 802000000, 62500, 62500, 38937500,
                169000000, 454000000, 0xa0, 0x90, 0x30, 0x8e},
        { "Temic PAL* auto (4006 FN5)", TEMIC, PAL,
	         55000000, 802000000, 62500, 62500, 38937500,
                169000000, 454000000, 0xa0, 0x90, 0x30, 0x8e},

        { "Temic PAL (4009 FR5)", TEMIC, PAL,
	         55000000, 802000000, 62500, 62500, 38937500,
                141000000, 464000000, 0xa0, 0x90, 0x30, 0x8e},
        { "Temic NTSC (4039 FR5)", TEMIC, NTSC,
	         55000000, 802000000, 62500, 62500, 45750000,
                158000000, 453000000, 0xa0, 0x90, 0x30, 0x8e},
        { "Temic PAL/SECAM multi (4046 FM5)", TEMIC, PAL,
	         55000000, 802000000, 62500, 62500, 38937500,
                169000000, 454000000, 0xa0, 0x90, 0x30, 0x8e},
        { "Philips PAL_DK", Philips, PAL,
	         55000000, 802000000, 62500, 62500, 38937500,
	        170000000, 450000000, 0xa0, 0x90, 0x30, 0x8e},

	{ "Philips PAL/SECAM multi (FQ1216ME)", Philips, PAL,
	         55000000, 802000000, 62500, 62500, 38937500,
	        170000000, 450000000, 0xa0,0x90,0x30,0x8e},
	{ "LG PAL_I+FM (TAPC-I001D)", LGINNOTEK, PAL_I,
	         55000000, 802000000, 62500, 62500, 38937500,
	        170000000, 450000000, 0xa0, 0x90, 0x30, 0x8e},
	{ "LG PAL_I (TAPC-I701D)", LGINNOTEK, PAL_I,
	         55000000, 802000000, 62500, 62500, 38937500,
	        170000000, 450000000, 0xa0, 0x90, 0x30, 0x8e},
	{ "LG NTSC+FM (TPI8NSR01F)", LGINNOTEK, NTSC,
	         55000000, 802000000, 62500, 62500, 45750000,
	        210000000, 497000000, 0xa0, 0x90, 0x30, 0x8e},

	{ "LG PAL_BG+FM (TPI8PSB01D)", LGINNOTEK, PAL,
                 55000000, 802000000, 62500, 62500, 38937500,
	        170000000, 450000000, 0xa0, 0x90, 0x30, 0x8e},
	{ "LG PAL_BG (TPI8PSB11D)", LGINNOTEK, PAL,
	         55000000, 802000000, 62500, 62500, 38937500,
	        170000000, 450000000, 0xa0, 0x90, 0x30, 0x8e},
	{ "Temic PAL* auto + FM (4009 FN5)", TEMIC, PAL,
	         55000000, 802000000, 62500, 62500, 38937500,
	        141000000, 464000000, 0xa0, 0x90, 0x30, 0x8e},
	{ "SHARP NTSC_JP (2U5JF5540)", SHARP, NTSC, /* 940=16*58.75 NTSC@Japan */
	         55000000, 802000000, 62500, 62500, 58750000,
	        137250000, 317250000, 0x01, 0x02, 0x08, 0x8e},

	{ "Samsung PAL TCPM9091PD27", Samsung, PAL,  /* from sourceforge v3tv */
	         55000000, 802000000, 62500, 62500, 38937500,
                169000000, 464000000, 0xA0, 0x90, 0x30, 0x8e},
	{ "MT2032 universal", Microtune,PAL|NTSC,
	         55000000, 802000000, 62500, 62500, 0,
                 0, 0, 0, 0, 0, 0},
	{ "Temic PAL_BG (4106 FH5)", TEMIC, PAL,
	         55000000, 802000000, 62500, 62500, 38937500,
                141000000, 464000000, 0xa0, 0x90, 0x30, 0x8e},
	{ "Temic PAL_DK/SECAM_L (4012 FY5)", TEMIC, PAL,
	         55000000, 802000000, 62500, 62500, 38937500,
                140250000, 463250000, 0x02, 0x04, 0x01, 0x8e},

	{ "Temic NTSC (4136 FY5)", TEMIC, NTSC,
	         55000000, 802000000, 62500, 62500, 45750000,
                158000000, 453000000, 0xa0, 0x90, 0x30, 0x8e},
        { "LG PAL (newer TAPC series)", LGINNOTEK, PAL,
	         55000000, 802000000, 62500, 62500, 38937500,
                170000000, 450000000, 0x01, 0x02, 0x08, 0x8e},
	

        {"SP5659", MITEL, DVBS,
	          100000000UL, 2700000000UL, 125000, 1, 479500000,
	         0xffffffffUL, 0xffffffffUL, 0x30, 0x30, 0x30, 0x95},
        {"SPXXXX", MITEL, DVBC,
	          40000000UL, 870000000UL, 62500, 1, 36125000,
	         174000000UL, 454000000UL, 0xa1, 0x92, 0x34, 0x8e},
        {"TSA5059", Philips, DVBS, 
	          100000000UL, 2700000000UL, 125000, 1, 0,
	         0xffffffffUL, 0xffffffffUL, 0x30, 0x30, 0x30, 0x95},
        {"TSA5522", Philips, DVBS, 
 	         100000000UL, 2700000000UL, 125000, 1, 0, 
	         0xffffffffUL, 0xffffffffUL, 0x00, 0x00, 0x00, 0x8e},
	{"TSA5060", Philips, DVBT, 
	          64000000UL,  1300000000UL, 166666, 1, 36000000,
	         0xffffffffUL, 0xffffffffUL, 0x00, 0x00, 0x00, 0x88},
	{"SP5668", MITEL, DVBT,   /*  ??????????????    */
	          64000000UL,  1300000000UL, 166666, 1, 36000000,
	         0xffffffffUL, 0xffffffffUL, 0x00, 0x00, 0x00, 0x88},
        {"SP5659C", MITEL, DVBC,
                  40000000UL, 870000000UL, 62500, 1, 36125000,
                 174000000UL, 470000000UL, 0xa1, 0x84, 0x81, 0x85},
	/*
        {"SPXXXX", MITEL, DVBC,
	          40000000UL, 870000000UL, 62500, 1, 38900000,
 	         174000000UL, 454000000UL, 0xa1, 0x92, 0x34, 0x8e},
	*/
};
#define TUNERS (sizeof(tuners)/sizeof(struct tunertype))

/* ---------------------------------------------------------------------- */

static int tuner_getstatus(struct i2c_client *c)
{
	unsigned char byte;

	if (1 != i2c_master_recv(c,&byte,1))
		return 0;
	return byte;
}

#define TUNER_POR       0x80
#define TUNER_FL        0x40
#define TUNER_MODE      0x38
#define TUNER_AFC       0x07

#define TUNER_STEREO    0x10 /* radio mode */
#define TUNER_SIGNAL    0x07 /* radio mode */

static int tuner_signal(struct i2c_client *c)
{
	return (tuner_getstatus(c) & TUNER_SIGNAL)<<13;
}

static int tuner_stereo(struct i2c_client *c)
{
	return (tuner_getstatus (c) & TUNER_STEREO);
}


static int tuner_islocked (struct i2c_client *c)
{
        return (tuner_getstatus (c) & TUNER_FL);
}

static int tuner_afcstatus (struct i2c_client *c)
{
        return (tuner_getstatus (c) & TUNER_AFC) - 2;
}

#if 0 /* unused */
static int tuner_mode (struct i2c_client *c)
{
        return (tuner_getstatus (c) & TUNER_MODE) >> 3;
}
#endif

static void set_tv_freq(struct i2c_client *c, u32 freq)
{
	u8 config;
        u32 div2;
	u16 div;
	struct tunertype *tun;
	struct tuner *t = c->data;
        unsigned char buffer[4];
	int rc;

	if (t->type == -1) {
		printk("tuner: tuner type not set\n");
		return;
	}

	tun=&tuners[t->type];

        freq*=tun->step; // turn into Hz

	if (freq < tun->thresh1) 
		config = tun->VHF_L;
	else if (freq < tun->thresh2) 
		config = tun->VHF_H;
	else
		config = tun->UHF;

#if 1   // Fix colorstandard mode change
	if (t->type == TUNER_PHILIPS_SECAM && t->mode)
		config |= tun->mode;
	else
		config &= ~tun->mode;

#else
		config &= ~tun->mode;
#endif

	div2=freq + tun->IFPCoff;
        div2/=tun->res;
        div=(u16)div2;

    /*
     * Philips FI1216MK2 remark from specification :
     * for channel selection involving band switching, and to ensure
     * smooth tuning to the desired channel without causing
     * unnecessary charge pump action, it is recommended to consider
     * the difference between wanted channel frequency and the
     * current channel frequency.  Unnecessary charge pump action
     * will result in very low tuning voltage which may drive the
     * oscillator to extreme conditions.
     */
    /*
     * Progfou: specification says to send config data before
     * frequency in case (wanted frequency < current frequency).
     */

	if (t->type == TUNER_PHILIPS_SECAM && freq < t->freq) {
		buffer[0] = tun->config;
		buffer[1] = config;
		buffer[2] = (div>>8) & 0x7f;
		buffer[3] = div      & 0xff;
	} else {
		buffer[0] = (div>>8) & 0x7f;
		buffer[1] = div      & 0xff;
		buffer[2] = tun->config;
		buffer[3] = config;
	}
dprintk ("buffer == 0x%02x 0x%02x 0x%02x 0x%02x\n",
	buffer[0], buffer[1], buffer[2], buffer[3]);
	if (4 != (rc = i2c_master_send(c,buffer,4)))
                printk("tuner: i2c i/o error: rc == %d (should be 4)\n",rc);

}

static void set_radio_freq(struct i2c_client *c, int freq)
{
	u8 config;
	u16 div;
	struct tunertype *tun;
	struct tuner *t = (struct tuner*)c->data;
        unsigned char buffer[4];
	int rc;

	if (t->type == -1) {
		printk("tuner: tuner type not set\n");
		return;
	}

	tun=&tuners[t->type];
	config = 0xa4 /* 0xa5 */; /* bit 0 is AFC (set) vs. RF-Signal (clear) */
	div=freq + (int)(16*10.7);
  	div&=0x7fff;

        buffer[0] = (div>>8) & 0x7f;
        buffer[1] = div      & 0xff;
        buffer[2] = tun->config;
        buffer[3] = config;
        if (4 != (rc = i2c_master_send(c,buffer,4)))
                printk("tuner: i2c i/o error: rc == %d (should be 4)\n",rc);

	if (debug) {
		current->state   = TASK_INTERRUPTIBLE;
		schedule_timeout(HZ/10);
		
		if (tuner_islocked (c))
			printk ("tuner: PLL locked\n");
		else
			printk ("tuner: PLL not locked\n");
		
		if (config & 1) {
			printk ("tuner: AFC: %d\n", tuner_afcstatus(c));
		} else {
			printk ("tuner: Signal: %d\n", tuner_signal(c));
		}
	}
}
/* ---------------------------------------------------------------------- */


static int tuner_attach(struct i2c_adapter *adap, int addr,
			unsigned short flags, int kind)
{
	struct tuner *t;
	struct i2c_client *client;

	if (this_adap > 0)
		return -1;
	this_adap++;
	
        client_template.adapter = adap;
        client_template.addr = addr;

        printk("tuner: chip found @ 0x%x\n",addr);

        if (NULL == (client = kmalloc(sizeof(struct i2c_client), GFP_KERNEL)))
                return -ENOMEM;
        memcpy(client,&client_template,sizeof(struct i2c_client));
        client->data = t = kmalloc(sizeof(struct tuner),GFP_KERNEL);
        if (NULL == t) {
                kfree(client);
                return -ENOMEM;
        }
        memset(t,0,sizeof(struct tuner));
	if (type >= 0 && type < TUNERS) {
		t->type = type;
		strncpy(client->name, tuners[t->type].name, sizeof(client->name));
	} else {
		t->type = -1;
	}
        i2c_attach_client(client);
	MOD_INC_USE_COUNT;

	return 0;
}

static int tuner_probe(struct i2c_adapter *adap)
{
	if (0 != addr) {
		normal_i2c_range[0] = addr;
		normal_i2c_range[1] = addr;
	}
	this_adap = 0;
	//if (adap->id == (I2C_ALGO_BIT | I2C_HW_B_BT848))
        if (1) 
		return i2c_probe(adap, &addr_data, tuner_attach);
        return 0;
}

static int tuner_detach(struct i2c_client *client)
{
	struct tuner *t = (struct tuner*)client->data;

	i2c_detach_client(client);
	kfree(t);
	kfree(client);
	MOD_DEC_USE_COUNT;
	return 0;
}

static int
tuner_command(struct i2c_client *client, unsigned int cmd, void *arg)
{
	struct tuner *t = (struct tuner*)client->data;
	unsigned int *iarg = (int*)arg;
#if 1
        u16 *sarg = (u16*)arg;
#endif

	switch (cmd) {

	/* --- configuration --- */
        case TUNER_SET_TYPE:
		if (t->type != -1)
			return 0;
		if (*iarg < 0 || *iarg >= TUNERS)
			return 0;
                t->type = *iarg;
                dprintk("tuner: type set to %d (%s)\n",
			t->type,tuners[t->type].name);
		strncpy(client->name, tuners[t->type].name, sizeof(client->name));
                break;
                
	/* --- v4l ioctls --- */
	/* take care: bttv does userspace copying, we'll get a
	   kernel pointer here... */
	case VIDIOCSCHAN:
	{
		struct video_channel *vc = arg;
		
		if (t->type == TUNER_PHILIPS_SECAM) {
			t->mode = (vc->norm == VIDEO_MODE_SECAM) ? 1 : 0;
			set_tv_freq(client,t->freq);
		}
		return 0;
	}
	case VIDIOCSFREQ:
	{
		unsigned long *v = arg;

		t->freq = *v;
		if (t->radio) {
			dprintk("tuner: radio freq set to %d.%02d\n",
				(*iarg)/16,(*iarg)%16*100/16);
			set_radio_freq(client,t->freq);
		} else {
			dprintk("tuner: tv freq set to %d.%02d\n",
				(*iarg)/16,(*iarg)%16*100/16);
			set_tv_freq(client,t->freq);
		}
		return 0;
	}
#if 1
	/* --- old, obsolete interface --- */
        case TUNER_SET_TVFREQ:
                dprintk("tuner: tv freq set to %d\n",(*iarg));
                set_tv_freq(client, *iarg);
                t->radio = 0;
                t->freq = *iarg;
                break;
	    
        case TUNER_SET_RADIOFREQ:
                dprintk("tuner: radio freq set to %d.%02d\n",
                        (*iarg)/16,(*iarg)%16*100/16);
                set_radio_freq(client,*iarg);
                t->radio = 1;
                t->freq = *iarg;
                break;
        case TUNER_SET_MODE:
                if (t->type != TUNER_PHILIPS_SECAM) {
                        dprintk("tuner: trying to change mode for other than TUNER_PHILIPS_SECAM\n");
                } else {
			int mode=(*sarg==VIDEO_MODE_SECAM)?1:0;
                        dprintk("tuner: mode set to %d\n", *sarg);
                        t->mode = mode;
                        set_tv_freq(client,t->freq);
                }
                break;
#endif
        default:
                return -EINVAL;
	}
	return 0;
}

/* ----------------------------------------------------------------------- */

void inc_use (struct i2c_client *client)
{
#ifdef MODULE
        MOD_INC_USE_COUNT;
#endif
}

void dec_use (struct i2c_client *client)
{
#ifdef MODULE
        MOD_DEC_USE_COUNT;
#endif
}
static struct i2c_driver driver = {
        "i2c TV tuner driver",
        I2C_DRIVERID_TUNER,
        I2C_DF_NOTIFY,
        tuner_probe,
        tuner_detach,
        tuner_command,
        inc_use,
        dec_use,
};

static struct i2c_client client_template =
{
	name:    "i2c tv tuner chip",          /* name       */
        id:      I2C_DRIVERID_TUNER,           /* ID         */
        flags:   0,
	addr:    0, /* interpret as 7Bit-Adr */
        adapter: NULL,
        driver:  &driver
};

EXPORT_NO_SYMBOLS;

#ifdef MODULE
int init_module(void)
#else
int i2c_tuner_init(void)
#endif
{
	i2c_add_driver(&driver);
	return 0;
}

#ifdef MODULE
void cleanup_module(void)
{
	i2c_del_driver(&driver);
}
#endif

/*
 * Overrides for Emacs so that we follow Linus's tabbing style.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-basic-offset: 8
 * End:
 */
