/*
 * dsyslog - a dumb syslog (e.g. syslog for people who have a clue)
 * Copyright (c) 2008 William Pitcock <nenolod@sacredspiral.co.uk>
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice is present in all copies.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "dsyslog.h"

struct dsyslog_source_constructor_map {
	gchar *type;
	dsyslog_source_constructor_t constructor;
};

static GList *source_list_ = NULL;
static GList *source_constructors = NULL;

void
dsyslog_source_register(gchar *type, dsyslog_source_constructor_t constructor)
{
	struct dsyslog_source_constructor_map *source;

	_ENTER;

	source = g_slice_new(struct dsyslog_source_constructor_map);

	source->type = type;
	source->constructor = constructor;

	source_constructors = g_list_prepend(source_constructors, source);

	_LEAVE;
}

void
dsyslog_source_unregister(gchar *type)
{
	GList *n;

	_ENTER;

	for (n = source_constructors; n != NULL; n = g_list_next(n))
	{
		struct dsyslog_source_constructor_map *cst = (struct dsyslog_source_constructor_map *) n->data;

		if (!g_ascii_strcasecmp(cst->type, type))
		{
			source_constructors = g_list_remove_all(source_constructors, cst);
			g_slice_free(struct dsyslog_source_constructor_map, cst);

			_LEAVE;
		}
	}

	_LEAVE;
}

void
dsyslog_source_setup(dsyslog_source_t *src)
{
	GList *n;

	_ENTER;

	for (n = source_constructors; n != NULL; n = g_list_next(n))
	{
		struct dsyslog_source_constructor_map *cst = (struct dsyslog_source_constructor_map *) n->data;

		if (!g_ascii_strcasecmp(cst->type, src->type))
		{
			cst->constructor(src);
			source_list_ = g_list_append(source_list_, src);

			_LEAVE;
		}
	}

	_LEAVE;
}

void
dsyslog_source_remove(dsyslog_source_t *src)
{
	_ENTER;

	source_list_ = g_list_remove_all(source_list_, src);

	if (src->destructor)
		src->destructor(src);

	g_free(src->type);
	g_free(src->host);
	g_free(src->path);

	g_slice_free(dsyslog_source_t, src);

	_LEAVE;
}

void
dsyslog_source_clear(void)
{
	GList *n, *n2;

	_ENTER;

	for (n = source_list_, n2 = g_list_next(source_list_); n != NULL; n = n2, n2 = g_list_next(n2)) {
		dsyslog_source_t *src = (dsyslog_source_t *) n->data;

		dsyslog_source_remove(src);
	}

	source_list_ = NULL;

	_LEAVE;
}
