/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

// ----------------------------------------------------------------------------------
//
// bwfilter, a simple filter that strips out Cr and Cb colour planes,
//           (sets Cr & Cb to 128) only leaving Y plane (black & white)
//
// 09-08-2002 - Jarl van Katwijk
//
// ----------------------------------------------------------------------------------


#include "../config.h"
#include "../src/drip.h"
#include "../encoder/plugin-loader.hh"
#include <gmodule.h>
// Custom includes:
#include <string.h>
#include "../encoder/fast_memcpy.hh"

#undef DEBUG

static gboolean init_done = FALSE;                 // Plugin initialised flag
static gint width,height;                          // Input Width\Height sizes
static gint xoff,yoff,xlen,ylen;                   // Offset and lenght values
static gdouble framerate;                           // Framerate (25.00, 29.97, etc)
static gdouble value1,value2,value3,value4,value5;  // Plugin defined values - these are loaded\saved by Drip
static guint8 *src_planes[3];                      // Input data planes (I420 colour space)
static guint8 *dst_planes[3];                      // Output data planes


extern "C" {



/* Return type (SPU,AUDIO,VIDEO) of filter */
G_MODULE_EXPORT module_type_e bwfilter_type(void) {
    return VIDEO;
}


/* Return phase (PRE,NORMAL,POST) of filter */
G_MODULE_EXPORT module_phase_e bwfilter_phase(void) {
    return POST;
}





/* Function which implements a GtkPlug to let user configure this plugin */ 
G_MODULE_EXPORT GtkWidget* bwfilter_gui(void) {
    static GtkWidget *gui = NULL;
    #ifdef DEBUG
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: bwfilter gui");
    #endif
    gui = gtk_label_new("\n\n\nbwfilter\n\n\nThis filter strips out the colour, leaving black&white\n");

    /* Clean & Exit */
    return gui;
}


/* This function is called for every (audio\video) frame.
   Here the actual filtering is done. */
G_MODULE_EXPORT guint8** bwfilter_apply(guint8** src, glong size, gulong SCR, gulong PTS) {
    static guint len = (guint)(width*height/4);

    #ifdef DEBUG
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: bwfilter apply (PTS=%lu,SCR=%lu)",PTS,SCR);
    #endif

    /* Clean Cr and Cb planes */
    memset(src[1], 128, len);  /* Clear Cr */
    memset(src[2], 128, len);  /* Clear Cb */

    /* Clean & Exit */
    return src; 
}


G_MODULE_EXPORT void bwfilter_init(gint w, gint h,
                                     gint xo, gint yo, gint xl, gint yl,
                                     gdouble fr) {
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: bwfilter init");
    /* Copy values */
    width = w; height = h;
    xoff = xo; yoff = yo; xlen = xl; ylen = yl;
    framerate = fr;

    /* Set init flag */
    init_done = TRUE;
    /* Clean & Exit */
    return;
}


/* Called when parameters need tobe reset to default, auto called when plugin used for 1st time */
G_MODULE_EXPORT void bwfilter_default(void) {                                             /* DEFINE! */
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: bwfilter default");
    /* Set value's to default */
    value1 = 0;
    value2 = 0;
    value3 = 0;
    value4 = 0;
    value5 = 0;
    /* Clean & Exit */
    return;
}


/* Called after encoding has ended. Variable can be reset etc. */
G_MODULE_EXPORT void bwfilter_destruct(void) {  
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: bwfilter destruct");
    /* Clean & Exit */
    return;
}






/* Return values of plugin (value1,value2,... value5) */
G_MODULE_EXPORT void bwfilter_values(gdouble* values) {                          /* DEFINE! */
    values[0] = value1;
    values[1] = value2;
    values[2] = value3;
    values[3] = value4;
    values[4] = value5;
    return;
}

/* Set plugins values */
G_MODULE_EXPORT void bwfilter_values_function(gdouble *v) { /* DEFINE */
    value1 = v[0];
    value2 = v[1];
    value3 = v[2];
    value4 = v[3];
    value5 = v[4];
    return;
}

/* g_module_check_init is automatically executed upon loading */
G_MODULE_EXPORT const gchar* g_module_check_init(GModule *module) {
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: bwfilter selftest passed");
    return NULL;
}

} //extern "C"


