/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * subpicture.c - DVD subpicture handling - Jarl van Katwijk
 *
 * HOWTO use this code:
 * - call subpicture_init 
 * - call subpicture_render with every mpeg2 package containing subpicture data
 * - call subpicture_overlay with every video frame that needs subpictures
 * this should apply the right subpicture frames for the video ones. 
 * Only 24 bits are supported now, edit overlay.c for other.
 *
 * TODO: add 8/16/32 bits rendering.
 * TODO: Implement PTS handling
 *
 *
 *
 *
 *
 * Mostly based on hard work by:
 *
 * Copyright (C) 2000   Samuel Hocevar <sam@via.ecp.fr>
 *                       and Michel Lespinasse <walken@via.ecp.fr>
 *
 * Lots of rearranging by:
 *	Aaron Holtzman <aholtzma@ess.engr.uvic.ca>
 *	Thomas Mirlacher <dent@cosy.sbg.ac.at>
 *		implemented reassembling
 *		cleaner implementation of SPU are saving
 *		overlaying (proof of concept for now)
 *		... and yes, it works now with oms
 *		added tranparency (provided by the SPU hdr) 
 *		changed structures for easy porting to MGAs DVD mode
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *                                                     
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <glib.h>
#include <time.h>
#include <signal.h>
#include <setjmp.h>
#include "overlay.h"
#include "spu.h"
#include <math.h>

//#define SPU_CRT // SPU packet CRC checking for doubles
//#define DEBUG

#define DISPLAY_INIT
#define REASSEMBLY_START        0
#define REASSEMBLY_MID          1
#define REASSEMBLY_UNNEEDED     2

#define CMD_SPU_MENU            0x00
#define CMD_SPU_SHOW            0x01
#define CMD_SPU_HIDE            0x02
#define CMD_SPU_SET_PALETTE     0x03
#define CMD_SPU_SET_ALPHA       0x04
#define CMD_SPU_SET_SIZE        0x05
#define CMD_SPU_SET_PXD_OFFSET  0x06
#define CMD_SPU_EOF             0xff

typedef guint uint_32;
typedef guchar uint_8;
struct reassembly_s {
        guint8 *buf;
        guint8 *buf_ptr;        // actual pointer to still empty buffer
        guint buf_len;
        guint cmd_offset;
} reassembly;

gchar* SPU_LD = NULL;
static struct spu_img_struct spu;
static guchar *buffer = NULL;
static guint data_size;
static guint field;             // which field we are currently decoding
static guint reassembly_flag = REASSEMBLY_START;
static gint fd; /* input subpicture filedescriptor */
guint spu_clut[16];
static guchar *data_buffer = NULL;
static gulong data_buffer_size = 0;
#ifdef SPU_CRT
static gulong CRClist[65535];
static guint CRClist_counter = 0;
#endif
static glong frames_to_show_current_overlay = 0;
static glong frames_to_show_current_overlay_value;
static gboolean show_subpicture = FALSE;
static gint subpicture_stream = -1;
static gint subpicture_x = -1;
static gint subpicture_y = -1;
static gdouble framerate;
static gdouble transparency[256];


/* - */
static inline guint spu_get_bits(guint bits, struct spu_img_struct *spu) {
    static guint data;
    static guint bits_left;
    guint ret = 0;

    if (!bits) {	// for realignment to next byte
        bits_left = 0;
    }
    while (bits) {
        if (bits > bits_left) {
            ret |= data << (bits - bits_left);
            bits -= bits_left;
            data = reassembly.buf[spu->offset[field]++];
            bits_left = 8;
        } else {
            bits_left -= bits;
            ret |= data >> (bits_left);
            data &= (1 << bits_left) - 1;
            bits = 0;
        }
    }

    return ret;	
}


/* Expand\Draw a colourId+len to SPU render buffer */
static inline void spu_put_pixel(struct spu_img_struct *spu, guint len, guint8 colorid,gboolean back) {
    gint offset = spu->_x + spu->_y * spu->width;
    guint8 *spu_data_ptr = &spu->data[offset];
    static guint pixel;

    spu->_x += len;
    if (back==TRUE) {
        /* Use colour from CLUT and transparency to build a SPU pixel */
        pixel = (spu->trans[colorid]<<4) + spu->clut_index[colorid];
    } else {
        /* Draw a 100% transparent pixel so nothing is drawn */
        pixel = 0x00;
    }

    /* Check for overflow */
    if (len+offset < 2000000) {
        /* Set pixels */
        memset(spu_data_ptr, pixel, len);
    } else {
        g_log(SPU_LD,G_LOG_LEVEL_ERROR,"SPU: Not drawing pixels outsite frame (x=%i,y=%i,len=%i)",spu->_x,spu->_y,len);
    }

    return;
}


/* SPU position values to next line */
static inline gint spu_next_line(struct spu_img_struct *spu) {
    spu_get_bits (0,spu); // byte align rle data
    spu->_x = 0;
    spu->_y++;
    field = (field+1) & 0x01; // Toggle fields
    if (spu->_y >= spu->height) {
        return -1;
    }
    return 0;
}


static inline struct reassembly_s *spu_reassembly(guint8 *pkt_data, guint pkt_len) {
    if (reassembly_flag == REASSEMBLY_UNNEEDED)
        reassembly_flag = REASSEMBLY_START;

    if (reassembly_flag == REASSEMBLY_START) {
        reassembly.buf_len = (((guint)pkt_data[0])<<8) | pkt_data[1];
        reassembly.cmd_offset = (((guint)pkt_data[2])<<8) | pkt_data[3];
        if (pkt_len >= reassembly.buf_len) {
            reassembly.buf = pkt_data;
            reassembly_flag = REASSEMBLY_UNNEEDED;
            return &reassembly;
        } else {
            reassembly.buf_ptr = reassembly.buf;
            memcpy(reassembly.buf_ptr, pkt_data, pkt_len);
            reassembly.buf_ptr += pkt_len;
            reassembly_flag = REASSEMBLY_MID;
        }
    } else {
        if ((reassembly.buf_ptr+pkt_len) > (reassembly.buf+reassembly.buf_len))
            pkt_len = reassembly.buf_len-(reassembly.buf_ptr-reassembly.buf);
        memcpy(reassembly.buf_ptr, pkt_data, pkt_len);
        reassembly.buf_ptr += pkt_len;
        if (reassembly.buf_ptr >= (reassembly.buf+reassembly.buf_len)) {
            reassembly_flag = REASSEMBLY_START;
            return &reassembly;
        }
    }
    return NULL;	
}


/* The time is given as an offset from the presentation time stamp
   and it is measured in number of fields. If we play a PAL movie
   the time for each field is 1/(2*25.00) seconds. */
gint spu_parse_header(struct spu_img_struct *spu, guint8 *pkt_data, guint pkt_len) {
    struct reassembly_s *reassembly;
    static guint8 *buf; 
    static guint DCSQ_offset;
    static guint prev_DCSQ_offset = -1;
    static guint i = 0;
    static spu_clut_t *clut;
    static spu_clut_t *trans;
    static gint x,y,xl,yl;
    static glong time_exe;

    if (!(reassembly = spu_reassembly(pkt_data,pkt_len))) {
        return -1;
    }
    buf = reassembly->buf;
    DCSQ_offset = reassembly->cmd_offset;

    while (DCSQ_offset != prev_DCSQ_offset) { /* Display Control Sequences */
        i = DCSQ_offset;
        time_exe = (((buf[i]<<8)+buf[i+1]) * framerate) / 80; /* /100 -> /80 , 20% longer display */
        spu->time_execute = (guint)time_exe;
        i += 2;
        prev_DCSQ_offset = DCSQ_offset;
        DCSQ_offset = (buf[i]<<8) + buf[i+1];
        i += 2;
        while (buf[i]!=CMD_SPU_EOF) {		// Command Sequence
            switch ((guint8)buf[i]) {
                case CMD_SPU_SHOW: // show subpicture
                    //DONT HONOR; show_subpicture = TRUE;
                    i++;
                    break;
                case CMD_SPU_HIDE: // hide subpicture
                    //DONT HONOR THIS; show_subpicture = FALSE;
                    i++;
                    break;
                case CMD_SPU_SET_PALETTE: // CLUT
                    clut = (spu_clut_t*)&buf[i+1];
                    spu->clut_index[0] = clut->entry0;
                    spu->clut_index[1] = clut->entry1;
                    spu->clut_index[2] = clut->entry2;
                    spu->clut_index[3] = clut->entry3;
                    i += 3;
                    #ifdef DEBUG
                    g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: set palette %i, %i, %i, %i",spu->clut_index[0],spu->clut_index[1],spu->clut_index[2],spu->clut_index[3]);
                    #endif
                    break;
                case CMD_SPU_SET_ALPHA: // transparency palette
                    trans = (spu_clut_t*)&buf[i+1];
                    if (trans->entry0!=0 || trans->entry1!=0 || trans->entry2!=0 || trans->entry3!=0) {
                        spu->trans[0] = trans->entry0;
                        spu->trans[1] = trans->entry1;
                        spu->trans[2] = trans->entry2;
                        spu->trans[3] = trans->entry3;
                        #ifdef DEBUG
                        g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: set transparency : %i, %i, %i, %i",spu->trans[0],spu->trans[1],spu->trans[2],spu->trans[3]);
                        #endif
                    }
                    i += 3;
                    break;
                case CMD_SPU_SET_SIZE: // image coordinates
                    x = (buf[i+1] << 4) | (buf[i+2] >> 4);
                    xl = (((buf[i+2] & 0x0f) << 8) | buf[i+3]);
                    y = (buf[i+4]  << 4) | (buf[i+5] >> 4);
                    yl = (((buf[i+5] & 0x0f) << 8) | buf[i+6]);
                    if (x>-1 && x<xl && y>-1 && y<yl && xl>-1 && xl<subpicture_x && yl>-1 && yl<subpicture_y) {
                        spu->x = x;
                        spu->y = y;
                        spu->width =  xl - spu->x + 1;
                        spu->height = yl - spu->y + 1;
                        #ifdef DEBUG
                        g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: set size: x=%i, y=%i, xl=%i, yl=%i, width=%i, height=%i (mem1=%i,mem2=%i,mem3=%i,mem4=%i,mem5=%i,mem6=%i)",spu->x,spu->y,xl,yl,spu->width,spu->height,buf[i+1],buf[i+2],buf[i+3],buf[i+4],buf[i+5],buf[i+6]);
                        #endif
                        /* Private stuff */
                        spu->_x = 0;
                        spu->_y = 0;
                    } else {
                        #ifdef DEBUG
                        g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: set size outside limits: x=%i, y=%i, xl=%i, yl=%i, width=%i, height=%i (mem1=%i,mem2=%i,mem3=%i,mem4=%i,mem5=%i,mem6=%i)",spu->x,spu->y,xl,yl,spu->width,spu->height,buf[i+1],buf[i+2],buf[i+3],buf[i+4],buf[i+5],buf[i+6]);
                        #endif
                    }

                    i += 7;
                    break;
                case CMD_SPU_SET_PXD_OFFSET:	// image 1 / image 2 offsets
                    spu->offset[0] = (((guint)buf[i+1]) << 8) | buf[i+2];
                    spu->offset[1] = (((guint)buf[i+3]) << 8) | buf[i+4];
                    i += 5;
                    break;
                /*case CMD_SPU_MENU:
                    spu->clut_index[0] = 0;
                    spu->clut_index[1] = 9;
                    spu->clut_index[2] = 8;
                    spu->clut_index[3] = 12;
                    spu->trans[0] = 0;
                    spu->trans[1] = spu->trans[2] = spu->trans[3] = 15;
                    i++;
                    break; */
                default:
                    i++;
                    break;
            }
        }
        i++; // lose the CMD_SPU_EOF code (no need to, really)
        /* Until we change the interface we parse all 'Command Sequence's 
           but just overwrite the data in spu. Should be a list instead. */
    }
    /* Here we should have a linked list of display commands ready to 
       be decoded/executed by later calling some spu???() */
    return 0;
}


void spu_parse_data(struct spu_img_struct *spu) {
    guint color = 0;
    guint len;
    guint vlc;

    field = 0;
    spu_get_bits(0, spu);	// Reset/init bit code
    while ((spu->offset[1] < reassembly.cmd_offset)) {
        vlc = spu_get_bits(4, spu);
        if (vlc < 0x0004) {
            vlc = (vlc<<4) | spu_get_bits (4, spu);
            if (vlc < 0x0010) {
                vlc = (vlc<<4) | spu_get_bits (4, spu);
                if (vlc < 0x0040) {
                    vlc = (vlc<<4) | spu_get_bits (4, spu);
                }
            }
        }
        color = vlc & 0x03;
        len = vlc>>2;
        if (len > (spu->width - spu->_x) || len == 0) {
            len = spu->width - spu->_x;
        }

        spu_put_pixel(spu,len,color,TRUE);
        if (spu->_x >= spu->width) {
            if (spu_next_line(spu) < 0) {
                goto clean_up;
            }
        }
    }
    /* Like the eof-line escape, fill the rest of the sp. with background */
    spu_put_pixel(spu,spu->width-spu->_x,color,FALSE);
    while (!spu_next_line(spu)) {
        spu_put_pixel(spu,spu->width-spu->_x,color,FALSE);
    }
    clean_up:
    reassembly_flag = REASSEMBLY_START;
    return;
}


/*
 * This function will be called during the processing of the input mpeg2 stream. 
 * Every call the background frame will be overlayed by the subpicture, and the
 * counter 'frames_to_show_current_overlay' will be lowered. When this counter
 * reaches 0, overlayingi the pubpicture to the nackground will NOT be done anymore.
 * The subpicture frames that are used are rendered by 'subpicture_render(guchar*)'
 */
void spu_overlay(guint8 *bg_frame[3], guint x, guint y, guint o) {
    static gint index;
    if (show_subpicture==TRUE) {
        index = (gint)(255*(gdouble)frames_to_show_current_overlay/(gdouble)frames_to_show_current_overlay_value);
        #ifdef DEBUG
        g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: transparency=%f",transparency[index]);
        #endif
        overlay(bg_frame, x, y, o, &spu, transparency[index]);
        frames_to_show_current_overlay--;
        if (frames_to_show_current_overlay==0) {
            /* When frames_to_show reaches 0 quit overlaying.
               NOTE that DVD can have this counter set 0, so
               the subpicture is set to -1 once it reaches 
               this check, and so the sub is shown untill the
               next subpicture is available! */
            show_subpicture=FALSE;
        }
    }
    return;
}


/*
 * This function is called by the mpeg2 demuxer: every subpicture package is passed on here.
 * Only the prefered subpicture stream is used (language) to fill the local buffer. Once the
 * buffer is filled with enough data to render the next subpicture frame, this is done and 
 * the function subpicture_overlay(...) will automatically use it to apply to the background
 * frames.
 */
void spu_render(guchar* data, glong pts) {
    guint pkt_len;
    guchar* data_current;
    gint i;
    guint data_size_total;
#ifdef SPU_CRT
    glong CRC = 0;
#endif 
        
    /* SPU initialised? If not exit */
    if (data_buffer==NULL) {
        #ifdef DEBUG
        g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: Data buffer not initialized yet");
        #endif
        return;
    }

    #ifdef DEBUG
    g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: Packet PTS value = %li",pts);
    #endif

    /* Check for language, based on values from subpicture_init */
    if (subpicture_stream != (data[23 + data[22]])) {
        /* not the prefered language, leave */
        #ifdef DEBUG
        g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: Packet not selected language (language:%i, package id:%i)",subpicture_stream,data[23 + data[22]]);
        #endif
        return;
    }
    #ifdef DEBUG
    g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: Packet of selected language (%i)",subpicture_stream);
    #endif


#ifdef SPU_CRT
    /* Calculate a CRC value for this package. CRC's are used to determine whether or not a 
     * package has already been processed. Somehow packages arrive multiple times here... */
    //TODO: CRC value too simple?? 
    for (i=0;i<(2048);i++) {
        CRC+=data[i];
    }

    /* CRC check current package */
    for (i=0;i<CRClist_counter;i++) {
        if (CRC==CRClist[i]) {
            g_log(SPU_LD,G_LOG_LEVEL_WARNING,"SPU: CRC hit, not using this package");
            return;
        }
    }

    /* CRC passed, put current CRC value in history */
    CRClist[CRClist_counter] = CRC;
    CRClist_counter++;
    /* Handle overflow: move last 1000 CRC value to begin and reset counter */
    if (CRClist_counter==65535) {
        CRClist_counter = 1000;
        for (i=0;i<1000;i++) {
            CRClist[i]=CRClist[i+64535];
        }
        /* NOTE: not cleaning of remaining entries, not needed due to counter */
    } 
#endif //SPU_CRT


    /* Add new data to data_buffer & recalc data_buffer_size */
    memcpy(data_buffer+data_buffer_size, data+24+data[22], 2048-(24+data[22]) );
    data_buffer_size += 2048-(24+data[22]);

    /* Parse off the padding bytes */
    data_current = data_buffer;
    pkt_len = data_current[0];
    while((pkt_len == 0xff) && (data_current<(data_buffer+data_buffer_size))) {
        data_current++;
        pkt_len = data_current[0];
    }
    if (pkt_len == 0xff) {
        g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: buffer overflow, resetting buffer and discarding package");
        memset(data_buffer,0,2000000);
        data_buffer_size = 0;
        return;
    }

    /* Packet length */
    pkt_len = (pkt_len << 8) + data_current[1];
    if (pkt_len == 0x0) {
        g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: pakket lenght is 0, resetting buffer and discarding package");
        memset(data_buffer,0,2000000);
        data_buffer_size = 0;
        return;
    } else if (pkt_len > 50000) {
        g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: pakket lenght > 50000, resetting buffer and discarding package");
        memset(data_buffer,0,2000000);
        data_buffer_size = 0;
        return;
    }

    /* Do we have enough data buffered? */
    if ((pkt_len+6)>data_buffer_size) {
        /* No, return. Package reassembling will continue with next spu */
        return;
    }

    /* Render new subpicture frame */
    show_subpicture = TRUE; /* default to SHOW when a new subpicture is ready.. TODO:???? */
    data_size = (((guint)data_current[2]) << 8) + data_current[3];
    data_size_total = (data_current[data_size+2] << 8) + data_current[data_size+3];
    /* Check packet size, allow +24 and +0 also.. */
    if ((data_size+24 != data_size_total) && (data_size != data_size_total)) {
        g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: datasize's mismatch (data_size=%i, data_size_total=%i), resetting buffer",data_size,data_size_total);
        /* reset buffer */
        memset(data_buffer,0,2000000);
        data_buffer_size = 0;
        return;
    } else {
        /* Process package */
        memset(spu.data,0,2000000);
        if (!spu_parse_header(&spu, data_current, pkt_len)) {
            spu_parse_data(&spu);
        }
        frames_to_show_current_overlay = spu.time_execute; /* displayed subpicture frames */
        frames_to_show_current_overlay_value = frames_to_show_current_overlay; // store for later usage
    }

    /* Update data_buffer & size */
    for (i=pkt_len+6; data_buffer[i]==0xff; i++) {;}

    /* TEMP: clean buffer */
    if (data_buffer_size<i) {
        /* more to clear as we got? -> reset buffer */
        g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: buffer too small to clear, more to clear as is actually there? Resetting buffer");
        memset(data_buffer,0,2000000);
        data_buffer_size = 0;
        return;
    }
    /* Rotate buffer */
    #ifdef DEBUG
    g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: Data added to buffer");
    #endif
    g_memmove(data_buffer, data_buffer+i, data_buffer_size-i );
    data_buffer_size -= i;
    return;
}

/* 
 * Initializes the subpicture handling to:
 * - language   : language, valued 0-31 ( the += 0x20 is done here as you can see
 * - sx         : width of source background frame
 * - sy         : height of source background frame
 * - FR         : Framerate
 * - clutfile   : Filename of 32x32bits colour table (16x RGB 24bits colour, 16x I420 12bits colour)
 * - log_domain : glib g_log() logging domain. 
 */
void spu_init(gint language, gint sx, gint sy, gint ctop, gint cbottom, gint cleft, gint cright, gdouble FR, gchar* clutfile, gchar* log_domain) {
    gint i;
 
    /* Publice log domain */
    SPU_LD = log_domain;
    g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: Logging to domain %s",SPU_LD);
#ifdef SPU_CRT
    /* Clear CRC table */
    for (i=0;i<65535;i++) {
        CRClist[i] = 0;
    }
#endif //SPU_CRT
    /* Read CLUT table */
    g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: Opening clut file %s",clutfile);
    if (clutfile!=NULL) {
        fd = open(clutfile,O_RDONLY);
        for (i=0;i<16;i++) {  // Read BGR colours (for backward compatibility
            read(fd,&spu_clut[i],sizeof(u_int));
            g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: BGR Colour %i : %x",i,spu_clut[i]);
        }
        for (i=0;i<16;i++) {  // Read YVU colours, these ones are used
            read(fd,&spu_clut[i],sizeof(u_int));
            g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: I420 Colour %i : %x",i,spu_clut[i]);
        }
        close(fd);
    } else {
        g_log(SPU_LD,G_LOG_LEVEL_WARNING,"SPU: Using default colour table. This wont produce correct colours. Use gnomedrip for DVD content caching to get the correct table.");
        /* Use static table, Pretty useless besides generating some response */
        spu_clut[0] = 0xffffff;
        spu_clut[1] = 0xE0E0E0;
        spu_clut[2] = 0xffffff;
        spu_clut[3] = 0x000000;
        spu_clut[4] = 0x005555;
        spu_clut[5] = 0x00ff00;
        spu_clut[6] = 0xcc2255;
        spu_clut[7] = 0xcc0055;
        spu_clut[8] = 0x404040;
        spu_clut[9] = 0x202020;
        spu_clut[10] = 0xb0b0b0;
        spu_clut[11] = 0xd0d0d0;
        spu_clut[12] = 0x606000;
        spu_clut[13] = 0x707000;
        spu_clut[14] = 0x808000;
        spu_clut[15] = 0x606000;
    }
    /* Reset variables */
    g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: X=%i, Y=%i, CT=%i, CB=%i",sx,sy,ctop,cbottom);
#ifdef SPU_CRT
    CRClist_counter = 0;
#endif
    subpicture_stream = language + 0x20;  /* language code + 0x20 */
    subpicture_x = sx;
    subpicture_y = sy;
    spu.cliptop = ctop;
    spu.clipbottom = cbottom;
    spu.clipleft = cleft;
    spu.clipright = cright;
    spu.xbase = spu.clipleft + spu.clipright;
    spu.ybase = spu.cliptop + spu.clipbottom;
    framerate = FR;
    show_subpicture = FALSE;

    /* Fill transparency table, this is used to fade in and fade out the overlayed SPU bitmaps */
    for (i=0;i<256;i++) {
        transparency[i] = 2.7 * cos((gdouble)(i-128)/128) - 1;
        if (transparency[i]>1) transparency[i] = 1;
    }

    /* Init Overlaying */
    overlay_init();

    /* Allocate buffers */
    if (buffer==NULL) buffer = (u_char*)malloc(2000000); /* holds rendered subpicture frame */
    if (data_buffer==NULL) data_buffer = (u_char*)malloc(2000000);
    if (spu.data==NULL) spu.data = malloc(2000000);
    if (reassembly.buf==NULL) reassembly.buf = malloc(2000000);
    #ifdef DEBUG
    g_log(SPU_LD,G_LOG_LEVEL_DEBUG,"SPU: Initialized SPU engine");
    #endif
    return;
}


