/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/extensions/instrumentation/annotation.hpp
 *
 * $Id: configuration.hpp,v 1.10 2005/06/23 09:54:18 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @brief [LEVEL: alpha] 
 */
#ifndef DIAGNOSTICS__EXTENSIONS__INSTRUMENTATION__ANNOTATION_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__EXTENSIONS__INSTRUMENTATION__ANNOTATION_HPP__INCLUDE_GUARD

#include <diagnostics/extensions/instrumentation/load_symbol.hpp>
#include <diagnostics/frame/level.hpp>
#include <diagnostics/frame/pp_config.hpp>
#include <diagnostics/macros/block_annotation.hpp>

#include <string>

/**
 * @brief Instruments a call to some C-procedure to generate a log
 * message on entering and leaving this procedure (@ref
 * ::diagnostics::TYPE_PROCEDURE_ENTER, @ref
 * ::diagnostics::TYPE_PROCEDURE_EXIT).
 *
 * The macro outpus a procedure with the signature of the original
 * procedure which generates the log messages, loads the procedure
 * from a shared library (on the first time calling), and calls the
 * procedure then.
 *
 * @note This SHOULD be thread safe -- load_symbol calls dlopen, and
 * the latter can be called more than once safely. Thus two threads
 * might concurrently load the same symbol, but the obtained and
 * stored pointer should be identical.
 * 
 * @param LEVEL is the @a ::diagnostics::Level_t of the log messages
 * to generated.
 * @param LIB_NAME is the name of library to provide the procedure
 * @param RETURN_TYPE is the return type of procedure
 * @param PROC is the procdure -- written as identifier 
 * @param ARGS are the arguemtns to the procdure, completely written
 * out with embracing parenthesis.
 * @param THROW is the throw declaration of the procdure.
 * @param MESG is the what string to be added into the generated log-messages,
 * prefixed by PROCDURE=<PROCEDURE SIGNATURE>.
 */ 
#define DIAGNOSTICS_BASE_INSTRUMENT_C_CALL(LEVEL,LIB_NAME,RETURN_TYPE,PROC,ARGS,THROW,CALL,MESG) \
RETURN_TYPE PROC ARGS THROW \
{ \
	DIAGNOSTICS_BASE_BLOCK_GUARD((LEVEL), \
                                 ::diagnostics::TYPE_PROCEDURE_ENTER, \
                                 ::diagnostics::TYPE_PROCEDURE_EXIT,0, \
								 ::std::string("PROCEDURE=\"") + DIAGNOSTICS_FUNC_NAME + "\" " + (MESG)); \
   typedef RETURN_TYPE (*Proc_t) ARGS ; \
   static Proc_t real_proc=NULL; \
   if(real_proc==NULL) \
   real_proc = (Proc_t)::diagnostics::internal::load_symbol((LIB_NAME),#PROC); \
   return (*real_proc) CALL; \
} void do_not_call_this_method_which_is_only_declared_for_a_semicolon(void)


#endif
