/* datastructures.h
 *
 * the important data structures and defines for denemo, a gtk+ frontend to
 * Lilypond, the GNU music typesetter
 *
 * (c) 1999, 2000, 2001 Matthew Hiller <matthew.hiller@yale.edu>
 *
 */

#ifndef DENEMO_DATASTRUCTURES

#define DENEMO_DATASTRUCTURES
using namespace::std;
/* Include the internationalization stuff.  */
/* Only if not on Cygwin, because libintl does not work with mingw */
/* G_OS_WIN32 is defined in gtk.h for mingw */

#include <gtk/gtk.h>
#include <gmodule.h>
#ifndef G_OS_WIN32
# include <libintl.h>
# include <locale.h>
# define _(String) gettext (String)
# ifdef gettext_noop
#  define N_(String) gettext_noop (String)
# else
#  define N_(String) (String)
# endif
#else
# define _(String) (String)
# define N_(String) (String)
# define setlocale(x,y)
# define bindtextdomain(x,y)
# define textdomain(x)
#endif




#include <gtk/gtk.h>
#include <glib.h>

#if GTK_MAJOR_VERSION <= 1
#define GtkTextBuffer GtkWidget
#define g_string_append_printf g_string_sprintfa
#endif


#define MAX_UNDOS 50
/* This unit of memory size is used all over the place.  */

#define SEVENGINTS (sizeof (gint) * 7)

/* and the following typedefs are basically here for so that it's
 * possible to understand what my code is doing -- just as much for
 * my sake as yours!
 *
 * What can I say; I've done a lot of programming in Java and
 * SML/NJ; 
 * I like my type names to convey information. */

/* The ->data part of each objnode presently points to a mudelaobject */

typedef GList objnode;


/* The idea here is to make everything recursive.  The dominant
   paradigm is a linked list.  Handy that there's such a nice
   precooked implementation of them in glib, eh?  Note, however, that
   the whole score isn't treated as a linked list of notes and other
   objects as it is in, say Rosegarden; instead, the program has a
   linked list of musical objects for each measure, and only then are
   measures linked into staffs.  That's my main beef with Rosegarden
   -- I don't tend to compose my stuff in the order in which it will
   eventually be played. As such, I like being able to start entering
   music in the middle of my score, then the beginning, then the end,
   or whatever, as appropriate.  */

enum objtype
{
  CHORD,
  TUPOPEN,
  TUPCLOSE,
  CLEF,
  TIMESIG,
  KEYSIG,
  BARLINE,
  LILYDIRECTIVE,
  COMMENT,
  STEMDIRECTIVE,
  MEASUREBREAK,
  DYNAMIC,
  GRACE_START,
  GRACE_END,
  LYRIC,
  FIGURE
};

enum headtype
{ 
  DENEMO_NORMAL_NOTEHEAD,
  DENEMO_CROSS_NOTEHEAD,
  DENEMO_HARMONIC_NOTEHEAD,
  DENEMO_DIAMOND_NOTEHEAD
};

enum input_mode {
  INPUTNORMAL,
  INPUTREST,
  INPUTBLANK,
  INPUTBLANKREST,
  REPLACE,
  TRAVERSE
};

enum action_type {
  ACTION_INSERT,
  ACTION_DELETE,
  ACTION_CHANGE
};

typedef struct grace
{
 
  gboolean on_beat;
  gint duration;
}
grace;
  
typedef struct note
{
  gint mid_c_offset;
  gint enshift;
  gboolean reversealign;
  gboolean showaccidental;
  /* Holds number of pixels to the left of the notehead that the
     accidental should be drawn at.  */
  gint position_of_accidental;
  /* Holds y co-ordinate for drawing rather than recalculating it each
     run through the drawing loop.  */
  gint y;
  /* Holds note head type.  */
  enum headtype noteheadtype;
  /* Holds x offset from default position.  */
  gint x_off;
  /* Holds y offset from default position.  */
  gint y_off;
 
}
note;

typedef struct chord
{
  GList *tones;			/* Holds pointers to actual note structures */
  /* test for new dynamic handling */
  GList *dynamics;
  gboolean has_dynamic;
  gint highestpitch;
  gint lowestpitch;
  gint highesty;
  gint lowesty;
  gint baseduration;
  gint numdots;
  gint sum_mid_c_offset;
  gint numtones;

  gboolean is_tied;
  gboolean is_stemup;
  gboolean is_reversealigned;
  gboolean slur_begin_p;
  gboolean slur_end_p;
  gboolean crescendo_begin_p;
  gboolean crescendo_end_p;
  gboolean diminuendo_begin_p;
  gboolean diminuendo_end_p;
  gboolean hasanacc;
  /*Flag to show whether note is a grace note*/
  gboolean is_grace;
  gboolean struck_through; /*Flag for a struck through stem*/
  gint stemy;		/* Stores the y position of the end of the stem */
  gboolean has_stacatto_p; 
  gboolean is_accented_p; 
  gboolean has_fermata_p; 
  gboolean has_tenuto_p;
  gboolean has_trill_p;
  gboolean has_turn_p;
  gboolean has_mordent_p;
  gboolean has_staccatissimo_p;
  gboolean has_marcato_p;
  gboolean has_ubow_p;
  gboolean has_dbow_p;
  gboolean has_rheel_p;
  gboolean has_lheel_p;
  gboolean has_rtoe_p;
  gboolean has_ltoe_p;

  /* Note highlighting flag if
   * true change notes colour
   */
  gboolean is_highlighted; 

  GString *lyric;
  gboolean is_syllable;
  gboolean center_lyric;
  gboolean is_figure; /* the reason for using this boolean is to exploit 
			 the fact that all the spacing and printing of 
			 figures can use the code for the CHORD type */
  gpointer figure; /* when this chord is a bass note 
		      (figure !=NULL && is_figure==FALSE) this
		      pointer points to an objnode in a FiguredBassStaff.
		      That objnode's data is a mudelaobject of type CHORD.
		      It is linked into the corresponding FiguredBassStaff if
		      one exists.
		      When this chord is_figure then figure is a 
		      GString* containing the
		      figures in lilypond format. */
}
chord;

/* Indicator that a tuplet is starting */

typedef struct tupopen
{
  gint numerator;
  gint denominator;
}
tupopen;

enum clefs
{
  DENEMO_TREBLE_CLEF,
  DENEMO_BASS_CLEF,
  DENEMO_ALTO_CLEF,
  DENEMO_G_8_CLEF,
  DENEMO_TENOR_CLEF,
  DENEMO_SOPRANO_CLEF
};

/* Indicator for a clef change */

typedef struct clef
{
  enum clefs type;
}
clef;

/* Indicator for a time-signature change. Only appears at the
 * beginning of a measure */

typedef struct timesig
{
  gint time1;
  gint time2;
}
timesig;

/* Indicator for a key-signature change. */

typedef struct keysig
{
  gint number;
  gboolean isminor;
  gint accs[7];
}
keysig;

/* Dynamic */

typedef struct dynamic 
{
  GString *type;
}
dynamic;

enum barline_type {
  ORDINARY_BARLINE,
  DOUBLE_BARLINE,
  END_BARLINE,
  OPENREPEAT_BARLINE,
  CLOSE_REPEAT,
  OPEN_CLOSE_REPEAT
};

typedef struct barline
{
  gint type;
}
barline;

/* Data structure for an arbitrary directive to be passed to
 * Lilypond that Denemo doesn't understand. This type is useful
 * in that the user can insert such directives into a mudela file
 * by hand and have Denemo respect them when it loads a file and
 * write them back when it saves it */


/* Lyric datastructure */
typedef struct lyric
{
  GString *lyrics;
  gint position;
  gboolean is_syllable;
  gboolean center_lyric;
}lyric;


typedef struct lilydirective
{
  GString *directive;
}
lilydirective;

enum stemdirections
{
  DENEMO_STEMDOWN=1,
  DENEMO_STEMBOTH,
  DENEMO_STEMUP
};

/* Indicator that the following music should be all stemup, all
 * stemdown, or stemmed normally */

typedef struct stemdirective
{
  enum stemdirections type;
}
stemdirective;

/* A comment in the mudela file; treated similarly to a lilydirective.  */

typedef struct mudelacomment
{
  GString *text;
}
mudelacomment;

/* obsolete union as been replaced by void pointer instead
 typedef union mudelaobjunion
{
  chord chordval;
  tupopen tupval;
  clef clefval;
  timesig timeval;
  keysig keyval;
  dynamic dynval;
  barline blval;
  lilydirective lilyval;
  mudelacomment commentval;
  stemdirective stemval;
  grace graceval;
}
mudelaobjunion;
 */

typedef struct mudelaobject
{
  enum objtype type;
  gchar *user_string;/* holds user's original text parsed to generated this 
			object */
  gint basic_durinticks;
  gint durinticks;
  gint starttick;
  gint starttickofnextnote;
  /* Allots extra space for accidentals or reverse-aligned notes if
   * the stem is down */
  gint space_before;
  gint minpixelsalloted;
  /* Holds x co-ordinate relative to the beginning of the measure. */
  gint x;
  gboolean isstart_beamgroup;
  gboolean isend_beamgroup;
  /* mudelaobjunion u; has become a pointer*/
  gpointer object;
  gboolean isinvisible;
}
mudelaobject;


/* The ->data part of each measurenode points to an objlist, which is
   a list of the musical objects in that measure. */

typedef GList measurenode;

typedef struct staff
{
  measurenode *measures;
  gint sclef;
  gint skey;
  gint skey_isminor;
  gint skeyaccs[7];
  gint stime1;
  gint stime2;
  gint leftmost_clefcontext;
  gint leftmost_keysigcontext;
  gint leftmost_keywidth;
  gint leftmost_time1context;
  gint leftmost_time2context;
  gint leftmost_keyaccs[7];
  gint leftmost_stem_directive;

  /*
   * Staff Parameters
   * Added Adam Tee 27/1/2000, 2001 
   */
  gint no_of_lines;
  gint transposition;
  gint pos_in_half_lines;
  gint volume;
  /* Back to Hiller stuff */
  GString *staff_name;
  /* RTS: I've introduced the staff name here, the other two are versions
     of the voice name; however I'm still sticking to the unwritten convention
     that each staff's voices are contiguous in si->thescore. Without this you
     can't have same named voices in different staffs. */
  GString *denemo_name;
  GString *lily_name;
  GString *midi_instrument;

  gint space_above;
  gint space_below;
  gint voicenumber;
  /* presently set to 2 for any non-primary voices; we might want to
   * change that, though */
  measurenode ** is_parasite; /* points to address of host staff's measures 
				 field if measures are owned by another 
				 staff */
}
staff;

/* The ->data part of each staffnode points to a staff structure */

typedef GList staffnode;
typedef staffnode *score;

/* a pair of staffs, used to relate two staffs together */
typedef struct staff_info
{
  staff *main_staff; /* eg the bass line or the vocal part */
  staff *related_staff; /* eg the figures for the bass or the lyrics*/
}
staff_info;


/* This structure is what's stored in the keymap data structure, both
   in the array and in the hash tables.  Note that some redundant
   information is stored in either case, but either one structure or
   the other will need or find it useful to have each piece of information. */

union func_union
{
  GtkFunction nocallback;
  GtkItemFactoryCallback1 callback;
};

struct keybinding_info
{
  gint keyval;
  gint state;
  gint command_number;
  gint callback_action;
  union func_union func;
};

struct keymap
{
  GList **commands;
  GHashTable *quick_lookup_hashes;
};

/* prefinfo holds information on user preferences. */

struct prefinfo
{
  GString *lilypath;
  GString *midiplayer;
  GString *audioplayer;
  gboolean playbackoutput;
  gboolean immediateplayback;
  gboolean lilyentrystyle;
  gboolean createclones;
  /*Auto save data*/
  gboolean autosave;
  gint autosave_timeout;
  struct keymap *the_keymap;
};

struct mudelaheaderinfo {
/* Mudela file header information */
  GString *title;
  GString *subtitle;
  GString *poet;
  GString *composer;
  GString *meter;
  GString *opus;
  GString *arranger;
  GString *instrument;
  GString *dedication;
  GString *piece;
  GString *head;
  GString *copyright;
  GString *footer;
  GString *tagline;
/*  GString *; */

};

/*
 * Undo data. holds data for undo 
 * Borrowed idea from GScore
 */
struct undo_data
{
  gpointer object;    /* pointer to object deleted */
  gint staffnum;      /* staff number */     
  gint measurenum;    /* measure number */
  gint position;      /* position in bar */
  enum action_type action; /*action type */

};
 
/* Redo data 
 * Contains data required for redo operation 
 * contents as per undo_data
 */
struct redo_data
{
  gpointer object;
  gint staffnum;
  gint measurenum;
  gint measure;
  gint position;
  enum action_type action;
};


/* struct scoreinfo is actually a somewhat misleading name.  It contains
 * within it a struct prefinfo *, and when Denemo supports editing
 * multiple scores simultaneously, each score's scoreinfo structure will
 * contain a pointer to the same single prefinfo structure.
 *
 * I originally hit on this idea as a hack -- everything in Denemo was
 * coded to pass around struct scoreinfo *s, and I didn't want to
 * change this around -- but then I realized that it was a reasonably
 * good idea.  Basically all operations need access to the scoreinfo
 * structure, and a handful need access to the prefinfo structure as
 * well. Only the preferences dialog needs access to _just_ the
 * prefinfo structure.  So, why not keep things simple and always pass
 * around scoreinfos? */

struct scoreinfo
{
  /* Fields used fairly directly for drawing */
  GtkWidget *window;
  GtkWidget *scorearea;
  GdkPixmap *pixmap;
  GtkObject *vadjustment;
  GtkWidget *vscrollbar;
  GtkObject *hadjustment;
  GtkWidget *hscrollbar;
  GtkWidget *menubar;
  GtkWidget *statusbar;
  /* text of lily input string corresponding to curobj */
  GtkTextBuffer *textbuffer; 
  GtkWidget *textwindow; /* top level window containing lily textwindow */
  GtkWidget *textview; /* text entry containing lily text to be edited */
  GtkWidget *musicdatabutton; /* button for editing music data textually */
  gulong sigid; /* signal handler id for when text is changed */

  GtkWidget *musictext; /* status bar with text of lily input string 
			   corresponding to music item at cursor */
  GList *curlilynode; /* the node of the lily parse tree on display 
			 in textwindow */
  GList *lily_file; /* root of lily file parse, see lilyparser.y etc  */
  
  gint status_context_id;
  gint leftmeasurenum; /* start at 1 */
  gint rightmeasurenum;/* start at 1 */
  gint top_staff;
  gint bottom_staff;
  gint measurewidth;
  /* List of all minimum measure widths */
  GList *measurewidths;
  gint widthtoworkwith;
  gint staffspace;

  /* Fields that have more to do with the data model and its manipulation,
   * though they may be modified by side-effects of the drawing routines */
  score thescore;
  
  staffnode *currentprimarystaff;
  staffnode *currentstaff;
  gint currentstaffnum;
  measurenode *currentmeasure;
  gint currentmeasurenum;/* start at 1 */
  objnode *currentobject;
  /* currentobject points to the note preceding the cursor when the
   * cursor is appending. == NULL only when currentmeasure is empty. */
  gint cursor_x;
  gint cursor_y;
  gint staffletter_y;
  gint maxkeywidth;
  gboolean cursor_appending;
  enum input_mode mode;
  gboolean cursoroffend;
  gint cursorclef;
  gint cursoraccs[7];
  gint cursortime1;
  gint cursortime2;
  gint curmeasureclef;
  gint curmeasurekey;
  gint curmeasureaccs[7];
  gint nextmeasureaccs[7];
  /* These are used for determining what accidentals should be there
   * if the cursor advances to the next measure from the next "insert chord"
   * operation */
  gint curmeasure_stem_directive;

  gboolean is_grace_mode;
  /* Is there a figured bass present, is so this relates the bass
   * with its figures staff, if one is present */
  staff_info * has_figures;
  /* Now stuff that's used for marking areas */
  gint markstaffnum;
  gint markmeasurenum;
  gint markcursor_x;
  gint firststaffmarked;
  gint laststaffmarked;
  gint firstmeasuremarked;
  gint lastmeasuremarked;
  gint firstobjmarked;
  gint lastobjmarked;

  gboolean haschanged;

  /*Temp field for testing for a slur*/
  gboolean is_slured;

  /* Fields used for MIDI playback */
  gint tempo;
  gint start;
  gint end;
  gint stafftoplay;

  struct mudelaheaderinfo *headerinfo;
  
  /* Filename */
  GString *filename;
  GString *autosavename;

  /* And the prefinfo pointer */
  struct prefinfo *prefs;
  
  GList *savebuffer;
  /* Holds List of Plugin Names */
  GList *plugins;
  gint plugincounter;

  /* a list of scoreinfos for each scoreblock in the (lilypond) file
     the last one is always the one currently at the address hardwired
     into the display functions.
     NULL if just one scoreblock in file */
  GList *scoreblocks;

  /*list of undo data */
  GList *undodata;
  gint undo_level; /*denotes undo level */
  GList *redodata;
  gint redo_level;
  gint undo_redo_mode;
  

};



typedef struct denemo_plugin
{
  void (* plugin_init) (struct scoreinfo *si); /*Initialisation Function*/
  void (* plugin_cleanup) (struct scoreinfo *si); /*Cleanup Function */
  GString *title; /*Name of plugin */
}denemo_plugin;


/*Forward Declaration*/
typedef struct _PluginData PluginData;
/* Structure borrowed from Gnumeric which 
 * allows us to unload a plugin.
 * This should render the denemo_plugin struct redundant.
 */
struct _PluginData {
  GModule *handle;
  void (*initialise) (struct scoreinfo *si, PluginData *pd);
  void (*clean_up)   (struct scoreinfo *si, PluginData *pd);
  gchar *title;
};


#define g_malloc(a) g_malloc0(a) /* for extensible debuggable code */

#endif  /* #ifndef DENEMO_DATASTRUCTURES  */
